<?php
declare(strict_types=1);


namespace DailyTarot\Rest;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Analytics\Tracker;
use DailyTarot\Calendar\DayEntry;
use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Frontend\ReadableRoutes;
use DailyTarot\Meaning\MeaningPackRepository;
use DailyTarot\Reading\ReadingComposer;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\RateLimit;
use DailyTarot\Support\CacheVersion;
use DailyTarot\Support\Log;

final class ReadingsController {

    private static function addCacheHeaders(\WP_REST_Response $res, string $etag, int $lastModifiedUnixTs, int $maxAgeSeconds = 300): \WP_REST_Response {
        $etag = trim($etag);
        if ($etag !== '') {
            $res->header('ETag', $etag);
        }
        if ($lastModifiedUnixTs > 0) {
            $res->header('Last-Modified', gmdate('D, d M Y H:i:s', $lastModifiedUnixTs) . ' GMT');
        }
        $maxAgeSeconds = max(0, (int)$maxAgeSeconds);
        $res->header('Cache-Control', 'public, max-age=' . $maxAgeSeconds);
        return $res;
    }

    public static function init(): void {
        add_action('rest_api_init', [__CLASS__, 'registerRoutes']);
    }

    public static function registerRoutes(): void {
        register_rest_route('dtarot/v1', '/readings/(?P<date>\d{4}-\d{2}-\d{2})', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'getByDate'],
            'permission_callback' => '__return_true',
            'args' => [
                'date' => [
                    'validate_callback' => fn($v) => is_string($v) && (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $v),
                ],
            ],
        ]);

        register_rest_route('dtarot/v1', '/readings/latest', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'getLatest'],
            'permission_callback' => '__return_true',
            'args' => [
                'limit' => [
                    'validate_callback' => fn($v) => is_numeric($v),
                    'default' => 1,
                ],
                'from' => [
                    'validate_callback' => fn($v) => $v === null || $v === '' || (is_string($v) && (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $v)),
                    'default' => '',
                ],
            ],
        ]);
    }

    public static function getByDate(\WP_REST_Request $req) {
        if (class_exists(RateLimit::class) && !RateLimit::hit('rest_reading_by_date', 120, 60)) {
            if (class_exists(Log::class)) {
                Log::add('warn', 'rate_limited', 'REST reading by-date rate limited', ['action' => 'rest_reading_by_date']);
            }
            return new \WP_Error('dtarot_rate_limited', __('Too many requests.','daily-tarot'), ['status' => 429]);
        }

        $date = (string)$req['date'];
        $entry = DayEntryService::getPublished($date);
        if (!$entry) {
            return new \WP_Error('dtarot_not_found', __('Not found.','daily-tarot'), ['status' => 404]);
        }

        if (class_exists(Tracker::class)) {
            Tracker::trackPublishedReading($entry, 'rest_by_date', $date);
        }

        $payload = self::formatReading($date, $entry);
        $res = new \WP_REST_Response($payload, 200);

        $lastmod = class_exists(CacheVersion::class) ? CacheVersion::getDayLastModified($date) : 0;
        $ver = class_exists(CacheVersion::class) ? CacheVersion::get() : 1;
        $etag = 'W/"dtarot-rest-reading-' . md5($ver . '|' . $date . '|' . $lastmod) . '"';
        return self::addCacheHeaders($res, $etag, $lastmod, 300);
    }

    public static function getLatest(\WP_REST_Request $req) {
        if (class_exists(RateLimit::class) && !RateLimit::hit('rest_reading_latest', 120, 60)) {
            if (class_exists(Log::class)) {
                Log::add('warn', 'rate_limited', 'REST reading latest rate limited', ['action' => 'rest_reading_latest']);
            }
            return new \WP_Error('dtarot_rate_limited', __('Too many requests.','daily-tarot'), ['status' => 429]);
        }

        $limit = (int)$req->get_param('limit');
        if ($limit <= 0) $limit = 1;
        if ($limit > 30) $limit = 30;

        $from = (string)$req->get_param('from');
        $items = DayEntryService::latestPublished($limit, $from);

        $out = [];
        foreach ($items as $date => $entry) {
            if (!is_string($date) || !($entry instanceof DayEntry)) continue;
            if (class_exists(Tracker::class)) {
                Tracker::trackPublishedReading($entry, 'rest_latest', $date);
            }
            $out[] = self::formatReading($date, $entry);
        }

        $payload = [
            'count' => count($out),
            'items' => $out,
        ];

        $res = new \WP_REST_Response($payload, 200);
        $ver = class_exists(CacheVersion::class) ? CacheVersion::get() : 1;
        $etag = 'W/"dtarot-rest-latest-' . md5($ver . '|' . $limit . '|' . $from) . '"';
        return self::addCacheHeaders($res, $etag, 0, 120);
    }

    private static function formatReading(string $date, DayEntry $entry): array {
        $deckId = $entry->deckId;
        $packId = $entry->packId;
        $cardId = $entry->cardId;

        $meaning = MeaningPackRepository::getMeaning($packId, $cardId);

        $content = $entry->content;
        $dailyText = $entry->dailyText;
        $fallback = ReadingComposer::applyMeaningFallback($content, $dailyText, $meaning);

        $deckSlug = '';
        if ($deckId > 0) {
            $p = get_post($deckId);
            if ($p && isset($p->post_name)) $deckSlug = (string)$p->post_name;
        }
        $packSlug = '';
        if ($packId > 0) {
            $p = get_post($packId);
            if ($p && isset($p->post_name)) $packSlug = (string)$p->post_name;
        }

        return [
            'date' => $date,
            'readable_url' => class_exists(ReadableRoutes::class) ? ReadableRoutes::urlForDate($date) : '',
            'deck_id' => $deckId,
            'deck_slug' => $deckSlug,
            'card_id' => $cardId,
            'card_name' => Cards::name($cardId),
            'pack_id' => $packId,
            'pack_slug' => $packSlug,
            'content' => $fallback['content'],
            'daily_text' => $fallback['daily_text'],
        ];
    }
}
