<?php
declare(strict_types=1);


namespace DailyTarot\Registry;
if (!defined('ABSPATH')) { exit; }


final class Cards {

    public const SYSTEM_TAROT = 'tarot';
    public const SYSTEM_LENORMAND = 'lenormand';
    public const SYSTEM_KIPPER = 'kipper';
    public const SYSTEM_GYPSY = 'gypsy';
    public const SYSTEM_KIPPER_FIN_DE_SIECLE = 'kipper_fin_de_siecle';

    /** @return array<string,string> */
    public static function systems(): array {
        return [
            self::SYSTEM_TAROT => __('Tarot','daily-tarot'),
            self::SYSTEM_LENORMAND => __('Lenormand','daily-tarot'),
            self::SYSTEM_KIPPER => __('Kipper (Standard)','daily-tarot'),
            self::SYSTEM_GYPSY => __('Gypsy (Tarot Gypsy, 36)','daily-tarot'),
            self::SYSTEM_KIPPER_FIN_DE_SIECLE => __('Kipper (Fin de Siècle, 39)','daily-tarot'),
        ];
    }

    public static function normalizeSystem(?string $system): string {
        $system = is_string($system) ? sanitize_key($system) : '';
        if ($system === '') return self::SYSTEM_TAROT;

        // Aliases for Fin de Siècle variant.
        if (in_array($system, ['kipper_fds','kipper_fin_de_siecle','kipper_fin_de_siecle_39'], true)) {
            return self::SYSTEM_KIPPER_FIN_DE_SIECLE;
        }

        return array_key_exists($system, self::systems()) ? $system : self::SYSTEM_TAROT;
    }

    /** @return array<string,string> */
    public static function all(): array {
        // Backwards-compatible default: Tarot
        return self::forSystem(self::SYSTEM_TAROT);
    }

    /** @return array<string,string> */
    public static function forSystem(string $system): array {
        $system = self::normalizeSystem($system);

        if ($system === self::SYSTEM_LENORMAND) {
            $cards = [
                'lenormand_01' => __('Rider','daily-tarot'),
                'lenormand_02' => __('Clover','daily-tarot'),
                'lenormand_03' => __('Ship','daily-tarot'),
                'lenormand_04' => __('House','daily-tarot'),
                'lenormand_05' => __('Tree','daily-tarot'),
                'lenormand_06' => __('Clouds','daily-tarot'),
                'lenormand_07' => __('Snake','daily-tarot'),
                'lenormand_08' => __('Coffin','daily-tarot'),
                'lenormand_09' => __('Bouquet','daily-tarot'),
                'lenormand_10' => __('Scythe','daily-tarot'),
                'lenormand_11' => __('Whip','daily-tarot'),
                'lenormand_12' => __('Birds','daily-tarot'),
                'lenormand_13' => __('Child','daily-tarot'),
                'lenormand_14' => __('Fox','daily-tarot'),
                'lenormand_15' => __('Bear','daily-tarot'),
                'lenormand_16' => __('Stars','daily-tarot'),
                'lenormand_17' => __('Stork','daily-tarot'),
                'lenormand_18' => __('Dog','daily-tarot'),
                'lenormand_19' => __('Tower','daily-tarot'),
                'lenormand_20' => __('Garden','daily-tarot'),
                'lenormand_21' => __('Mountain','daily-tarot'),
                'lenormand_22' => __('Crossroads','daily-tarot'),
                'lenormand_23' => __('Mice','daily-tarot'),
                'lenormand_24' => __('Heart','daily-tarot'),
                'lenormand_25' => __('Ring','daily-tarot'),
                'lenormand_26' => __('Book','daily-tarot'),
                'lenormand_27' => __('Letter','daily-tarot'),
                'lenormand_28' => __('Man','daily-tarot'),
                'lenormand_29' => __('Woman','daily-tarot'),
                'lenormand_30' => __('Lily','daily-tarot'),
                'lenormand_31' => __('Sun','daily-tarot'),
                'lenormand_32' => __('Moon','daily-tarot'),
                'lenormand_33' => __('Key','daily-tarot'),
                'lenormand_34' => __('Fish','daily-tarot'),
                'lenormand_35' => __('Anchor','daily-tarot'),
                'lenormand_36' => __('Cross','daily-tarot'),
            ];

            /** @var array<string,string> $cards */
            $cards = apply_filters('dtarot_cards_lenormand', $cards);
            return $cards;
        }

        if ($system === self::SYSTEM_KIPPER) {
            $cards = [
                'kipper_01' => __('Main Male','daily-tarot'),
                'kipper_02' => __('Main Female','daily-tarot'),
                'kipper_03' => __('Marriage','daily-tarot'),
                'kipper_04' => __('Meeting','daily-tarot'),
                'kipper_05' => __('Good Gentleman','daily-tarot'),
                'kipper_06' => __('Good Lady','daily-tarot'),
                'kipper_07' => __('Letter','daily-tarot'),
                'kipper_08' => __('False Person','daily-tarot'),
                'kipper_09' => __('Change','daily-tarot'),
                'kipper_10' => __('Journey','daily-tarot'),
                'kipper_11' => __('Win Lots','daily-tarot'),
                'kipper_12' => __('Rich Girl','daily-tarot'),
                'kipper_13' => __('Rich Gentleman','daily-tarot'),
                'kipper_14' => __('Sad News','daily-tarot'),
                'kipper_15' => __('Good Outcome in Love','daily-tarot'),
                'kipper_16' => __('His Thoughts','daily-tarot'),
                'kipper_17' => __('Receiving a Gift','daily-tarot'),
                'kipper_18' => __('Small Child','daily-tarot'),
                'kipper_19' => __('Death','daily-tarot'),
                'kipper_20' => __('House','daily-tarot'),
                'kipper_21' => __('Living Room','daily-tarot'),
                'kipper_22' => __('Military Person','daily-tarot'),
                'kipper_23' => __('Court','daily-tarot'),
                'kipper_24' => __('Theft','daily-tarot'),
                'kipper_25' => __('High Honors','daily-tarot'),
                'kipper_26' => __('Great Fortune','daily-tarot'),
                'kipper_27' => __('Unexpected Money','daily-tarot'),
                'kipper_28' => __('Expectation','daily-tarot'),
                'kipper_29' => __('Prison','daily-tarot'),
                'kipper_30' => __('Official Person','daily-tarot'),
                'kipper_31' => __('Short Illness','daily-tarot'),
                'kipper_32' => __('Grief','daily-tarot'),
                'kipper_33' => __('Gloomy Thoughts','daily-tarot'),
                'kipper_34' => __('Work','daily-tarot'),
                'kipper_35' => __('Long Road','daily-tarot'),
                'kipper_36' => __('Hope','daily-tarot'),
            ];

            /** @var array<string,string> $cards */
            $cards = apply_filters('dtarot_cards_kipper', $cards);
            return $cards;
        }

        if ($system === self::SYSTEM_GYPSY) {
            // Tarot Gypsy deck (legacy names) with stable IDs gypsy_01..36.
            $cards = [
                'gypsy_01' => __('Main Male','daily-tarot'),
                'gypsy_02' => __('Main Female','daily-tarot'),
                'gypsy_03' => __('Marriage','daily-tarot'),
                'gypsy_04' => __('Merriment','daily-tarot'),
                'gypsy_05' => __('Widower','daily-tarot'),
                'gypsy_06' => __('Widow','daily-tarot'),
                'gypsy_07' => __('Letter','daily-tarot'),
                'gypsy_08' => __('Falseness','daily-tarot'),
                'gypsy_09' => __('Unexpected Joy','daily-tarot'),
                'gypsy_10' => __('Journey','daily-tarot'),
                'gypsy_11' => __('Money','daily-tarot'),
                'gypsy_12' => __('Visit','daily-tarot'),
                'gypsy_13' => __('Loss','daily-tarot'),
                'gypsy_14' => __('Sadness','daily-tarot'),
                'gypsy_15' => __('Love','daily-tarot'),
                'gypsy_16' => __('Thought','daily-tarot'),
                'gypsy_17' => __('Gift','daily-tarot'),
                'gypsy_18' => __('Baby','daily-tarot'),
                'gypsy_19' => __('Death','daily-tarot'),
                'gypsy_20' => __('House','daily-tarot'),
                'gypsy_21' => __('Ecclesiastic','daily-tarot'),
                'gypsy_22' => __('Officer','daily-tarot'),
                'gypsy_23' => __('Judge','daily-tarot'),
                'gypsy_24' => __('Thief','daily-tarot'),
                'gypsy_25' => __('Message','daily-tarot'),
                'gypsy_26' => __('Fortune','daily-tarot'),
                'gypsy_27' => __('Some Money','daily-tarot'),
                'gypsy_28' => __('Desire','daily-tarot'),
                'gypsy_29' => __('Misfortune','daily-tarot'),
                'gypsy_30' => __('Jealousy','daily-tarot'),
                'gypsy_31' => __('Malady','daily-tarot'),
                'gypsy_32' => __('Anger','daily-tarot'),
                'gypsy_33' => __('Enemy','daily-tarot'),
                'gypsy_34' => __('Constancy','daily-tarot'),
                'gypsy_35' => __('Fidelity','daily-tarot'),
                'gypsy_36' => __('Hope','daily-tarot'),
            ];

            /** @var array<string,string> $cards */
            $cards = apply_filters('dtarot_cards_gypsy', $cards);
            return $cards;
        }

        if ($system === self::SYSTEM_KIPPER_FIN_DE_SIECLE) {
            // Default list: standard 36 Kipper cards + 3 extra placeholders.
            // If you have the exact 39-card Fin de Siècle titles, override via the filter.
            $cards = [
                'kipper_fds_01' => __('Main Person Male','daily-tarot'),
                'kipper_fds_02' => __('Main Person Female','daily-tarot'),
                'kipper_fds_03' => __('Marriage','daily-tarot'),
                'kipper_fds_04' => __('Courtship','daily-tarot'),
                'kipper_fds_05' => __('Mature Man','daily-tarot'),
                'kipper_fds_06' => __('Mature Woman','daily-tarot'),
                'kipper_fds_07' => __('Message','daily-tarot'),
                'kipper_fds_08' => __('False Person','daily-tarot'),
                'kipper_fds_09' => __('A Change','daily-tarot'),
                'kipper_fds_10' => __('Journey','daily-tarot'),
                'kipper_fds_11' => __('Sudden Wealth','daily-tarot'),
                'kipper_fds_12' => __('Privileged Lady','daily-tarot'),
                'kipper_fds_13' => __('Wealthy Man','daily-tarot'),
                'kipper_fds_14' => __('Message of Concern','daily-tarot'),
                'kipper_fds_15' => __('Lovers','daily-tarot'),
                'kipper_fds_16' => __('Thoughts','daily-tarot'),
                'kipper_fds_17' => __('Gift','daily-tarot'),
                'kipper_fds_18' => __('Child','daily-tarot'),
                'kipper_fds_19' => __('Coffin','daily-tarot'),
                'kipper_fds_20' => __('House','daily-tarot'),
                'kipper_fds_21' => __('Family Room','daily-tarot'),
                'kipper_fds_22' => __('Official Person','daily-tarot'),
                'kipper_fds_23' => __('Court','daily-tarot'),
                'kipper_fds_24' => __('Theft','daily-tarot'),
                'kipper_fds_25' => __('High Honors','daily-tarot'),
                'kipper_fds_26' => __('Great Fortune','daily-tarot'),
                'kipper_fds_27' => __('Unexpected Income','daily-tarot'),
                'kipper_fds_28' => __('Expectation','daily-tarot'),
                'kipper_fds_29' => __('Imprisonment','daily-tarot'),
                'kipper_fds_30' => __('Adjudication','daily-tarot'),
                'kipper_fds_31' => __('Bad Health','daily-tarot'),
                'kipper_fds_32' => __('Despair','daily-tarot'),
                'kipper_fds_33' => __('Concern','daily-tarot'),
                'kipper_fds_34' => __('Occupation','daily-tarot'),
                'kipper_fds_35' => __('Pathway','daily-tarot'),
                'kipper_fds_36' => __('Distant Horizons','daily-tarot'),
                'kipper_fds_37' => __('Poverty','daily-tarot'),
                'kipper_fds_38' => __('Toil & Labour','daily-tarot'),
                'kipper_fds_39' => __('Community','daily-tarot'),
            ];

            /** @var array<string,string> $cards */
            $cards = apply_filters('dtarot_cards_kipper_fin_de_siecle', $cards);
            return $cards;
        }

        // Tarot (78): Major + Minor Arcana
        $cards = [
            'tarot_major_0'  => __('The Fool','daily-tarot'),
            'tarot_major_1'  => __('The Magician','daily-tarot'),
            'tarot_major_2'  => __('The High Priestess','daily-tarot'),
            'tarot_major_3'  => __('The Empress','daily-tarot'),
            'tarot_major_4'  => __('The Emperor','daily-tarot'),
            'tarot_major_5'  => __('The Hierophant','daily-tarot'),
            'tarot_major_6'  => __('The Lovers','daily-tarot'),
            'tarot_major_7'  => __('The Chariot','daily-tarot'),
            'tarot_major_8'  => __('Strength','daily-tarot'),
            'tarot_major_9'  => __('The Hermit','daily-tarot'),
            'tarot_major_10' => __('Wheel of Fortune','daily-tarot'),
            'tarot_major_11' => __('Justice','daily-tarot'),
            'tarot_major_12' => __('The Hanged Man','daily-tarot'),
            'tarot_major_13' => __('Death','daily-tarot'),
            'tarot_major_14' => __('Temperance','daily-tarot'),
            'tarot_major_15' => __('The Devil','daily-tarot'),
            'tarot_major_16' => __('The Tower','daily-tarot'),
            'tarot_major_17' => __('The Star','daily-tarot'),
            'tarot_major_18' => __('The Moon','daily-tarot'),
            'tarot_major_19' => __('The Sun','daily-tarot'),
            'tarot_major_20' => __('Judgement','daily-tarot'),
            'tarot_major_21' => __('The World','daily-tarot'),
        ];

        // Minor Arcana: Wands
        $cards = array_merge($cards, [
            'tarot_wands_01' => __('Ace of Wands','daily-tarot'),
            'tarot_wands_02' => __('Two of Wands','daily-tarot'),
            'tarot_wands_03' => __('Three of Wands','daily-tarot'),
            'tarot_wands_04' => __('Four of Wands','daily-tarot'),
            'tarot_wands_05' => __('Five of Wands','daily-tarot'),
            'tarot_wands_06' => __('Six of Wands','daily-tarot'),
            'tarot_wands_07' => __('Seven of Wands','daily-tarot'),
            'tarot_wands_08' => __('Eight of Wands','daily-tarot'),
            'tarot_wands_09' => __('Nine of Wands','daily-tarot'),
            'tarot_wands_10' => __('Ten of Wands','daily-tarot'),
            'tarot_wands_11' => __('Page of Wands','daily-tarot'),
            'tarot_wands_12' => __('Knight of Wands','daily-tarot'),
            'tarot_wands_13' => __('Queen of Wands','daily-tarot'),
            'tarot_wands_14' => __('King of Wands','daily-tarot'),
        ]);

        // Minor Arcana: Cups
        $cards = array_merge($cards, [
            'tarot_cups_01' => __('Ace of Cups','daily-tarot'),
            'tarot_cups_02' => __('Two of Cups','daily-tarot'),
            'tarot_cups_03' => __('Three of Cups','daily-tarot'),
            'tarot_cups_04' => __('Four of Cups','daily-tarot'),
            'tarot_cups_05' => __('Five of Cups','daily-tarot'),
            'tarot_cups_06' => __('Six of Cups','daily-tarot'),
            'tarot_cups_07' => __('Seven of Cups','daily-tarot'),
            'tarot_cups_08' => __('Eight of Cups','daily-tarot'),
            'tarot_cups_09' => __('Nine of Cups','daily-tarot'),
            'tarot_cups_10' => __('Ten of Cups','daily-tarot'),
            'tarot_cups_11' => __('Page of Cups','daily-tarot'),
            'tarot_cups_12' => __('Knight of Cups','daily-tarot'),
            'tarot_cups_13' => __('Queen of Cups','daily-tarot'),
            'tarot_cups_14' => __('King of Cups','daily-tarot'),
        ]);

        // Minor Arcana: Swords
        $cards = array_merge($cards, [
            'tarot_swords_01' => __('Ace of Swords','daily-tarot'),
            'tarot_swords_02' => __('Two of Swords','daily-tarot'),
            'tarot_swords_03' => __('Three of Swords','daily-tarot'),
            'tarot_swords_04' => __('Four of Swords','daily-tarot'),
            'tarot_swords_05' => __('Five of Swords','daily-tarot'),
            'tarot_swords_06' => __('Six of Swords','daily-tarot'),
            'tarot_swords_07' => __('Seven of Swords','daily-tarot'),
            'tarot_swords_08' => __('Eight of Swords','daily-tarot'),
            'tarot_swords_09' => __('Nine of Swords','daily-tarot'),
            'tarot_swords_10' => __('Ten of Swords','daily-tarot'),
            'tarot_swords_11' => __('Page of Swords','daily-tarot'),
            'tarot_swords_12' => __('Knight of Swords','daily-tarot'),
            'tarot_swords_13' => __('Queen of Swords','daily-tarot'),
            'tarot_swords_14' => __('King of Swords','daily-tarot'),
        ]);

        // Minor Arcana: Pentacles
        $cards = array_merge($cards, [
            'tarot_pentacles_01' => __('Ace of Pentacles','daily-tarot'),
            'tarot_pentacles_02' => __('Two of Pentacles','daily-tarot'),
            'tarot_pentacles_03' => __('Three of Pentacles','daily-tarot'),
            'tarot_pentacles_04' => __('Four of Pentacles','daily-tarot'),
            'tarot_pentacles_05' => __('Five of Pentacles','daily-tarot'),
            'tarot_pentacles_06' => __('Six of Pentacles','daily-tarot'),
            'tarot_pentacles_07' => __('Seven of Pentacles','daily-tarot'),
            'tarot_pentacles_08' => __('Eight of Pentacles','daily-tarot'),
            'tarot_pentacles_09' => __('Nine of Pentacles','daily-tarot'),
            'tarot_pentacles_10' => __('Ten of Pentacles','daily-tarot'),
            'tarot_pentacles_11' => __('Page of Pentacles','daily-tarot'),
            'tarot_pentacles_12' => __('Knight of Pentacles','daily-tarot'),
            'tarot_pentacles_13' => __('Queen of Pentacles','daily-tarot'),
            'tarot_pentacles_14' => __('King of Pentacles','daily-tarot'),
        ]);

        /** @var array<string,string> $cards */
        $cards = apply_filters('dtarot_cards_tarot', $cards);
        return $cards;
    }

    /** @return array<string,string> */
    public static function allMerged(): array {
        $out = [];
        foreach (array_keys(self::systems()) as $system) {
            foreach (self::forSystem($system) as $id => $name) {
                $out[$id] = $name;
            }
        }
        return $out;
    }

    public static function name(string $id): string {
        $id = sanitize_text_field($id);
        if ($id === '') return '';

        $all = self::allMerged();
        if (isset($all[$id])) {
            return (string)$all[$id];
        }

        // Legacy alias: some stored readings used gypsy_XX IDs.
        if (preg_match('/^gypsy_(\d{2})$/', $id, $m)) {
            $alias = 'kipper_' . $m[1];
            return isset($all[$alias]) ? (string)$all[$alias] : $id;
        }

        return $id;
    }

    /**
     * Returns alternate IDs for cross-compat between Gypsy and Kipper numeric cards.
     *
     * Used for lookups like deck image maps and legacy stored readings.
     *
     * @return array<int,string>
     */
    public static function kipperGypsyAliases(string $id): array {
        $id = sanitize_text_field($id);
        if ($id === '') return [];

        $out = [$id];

        if (preg_match('/^gypsy_(\d{2})$/', $id, $m)) {
            $out[] = 'kipper_' . $m[1];
        } elseif (preg_match('/^kipper_(\d{2})$/', $id, $m)) {
            $out[] = 'gypsy_' . $m[1];
        }

        return array_values(array_unique($out));
    }

    /**
     * Returns structured metadata for a card.
     *
     * Intended for storing system-specific fields beyond the plain display name
     * (e.g., Lenormand playing-card insets and additional meaning facets).
     *
     * @return array<string,mixed>
     */
    public static function meta(string $id): array {
        $id = sanitize_text_field($id);
        if ($id === '') return [];

        // Legacy alias: early builds used gypsy_XX IDs for this Kipper deck.
        if (preg_match('/^gypsy_(\d{2})$/', $id, $m)) {
            $id = 'kipper_' . $m[1];
        }

        if (str_starts_with($id, 'lenormand_')) {
            $meta = self::lenormandMeta();
            $out = $meta[$id] ?? [];

            // Allow portable meta overrides (e.g., imported from backups/packs).
            if (function_exists('get_option')) {
                $overrides = get_option('dtarot_card_meta_lenormand', []);
                if (is_array($overrides) && isset($overrides[$id]) && is_array($overrides[$id])) {
                    // Shallow merge is sufficient for our current schema.
                    $out = array_merge($out, $overrides[$id]);
                }
            }

            /**
             * Filter: customize Lenormand metadata.
             *
             * @param array<string,mixed> $out
             * @param string $id
             */
            return apply_filters('dtarot_card_meta_lenormand', $out, $id);
        }

        return [];
    }

    /**
     * Playing-card suit associations (Lenormand overlay suits).
     *
     * @return array<string,string>
     */
    public static function playingSuitAssociations(): array {
        return [
            'hearts' => __('Love, relationships, family, home life, feelings','daily-tarot'),
            'spades' => __('Society, government, travel, business, structure','daily-tarot'),
            'diamonds' => __('Enterprise, fortune, misfortune, wisdom, dynamism','daily-tarot'),
            'clubs' => __('Survival, hardship, trouble, responsibilities, complications','daily-tarot'),
        ];
    }

    /**
     * Playing-card rank associations (Lenormand overlay ranks).
     *
     * @return array<string,string>
     */
    public static function playingRankAssociations(): array {
        return [
            'A' => __('Beginnings, origin, answer','daily-tarot'),
            '6' => __('Commitment, attainment, out of control','daily-tarot'),
            '7' => __('Assessment, communication, response','daily-tarot'),
            '8' => __('Community, status, resolution','daily-tarot'),
            '9' => __('Movement, transactions, behavior','daily-tarot'),
            '10' => __('Connections, guidance, advice','daily-tarot'),
            'J' => __('Energy, excitement, interaction','daily-tarot'),
            'Q' => __('Resources, nurturing, methods','daily-tarot'),
            'K' => __('Maturity, control, dominion','daily-tarot'),
        ];
    }

    /**
     * Lenormand system: subject/modifier/extended + playing-card inset.
     *
     * Note: This map currently includes the user-provided content for cards 10–36.
     * Cards 01–09 intentionally omitted until provided.
     *
     * @return array<string,array<string,mixed>>
     */
    private static function lenormandMeta(): array {
        return [
            'lenormand_01' => [
                'playing' => ['rank' => '9', 'suit' => 'hearts'],
                'subject' => 'News, message, delivery, visitor',
                'modifier' => 'Speedy, movement, passionate, athletic, action, updates, declaration',
                'extended' => 'Approach, communication, action, renewal, imminent change, return, declaration, motorcycle, someone new',
            ],
            'lenormand_02' => [
                'playing' => ['rank' => '6', 'suit' => 'diamonds'],
                'subject' => 'Opportunity, luck, reward, chance',
                'modifier' => 'Hopeful, excited, optimistic, gambling, pleasant surprise',
                'extended' => 'Stroke of luck, lucky break, risk and reward, second chance, fling, fleeting, bonus, bargain, boost, wild card',
            ],
            'lenormand_03' => [
                'playing' => ['rank' => '10', 'suit' => 'spades'],
                'subject' => 'Travel, farewell, distance, journey',
                'modifier' => 'Change, movement, adventurous, seeking, risk taking, transition',
                'extended' => 'Transfer, foreign location, long distance trip, physical distance, separation, exploration, overseas, international',
            ],
            'lenormand_04' => [
                'playing' => ['rank' => 'K', 'suit' => 'hearts'],
                'subject' => 'Home, family, safety, tradition, property',
                'modifier' => 'Stable, comfortable, domestic affairs',
                'extended' => 'Small building, real estate, sanctuary, address, comfort zone, small business, property',
            ],
            'lenormand_05' => [
                'playing' => ['rank' => '7', 'suit' => 'hearts'],
                'subject' => 'Health, growth, medicine, life, vitality',
                'modifier' => 'Healthy, grounded, spiritual, well being, healing, growth, process, accumulation',
                'extended' => 'Development, physical health, environment, patience, hospital, spiritual place, ancestry',
            ],
            'lenormand_06' => [
                'playing' => ['rank' => 'K', 'suit' => 'clubs'],
                'subject' => 'Misunderstanding, secrets, chaos, illusion',
                'modifier' => 'Confused, doubtful, insecure, hidden, contradiction, uncertain',
                'extended' => 'Nebulous, ambiguity, inner conflict, chaotic situation, instability, pollution, mental cloudiness, discomfort',
            ],
            'lenormand_07' => [
                'playing' => ['rank' => 'Q', 'suit' => 'clubs'],
                'subject' => 'Problem, desire, deception, betrayal, issue',
                'modifier' => 'Jealous, sexual, seductive, dysfunctional, sabotaged, detour, trouble',
                'extended' => 'Complication, rival, traitor, disorder, deceit',
            ],
            'lenormand_08' => [
                'playing' => ['rank' => '9', 'suit' => 'diamonds'],
                'subject' => 'Grief, ending, death',
                'modifier' => 'Detached, mournful, depressed, transformative, negative, wasted',
                'extended' => 'Cancellation, stagnation, trash, lacking, vanished, detached, empty container, graveyard',
            ],
            'lenormand_09' => [
                'playing' => ['rank' => 'Q', 'suit' => 'spades'],
                'subject' => 'Social life, gift, grace, beauty, charm, blessing',
                'modifier' => 'Aesthetic, refined, beautiful, charming, inviting, celebratory, cosmetic, joyful, positive',
                'extended' => 'Grace, wish granted, treat, pampered, appreciation, fun',
            ],
            'lenormand_10' => [
                'playing' => ['rank' => 'J', 'suit' => 'diamonds'],
                'subject' => 'Separation, decision, danger, removal, reduction',
                'modifier' => 'Sudden, shocking, dangerous, definitive, surgical, interruptive',
                'extended' => 'Weapon, clean break, termination, rejection, accident, surgery, farmlands, injury, execution',
            ],
            'lenormand_11' => [
                'playing' => ['rank' => 'J', 'suit' => 'clubs'],
                'subject' => 'Conflict, discipline, hostility, threat, struggles, hardship',
                'modifier' => 'Scolding, argumentative, angry, violent, abusive, punishing',
                'extended' => 'Physical activity, anger, discord, strife, dispute, sex, repetitive labor, competition',
            ],
            'lenormand_12' => [
                'playing' => ['rank' => '7', 'suit' => 'diamonds'],
                'subject' => 'Communication, companionship, negotiation, meeting, conversation',
                'modifier' => 'Restless, anxious, gossipy, chatty, excited',
                'extended' => 'Texting, calling, interview, debate, speech, date, visit, counseling',
            ],
            'lenormand_13' => [
                'playing' => ['rank' => 'J', 'suit' => 'spades'],
                'subject' => 'Children, new beginning, baby, childhood, innocence, youth',
                'modifier' => 'Simple, naive, trusting, youthful, playful',
                'extended' => 'Lacking experience, enthusiastic, playground, school, learning, spontaneous, wonder, infancy',
            ],
            'lenormand_14' => [
                'playing' => ['rank' => '9', 'suit' => 'clubs'],
                'subject' => 'Job, lies, work, caution, intelligence, survival',
                'modifier' => 'Deceitful, discerning, criminal, disguised, clever, cunning, specialized',
                'extended' => 'Skillful, employee, caution, awareness, analytical, specialized, detecting, con artist, investigation, fraud, trickster',
            ],
            'lenormand_15' => [
                'playing' => ['rank' => '10', 'suit' => 'clubs'],
                'subject' => 'Leader, boss, strength, wealth, finances, power, security',
                'modifier' => 'Resourceful, protective, managerial, dominant, influential, capitalistic, abundant',
                'extended' => 'Reserves, savings, money, goods, possessions, stock, provider',
            ],
            'lenormand_16' => [
                'playing' => ['rank' => '6', 'suit' => 'hearts'],
                'subject' => 'Dreams, progress, guidance, direction, navigation, clarity, ambition',
                'modifier' => 'Hopeful, inspiring, optimistic, safe, transparent, idealistic, charted territory',
                'extended' => 'Plans, design, signs, safe passage, clear skies, strategy, goals, innovation, science, magic, astrology',
            ],
            'lenormand_17' => [
                'playing' => ['rank' => 'Q', 'suit' => 'hearts'],
                'subject' => 'Transition, relocation, improvement, advancement, recovery, fresh start, new beginning',
                'modifier' => 'Graceful, new, dynamic, upgraded, promoted, gain, flexibility, adaptability',
                'extended' => 'Positive change, pregnancy, birth, addition to household, progressive, upgrade',
            ],
            'lenormand_18' => [
                'playing' => ['rank' => '10', 'suit' => 'hearts'],
                'subject' => 'Friend, pet, partner, companion, ally',
                'modifier' => 'Devoted, loyal, supportive, consultative, faithful, acquainted',
                'extended' => 'Known person, advisor, intimacy, support',
            ],
            'lenormand_19' => [
                'playing' => ['rank' => '6', 'suit' => 'spades'],
                'subject' => 'Government, ego, bureaucracy, rules, organizations, status',
                'modifier' => 'Established, lonely, arrogant, authoritative, orderly, legal, corporate, captive, official',
                'extended' => 'Legal matters, company, institution, boundaries, hierarchy, standards, large building, high rise, quarantine',
            ],
            'lenormand_20' => [
                'playing' => ['rank' => '8', 'suit' => 'spades'],
                'subject' => 'Community, event, party, gathering, social life, public area, groups',
                'modifier' => 'Popular, performative, cultured, social, public',
                'extended' => 'Society, audience, seminar, team, class, nature, social media, club, crowd',
            ],
            'lenormand_21' => [
                'playing' => ['rank' => '8', 'suit' => 'clubs'],
                'subject' => 'Obstacle, delay, inactivity, barrier',
                'modifier' => 'Stuck, stubborn, challenging, blocked, isolated, immobile, solid, discouraged, aloof',
                'extended' => 'Enemy, impasse, apathy, silence, denial, discouraged, indifferent, emotionally distant',
            ],
            'lenormand_22' => [
                'playing' => ['rank' => 'Q', 'suit' => 'diamonds'],
                'subject' => 'Decision, trip, options, junction, pathways',
                'modifier' => 'Hesitant, indecisive, independent, searching, diverse, multiple',
                'extended' => 'Alternatives, choices, turning point, escape route, changed direction, mediation, passage',
            ],
            'lenormand_23' => [
                'playing' => ['rank' => '7', 'suit' => 'clubs'],
                'subject' => 'Loss, disease, decay, deterioration, worries, niggles',
                'modifier' => 'Costly, stressed, damaged, eroded, nervous, parasitic, toxic, irritating',
                'extended' => 'Damage, theft, destruction, fearful, nuisance',
            ],
            'lenormand_24' => [
                'playing' => ['rank' => 'J', 'suit' => 'hearts'],
                'subject' => 'Love, romance, desire, fondness',
                'modifier' => 'Forgiving, gentle, caring, compatible, generous, happy, passionate, compassionate',
                'extended' => 'Affection, preference, favorite, bliss, kind, sharing, love life',
            ],
            'lenormand_25' => [
                'playing' => ['rank' => 'A', 'suit' => 'clubs'],
                'subject' => 'Marriage, contracts, partnership, offer, agreement',
                'modifier' => 'Committed, stable, promising, trusted, guaranteed, merging',
                'extended' => 'Valuables, obligation, alliance, union, merger, co-operation, gift, jewelry, guarantee',
            ],
            'lenormand_26' => [
                'playing' => ['rank' => '10', 'suit' => 'diamonds'],
                'subject' => 'Education, research, knowledge, report, learning secrets, information',
                'modifier' => 'Informed, secret, knowledgeable, evaluated, teaching, memorized, filed, accounted for',
                'extended' => 'Data, journal, newspaper, accounting, project, assignment, audit, studies, mystery, memory',
            ],
            'lenormand_27' => [
                'playing' => ['rank' => '7', 'suit' => 'spades'],
                'subject' => 'Document, conversation, results, evidence, message, written communication',
                'modifier' => 'Communicative, expressive, written, shared news, corresponded',
                'extended' => 'Mail, correspondence, contract, award, certificate, invoice, paper trail, article',
            ],
            'lenormand_28' => [
                'playing' => ['rank' => 'A', 'suit' => 'hearts'],
                'is_signifier' => true,
                'subject' => 'The Querent, a man or men in the querent’s life',
                'modifier' => 'Masculine',
                'extended' => 'N/A',
            ],
            'lenormand_29' => [
                'playing' => ['rank' => 'A', 'suit' => 'spades'],
                'is_signifier' => true,
                'subject' => 'The Querent, a woman or women in the querent’s life',
                'modifier' => 'Feminine',
                'extended' => 'N/A',
            ],
            'lenormand_30' => [
                'playing' => ['rank' => 'K', 'suit' => 'spades'],
                'subject' => 'Retirement, peace, serenity, release, age, experience, wisdom',
                'modifier' => 'Peaceful, calm, wise, older, sensual, vintage, contemplated, slow paced, experienced',
                'extended' => 'Nursing home, parents, elders, tradition',
            ],
            'lenormand_31' => [
                'playing' => ['rank' => 'A', 'suit' => 'diamonds'],
                'subject' => 'Success, recognition, energy, victory, positivity, happiness',
                'modifier' => 'Happy, fortunate, warm, illuminated, glorious, vital, enthusiastic, motivated, courageous, confident',
                'extended' => 'Achievement, excellence, charismatic, heat, desert, conscious, will, exposure, light, visibility',
            ],
            'lenormand_32' => [
                'playing' => ['rank' => '8', 'suit' => 'hearts'],
                'subject' => 'Subconscious, imagination, intimacy, fantasy, dreams, desire, emotions, feelings',
                'modifier' => 'Artistic, emotional, attractive, psychic, euphoric, romantic, seductive, sensitive, creative',
                'extended' => 'Recognition, imagination, subconscious, intuitive',
            ],
            'lenormand_33' => [
                'playing' => ['rank' => '8', 'suit' => 'diamonds'],
                'subject' => 'Resolution, mastery, breakthrough, access, fate',
                'modifier' => 'Open, liberated, destined, certain, significant, inspired',
                'extended' => 'Solution, password, insight, clue, miracle, security, discovery, revelation',
            ],
            'lenormand_34' => [
                'playing' => ['rank' => 'K', 'suit' => 'diamonds'],
                'subject' => 'Wealth, commerce, water, transaction, business',
                'modifier' => 'Abundant, luxurious, independent, free, moving, flexible, expansive',
                'extended' => 'Self-employment, investment, industry, trade, exchange, liquid, pool, coastal, freedom, consulting, water',
            ],
            'lenormand_35' => [
                'playing' => ['rank' => '9', 'suit' => 'spades'],
                'subject' => 'Foundations, achievement, standpoint, stability, long-lasting',
                'modifier' => 'Faithful, resilient, secure, stable, determined, settled, persevering',
                'extended' => 'Base, legacy, immigration, fixed point',
            ],
            'lenormand_36' => [
                'playing' => ['rank' => '6', 'suit' => 'clubs'],
                'subject' => 'Religion, principles, sacrifice, guilt, burden, responsibility',
                'modifier' => 'Dutiful, suffering, burdened, remorseful, traumatized, selfless, devoted, despairing, distressed',
                'extended' => 'Pain, sadness, regret, difficulty, ordeal, faith, mercy, holy place, altar, trauma, victim, grief',
            ],
        ];
    }
}
