<?php
declare(strict_types=1);


namespace DailyTarot\Reading;
if (!defined('ABSPATH')) { exit; }


final class ReadingComposer {

    /**
     * Applies meaning-pack fallback.
     *
     * Rules (matches current frontend behavior):
     * - If $content is empty: prefer meaning.short, else meaning.upright
     * - If $dailyText is empty: prefer meaning.long
     *
     * @param array{upright?:string,short?:string,long?:string} $meaning
     * @return array{content:string,daily_text:string}
     */
    public static function applyMeaningFallback(string $content, string $dailyText, array $meaning): array {
        $content = (string)$content;
        $dailyText = (string)$dailyText;

        $short = isset($meaning['short']) && is_string($meaning['short']) ? $meaning['short'] : '';
        $upright = isset($meaning['upright']) && is_string($meaning['upright']) ? $meaning['upright'] : '';
        $long = isset($meaning['long']) && is_string($meaning['long']) ? $meaning['long'] : '';

        if ($content === '') {
            if ($short !== '') {
                $content = $short;
            } elseif ($upright !== '') {
                $content = $upright;
            }
        }

        if ($dailyText === '') {
            if ($long !== '') {
                $dailyText = $long;
            }
        }

        return ['content' => $content, 'daily_text' => $dailyText];
    }

    /**
     * Builds a best-effort meta description source.
     *
     * @param array{content?:string,daily_text?:string} $entry
     * @param array{upright?:string,short?:string,long?:string} $meaning
     */
    public static function metaSource(array $entry, array $meaning): string {
        $raw = isset($entry['daily_text']) && is_string($entry['daily_text']) ? $entry['daily_text'] : '';
        if ($raw === '') {
            $raw = isset($entry['content']) && is_string($entry['content']) ? $entry['content'] : '';
        }
        if ($raw === '') {
            $long = isset($meaning['long']) && is_string($meaning['long']) ? $meaning['long'] : '';
            $short = isset($meaning['short']) && is_string($meaning['short']) ? $meaning['short'] : '';
            $upright = isset($meaning['upright']) && is_string($meaning['upright']) ? $meaning['upright'] : '';
            $raw = $long !== '' ? $long : ($short !== '' ? $short : $upright);
        }
        return (string)$raw;
    }

    /**
     * Builds a best-effort meta description (OG/Twitter/HTML description).
     *
     * For social sharing, we prefer the shortest meaningful summary first.
     * Order:
     * - entry.content (intro)
     * - meaning.short
     * - meaning.upright
     * - entry.daily_text (expanded)
     * - meaning.long
     */
    public static function buildMetaDescription(array $entry, array $meaning, int $maxLen = 160): string {
        $parts = [
            (isset($entry['content']) && is_string($entry['content'])) ? $entry['content'] : '',
            (isset($meaning['short']) && is_string($meaning['short'])) ? $meaning['short'] : '',
            (isset($meaning['upright']) && is_string($meaning['upright'])) ? $meaning['upright'] : '',
            (isset($entry['daily_text']) && is_string($entry['daily_text'])) ? $entry['daily_text'] : '',
            (isset($meaning['long']) && is_string($meaning['long'])) ? $meaning['long'] : '',
        ];

        $raw = '';
        foreach ($parts as $p) {
            $p = is_string($p) ? trim($p) : '';
            if ($p === '') continue;
            $raw = $p;
            break;
        }
        if ($raw === '') return '';

        $desc = trim(wp_strip_all_tags($raw));
        if ($desc === '') return '';

        // Normalize whitespace for more stable share previews.
        $desc = preg_replace('/\s+/u', ' ', $desc);
        $desc = is_string($desc) ? trim($desc) : '';
        if ($desc === '') return '';

        $maxLen = (int)$maxLen;
        if ($maxLen <= 0) $maxLen = 160;
        if (mb_strlen($desc) > $maxLen) {
            $desc = mb_substr($desc, 0, max(0, $maxLen - 1)) . '…';
        }

        return $desc;
    }
}
