<?php
declare(strict_types=1);

namespace DailyTarot;

if (!defined('ABSPATH')) { exit; }

use DailyTarot\Admin\Menu;
use DailyTarot\Admin\Ajax;
use DailyTarot\Admin\Backup;
use DailyTarot\Admin\StarterDecksPrompt;
use DailyTarot\Admin\Pages\Analytics;
use DailyTarot\CPT\Deck;
use DailyTarot\CPT\MeaningPack;
use DailyTarot\CPT\ReadingType;
use DailyTarot\CPT\Booking;
use DailyTarot\Frontend\Shortcodes;
use DailyTarot\Frontend\Booking as BookingFrontend;
use DailyTarot\Frontend\ReadableRoutes;
use DailyTarot\Frontend\ShareTracking;
use DailyTarot\Seo\Yoast;
use DailyTarot\Seo\RankMath;
use DailyTarot\Seo\Sitemap;
use DailyTarot\Seo\Schema;
use DailyTarot\Blocks\Blocks;
use DailyTarot\Calendar\DayEntryStorageAdmin;
use DailyTarot\Calendar\DayEntryTable;
use DailyTarot\Analytics\ShareTable;
use DailyTarot\Meaning\DefaultMeaningPack;
use DailyTarot\Meaning\KipperBasicsMeaningPack;
use DailyTarot\Meaning\KipperStandardMeaningPack;
use DailyTarot\Automation\Scheduler;
use DailyTarot\Automation\DailyJob;
use DailyTarot\Automation\AutomationSettings;
use DailyTarot\Automation\AiPrefillScheduler;
use DailyTarot\Automation\AiPrefillJob;
use DailyTarot\Automation\AiPrefillSettings;
use DailyTarot\Admin\CalendarPublishTimes;
use DailyTarot\Rest\ReadingsController;
use DailyTarot\Rest\AiProviderController;
use DailyTarot\Admin\UiSettings;
use DailyTarot\Support\AiProviderSettings;
use DailyTarot\Support\EmailCtaSettings;
use DailyTarot\Support\EmailCtaStore;
use DailyTarot\Support\ShortcodeSettings;
use DailyTarot\Support\ShareImageSettings;
use DailyTarot\Support\BookingSettings;
use DailyTarot\Support\DefaultDecks;
use DailyTarot\Support\DefaultMeaningPacks;
use DailyTarot\Support\CachePurge;
use DailyTarot\Support\DefaultReadingTypes;
use DailyTarot\Support\SpreadMeaningPacks;
use DailyTarot\Support\SpreadScanner;
use DailyTarot\Support\Upgrade;
use DailyTarot\Admin\SpreadAdmin;
use DailyTarot\Support\License;
use DailyTarot\Admin\Pages\Feedback;
use DailyTarot\Admin\Pages\Settings;
use DailyTarot\Support\CacheVersion;
use DailyTarot\Support\PostTypes;
use DailyTarot\Support\RelatedLinks;
use DailyTarot\Support\ReviewPrompt;

final class Plugin {

    private const OPT_REWRITE_FLUSHED_VER = 'dtarot_rewrite_flushed_ver';
    private const OPT_REWRITE_NEEDS_FLUSH = 'dtarot_rewrite_needs_flush_ver';

    /**
     * Bump this only when rewrite rules change.
     *
     * (Not every plugin version requires a permalinks flush.)
     */
    private const REWRITE_RULES_VERSION = '2';

    public static function registerAllRewrites(): void {
        if (class_exists(ReadableRoutes::class)) {
            ReadableRoutes::register_rewrites();
        }
        if (class_exists(Sitemap::class)) {
            Sitemap::registerRewrites();
        }
    }

    public static function maybeFlushRewritesOnUpgrade(): void {
        // Only do this in admin, and only for admins.
        if (!is_admin() || !current_user_can('manage_options')) return;

        $last = (string)get_option(self::OPT_REWRITE_FLUSHED_VER, '');
        $target = (string)self::REWRITE_RULES_VERSION;
        if ($last === $target) {
            delete_option(self::OPT_REWRITE_NEEDS_FLUSH);
            return;
        }

        // Flag for a one-click admin notice (avoid auto-flushing on every update).
        update_option(self::OPT_REWRITE_NEEDS_FLUSH, $target, false);
    }

    public static function maybeShowRewriteNotice(): void {
        if (!is_admin() || !current_user_can('manage_options')) return;

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only query arg used to show an admin notice.
        $ok = isset($_GET['dtarot_rewrites']) ? sanitize_key((string)wp_unslash($_GET['dtarot_rewrites'])) : '';
        if ($ok === 'ok') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Daily Tarot permalinks updated.','daily-tarot') . '</p></div>';
            return;
        }

        $needs = (string)get_option(self::OPT_REWRITE_NEEDS_FLUSH, '');
        $target = (string)self::REWRITE_RULES_VERSION;
        if ($needs !== $target) return;

        $url = wp_nonce_url(admin_url('admin-post.php?action=dtarot_flush_rewrites'), 'dtarot_flush_rewrites');
        echo '<div class="notice notice-warning">';
        echo '<p><strong>' . esc_html__('Daily Tarot needs to refresh permalinks.','daily-tarot') . '</strong> ';
        echo esc_html__('This is required after some updates that change readable routes/sitemap URLs.','daily-tarot') . '</p>';
        echo '<p><a class="button button-primary" href="' . esc_url($url) . '">' . esc_html__('Refresh permalinks','daily-tarot') . '</a></p>';
        echo '</div>';
    }

    public static function handleFlushRewrites(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_flush_rewrites');

        self::registerAllRewrites();

        flush_rewrite_rules(false);
        update_option(self::OPT_REWRITE_FLUSHED_VER, (string)self::REWRITE_RULES_VERSION, false);
        delete_option(self::OPT_REWRITE_NEEDS_FLUSH);

        $to = wp_get_referer();
        if (!$to) $to = admin_url('admin.php?page=daily-tarot');
        $to = add_query_arg(['dtarot_rewrites' => 'ok'], $to);
        wp_safe_redirect($to);
        exit;
    }

    public static function init(): void {
        add_action('init', [__CLASS__, 'lateInit']);

        // Migrate legacy CPT keys to plugin-specific dtarot_ keys before anything queries them.
        add_action('init', [PostTypes::class, 'migrateLegacyPostTypes'], -1);

        // Cache version bumping when plugin-relevant content changes.
        add_action('save_post_' . PostTypes::DECK, [__CLASS__, 'handleContentChanged'], 10, 3);
        add_action('save_post_' . PostTypes::MEANING_PACK, [__CLASS__, 'handleContentChanged'], 10, 3);
        add_action('save_post_' . PostTypes::READING_TYPE, [__CLASS__, 'handleContentChanged'], 10, 3);

        // Back-compat (pre-migration hooks).
        add_action('save_post_' . PostTypes::LEGACY_DECK, [__CLASS__, 'handleContentChanged'], 10, 3);
        add_action('save_post_' . PostTypes::LEGACY_MEANING_PACK, [__CLASS__, 'handleContentChanged'], 10, 3);
        add_action('save_post_' . PostTypes::LEGACY_READING_TYPE, [__CLASS__, 'handleContentChanged'], 10, 3);
        add_action('updated_option', [__CLASS__, 'handleOptionUpdated'], 10, 3);

        // robots.txt sitemap discovery (works even without SEO plugins).
        add_filter('robots_txt', [__CLASS__, 'filterRobotsTxt'], 10, 2);

        // Prevent self-pingbacks (internal links creating pingback comments).
        add_action('pre_ping', [__CLASS__, 'filterPrePing']);

        // Calendar storage schema (custom DB table) - safe no-op if DB not ready.
        if (class_exists(DayEntryTable::class)) {
            DayEntryTable::ensureSchema();
        }
        if (class_exists(ShareTable::class)) {
            ShareTable::ensureSchema();
        }

        // CPTs
        Deck::register();
        MeaningPack::register();
        ReadingType::register();
        Booking::register();

        // Shortcodes should register everywhere (frontend + editor previews).
        // Be defensive on case-sensitive servers / partial uploads.
        if (!class_exists(Shortcodes::class)) {
            $file = DTAROT_PATH . 'includes/Frontend/Shortcodes.php';
            if (is_readable($file)) {
                require_once $file;
            }
        }

        if (class_exists(Shortcodes::class)) {
            Shortcodes::init();
        } else {
            // Fallback: prevent raw [daily_tarot] output if the file is missing.
            add_shortcode('daily_tarot', function (): string {
                // Keep frontend clean; show a helpful message only to admins.
                if (is_user_logged_in() && current_user_can('manage_options')) {
                    return '<div class="dtarot-frontend dtarot-frontend-empty">'.esc_html__('Daily Tarot shortcode is unavailable (plugin files missing). Re-upload the plugin.','daily-tarot').'</div>';
                }
                return '';
            });
        }

        // Booking frontend handlers (AJAX + form submit).
        if (!class_exists(BookingFrontend::class)) {
            $file = DTAROT_PATH . 'includes/Frontend/Booking.php';
            if (is_readable($file)) {
                require_once $file;
            }
        }
        if (class_exists(BookingFrontend::class)) {
            BookingFrontend::init();
        }
        if (class_exists(ShareTracking::class)) {
            ShareTracking::init();
        }

        // Gutenberg blocks (server-rendered wrappers around shortcodes)
        if (class_exists(Blocks::class)) {
            Blocks::init();
        }

        // SEO-friendly readable routes (pretty permalinks for expanded daily pages)
        if (!class_exists(ReadableRoutes::class)) {
            $file = DTAROT_PATH . 'includes/Frontend/ReadableRoutes.php';
            if (is_readable($file)) {
                require_once $file;
            }
        }
        if (class_exists(ReadableRoutes::class)) {
            ReadableRoutes::init();
        }

        // Cache purge integration (auto-detect pages with our shortcode/block).
        if (class_exists(CachePurge::class)) {
            CachePurge::init();
        }

        // Optional: Yoast SEO integration (title/description/canonical/OG for readable routes).
        if (class_exists(Yoast::class)) {
            Yoast::init();
        }

        // Optional: Rank Math SEO integration (title/description/canonical/OG for readable routes).
        if (class_exists(RankMath::class)) {
            RankMath::init();
        }

        // Public sitemap for readable daily URLs.
        if (class_exists(Sitemap::class)) {
            Sitemap::init();
        }

        // Optional JSON-LD schema for readable routes.
        if (class_exists(Schema::class)) {
            Schema::init();
        }

        // Delay translation-backed defaults until init (see lateInit()).

        // Optional built-in packs for Kipper/Gypsy systems.
        if (!class_exists(KipperBasicsMeaningPack::class)) {
            $file = DTAROT_PATH . 'includes/Meaning/KipperBasicsMeaningPack.php';
            if (is_readable($file)) {
                require_once $file;
            }
        }
        if (!class_exists(KipperStandardMeaningPack::class)) {
            $file = DTAROT_PATH . 'includes/Meaning/KipperStandardMeaningPack.php';
            if (is_readable($file)) {
                require_once $file;
            }
        }
        // Packs are ensured in lateInit().

        // Public REST API (published readings only)
        if (class_exists(ReadingsController::class)) {
            ReadingsController::init();
        }

        // Optional: built-in AI provider endpoint (same-site provider).
        if (class_exists(AiProviderController::class)) {
            AiProviderController::init();
        }

        // Automation (WP-Cron)
        if (class_exists(Scheduler::class)) {
            Scheduler::init();
        }
        if (class_exists(DailyJob::class)) {
            DailyJob::init();
        }

        // AI Prefill (WP-Cron)
        if (class_exists(AiPrefillScheduler::class)) {
            AiPrefillScheduler::init();
        }
        if (class_exists(AiPrefillJob::class)) {
            AiPrefillJob::init();
        }

        // Admin
        if (is_admin()) {
            add_action('admin_init', [__CLASS__, 'maybeFlushRewritesOnUpgrade']);
            add_action('admin_notices', [__CLASS__, 'maybeShowRewriteNotice']);

            if (class_exists(ReviewPrompt::class)) {
                ReviewPrompt::init();
                add_action('admin_post_dtarot_review_prompt', [ReviewPrompt::class, 'handleAction']);
            }

            Menu::init();
            if (class_exists(\DailyTarot\Admin\PluginsPage::class)) {
                \DailyTarot\Admin\PluginsPage::init();
            }
            Ajax::init();
            Backup::init();
            StarterDecksPrompt::init();
            \DailyTarot\Admin\DeckMeta::init();
            \DailyTarot\Admin\MeaningPackMeta::init();
            \DailyTarot\Admin\ReadingTypeMeta::init();
            \DailyTarot\Admin\BookingMeta::init();
            \DailyTarot\Admin\BookingActions::init();

            // Pro-only: suggest internal links while editing WP posts/pages.
            if (class_exists(\DailyTarot\Admin\PostLinkSuggestions::class)) {
                \DailyTarot\Admin\PostLinkSuggestions::init();
            }

            // Admin-post handlers (Settings → Automation)
            if (class_exists(AutomationSettings::class)) {
                add_action('admin_post_dtarot_automation_save', [AutomationSettings::class, 'handleSave']);
                add_action('admin_post_dtarot_automation_run_now', [AutomationSettings::class, 'handleRunNow']);
                add_action('admin_post_dtarot_automation_reschedule', [AutomationSettings::class, 'handleReschedule']);
                add_action('admin_post_dtarot_automation_email_preview', [AutomationSettings::class, 'handleEmailPreview']);
                add_action('admin_post_dtarot_automation_test_email', [AutomationSettings::class, 'handleTestEmail']);
                add_action('admin_post_dtarot_automation_publish_for_date', [AutomationSettings::class, 'handlePublishForDate']);
            }

            // Admin-post handlers (AI → Automation)
            if (class_exists(AiPrefillSettings::class)) {
                add_action('admin_post_dtarot_ai_prefill_save', [AiPrefillSettings::class, 'handleSave']);
            }

            // Admin-post handlers (Settings → Diagnostics)
            add_action('admin_post_dtarot_calendar_migrate_table', [DayEntryStorageAdmin::class, 'handleMigrateToTable']);
            add_action('admin_post_dtarot_diagnostics_fetch_public', [Settings::class, 'handleFetchPublicDiagnostics']);
            add_action('admin_post_dtarot_log_export', [Settings::class, 'handleLogExport']);
            add_action('admin_post_dtarot_log_clear', [Settings::class, 'handleLogClear']);

            // Admin-post handlers (Calendar → Publish times)
            if (class_exists(CalendarPublishTimes::class)) {
                add_action('admin_post_dtarot_calendar_publish_times_save', [CalendarPublishTimes::class, 'handleSave']);
            }

            // Admin-post handlers (Settings → UI)
            add_action('admin_post_dtarot_ui_save', [UiSettings::class, 'handleSave']);

            // Admin-post handlers (Settings → Pro → AI provider)
            if (class_exists(AiProviderSettings::class)) {
                add_action('admin_post_dtarot_ai_provider_save', [AiProviderSettings::class, 'handleSave']);
            }

            // Admin-post handlers (Settings → CTA)
            add_action('admin_post_dtarot_email_cta_save', [EmailCtaSettings::class, 'handleSave']);

            // Admin-post handlers (Settings → Shortcode)
            add_action('admin_post_dtarot_shortcode_save', [ShortcodeSettings::class, 'handleSave']);
            add_action('admin_post_dtarot_share_image_save', [ShareImageSettings::class, 'handleSave']);

            // Admin-post handlers (Settings → Shortcode → Related links)
            if (class_exists(RelatedLinks::class)) {
                add_action('admin_post_dtarot_related_links_save', [RelatedLinks::class, 'handleSave']);
            }

            // Admin-post handlers (Settings → Bookings)
            add_action('admin_post_dtarot_booking_settings_save', [BookingSettings::class, 'handleSave']);

            // Admin-post handlers (Spreads)
            add_action('admin_post_dtarot_spreads_scan', [SpreadAdmin::class, 'handleScan']);
            add_action('admin_post_dtarot_spread_mapping_save', [SpreadAdmin::class, 'handleMappingSave']);
            add_action('admin_post_dtarot_spread_options_save', [SpreadAdmin::class, 'handleOptionsSave']);
            add_action('admin_post_dtarot_spread_pack_import', [SpreadAdmin::class, 'handlePackImport']);
            add_action('admin_post_dtarot_spread_pack_save', [SpreadAdmin::class, 'handlePackSave']);
            add_action('admin_post_dtarot_spread_pack_create', [SpreadAdmin::class, 'handlePackCreate']);

            // Admin-post handlers (Analytics)
            if (class_exists(Analytics::class)) {
                add_action('admin_post_dtarot_analytics_export', [Analytics::class, 'handleExport']);
            }

            // Admin-post handlers (Decks → Default deck per system)
            add_action('admin_post_dtarot_set_default_deck', [DefaultDecks::class, 'handleSetDefault']);

            // Admin-post handlers (Meaning Packs → Default meaning pack per system)
            add_action('admin_post_dtarot_set_default_pack', [DefaultMeaningPacks::class, 'handleSetDefault']);

            // Admin-post handlers (Dev mode → force Free/Pro)
            add_action('admin_post_dtarot_license_sync', [License::class, 'handleSync']);
            add_action('admin_post_dtarot_feedback_submit', [Feedback::class, 'handleSubmit']);

            // Admin-post handlers (Permalinks)
            add_action('admin_post_dtarot_flush_rewrites', [__CLASS__, 'handleFlushRewrites']);
        }

        // SEO plugin sitemap index integration.
        // This keeps Daily Tarot URLs discoverable without custom SEO settings.
        add_filter('wpseo_sitemap_index', [__CLASS__, 'filterYoastSitemapIndex']);
        add_filter('rank_math/sitemap/index', [__CLASS__, 'filterRankMathSitemapIndex']);

        // Frontend CTA submit (wp_mail)
        add_action('admin_post_dtarot_email_cta_submit', [EmailCtaSettings::class, 'handleSubmit']);
        add_action('admin_post_nopriv_dtarot_email_cta_submit', [EmailCtaSettings::class, 'handleSubmit']);

        if (is_admin() && class_exists(EmailCtaStore::class)) {
            add_action('admin_post_dtarot_email_cta_export', [EmailCtaStore::class, 'handleExport']);
        }

        if (class_exists(SpreadScanner::class)) {
            SpreadScanner::init();
        }

        if (is_admin()) {
            add_action('enqueue_block_editor_assets', [__CLASS__, 'enqueueBlockEditorAssets']);
        }
    }

    public static function lateInit(): void {
        // Silent one-time upgrade migrations.
        if (class_exists(Upgrade::class)) {
            Upgrade::maybeRun();
        }

        // Ensure we always have at least one meaning pack available.
        if (class_exists(DefaultMeaningPack::class)) {
            DefaultMeaningPack::ensureExists();
        }

        // Ensure default spread meaning pack exists.
        if (class_exists(SpreadMeaningPacks::class)) {
            SpreadMeaningPacks::ensureDefaultPackExists();
        }

        // Optional built-in packs for Kipper/Gypsy systems.
        if (class_exists(KipperBasicsMeaningPack::class)) {
            KipperBasicsMeaningPack::ensureExists();
        }
        if (class_exists(KipperStandardMeaningPack::class)) {
            KipperStandardMeaningPack::ensureExists();
        }

        // Ensure default reading types exist.
        if (class_exists(DefaultReadingTypes::class)) {
            DefaultReadingTypes::ensureExists();
        }
    }

    public static function enqueueBlockEditorAssets(): void {
        $cssFile = DTAROT_PATH . 'assets/frontend.css';
        $cssVer = is_readable($cssFile) ? (string)@filemtime($cssFile) : DTAROT_VERSION;
        wp_enqueue_style('dtarot-frontend', DTAROT_URL.'assets/frontend.css', [], $cssVer);

        $jsFile = DTAROT_PATH . 'assets/frontend.js';
        $ver = is_readable($jsFile) ? (string)@filemtime($jsFile) : DTAROT_VERSION;
        wp_enqueue_script('dtarot-frontend-js', DTAROT_URL.'assets/frontend.js', [], $ver, true);

        wp_localize_script('dtarot-frontend-js', 'DTAROT_FRONTEND', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('dtarot_booking'),
            'shareNonce' => wp_create_nonce('dtarot_share'),
            'i18n' => [
                'loading' => __('Loading...','daily-tarot'),
                'noSlots' => __('No available times for this date.','daily-tarot'),
                'selectTime' => __('Select a time','daily-tarot'),
            ],
        ]);
    }

    public static function filterYoastSitemapIndex(string $index): string {
        if (!class_exists(Sitemap::class)) return $index;
        $loc = Sitemap::url();
        if ($loc === '') return $index;

        $entry = "\n<sitemap><loc>" . esc_url($loc) . "</loc></sitemap>\n";
        return $index . $entry;
    }

    /**
     * Rank Math filter receives the XML index as a string.
     */
    public static function filterRankMathSitemapIndex(string $index): string {
        if (!class_exists(Sitemap::class)) return $index;
        $loc = Sitemap::url();
        if ($loc === '') return $index;

        $entry = "\n<sitemap><loc>" . esc_url($loc) . "</loc></sitemap>\n";
        return $index . $entry;
    }

    public static function handleContentChanged(int $postId, \WP_Post $post, bool $update): void {
        if (!class_exists(CacheVersion::class)) return;
        if (wp_is_post_revision($postId)) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        CacheVersion::bump();
    }

    /**
     * Bump cache when Daily Tarot settings change.
     *
     * @param mixed $old
     * @param mixed $value
     */
    public static function handleOptionUpdated(string $option, $old, $value): void {
        if (!class_exists(CacheVersion::class)) return;
        if ($option === 'dtarot_cache_ver') return;
        if (str_starts_with($option, 'dtarot_day_lastmod_')) return;

        if (str_starts_with($option, 'dtarot_')) {
            CacheVersion::bump();
        }
    }

    public static function filterRobotsTxt(string $output, bool $public): string {
        if (!$public) return $output;
        $enabled = (bool)apply_filters('dtarot_robots_sitemap_enabled', true);
        if (!$enabled) return $output;
        if (!class_exists(Sitemap::class)) return $output;

        $url = Sitemap::url();
        if ($url === '') return $output;

        if (stripos($output, 'Sitemap:') !== false && stripos($output, $url) !== false) {
            return $output;
        }

        $output = rtrim($output) . "\nSitemap: " . $url . "\n";
        return $output;
    }

    /**
     * Remove internal links from the pingback queue to avoid self-pingback comments.
     *
     * @param array<int,string> $links
     */
    public static function filterPrePing(array &$links): void {
        if (!$links) return;

        $homeHost = wp_parse_url(home_url(), PHP_URL_HOST);
        $siteHost = wp_parse_url(site_url(), PHP_URL_HOST);
        $hosts = array_values(array_filter([
            is_string($homeHost) ? strtolower($homeHost) : '',
            is_string($siteHost) ? strtolower($siteHost) : '',
        ]));
        if (!$hosts) return;

        foreach ($links as $i => $link) {
            if (!is_string($link) || $link === '') continue;
            $h = wp_parse_url($link, PHP_URL_HOST);
            if (!is_string($h) || $h === '') continue;
            $h = strtolower($h);
            if (in_array($h, $hosts, true)) {
                unset($links[$i]);
            }
        }
        // Keep array tidy.
        $links = array_values($links);
    }
}
