<?php
declare(strict_types=1);


namespace DailyTarot\Meaning;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Admin\MeaningPackMeta;
use DailyTarot\Support\PostTypes;

/**
 * Reads meaning-pack data (text-only) mapped by canonical card IDs.
 */
final class MeaningPackRepository {

    /**
        * @return array{upright:string,reversed:string,keywords:string,correspondences:string,symbols:string,short:string,long:string}
     */
    public static function getMeaning(int $packId, string $cardId): array {
        if ($packId <= 0 || $cardId === '') {
            return self::emptyMeaning();
        }

        $post = get_post($packId);
        if (!$post || !isset($post->post_type) || !PostTypes::isMeaningPackType((string)$post->post_type)) {
            return self::emptyMeaning();
        }

        $all = get_post_meta($packId, MeaningPackMeta::META_KEY, true);
        if (!is_array($all)) {
            return self::emptyMeaning();
        }

        $m = $all[$cardId] ?? null;
        if (!is_array($m)) {
            return self::emptyMeaning();
        }

        return self::sanitizeMeaning($m);
    }

    /**
     * @return array{upright:string,reversed:string,keywords:string,correspondences:string,symbols:string,short:string,long:string}
     */
    public static function emptyMeaning(): array {
        return [
            'upright' => '',
            'reversed' => '',
            'keywords' => '',
            'correspondences' => '',
            'symbols' => '',
            'short' => '',
            'long' => '',
        ];
    }

    /** @param array<string,mixed> $m
     *  @return array{upright:string,reversed:string,keywords:string,correspondences:string,symbols:string,short:string,long:string}
     */
    private static function sanitizeMeaning(array $m): array {
        $out = self::emptyMeaning();

        foreach (array_keys($out) as $k) {
            if (isset($m[$k]) && !is_array($m[$k])) {
                $out[$k] = (string)$m[$k];
            }
        }

        return $out;
    }
}
