<?php
declare(strict_types=1);


namespace DailyTarot\Meaning;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Admin\MeaningPackMeta;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;

final class KipperStandardMeaningPack {
    private const META_BUILTIN_KEY = '_dtarot_builtin_pack';
    private const BUILTIN_SLUG = 'kipper_standard_v2';

    /**
     * Legacy built-in slugs used by earlier builds.
     *
     * Note: earlier builds used the slugs kipper_basics_v1 / gypsy_basics_v1 for the
     * Tarot Gypsy deck. Those slugs are now owned by the Gypsy system, so Standard Kipper
     * does not migrate them.
     */
    private const LEGACY_BUILTIN_SLUGS = [
        // (intentionally empty)
    ];

    /**
     * Creates the built-in Standard Kipper meaning pack if missing.
     *
     * If a legacy built-in Kipper pack is found, it is migrated in-place to the
     * new slug and updated to the current built-in data.
     *
     * @return int Post ID, or 0 on failure.
     */
    public static function ensureExists(): int {
        $existing = self::getIdBySlug(self::BUILTIN_SLUG);
        if ($existing > 0) {
            return $existing;
        }

        foreach (self::LEGACY_BUILTIN_SLUGS as $legacySlug) {
            $legacy = self::getIdBySlug($legacySlug);
            if ($legacy <= 0) continue;

            update_post_meta((int)$legacy, self::META_BUILTIN_KEY, self::BUILTIN_SLUG);
            update_post_meta((int)$legacy, MeaningPackMeta::META_SYSTEM, Cards::SYSTEM_KIPPER);

            wp_update_post([
                'ID' => (int)$legacy,
                'post_title' => __('Kipper — Standard','daily-tarot'),
            ]);

            update_post_meta((int)$legacy, MeaningPackMeta::META_KEY, self::data());
            self::migrateLegacyIdsInPlace((int)$legacy);
            return (int)$legacy;
        }

        $postId = wp_insert_post([
            'post_type' => PostTypes::MEANING_PACK,
            'post_status' => 'publish',
            'post_title' => __('Kipper — Standard','daily-tarot'),
        ], true);

        if (is_wp_error($postId) || !$postId) {
            return 0;
        }

        update_post_meta((int)$postId, self::META_BUILTIN_KEY, self::BUILTIN_SLUG);
        update_post_meta((int)$postId, MeaningPackMeta::META_SYSTEM, Cards::SYSTEM_KIPPER);
        update_post_meta((int)$postId, MeaningPackMeta::META_KEY, self::data());

        return (int)$postId;
    }

    /** @return int */
    private static function getIdBySlug(string $slug): int {
        $q = new \WP_Query([
            'post_type' => PostTypes::meaningPackTypes(),
            'post_status' => ['publish','draft'],
            'posts_per_page' => 1,
            'fields' => 'ids',
            'meta_key' => self::META_BUILTIN_KEY, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- 1-row lookup.
            'meta_value' => $slug, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- 1-row lookup.
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ]);

        if (!empty($q->posts[0])) {
            return (int)$q->posts[0];
        }

        return 0;
    }

    /**
     * Migrates legacy (gypsy_XX) keys to canonical (kipper_XX) in-place.
     *
     * This should be safe even after seeding built-in data.
     */
    private static function migrateLegacyIdsInPlace(int $packId): void {
        if ($packId <= 0) return;

        $all = get_post_meta($packId, MeaningPackMeta::META_KEY, true);
        if (!is_array($all) || empty($all)) return;

        $changed = false;
        $updated = [];
        foreach ($all as $cardId => $meaning) {
            if (is_string($cardId) && preg_match('/^gypsy_(\\d{2})$/', $cardId, $m)) {
                $updated['kipper_' . $m[1]] = $meaning;
                $changed = true;
                continue;
            }
            $updated[$cardId] = $meaning;
        }

        if ($changed) {
            update_post_meta($packId, MeaningPackMeta::META_KEY, $updated);
        }
    }

    /**
     * @return array<string,array<string,string>>
     */
    private static function data(): array {
        $intro = 'Some basics of card meaning. Further study will be needed to learn how to apply these card meanings to your spreads.';

        $makeLong = static function (array $sections, bool $includeIntro = false) use ($intro): string {
            $out = '';
            if ($includeIntro) {
                $out .= '<p>' . esc_html($intro) . '</p>';
            }
            foreach ($sections as $label => $value) {
                if ($value === '') continue;
                $out .= '<p><strong>' . esc_html($label) . ':</strong> ' . esc_html($value) . '</p>';
            }
            return $out;
        };

        return [
            'kipper_01' => [
                'keywords' => 'male querent, important man, identity, direction, initiative, focus',
                'upright' => 'The male querent or the most important man in the situation.',
                'reversed' => '',
                'short' => 'Key male figure.',
                'long' => $makeLong([
                    'Meaning' => 'The male querent or the most important man in the situation.',
                    'Themes' => 'Identity, personal direction, ego, initiative.',
                    'Note' => 'Focus card: events revolve around him.',
                ], true),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_02' => [
                'keywords' => 'female querent, important woman, emotions, inner life, relationships',
                'upright' => 'The female querent or the most important woman in the situation.',
                'reversed' => '',
                'short' => 'Key female figure.',
                'long' => $makeLong([
                    'Meaning' => 'The female querent or the most important woman in the situation.',
                    'Themes' => 'Emotional focus, personal life, inner experience.',
                    'Note' => 'Central figure in relationship questions.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_03' => [
                'keywords' => 'union, partnership, contract, agreement, commitment, bond',
                'upright' => 'Union, partnership, contract, or agreement between two parties.',
                'reversed' => '',
                'short' => 'Union and agreement.',
                'long' => $makeLong([
                    'Meaning' => 'Union, partnership, contract, agreement.',
                    'Themes' => 'Emotional or practical bond between two parties.',
                    'Note' => 'Can indicate commitment or separation depending on surrounding cards.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_04' => [
                'keywords' => 'meeting, gathering, encounter, appointment, reunion, networking',
                'upright' => 'A meeting, encounter, or social interaction brings things together.',
                'reversed' => '',
                'short' => 'Meeting and connection.',
                'long' => $makeLong([
                    'Meaning' => 'Gathering, encounter, social interaction.',
                    'Themes' => 'Appointments, networking, reunions.',
                    'Note' => 'Things come together physically or socially.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_05' => [
                'keywords' => 'helpful man, support, protection, mentor, reliable partner, benefactor',
                'upright' => 'A helpful, kind, reliable man brings support or protection.',
                'reversed' => '',
                'short' => 'Supportive man.',
                'long' => $makeLong([
                    'Meaning' => 'Helpful, kind, reliable man.',
                    'Themes' => 'Support, protection, positive male influence.',
                    'Note' => 'Mentor, benefactor, trustworthy partner.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_06' => [
                'keywords' => 'supportive woman, kindness, nurturing, ally, emotional help, protector',
                'upright' => 'A caring, supportive woman offers kindness and emotional help.',
                'reversed' => '',
                'short' => 'Supportive woman.',
                'long' => $makeLong([
                    'Meaning' => 'Caring, supportive woman.',
                    'Themes' => 'Emotional help, kindness, nurturing energy.',
                    'Note' => 'Female ally or protector.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_07' => [
                'keywords' => 'message, news, email, document, letter, contract, announcement',
                'upright' => 'A message or document brings information and decisions.',
                'reversed' => '',
                'short' => 'News and documents.',
                'long' => $makeLong([
                    'Meaning' => 'Message, news, email, document.',
                    'Themes' => 'Communication that brings information.',
                    'Note' => 'Contracts, written decisions, announcements.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_08' => [
                'keywords' => 'deception, dishonesty, hidden motives, mask, misleading, warning',
                'upright' => 'Deception or hidden motives are present; verify facts and intentions.',
                'reversed' => '',
                'short' => 'Deception warning.',
                'long' => $makeLong([
                    'Meaning' => 'Deception, dishonesty, hidden motives.',
                    'Themes' => 'Someone wearing a mask — or a misleading situation.',
                    'Note' => 'Warning to verify facts and intentions.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_09' => [
                'keywords' => 'change, transformation, transition, shift, new direction, movement',
                'upright' => 'A shift or transition changes the direction of events.',
                'reversed' => '',
                'short' => 'Change and transition.',
                'long' => $makeLong([
                    'Meaning' => 'Transformation, shift, transition.',
                    'Themes' => 'Life direction alters.',
                    'Note' => 'Movement from one state to another.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_10' => [
                'keywords' => 'travel, distance, movement, relocation, progress, separation',
                'upright' => 'Travel or movement—literal or metaphorical—creates distance and progress.',
                'reversed' => '',
                'short' => 'Travel and movement.',
                'long' => $makeLong([
                    'Meaning' => 'Travel, distance, movement.',
                    'Themes' => 'Physical trip or metaphorical life path.',
                    'Note' => 'Separation, relocation, progress through space.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_11' => [
                'keywords' => 'success, abundance, luck, gain, winning, reward, opportunity',
                'upright' => 'A major gain or lucky opportunity brings abundance and success.',
                'reversed' => '',
                'short' => 'Big win and gain.',
                'long' => $makeLong([
                    'Meaning' => 'Major gain, success, abundance.',
                    'Themes' => 'Luck, winning, fortunate opportunity.',
                    'Note' => 'Can indicate financial or symbolic reward.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_12' => [
                'keywords' => 'confident woman, comfort, charm, lifestyle, privilege, social visibility',
                'upright' => 'A confident woman or a socially visible influence tied to comfort and status.',
                'reversed' => '',
                'short' => 'Comfort and social ease.',
                'long' => $makeLong([
                    'Meaning' => 'Confident woman, social ease, comfort.',
                    'Themes' => 'Beauty, charm, lifestyle, privilege.',
                    'Note' => 'Sometimes a rival or socially visible woman.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_13' => [
                'keywords' => 'influential man, wealth, authority, ambition, status, power dynamics',
                'upright' => 'An influential man, wealth, or authority affects outcomes and power dynamics.',
                'reversed' => '',
                'short' => 'Wealth and authority.',
                'long' => $makeLong([
                    'Meaning' => 'Influential man, wealth, authority.',
                    'Themes' => 'Career focus, ambition, status.',
                    'Note' => 'Power dynamics; success through position.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_14' => [
                'keywords' => 'sad news, disappointment, painful truth, reality check, clarity',
                'upright' => 'Disappointing information arrives; a truth that hurts but clarifies.',
                'reversed' => '',
                'short' => 'Disappointing news.',
                'long' => $makeLong([
                    'Meaning' => 'Disappointing or painful information.',
                    'Themes' => 'Emotional impact, letdown, reality check.',
                    'Note' => 'Truth that hurts but clarifies.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_15' => [
                'keywords' => 'love outcome, harmony, reconciliation, fulfillment, happy ending',
                'upright' => 'A happy emotional resolution: harmony, reconciliation, fulfillment.',
                'reversed' => '',
                'short' => 'Favorable love outcome.',
                'long' => $makeLong([
                    'Meaning' => 'Happy emotional resolution.',
                    'Themes' => 'Harmony, reconciliation, emotional fulfillment.',
                    'Note' => 'Favorable ending, especially in relationships.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_16' => [
                'keywords' => 'thoughts, plans, strategy, analysis, concerns, mental focus, unspoken',
                'upright' => 'Thoughts, plans, and concerns—what someone is thinking but not saying.',
                'reversed' => '',
                'short' => 'Plans and inner thoughts.',
                'long' => $makeLong([
                    'Meaning' => 'Thoughts, plans, concerns, mental focus.',
                    'Themes' => 'Strategy, analysis, internal dialogue.',
                    'Note' => 'What someone is thinking but not saying.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_17' => [
                'keywords' => 'gift, benefit, favor, bonus, kindness, compensation, help',
                'upright' => 'A gift, favor, or unexpected benefit arrives.',
                'reversed' => '',
                'short' => 'Gift and benefit.',
                'long' => $makeLong([
                    'Meaning' => 'Gift, benefit, favor.',
                    'Themes' => 'Something given freely or unexpectedly.',
                    'Note' => 'Compensation, kindness, bonus.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_18' => [
                'keywords' => 'child, new beginning, innocence, vulnerability, fresh start, learning',
                'upright' => 'A new beginning: something young, fragile, or still developing.',
                'reversed' => '',
                'short' => 'New start and innocence.',
                'long' => $makeLong([
                    'Meaning' => 'New beginning, innocence, vulnerability.',
                    'Themes' => 'Something young, undeveloped, or fragile.',
                    'Note' => 'Fresh start, learning phase.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_19' => [
                'keywords' => 'ending, closure, transformation, irreversible change, letting go',
                'upright' => 'An ending or closure: one phase ends so another may begin.',
                'reversed' => '',
                'short' => 'Endings and transformation.',
                'long' => $makeLong([
                    'Meaning' => 'Ending, closure, irreversible change.',
                    'Themes' => 'One phase ends so another may begin.',
                    'Note' => 'Letting go, transformation.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_20' => [
                'keywords' => 'home, family, property, stability, security, foundations, private life',
                'upright' => 'Home and stability: property, security, and private life matters.',
                'reversed' => '',
                'short' => 'Home and security.',
                'long' => $makeLong([
                    'Meaning' => 'Home, family, stability.',
                    'Themes' => 'Property, private life, security.',
                    'Note' => 'Foundations, comfort, domestic matters.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_21' => [
                'keywords' => 'inner circle, everyday life, familiar, comfort zone, social ease',
                'upright' => 'Your inner circle and everyday life; staying within familiar boundaries.',
                'reversed' => '',
                'short' => 'Inner circle and comfort.',
                'long' => $makeLong([
                    'Meaning' => 'Inner circle, everyday life.',
                    'Themes' => 'Familiar environment, social comfort.',
                    'Note' => 'Staying within known boundaries.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_22' => [
                'keywords' => 'discipline, authority, duty, pressure, conflict, rules, structure',
                'upright' => 'Discipline and authority: duty, pressure, and strict structure.',
                'reversed' => '',
                'short' => 'Duty and authority.',
                'long' => $makeLong([
                    'Meaning' => 'Discipline, authority, duty.',
                    'Themes' => 'Conflict, endurance, pressure.',
                    'Note' => 'Strict rules, structured environment.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_23' => [
                'keywords' => 'legal matters, justice, judgement, contracts, responsibility, rules',
                'upright' => 'Legal or official matters: justice, formal judgment, accountability.',
                'reversed' => '',
                'short' => 'Legal and official decisions.',
                'long' => $makeLong([
                    'Meaning' => 'Legal matters, justice, formal judgment.',
                    'Themes' => 'Contracts, responsibility, official decisions.',
                    'Note' => 'Fairness, accountability, rules.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_24' => [
                'keywords' => 'theft, loss, betrayal, trust, taken away, misuse',
                'upright' => 'Loss or betrayal: something is taken away or misused.',
                'reversed' => '',
                'short' => 'Loss and betrayal.',
                'long' => $makeLong([
                    'Meaning' => 'Loss, deception, something taken away.',
                    'Themes' => 'Betrayal, misuse of trust.',
                    'Note' => 'Loss of time, money, energy, or opportunity.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_25' => [
                'keywords' => 'recognition, achievement, reputation, promotion, prestige, status',
                'upright' => 'Recognition and achievement: promotion, success, rising status.',
                'reversed' => '',
                'short' => 'Recognition and status.',
                'long' => $makeLong([
                    'Meaning' => 'Recognition, achievement, reputation.',
                    'Themes' => 'Promotion, success, public acknowledgment.',
                    'Note' => 'Rising status, prestige.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_26' => [
                'keywords' => 'luck, protection, destiny, breakthrough, favorable outcome, fortune',
                'upright' => 'Major luck and protection: a favorable breakthrough even despite obstacles.',
                'reversed' => '',
                'short' => 'Great fortune and protection.',
                'long' => $makeLong([
                    'Meaning' => 'Major luck, protection, favorable destiny.',
                    'Themes' => 'Positive breakthrough.',
                    'Note' => 'Things work out despite obstacles.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_27' => [
                'keywords' => 'unexpected money, bonus, refund, surprise income, financial help',
                'upright' => 'Surprise income or financial help arrives unexpectedly.',
                'reversed' => '',
                'short' => 'Surprise income.',
                'long' => $makeLong([
                    'Meaning' => 'Surprise income, refund, bonus.',
                    'Themes' => 'Financial help from an unexpected source.',
                    'Note' => 'Sudden material improvement.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_28' => [
                'keywords' => 'waiting, expectation, anticipation, pause, preparation, timing',
                'upright' => 'Waiting and anticipation: a pause while things form and timing ripens.',
                'reversed' => '',
                'short' => 'Waiting and anticipation.',
                'long' => $makeLong([
                    'Meaning' => 'Waiting, anticipation, pause.',
                    'Themes' => 'Preparation phase.',
                    'Note' => 'Things are forming but not ready yet.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_29' => [
                'keywords' => 'restriction, limitation, confinement, blocked, controlled, stuck',
                'upright' => 'Restriction or limitation: feeling stuck, blocked, or controlled.',
                'reversed' => '',
                'short' => 'Restriction and blockage.',
                'long' => $makeLong([
                    'Meaning' => 'Restriction, limitation, confinement.',
                    'Themes' => 'Feeling stuck, blocked, controlled.',
                    'Note' => 'External circumstances prevent movement.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_30' => [
                'keywords' => 'authority, institution, bureaucracy, paperwork, approval, administration',
                'upright' => 'An authority figure or institution influences outcomes through formal channels.',
                'reversed' => '',
                'short' => 'Authority and institutions.',
                'long' => $makeLong([
                    'Meaning' => 'Authority figure, bureaucrat, institution.',
                    'Themes' => 'Paperwork, administration, formal channels.',
                    'Note' => 'Someone who must approve or decide.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_31' => [
                'keywords' => 'short illness, burnout, weakness, delay, recovery, rest',
                'upright' => 'Temporary weakness or burnout causes delays; rest and recovery are needed.',
                'reversed' => '',
                'short' => 'Temporary illness and delay.',
                'long' => $makeLong([
                    'Meaning' => 'Temporary weakness, burnout.',
                    'Themes' => 'Delay due to lack of energy.',
                    'Note' => 'Need for rest and recovery.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_32' => [
                'keywords' => 'grief, sadness, mourning, sorrow, regret, emotional heaviness',
                'upright' => 'Grief and emotional pain: processing loss, disappointment, or regret.',
                'reversed' => '',
                'short' => 'Grief and sadness.',
                'long' => $makeLong([
                    'Meaning' => 'Sadness, emotional pain, mourning.',
                    'Themes' => 'Disappointment, sorrow, emotional heaviness.',
                    'Note' => 'Processing loss or regret.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_33' => [
                'keywords' => 'anxiety, worry, pessimism, fear, doubt, mental stress',
                'upright' => 'Anxiety and worry cloud perception; fear-driven thinking needs grounding.',
                'reversed' => '',
                'short' => 'Worry and anxiety.',
                'long' => $makeLong([
                    'Meaning' => 'Anxiety, worry, pessimism.',
                    'Themes' => 'Mental stress, fear-driven thinking.',
                    'Note' => 'Inner doubts clouding perception.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_34' => [
                'keywords' => 'work, effort, labor, responsibility, routine, productivity, duty',
                'upright' => 'Work and responsibility: consistent effort brings results over time.',
                'reversed' => '',
                'short' => 'Work and effort.',
                'long' => $makeLong([
                    'Meaning' => 'Effort, labor, responsibility.',
                    'Themes' => 'Consistent work brings results.',
                    'Note' => 'Focus on duty, productivity, routine.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_35' => [
                'keywords' => 'long road, long-term, endurance, distance, patience, steady progress',
                'upright' => 'A long-term process: slow but steady progress, patience required.',
                'reversed' => '',
                'short' => 'Long-term progress.',
                'long' => $makeLong([
                    'Meaning' => 'Long-term process, distance, endurance.',
                    'Themes' => 'Slow but steady progress.',
                    'Note' => 'Patience required; future-oriented outcome.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_36' => [
                'keywords' => 'hope, faith, optimism, healing, renewal, light, improvement',
                'upright' => 'Hope and healing: trust in future improvement and light after darkness.',
                'reversed' => '',
                'short' => 'Hope and renewal.',
                'long' => $makeLong([
                    'Meaning' => 'Faith, optimism, healing.',
                    'Themes' => 'Light after darkness.',
                    'Note' => 'Trust in future improvement.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
        ];
    }
}
