<?php
declare(strict_types=1);


namespace DailyTarot\Meaning;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Admin\MeaningPackMeta;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;

final class KipperBasicsMeaningPack {
    private const META_BUILTIN_KEY = '_dtarot_builtin_pack';
    private const BUILTIN_SLUG = 'kipper_basics_v1';
    private const LEGACY_BUILTIN_SLUG = 'gypsy_basics_v1';

    /**
        * Creates the built-in Gypsy (Tarot Gypsy) basics pack if missing.
        *
        * Compatibility notes:
        * - Older installs stored this pack as Kipper (system + IDs like kipper_XX).
        * - New installs use the Gypsy system with IDs gypsy_XX.
        *
        * We keep the built-in slug for continuity, but migrate stored keys to gypsy_XX.
     *
     * @return int Post ID, or 0 on failure.
     */
    public static function ensureExists(): int {
        $existing = self::getIdBySlug(self::BUILTIN_SLUG);
        if ($existing > 0) {
            self::migrateInPlace($existing);
            return $existing;
        }

        $legacy = self::getIdBySlug(self::LEGACY_BUILTIN_SLUG);
        if ($legacy > 0) {
            update_post_meta((int)$legacy, self::META_BUILTIN_KEY, self::BUILTIN_SLUG);
            update_post_meta((int)$legacy, MeaningPackMeta::META_SYSTEM, Cards::SYSTEM_GYPSY);

            wp_update_post([
                'ID' => (int)$legacy,
                'post_title' => __('Gypsy — Basics','daily-tarot'),
            ]);

            self::migrateInPlace($legacy);
            return $legacy;
        }

        $postId = wp_insert_post([
            'post_type' => PostTypes::MEANING_PACK,
            'post_status' => 'publish',
            'post_title' => __('Gypsy — Basics','daily-tarot'),
        ], true);

        if (is_wp_error($postId) || !$postId) {
            return 0;
        }

        update_post_meta((int)$postId, self::META_BUILTIN_KEY, self::BUILTIN_SLUG);
        update_post_meta((int)$postId, MeaningPackMeta::META_SYSTEM, Cards::SYSTEM_GYPSY);
        update_post_meta((int)$postId, MeaningPackMeta::META_KEY, self::dataGypsy());

        return (int)$postId;
    }

    /** @return int */
    private static function getIdBySlug(string $slug): int {
        $q = new \WP_Query([
            'post_type' => PostTypes::meaningPackTypes(),
            'post_status' => ['publish','draft'],
            'posts_per_page' => 1,
            'fields' => 'ids',
            'meta_key' => self::META_BUILTIN_KEY, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- 1-row lookup.
            'meta_value' => $slug, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- 1-row lookup.
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ]);

        if (!empty($q->posts[0])) {
            return (int)$q->posts[0];
        }

        return 0;
    }

    private static function migrateInPlace(int $packId): void {
        if ($packId <= 0) return;

        // Ensure correct system + title on existing installs (idempotent).
        $needsUpdate = false;
        $ps = Cards::normalizeSystem((string)get_post_meta($packId, MeaningPackMeta::META_SYSTEM, true));
        if ($ps !== Cards::SYSTEM_GYPSY) {
            update_post_meta($packId, MeaningPackMeta::META_SYSTEM, Cards::SYSTEM_GYPSY);
            $needsUpdate = true;
        }

        $p = get_post($packId);
        if ($p && is_object($p) && isset($p->post_title)) {
            $want = __('Gypsy — Basics','daily-tarot');
            if ((string)$p->post_title !== $want) {
                wp_update_post([
                    'ID' => (int)$packId,
                    'post_title' => $want,
                ]);
                $needsUpdate = true;
            }
        }

        $all = get_post_meta($packId, MeaningPackMeta::META_KEY, true);
        if (!is_array($all) || empty($all)) {
            update_post_meta($packId, MeaningPackMeta::META_KEY, self::dataGypsy());
            return;
        }

        $changed = false;
        $updated = $all;
        foreach ($all as $cardId => $meaning) {
            if (!is_string($cardId)) continue;

            // Old installs stored this pack keyed by kipper_XX; migrate to gypsy_XX.
            if (preg_match('/^kipper_(\\d{2})$/', $cardId, $m)) {
                $newId = 'gypsy_' . $m[1];
                if (!isset($updated[$newId])) {
                    $updated[$newId] = $meaning;
                }
                unset($updated[$cardId]);
                $changed = true;
            }
        }

        if ($changed || $needsUpdate) {
            update_post_meta($packId, MeaningPackMeta::META_KEY, $updated);
        }
    }

    /**
     * Converts the original data (keyed by kipper_XX) into Gypsy IDs (gypsy_XX).
     *
     * @return array<string,mixed>
     */
    private static function dataGypsy(): array {
        $base = self::data();
        $out = [];
        foreach ($base as $cardId => $meaning) {
            if (!is_string($cardId)) continue;
            if (preg_match('/^kipper_(\\d{2})$/', $cardId, $m)) {
                $out['gypsy_' . $m[1]] = $meaning;
            } else {
                $out[$cardId] = $meaning;
            }
        }
        return $out;
    }

    /**
     * @return array<string,array<string,string>>
     */
    private static function data(): array {
        $intro = "Some basics of card meaning. Further study will be needed to learn how to apply these card meanings to your spreads (e.g., using the directions of the images on the cards to modify meaning and help with interpretation of the whole spread).";

        $makeLong = static function (array $sections, bool $includeIntro = false) use ($intro): string {
            $out = '';
            if ($includeIntro) {
                $out .= '<p>' . esc_html($intro) . '</p>';
            }
            foreach ($sections as $label => $value) {
                if ($value === '') continue;
                $out .= '<p><strong>' . esc_html($label) . ':</strong> ' . esc_html($value) . '</p>';
            }
            return $out;
        };

        return [
            'kipper_01' => [
                'keywords' => 'significator, male querent, partner',
                'upright' => 'Male querent or partner of a female querent.',
                'reversed' => '',
                'short' => 'Significator (male).',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Direction' => 'Person card has direction — auspicious in front (where its facing), inauspicious behind.',
                    'Core' => 'Male querent or partner of Female querent',
                    'Type' => 'Start',
                    'Meaning' => 'This card does not have an extra meaning as it is the Significator for your Querent.',
                ], true),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_02' => [
                'keywords' => 'significator, female querent, partner',
                'upright' => 'Female querent or partner of a male querent.',
                'reversed' => '',
                'short' => 'Significator (female).',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Direction' => 'Person card has direction — auspicious in front (where its facing), inauspicious behind.',
                    'Core' => 'Female querent or partner of male querent',
                    'Type' => 'Start',
                    'Meaning' => 'This card does not have an extra meaning as it is the Significator for your Querent.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_03' => [
                'keywords' => 'marriage, connection, contract, couple, partnership, commitment',
                'upright' => 'Connection, commitment, contracts, partnership.',
                'reversed' => '',
                'short' => 'Connection and commitment.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Connection',
                    'Type' => 'Connector (connects cards that flank on either side)',
                    'Meaning' => 'Marriage, connection, contract, coming together, couple, partnership, commitment',
                    'Person' => 'Married, connected, clingy, dependent, dual personality',
                    'Significator' => 'Marriage, connection, contract',
                    'Health' => 'Treatment, health commitment, health insurance',
                    'Love' => 'Marriage, firm relationship, commitment',
                    'Work' => 'Partnership, work contract, connection',
                    'General' => 'Shows connections of all types; cards to left & right show what’s bound or connected.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_04' => [
                'keywords' => 'togetherness, harmony, parties, friends, openness, sharing',
                'upright' => 'Joyful time together; harmony and good company.',
                'reversed' => '',
                'short' => 'Happy fun times together.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Togetherness',
                    'Type' => 'Connector (joins cards on L & R with joy)',
                    'Meaning' => 'Spending joyful time together, harmony, parties, friends, generally being at one with company, openness, sharing.',
                    'Person' => 'Joyful, funny, charming, lively, loveable.',
                    'Significator' => 'Togetherness, what joyful time will you spend with others?',
                    'Health' => 'Positive wellness & recuperation spent with others',
                    'Love' => 'A loving time together, being at one.',
                    'Work' => 'Good partnerships & positive working environment',
                    'General' => 'Connector card; connects the 2 flanking cards in a joyous feeling.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_05' => [
                'keywords' => 'older man, support, trust, father, boss, regrets',
                'upright' => 'Older, well-respected man; support and trust.',
                'reversed' => '',
                'short' => 'Older supportive man.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Older gent',
                    'Type' => 'Direction (faces R is auspicious; L is inauspicious)',
                    'Meaning' => 'Older gent; only a widower if seen with Death, Enemy, Illness or Misfortune cards.',
                    'Person' => 'Separated gent, alone, regrets, nurturing, widower, single father, fatherly',
                    'Significator' => 'Father, grandpa, older male relative, boss.',
                    'Health' => 'A carer, being cared for',
                    'Love' => 'An older gent',
                    'Work' => 'Boss, carer, doctor',
                    'General' => 'Not necessarily bereaved; can also mean regrets if not a person.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_06' => [
                'keywords' => 'older woman, support, trust, mother, boss',
                'upright' => 'Older, well-respected woman; support and trust.',
                'reversed' => '',
                'short' => 'Older supportive woman.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Older lady',
                    'Type' => 'Direction (faces L is auspicious; R is inauspicious)',
                    'Meaning' => 'Older lady; only a widow if seen with Death, Enemy, Illness or Misfortune cards.',
                    'Person' => 'Separated lady, alone, nurturing, widow, single mother, motherly',
                    'Significator' => 'Mother, grandmother, older female relative, boss',
                    'Health' => 'A carer, being cared for',
                    'Love' => 'An older woman',
                    'Work' => 'Boss, Carer, Nun, Doctor',
                    'General' => 'Not necessarily bereaved; partner card to Widower; generationally mother/grandmother.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_07' => [
                'keywords' => 'news, applications, correspondence, invitation, papers',
                'upright' => 'Positive news; written correspondence and applications.',
                'reversed' => '',
                'short' => 'Written news and messages.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Written communication, superficiality',
                    'Type' => 'Stop',
                    'Meaning' => 'Positive news is on its way, applications of all kinds, all written correspondence, invitation.',
                    'Person' => 'Superficial, gossipy, chatty',
                    'Significator' => 'Written communication',
                    'Health' => 'Good results',
                    'Love' => 'Invitation, flirting',
                    'Work' => 'Applications, CV, job offer',
                    'Timing' => 'Depends on placement (approx. 5 days)',
                    'General' => 'Any pieces of paper; even with negative cards, it shines positivity via good communication.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_08' => [
                'keywords' => 'wrongness, mistrust, hidden, deception, misunderstanding',
                'upright' => 'Something is wrong; don’t trust appearances.',
                'reversed' => '',
                'short' => 'Something is not what it seems.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Wrongness',
                    'Type' => 'Direction (something false/wrong with card on L)',
                    'Meaning' => 'Something’s wrong; don’t trust it or them; hidden; not what it seems; wrong impression of somebody (good or bad).',
                    'Person' => 'Dishonest, untrustworthy, dissatisfied',
                    'Significator' => 'Wrongness',
                    'Health' => 'Wrong diagnosis',
                    'Love' => 'Dishonesty in the relationship',
                    'Work' => 'Not the right workplace; job is wrong; something wrong in the workplace.',
                    'Timing' => 'In timing: “wrong time” to do something.',
                    'General' => 'Left card receives full effect; look to right to see where wrongness comes from.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_09' => [
                'keywords' => 'happiness, joy, surprise, upturn, lucky break',
                'upright' => 'A sudden positive upturn; a pleasant surprise.',
                'reversed' => '',
                'short' => 'Sudden joy and luck.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Positive upturn',
                    'Type' => 'Direction (positive to all cards; L receives most of the sudden improvement)',
                    'Meaning' => 'Happiness, joy, surprise, upturn, lucky break, any positive surprise, a great moment.',
                    'Person' => 'Happy go-lucky, carefree, happy, bringer of good luck.',
                    'Significator' => 'Upturn',
                    'Timing' => 'Suddenly / Summer',
                    'Health' => 'Sudden healing',
                    'Love' => 'Marriage proposal, surprise love, an upturn.',
                    'Work' => 'Sudden improvements, promotion',
                    'General' => 'Very auspicious; can also indicate a “nasty surprise” if in an inauspicious position.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_10' => [
                'keywords' => 'journey, change, movement, development, travel',
                'upright' => 'Movement and change; travel or life experiences.',
                'reversed' => '',
                'short' => 'Movement and development.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Experience & change',
                    'Type' => 'Direction (moves from L to R)',
                    'Meaning' => 'A journey, experiences, change, developments, holiday, travelling, movement, development, move, car, ship, plane etc.',
                    'Person' => 'Changeable, well-travelled, sporty',
                    'Significator' => 'Experience & change',
                    'Timing' => 'Movement starts now',
                    'Health' => 'Recharging, health spa',
                    'Love' => 'Long distance love, love getaway',
                    'Work' => 'Lots of movement at work, work travel',
                    'General' => 'Shows movement from L to R; can be literal travel or change between experiences.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_11' => [
                'keywords' => 'wealth, money, abundance, profit, inheritance',
                'upright' => 'Material wealth and abundance; positive resources.',
                'reversed' => '',
                'short' => 'Abundance and finances.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Abundance',
                    'Type' => 'Stop',
                    'Meaning' => 'Material wealth, money, profit, riches, wish fulfilment, achievement, abundance, bank, large sums of money, stocks, inheritance etc.',
                    'Person' => 'Affluent, generous, supportive, noble, gentry, banker',
                    'Significator' => 'Finances',
                    'Health' => 'Overindulgence, diabetes, gout, overweight',
                    'Love' => 'A rich love life, partner/marriage is abundant',
                    'Work' => 'High salary, sales, banking, finance etc.',
                    'General' => 'Shows abundance in general, material or emotional; a positive wealth in any topic.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_12' => [
                'keywords' => 'visit, invitation, meetings, new possibilities',
                'upright' => 'Surprise visit; meetings and new possibilities.',
                'reversed' => '',
                'short' => 'Visits and meetings.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Togetherness',
                    'Type' => 'Connector (cards on either side visit together)',
                    'Meaning' => 'Surprise visit / visits of all sorts, invitation to a party, meet/communicate/get to know others. New possibilities, something new comes your way, coming & going.',
                    'Person' => 'Communicative, sociable, chatty, charming',
                    'Significator' => 'Meetings',
                    'Timing' => '2 weeks and Spring',
                    'Health' => 'Appointments, consultations',
                    'Love' => 'Loving meetings, dates',
                    'Work' => 'Meetings, interviews, busy times',
                    'General' => 'Connects the two flanking cards; includes appointments, interviews, meetings.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_13' => [
                'keywords' => 'loss, shortcomings, self-imposed, separation',
                'upright' => 'Self-imposed loss; losing something of your own accord.',
                'reversed' => '',
                'short' => 'A loss you cause.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Loss',
                    'Type' => 'Connector',
                    'Meaning' => 'You’ll lose something of your own accord, losses of all sorts that are deserved, short-comings',
                    'Person' => 'Slow-witted, dim, disappointed, bitter, lack of self-confidence & little self-control.',
                    'Significator' => 'Self-imposed loss',
                    'Health' => 'Loss of bodily function due to own actions, deficiencies',
                    'Love' => 'Loss of love, separation',
                    'Work' => 'Loss of job',
                    'General' => 'Unlike Theft, Loss is something you lose yourself; connects the cards that create the loss.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_14' => [
                'keywords' => 'sadness, grief, disappointment, tears, depression',
                'upright' => 'Sadness, grief, and disappointment; cause-and-effect.',
                'reversed' => '',
                'short' => 'A period of sadness.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Sadness',
                    'Type' => 'Direction (cause behind; effect in front)',
                    'Meaning' => 'Sadness, grief, disappointment, gloominess, distressed, troubled, depressed, sorrow, anguish, tears',
                    'Person' => 'Sad, down, sensitive, pale',
                    'Significator' => 'Sadness',
                    'Timing' => 'Autumn',
                    'Health' => 'Chronic illness, depression',
                    'Love' => 'Unrequited love, unhappy relationship',
                    'Work' => 'Depressed with work, work stress',
                    'General' => 'Card behind is cause; card in front is affected by the sadness.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_15' => [
                'keywords' => 'love, happiness, deep feelings, good outcome',
                'upright' => 'Love and positive outcomes in all areas.',
                'reversed' => '',
                'short' => 'Love; a “yes” card.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Love',
                    'Type' => 'Direction (shines love on all cards)',
                    'Meaning' => 'Love, happiness, good times, deep feelings',
                    'Person' => 'Loving, warm, nice',
                    'Significator' => 'Love, good outcome in love',
                    'Health' => 'Heart, lovesickness',
                    'Love' => 'Trump card for all things love',
                    'Work' => 'Good outcome in your work life',
                    'General' => 'Also positive outcome in life generally; often indicates a Yes.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_16' => [
                'keywords' => 'thoughts, plans, mental, consideration',
                'upright' => 'Thoughts, planning, and mental activity.',
                'reversed' => '',
                'short' => 'Thinking and planning.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Thoughts',
                    'Type' => 'Direction (thinking of card to L)',
                    'Meaning' => 'Anything of a mental nature, somebody is thinking of you',
                    'Person' => 'Thoughtful, quiet, introvert, balanced, relaxed, calculated',
                    'Significator' => 'Thoughts, plans',
                    'Timing' => 'Planning process, early stages',
                    'Health' => 'Hypochondria',
                    'Love' => 'Romantic thoughts',
                    'Work' => 'Plans; positions involving planning',
                    'General' => 'Practical planning; advice to think twice.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_17' => [
                'keywords' => 'gift, surprise, compliment, festiveness',
                'upright' => 'A gift or surprise; something comes your way.',
                'reversed' => '',
                'short' => 'A surprise (good or bad).',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Gift',
                    'Type' => 'Stop',
                    'Meaning' => 'Receive a gift, something good is coming your way, receive a compliment, unplanned positive occurrences, festiveness, birthdays',
                    'Person' => 'Nice, pleasant, surprising, well mannered',
                    'Significator' => 'Surprise, gift',
                    'Timing' => 'Birthday / Holidays',
                    'Health' => 'Recuperation, positive upturn',
                    'Love' => 'Gift of love, loving gifts',
                    'Work' => 'Easy time at work, praise, accolade, commendation',
                    'General' => 'Gift brings a surprise that can be nice or nasty; unlike Unexpected Joy which is positive.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_18' => [
                'keywords' => 'child, baby, new, fresh start, small',
                'upright' => 'A child, a new beginning, or something small/new.',
                'reversed' => '',
                'short' => 'New beginnings or children.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Baby, child, new',
                    'Type' => 'Stop (can be directional as a significator)',
                    'Meaning' => 'A baby, child, pregnancy (with hope), new, fresh start, young, small',
                    'Person' => 'A child, childish, naive, newbie, easy, uncomplicated, young',
                    'Significator' => 'New project, new beginning, children',
                    'Timing' => 'Beginning',
                    'Health' => 'Pregnancy (only with other cards to confirm), small illness, childhood illness',
                    'Love' => 'New love, childhood sweetheart',
                    'Work' => 'New job, new project, new start',
                    'General' => 'Significator for a child; otherwise new situations starting.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_19' => [
                'keywords' => 'end, goodbye, warning, stop, separation',
                'upright' => 'An ending; leaving the old behind.',
                'reversed' => '',
                'short' => 'An end and a new phase.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'End',
                    'Type' => 'Direction (looks to L; lights way to R)',
                    'Meaning' => 'End, goodbye, warning, leave the old behind, death, split, stop',
                    'Person' => 'Cold, deathly pale, silent, bald',
                    'Significator' => 'Ending',
                    'Timing' => 'End / end of year',
                    'Health' => 'Death, chronic illness, warning of impending illness',
                    'Love' => 'End of a relationship, separation',
                    'Work' => 'Retirement, end of a job, end of a project',
                    'General' => 'Rarely literal death; more likely an end to a situation.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_20' => [
                'keywords' => 'home, family, stability, roots, safety',
                'upright' => 'Home, family roots, stability and safety.',
                'reversed' => '',
                'short' => 'Home and stability.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Home',
                    'Type' => 'Stop',
                    'Meaning' => 'Home, family, building, stability, independent, roots, where you come from',
                    'Person' => 'Homebody, dependable, stable, trustworthy, family person, grounded',
                    'Significator' => 'Home, house',
                    'Timing' => '1–4 months',
                    'Health' => 'The whole body, stable health',
                    'Love' => 'Family, stable & grounded love',
                    'Work' => 'Home office, estate agent, office building, stable environment',
                    'General' => 'Homes, offices, any four walls; also roots and where we feel safe.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_21' => [
                'keywords' => 'spiritual, karma, advisor, church, blessing',
                'upright' => 'Spiritual guidance; belief systems and inner self.',
                'reversed' => '',
                'short' => 'Spiritual advice and belief.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Spiritual',
                    'Type' => 'Direction (person card)',
                    'Meaning' => 'A spiritual person, open attitude to spiritual belief systems, karma, karmic return, life coach, advisor, spiritual leader, church, god, spirit realms',
                    'Person' => 'Priest, spiritual advisor, open, religious, arrogant',
                    'Significator' => 'Spiritual advice',
                    'Timing' => 'Winter',
                    'Health' => 'The mind, meditation, psyche, spiritual wellbeing',
                    'Love' => 'Soul mates, twin flames, spiritual love, a priest',
                    'Work' => 'A spiritual workplace or job (reiki, yoga etc.)',
                    'General' => 'Spiritual search; can also denote inner self and powerful ally or enemy depending on position.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_22' => [
                'keywords' => 'authority, rules, discipline, official, regulation',
                'upright' => 'An official person or situation; rules and discipline.',
                'reversed' => '',
                'short' => 'Authority and regulations.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Power',
                    'Type' => 'Direction (person card)',
                    'Meaning' => 'Any official matters, all uniformed positions, influential person, directness, correctness, going by the rules, regulations, stiff upper lip, boss, selfish',
                    'Person' => 'Neat, direct, powerful, influential, boss, regulator, civil service, pilot, uniformed, officer',
                    'Significator' => 'Discipline, rules, blockages, restrictions',
                    'Timing' => '2–3 years',
                    'Health' => 'Workaholic, war wounds',
                    'Love' => 'Rival in love or discipline in the relationship',
                    'Work' => 'Civil service, police, pilot, boss, manager',
                    'General' => 'Discipline and rules; can block progress or indicate a uniformed person.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_23' => [
                'keywords' => 'justice, rights, fairness, decision, balance',
                'upright' => 'Fairness, justice, and decisions that restore balance.',
                'reversed' => '',
                'short' => 'Justice and decisions.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Justice',
                    'Type' => 'Direction (balances L & R; R is “just/right”)',
                    'Meaning' => 'Judge, juristic matters, rights, justice, fairness, process, decisions',
                    'Person' => 'Judge, righteous, lawyer, fair, unbiased, impartial',
                    'Significator' => 'Decisions, advice',
                    'Health' => 'OCD',
                    'Love' => 'Fair relationship, legal union',
                    'Work' => 'Decisions from above our pay grade affect our work',
                    'General' => 'Often foretells reliance on decisions from above status or external source.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_24' => [
                'keywords' => 'theft, loss, reduction, negative people',
                'upright' => 'A loss you cannot control; something is taken away.',
                'reversed' => '',
                'short' => 'Something is stolen or removed.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Theft',
                    'Type' => 'Direction (steals from R; affects L)',
                    'Meaning' => 'Something will be stolen, negative people around, all types of losses, reduction',
                    'Person' => 'Thief, pessimist, false, liar, taker, criminal',
                    'Significator' => 'Loss, theft',
                    'Health' => 'Loss of something, removal, spiritual problems, reduction',
                    'Love' => 'Something is not right in the relationship',
                    'Work' => 'Beware of theft in the workplace',
                    'General' => 'Loss the querent cannot control; steals from the card to the right.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_25' => [
                'keywords' => 'message, news, communication, tidings',
                'upright' => 'News and communication of all kinds; context decides tone.',
                'reversed' => '',
                'short' => 'Messages and tidings.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'News',
                    'Type' => 'Direction (sent message to the R)',
                    'Meaning' => 'Messages, tidings, news, information, knowledge, understanding; all communication including email/fax/PM/chat/phone calls etc.',
                    'Person' => 'Communicative, experienced, knowledgeable, intuitive',
                    'Significator' => 'Communication',
                    'Timing' => 'Soon / anon',
                    'Health' => 'Results, prescription, diagnosis',
                    'Love' => 'Love letters',
                    'Work' => 'Work communication of all kinds; reporter, middleman',
                    'General' => 'Nature of news depends on surrounding cards; similar to Lenormand Rider as messenger.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_26' => [
                'keywords' => 'fortune, good luck, success, trump',
                'upright' => 'Great fortune; good luck and success.',
                'reversed' => '',
                'short' => 'Trump card: good luck.',
                'long' => $makeLong([
                    'Value' => 'Positive (Trump Card)',
                    'Core' => 'Great fortune',
                    'Type' => 'Direction (shines well on all cards; more on R)',
                    'Meaning' => 'This card trumps all cards around and shines positive influence even over negative cards. Good luck in all endeavours. Success.',
                    'Person' => 'Happy, optimistic, successful',
                    'Significator' => 'Good luck — what will go well',
                    'Timing' => 'Soon',
                    'Health' => 'Total rejuvenation, good health',
                    'Love' => 'Partner for life or great fortune finding a partner',
                    'Work' => 'Good luck in the work arena',
                    'General' => 'Every deck has a trump card; this one brightens the surrounding cards.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_27' => [
                'keywords' => 'small money, improvements, balance, steady',
                'upright' => 'Small improvements and steady gains.',
                'reversed' => '',
                'short' => 'Slow and sure improvement.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Small improvements',
                    'Type' => 'Stop',
                    'Meaning' => 'Small financial improvements, household finances, small change, money',
                    'Person' => 'Frugal, housewife/husband, bookkeeper (not accountant), neat',
                    'Significator' => 'Household finances, day to day finances',
                    'Timing' => 'Slow & sure',
                    'Health' => 'Inner wellbeing, balanced',
                    'Love' => 'Benefit financially from a loved one',
                    'Work' => 'Salary increase, bonus',
                    'General' => 'Doesn’t have to be financial; look around it for where improvement occurs.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_28' => [
                'keywords' => 'desire, yearning, expectation, patience',
                'upright' => 'Longing and expectation; waiting and hoping.',
                'reversed' => '',
                'short' => 'Yearning and patience.',
                'long' => $makeLong([
                    'Value' => 'Neutral',
                    'Core' => 'Desire',
                    'Type' => 'Direction (L shows what to expect within ~3 months)',
                    'Meaning' => 'What you long for or desire, yearning for the future, expectation',
                    'Person' => 'Patient, dreamy, away with the fairies, alone',
                    'Significator' => 'What your querent wishes for',
                    'Timing' => '3 months approx.',
                    'Health' => 'Psychological illnesses',
                    'Love' => 'Yearn for love, long for your partner',
                    'Work' => 'Long for a new job or better environment',
                    'General' => 'Dreams and expectations; can show what to expect in a situation.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_29' => [
                'keywords' => 'misfortune, difficulties, trouble, danger, accident',
                'upright' => 'Difficulties and trouble; a warning to prepare.',
                'reversed' => '',
                'short' => 'Difficulties ahead.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Misfortune',
                    'Type' => 'Direction (L/above show cause; R/below show how to save self)',
                    'Meaning' => 'Difficulties, trouble, disaster, adversity, accident, danger, ruin, loss',
                    'Person' => 'Awkward, clumsy, desperate, frantic, erratic',
                    'Significator' => 'Misfortune, difficulties',
                    'Health' => 'Accident, injuries',
                    'Love' => 'Bad luck in love, unhappy love',
                    'Work' => 'Dismissal, bullying, misfortune at work',
                    'General' => 'Ranges from accident to inconvenience; warning of difficulties to come.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_30' => [
                'keywords' => 'jealousy, envy, distrust, rival',
                'upright' => 'Jealousy and lack of trust; warning about envy.',
                'reversed' => '',
                'short' => 'Jealousy and distrust.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Jealousy',
                    'Type' => 'Connector',
                    'Meaning' => 'Jealousy, lacking trust, somebody is jealous, envy',
                    'Person' => 'Resentful, jealous, envious, rival',
                    'Significator' => 'Where trust is lacking / what jealousy',
                    'Health' => 'Jealousy making you ill, low self-esteem',
                    'Love' => 'Be careful how you judge your partner',
                    'Work' => 'Jealousy in the workplace or bullying',
                    'General' => 'Warning of envious people; can reflect our own insecurity.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_31' => [
                'keywords' => 'illness, burden, weakness, rest, bed',
                'upright' => 'Illness or burden; time out and rest.',
                'reversed' => '',
                'short' => 'A burden or illness.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Illness',
                    'Type' => 'Direction (L shows malady/down patch)',
                    'Meaning' => 'Illness, burden, malady, bed, bedroom, shows where problems lie, weakness, time out, well-deserved rest.',
                    'Person' => 'Weak, delicate, ill, pale, needs care',
                    'Significator' => 'Problems; find the cause; an illness',
                    'Timing' => 'Within 1 year',
                    'Health' => 'Illness in general',
                    'Love' => 'A down patch or the bed',
                    'Work' => 'Work-related illness, time-out at work',
                    'General' => 'A condition; can be emotional; also bed/bedroom.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_32' => [
                'keywords' => 'anger, vexation, annoyance, strife, trouble',
                'upright' => 'Annoyance and strife; cause-and-effect conflict.',
                'reversed' => '',
                'short' => 'Vexation and arguments.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Annoyance',
                    'Type' => 'Connector (cause & effect)',
                    'Meaning' => 'Aggression, vexation, annoyance, strife, lack of patience, temper, trouble, disagreement, hindrances, turbulent situations',
                    'Person' => 'Grumpy, argumentative, sullen, aggressive, cross',
                    'Significator' => 'Trouble, strife, hindrances',
                    'Health' => 'Anger issues, stomach issues, sleep issues',
                    'Love' => 'Trouble and strife in love',
                    'Work' => 'Arguments / disagreements at work',
                    'General' => '“Verdruss” = vexation; warning to adjust our reaction and approach.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_33' => [
                'keywords' => 'enemy, warning, negative influence, obstacle',
                'upright' => 'Warning: something works against you; bad influence.',
                'reversed' => '',
                'short' => 'A warning and obstacle.',
                'long' => $makeLong([
                    'Value' => 'Negative',
                    'Core' => 'Warning',
                    'Type' => 'Direction (warning of card to R)',
                    'Meaning' => 'Warning! Do not underestimate the situation! Somebody or something is working against you. Negative energies around. Bad influence.',
                    'Person' => 'Negative, dishonest, unfriendly, bad attitude, dark in nature, dominant',
                    'Significator' => 'Warning — look for trouble',
                    'Timing' => 'Sudden',
                    'Health' => 'Foreign bodies; negative thinking leading to illness',
                    'Love' => 'Warning of enemies or negative influences from others',
                    'Work' => 'Enemies in the workplace',
                    'General' => 'Real-life enemy or obstacle; can be external or self-made; blocks your way.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_34' => [
                'keywords' => 'constancy, stability, patience, persistence, work',
                'upright' => 'Long-lasting stability; persistence and patience win.',
                'reversed' => '',
                'short' => 'Stable “yes” card.',
                'long' => $makeLong([
                    'Value' => 'Positive (long lasting)',
                    'Core' => 'Constancy',
                    'Type' => 'Stop',
                    'Meaning' => 'Positively long lasting, stable, constant, persistence wins the day, you’re on the right track, be patient, focus on the goal, stable work life, working towards goals',
                    'Person' => 'Stable, accountable, hardworking, patient',
                    'Significator' => 'Workplace/job; long-term prospect',
                    'Timing' => '1–2 years (future card)',
                    'Health' => 'Overcome worries or long-lasting illness with a good outcome',
                    'Love' => 'Stable & trustworthy',
                    'Work' => 'Good outcome at work; stable environment',
                    'General' => 'Certain answer / Yes card; slow and steady stability; also the work card.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_35' => [
                'keywords' => 'fidelity, loyalty, true friends, trust',
                'upright' => 'Loyalty and faithful support; true friends.',
                'reversed' => '',
                'short' => 'Loyalty and support.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Loyalty',
                    'Type' => 'Direction',
                    'Meaning' => 'Loyalty, true friends, fidelity, faithful, trustworthy',
                    'Person' => 'Loyal, true, long term, a good person, honest, sincere',
                    'Significator' => 'Loyalty, friendships, pets',
                    'Timing' => 'A long time',
                    'Health' => 'Chronic pain, long term illness',
                    'Love' => 'Faithful love',
                    'Work' => 'Loyal colleagues, long term employment',
                    'General' => 'Long-lasting friendship and loyalty; reassess friendships if far from querent card.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
            'kipper_36' => [
                'keywords' => 'hope, goals, dreams, future, spirituality',
                'upright' => 'Hope and possibility; goals and expectations.',
                'reversed' => '',
                'short' => 'Hope made possible.',
                'long' => $makeLong([
                    'Value' => 'Positive',
                    'Core' => 'Hope',
                    'Type' => 'Direction',
                    'Meaning' => 'Hopes & dreams, goals & expectations, what you want out of life, abroad, future, spirituality, long distance travel, addiction',
                    'Person' => 'Positive, open, future-orientated, well-travelled, profound & deep',
                    'Significator' => 'Hopes & dreams',
                    'Health' => 'Hope for recuperation, addiction',
                    'Love' => 'A new love or return of an old love',
                    'Work' => 'Positive change or improvements in the workplace',
                    'General' => 'Different from Desire: Hope is the opportunity to make the goal happen.',
                ]),
                'correspondences' => '',
                'symbols' => '',
            ],
        ];
    }
}
