<?php
declare(strict_types=1);


namespace DailyTarot\Meaning;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Admin\MeaningPackMeta;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;

final class DefaultMeaningPack {
    public const META_IS_DEFAULT = '_dtarot_is_default_pack';

    /**
     * Creates the default pack if missing.
     *
     * @return int Post ID of the default meaning pack, or 0 if creation failed.
     */
    public static function ensureExists(): int {
        $existing = self::getId();
        if ($existing > 0) {
            self::maybeRename($existing);
            self::seed($existing);
            return $existing;
        }

        $postId = wp_insert_post([
            'post_type' => PostTypes::MEANING_PACK,
            'post_status' => 'publish',
            'post_title' => __('Tarot - Basics','daily-tarot'),
        ], true);

        if (is_wp_error($postId) || !$postId) {
            return 0;
        }

        update_post_meta((int)$postId, self::META_IS_DEFAULT, '1');
        update_post_meta((int)$postId, MeaningPackMeta::META_SYSTEM, Cards::SYSTEM_TAROT);
        self::seed((int)$postId);

        return (int)$postId;
    }

    /** @return int */
    public static function getId(): int {
        $id = apply_filters('dtarot_default_meaning_pack_id', 0);
        if (is_int($id) && $id > 0) return $id;
        if (is_string($id) && ctype_digit($id) && (int)$id > 0) return (int)$id;

        $q = new \WP_Query([
            'post_type' => PostTypes::meaningPackTypes(),
            'post_status' => ['publish','draft'],
            'posts_per_page' => 1,
            'fields' => 'ids',
            'meta_key' => self::META_IS_DEFAULT, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- 1-row lookup.
            'meta_value' => '1', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- 1-row lookup.
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ]);

        if (!empty($q->posts[0])) {
            return (int)$q->posts[0];
        }

        return 0;
    }

    public static function isDefault(int $packId): bool {
        if ($packId <= 0) return false;
        return get_post_meta($packId, self::META_IS_DEFAULT, true) === '1';
    }

    private static function maybeRename(int $packId): void {
        if ($packId <= 0) return;
        $title = get_the_title($packId);
        if (!is_string($title)) $title = '';
        if ($title === 'Tarot - Basics') return;

        wp_update_post([
            'ID' => $packId,
            'post_title' => __('Tarot - Basics','daily-tarot'),
        ]);
    }

    private static function seed(int $packId): void {
        $existing = get_post_meta($packId, MeaningPackMeta::META_KEY, true);
        $seed = self::data();
        if (is_array($existing) && !empty($existing)) {
            // Do not overwrite user content; only fill missing cards.
            $merged = $existing;
            foreach ($seed as $cardId => $meaning) {
                if (!isset($merged[$cardId]) || !is_array($merged[$cardId])) {
                    $merged[$cardId] = $meaning;
                }
            }
            update_post_meta($packId, MeaningPackMeta::META_KEY, $merged);
            return;
        }

        update_post_meta($packId, MeaningPackMeta::META_KEY, $seed);
    }

    /**
     * Starter meanings + symbols + correspondences.
     *
     * Notes:
     * - This content is original and intentionally concise.
     * - Correspondences are common Golden Dawn style attributions and may vary by tradition.
     *
     * @return array<string,array<string,string>>
     */
    private static function data(): array {
        return [
            'tarot_major_0' => [
                'keywords' => 'beginnings, innocence, leap of faith, freedom, curiosity',
                'upright' => 'A fresh start invites trust in the journey and learning by doing.',
                'reversed' => 'Hesitation or recklessness calls for grounding before the next step.',
                'short' => 'New path, open heart, mindful risk.',
                'long' => '<p>The Fool opens a new chapter and asks for presence and openness. Choose the option that expands your world while staying aware of basics.</p><p>Reversed, slow down, check essentials, and avoid rushing into avoidable pitfalls.</p>',
                'correspondences' => 'Element: Air; Planet: Uranus (often); Path: 11 (Hermetic/Qabalah).',
                'symbols' => 'cliff edge, small bundle, white rose, loyal companion, bright sun',
            ],
            'tarot_major_1' => [
                'keywords' => 'will, skill, manifestation, focus, initiative',
                'upright' => 'Your attention is power; use your tools with intention and clarity.',
                'reversed' => 'Scattered focus or manipulation warns against shortcuts.',
                'short' => 'Direct your will with clear focus.',
                'long' => '<p>The Magician highlights agency. You already have what you need if you gather it and commit.</p><p>Reversed, simplify, align your motives, and rebuild trust in your process.</p>',
                'correspondences' => 'Planet: Mercury; Element: Air; Path: 12 (Hermetic/Qabalah).',
                'symbols' => 'raised wand, pointing hand, infinity sign, table of tools, garden',
            ],
            'tarot_major_2' => [
                'keywords' => 'intuition, inner knowing, mystery, patience, receptivity',
                'upright' => 'Listen beneath the noise; your instincts already understand.',
                'reversed' => 'Inner noise or secrecy suggests checking assumptions.',
                'short' => 'Quiet intuition guides the next step.',
                'long' => '<p>The High Priestess emphasizes what is sensed rather than spoken. You may not have full proof yet, but you have a feeling worth respecting.</p><p>Reversed, seek clarity, ask direct questions, and avoid acting on half truths.</p>',
                'correspondences' => 'Planet: Moon; Element: Water; Path: 13 (Hermetic/Qabalah).',
                'symbols' => 'veil, pillars, scroll, crescent moon, pomegranates',
            ],
            'tarot_major_3' => [
                'keywords' => 'abundance, creativity, nurture, comfort, growth',
                'upright' => 'Care for what you want to grow; attention and warmth create results.',
                'reversed' => 'Overgiving or stagnation calls for balance and boundaries.',
                'short' => 'Nurture brings abundance and ease.',
                'long' => '<p>The Empress supports creative momentum and embodied well being. Feed the project, relationship, or habit that feeds you back.</p><p>Reversed, restore balance, prioritize self care, and avoid smothering what you love.</p>',
                'correspondences' => 'Planet: Venus; Zodiac: Taurus or Libra; Path: 14 (Hermetic/Qabalah).',
                'symbols' => 'crown of stars, wheat, flowing water, lush garden, heart shield',
            ],
            'tarot_major_4' => [
                'keywords' => 'structure, leadership, boundaries, stability, responsibility',
                'upright' => 'Make a firm decision and protect your priorities with clear boundaries.',
                'reversed' => 'Rigidity or avoidance of responsibility blocks progress.',
                'short' => 'Order and boundaries create stability.',
                'long' => '<p>The Emperor calls for ownership: define the rules, the timeline, and the next action. This is about reliability, not control.</p><p>Reversed, loosen a rigid stance or step up where you have been avoiding duty.</p>',
                'correspondences' => 'Zodiac: Aries; Planet: Mars (via Aries); Path: 15 (Hermetic/Qabalah).',
                'symbols' => 'stone throne, armor, ram motifs, scepter, mountain',
            ],
            'tarot_major_5' => [
                'keywords' => 'tradition, learning, guidance, belief, shared values',
                'upright' => 'Seek solid teaching and align your choices with your true values.',
                'reversed' => 'Outdated rules or rebellion without purpose signal a reset.',
                'short' => 'Learn from what has been tested.',
                'long' => '<p>The Hierophant encourages study, mentorship, and ethical alignment. A trusted framework can help you move forward without reinventing everything.</p><p>Reversed, question dogma, keep what works, and release what no longer fits.</p>',
                'correspondences' => 'Zodiac: Taurus; Element: Earth; Path: 16 (Hermetic/Qabalah).',
                'symbols' => 'keys, pillars, blessing hand, acolytes, triple cross',
            ],
            'tarot_major_6' => [
                'keywords' => 'choice, union, alignment, attraction, honesty',
                'upright' => 'Choose what matches your heart and your integrity, not just what is easy.',
                'reversed' => 'Misalignment or mixed signals call for honest decisions.',
                'short' => 'Alignment over convenience.',
                'long' => '<p>The Lovers is about values based choice. Whether in romance, work, or self care, the right option is the one you can respect tomorrow.</p><p>Reversed, clarify priorities, repair trust, or choose a new path.</p>',
                'correspondences' => 'Zodiac: Gemini; Element: Air; Path: 17 (Hermetic/Qabalah).',
                'symbols' => 'angel, garden, mountain, sun, paired figures',
            ],
            'tarot_major_7' => [
                'keywords' => 'drive, discipline, victory, focus, direction',
                'upright' => 'Momentum comes from steering your energy; pick a direction and commit.',
                'reversed' => 'Scattered effort or stubbornness slows progress.',
                'short' => 'Choose a direction, then move.',
                'long' => '<p>The Chariot rewards disciplined motion. Conflicting impulses can be harnessed when you decide what matters most.</p><p>Reversed, slow the pace, remove distractions, and regain control.</p>',
                'correspondences' => 'Zodiac: Cancer; Element: Water; Path: 18 (Hermetic/Qabalah).',
                'symbols' => 'chariot, canopy of stars, opposing forces, armor, reins',
            ],
            'tarot_major_8' => [
                'keywords' => 'courage, compassion, calm power, patience, resilience',
                'upright' => 'Strength is gentle consistency; meet the day with steady courage.',
                'reversed' => 'Self doubt or volatility calls for compassion and regulation.',
                'short' => 'Soft power wins today.',
                'long' => '<p>Strength asks you to lead yourself with kindness. Bravery is the ability to stay present and choose your response.</p><p>Reversed, rebuild confidence through small wins and calmer self talk.</p>',
                'correspondences' => 'Zodiac: Leo; Element: Fire; Path: 19 (Hermetic/Qabalah).',
                'symbols' => 'lion, infinity sign, calm figure, open hands, white garments',
            ],
            'tarot_major_9' => [
                'keywords' => 'reflection, wisdom, solitude, guidance, discernment',
                'upright' => 'Step back to see clearly; your next move becomes obvious in quiet.',
                'reversed' => 'Isolation or avoidance suggests reconnecting with support.',
                'short' => 'Quiet reveals the path.',
                'long' => '<p>The Hermit supports reflection and careful discernment. Not everything needs a fast answer today.</p><p>Reversed, step out of the cave, seek feedback, and rejoin the flow.</p>',
                'correspondences' => 'Zodiac: Virgo; Element: Earth; Path: 20 (Hermetic/Qabalah).',
                'symbols' => 'lantern, staff, mountain, cloak, star/light',
            ],
            'tarot_major_10' => [
                'keywords' => 'change, cycles, luck, turning point, timing',
                'upright' => 'A shift is underway; adapt quickly and let timing work in your favor.',
                'reversed' => 'Resistance to change or bad timing suggests patience.',
                'short' => 'A turning point: stay flexible.',
                'long' => '<p>The Wheel of Fortune reminds you that seasons change. What is stuck can move, and what is easy can require attention.</p><p>Reversed, focus on what you can control and ride out the cycle.</p>',
                'correspondences' => 'Planet: Jupiter; Element: Fire; Path: 21 (Hermetic/Qabalah).',
                'symbols' => 'wheel, four figures, rising/falling creatures, clouds, letters',
            ],
            'tarot_major_11' => [
                'keywords' => 'fairness, truth, balance, accountability, clarity',
                'upright' => 'Be honest with yourself and others; clear truth leads to clean outcomes.',
                'reversed' => 'Bias or avoidance delays resolution.',
                'short' => 'Truth and balance restore order.',
                'long' => '<p>Justice asks for clear eyed accountability. Choices have consequences, and you can correct course with integrity.</p><p>Reversed, check for blind spots and restore fairness.</p>',
                'correspondences' => 'Zodiac: Libra; Element: Air; Path: 22 (Hermetic/Qabalah).',
                'symbols' => 'scales, sword, pillars, square seat, calm gaze',
            ],
            'tarot_major_12' => [
                'keywords' => 'pause, surrender, new perspective, sacrifice, reframing',
                'upright' => 'Stop forcing the outcome; shift perspective and let insight arrive.',
                'reversed' => 'Stalling or martyrdom signals time to move.',
                'short' => 'Pause. Reframe. Then proceed.',
                'long' => '<p>The Hanged Man invites a deliberate pause. When you step out of urgency, you see what you could not see before.</p><p>Reversed, make a clear choice and release a stale sacrifice.</p>',
                'correspondences' => 'Element: Water; Planet: Neptune (modern); Path: 23 (Hermetic/Qabalah).',
                'symbols' => 'hanging figure, halo, crossed leg, rope, calm expression',
            ],
            'tarot_major_13' => [
                'keywords' => 'transformation, endings, renewal, release, rebirth',
                'upright' => 'Release what is complete; space opens for the next phase to begin.',
                'reversed' => 'Resistance or fear of change slows renewal.',
                'short' => 'Let go to level up.',
                'long' => '<p>Death is the art of clean endings. Something is ready to change, and resisting only prolongs discomfort.</p><p>Reversed, allow a smaller release to unlock movement.</p>',
                'correspondences' => 'Zodiac: Scorpio; Element: Water; Path: 24 (Hermetic/Qabalah).',
                'symbols' => 'skeletal rider, banner/rose, dawn, fallen crown, river',
            ],
            'tarot_major_14' => [
                'keywords' => 'balance, moderation, healing, harmony, integration',
                'upright' => 'Small steady adjustments bring you back into balance.',
                'reversed' => 'Imbalance or extremes call for recalibration.',
                'short' => 'Balance is built, not found.',
                'long' => '<p>Temperance supports healing through moderation and integration. Mix patience with action; blend intuition with logic.</p><p>Reversed, remove excess and restore a sustainable rhythm.</p>',
                'correspondences' => 'Zodiac: Sagittarius; Element: Fire; Path: 25 (Hermetic/Qabalah).',
                'symbols' => 'angel, two cups, flowing water, one foot on land, iris',
            ],
            'tarot_major_15' => [
                'keywords' => 'attachment, temptation, shadow, desire, material focus',
                'upright' => 'Notice what has a grip on you; freedom starts with honest awareness.',
                'reversed' => 'Release or recovery is possible with clear choices.',
                'short' => 'Name the attachment; reclaim choice.',
                'long' => '<p>The Devil highlights binding patterns: habits, fears, or pleasures that quietly run the show.</p><p>Reversed, cut one tie, set one boundary, and reclaim autonomy.</p>',
                'correspondences' => 'Zodiac: Capricorn; Element: Earth; Path: 26 (Hermetic/Qabalah).',
                'symbols' => 'chains, pedestal, horned figure, torch, mirrored pair',
            ],
            'tarot_major_16' => [
                'keywords' => 'shock, liberation, truth, disruption, breakthrough',
                'upright' => 'What falls away makes room for what is real; rebuild on truth.',
                'reversed' => 'A delayed shake up or inner break through is arriving.',
                'short' => 'A truth break opens a new path.',
                'long' => '<p>The Tower clears what cannot last. While it may feel sudden, the outcome is honesty and freedom.</p><p>Reversed, prepare for change and release weak foundations.</p>',
                'correspondences' => 'Planet: Mars; Element: Fire; Path: 27 (Hermetic/Qabalah).',
                'symbols' => 'lightning, crown falling, crumbling tower, figures descending, flames',
            ],
            'tarot_major_17' => [
                'keywords' => 'hope, renewal, guidance, inspiration, calm',
                'upright' => 'Restore hope; gentle consistency brings your vision closer.',
                'reversed' => 'Doubt or fatigue asks for rest and honest care.',
                'short' => 'Hope returns when you keep going.',
                'long' => '<p>The Star is a quiet promise: healing is possible, and direction returns. Choose routines that nourish you.</p><p>Reversed, reconnect with a simple practice and rebuild faith step by step.</p>',
                'correspondences' => 'Zodiac: Aquarius; Element: Air; Path: 28 (Hermetic/Qabalah).',
                'symbols' => 'star, water jars, pool/stream, kneeling figure, bird',
            ],
            'tarot_major_18' => [
                'keywords' => 'dreams, uncertainty, intuition, illusion, sensitivity',
                'upright' => 'Not everything is clear yet; move gently and trust your instincts.',
                'reversed' => 'Confusion lifts as facts come into view.',
                'short' => 'Walk slowly through uncertainty.',
                'long' => '<p>The Moon asks you to navigate ambiguity with care. Feelings are real data, but they still require interpretation.</p><p>Reversed, verify information and choose clarity over fear.</p>',
                'correspondences' => 'Zodiac: Pisces (modern) or Moon; Element: Water; Path: 29 (Hermetic/Qabalah).',
                'symbols' => 'moon, towers, path, dog and wolf, crustacean',
            ],
            'tarot_major_19' => [
                'keywords' => 'joy, clarity, success, vitality, confidence',
                'upright' => 'Choose openness; what is true becomes easier, and energy returns.',
                'reversed' => 'Clouds or delay ask for patience and perspective.',
                'short' => 'Clarity and joy are available.',
                'long' => '<p>The Sun brings warmth and visibility. Share your work, speak plainly, and celebrate what is going right.</p><p>Reversed, seek simple joy and do not hide your light.</p>',
                'correspondences' => 'Planet: Sun; Element: Fire; Path: 30 (Hermetic/Qabalah).',
                'symbols' => 'sun, child, horse, sunflowers, red banner',
            ],
            'tarot_major_20' => [
                'keywords' => 'awakening, calling, forgiveness, renewal, evaluation',
                'upright' => 'Answer the call; make peace with the past and step into what is next.',
                'reversed' => 'Self doubt or avoidance delays a needed decision.',
                'short' => 'A wake up moment: choose renewal.',
                'long' => '<p>Judgement is about honest review and fresh commitment. You can learn from the past without living in it.</p><p>Reversed, forgive yourself, correct course, and recommit.</p>',
                'correspondences' => 'Element: Fire; Planet: Pluto (modern); Path: 31 (Hermetic/Qabalah).',
                'symbols' => 'trumpet, rising figures, clouds, flag/banner, ocean',
            ],
            'tarot_major_21' => [
                'keywords' => 'completion, integration, mastery, wholeness, arrival',
                'upright' => 'You are closing a cycle; honor what you built and integrate the lesson.',
                'reversed' => 'Loose ends or delays ask for final adjustments.',
                'short' => 'Completion and integration.',
                'long' => '<p>The World celebrates completion and coherence. A project, phase, or identity is coming together in a meaningful way.</p><p>Reversed, finish the last details and prepare to step forward.</p>',
                'correspondences' => 'Planet: Saturn or Element: Earth; Path: 32 (Hermetic/Qabalah).',
                'symbols' => 'laurel wreath, four living beings, dance/figure, wands, ribbon',
            ],
            'tarot_wands_01' => [
                'keywords' => 'spark, inspiration, initiative, desire, growth',
                'upright' => 'A new spark arrives; take the first bold step.',
                'reversed' => 'A delay or lack of drive asks for rest and clarity.',
                'short' => 'A spark of action and desire.',
                'long' => '<p>The Ace of Wands lights the match for a new project, passion, or direction. Start small, but start now.</p><p>Reversed, energy is low or scattered, so simplify and rekindle the flame.</p>',
                'correspondences' => 'Element: Fire; Number: 1 (ace).',
                'symbols' => 'sprouting wand, hand from cloud, leaves, landscape',
            ],
            'tarot_wands_02' => [
                'keywords' => 'planning, vision, decisions, progress, horizons',
                'upright' => 'Look ahead and choose the path with the widest growth.',
                'reversed' => 'Fear of change or narrow focus limits progress.',
                'short' => 'Plan the journey and commit.',
                'long' => '<p>The Two of Wands is a moment of strategy and choice. Your next move shapes the horizon.</p><p>Reversed, you may be overthinking or holding back. Decide and act.</p>',
                'correspondences' => 'Element: Fire; Number: 2.',
                'symbols' => 'globe, wall, two wands, distant landscape',
            ],
            'tarot_wands_03' => [
                'keywords' => 'momentum, expansion, foresight, trade, progress',
                'upright' => 'Momentum builds; keep your long range vision steady.',
                'reversed' => 'Delays or scattered effort suggest recalibration.',
                'short' => 'Expansion is underway.',
                'long' => '<p>The Three of Wands signals growth through planning and outreach. You are not stuck; you are building.</p><p>Reversed, adjust the plan and stay consistent with your direction.</p>',
                'correspondences' => 'Element: Fire; Number: 3.',
                'symbols' => 'shoreline, ships, three wands, lookout',
            ],
            'tarot_wands_04' => [
                'keywords' => 'celebration, stability, home, community, milestone',
                'upright' => 'Celebrate progress and enjoy supportive connections.',
                'reversed' => 'Unstable foundations or tension at home need care.',
                'short' => 'A milestone worth honoring.',
                'long' => '<p>The Four of Wands highlights joy, ceremony, and a stable base. Share the win.</p><p>Reversed, resolve conflict at home or rebuild the structure that holds you.</p>',
                'correspondences' => 'Element: Fire; Number: 4.',
                'symbols' => 'garland, four wands, home, dancing figures',
            ],
            'tarot_wands_05' => [
                'keywords' => 'conflict, competition, tension, friction, growth',
                'upright' => 'Healthy challenge sharpens skill; stay focused and fair.',
                'reversed' => 'Avoidable conflict or inner chaos needs calm.',
                'short' => 'Friction that builds strength.',
                'long' => '<p>The Five of Wands reflects lively competition or mixed agendas. Use it to learn and improve.</p><p>Reversed, reduce drama, clarify roles, and restore cooperation.</p>',
                'correspondences' => 'Element: Fire; Number: 5.',
                'symbols' => 'five wands, sparring figures, open field',
            ],
            'tarot_wands_06' => [
                'keywords' => 'recognition, victory, confidence, progress, pride',
                'upright' => 'A win is visible; accept recognition with humility.',
                'reversed' => 'Ego or delayed praise suggests patience and focus.',
                'short' => 'Success and public support.',
                'long' => '<p>The Six of Wands brings a clear win and supportive attention. Keep leading with integrity.</p><p>Reversed, refocus on your work and let results speak.</p>',
                'correspondences' => 'Element: Fire; Number: 6.',
                'symbols' => 'laurel wreath, horse, staff, crowd',
            ],
            'tarot_wands_07' => [
                'keywords' => 'defense, courage, standing ground, challenge, resilience',
                'upright' => 'Hold your ground; your position is worth protecting.',
                'reversed' => 'Fatigue or overwhelm suggests regrouping.',
                'short' => 'Stand firm under pressure.',
                'long' => '<p>The Seven of Wands calls for courage and clear boundaries. You can handle this challenge.</p><p>Reversed, simplify commitments and get support.</p>',
                'correspondences' => 'Element: Fire; Number: 7.',
                'symbols' => 'high ground, staff, opposing wands, stance',
            ],
            'tarot_wands_08' => [
                'keywords' => 'speed, movement, messages, alignment, flow',
                'upright' => 'Things move fast; act while the path is open.',
                'reversed' => 'Delays or mixed signals require patience.',
                'short' => 'Momentum and swift progress.',
                'long' => '<p>The Eight of Wands signals quick movement and clear communication. Stay ready and responsive.</p><p>Reversed, slow down and confirm details before moving.</p>',
                'correspondences' => 'Element: Fire; Number: 8.',
                'symbols' => 'eight wands in flight, open sky, distant hills',
            ],
            'tarot_wands_09' => [
                'keywords' => 'persistence, boundaries, resilience, endurance, caution',
                'upright' => 'You are close to the finish; protect your energy.',
                'reversed' => 'Exhaustion or mistrust calls for rest.',
                'short' => 'Keep going, protect your limits.',
                'long' => '<p>The Nine of Wands is a sign of perseverance and cautious strength. You have come far.</p><p>Reversed, do not fight alone. Recover and reinforce support.</p>',
                'correspondences' => 'Element: Fire; Number: 9.',
                'symbols' => 'bandaged figure, nine wands, guarded stance',
            ],
            'tarot_wands_10' => [
                'keywords' => 'burden, responsibility, pressure, effort, completion',
                'upright' => 'A heavy load can be carried, but it must be shared or simplified.',
                'reversed' => 'Overload or avoidance signals a need to delegate.',
                'short' => 'Heavy load, near the end.',
                'long' => '<p>The Ten of Wands shows a final push with too much on your back. Prioritize and trim.</p><p>Reversed, release what is not yours and ask for help.</p>',
                'correspondences' => 'Element: Fire; Number: 10.',
                'symbols' => 'stacked wands, bent posture, distant home',
            ],
            'tarot_wands_11' => [
                'keywords' => 'curiosity, exploration, enthusiasm, messages, risk',
                'upright' => 'A fresh idea arrives; explore with playful confidence.',
                'reversed' => 'Impulsiveness or delays ask for maturity.',
                'short' => 'Curious and bold beginnings.',
                'long' => '<p>The Page of Wands brings a message of inspiration and a willingness to experiment. Say yes to learning.</p><p>Reversed, slow down, plan, and avoid empty promises.</p>',
                'correspondences' => 'Element: Fire of Earth (Page); Suit: Fire.',
                'symbols' => 'wand, desert, youthful figure, salamanders',
            ],
            'tarot_wands_12' => [
                'keywords' => 'action, adventure, courage, passion, drive',
                'upright' => 'Act decisively and follow the call of adventure.',
                'reversed' => 'Recklessness or burnout suggests pacing.',
                'short' => 'Bold action and forward motion.',
                'long' => '<p>The Knight of Wands charges forward with confidence and ambition. Use momentum wisely.</p><p>Reversed, avoid haste, temper impulsive moves, and conserve energy.</p>',
                'correspondences' => 'Element: Fire of Air (Knight); Suit: Fire.',
                'symbols' => 'rearing horse, wand, flames, desert',
            ],
            'tarot_wands_13' => [
                'keywords' => 'confidence, warmth, creativity, leadership, charisma',
                'upright' => 'Lead with warmth and creative confidence.',
                'reversed' => 'Insecurity or jealousy can distort your impact.',
                'short' => 'Warm leadership and creativity.',
                'long' => '<p>The Queen of Wands shines with courage and generous spirit. Share your vision and invite others in.</p><p>Reversed, rebuild confidence and avoid controlling behavior.</p>',
                'correspondences' => 'Element: Fire of Water (Queen); Suit: Fire.',
                'symbols' => 'sunflower, throne, cat, wand',
            ],
            'tarot_wands_14' => [
                'keywords' => 'vision, mastery, integrity, leadership, inspiration',
                'upright' => 'Lead with purpose and steady conviction.',
                'reversed' => 'Dominance or scattered priorities call for humility.',
                'short' => 'Commanding leadership and vision.',
                'long' => '<p>The King of Wands turns inspiration into action and strategy. Step up and guide with integrity.</p><p>Reversed, focus on one direction and avoid power struggles.</p>',
                'correspondences' => 'Element: Fire of Fire (King); Suit: Fire.',
                'symbols' => 'throne, wand, lion motifs, salamanders',
            ],
            'tarot_cups_01' => [
                'keywords' => 'new feelings, love, intuition, openness, healing',
                'upright' => 'An emotional opening invites connection and renewal.',
                'reversed' => 'Blocked feelings suggest gentle self care.',
                'short' => 'New heart energy and openness.',
                'long' => '<p>The Ace of Cups offers a fresh wave of emotion, compassion, or creativity. Allow yourself to receive.</p><p>Reversed, protect your heart and tend to what needs healing.</p>',
                'correspondences' => 'Element: Water; Number: 1 (ace).',
                'symbols' => 'overflowing cup, dove, lotus, water',
            ],
            'tarot_cups_02' => [
                'keywords' => 'partnership, harmony, mutual respect, attraction, unity',
                'upright' => 'Mutual respect and honest exchange create harmony.',
                'reversed' => 'Imbalance or miscommunication needs repair.',
                'short' => 'A balanced and equal bond.',
                'long' => '<p>The Two of Cups supports partnership, shared values, and sincere connection. Show up fully.</p><p>Reversed, address misalignment and rebuild trust.</p>',
                'correspondences' => 'Element: Water; Number: 2.',
                'symbols' => 'two cups, caduceus, lion, shared gaze',
            ],
            'tarot_cups_03' => [
                'keywords' => 'celebration, friendship, community, joy, support',
                'upright' => 'Celebrate with those who support you.',
                'reversed' => 'Overindulgence or isolation calls for balance.',
                'short' => 'Joy shared with community.',
                'long' => '<p>The Three of Cups highlights friendship and shared joy. Lean into supportive circles.</p><p>Reversed, avoid excess and reconnect with what feels nourishing.</p>',
                'correspondences' => 'Element: Water; Number: 3.',
                'symbols' => 'three cups raised, dancing figures, fruit',
            ],
            'tarot_cups_04' => [
                'keywords' => 'apathy, contemplation, dissatisfaction, pause, reevaluation',
                'upright' => 'Pause and reevaluate what actually fulfills you.',
                'reversed' => 'A new offer or interest breaks the lull.',
                'short' => 'Contemplation before choosing.',
                'long' => '<p>The Four of Cups signals a quiet pause and a need to reassess. Notice what you are overlooking.</p><p>Reversed, reengage with life and accept a fresh opportunity.</p>',
                'correspondences' => 'Element: Water; Number: 4.',
                'symbols' => 'three cups, offered cup, seated figure, tree',
            ],
            'tarot_cups_05' => [
                'keywords' => 'grief, regret, loss, sadness, acceptance',
                'upright' => 'Honor the loss, but do not ignore what remains.',
                'reversed' => 'Healing and perspective begin to return.',
                'short' => 'Grief with a path forward.',
                'long' => '<p>The Five of Cups acknowledges disappointment and grief. Let the feeling move through you.</p><p>Reversed, recovery starts when you recognize remaining support.</p>',
                'correspondences' => 'Element: Water; Number: 5.',
                'symbols' => 'spilled cups, cloaked figure, bridge, river',
            ],
            'tarot_cups_06' => [
                'keywords' => 'memory, nostalgia, kindness, innocence, reunion',
                'upright' => 'Simple kindness and nostalgia bring comfort.',
                'reversed' => 'Living in the past blocks present growth.',
                'short' => 'Sweet memories and gentle giving.',
                'long' => '<p>The Six of Cups brings warmth, memories, and a tender exchange. Offer kindness without conditions.</p><p>Reversed, release old patterns and focus on today.</p>',
                'correspondences' => 'Element: Water; Number: 6.',
                'symbols' => 'flower cups, children, courtyard, nostalgia',
            ],
            'tarot_cups_07' => [
                'keywords' => 'choices, imagination, illusion, desire, options',
                'upright' => 'Many options appear; choose with discernment.',
                'reversed' => 'Clarity returns as you narrow priorities.',
                'short' => 'Choices and tempting visions.',
                'long' => '<p>The Seven of Cups highlights imagination and many possibilities. Evaluate reality before committing.</p><p>Reversed, simplify and choose the option that is real and sustainable.</p>',
                'correspondences' => 'Element: Water; Number: 7.',
                'symbols' => 'floating cups, clouds, treasures, shadows',
            ],
            'tarot_cups_08' => [
                'keywords' => 'departure, searching, letting go, solitude, transition',
                'upright' => 'Walk away from what no longer feeds you.',
                'reversed' => 'Avoidance or fear of change keeps you stuck.',
                'short' => 'Leave to find deeper meaning.',
                'long' => '<p>The Eight of Cups marks a conscious departure from something incomplete. Seek what is true for you.</p><p>Reversed, clarify your reasons and do not abandon what still matters.</p>',
                'correspondences' => 'Element: Water; Number: 8.',
                'symbols' => 'stacked cups, moon, mountains, departing figure',
            ],
            'tarot_cups_09' => [
                'keywords' => 'contentment, satisfaction, pleasure, gratitude, wishes',
                'upright' => 'Enjoy what you have built and express gratitude.',
                'reversed' => 'Overindulgence or dissatisfaction needs realignment.',
                'short' => 'Contentment and wishes fulfilled.',
                'long' => '<p>The Nine of Cups brings pleasure and emotional satisfaction. Let yourself enjoy the moment.</p><p>Reversed, seek balance and avoid chasing surface comfort.</p>',
                'correspondences' => 'Element: Water; Number: 9.',
                'symbols' => 'row of cups, seated figure, rich fabric',
            ],
            'tarot_cups_10' => [
                'keywords' => 'harmony, family, fulfillment, peace, joy',
                'upright' => 'Emotional fulfillment and harmony are available.',
                'reversed' => 'Tension or unmet needs call for honest conversation.',
                'short' => 'Emotional harmony and shared joy.',
                'long' => '<p>The Ten of Cups speaks of deep connection, peace, and shared joy. Appreciate the bonds you have.</p><p>Reversed, address what is unspoken and rebuild alignment.</p>',
                'correspondences' => 'Element: Water; Number: 10.',
                'symbols' => 'rainbow, family, home, cups in sky',
            ],
            'tarot_cups_11' => [
                'keywords' => 'curiosity, openness, messages, sensitivity, wonder',
                'upright' => 'A heartfelt message or creative idea arrives.',
                'reversed' => 'Moodiness or insecurity asks for grounding.',
                'short' => 'Tender curiosity and new feelings.',
                'long' => '<p>The Page of Cups invites gentle exploration of feelings and creativity. Stay open to a kind surprise.</p><p>Reversed, steady your emotions and avoid mixed signals.</p>',
                'correspondences' => 'Element: Water of Earth (Page); Suit: Water.',
                'symbols' => 'cup with fish, coastal shore, youthful figure',
            ],
            'tarot_cups_12' => [
                'keywords' => 'romance, imagination, charm, pursuit, idealism',
                'upright' => 'Follow the heart with grace and creativity.',
                'reversed' => 'Fantasy or inconsistency clouds judgment.',
                'short' => 'Romantic pursuit and inspiration.',
                'long' => '<p>The Knight of Cups moves with charm and emotional conviction. Act on what you truly feel.</p><p>Reversed, check reality and avoid promises you cannot keep.</p>',
                'correspondences' => 'Element: Water of Air (Knight); Suit: Water.',
                'symbols' => 'horse, cup, river, winged helmet',
            ],
            'tarot_cups_13' => [
                'keywords' => 'empathy, compassion, calm, intuition, support',
                'upright' => 'Offer steady compassion and trust your intuition.',
                'reversed' => 'Emotional overload or avoidance needs boundaries.',
                'short' => 'Compassionate and intuitive care.',
                'long' => '<p>The Queen of Cups leads with empathy and emotional intelligence. Listen deeply.</p><p>Reversed, protect your energy and avoid absorbing what is not yours.</p>',
                'correspondences' => 'Element: Water of Water (Queen); Suit: Water.',
                'symbols' => 'ornate cup, sea, throne, shells',
            ],
            'tarot_cups_14' => [
                'keywords' => 'balance, diplomacy, wisdom, emotional mastery, support',
                'upright' => 'Lead with steady emotional wisdom and fairness.',
                'reversed' => 'Moodiness or detachment suggests recalibration.',
                'short' => 'Emotional mastery and calm leadership.',
                'long' => '<p>The King of Cups combines compassion with clear judgment. Offer guidance that steadies the room.</p><p>Reversed, address suppressed feelings and restore balance.</p>',
                'correspondences' => 'Element: Water of Fire (King); Suit: Water.',
                'symbols' => 'throne on water, cup, scepter, fish',
            ],
            'tarot_swords_01' => [
                'keywords' => 'clarity, truth, insight, decision, breakthrough',
                'upright' => 'A clear truth cuts through confusion.',
                'reversed' => 'Miscommunication or confusion delays action.',
                'short' => 'Clarity and decisive insight.',
                'long' => '<p>The Ace of Swords brings a fresh mental breakthrough. Speak clearly and choose the honest path.</p><p>Reversed, slow down, verify facts, and avoid sharp words.</p>',
                'correspondences' => 'Element: Air; Number: 1 (ace).',
                'symbols' => 'sword, crown, clouds, mountain',
            ],
            'tarot_swords_02' => [
                'keywords' => 'indecision, balance, stalemate, choice, truce',
                'upright' => 'Weigh both sides before making a calm decision.',
                'reversed' => 'Avoidance ends; choose and move forward.',
                'short' => 'Pause and consider the choice.',
                'long' => '<p>The Two of Swords is a temporary pause to gather information. Stay neutral and listen.</p><p>Reversed, break the stalemate with a clear decision.</p>',
                'correspondences' => 'Element: Air; Number: 2.',
                'symbols' => 'blindfold, crossed swords, moon, sea',
            ],
            'tarot_swords_03' => [
                'keywords' => 'heartbreak, sorrow, truth, grief, release',
                'upright' => 'A painful truth asks for honest processing.',
                'reversed' => 'Healing begins as you release the grief.',
                'short' => 'Painful truth and release.',
                'long' => '<p>The Three of Swords acknowledges heartbreak or disappointment. Let yourself feel and name it.</p><p>Reversed, recovery begins with self compassion and support.</p>',
                'correspondences' => 'Element: Air; Number: 3.',
                'symbols' => 'three swords, heart, rain clouds',
            ],
            'tarot_swords_04' => [
                'keywords' => 'rest, recovery, pause, contemplation, retreat',
                'upright' => 'Rest and recovery are required before the next step.',
                'reversed' => 'Restlessness or burnout suggests real downtime.',
                'short' => 'Rest, restore, then move.',
                'long' => '<p>The Four of Swords is a deliberate pause for healing. Step back to regain strength.</p><p>Reversed, slow down before stress forces a stop.</p>',
                'correspondences' => 'Element: Air; Number: 4.',
                'symbols' => 'tomb, stained glass, sword above',
            ],
            'tarot_swords_05' => [
                'keywords' => 'conflict, tension, winning at a cost, ego, discord',
                'upright' => 'Conflict may bring a win, but check the cost.',
                'reversed' => 'Reconciliation or release of conflict is possible.',
                'short' => 'A win with consequences.',
                'long' => '<p>The Five of Swords warns against hollow victories. Choose integrity over ego.</p><p>Reversed, seek peace, apologize, or walk away from a toxic battle.</p>',
                'correspondences' => 'Element: Air; Number: 5.',
                'symbols' => 'scattered swords, wind, tense figures',
            ],
            'tarot_swords_06' => [
                'keywords' => 'transition, recovery, passage, relief, movement',
                'upright' => 'Move toward calmer waters and clearer thinking.',
                'reversed' => 'Resistance to change slows relief.',
                'short' => 'Transition toward peace.',
                'long' => '<p>The Six of Swords marks a necessary transition. Leave turbulence behind and move toward stability.</p><p>Reversed, release resistance and allow help.</p>',
                'correspondences' => 'Element: Air; Number: 6.',
                'symbols' => 'boat, river, cloaked figures, swords',
            ],
            'tarot_swords_07' => [
                'keywords' => 'strategy, secrecy, caution, independence, risk',
                'upright' => 'Use strategy and discretion, but avoid deception.',
                'reversed' => 'Exposure or self sabotage suggests honesty.',
                'short' => 'Strategy needs integrity.',
                'long' => '<p>The Seven of Swords highlights stealth or unconventional tactics. Be smart, not shady.</p><p>Reversed, come clean, regroup, and rebuild trust.</p>',
                'correspondences' => 'Element: Air; Number: 7.',
                'symbols' => 'sneaking figure, swords, tents',
            ],
            'tarot_swords_08' => [
                'keywords' => 'restriction, fear, overwhelm, self doubt, limits',
                'upright' => 'Mental barriers can be loosened with small steps.',
                'reversed' => 'Release and freedom are within reach.',
                'short' => 'A mental bind that can be undone.',
                'long' => '<p>The Eight of Swords shows a feeling of being trapped by fear or assumptions. Test the limits.</p><p>Reversed, ask for help and take one clear freeing action.</p>',
                'correspondences' => 'Element: Air; Number: 8.',
                'symbols' => 'bound figure, swords, blindfold, marsh',
            ],
            'tarot_swords_09' => [
                'keywords' => 'anxiety, worry, guilt, sleeplessness, rumination',
                'upright' => 'Worry feels intense; name it and seek support.',
                'reversed' => 'Relief arrives as you face the fear.',
                'short' => 'Anxiety asks for care and honesty.',
                'long' => '<p>The Nine of Swords reflects sleepless thoughts and guilt. Be gentle with yourself.</p><p>Reversed, perspective returns and the mind begins to quiet.</p>',
                'correspondences' => 'Element: Air; Number: 9.',
                'symbols' => 'bed, swords on wall, night, hands to face',
            ],
            'tarot_swords_10' => [
                'keywords' => 'ending, release, exhaustion, truth, conclusion',
                'upright' => 'A cycle ends; accept the truth and release.',
                'reversed' => 'Recovery and renewal begin after the low point.',
                'short' => 'An ending clears the way.',
                'long' => '<p>The Ten of Swords marks a final ending and a need to let go. The worst is behind you.</p><p>Reversed, healing starts and a new dawn appears.</p>',
                'correspondences' => 'Element: Air; Number: 10.',
                'symbols' => 'ten swords, sunrise, still figure',
            ],
            'tarot_swords_11' => [
                'keywords' => 'curiosity, honesty, study, ideas, messages',
                'upright' => 'A curious mind seeks truth and learning.',
                'reversed' => 'Gossip or haste warns against careless words.',
                'short' => 'Curious and direct learning.',
                'long' => '<p>The Page of Swords brings a message or idea that needs investigation. Ask questions.</p><p>Reversed, slow down and check facts before speaking.</p>',
                'correspondences' => 'Element: Air of Earth (Page); Suit: Air.',
                'symbols' => 'raised sword, wind, youthful figure',
            ],
            'tarot_swords_12' => [
                'keywords' => 'decisive action, urgency, focus, drive, logic',
                'upright' => 'Act with clear logic and direct focus.',
                'reversed' => 'Impulsiveness or aggression needs restraint.',
                'short' => 'Fast action and sharp focus.',
                'long' => '<p>The Knight of Swords charges toward a goal with determination. Use the momentum wisely.</p><p>Reversed, avoid rash decisions and temper sharp speech.</p>',
                'correspondences' => 'Element: Air of Air (Knight); Suit: Air.',
                'symbols' => 'charging horse, sword, stormy sky',
            ],
            'tarot_swords_13' => [
                'keywords' => 'discernment, boundaries, clarity, independence, truth',
                'upright' => 'Lead with clear boundaries and fair judgment.',
                'reversed' => 'Coldness or bitterness asks for softness.',
                'short' => 'Clear mind and honest boundaries.',
                'long' => '<p>The Queen of Swords sees what is real and communicates it with clarity. Speak plainly and kindly.</p><p>Reversed, soften rigidity and reconnect with empathy.</p>',
                'correspondences' => 'Element: Air of Water (Queen); Suit: Air.',
                'symbols' => 'raised sword, birds, throne, clouds',
            ],
            'tarot_swords_14' => [
                'keywords' => 'authority, strategy, truth, leadership, discipline',
                'upright' => 'Lead with logic, fairness, and ethical authority.',
                'reversed' => 'Tyranny or rigidity suggests reevaluation.',
                'short' => 'Strategic and disciplined leadership.',
                'long' => '<p>The King of Swords brings order through clear standards and decisive action. Choose principles over emotion.</p><p>Reversed, avoid harsh judgment and seek balance.</p>',
                'correspondences' => 'Element: Air of Fire (King); Suit: Air.',
                'symbols' => 'throne, sword, butterflies, storm clouds',
            ],
            'tarot_pentacles_01' => [
                'keywords' => 'opportunity, resources, stability, health, prosperity',
                'upright' => 'A practical opportunity arrives; plant the seed.',
                'reversed' => 'Missed chances or instability suggests caution.',
                'short' => 'A solid new opportunity.',
                'long' => '<p>The Ace of Pentacles brings a tangible start in work, money, or health. Build steadily.</p><p>Reversed, ground your plan and avoid risky shortcuts.</p>',
                'correspondences' => 'Element: Earth; Number: 1 (ace).',
                'symbols' => 'pentacle, garden, archway, hand from cloud',
            ],
            'tarot_pentacles_02' => [
                'keywords' => 'balance, adaptability, finances, rhythm, priorities',
                'upright' => 'Balance resources and keep your priorities flexible.',
                'reversed' => 'Overcommitment or imbalance needs adjustment.',
                'short' => 'Juggling with steady rhythm.',
                'long' => '<p>The Two of Pentacles asks for flexible planning and steady pacing. Keep it simple.</p><p>Reversed, reduce obligations and stabilize your routine.</p>',
                'correspondences' => 'Element: Earth; Number: 2.',
                'symbols' => 'juggling figure, infinity loop, ships, coins',
            ],
            'tarot_pentacles_03' => [
                'keywords' => 'teamwork, skill, learning, craft, recognition',
                'upright' => 'Collaboration and skill bring solid results.',
                'reversed' => 'Misalignment or poor quality needs correction.',
                'short' => 'Craft and teamwork succeed.',
                'long' => '<p>The Three of Pentacles highlights shared work and skillful execution. Learn and contribute.</p><p>Reversed, clarify roles and raise standards.</p>',
                'correspondences' => 'Element: Earth; Number: 3.',
                'symbols' => 'cathedral, tools, apprentices, pentacles',
            ],
            'tarot_pentacles_04' => [
                'keywords' => 'security, saving, control, stability, possessiveness',
                'upright' => 'Protect your resources, but do not cling too tightly.',
                'reversed' => 'Fear of loss or overspending needs balance.',
                'short' => 'Security with a risk of rigidity.',
                'long' => '<p>The Four of Pentacles is about stability and cautious stewardship. Use resources wisely.</p><p>Reversed, loosen control and trust a healthy flow.</p>',
                'correspondences' => 'Element: Earth; Number: 4.',
                'symbols' => 'coins, crown, city, guarded stance',
            ],
            'tarot_pentacles_05' => [
                'keywords' => 'hardship, lack, worry, support, resilience',
                'upright' => 'Hard times call for support and practical steps.',
                'reversed' => 'Relief appears; accept help and rebuild.',
                'short' => 'Hardship with help nearby.',
                'long' => '<p>The Five of Pentacles reflects material or emotional strain. You are not alone, even if it feels so.</p><p>Reversed, recovery begins through support and resourcefulness.</p>',
                'correspondences' => 'Element: Earth; Number: 5.',
                'symbols' => 'snow, stained glass, limping figures, coins',
            ],
            'tarot_pentacles_06' => [
                'keywords' => 'generosity, balance, giving, receiving, fairness',
                'upright' => 'Share resources with fairness and gratitude.',
                'reversed' => 'Imbalance or strings attached require honesty.',
                'short' => 'Give and receive in balance.',
                'long' => '<p>The Six of Pentacles is about equitable exchange. Be generous and accept help.</p><p>Reversed, check power dynamics and set clear terms.</p>',
                'correspondences' => 'Element: Earth; Number: 6.',
                'symbols' => 'scales, coins, giving figure, kneeling figures',
            ],
            'tarot_pentacles_07' => [
                'keywords' => 'patience, assessment, growth, investment, progress',
                'upright' => 'Results take time; review progress and adjust.',
                'reversed' => 'Impatience or lack of reward needs recalibration.',
                'short' => 'Patience and long term growth.',
                'long' => '<p>The Seven of Pentacles signals a pause to assess growth. Keep tending what matters.</p><p>Reversed, reset expectations and refine your effort.</p>',
                'correspondences' => 'Element: Earth; Number: 7.',
                'symbols' => 'vine with coins, tool, thoughtful figure',
            ],
            'tarot_pentacles_08' => [
                'keywords' => 'craft, diligence, mastery, practice, focus',
                'upright' => 'Skill grows through steady practice and attention.',
                'reversed' => 'Perfectionism or boredom calls for new focus.',
                'short' => 'Dedicated work and skill building.',
                'long' => '<p>The Eight of Pentacles rewards disciplined effort. Stay focused on quality.</p><p>Reversed, avoid burnout and reconnect with purpose.</p>',
                'correspondences' => 'Element: Earth; Number: 8.',
                'symbols' => 'workbench, carved coins, tools, focus',
            ],
            'tarot_pentacles_09' => [
                'keywords' => 'independence, refinement, comfort, self reliance, rewards',
                'upright' => 'Enjoy the rewards of your steady work.',
                'reversed' => 'Dependence or isolation suggests realignment.',
                'short' => 'Self reliance and earned comfort.',
                'long' => '<p>The Nine of Pentacles highlights earned comfort and refined taste. Appreciate your progress.</p><p>Reversed, reconnect with support or balance material focus.</p>',
                'correspondences' => 'Element: Earth; Number: 9.',
                'symbols' => 'garden, falcon, grapes, pentacles',
            ],
            'tarot_pentacles_10' => [
                'keywords' => 'legacy, family, stability, wealth, tradition',
                'upright' => 'Long term stability and legacy are in focus.',
                'reversed' => 'Family tension or instability needs attention.',
                'short' => 'Legacy and long term stability.',
                'long' => '<p>The Ten of Pentacles points to lasting structures, family support, and material security. Think long term.</p><p>Reversed, address weak foundations and update traditions.</p>',
                'correspondences' => 'Element: Earth; Number: 10.',
                'symbols' => 'family scene, archway, coins, dogs',
            ],
            'tarot_pentacles_11' => [
                'keywords' => 'learning, opportunity, practicality, study, growth',
                'upright' => 'A practical path opens; learn and build skills.',
                'reversed' => 'Lack of focus or procrastination delays progress.',
                'short' => 'A grounded new path.',
                'long' => '<p>The Page of Pentacles brings a tangible opportunity to learn or earn. Start with basics.</p><p>Reversed, commit to a plan and stay consistent.</p>',
                'correspondences' => 'Element: Earth of Earth (Page); Suit: Earth.',
                'symbols' => 'coin, field, youthful figure, flowers',
            ],
            'tarot_pentacles_12' => [
                'keywords' => 'responsibility, steady work, patience, reliability, routine',
                'upright' => 'Steady progress comes from reliable effort.',
                'reversed' => 'Stagnation or stubbornness needs adjustment.',
                'short' => 'Slow and steady progress.',
                'long' => '<p>The Knight of Pentacles is methodical and dependable. Keep showing up.</p><p>Reversed, break the rut and add fresh momentum.</p>',
                'correspondences' => 'Element: Earth of Air (Knight); Suit: Earth.',
                'symbols' => 'horse, coin, plowed field, stillness',
            ],
            'tarot_pentacles_13' => [
                'keywords' => 'nurture, practicality, abundance, care, resourcefulness',
                'upright' => 'Create comfort through practical care and kindness.',
                'reversed' => 'Overgiving or insecurity needs boundaries.',
                'short' => 'Nurturing and practical abundance.',
                'long' => '<p>The Queen of Pentacles blends warmth with resourcefulness. Build a stable, supportive environment.</p><p>Reversed, protect your energy and avoid overextending.</p>',
                'correspondences' => 'Element: Earth of Water (Queen); Suit: Earth.',
                'symbols' => 'throne, rabbit, coin, garden',
            ],
            'tarot_pentacles_14' => [
                'keywords' => 'mastery, discipline, security, leadership, stewardship',
                'upright' => 'Lead with grounded authority and long term vision.',
                'reversed' => 'Material rigidity or control needs balance.',
                'short' => 'Grounded leadership and stewardship.',
                'long' => '<p>The King of Pentacles represents mastery of the material world. Use resources wisely and plan for the long term.</p><p>Reversed, loosen control and invest in what truly matters.</p>',
                'correspondences' => 'Element: Earth of Fire (King); Suit: Earth.',
                'symbols' => 'throne, grapes, bulls, coin',
            ],
        ];
    }
}
