<?php
declare(strict_types=1);


namespace DailyTarot\Frontend;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.DateTime.RestrictedFunctions.date_date

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Analytics\Tracker;
use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Meaning\MeaningPackRepository;
use DailyTarot\Reading\ReadingComposer;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\DefaultDecks;
use DailyTarot\Support\EmailCtaSettings;
use DailyTarot\Support\ShortcodeSettings;
use DailyTarot\Support\OnlineVisitors;
use DailyTarot\Support\SpreadPresets;
use DailyTarot\Support\SpreadMeaningPacks;
use DailyTarot\Support\SpreadSettings;
use DailyTarot\Support\SpreadMappings;
use DailyTarot\Support\BookingSettings;
use DailyTarot\Support\ShareImageSettings;
use DailyTarot\Support\RelatedLinks;
use DailyTarot\Support\PostTypes;

final class Shortcodes {

    /**
     * Resolve a deck card image URL from the stored _dtarot_cards map.
     *
     * Supports Gypsy/Kipper alias IDs so legacy decks keep their image connections.
     *
     * @param array<string,mixed> $imgs
     */
    private static function resolveDeckCardUrlFromMap(array $imgs, string $cardId): string {
        if (!$imgs || $cardId === '') return '';

        foreach (Cards::kipperGypsyAliases($cardId) as $id) {
            if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
            $u = trim((string)$imgs[$id]);
            if ($u !== '') return $u;
        }

        return '';
    }

    /**
     * Shared renderer for deck-like card grids.
     *
     * @param array<string,string> $cards cardId => cardName
     */
    private static function renderDeckGrid(int $deckId, int $cols, array $cards): string {
        if ($deckId <= 0) return '';
        if ($cols < 2) $cols = 2;
        if ($cols > 10) $cols = 10;

        self::enqueueFrontendAssets();

        $imgs = get_post_meta($deckId, '_dtarot_cards', true);
        if (!is_array($imgs)) $imgs = [];
        $back = (string)get_post_meta($deckId, '_dtarot_back', true);

        $style = '--dtarot-cols:'.$cols.';';
        $html = '<div class="dtarot-deck-grid" style="'.esc_attr($style).'">';
        foreach ($cards as $cardId => $name) {
            $url = esc_url(self::resolveDeckCardUrlFromMap($imgs, (string)$cardId));
            $href = self::cardDetailUrlFromDeck($deckId, (string)$cardId);
            $html .= '<div class="dtarot-card-tile">';
            $html .= '<div class="dtarot-card-media">';
            if ($url !== '') {
                if ($back !== '') {
                    $html .= self::renderFlip($url, $back, $href);
                } else {
                    $img = '<img src="'.$url.'" alt="" />';
                    if ($href !== '') {
                        $img = '<a class="dtarot-card-link" href="' . esc_url($href) . '">' . $img . '</a>';
                    }
                    $html .= $img;
                }
            } else {
                $html .= '<div class="dtarot-card-media-empty" aria-hidden="true"></div>';
            }
            $html .= '</div>';
            if ($href !== '') {
                $html .= '<div class="dtarot-card-caption"><a class="dtarot-card-link" href="' . esc_url($href) . '">' . esc_html($name) . '</a></div>';
            } else {
                $html .= '<div class="dtarot-card-caption">'.esc_html($name).'</div>';
            }
            $html .= '</div>';
        }
        $html .= '</div>';
        return $html;
    }

    /** @param array<string,string> $cards */
    private static function filterTarotMajors(array $cards): array {
        $out = [];
        foreach ($cards as $cardId => $name) {
            if (is_string($cardId) && str_starts_with($cardId, 'tarot_major_')) {
                $out[$cardId] = $name;
            }
        }
        return $out;
    }

    /** @param array<string,string> $cards */
    private static function filterTarotMinors(array $cards): array {
        $out = [];
        foreach ($cards as $cardId => $name) {
            if (!is_string($cardId)) continue;
            if (
                str_starts_with($cardId, 'tarot_wands_') ||
                str_starts_with($cardId, 'tarot_cups_') ||
                str_starts_with($cardId, 'tarot_swords_') ||
                str_starts_with($cardId, 'tarot_pentacles_')
            ) {
                $out[$cardId] = $name;
            }
        }
        return $out;
    }

    /** @param array<string,string> $cards */
    private static function filterTarotMinorsBySuit(array $cards, string $suit): array {
        $suit = sanitize_key($suit);
        if ($suit === 'coins') $suit = 'pentacles';

        $prefix = '';
        if ($suit === 'wands') $prefix = 'tarot_wands_';
        if ($suit === 'cups') $prefix = 'tarot_cups_';
        if ($suit === 'swords') $prefix = 'tarot_swords_';
        if ($suit === 'pentacles') $prefix = 'tarot_pentacles_';
        if ($prefix === '') return self::filterTarotMinors($cards);

        $out = [];
        foreach ($cards as $cardId => $name) {
            if (is_string($cardId) && str_starts_with($cardId, $prefix)) {
                $out[$cardId] = $name;
            }
        }
        return $out;
    }

    private static function enqueueFrontendAssets(): void {
        static $localized = false;
        $cssFile = DTAROT_PATH . 'assets/frontend.css';
        $cssVer = is_readable($cssFile) ? (string)@filemtime($cssFile) : DTAROT_VERSION;
        wp_enqueue_style('dtarot-frontend', DTAROT_URL.'assets/frontend.css', [], $cssVer);

        $shareCss = DTAROT_PATH . 'assets/share-image.css';
        if (is_readable($shareCss)) {
            $shareVer = (string)@filemtime($shareCss);
            wp_enqueue_style('dtarot-share-image', DTAROT_URL.'assets/share-image.css', [], $shareVer);
        }

        $jsFile = DTAROT_PATH . 'assets/frontend.js';
        $ver = is_readable($jsFile) ? (string)@filemtime($jsFile) : DTAROT_VERSION;
        wp_enqueue_script('dtarot-frontend-js', DTAROT_URL.'assets/frontend.js', [], $ver, true);

        $shareJs = DTAROT_PATH . 'assets/share-image.js';
        if (is_readable($shareJs)) {
            $shareVer = (string)@filemtime($shareJs);
            wp_enqueue_script('dtarot-share-image', DTAROT_URL.'assets/share-image.js', [], $shareVer, true);
        }

        if (!$localized) {
            wp_localize_script('dtarot-frontend-js', 'DTAROT_FRONTEND', [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('dtarot_booking'),
                'shareNonce' => wp_create_nonce('dtarot_share'),
                'shareImage' => class_exists(ShareImageSettings::class) ? ShareImageSettings::get() : [],
                'analyticsEnabled' => (bool)apply_filters('dtarot_analytics_enabled', true),
                'analytics' => [
                    'pageType' => 'embed',
                    'url' => (string)get_permalink(),
                ],
                'i18n' => [
                    'loading' => __('Loading...','daily-tarot'),
                    'noSlots' => __('No available times for this date.','daily-tarot'),
                    'selectTime' => __('Select a time','daily-tarot'),
                ],
            ]);
            $localized = true;
        }
    }

    private static function renderFlip(string $frontUrl, string $backUrl, string $href = ''): string {
        $frontUrl = esc_url(trim($frontUrl));
        $backUrl = esc_url(trim($backUrl));
        $href = esc_url(trim($href));

        // Only enable flip if both URLs validate to something non-empty.
        if ($frontUrl === '' || $backUrl === '') {
            return $frontUrl !== '' ? '<img src="' . $frontUrl . '" alt="" />' : '';
        }

        $hrefAttr = $href !== '' ? ' data-dtarot-href="' . $href . '"' : '';

        return '<div class="dtarot-card-flip" data-dtarot-flip' . $hrefAttr . ' role="button" tabindex="0" aria-label="'.esc_attr__('Flip card','daily-tarot').'">'
            . '<div class="dtarot-card-flip-inner">'
            . '<div class="dtarot-card-face dtarot-card-front"><img src="' . $frontUrl . '" alt="" /></div>'
            . '<div class="dtarot-card-face dtarot-card-back"><img src="' . $backUrl . '" alt="" /></div>'
            . '</div>'
            . '</div>';
    }

    private static function resolveEmptyBackUrl(string $mode, int $deckId): string {
        $mode = sanitize_key($mode);
        if (!in_array($mode, ['random','default'], true)) {
            $mode = 'random';
        }

        $deckId = $deckId > 0 ? $deckId : 0;
        if ($mode === 'default') {
            if ($deckId > 0) {
                $p = get_post($deckId);
                if ($p && isset($p->post_type) && PostTypes::isDeckType((string)$p->post_type)) {
                    $back = (string)get_post_meta($deckId, '_dtarot_back', true);
                    if ($back !== '') {
                        return esc_url($back);
                    }
                }
            }
            $deckId = self::pickFallbackDeckId(false);
        } else {
            $deckId = self::pickFallbackDeckId(true);
        }

        if ($deckId <= 0) return '';
        $back = (string)get_post_meta($deckId, '_dtarot_back', true);
        return $back !== '' ? esc_url($back) : '';
    }

    private static function pickFallbackDeckId(bool $random): int {
        $query = [
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => 1,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => $random ? 'rand' : 'date',
            'order' => $random ? 'ASC' : 'DESC',
            'meta_query' => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Returns 1 id and is used as a fallback.
                [
                    'key' => '_dtarot_back',
                    'compare' => '!=',
                    'value' => '',
                ],
            ],
        ];
        $decks = get_posts($query);
        if (!$decks || !isset($decks[0]->ID)) return 0;
        return (int)$decks[0]->ID;
    }

    private static function cardDetailUrlFromDeck(int $deckId, string $cardId): string {
        if ($deckId <= 0 || $cardId === '') return '';
        if (!class_exists(ReadableRoutes::class)) return '';

        $system = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        if ($system === '') $system = Cards::SYSTEM_TAROT;

        return ReadableRoutes::urlForCard($system, $cardId);
    }

    private static function renderLenormandMetaBlock(string $cardId): string {
        $cardId = sanitize_text_field($cardId);
        if ($cardId === '' || !str_starts_with($cardId, 'lenormand_')) return '';

        $meta = Cards::meta($cardId);
        if (!$meta) return '';

        $subject = isset($meta['subject']) && is_string($meta['subject']) ? trim($meta['subject']) : '';
        $modifier = isset($meta['modifier']) && is_string($meta['modifier']) ? trim($meta['modifier']) : '';
        $extended = isset($meta['extended']) && is_string($meta['extended']) ? trim($meta['extended']) : '';

        $rank = '';
        $suit = '';
        if (isset($meta['playing']) && is_array($meta['playing'])) {
            $rank = isset($meta['playing']['rank']) && is_string($meta['playing']['rank']) ? trim($meta['playing']['rank']) : '';
            $suit = isset($meta['playing']['suit']) && is_string($meta['playing']['suit']) ? trim($meta['playing']['suit']) : '';
        }

        $isSignifier = !empty($meta['is_signifier']);

        if ($subject === '' && $modifier === '' && $extended === '' && $rank === '' && $suit === '' && !$isSignifier) {
            return '';
        }

        $suitLabels = [
            'hearts' => __('Hearts','daily-tarot'),
            'spades' => __('Spades','daily-tarot'),
            'diamonds' => __('Diamonds','daily-tarot'),
            'clubs' => __('Clubs','daily-tarot'),
        ];
        $suitLabel = $suit !== '' ? ($suitLabels[$suit] ?? $suit) : '';

        $html = '<div class="dtarot-lenormand-meta">';

        if ($isSignifier) {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Signifier','daily-tarot').'</span></div>';
        }

        if ($rank !== '' || $suitLabel !== '') {
            $inset = trim($rank . ' ' . $suitLabel);
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Playing card inset','daily-tarot').':</span> '.esc_html($inset).'</div>';
        }

        if ($subject !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Subject','daily-tarot').':</span> '.esc_html($subject).'</div>';
        }
        if ($modifier !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Modifier','daily-tarot').':</span> '.esc_html($modifier).'</div>';
        }
        if ($extended !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Extended','daily-tarot').':</span> '.esc_html($extended).'</div>';
        }

        $html .= '</div>';
        return $html;
    }

    /**
     * @param array{date?:string,post_id?:int,deck_id?:int,card_id?:string,preset?:string,pack?:string,share_style?:string,title?:string,excerpt?:string,readable_url?:string,card_img?:string} $meta
     */
    public static function renderShareLinks(string $url, string $context, array $meta = []): string {
        $url = esc_url(trim($url));
        if ($url === '') return '';

        $encoded = rawurlencode($url);
        $context = sanitize_key($context);

        $shareStyle = '';
        if (!empty($meta['share_style']) && is_string($meta['share_style'])) {
            $shareStyle = sanitize_key($meta['share_style']);
        } elseif (class_exists(ShortcodeSettings::class)) {
            $settings = ShortcodeSettings::get();
            $shareStyle = isset($settings['share_style']) && is_string($settings['share_style']) ? (string)$settings['share_style'] : '';
        }
        if (!in_array($shareStyle, ['text','arcana-sigils','crystal-aura','tarot-card'], true)) {
            $shareStyle = 'text';
        }

        $attrs = [
            'data-dtarot-share' => '1',
            'data-dtarot-context' => $context,
            'data-dtarot-url' => $url,
        ];
        if (!empty($meta['date']) && is_string($meta['date'])) {
            $attrs['data-dtarot-date'] = (string)$meta['date'];
        }
        if (!empty($meta['readable_url']) && is_string($meta['readable_url'])) {
            $attrs['data-dtarot-readable'] = (string)$meta['readable_url'];
        } elseif (!empty($meta['date']) && is_string($meta['date']) && class_exists(ReadableRoutes::class)) {
            $readable = ReadableRoutes::urlForDate((string)$meta['date']);
            if ($readable !== '') {
                $attrs['data-dtarot-readable'] = $readable;
            }
        }
        if (!empty($meta['post_id'])) {
            $attrs['data-dtarot-post-id'] = (string)(int)$meta['post_id'];
        }
        if (!empty($meta['deck_id'])) {
            $attrs['data-dtarot-deck'] = (string)(int)$meta['deck_id'];
        }
        if (!empty($meta['card_id']) && is_string($meta['card_id'])) {
            $attrs['data-dtarot-card'] = (string)$meta['card_id'];
        }
        if (!empty($meta['preset']) && is_string($meta['preset'])) {
            $attrs['data-dtarot-preset'] = (string)$meta['preset'];
        }
        if (!empty($meta['pack']) && is_string($meta['pack'])) {
            $attrs['data-dtarot-pack'] = (string)$meta['pack'];
        }
        if (!empty($meta['title']) && is_string($meta['title'])) {
            $attrs['data-dtarot-title'] = (string)$meta['title'];
        }
        if (!empty($meta['excerpt']) && is_string($meta['excerpt'])) {
            $attrs['data-dtarot-excerpt'] = (string)$meta['excerpt'];
        }
        if (!empty($meta['card_img']) && is_string($meta['card_img'])) {
            $attrs['data-dtarot-card-img'] = (string)$meta['card_img'];
        }

        $attrString = '';
        foreach ($attrs as $k => $v) {
            $attrString .= ' ' . $k . '="' . esc_attr($v) . '"';
        }

        $shareClass = 'dtarot-share';
        if ($shareStyle !== 'text') {
            $shareClass .= ' dtarot-share--icons dtarot-share--' . $shareStyle;
        }

        $html = '<div class="' . esc_attr($shareClass) . '"' . $attrString . '>';

        if ($shareStyle === 'text') {
            $html .= '<button type="button" class="dtarot-share-link dtarot-share-copy" data-dtarot-share-link="copy">' . esc_html__('Copy link','daily-tarot') . '</button>';
            $html .= '<span class="dtarot-share-sep">|</span>';
            $html .= '<a class="dtarot-share-link" data-dtarot-share-link="link" href="'.$url.'">'.esc_html__('Link','daily-tarot').'</a>';
            $html .= '<span class="dtarot-share-sep">|</span>';
            $html .= '<a class="dtarot-share-link" data-dtarot-share-link="facebook" target="_blank" rel="noopener" href="https://www.facebook.com/sharer/sharer.php?u='.$encoded.'">'.esc_html__('Facebook','daily-tarot').'</a>';
            $html .= '<span class="dtarot-share-sep">|</span>';
            $html .= '<a class="dtarot-share-link" data-dtarot-share-link="instagram" target="_blank" rel="noopener" href="https://www.instagram.com/">'.esc_html__('Instagram','daily-tarot').'</a>';
            $html .= '<span class="dtarot-share-sep">|</span>';
            $html .= '<a class="dtarot-share-link" data-dtarot-share-link="x" target="_blank" rel="noopener" href="https://twitter.com/intent/tweet?url='.$encoded.'">'.esc_html__('X','daily-tarot').'</a>';
        } else {
            $icons = [
                'arcana-sigils' => [
                    'dir' => 'icons/arcana-sigils',
                    'copy' => 'as-copy.webp',
                    'facebook' => 'as-facebook.webp',
                    'instagram' => 'as-instagram.webp',
                    'whatsapp' => 'as-whatsapp.webp',
                    'x' => 'as-twitter.webp',
                ],
                'crystal-aura' => [
                    'dir' => 'icons/crystal-aura',
                    'copy' => 'ca-copy.webp',
                    'facebook' => 'ca-facebook.webp',
                    'instagram' => 'ca-instagram.webp',
                    'whatsapp' => 'ca-whatsapp.webp',
                    'x' => 'ca-twitter.webp',
                ],
                'tarot-card' => [
                    'dir' => 'icons/tarot_card',
                    'copy' => 'tc-copy.webp',
                    'facebook' => 'tc-facebook.webp',
                    'instagram' => 'tc-instagram.webp',
                    'whatsapp' => 'tx-whatsapp.webp',
                    'x' => 'tc-twitter.webp',
                ],
            ];

            $set = $icons[$shareStyle] ?? null;
            $dir = is_array($set) && isset($set['dir']) ? (string)$set['dir'] : '';
            $basePath = $dir !== '' ? 'assets/' . $dir . '/' : '';
            $baseUrl = $basePath !== '' ? DTAROT_URL . $basePath : '';

            $iconUrl = static function(string $file) use ($baseUrl): string {
                if ($baseUrl === '' || $file === '') return '';
                return $baseUrl . $file;
            };

            $copyIcon = (is_array($set) && isset($set['copy'])) ? $iconUrl((string)$set['copy']) : '';
            $facebookIcon = (is_array($set) && isset($set['facebook'])) ? $iconUrl((string)$set['facebook']) : '';
            $instagramIcon = (is_array($set) && isset($set['instagram'])) ? $iconUrl((string)$set['instagram']) : '';
            $whatsappIcon = (is_array($set) && isset($set['whatsapp'])) ? $iconUrl((string)$set['whatsapp']) : '';
            $xIcon = (is_array($set) && isset($set['x'])) ? $iconUrl((string)$set['x']) : '';

            if ($copyIcon !== '') {
                $html .= '<button type="button" class="dtarot-share-link dtarot-share-icon dtarot-share-copy" data-dtarot-share-link="copy"><img src="' . esc_url($copyIcon) . '" alt="' . esc_attr__('Copy link','daily-tarot') . '" /></button>';
            }
            if ($facebookIcon !== '') {
                $html .= '<a class="dtarot-share-link dtarot-share-icon" data-dtarot-share-link="facebook" target="_blank" rel="noopener" href="https://www.facebook.com/sharer/sharer.php?u='.$encoded.'"><img src="' . esc_url($facebookIcon) . '" alt="' . esc_attr__('Facebook','daily-tarot') . '" /></a>';
            }
            if ($instagramIcon !== '') {
                $html .= '<a class="dtarot-share-link dtarot-share-icon" data-dtarot-share-link="instagram" target="_blank" rel="noopener" href="https://www.instagram.com/?url='.$encoded.'"><img src="' . esc_url($instagramIcon) . '" alt="' . esc_attr__('Instagram','daily-tarot') . '" /></a>';
            }
            if ($whatsappIcon !== '') {
                $html .= '<a class="dtarot-share-link dtarot-share-icon" data-dtarot-share-link="whatsapp" target="_blank" rel="noopener" href="https://wa.me/?text='.$encoded.'"><img src="' . esc_url($whatsappIcon) . '" alt="' . esc_attr__('WhatsApp','daily-tarot') . '" /></a>';
            }
            if ($xIcon !== '') {
                $html .= '<a class="dtarot-share-link dtarot-share-icon" data-dtarot-share-link="x" target="_blank" rel="noopener" href="https://twitter.com/intent/tweet?url='.$encoded.'"><img src="' . esc_url($xIcon) . '" alt="' . esc_attr__('X','daily-tarot') . '" /></a>';
            }
        }

        $html .= '</div>';

        return $html;
    }

    public static function init(): void {
        add_shortcode('daily_tarot', [__CLASS__, 'render_daily_tarot']);
        add_shortcode('dtarot_decks', [__CLASS__, 'render_decks']);
        add_shortcode('dtarot_deck', [__CLASS__, 'render_deck']);
        add_shortcode('dtarot_majors', [__CLASS__, 'render_tarot_majors']);
        add_shortcode('dtarot_minors', [__CLASS__, 'render_tarot_minors']);
        add_shortcode('dtarot_email_cta', [__CLASS__, 'render_email_cta']);
        add_shortcode('dtarot_card', [__CLASS__, 'render_card']);
        add_shortcode('dtarot_spread', [__CLASS__, 'render_spread']);
        add_shortcode('dtarot_month', [__CLASS__, 'render_month']);
        add_shortcode('dtarot_booking', [__CLASS__, 'render_booking']);
        add_shortcode('dtarot_booking_button', [__CLASS__, 'render_booking_button']);
        add_shortcode('dtarot_booking_teaser', [__CLASS__, 'render_booking_teaser']);
    }

    /**
     * Shortcode: [dtarot_email_cta action="https://xxx.list-manage.com/subscribe/post?u=...&id=..."]
     *
     * Minimal email capture form intended for Mailchimp (or similar hosted forms).
     * This shortcode does NOT store emails in WordPress.
     */
    public static function render_email_cta($atts = []): string {
        $atts = shortcode_atts([
            'action' => '',
            'text' => __('Input your email for daily cards and more.','daily-tarot'),
            'placeholder' => __('Your email','daily-tarot'),
            'button' => __('Subscribe','daily-tarot'),
            'disclaimer' => '',
            'target_blank' => '1',
            'honeypot_name' => '',
        ], (array)$atts, 'dtarot_email_cta');

        $actionRaw = is_string($atts['action']) ? trim($atts['action']) : '';
        $honeypotName = isset($atts['honeypot_name']) && is_string($atts['honeypot_name']) ? sanitize_key($atts['honeypot_name']) : '';
        $provider = '';

        // If the action URL isn't provided in the shortcode, fall back to Settings → CTA.
        if ($actionRaw === '' && class_exists(EmailCtaSettings::class)) {
            $s = EmailCtaSettings::get();
            $provider = isset($s['provider']) && is_string($s['provider']) ? (string)$s['provider'] : '';
            if ($provider === 'mailchimp') {
                if (isset($s['action_url']) && is_string($s['action_url'])) {
                    $actionRaw = trim($s['action_url']);
                }
                if ($honeypotName === '' && isset($s['honeypot_name']) && is_string($s['honeypot_name'])) {
                    $honeypotName = sanitize_key($s['honeypot_name']);
                }
            } elseif ($provider === 'wp') {
                $actionRaw = admin_url('admin-post.php');
                if ($honeypotName === '' && isset($s['honeypot_name']) && is_string($s['honeypot_name'])) {
                    $honeypotName = sanitize_key($s['honeypot_name']);
                }
            }
        }

        $useWp = ($provider === 'wp' && $actionRaw !== '');
        $action = esc_url($actionRaw);
        if ($action === '') {
            if (function_exists('current_user_can') && current_user_can('manage_options')) {
                return '<div class="dtarot-frontend dtarot-frontend-empty">'
                    . esc_html__('Email CTA is missing its provider configuration. Set it in Daily Tarot → Settings → CTA, or add `action` to the shortcode.','daily-tarot')
                    . '</div>';
            }
            return '';
        }

        $text = isset($atts['text']) && is_string($atts['text']) ? trim($atts['text']) : '';
        $placeholder = isset($atts['placeholder']) && is_string($atts['placeholder']) ? trim($atts['placeholder']) : '';
        $button = isset($atts['button']) && is_string($atts['button']) ? trim($atts['button']) : '';
        $disclaimer = isset($atts['disclaimer']) && is_string($atts['disclaimer']) ? trim($atts['disclaimer']) : '';

        $targetBlank = is_string($atts['target_blank']) && in_array(strtolower(trim($atts['target_blank'])), ['1','true','yes','on'], true);
        if ($useWp) $targetBlank = false;
        $targetAttr = $targetBlank ? ' target="_blank" rel="noopener noreferrer"' : '';

        self::enqueueFrontendAssets();

        $html = '<div class="dtarot-frontend dtarot-email-cta">';
        if ($useWp) {
            $ctaStatus = isset($_GET['dtarot_email_cta']) ? sanitize_key((string)wp_unslash($_GET['dtarot_email_cta'])) : '';
            if ($ctaStatus === 'ok') {
                $html .= '<div class="dtarot-email-cta-success">' . esc_html__('Thanks! You are on the list.','daily-tarot') . '</div>';
            } elseif ($ctaStatus === 'invalid') {
                $html .= '<div class="dtarot-email-cta-error">' . esc_html__('Please enter a valid email address.','daily-tarot') . '</div>';
            }
        }
        if ($text !== '') {
            $html .= '<div class="dtarot-email-cta-text">' . esc_html($text) . '</div>';
        }

        $html .= '<form class="dtarot-email-cta-form" action="' . $action . '" method="post"' . $targetAttr . ' novalidate>';
        if ($useWp) {
            $html .= '<input type="hidden" name="action" value="dtarot_email_cta_submit" />';
            $html .= wp_nonce_field('dtarot_email_cta_submit', '_wpnonce', true, false);
        }
        $html .= '<label class="dtarot-email-cta-label" for="dtarot-email-cta-email">' . esc_html__('Email','daily-tarot') . '</label>';
        $html .= '<div class="dtarot-email-cta-row">';
        $emailField = $useWp ? 'dtarot_email' : 'EMAIL';
        $html .= '<input class="dtarot-email-cta-input" type="email" id="dtarot-email-cta-email" name="' . esc_attr($emailField) . '" required autocomplete="email" placeholder="' . esc_attr($placeholder) . '" />';
        $html .= '<button class="dtarot-daily-button dtarot-email-cta-button" type="submit">' . esc_html($button) . '</button>';
        $html .= '</div>';

        // Optional honeypot field if the user copied it from Mailchimp embed code.
        if ($honeypotName !== '') {
            $html .= '<div style="position:absolute;left:-5000px;" aria-hidden="true">'
                . '<input type="text" name="' . esc_attr($honeypotName) . '" tabindex="-1" value="" />'
                . '</div>';
        }

        if ($disclaimer !== '') {
            $html .= '<div class="dtarot-email-cta-disclaimer">' . esc_html($disclaimer) . '</div>';
        }

        $html .= '</form>';
        $html .= '</div>';
        return $html;
    }

    /**
     * Shortcode: [dtarot_month]
     *
     * Renders a month grid; published days link to the readable URL.
     */
    public static function render_month($atts = []): string {
        $atts = shortcode_atts([
            'year' => '',
            'month' => '',
            'show_card' => '1',
            'show_excerpt' => '0',
        ], (array)$atts, 'dtarot_month');

        $year = is_string($atts['year']) ? (int)trim($atts['year']) : 0;
        $month = is_string($atts['month']) ? (int)trim($atts['month']) : 0;

        // Allow month navigation via query args when shortcode attrs are not set.
        if ($year <= 0 && isset($_GET['dtarot_year'])) {
            $year = (int)sanitize_text_field((string)wp_unslash($_GET['dtarot_year']));
        }
        if ($month <= 0 && isset($_GET['dtarot_month'])) {
            $month = (int)sanitize_text_field((string)wp_unslash($_GET['dtarot_month']));
        }

        $nowTs = function_exists('current_time') ? (int)current_time('timestamp') : time();
        if ($year <= 0) $year = (int)gmdate('Y', $nowTs);
        if ($month <= 0) $month = (int)gmdate('n', $nowTs);
        if ($month < 1) $month = 1;
        if ($month > 12) $month = 12;

        $showCard = is_string($atts['show_card']) && in_array(strtolower(trim($atts['show_card'])), ['1','true','yes','on'], true);
        $showExcerpt = is_string($atts['show_excerpt']) && in_array(strtolower(trim($atts['show_excerpt'])), ['1','true','yes','on'], true);

        $firstTs = strtotime(sprintf('%04d-%02d-01 00:00:00', $year, $month));
        if ($firstTs === false) {
            return '';
        }

        // Week starts Monday (1..7)
        $daysInMonth = (int)date('t', $firstTs);
        $startDow = (int)date('N', $firstTs);

        self::enqueueFrontendAssets();

        $monthTitle = function_exists('wp_date') ? (string)wp_date('F Y', $firstTs) : (string)date_i18n('F Y', $firstTs);

        $prevMonth = $month - 1;
        $prevYear = $year;
        if ($prevMonth < 1) {
            $prevMonth = 12;
            $prevYear = $year - 1;
        }
        $nextMonth = $month + 1;
        $nextYear = $year;
        if ($nextMonth > 12) {
            $nextMonth = 1;
            $nextYear = $year + 1;
        }

        $baseUrl = '';
        if (function_exists('get_permalink')) {
            $qid = function_exists('get_queried_object_id') ? (int)get_queried_object_id() : 0;
            $baseUrl = $qid > 0 ? (string)get_permalink($qid) : (string)get_permalink();
        }
        if ($baseUrl === '') {
            $req = '/';
            if (isset($_SERVER['REQUEST_URI'])) {
                $req = sanitize_text_field((string)wp_unslash($_SERVER['REQUEST_URI']));
                $req = trim($req);
                if ($req === '') {
                    $req = '/';
                } elseif (substr($req, 0, 1) !== '/') {
                    $req = '/' . ltrim($req, '/');
                }
            }
            $baseUrl = function_exists('home_url') ? (string)home_url($req) : $req;
        }
        $baseUrl = remove_query_arg(['dtarot_year','dtarot_month'], $baseUrl);

        $prevUrl = add_query_arg(['dtarot_year' => (string)$prevYear, 'dtarot_month' => (string)$prevMonth], $baseUrl);
        $nextUrl = add_query_arg(['dtarot_year' => (string)$nextYear, 'dtarot_month' => (string)$nextMonth], $baseUrl);
        $todayUrl = $baseUrl;

        $html = '<div class="dtarot-month">';
        $html .= '<div class="dtarot-month-nav">'
            . '<a class="dtarot-month-nav-link" href="' . esc_url($prevUrl) . '">'.esc_html__('← Previous','daily-tarot').'</a>'
            . '<a class="dtarot-month-nav-link" href="' . esc_url($todayUrl) . '">'.esc_html__('Today','daily-tarot').'</a>'
            . '<a class="dtarot-month-nav-link" href="' . esc_url($nextUrl) . '">'.esc_html__('Next →','daily-tarot').'</a>'
            . '</div>';
        $html .= '<div class="dtarot-month-title">' . esc_html($monthTitle) . '</div>';
        $html .= '<div class="dtarot-month-grid" role="grid">';

        // Empty cells before day 1
        for ($i = 1; $i < $startDow; $i++) {
            $html .= '<div class="dtarot-month-cell dtarot-month-empty" aria-hidden="true"></div>';
        }

        $rangeStart = sprintf('%04d-%02d-01', $year, $month);
        $rangeEnd = sprintf('%04d-%02d-%02d', $year, $month, $daysInMonth);
        $published = DayEntryService::publishedRange($rangeStart, $rangeEnd);

        for ($d = 1; $d <= $daysInMonth; $d++) {
            $date = sprintf('%04d-%02d-%02d', $year, $month, $d);
            $entry = $published[$date] ?? null;

            $cardName = '';
            $url = '';
            $excerpt = '';
            if ($entry) {
                $arr = $entry->toArray();
                $cardId = (string)($arr['card'] ?? '');
                $cardName = $cardId !== '' ? Cards::name($cardId) : '';
                if ($showExcerpt) {
                    $raw = isset($arr['content']) && is_string($arr['content']) ? (string)$arr['content'] : '';
                    $raw = trim(preg_replace('/\s+/', ' ', wp_strip_all_tags($raw)) ?? $raw);
                    if ($raw !== '') {
                        $excerpt = (mb_strlen($raw) > 120) ? (mb_substr($raw, 0, 117) . '…') : $raw;
                    }
                }
                if (class_exists(ReadableRoutes::class)) {
                    $url = ReadableRoutes::urlForDate($date);
                }
            }

            $html .= '<div class="dtarot-month-cell">';

            if ($url !== '') {
                $html .= '<a class="dtarot-month-link" href="' . esc_url($url) . '">';
                $html .= '<div class="dtarot-month-day">' . esc_html((string)$d) . '</div>';
                if ($showCard && $cardName !== '') {
                    $html .= '<div class="dtarot-month-card">' . esc_html($cardName) . '</div>';
                }
                if ($showExcerpt && $excerpt !== '') {
                    $html .= '<div class="dtarot-month-excerpt">' . esc_html($excerpt) . '</div>';
                }
                $html .= '</a>';
            } else {
                $html .= '<div class="dtarot-month-day">' . esc_html((string)$d) . '</div>';
            }

            $html .= '</div>';
        }

        $html .= '</div>';
        $html .= '</div>';
        $postId = (isset($GLOBALS['post']) && is_object($GLOBALS['post']) && isset($GLOBALS['post']->ID)) ? (int)$GLOBALS['post']->ID : 0;
        $html .= self::renderShareLinks((string)get_permalink(), 'month', [
            'post_id' => $postId,
        ]);
        return $html;
    }

    /**
     * Shortcode: [daily_tarot]
     * Optional: [daily_tarot date="2025-12-19"]
     */
    public static function render_daily_tarot($atts = []): string {
        $defaultEmptyText = __('No card has been drawn for today. While you wait, here\'s a gentle reminder: "The future depends on what you do today."','daily-tarot');
        $shortcodeDefaults = [
            'card_position' => '',
            'text_mode' => '',
            'excerpt_words' => '',
            'read_more_label' => '',
            'layout' => '',
            'show_share' => '',
            'share_style' => '',
            'theme' => '',
        ];
        $settings = class_exists(ShortcodeSettings::class) ? ShortcodeSettings::get() : [
            'card_position' => 'left',
            'text_mode' => 'full',
            'excerpt_words' => 40,
            'read_more_label' => 'Read more',
            'layout_default' => 'minimal',
            'show_share' => '0',
            'share_style' => 'text',
            'theme_style' => 'minimal',
            'empty_card_mode' => 'random',
            'empty_deck_id' => 0,
            'empty_text' => $defaultEmptyText,
        ];

        $atts = shortcode_atts([
            'date' => '',
            'mode' => 'latest',
            'days' => '7',
            'show_date' => '0',
            'layout' => '',
            'show_share' => '',
            'share_style' => '',
            'readable_url' => '',
        ] + $shortcodeDefaults, (array)$atts, 'daily_tarot');

        $date = is_string($atts['date']) ? trim($atts['date']) : '';

        // Allow readable pages to pass date via query param.
        if ($date === '' && isset($_GET['dtarot_date'])) {
            $candidate = sanitize_text_field((string)wp_unslash($_GET['dtarot_date']));
            if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $candidate)) {
                $date = $candidate;
            }
        }

        $mode = is_string($atts['mode']) ? strtolower(trim($atts['mode'])) : 'latest';

        // If CTA links back to the same page, allow query param to force readable view.
        $forceReadable = isset($_GET['dtarot']) && (sanitize_text_field((string)wp_unslash($_GET['dtarot'])) === 'readable');
        if ($forceReadable) {
            $mode = 'readable';
        }
        $days = is_string($atts['days']) ? (int)$atts['days'] : 7;
        if ($days <= 0) $days = 7;
        if ($days > 90) $days = 90;

        $showDate = is_string($atts['show_date']) && in_array(strtolower(trim($atts['show_date'])), ['1','true','yes','on'], true);

        $layout = is_string($atts['layout']) ? strtolower(trim($atts['layout'])) : '';
        if ($layout === '') {
            $layout = (string)$settings['layout_default'];
        }
        if (!in_array($layout, ['minimal','hero'], true)) $layout = 'minimal';

        $showShare = false;
        if (is_string($atts['show_share']) && trim($atts['show_share']) !== '') {
            $showShare = in_array(strtolower(trim($atts['show_share'])), ['1','true','yes','on'], true);
        } else {
            $showShare = ((string)$settings['show_share'] === '1');
        }

        $shareStyle = is_string($atts['share_style']) ? sanitize_key($atts['share_style']) : '';
        if ($shareStyle === '') {
            $shareStyle = isset($settings['share_style']) && is_string($settings['share_style']) ? (string)$settings['share_style'] : 'text';
        }
        if (!in_array($shareStyle, ['text','arcana-sigils','crystal-aura','tarot-card'], true)) {
            $shareStyle = 'text';
        }

        if ($date === '') {
            $date = function_exists('wp_date') ? (string)wp_date('Y-m-d') : (string)current_time('Y-m-d');
        }

        $readableUrl = is_string($atts['readable_url']) ? trim($atts['readable_url']) : '';
        $cardPosition = is_string($atts['card_position']) ? sanitize_key($atts['card_position']) : '';
        if ($cardPosition === '') {
            $cardPosition = (string)$settings['card_position'];
        }
        if (!in_array($cardPosition, ['left','right','center'], true)) {
            $cardPosition = 'left';
        }

        $textMode = is_string($atts['text_mode']) ? sanitize_key($atts['text_mode']) : '';
        if ($textMode === '') {
            $textMode = (string)$settings['text_mode'];
        }
        if (!in_array($textMode, ['full','excerpt','read_more','none'], true)) {
            $textMode = 'full';
        }

        $excerptWords = is_string($atts['excerpt_words']) ? (int)$atts['excerpt_words'] : 0;
        if ($excerptWords <= 0) {
            $excerptWords = (int)$settings['excerpt_words'];
        }
        if ($excerptWords < 10) $excerptWords = 10;
        if ($excerptWords > 200) $excerptWords = 200;

        $readMoreLabel = is_string($atts['read_more_label']) ? trim($atts['read_more_label']) : '';
        if ($readMoreLabel === '') {
            $readMoreLabel = (string)$settings['read_more_label'];
        }
        $readMoreLabel = sanitize_text_field($readMoreLabel);
        if ($readMoreLabel === '') $readMoreLabel = 'Read more';

        $themeStyle = is_string($atts['theme']) ? sanitize_key($atts['theme']) : '';
        if ($themeStyle === '') {
            $themeStyle = (string)$settings['theme_style'];
        }
        if (!in_array($themeStyle, ['minimal','mystic','modern'], true)) {
            $themeStyle = 'minimal';
        }
        $themeClass = ' dtarot-theme-' . $themeStyle;

        $emptyCardMode = isset($settings['empty_card_mode']) && is_string($settings['empty_card_mode']) ? sanitize_key($settings['empty_card_mode']) : 'random';
        if (!in_array($emptyCardMode, ['random','default'], true)) {
            $emptyCardMode = 'random';
        }
        $emptyDeckId = isset($settings['empty_deck_id']) ? (int)$settings['empty_deck_id'] : 0;
        if ($emptyDeckId < 0) $emptyDeckId = 0;
        $emptyText = isset($settings['empty_text']) && is_string($settings['empty_text']) ? trim($settings['empty_text']) : $defaultEmptyText;
        if ($emptyText === '') $emptyText = $defaultEmptyText;

        // Enqueue minimal styles only when shortcode is used.
        self::enqueueFrontendAssets();

        if ($mode === 'archive') {
            $items = DayEntryService::latestPublished($days, $date);
            if (!$items) {
                return '<div class="dtarot-frontend dtarot-frontend-empty">'.esc_html__('No published readings found.','daily-tarot').'</div>';
            }

            if (class_exists(OnlineVisitors::class)) {
                OnlineVisitors::record();
            }

            $html = '<div class="dtarot-frontend dtarot-frontend-archive' . esc_attr($themeClass) . '">';
            foreach ($items as $d => $entry) {
                // Archive stays minimal for now.
                $html .= self::renderReading($entry->toArray(), $showDate ? $d : '', 'minimal', $showShare, $d, $readableUrl, $cardPosition, $textMode, $excerptWords, $readMoreLabel, $shareStyle);
            }
            $html .= '</div>';
            return $html;
        }

        if ($mode === 'readable') {
            $entry = DayEntryService::getPublished($date);
            if (!$entry) {
                return self::renderEmptyDaily($date, 'minimal', true, $showShare, $themeStyle, $cardPosition, $textMode, $excerptWords, $readMoreLabel, $emptyText, $emptyCardMode, $emptyDeckId);
            }

            if (class_exists(Tracker::class)) {
                Tracker::trackPublishedReading($entry, 'shortcode_readable', $date);
            }
            if (class_exists(OnlineVisitors::class)) {
                OnlineVisitors::record();
            }

            return self::renderReadable($entry->toArray(), $date, $themeStyle, $shareStyle);
        }

        // Default: latest (single)
        $entry = DayEntryService::getPublished($date);
        if (!$entry) {
            return self::renderEmptyDaily($date, $layout, $showDate, $showShare, $themeStyle, $cardPosition, $textMode, $excerptWords, $readMoreLabel, $emptyText, $emptyCardMode, $emptyDeckId);
        }

        if (class_exists(Tracker::class)) {
            Tracker::trackPublishedReading($entry, $layout === 'hero' ? 'shortcode_hero' : 'shortcode_latest', $date);
        }
        if (class_exists(OnlineVisitors::class)) {
            OnlineVisitors::record();
        }

        // Latest
        if ($layout === 'hero') {
            return self::renderHero($entry->toArray(), $date, $showShare, $readableUrl, $themeStyle, $shareStyle);
        }

        return '<div class="dtarot-frontend' . esc_attr($themeClass) . '">'.self::renderReading($entry->toArray(), $showDate ? $date : '', 'minimal', $showShare, $date, $readableUrl, $cardPosition, $textMode, $excerptWords, $readMoreLabel, $shareStyle).'</div>';
    }

    private static function renderEmptyDaily(
        string $date,
        string $layout,
        bool $showDate,
        bool $showShare,
        string $themeStyle,
        string $cardPosition,
        string $textMode,
        int $excerptWords,
        string $readMoreLabel,
        string $emptyText,
        string $emptyCardMode,
        int $emptyDeckId
    ): string {
        $emptyText = trim($emptyText);
        $legacy = __('No card has been drawn for today. While you wait, here\'s a gentle reminder: "The future depends on what you do today."','daily-tarot');
        $useQuoteBox = ($emptyText === '' || $emptyText === $legacy);
        $contentHtml = '';
        $shareExcerpt = '';
        if ($textMode !== 'none') {
            if ($useQuoteBox) {
                $intro = __('No card has been drawn for today. While you wait, here\'s a gentle reminder:','daily-tarot');
                $quote = self::pickEmptyQuote();
                $contentHtml = '<div class="dtarot-empty-intro">' . esc_html($intro) . '</div>'
                    . '<div class="dtarot-empty-quote">"' . esc_html($quote) . '"</div>';
                $shareExcerpt = $intro . ' ' . $quote;
            } else {
                $contentHtml = self::formatReadingContent($emptyText, $textMode, $excerptWords, $readMoreLabel, '');
                $shareExcerpt = self::shareExcerpt($emptyText, 220);
            }
        }
        $title = __('No card drawn yet','daily-tarot');
        $backUrl = self::resolveEmptyBackUrl($emptyCardMode, $emptyDeckId);

        $themeStyle = in_array($themeStyle, ['minimal','mystic','modern'], true) ? $themeStyle : 'minimal';
        if ($layout === 'hero') {
            $ts = strtotime($date . ' 00:00:00');
            $prettyDate = $date;
            if ($ts !== false) {
                if (function_exists('wp_date')) {
                    $prettyDate = (string)wp_date('l, F jS, Y', $ts);
                } else {
                    $prettyDate = (string)date_i18n('l, F jS, Y', $ts);
                }
            }

            $html = '<section class="dtarot-daily-hero dtarot-theme-' . esc_attr($themeStyle) . '">';
            $html .= '<div class="dtarot-daily-inner">';
            $html .= '<header class="dtarot-daily-header">';
            $html .= '<h2 class="dtarot-daily-kicker">'.esc_html__('Card of the Day','daily-tarot').'</h2>';
            $html .= '<div class="dtarot-daily-date">'.esc_html($prettyDate).'</div>';
            $html .= '</header>';
            $html .= '<div class="dtarot-daily-divider"></div>';

            $html .= '<div class="dtarot-daily-body">';
            $html .= '<div class="dtarot-daily-left">';
            if ($backUrl !== '') {
                $html .= '<div class="dtarot-daily-card"><img src="' . $backUrl . '" alt="" /></div>';
            }
            $html .= '</div>';

            $html .= '<div class="dtarot-daily-right">';
            $html .= '<h3 class="dtarot-daily-title">'.esc_html($title).'</h3>';
            if ($contentHtml !== '') {
                $html .= '<div class="dtarot-daily-text">'.wp_kses_post($contentHtml).'</div>';
            }
            $html .= '</div>';
            $html .= '</div>';
            if ($showShare) {
                $postId = (isset($GLOBALS['post']) && is_object($GLOBALS['post']) && isset($GLOBALS['post']->ID)) ? (int)$GLOBALS['post']->ID : 0;
                $html .= '<footer class="dtarot-daily-footer">';
                $html .= self::renderShareLinks((string)get_permalink(), 'daily', [
                    'date' => $date,
                    'post_id' => $postId,
                    'title' => $title,
                    'excerpt' => $shareExcerpt,
                ]);
                $html .= '</footer>';
            }
            $html .= '</div>';
            $html .= '</section>';
            return $html;
        }

        $themeClass = ' dtarot-theme-' . $themeStyle;
        $rowClass = 'dtarot-frontend-row dtarot-frontend-pos-' . $cardPosition;
        $html = '<div class="dtarot-frontend' . esc_attr($themeClass) . '">';
        $html .= '<div class="' . esc_attr($rowClass) . '">';

        if ($backUrl !== '') {
            $html .= '<div class="dtarot-frontend-left"><img src="' . $backUrl . '" alt="" /></div>';
        } else {
            $html .= '<div class="dtarot-frontend-left dtarot-frontend-left-empty"></div>';
        }

        $html .= '<div class="dtarot-frontend-right">';
        if ($showDate) {
            $html .= '<div class="dtarot-frontend-meta">'.esc_html($date).'</div>';
        }
        $html .= '<div class="dtarot-frontend-title">'.esc_html($title).'</div>';
        if ($contentHtml !== '') {
            $html .= '<div class="dtarot-frontend-content">'.$contentHtml.'</div>';
        }
        if ($showShare) {
            $postId = (isset($GLOBALS['post']) && is_object($GLOBALS['post']) && isset($GLOBALS['post']->ID)) ? (int)$GLOBALS['post']->ID : 0;
            $html .= self::renderShareLinks((string)get_permalink(), 'daily', [
                'date' => $date,
                'post_id' => $postId,
                'title' => $title,
                'excerpt' => $shareExcerpt,
            ]);
        }
        $html .= '</div>';
        $html .= '</div>';
        $html .= '</div>';
        return $html;
    }

    /** @param array<string,mixed> $entry */
    private static function renderReading(
        array $entry,
        string $dateLabel = '',
        string $layout = 'minimal',
        bool $showShare = false,
        string $date = '',
        string $readableUrlBase = '',
        string $cardPosition = 'left',
        string $textMode = 'full',
        int $excerptWords = 40,
        string $readMoreLabel = 'Read more',
        string $shareStyle = 'text'
    ): string {
        $cardId = (string)($entry['card'] ?? '');
        $cardName = Cards::name($cardId);

        $imgUrl = '';
        $deckId = (int)($entry['deck'] ?? 0);
        $back = '';
        $overrideUrl = isset($entry['image_override_url']) ? trim((string)$entry['image_override_url']) : '';
        if ($overrideUrl !== '') {
            $imgUrl = esc_url($overrideUrl);
            $back = '';
        } elseif ($deckId > 0 && $cardId !== '') {
            $imgs = get_post_meta($deckId, '_dtarot_cards', true);
            if (is_array($imgs)) {
                $imgUrl = esc_url(self::resolveDeckCardUrlFromMap($imgs, (string)$cardId));
            }
            $back = (string)get_post_meta($deckId, '_dtarot_back', true);
        }

        $cardUrl = self::cardDetailUrlFromDeck($deckId, $cardId);
        if ($date !== '' && class_exists(ReadableRoutes::class)) {
            $dailyUrl = ReadableRoutes::urlForDate($date, $readableUrlBase);
            if ($dailyUrl !== '') {
                $cardUrl = $dailyUrl;
            }
        }

        $rowClass = 'dtarot-frontend-row dtarot-frontend-pos-' . $cardPosition;
        $html = '<div class="' . esc_attr($rowClass) . '">';

        if ($imgUrl !== '') {
            if ($back !== '') {
                $html .= '<div class="dtarot-frontend-left">' . self::renderFlip($imgUrl, $back, $cardUrl) . '</div>';
            } else {
                $img = '<img src="'.$imgUrl.'" alt="" />';
                if ($cardUrl !== '') {
                    $img = '<a class="dtarot-card-link" href="' . esc_url($cardUrl) . '">' . $img . '</a>';
                }
                $html .= '<div class="dtarot-frontend-left">' . $img . '</div>';
            }
        } else {
            $html .= '<div class="dtarot-frontend-left dtarot-frontend-left-empty"></div>';
        }

        $html .= '<div class="dtarot-frontend-right">';

        if ($dateLabel !== '') {
            $html .= '<div class="dtarot-frontend-meta">'.esc_html($dateLabel).'</div>';
        }

        if ($cardUrl !== '') {
            $html .= '<div class="dtarot-frontend-title"><a class="dtarot-card-link" href="' . esc_url($cardUrl) . '">' . esc_html($cardName) . '</a></div>';
        } else {
            $html .= '<div class="dtarot-frontend-title">'.esc_html($cardName).'</div>';
        }

        $html .= self::renderLenormandMetaBlock($cardId);

        $content = (string)($entry['content'] ?? '');
        if ($content === '') {
            $packId = (int)($entry['pack'] ?? 0);
            $meaning = MeaningPackRepository::getMeaning($packId, $cardId);
            // Minimal layout uses short if available, else upright.
            if ($meaning['short'] !== '') {
                $content = $meaning['short'];
            } elseif ($meaning['upright'] !== '') {
                $content = $meaning['upright'];
            }
        }
        $shareExcerpt = self::shareExcerpt($content, 220);

        $contentHtml = self::formatReadingContent($content, $textMode, $excerptWords, $readMoreLabel, $cardUrl);
        if ($contentHtml !== '') {
            $html .= '<div class="dtarot-frontend-content">'.$contentHtml.'</div>';
        }

        if (class_exists(RelatedLinks::class)) {
            $html .= RelatedLinks::renderBlock($cardId, $cardName);
        }

        if ($showShare) {
            $postId = (isset($GLOBALS['post']) && is_object($GLOBALS['post']) && isset($GLOBALS['post']->ID)) ? (int)$GLOBALS['post']->ID : 0;
            $html .= self::renderShareLinks((string)get_permalink(), 'daily', [
                'date' => $date,
                'post_id' => $postId,
                'deck_id' => $deckId,
                'card_id' => $cardId,
                'readable_url' => $date !== '' ? $cardUrl : '',
                'share_style' => $shareStyle,
                'title' => $cardName,
                'excerpt' => $shareExcerpt,
                'card_img' => $imgUrl,
            ]);
        }

        $html .= '</div>';
        $html .= '</div>';
        return $html;
    }

    private static function shareExcerpt(string $content, int $maxChars = 200): string {
        $content = trim((string)wp_strip_all_tags($content));
        if ($content === '') return '';
        $content = preg_replace('/\s+/', ' ', $content) ?? $content;
        if (mb_strlen($content) <= $maxChars) return $content;
        return mb_substr($content, 0, $maxChars - 3) . '...';
    }

    private static function pickEmptyQuote(): string {
        $quotes = [
            __('The future depends on what you do today.','daily-tarot'),
            __('Trust the timing of your life.','daily-tarot'),
            __('Small steps every day lead to big change.','daily-tarot'),
            __('What is meant for you will not pass you by.','daily-tarot'),
            __('Listen closely to what your intuition whispers.','daily-tarot'),
            __('Your path is unfolding with every breath.','daily-tarot'),
            __('Clarity arrives when you allow stillness.','daily-tarot'),
            __('Let the day reveal itself step by step.','daily-tarot'),
            __('Patience is a form of power.','daily-tarot'),
            __('A quiet mind hears the next right move.','daily-tarot'),
            __('The answers you seek are already within you.','daily-tarot'),
            __('Let go of haste and the message will find you.','daily-tarot'),
            __('Even pauses are part of the journey.','daily-tarot'),
            __('Open hands receive more than clenched ones.','daily-tarot'),
            __('What you nurture today grows tomorrow.','daily-tarot'),
            __('Breathe in calm, breathe out doubt.','daily-tarot'),
            __('You are exactly where you need to be.','daily-tarot'),
            __('Softness is not weakness; it is wisdom.','daily-tarot'),
            __('The next sign appears when you slow down.','daily-tarot'),
            __('Every ending clears space for a beginning.','daily-tarot'),
            __('Choose grace over rush.','daily-tarot'),
            __('The universe speaks in gentle nudges.','daily-tarot'),
            __('You do not have to force what is meant for you.','daily-tarot'),
            __('Rest is part of the work.','daily-tarot'),
            __('The present moment is your compass.','daily-tarot'),
            __('Let curiosity lead you forward.','daily-tarot'),
            __('You are supported, even in the quiet.','daily-tarot'),
            __('Hope is a daily practice.','daily-tarot'),
            __('Trust the pause between breaths.','daily-tarot'),
            __('Growth happens below the surface.','daily-tarot'),
            __('Your energy shapes your outcome.','daily-tarot'),
            __('Gentle progress is still progress.','daily-tarot'),
            __('Your intuition already knows the way.','daily-tarot'),
            __('Make room for the unexpected blessing.','daily-tarot'),
            __('Alignment feels calm, not frantic.','daily-tarot'),
            __('You are allowed to take your time.','daily-tarot'),
            __('There is strength in choosing peace.','daily-tarot'),
            __('Your next step will reveal itself.','daily-tarot'),
            __('This moment is enough.','daily-tarot'),
            __('Let the day soften your edges.','daily-tarot'),
            __('You are learning even when you wait.','daily-tarot'),
            __('Trust the gentle rhythm of your life.','daily-tarot'),
            __('Stillness makes room for insight.','daily-tarot'),
            __('Let today be simple and true.','daily-tarot'),
            __('You are guided by what feels honest.','daily-tarot'),
            __('Your heart knows how to navigate.','daily-tarot'),
            __('Be patient with the unfolding.','daily-tarot'),
            __('The message arrives in its own time.','daily-tarot'),
            __('Peace is a powerful choice.','daily-tarot'),
            __('Trust the process, not the pressure.','daily-tarot'),
            __('Your presence is your power.','daily-tarot'),
            __('Listen to the quiet, it is speaking.','daily-tarot'),
            __('Let your breath anchor you.','daily-tarot'),
            __('Each day teaches you in new ways.','daily-tarot'),
            __('You are allowed to begin again.','daily-tarot'),
            __('The right moment cannot miss you.','daily-tarot'),
        ];
        $index = array_rand($quotes);
        return $quotes[$index] ?? $quotes[0];
    }

    private static function formatReadingContent(string $content, string $textMode, int $excerptWords, string $readMoreLabel, string $cardUrl): string {
        $content = trim($content);
        if ($content === '' || $textMode === 'none') return '';

        if ($textMode === 'full') {
            return wp_kses_post($content);
        }

        $excerpt = wp_strip_all_tags($content);
        $excerpt = wp_trim_words($excerpt, $excerptWords, '...');
        if ($excerpt === '') return '';

        if ($textMode === 'excerpt') {
            return esc_html($excerpt);
        }

        $link = '';
        if ($cardUrl !== '') {
            $link = ' <a class="dtarot-frontend-read-more" href="' . esc_url($cardUrl) . '">' . esc_html($readMoreLabel) . '</a>';
        }
        return esc_html($excerpt) . $link;
    }

    /** @param array<string,mixed> $entry */
    /** @param array<string,mixed> $entry */
    private static function renderHero(array $entry, string $date, bool $showShare, string $readableUrl = '', string $themeStyle = 'minimal', string $shareStyle = 'text'): string {
        self::enqueueFrontendAssets();

        $cardId = (string)($entry['card'] ?? '');
        $cardName = Cards::name($cardId);

        $imgUrl = '';
        $deckId = (int)($entry['deck'] ?? 0);
        $back = '';
        $overrideUrl = isset($entry['image_override_url']) ? trim((string)$entry['image_override_url']) : '';
        if ($overrideUrl !== '') {
            $imgUrl = esc_url($overrideUrl);
            $back = '';
        } elseif ($deckId > 0 && $cardId !== '') {
            $imgs = get_post_meta($deckId, '_dtarot_cards', true);
            if (is_array($imgs)) {
                $imgUrl = esc_url(self::resolveDeckCardUrlFromMap($imgs, (string)$cardId));
            }
            $back = (string)get_post_meta($deckId, '_dtarot_back', true);
        }

        $cardUrl = self::cardDetailUrlFromDeck($deckId, $cardId);

        $ts = strtotime($date . ' 00:00:00');
        $prettyDate = $date;
        if ($ts !== false) {
            if (function_exists('wp_date')) {
                $prettyDate = (string)wp_date('l, F jS, Y', $ts);
            } else {
                $prettyDate = (string)date_i18n('l, F jS, Y', $ts);
            }
        }

        $raw = (string)($entry['content'] ?? '');
        if (trim($raw) === '') {
            $packId = (int)($entry['pack'] ?? 0);
            $meaning = MeaningPackRepository::getMeaning($packId, $cardId);
            // Hero: short becomes subtitle; long becomes body. Fallback to upright.
            $subtitle = $meaning['short'] !== '' ? $meaning['short'] : '';
            $body = $meaning['long'] !== '' ? $meaning['long'] : $meaning['upright'];
            if ($subtitle !== '' && $body !== '') {
                $raw = '<p>' . esc_html($subtitle) . '</p>' . $body;
            } elseif ($body !== '') {
                $raw = $body;
            }
        }
        [$subtitleText, $bodyHtml] = self::splitHeroContent($raw);
        $shareContent = trim($subtitleText . ' ' . wp_strip_all_tags($bodyHtml));
        $shareExcerpt = self::shareExcerpt($shareContent, 220);

        if (class_exists(ReadableRoutes::class)) {
            $readableUrl = ReadableRoutes::urlForDate($date, $readableUrl);
        } else {
            if ($readableUrl === '') {
                $readableUrl = (string)add_query_arg([
                    'dtarot' => 'readable',
                    'dtarot_date' => $date,
                ], (string)get_permalink());
            } else {
                $readableUrl = (string)add_query_arg([
                    'dtarot_date' => $date,
                ], $readableUrl);
            }
        }

        if ($readableUrl !== '') {
            $cardUrl = $readableUrl;
        }

        $themeStyle = in_array($themeStyle, ['minimal','mystic','modern'], true) ? $themeStyle : 'minimal';
        $html = '<section class="dtarot-daily-hero dtarot-theme-' . esc_attr($themeStyle) . '">';
        $html .= '<div class="dtarot-daily-inner">';
        $html .= '<header class="dtarot-daily-header">';
    $html .= '<h2 class="dtarot-daily-kicker">'.esc_html__('Card of the Day','daily-tarot').'</h2>';
        $html .= '<div class="dtarot-daily-date">'.esc_html($prettyDate).'</div>';
        $html .= '</header>';
        $html .= '<div class="dtarot-daily-divider"></div>';

        $html .= '<div class="dtarot-daily-body">';

        $html .= '<div class="dtarot-daily-left">';
        if ($imgUrl !== '') {
            if ($back !== '') {
                $html .= '<div class="dtarot-daily-card">' . self::renderFlip($imgUrl, $back, $cardUrl) . '</div>';
            } else {
                $img = '<img src="'.$imgUrl.'" alt="" />';
                if ($cardUrl !== '') {
                    $img = '<a class="dtarot-card-link" href="' . esc_url($cardUrl) . '">' . $img . '</a>';
                }
                $html .= '<div class="dtarot-daily-card">' . $img . '</div>';
            }
        }
        $html .= '</div>';

        $html .= '<div class="dtarot-daily-right">';
        if ($cardUrl !== '') {
            $html .= '<h3 class="dtarot-daily-title"><a class="dtarot-card-link" href="' . esc_url($cardUrl) . '">' . esc_html($cardName) . '</a></h3>';
        } else {
            $html .= '<h3 class="dtarot-daily-title">'.esc_html($cardName).'</h3>';
        }

        $html .= self::renderLenormandMetaBlock($cardId);

        if ($subtitleText !== '') {
            $html .= '<div class="dtarot-daily-subtitle">'.esc_html($subtitleText).'</div>';
        }
        if ($bodyHtml !== '') {
            $html .= '<div class="dtarot-daily-text">'.wp_kses_post($bodyHtml).'</div>';
        }
        $html .= '</div>';

        $html .= '</div>';

        $html .= '<footer class="dtarot-daily-footer">';
        $html .= '<div class="dtarot-daily-cta">';
        $html .= '<a class="dtarot-daily-button" href="'.esc_url($readableUrl).'">'.esc_html__('Read more','daily-tarot').'</a>';
        $html .= '</div>';
        if ($showShare) {
            $url = (string)get_permalink();
            $postId = (isset($GLOBALS['post']) && is_object($GLOBALS['post']) && isset($GLOBALS['post']->ID)) ? (int)$GLOBALS['post']->ID : 0;
            $html .= self::renderShareLinks($url, 'daily', [
                'date' => $date,
                'post_id' => $postId,
                'deck_id' => $deckId,
                'card_id' => $cardId,
                'readable_url' => $readableUrl,
                'share_style' => $shareStyle,
                'title' => $cardName,
                'excerpt' => $shareExcerpt,
                'card_img' => $imgUrl,
            ]);
        }
        $html .= '</footer>';

        $html .= '</div>';
        $html .= '</section>';
        return $html;
    }

    /** @param array<string,mixed> $entry */
    private static function renderReadable(array $entry, string $date, string $themeStyle = 'minimal', string $shareStyle = 'text'): string {
        self::enqueueFrontendAssets();

        $prevDate = DayEntryService::previousPublishedDate($date);
        $nextDate = DayEntryService::nextPublishedDate($date);
        $prevUrl = ($prevDate !== '' && class_exists(ReadableRoutes::class)) ? ReadableRoutes::urlForDate($prevDate) : '';
        $nextUrl = ($nextDate !== '' && class_exists(ReadableRoutes::class)) ? ReadableRoutes::urlForDate($nextDate) : '';

        $cardId = (string)($entry['card'] ?? '');
        $cardName = Cards::name($cardId);

        $imgUrl = '';
        $deckId = (int)($entry['deck'] ?? 0);
        $back = '';
        $overrideUrl = isset($entry['image_override_url']) ? trim((string)$entry['image_override_url']) : '';
        if ($overrideUrl !== '') {
            $imgUrl = esc_url($overrideUrl);
            $back = '';
        } elseif ($deckId > 0 && $cardId !== '') {
            $imgs = get_post_meta($deckId, '_dtarot_cards', true);
            if (is_array($imgs)) {
                $imgUrl = esc_url(self::resolveDeckCardUrlFromMap($imgs, (string)$cardId));
            }
            $back = (string)get_post_meta($deckId, '_dtarot_back', true);
        }

        $cardUrl = self::cardDetailUrlFromDeck($deckId, $cardId);

        $content = (string)($entry['content'] ?? '');
        $dailyText = (string)($entry['daily_text'] ?? '');

        $packId = (int)($entry['pack'] ?? 0);
        $meaning = MeaningPackRepository::getMeaning($packId, $cardId);

        $fallback = ReadingComposer::applyMeaningFallback($content, $dailyText, $meaning);
        $content = $fallback['content'];
        $dailyText = $fallback['daily_text'];

        $themeStyle = in_array($themeStyle, ['minimal','mystic','modern'], true) ? $themeStyle : 'minimal';
        $html = '<div class="dtarot-frontend dtarot-readable dtarot-theme-' . esc_attr($themeStyle) . '">';

        if ($prevUrl !== '' || $nextUrl !== '') {
            $html .= '<nav class="dtarot-readable-nav" aria-label="' . esc_attr__('Daily navigation','daily-tarot') . '">';
            if ($prevUrl !== '') {
                $html .= '<a class="dtarot-readable-nav-link dtarot-readable-nav-prev" href="' . esc_url($prevUrl) . '">← ' . esc_html__('Previous','daily-tarot') . '</a>';
            } else {
                $html .= '<span class="dtarot-readable-nav-spacer"></span>';
            }
            if ($nextUrl !== '') {
                $html .= '<a class="dtarot-readable-nav-link dtarot-readable-nav-next" href="' . esc_url($nextUrl) . '">' . esc_html__('Next','daily-tarot') . ' →</a>';
            }
            $html .= '</nav>';
        }
        $html .= '<div class="dtarot-readable-row">';
        if ($imgUrl !== '') {
            if ($back !== '') {
                $html .= '<div class="dtarot-readable-left">' . self::renderFlip($imgUrl, $back, $cardUrl) . '</div>';
            } else {
                $img = '<img src="'.$imgUrl.'" alt="" />';
                if ($cardUrl !== '') {
                    $img = '<a class="dtarot-card-link" href="' . esc_url($cardUrl) . '">' . $img . '</a>';
                }
                $html .= '<div class="dtarot-readable-left">' . $img . '</div>';
            }
        }
        $html .= '<div class="dtarot-readable-right">';
        if ($cardUrl !== '') {
            $html .= '<h2 class="dtarot-readable-title"><a class="dtarot-card-link" href="' . esc_url($cardUrl) . '">' . esc_html($cardName) . '</a></h2>';
        } else {
            $html .= '<h2 class="dtarot-readable-title">'.esc_html($cardName).'</h2>';
        }

        $html .= self::renderLenormandMetaBlock($cardId);

        if ($content !== '') {
            $html .= '<div class="dtarot-readable-intro">'.wp_kses_post($content).'</div>';
        }
        if ($dailyText !== '') {
            $html .= '<details class="dtarot-readmore">';
            $html .= '<summary class="dtarot-readmore-summary">' . esc_html__('Read more','daily-tarot') . '</summary>';
            $html .= '<div class="dtarot-readable-expanded">'.wp_kses_post($dailyText).'</div>';
            $html .= '</details>';
        }

        $postId = (isset($GLOBALS['post']) && is_object($GLOBALS['post']) && isset($GLOBALS['post']->ID)) ? (int)$GLOBALS['post']->ID : 0;
        $shareSource = $content !== '' ? $content : $dailyText;
        $shareExcerpt = self::shareExcerpt($shareSource, 220);
        $html .= '<div class="dtarot-readable-share">';
        $html .= self::renderShareLinks((string)get_permalink(), 'daily', [
            'date' => $date,
            'post_id' => $postId,
            'deck_id' => $deckId,
            'card_id' => $cardId,
            'readable_url' => class_exists(ReadableRoutes::class) ? ReadableRoutes::urlForDate($date) : '',
            'share_style' => $shareStyle,
            'title' => $cardName,
            'excerpt' => $shareExcerpt,
            'card_img' => $imgUrl,
        ]);
        $html .= '</div>';

        $html .= '</div>';
        $html .= '</div>';
        $html .= '</div>';
        return $html;
    }

    /**
     * Attempts to extract a short subtitle from the first paragraph and returns the remaining HTML.
     *
     * @return array{0:string,1:string} [subtitleText, bodyHtml]
     */
    private static function splitHeroContent(string $html): array {
        $html = trim($html);
        if ($html === '') return ['', ''];

        if (preg_match('/^\s*<p[^>]*>(.*?)<\/p>\s*(.*)$/is', $html, $m)) {
            $subtitle = trim(wp_strip_all_tags((string)$m[1]));
            $rest = trim((string)$m[2]);
            // Avoid super-long subtitle.
            if (mb_strlen($subtitle) > 120) {
                return ['', $html];
            }
            return [$subtitle, $rest];
        }

        // If no paragraph structure, leave as body.
        return ['', $html];
    }

    /** Shortcode: [dtarot_decks] */
    public static function render_decks($atts = []): string {
        self::enqueueFrontendAssets();

        $atts = shortcode_atts([
            // Optional: tarot|kipper|lenormand
            'system' => '',
            // If defaults exist, show only default deck(s). Use "0" to force showing all.
            'default_only' => '1',
        ], (array)$atts, 'dtarot_decks');

        $system = is_string($atts['system']) ? Cards::normalizeSystem($atts['system']) : '';
        $defaultOnly = !isset($atts['default_only']) || !is_string($atts['default_only']) || trim($atts['default_only']) !== '0';

        $query = [
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft'],
        ];

        // If a system is specified, prefer showing that system's default deck.
        if ($system !== '') {
            $query['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Limited query, optionally narrowed further by include.
                [
                    'key' => '_dtarot_system',
                    'value' => $system,
                    'compare' => '=',
                ],
            ];

            if ($defaultOnly) {
                $defaultId = DefaultDecks::get($system);
                if ($defaultId > 0) {
                    $query['include'] = [$defaultId];
                    $query['orderby'] = 'post__in';
                }
            }
        } else {
            // No system specified: if any defaults are set, show only those defaults (one per system).
            if ($defaultOnly) {
                $ids = [];
                // One per family (Tarot / Lenormand / Kipper).
                $candidates = [Cards::SYSTEM_TAROT, Cards::SYSTEM_LENORMAND, Cards::SYSTEM_KIPPER];
                foreach ($candidates as $sys) {
                    $id = DefaultDecks::get($sys);
                    if ($id > 0) $ids[] = $id;
                }
                if ($ids) {
                    $query['include'] = array_values(array_unique($ids));
                    $query['orderby'] = 'post__in';
                }
            }
        }

        $decks = get_posts($query);

        if (!$decks) {
            return '<div class="dtarot-frontend dtarot-frontend-empty">'.esc_html__('No decks found.','daily-tarot').'</div>';
        }

        $html = '<div class="dtarot-decks">';
        foreach ($decks as $d) {
            $title = $d->post_title ?: __('(no title)','daily-tarot');
            $back = (string)get_post_meta($d->ID, '_dtarot_back', true);

            $html .= '<div class="dtarot-deck-card">';
            if ($back) {
                $html .= '<div class="dtarot-deck-thumb"><img src="'.esc_url($back).'" alt="" /></div>';
            }
            $html .= '<div class="dtarot-deck-title">'.esc_html($title).'</div>';
            $html .= '</div>';
        }
        $html .= '</div>';

        return $html;
    }

    /**
     * Shortcode: [dtarot_deck id="123"]
     * Optional: [dtarot_deck id="123" columns="6"]
     */
    public static function render_deck($atts = []): string {
        $atts = shortcode_atts([
            'id' => '0',
            // Used when id is omitted: tarot|kipper|lenormand
            'system' => Cards::SYSTEM_TAROT,
            'columns' => '6',
        ], (array)$atts, 'dtarot_deck');

        $rawId = is_string($atts['id']) ? trim($atts['id']) : '0';
        $deckId = 0;
        if ($rawId !== '' && $rawId !== '0' && $rawId !== 'default') {
            $deckId = (int)$rawId;
        }

        if ($deckId <= 0) {
            $system = is_string($atts['system']) ? Cards::normalizeSystem($atts['system']) : Cards::SYSTEM_TAROT;
            if ($system === '') $system = Cards::SYSTEM_TAROT;

            $deckId = DefaultDecks::get($system);
            if ($deckId <= 0) {
                $deckId = self::findFirstDeckForSystem($system);
            }
        }

        if ($deckId <= 0) return '';

        $cols = is_string($atts['columns']) ? (int)$atts['columns'] : 6;
        if ($cols < 2) $cols = 2;
        if ($cols > 10) $cols = 10;

        $system = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        $cards = Cards::forSystem($system);
        return self::renderDeckGrid($deckId, $cols, $cards);
    }

    private static function findFirstDeckForSystem(string $system): int {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return 0;

        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => 50,
            'post_status' => ['publish','draft'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);
        foreach ((array)$decks as $p) {
            if (!isset($p->ID)) continue;
            $pid = (int)$p->ID;
            $ps = Cards::normalizeSystem((string)get_post_meta($pid, '_dtarot_system', true));
            if ($ps === $system) return $pid;
        }
        return 0;
    }

    /**
     * Shortcode: [dtarot_majors id="123" columns="6"]
     *
     * Renders the Major Arcana for a Tarot deck.
     */
    public static function render_tarot_majors($atts = []): string {
        $atts = shortcode_atts([
            'id' => '0',
            'deck_id' => '0',
            'columns' => '6',
        ], (array)$atts, 'dtarot_majors');

        $deckId = 0;
        $rawId = is_string($atts['id']) ? trim($atts['id']) : '0';
        if ($rawId !== '' && $rawId !== '0' && $rawId !== 'default') {
            $deckId = (int)$rawId;
        }
        if ($deckId <= 0 && is_string($atts['deck_id'])) {
            $rawDeckId = trim($atts['deck_id']);
            if ($rawDeckId !== '' && $rawDeckId !== '0' && $rawDeckId !== 'default') {
                $deckId = (int)$rawDeckId;
            }
        }
        if ($deckId <= 0) {
            $deckId = DefaultDecks::get(Cards::SYSTEM_TAROT);
            if ($deckId <= 0) {
                $deckId = self::findFirstDeckForSystem(Cards::SYSTEM_TAROT);
            }
        }
        if ($deckId <= 0) return '';

        $cols = is_string($atts['columns']) ? (int)$atts['columns'] : 6;
        if ($cols < 2) $cols = 2;
        if ($cols > 10) $cols = 10;

        $system = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        if ($system !== Cards::SYSTEM_TAROT) {
            return '<div class="dtarot-frontend dtarot-frontend-empty">'.esc_html__('This shortcode works with Tarot decks only.','daily-tarot').'</div>';
        }

        $cards = self::filterTarotMajors(Cards::forSystem(Cards::SYSTEM_TAROT));
        return self::renderDeckGrid($deckId, $cols, $cards);
    }

    /**
     * Shortcode: [dtarot_minors id="123" columns="6"]
     *
     * Renders the Minor Arcana for a Tarot deck.
     */
    public static function render_tarot_minors($atts = []): string {
        $atts = shortcode_atts([
            'id' => '0',
            'deck_id' => '0',
            'columns' => '6',
            'suit' => '',
        ], (array)$atts, 'dtarot_minors');

        $deckId = 0;
        $rawId = is_string($atts['id']) ? trim($atts['id']) : '0';
        if ($rawId !== '' && $rawId !== '0' && $rawId !== 'default') {
            $deckId = (int)$rawId;
        }
        if ($deckId <= 0 && is_string($atts['deck_id'])) {
            $rawDeckId = trim($atts['deck_id']);
            if ($rawDeckId !== '' && $rawDeckId !== '0' && $rawDeckId !== 'default') {
                $deckId = (int)$rawDeckId;
            }
        }
        if ($deckId <= 0) {
            $deckId = DefaultDecks::get(Cards::SYSTEM_TAROT);
            if ($deckId <= 0) {
                $deckId = self::findFirstDeckForSystem(Cards::SYSTEM_TAROT);
            }
        }
        if ($deckId <= 0) return '';

        $cols = is_string($atts['columns']) ? (int)$atts['columns'] : 6;
        if ($cols < 2) $cols = 2;
        if ($cols > 10) $cols = 10;

        $system = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        if ($system !== Cards::SYSTEM_TAROT) {
            return '<div class="dtarot-frontend dtarot-frontend-empty">'.esc_html__('This shortcode works with Tarot decks only.','daily-tarot').'</div>';
        }

        $suit = is_string($atts['suit']) ? trim($atts['suit']) : '';
        $cards = self::filterTarotMinorsBySuit(Cards::forSystem(Cards::SYSTEM_TAROT), $suit);
        return self::renderDeckGrid($deckId, $cols, $cards);
    }

    /**
     * Shortcode: [dtarot_card deck_id="123" card="maj_00"]
     */
    public static function render_card($atts = []): string {
        $atts = shortcode_atts([
            'deck_id' => '0',
            'card' => '',
            'title' => '1',
            'link' => '1',
        ], (array)$atts, 'dtarot_card');

        $deckId = is_string($atts['deck_id']) ? (int)$atts['deck_id'] : 0;
        $cardId = is_string($atts['card']) ? trim($atts['card']) : '';
        if ($deckId <= 0 || $cardId === '') return '';

        $cardName = Cards::name($cardId);

        $imgs = get_post_meta($deckId, '_dtarot_cards', true);
        if (!is_array($imgs)) return '';
        $frontUrl = self::resolveDeckCardUrlFromMap($imgs, (string)$cardId);
        if ($frontUrl === '') return '';

        self::enqueueFrontendAssets();

        $showTitle = is_string($atts['title']) && in_array(strtolower(trim($atts['title'])), ['1','true','yes','on'], true);
        $link = is_string($atts['link']) && in_array(strtolower(trim($atts['link'])), ['1','true','yes','on'], true);

        $back = (string)get_post_meta($deckId, '_dtarot_back', true);
        $href = $link ? self::cardDetailUrlFromDeck($deckId, $cardId) : '';

        $html = '<div class="dtarot-single-card">';
        $frontUrl = esc_url($frontUrl);
        if ($back !== '') {
            $html .= '<div class="dtarot-single-card-img">' . self::renderFlip($frontUrl, $back, $href) . '</div>';
        } else {
            $img = '<img src="'.$frontUrl.'" alt="" />';
            if ($href !== '') {
                $img = '<a class="dtarot-card-link" href="' . esc_url($href) . '">' . $img . '</a>';
            }
            $html .= '<div class="dtarot-single-card-img">' . $img . '</div>';
        }
        if ($showTitle) {
            if ($href !== '') {
                $html .= '<div class="dtarot-single-card-title"><a class="dtarot-card-link" href="' . esc_url($href) . '">' . esc_html($cardName) . '</a></div>';
            } else {
                $html .= '<div class="dtarot-single-card-title">'.esc_html($cardName).'</div>';
            }
        }

        $html .= self::renderLenormandMetaBlock($cardId);

        $html .= '</div>';
        return $html;
    }

    /**
     * Shortcode: [dtarot_spread deck_id="123" cards="maj_00,maj_01,maj_02" columns="3" titles="0"]
     *
     * - `cards` is a comma-separated list of fixed card IDs.
     * - `columns` controls grid columns (2..10).
     * - `titles` toggles captions under images.
     */
    public static function render_spread($atts = []): string {
        $atts = shortcode_atts([
            'deck_id' => '0',
            'cards' => '',
            'columns' => '3',
            'titles' => '',
            'link' => '',
            'preset' => '',
            'spread_pack' => '',
            'card_width' => '',
            'card_gap' => '',
            'card_width_offset' => '',
            'card_gap_offset' => '',
            'show_labels' => '',
            'show_meanings' => '',
            'show_titles' => '',
        ], (array)$atts, 'dtarot_spread');

        $deckId = is_string($atts['deck_id']) ? (int)$atts['deck_id'] : 0;
        $cardsRaw = is_string($atts['cards']) ? trim($atts['cards']) : '';

        $cols = is_string($atts['columns']) ? (int)$atts['columns'] : 3;
        if ($cols < 2) $cols = 2;
        if ($cols > 10) $cols = 10;

        $settings = class_exists(SpreadSettings::class) ? SpreadSettings::get() : [
            'default_preset' => 'three_card',
            'default_pack' => 'default',
            'default_deck_id' => 0,
            'show_titles' => '1',
            'show_labels' => '1',
            'show_meanings' => '1',
            'link_cards' => '1',
        ];

        $showTitles = true;

        $link = false;

        $showLabels = is_string($atts['show_labels']) && trim($atts['show_labels']) !== ''
            ? in_array(strtolower(trim($atts['show_labels'])), ['1','true','yes','on'], true)
            : ((string)($settings['show_labels'] ?? '1') === '1');
        $showMeanings = is_string($atts['show_meanings']) && trim($atts['show_meanings']) !== ''
            ? in_array(strtolower(trim($atts['show_meanings'])), ['1','true','yes','on'], true)
            : ((string)($settings['show_meanings'] ?? '1') === '1');

        $preset = is_string($atts['preset']) ? sanitize_key($atts['preset']) : '';
        $packId = is_string($atts['spread_pack']) ? sanitize_key($atts['spread_pack']) : '';
        $cardWidth = is_string($atts['card_width']) ? (int)trim($atts['card_width']) : 0;
        if ($cardWidth < 0) $cardWidth = 0;
        $cardGap = is_string($atts['card_gap']) ? (int)trim($atts['card_gap']) : 0;
        if ($cardGap < 0) $cardGap = 0;
        $cardWidthOffset = is_string($atts['card_width_offset']) ? (int)trim($atts['card_width_offset']) : 0;
        if ($cardWidthOffset < -50) $cardWidthOffset = -50;
        if ($cardWidthOffset > 50) $cardWidthOffset = 50;
        $cardGapOffset = is_string($atts['card_gap_offset']) ? (int)trim($atts['card_gap_offset']) : 0;
        if ($cardGapOffset < -29) $cardGapOffset = -29;
        if ($cardGapOffset > 50) $cardGapOffset = 50;

        $postId = 0;
        if (isset($GLOBALS['post']) && is_object($GLOBALS['post']) && isset($GLOBALS['post']->ID)) {
            $postId = (int)$GLOBALS['post']->ID;
        }
        static $spreadRenderCount = [];
        if ($postId > 0) {
            if (!isset($spreadRenderCount[$postId])) $spreadRenderCount[$postId] = 0;
            $index = $spreadRenderCount[$postId];
            $spreadRenderCount[$postId]++;
            if ($preset === '' || $packId === '' || $deckId <= 0) {
                $mapping = class_exists(SpreadMappings::class) ? SpreadMappings::forPost($postId) : [];
                if (isset($mapping[$index]) && is_array($mapping[$index])) {
                    if ($preset === '' && isset($mapping[$index]['preset'])) {
                        $preset = sanitize_key((string)$mapping[$index]['preset']);
                    }
                    if ($packId === '' && isset($mapping[$index]['pack'])) {
                        $packId = sanitize_key((string)$mapping[$index]['pack']);
                    }
                    if ($deckId <= 0 && isset($mapping[$index]['deck_id'])) {
                        $deckId = (int)$mapping[$index]['deck_id'];
                    }
                }
            }
        }

        if ($preset === '') $preset = (string)($settings['default_preset'] ?? 'three_card');
        if ($packId === '') $packId = (string)($settings['default_pack'] ?? 'default');
        if ($deckId <= 0) $deckId = (int)($settings['default_deck_id'] ?? 0);

        if ($deckId <= 0) {
            $deckId = DefaultDecks::get(Cards::SYSTEM_TAROT);
            if ($deckId <= 0) {
                $deckId = self::findFirstDeckForSystem(Cards::SYSTEM_TAROT);
            }
        }
        if ($deckId <= 0) return '';

        $cardIds = $cardsRaw !== '' ? array_values(array_filter(array_map('trim', explode(',', $cardsRaw)), static fn($v) => $v !== '')) : [];

        $cardMap = Cards::allMerged();
        $imgs = get_post_meta($deckId, '_dtarot_cards', true);
        if (!is_array($imgs)) $imgs = [];

        $deckSystem = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));

        self::enqueueFrontendAssets();

        $back = (string)get_post_meta($deckId, '_dtarot_back', true);

        $presetDef = class_exists(SpreadPresets::class) ? SpreadPresets::get($preset) : null;
        $presetSlots = ($presetDef && isset($presetDef['slots']) && is_array($presetDef['slots'])) ? $presetDef['slots'] : [];
        $pack = class_exists(SpreadMeaningPacks::class) ? SpreadMeaningPacks::get($packId) : null;
        $packSlots = (is_array($pack) && isset($pack['spreads'][$preset]['slots']) && is_array($pack['spreads'][$preset]['slots']))
            ? $pack['spreads'][$preset]['slots'] : [];

        if (!$cardIds) {
            $available = [];
            foreach ($imgs as $id => $url) {
                if (!is_string($id) || !is_string($url) || trim($url) === '') continue;

                // If the deck's stored image keys don't match its system IDs (legacy Gypsy/Kipper split),
                // map them so the selected card IDs use the deck's registry.
                if ($deckSystem === Cards::SYSTEM_GYPSY && preg_match('/^kipper_(\d{2})$/', $id, $m)) {
                    $id = 'gypsy_' . $m[1];
                } elseif ($deckSystem === Cards::SYSTEM_KIPPER && preg_match('/^gypsy_(\d{2})$/', $id, $m)) {
                    $id = 'kipper_' . $m[1];
                }
                $available[] = $id;
            }
            if (!$available) return '';
            $slotCount = count($presetSlots);
            if ($slotCount <= 0) $slotCount = 3;
            shuffle($available);
            $cardIds = array_slice($available, 0, $slotCount);
        }

        $overlapSlots = [];
        $hasPositions = false;
        $gridCols = 13;
        $gridRows = 13;
        $colSpan = 2;
        foreach ($presetSlots as $i => $slot) {
            if (!is_array($slot)) continue;
            if (isset($slot['x'], $slot['y'])) $hasPositions = true;
            if (empty($slot['overlap'])) continue;
            $overlapSlots[(int)$i] = $slot;
        }
        $hasOverlap = !empty($overlapSlots);

        $style = '--dtarot-cols:'.$cols.';';
        if ($cardWidth > 0) {
            if ($cardWidth < 120) $cardWidth = 120;
            if ($cardWidth > 320) $cardWidth = 320;
            $style .= '--dtarot-spread-card-w:'.$cardWidth.'px;';
        } elseif ($cardWidthOffset !== 0) {
            $style .= '--dtarot-spread-card-w:calc(clamp(160px, 22vw, 240px) - 56px + '.$cardWidthOffset.'px);';
        }
        if ($cardGap > 0) {
            if ($cardGap < 4) $cardGap = 4;
            if ($cardGap > 80) $cardGap = 80;
            $style .= '--dtarot-spread-gap:'.$cardGap.'px;';
        } elseif ($cardGapOffset !== 0) {
            $style .= '--dtarot-spread-gap:calc(2rem + '.$cardGapOffset.'px);';
        }
        if ($hasPositions) {
            $slotCount = count($presetSlots);
            $colSpan = 3;
            if ($slotCount >= 10) {
                $colSpan = 2;
            } elseif ($slotCount >= 7) {
                $colSpan = 2;
            }
            $style .= '--dtarot-grid-cols:'.$gridCols.';--dtarot-grid-rows:'.$gridRows.';--dtarot-col-span:'.$colSpan.';';
        }
        if ($hasOverlap) {
            $count = count($presetSlots);
            $heightMult = 3.2;
            if ($count >= 10) {
                $heightMult = 5.4;
            } elseif ($count >= 8) {
                $heightMult = 4.8;
            } elseif ($count >= 6) {
                $heightMult = 4.2;
            }
            $style .= '--dtarot-spread-height-mult:'.$heightMult.';';
        }

        $renderCard = static function (int $slotIndex, string $cardId, bool $overlapSlot, bool $renderImage) use ($cardMap, $imgs, $back, $link, $showLabels, $showTitles, $showMeanings, $packSlots, $presetSlots, $deckId, $hasPositions): string {
            if (!isset($cardMap[$cardId])) return '';

            $name = (string)$cardMap[$cardId];
            $url = esc_url(Shortcodes::resolveDeckCardUrlFromMap($imgs, (string)$cardId));
            $href = $link ? Shortcodes::cardDetailUrlFromDeck($deckId, (string)$cardId) : '';

            $styleAttr = '';
            $slotLabel = '';
            $slotMeaning = '';
            if (isset($packSlots[$slotIndex]) && is_array($packSlots[$slotIndex])) {
                $slotLabel = isset($packSlots[$slotIndex]['label']) && is_string($packSlots[$slotIndex]['label']) ? (string)$packSlots[$slotIndex]['label'] : '';
                $slotMeaning = isset($packSlots[$slotIndex]['meaning']) && is_string($packSlots[$slotIndex]['meaning']) ? (string)$packSlots[$slotIndex]['meaning'] : '';
            }
            if ($slotLabel === '' && isset($presetSlots[$slotIndex]['label']) && is_string($presetSlots[$slotIndex]['label'])) {
                $slotLabel = (string)$presetSlots[$slotIndex]['label'];
            }

            $styleAttr .= ' data-dtarot-slot="'.(int)$slotIndex.'"';
            if ($hasPositions && isset($presetSlots[$slotIndex]['x'], $presetSlots[$slotIndex]['y'])) {
                $x = (float)$presetSlots[$slotIndex]['x'];
                $y = (float)$presetSlots[$slotIndex]['y'];
                if ($x < 0) $x = 0;
                if ($x > 1) $x = 1;
                if ($y < 0) $y = 0;
                if ($y > 1) $y = 1;

                $gridCols = 13;
                $gridRows = 13;
                $col = (int)round($x * ($gridCols - 1)) + 1;
                $row = (int)round($y * ($gridRows - 1)) + 1;
                if ($col < 1) $col = 1;
                if ($col > ($gridCols - 1)) $col = $gridCols - 1;
                if ($row < 1) $row = 1;
                if ($row > $gridRows) $row = $gridRows;

                $styleAttr .= ' style="--dtarot-col:'.$col.';--dtarot-row:'.$row.';"';
            }

            $class = 'dtarot-spread-card';
            if ($overlapSlot) {
                $class .= ' is-overlap-slot';
            }
            $out = '<div class="'.esc_attr($class).'"'.$styleAttr.'>';
            if ($showLabels && $slotLabel !== '') {
                $out .= '<div class="dtarot-spread-label">' . esc_html($slotLabel) . '</div>';
            }
            $out .= '<div class="dtarot-spread-card-media">';
            if ($renderImage) {
                if ($url !== '') {
                    if ($back !== '') {
                        $out .= Shortcodes::renderFlip($url, $back, $href);
                    } else {
                        $img = '<img src="'.$url.'" alt="" />';
                        if ($href !== '') {
                            $img = '<a class="dtarot-card-link" href="' . esc_url($href) . '">' . $img . '</a>';
                        }
                        $out .= $img;
                    }
                } else {
                    $out .= '<div class="dtarot-spread-missing"></div>';
                }
            } else {
                $out .= '<div class="dtarot-spread-card-placeholder" aria-hidden="true"></div>';
            }
            $out .= '</div>';
            $out .= '<div class="dtarot-spread-card-meaning"><div class="dtarot-spread-card-meaning-inner">';
            if ($showTitles) {
                if ($href !== '') {
                    $out .= '<div class="dtarot-spread-title"><a class="dtarot-card-link" href="' . esc_url($href) . '">' . esc_html($name) . '</a></div>';
                } else {
                    $out .= '<div class="dtarot-spread-title">'.esc_html($name).'</div>';
                }
            }
            if ($showMeanings && $slotMeaning !== '') {
                $out .= '<div class="dtarot-spread-meaning">' . wp_kses_post($slotMeaning) . '</div>';
            }
            $out .= '</div></div>';
            $out .= '</div>';
            return $out;
        };

        $html = '<div class="dtarot-spread" style="'.esc_attr($style).'" data-preset="'.esc_attr($preset).'" data-pack="'.esc_attr($packId).'"' . ($hasOverlap ? ' data-has-overlap="1"' : '') . ($hasPositions ? ' data-has-positions="1"' : '') . '>';
        $html .= '<div class="dtarot-spread-body">';
        $html .= '<div class="dtarot-spread-grid">';

        $overlapItems = [];
        foreach ($cardIds as $slotIndex => $cardId) {
            $isOverlap = isset($overlapSlots[$slotIndex]);
            if ($isOverlap) {
                $overlapItems[] = [(int)$slotIndex, (string)$cardId];
            }
            $html .= $renderCard((int)$slotIndex, (string)$cardId, $isOverlap, true);
        }

        $html .= '</div>';

        if ($hasOverlap) {
            $html .= '<div class="dtarot-spread-overlap">';
            foreach ($overlapItems as $item) {
                $slotIndex = $item[0];
                $cardId = $item[1];
                if (!isset($presetSlots[$slotIndex]) || !is_array($presetSlots[$slotIndex])) continue;
                $slot = $presetSlots[$slotIndex];
                if (!isset($slot['x'], $slot['y'])) continue;

                $x = (float)$slot['x'];
                $y = (float)$slot['y'];
                if ($x < 0) $x = 0;
                if ($x > 1) $x = 1;
                if ($y < 0) $y = 0;
                if ($y > 1) $y = 1;
                $z = isset($slot['zIndex']) ? (int)$slot['zIndex'] : ($slotIndex + 1);

                if ($hasPositions) {
                    $col = (int)round($x * ($gridCols - 1)) + 1;
                    $row = (int)round($y * ($gridRows - 1)) + 1;
                    if ($col < 1) $col = 1;
                    if ($col > ($gridCols - 1)) $col = $gridCols - 1;
                    if ($row < 1) $row = 1;
                    if ($row > $gridRows) $row = $gridRows;

                    $x = ($col - 1 + ($colSpan / 2)) / ($gridCols - 1);
                    $y = ($row - 1 + 0.5) / ($gridRows - 1);
                    if ($x < 0) $x = 0;
                    if ($x > 1) $x = 1;
                    if ($y < 0) $y = 0;
                    if ($y > 1) $y = 1;
                }

                $name = isset($cardMap[$cardId]) ? (string)$cardMap[$cardId] : '';
                $url = esc_url(Shortcodes::resolveDeckCardUrlFromMap($imgs, (string)$cardId));
                $href = $link ? Shortcodes::cardDetailUrlFromDeck($deckId, (string)$cardId) : '';

                $html .= '<div class="dtarot-spread-overlap-card" data-dtarot-slot="'.(int)$slotIndex.'" style="--dtarot-x:'.$x.';--dtarot-y:'.$y.';--dtarot-z:'.$z.';">';
                $html .= '<div class="dtarot-spread-card-media">';
                if ($url !== '') {
                    if ($back !== '') {
                        $html .= Shortcodes::renderFlip($url, $back, $href);
                    } else {
                        $img = '<img src="'.$url.'" alt="" />';
                        if ($href !== '') {
                            $img = '<a class="dtarot-card-link" href="' . esc_url($href) . '">' . $img . '</a>';
                        }
                        $html .= $img;
                    }
                } else {
                    $html .= '<div class="dtarot-spread-missing"></div>';
                }
                $html .= '</div>';
                $html .= '</div>';
            }
            $html .= '</div>';
        }

        $html .= '</div>';
        $html .= '</div>';
        return $html;
    }

    public static function render_booking($atts = []): string {
        self::enqueueFrontendAssets();

        $types = get_posts([
            'post_type' => PostTypes::readingTypeTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'menu_order title',
            'order' => 'ASC',
        ]);

        if (!$types) {
            return '<div class="dtarot-frontend dtarot-frontend-empty">' . esc_html__('No reading types available yet.','daily-tarot') . '</div>';
        }

        $settings = class_exists(BookingSettings::class) ? BookingSettings::get() : [];
        $mode = isset($settings['mode']) ? (string)$settings['mode'] : 'request';
        $paymentMode = isset($settings['payment_mode']) ? (string)$settings['payment_mode'] : 'none';
        $paymentProvider = isset($settings['payment_provider']) ? (string)$settings['payment_provider'] : 'paypal';
        $paypalUrl = isset($settings['paypal_url']) ? (string)$settings['paypal_url'] : '';
        $stripeUrl = isset($settings['stripe_url']) ? (string)$settings['stripe_url'] : '';
        $ctaTitle = ($mode === 'instant') ? __('Book Instantly','daily-tarot') : __('Request a Reading','daily-tarot');
        $styleOverride = is_array($atts) && isset($atts['style']) ? sanitize_key((string)$atts['style']) : '';
        $style = $styleOverride !== '' ? $styleOverride : (isset($settings['style']) ? sanitize_key((string)$settings['style']) : 'modern');
        if (!in_array($style, ['modern','mystic','minimal'], true)) {
            $style = 'modern';
        }

        $notice = isset($_GET['dtarot_booking']) ? sanitize_key((string)wp_unslash($_GET['dtarot_booking'])) : '';
        $msg = '';
        if ($notice === 'ok') {
            $msg = __('Your request has been received. We will follow up soon.','daily-tarot');
            if ($mode === 'instant') {
                $msg = __('Your reading is confirmed. Check your email for details.','daily-tarot');
            }
        } elseif ($notice === 'error') {
            $msg = __('We could not submit your request. Please try another time.','daily-tarot');
        }

        ob_start();
        ?>
        <div class="dtarot-frontend dtarot-booking dtarot-booking-style-<?php echo esc_attr($style); ?>">
            <h2 class="dtarot-booking-title"><?php echo esc_html($ctaTitle); ?></h2>
            <?php if ($msg !== ''): ?>
                <div class="dtarot-booking-note"><?php echo esc_html($msg); ?></div>
            <?php endif; ?>
            <form class="dtarot-booking-form" method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('dtarot_booking_submit', 'dtarot_booking_nonce'); ?>
                <input type="hidden" name="action" value="dtarot_booking_submit" />
                <input type="hidden" name="booking_start_utc" value="" />
                <input type="hidden" name="booking_timezone" value="" />
                <input type="hidden" name="payment_provider" value="<?php echo esc_attr($paymentProvider); ?>" />

                <div class="dtarot-booking-section">
                    <h3><?php esc_html_e('Choose a reading','daily-tarot'); ?></h3>
                    <div class="dtarot-booking-types">
                        <?php foreach ($types as $t) :
                            $duration = (int)get_post_meta($t->ID, '_dtarot_reading_duration', true);
                            if ($duration <= 0) $duration = 30;
                            $price = (string)get_post_meta($t->ID, '_dtarot_reading_price', true);
                            $icon = (string)get_post_meta($t->ID, '_dtarot_reading_icon', true);
                            $desc = (string)$t->post_content;
                        ?>
                            <label class="dtarot-booking-type">
                                <input type="radio" name="reading_type" value="<?php echo (int)$t->ID; ?>" required />
                                <span class="dtarot-booking-type-card">
                                    <span class="dtarot-booking-type-title">
                                        <?php if ($icon !== ''): ?><span class="dtarot-booking-type-icon"><?php echo esc_html($icon); ?></span><?php endif; ?>
                                        <?php echo esc_html($t->post_title); ?>
                                    </span>
                                    <span class="dtarot-booking-type-meta">
                                        <?php echo esc_html(sprintf(__('%d mins','daily-tarot'), $duration)); ?>
                                        <?php if ($price !== ''): ?>
                                            <span class="dtarot-booking-type-price"><?php echo esc_html($price); ?></span>
                                        <?php endif; ?>
                                    </span>
                                    <?php if ($desc !== ''): ?>
                                        <span class="dtarot-booking-type-desc"><?php echo esc_html(wp_strip_all_tags($desc)); ?></span>
                                    <?php endif; ?>
                                </span>
                            </label>
                        <?php endforeach; ?>
                    </div>
                </div>

                <div class="dtarot-booking-section">
                    <h3><?php esc_html_e('Pick a time','daily-tarot'); ?></h3>
                    <div class="dtarot-booking-row">
                        <label>
                            <span><?php esc_html_e('Date','daily-tarot'); ?></span>
                            <input type="date" name="booking_date" required />
                        </label>
                        <label>
                            <span><?php esc_html_e('Time','daily-tarot'); ?></span>
                            <select name="booking_time" required>
                                <option value=""><?php esc_html_e('Select a time','daily-tarot'); ?></option>
                            </select>
                        </label>
                    </div>
                    <p class="description"><?php esc_html_e('Times are shown in your timezone.','daily-tarot'); ?></p>
                </div>

                <?php if ($paymentMode !== 'none') : ?>
                    <div class="dtarot-booking-section">
                        <h3><?php esc_html_e('Payment','daily-tarot'); ?></h3>
                        <?php if ($paymentMode === 'after') : ?>
                            <p class="description"><?php esc_html_e('You will receive a payment link after your reading is confirmed.','daily-tarot'); ?></p>
                        <?php else : ?>
                            <p class="description"><?php esc_html_e('Please complete payment before submitting your request.','daily-tarot'); ?></p>
                            <div class="dtarot-booking-pay">
                                <?php if ($paymentProvider === 'paypal' || $paymentProvider === 'both') : ?>
                                    <?php if ($paypalUrl !== '') : ?>
                                        <a class="dtarot-booking-pay-btn" href="<?php echo esc_url($paypalUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Pay with PayPal','daily-tarot'); ?></a>
                                    <?php endif; ?>
                                <?php endif; ?>
                                <?php if ($paymentProvider === 'stripe' || $paymentProvider === 'both') : ?>
                                    <?php if ($stripeUrl !== '') : ?>
                                        <a class="dtarot-booking-pay-btn" href="<?php echo esc_url($stripeUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Pay with Stripe','daily-tarot'); ?></a>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </div>
                            <?php if ($paymentProvider === 'both') : ?>
                                <label class="dtarot-booking-full">
                                    <span><?php esc_html_e('Payment method used','daily-tarot'); ?></span>
                                    <select name="payment_provider">
                                        <option value="paypal"><?php esc_html_e('PayPal','daily-tarot'); ?></option>
                                        <option value="stripe"><?php esc_html_e('Stripe','daily-tarot'); ?></option>
                                    </select>
                                </label>
                            <?php endif; ?>
                            <label class="dtarot-booking-full">
                                <span><?php esc_html_e('Payment reference (optional)','daily-tarot'); ?></span>
                                <input type="text" name="payment_ref" />
                            </label>
                            <label class="dtarot-booking-consent">
                                <input type="checkbox" name="payment_confirm" value="1" required />
                                <span><?php esc_html_e('I have completed the payment.','daily-tarot'); ?></span>
                            </label>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <div class="dtarot-booking-section">
                    <h3><?php esc_html_e('Your details','daily-tarot'); ?></h3>
                    <div class="dtarot-booking-row">
                        <label>
                            <span><?php esc_html_e('Name','daily-tarot'); ?></span>
                            <input type="text" name="booking_name" required />
                        </label>
                        <label>
                            <span><?php esc_html_e('Email','daily-tarot'); ?></span>
                            <input type="email" name="booking_email" required />
                        </label>
                    </div>
                    <label class="dtarot-booking-full">
                        <span><?php esc_html_e('Question or topic (optional)','daily-tarot'); ?></span>
                        <textarea name="booking_question" rows="4"></textarea>
                    </label>
                    <label class="dtarot-booking-consent">
                        <input type="checkbox" name="booking_consent" value="1" required />
                        <span><?php esc_html_e('Tarot readings are for guidance only and not medical, legal, or financial advice.','daily-tarot'); ?></span>
                    </label>
                </div>

                <div class="dtarot-booking-actions">
                    <button type="submit" class="dtarot-booking-submit"><?php echo esc_html($ctaTitle); ?></button>
                </div>
            </form>
        </div>
        <?php
        return (string)ob_get_clean();
    }

    public static function render_booking_button($atts = []): string {
        $atts = is_array($atts) ? $atts : [];
        $label = isset($atts['label']) ? sanitize_text_field((string)$atts['label']) : '';
        if ($label === '') {
            $label = __('Request a Reading','daily-tarot');
        }
        $style = isset($atts['style']) ? sanitize_key((string)$atts['style']) : '';
        if ($style !== '' && !in_array($style, ['modern','mystic','minimal'], true)) {
            $style = '';
        }

        $id = 'dtarot-booking-modal-' . wp_generate_password(6, false, false);
        $modalClass = 'dtarot-booking-modal';

        $form = self::render_booking($style !== '' ? ['style' => $style] : []);

        return '<div class="dtarot-booking-trigger">' .
            '<button type="button" class="dtarot-booking-open" data-dtarot-modal-open="' . esc_attr($id) . '">' . esc_html($label) . '</button>' .
            '</div>' .
            '<div id="' . esc_attr($id) . '" class="' . esc_attr($modalClass) . '" role="dialog" aria-modal="true" aria-hidden="true">' .
                '<div class="dtarot-booking-modal-backdrop" data-dtarot-modal-close="' . esc_attr($id) . '"></div>' .
                '<div class="dtarot-booking-modal-panel">' .
                    '<button type="button" class="dtarot-booking-modal-close" data-dtarot-modal-close="' . esc_attr($id) . '" aria-label="' . esc_attr__('Close','daily-tarot') . '">×</button>' .
                    $form .
                '</div>' .
            '</div>';
    }

    public static function render_booking_teaser($atts = []): string {
        $atts = is_array($atts) ? $atts : [];
        $title = isset($atts['title']) ? sanitize_text_field((string)$atts['title']) : '';
        if ($title === '') {
            $title = __('Daily Tarot Reading','daily-tarot');
        }
        $text = isset($atts['text']) ? sanitize_text_field((string)$atts['text']) : '';
        if ($text === '') {
            $text = __('Get clarity and guidance from a personal tarot reading.','daily-tarot');
        }
        $label = isset($atts['label']) ? sanitize_text_field((string)$atts['label']) : '';
        if ($label === '') {
            $label = __('Request a Reading','daily-tarot');
        }
        $style = isset($atts['style']) ? sanitize_key((string)$atts['style']) : '';
        if ($style !== '' && !in_array($style, ['modern','mystic','minimal'], true)) {
            $style = '';
        }

        $types = get_posts([
            'post_type' => PostTypes::readingTypeTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'menu_order title',
            'order' => 'ASC',
        ]);

        if (!$types) {
            return '<div class="dtarot-frontend dtarot-frontend-empty">' . esc_html__('No reading types available yet.','daily-tarot') . '</div>';
        }

        $id = 'dtarot-booking-modal-' . wp_generate_password(6, false, false);
        $form = self::render_booking($style !== '' ? ['style' => $style] : []);

        $list = '<ul class="dtarot-booking-teaser-list">';
        foreach ($types as $t) {
            $duration = (int)get_post_meta($t->ID, '_dtarot_reading_duration', true);
            if ($duration <= 0) $duration = 30;
            $list .= '<li>' . esc_html($t->post_title) . ' <span>(' . (int)$duration . ' ' . esc_html__('mins','daily-tarot') . ')</span></li>';
        }
        $list .= '</ul>';

        $class = 'dtarot-frontend dtarot-booking-teaser';
        if ($style !== '') {
            $class .= ' dtarot-booking-teaser-style-' . $style;
        }

        return '<div class="' . esc_attr($class) . '">' .
            '<div class="dtarot-booking-teaser-title">' . esc_html($title) . '</div>' .
            '<div class="dtarot-booking-teaser-text">' . esc_html($text) . '</div>' .
            $list .
            '<button type="button" class="dtarot-booking-teaser-btn" data-dtarot-modal-open="' . esc_attr($id) . '">' . esc_html($label) . '</button>' .
            '</div>' .
            '<div id="' . esc_attr($id) . '" class="dtarot-booking-modal" role="dialog" aria-modal="true" aria-hidden="true">' .
                '<div class="dtarot-booking-modal-backdrop" data-dtarot-modal-close="' . esc_attr($id) . '"></div>' .
                '<div class="dtarot-booking-modal-panel">' .
                    '<button type="button" class="dtarot-booking-modal-close" data-dtarot-modal-close="' . esc_attr($id) . '" aria-label="' . esc_attr__('Close','daily-tarot') . '">×</button>' .
                    $form .
                '</div>' .
            '</div>';
    }
}
