<?php
declare(strict_types=1);

namespace DailyTarot\Frontend;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Analytics\ShareStore;
use DailyTarot\Support\RateLimit;
use DailyTarot\Support\Log;

final class ShareTracking {

    public static function init(): void {
        add_action('wp_ajax_dtarot_share_track', [__CLASS__, 'track']);
        add_action('wp_ajax_nopriv_dtarot_share_track', [__CLASS__, 'track']);
    }

    public static function track(): void {
        if (class_exists(RateLimit::class) && !RateLimit::hit('share_track', 30, 60)) {
            if (class_exists(Log::class)) {
                Log::add('warn', 'rate_limited', 'Share tracking rate limited', ['action' => 'share_track']);
            }
            header('Retry-After: ' . (string)RateLimit::defaultRetryAfterSeconds(60));
            wp_send_json_error(['message' => 'rate_limited'], 429);
        }

        $nonce = isset($_POST['nonce']) ? sanitize_text_field((string)wp_unslash($_POST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'dtarot_share')) {
            wp_send_json_error(['message' => 'bad nonce'], 403);
        }

        if (!class_exists(ShareStore::class)) {
            wp_send_json_error(['message' => 'unavailable'], 503);
        }

        $event = [
            'context' => isset($_POST['context']) ? sanitize_key((string)wp_unslash($_POST['context'])) : '',
            'platform' => isset($_POST['platform']) ? sanitize_key((string)wp_unslash($_POST['platform'])) : '',
            'url' => isset($_POST['url']) ? esc_url_raw((string)wp_unslash($_POST['url'])) : '',
            'page' => isset($_POST['page']) ? sanitize_text_field((string)wp_unslash($_POST['page'])) : '',
            'date' => isset($_POST['date']) ? sanitize_text_field((string)wp_unslash($_POST['date'])) : '',
            'post_id' => isset($_POST['post_id']) ? absint($_POST['post_id']) : 0,
            'deck_id' => isset($_POST['deck_id']) ? absint($_POST['deck_id']) : 0,
            'card_id' => isset($_POST['card_id']) ? sanitize_text_field((string)wp_unslash($_POST['card_id'])) : '',
            'preset' => isset($_POST['preset']) ? sanitize_key((string)wp_unslash($_POST['preset'])) : '',
            'pack' => isset($_POST['pack']) ? sanitize_key((string)wp_unslash($_POST['pack'])) : '',
        ];

        $ok = ShareStore::record($event);
        if (!$ok) {
            wp_send_json_error(['message' => 'save_failed'], 500);
        }

        wp_send_json_success(['message' => 'ok']);
    }
}
