<?php
declare(strict_types=1);


namespace DailyTarot\Frontend;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Meaning\MeaningPackRepository;
use DailyTarot\Reading\ReadingComposer;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\DefaultMeaningPacks;
use DailyTarot\Support\DefaultDecks;
use DailyTarot\Support\PostTypes;
use DailyTarot\Support\OnlineVisitors;
use DailyTarot\Support\ShortcodeSettings;
use DailyTarot\Support\ShareImageSettings;
use DailyTarot\Support\RelatedLinks;
use DailyTarot\Support\CacheVersion;

final class ReadableRoutes {

    private static function cacheVer(): int {
        return class_exists(CacheVersion::class) ? CacheVersion::get() : 1;
    }

    private static function maybeHandleConditionalGet(string $etag, int $lastModifiedUnixTs): void {
        if ($etag !== '') {
            header('ETag: ' . $etag);
        }

        if ($lastModifiedUnixTs > 0) {
            header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $lastModifiedUnixTs) . ' GMT');
        }

        $ims = isset($_SERVER['HTTP_IF_MODIFIED_SINCE']) ? (string)$_SERVER['HTTP_IF_MODIFIED_SINCE'] : '';
        if ($lastModifiedUnixTs > 0 && $ims !== '') {
            $imsTs = strtotime($ims);
            if (is_int($imsTs) && $imsTs >= $lastModifiedUnixTs) {
                status_header(304);
                exit;
            }
        }

        $inm = isset($_SERVER['HTTP_IF_NONE_MATCH']) ? (string)$_SERVER['HTTP_IF_NONE_MATCH'] : '';
        if ($etag !== '' && $inm !== '' && trim($inm) === $etag) {
            status_header(304);
            exit;
        }
    }

    /**
     * @return array<int,array{deck_id:int,title:string,img_url:string,is_active:bool}>
     */
    private static function getDeckVariantsForCard(string $system, string $cardId, int $activeDeckId): array {
        $system = Cards::normalizeSystem($system);
        $cardId = sanitize_text_field($cardId);
        if ($system === '' || $cardId === '') return [];

        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);

        $out = [];
        foreach ((array)$decks as $p) {
            if (!isset($p->ID)) continue;
            $deckId = (int)$p->ID;
            $ps = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
            if ($ps !== $system) continue;

            $imgs = get_post_meta($deckId, '_dtarot_cards', true);
            if (!is_array($imgs) || empty($imgs[$cardId]) || !is_string($imgs[$cardId])) continue;

            $url = trim((string)$imgs[$cardId]);
            if ($url === '') continue;

            $out[] = [
                'deck_id' => $deckId,
                'title' => (string)(get_the_title($deckId) ?: ''),
                'img_url' => esc_url($url),
                'is_active' => ($activeDeckId > 0 && $deckId === $activeDeckId),
            ];
        }

        // Put the active deck first.
        usort($out, function (array $a, array $b): int {
            if (!empty($a['is_active']) && empty($b['is_active'])) return -1;
            if (empty($a['is_active']) && !empty($b['is_active'])) return 1;
            return 0;
        });

        return $out;
    }

    private static function shareExcerpt(string $content, int $maxChars = 200): string {
        $content = trim((string)wp_strip_all_tags($content));
        if ($content === '') return '';
        $content = preg_replace('/\s+/', ' ', $content) ?? $content;
        if (mb_strlen($content) <= $maxChars) return $content;
        return mb_substr($content, 0, $maxChars - 3) . '...';
    }

    /**
     * @return array<int,array{card_id:string,name:string,img_url:string,url:string,is_current:bool}>
     */
    private static function getOtherCardsFromDeck(string $system, int $deckId, string $currentCardId): array {
        $system = Cards::normalizeSystem($system);
        $currentCardId = sanitize_text_field($currentCardId);
        if ($system === '' || $deckId <= 0) return [];

        $imgs = get_post_meta($deckId, '_dtarot_cards', true);
        if (!is_array($imgs)) $imgs = [];

        $cards = Cards::forSystem($system);
        if (!$cards) return [];

        $all = [];
        foreach ($cards as $cardId => $name) {
            if (!is_string($cardId)) continue;
            if (empty($imgs[$cardId]) || !is_string($imgs[$cardId])) continue;
            $imgUrl = trim((string)$imgs[$cardId]);
            if ($imgUrl === '') continue;

            $url = add_query_arg(['deck_id' => (string)$deckId], self::urlForCard($system, (string)$cardId));
            $all[] = [
                'card_id' => (string)$cardId,
                'name' => (string)$name,
                'img_url' => esc_url($imgUrl),
                'url' => esc_url($url),
                'is_current' => ($currentCardId !== '' && (string)$cardId === $currentCardId),
            ];
        }

        // Rotate list so the first items are "new" (after the current card), and remove the current card.
        $idx = -1;
        for ($i = 0; $i < count($all); $i++) {
            if (!empty($all[$i]['is_current'])) {
                $idx = $i;
                break;
            }
        }

        if ($idx >= 0 && count($all) > 1) {
            $after = array_slice($all, $idx + 1);
            $before = array_slice($all, 0, $idx);
            $all = array_merge($after, $before);
        }

        $out = [];
        foreach ($all as $row) {
            if (!empty($row['is_current'])) continue;
            $out[] = $row;
        }
        return $out;
    }

    public static function init(): void {
        add_action('init', [__CLASS__, 'register_rewrites']);
        add_filter('query_vars', [__CLASS__, 'add_query_vars']);
        add_filter('redirect_canonical', [__CLASS__, 'maybe_disable_canonical_redirect'], 10, 2);
        add_filter('pre_get_document_title', [__CLASS__, 'maybe_override_title']);
        add_action('template_redirect', [__CLASS__, 'maybe_404']);
        add_action('template_redirect', [__CLASS__, 'maybe_render_readable'], 11);
        add_action('template_redirect', [__CLASS__, 'maybe_render_card'], 11);
        add_action('wp_head', [__CLASS__, 'maybe_output_meta'], 1);
    }

    public static function slug(): string {
        $slug = apply_filters('dtarot_readable_slug', 'card-of-the-day');
        $slug = is_string($slug) ? trim($slug) : 'card-of-the-day';
        $slug = trim($slug, "/ \t\n\r\0\x0B");
        if ($slug === '') $slug = 'card-of-the-day';
        return $slug;
    }

    public static function isReadableRequest(): bool {
        $mode = get_query_var('dtarot');
        $date = get_query_var('dtarot_date');
        return (is_string($mode) && $mode === 'readable') && (is_string($date) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $date));
    }

    public static function isCardRequest(): bool {
        $mode = get_query_var('dtarot');
        $system = get_query_var('dtarot_system');
        $card = get_query_var('dtarot_card');

        if (!is_string($mode) || $mode !== 'card') return false;
        if (!is_string($system) || $system === '') return false;
        if (!is_string($card) || $card === '') return false;

        return true;
    }

    public static function cardsSlug(): string {
        $slug = apply_filters('dtarot_cards_slug', 'cards');
        $slug = is_string($slug) ? trim($slug) : 'cards';
        $slug = trim($slug, "/ \t\n\r\0\x0B");
        if ($slug === '') $slug = 'cards';
        return $slug;
    }

    public static function urlForCard(string $system, string $cardId, string $baseUrl = ''): string {
        $system = Cards::normalizeSystem($system);
        $cardId = sanitize_text_field($cardId);
        if ($system === '' || $cardId === '') return '';

        $cards = Cards::forSystem($system);
        if (!isset($cards[$cardId])) return '';

        if ($baseUrl !== '') {
            $baseUrl = trim($baseUrl);
            $parts = wp_parse_url($baseUrl);
            if (is_array($parts) && !empty($parts['scheme']) && !empty($parts['host'])) {
                $path = isset($parts['path']) ? (string)$parts['path'] : '';
                $path = rtrim($path, '/');
                $rebuilt = $parts['scheme'].'://'.$parts['host'];
                if (!empty($parts['port'])) $rebuilt .= ':' . (int)$parts['port'];
                $rebuilt .= $path . '/';
                return $rebuilt . rawurlencode($system) . '/' . rawurlencode($cardId) . '/';
            }

            $baseUrl = preg_replace('/[?#].*$/', '', $baseUrl);
            $baseUrl = rtrim((string)$baseUrl, '/');
            return $baseUrl . '/' . rawurlencode($system) . '/' . rawurlencode($cardId) . '/';
        }

        return home_url(trailingslashit(self::cardsSlug()) . rawurlencode($system) . '/' . rawurlencode($cardId) . '/');
    }

    public static function renderCardEmbed(string $system, string $cardId, int $deckId = 0, int $packId = 0): string {
        $system = Cards::normalizeSystem($system);
        $cardId = sanitize_text_field($cardId);
        if ($system === '' || $cardId === '') return '';

        $cards = Cards::forSystem($system);
        if (!isset($cards[$cardId])) return '';

        if ($deckId > 0 && !self::isDeckForSystem($deckId, $system)) {
            $deckId = 0;
        }
        if ($packId > 0 && !self::isPackForSystem($packId, $system)) {
            $packId = 0;
        }

        if ($deckId <= 0) {
            $deckId = self::findFirstDeckForSystem($system);
        }
        if ($packId <= 0) {
            $packId = self::findFirstPackForSystem($system);
        }

        // Ensure our frontend CSS/JS is available.
        wp_enqueue_style('dtarot-frontend', DTAROT_URL.'assets/frontend.css', [], DTAROT_VERSION);
        $jsFile = DTAROT_PATH . 'assets/frontend.js';
        $ver = is_readable($jsFile) ? (string)@filemtime($jsFile) : DTAROT_VERSION;
        wp_enqueue_script('dtarot-frontend-js', DTAROT_URL.'assets/frontend.js', [], $ver, true);

        $meaning = $packId > 0 ? MeaningPackRepository::getMeaning($packId, $cardId) : MeaningPackRepository::emptyMeaning();
        $fallback = ReadingComposer::applyMeaningFallback('', '', $meaning);

        return self::renderCardDetailHtml($system, $cardId, $deckId, $packId, $meaning, $fallback['content'], $fallback['daily_text'], 'h2', '');
    }

    /**
     * Pretty URL builder for readable pages.
     *
     * If $baseUrl is empty, uses /{slug}/{YYYY-MM-DD}/.
     */
    public static function urlForDate(string $date, string $baseUrl = ''): string {
        $date = trim($date);
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            return '';
        }

        if ($baseUrl !== '') {
            $baseUrl = trim($baseUrl);
            // Strip query/hash so we always output a clean canonical URL.
            $parts = wp_parse_url($baseUrl);
            if (is_array($parts) && !empty($parts['scheme']) && !empty($parts['host'])) {
                $path = isset($parts['path']) ? (string)$parts['path'] : '';
                $path = rtrim($path, '/');
                $rebuilt = $parts['scheme'].'://'.$parts['host'];
                if (!empty($parts['port'])) $rebuilt .= ':' . (int)$parts['port'];
                $rebuilt .= $path . '/';
                return $rebuilt . $date . '/';
            }

            $baseUrl = preg_replace('/[?#].*$/', '', $baseUrl);
            $baseUrl = rtrim((string)$baseUrl, '/');
            return $baseUrl . '/' . $date . '/';
        }

        return home_url(trailingslashit(self::slug()) . $date . '/');
    }

    public static function register_rewrites(): void {
        $slug = self::slug();
        // /{slug}/YYYY-MM-DD/
        add_rewrite_rule(
            '^' . preg_quote($slug, '/') . '/(\\d{4}-\\d{2}-\\d{2})/?$',
            'index.php?dtarot=readable&dtarot_date=$matches[1]',
            'top'
        );

        // /cards/{system}/{card_id}/
        $cardsSlug = self::cardsSlug();
        add_rewrite_rule(
            '^' . preg_quote($cardsSlug, '/') . '/([a-z0-9_-]+)/([a-z0-9_-]+)/?$',
            'index.php?dtarot=card&dtarot_system=$matches[1]&dtarot_card=$matches[2]',
            'top'
        );
    }

    /** @param array<int,string> $vars */
    public static function add_query_vars(array $vars): array {
        $vars[] = 'dtarot';
        $vars[] = 'dtarot_date';
        $vars[] = 'dtarot_system';
        $vars[] = 'dtarot_card';
        return $vars;
    }

    /**
     * Prevent WP from "helpfully" redirecting our dated URL back to the base page.
     */
    public static function maybe_disable_canonical_redirect($redirectUrl, $requestedUrl) {
        if (self::isReadableRequest() || self::isCardRequest()) {
            return false;
        }
        return $redirectUrl;
    }

    public static function maybe_override_title(string $title): string {
        if (self::isReadableRequest()) {
            $date = (string)get_query_var('dtarot_date');
            $entry = DayEntryService::getPublished($date);
            if (!$entry) {
                return $title;
            }

            $entryArr = $entry->toArray();
            $cardId = (string)($entryArr['card'] ?? '');
            $cardName = Cards::name($cardId);

            $ts = strtotime($date . ' 00:00:00');
            $pretty = $date;
            if ($ts !== false) {
                $pretty = function_exists('wp_date') ? (string)wp_date('F j, Y', $ts) : (string)date_i18n('F j, Y', $ts);
            }

            return $cardName . ' — ' . $pretty;
        }

        if (self::isCardRequest()) {
            [$system, $cardId] = self::getCardRequestVars();
            if ($system === '' || $cardId === '') return $title;
            return Cards::name($cardId);
        }

        return $title;
    }

    public static function maybe_404(): void {
        if (self::isReadableRequest()) {
            $date = (string)get_query_var('dtarot_date');
            $entry = DayEntryService::getPublished($date);
            if ($entry) return;

            // Optional fallback: redirect to the latest published day.
            $fallbackEnabled = (bool)apply_filters('dtarot_readable_fallback_latest', true);
            if ($fallbackEnabled) {
                $latest = DayEntryService::latestPublished(1);
                if ($latest) {
                    $keys = array_keys($latest);
                    $latestDate = isset($keys[0]) && is_string($keys[0]) ? (string)$keys[0] : '';
                    $to = $latestDate !== '' ? self::urlForDate($latestDate) : '';
                    if ($to !== '') {
                        $code = (int)apply_filters('dtarot_readable_fallback_status', 302);
                        if ($code < 300 || $code > 399) $code = 302;
                        wp_safe_redirect($to, $code);
                        exit;
                    }
                }
            }

            global $wp_query;
            if ($wp_query) {
                $wp_query->set_404();
            }
            status_header(404);
            nocache_headers();
            return;
        }

        if (self::isCardRequest()) {
            if (self::isValidCardRequest()) return;

            global $wp_query;
            if ($wp_query) {
                $wp_query->set_404();
            }
            status_header(404);
            nocache_headers();
            return;
        }
    }

    public static function maybe_output_meta(): void {
        // If an SEO plugin is active, avoid duplicate meta output.
        if (defined('WPSEO_VERSION') || class_exists('WPSEO_Options') || defined('RANK_MATH_VERSION') || class_exists('RankMath\\Helper')) {
            return;
        }

        if (self::isReadableRequest()) {
            $date = (string)get_query_var('dtarot_date');
            $entry = DayEntryService::getPublished($date);
            if (!$entry) return;

            $entryArr = $entry->toArray();

            $canonical = self::urlForDate($date);
            if ($canonical !== '') {
                echo '<link rel="canonical" href="' . esc_url($canonical) . '" />' . "\n";
            }

            $packId = (int)($entryArr['pack'] ?? 0);
            $cardId = (string)($entryArr['card'] ?? '');
            $meaning = MeaningPackRepository::getMeaning($packId, $cardId);
            $desc = ReadingComposer::buildMetaDescription($entryArr, $meaning, 160);
            if ($desc !== '') echo '<meta name="description" content="' . esc_attr($desc) . '" />' . "\n";

            $deckId = (int)($entryArr['deck'] ?? 0);
            $imgUrl = '';
            $overrideUrl = isset($entryArr['image_override_url']) ? (string)$entryArr['image_override_url'] : '';
            if ($overrideUrl !== '') {
                $imgUrl = esc_url($overrideUrl);
            } elseif ($deckId > 0 && $cardId !== '') {
                $imgs = get_post_meta($deckId, '_dtarot_cards', true);
                if (is_array($imgs) && $imgs) {
                    foreach (Cards::kipperGypsyAliases($cardId) as $id) {
                        if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
                        $u = trim((string)$imgs[$id]);
                        if ($u !== '') {
                            $imgUrl = esc_url($u);
                            break;
                        }
                    }
                }
            }

            $title = Cards::name($cardId);
            $url = $canonical;

            echo '<meta property="og:type" content="article" />' . "\n";
            echo '<meta property="og:title" content="' . esc_attr($title) . '" />' . "\n";
            if ($desc !== '') echo '<meta property="og:description" content="' . esc_attr($desc) . '" />' . "\n";
            if ($url !== '') echo '<meta property="og:url" content="' . esc_url($url) . '" />' . "\n";
            if ($imgUrl !== '') echo '<meta property="og:image" content="' . esc_url($imgUrl) . '" />' . "\n";

            echo '<meta name="twitter:card" content="' . esc_attr($imgUrl !== '' ? 'summary_large_image' : 'summary') . '" />' . "\n";
            echo '<meta name="twitter:title" content="' . esc_attr($title) . '" />' . "\n";
            if ($desc !== '') echo '<meta name="twitter:description" content="' . esc_attr($desc) . '" />' . "\n";
            if ($imgUrl !== '') echo '<meta name="twitter:image" content="' . esc_url($imgUrl) . '" />' . "\n";
            return;
        }

        if (self::isCardRequest()) {
            if (!self::isValidCardRequest()) return;

            [$system, $cardId] = self::getCardRequestVars();
            $canonical = self::urlForCard($system, $cardId);
            if ($canonical !== '') {
                echo '<link rel="canonical" href="' . esc_url($canonical) . '" />' . "\n";
            }

            [$deckId, $packId] = self::getCardOverrides($system);
            if ($deckId <= 0) $deckId = self::findFirstDeckForSystem($system);
            if ($packId <= 0) $packId = self::findFirstPackForSystem($system);

            $meaning = $packId > 0 ? MeaningPackRepository::getMeaning($packId, $cardId) : MeaningPackRepository::emptyMeaning();
            $desc = ReadingComposer::buildMetaDescription([], $meaning, 160);
            if ($desc !== '') echo '<meta name="description" content="' . esc_attr($desc) . '" />' . "\n";

            $imgUrl = '';
            if ($deckId > 0) {
                $imgs = get_post_meta($deckId, '_dtarot_cards', true);
                if (is_array($imgs) && $imgs) {
                    foreach (Cards::kipperGypsyAliases($cardId) as $id) {
                        if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
                        $u = trim((string)$imgs[$id]);
                        if ($u !== '') {
                            $imgUrl = esc_url($u);
                            break;
                        }
                    }
                }
            }

            $title = Cards::name($cardId);
            $url = $canonical;

            echo '<meta property="og:type" content="article" />' . "\n";
            echo '<meta property="og:title" content="' . esc_attr($title) . '" />' . "\n";
            if ($desc !== '') echo '<meta property="og:description" content="' . esc_attr($desc) . '" />' . "\n";
            if ($url !== '') echo '<meta property="og:url" content="' . esc_url($url) . '" />' . "\n";
            if ($imgUrl !== '') echo '<meta property="og:image" content="' . esc_url($imgUrl) . '" />' . "\n";

            echo '<meta name="twitter:card" content="' . esc_attr($imgUrl !== '' ? 'summary_large_image' : 'summary') . '" />' . "\n";
            echo '<meta name="twitter:title" content="' . esc_attr($title) . '" />' . "\n";
            if ($desc !== '') echo '<meta name="twitter:description" content="' . esc_attr($desc) . '" />' . "\n";
            if ($imgUrl !== '') echo '<meta name="twitter:image" content="' . esc_url($imgUrl) . '" />' . "\n";
            return;
        }
    }

    public static function maybe_render_readable(): void {
        if (!self::isReadableRequest()) return;

        $date = (string)get_query_var('dtarot_date');
        $entry = DayEntryService::getPublished($date);
        if (!$entry) return; // maybe_404 will handle.

        // Cache the full rendered HTML for performance.
        // Keyed by date + dtarot_cache_ver so updates invalidate naturally.
        $cacheKey = 'dtarot_readable_html_' . self::cacheVer() . '_' . $date;
        $cachedHtml = get_transient($cacheKey);
        if (is_string($cachedHtml) && $cachedHtml !== '') {
            $etag = 'W/"dtarot-readable-' . md5($cachedHtml) . '"';
            $lastmod = class_exists(CacheVersion::class) ? CacheVersion::getDayLastModified($date) : 0;
            self::maybeHandleConditionalGet($etag, $lastmod);

            status_header(200);
            nocache_headers();
            global $wp_query;
            if ($wp_query) {
                $wp_query->is_404 = false;
            }
            get_header();
            echo '<main id="primary" class="site-main">' . $cachedHtml . '</main>';
            get_footer();
            exit;
        }

        $entryArr = $entry->toArray();

        $cardId = (string)($entryArr['card'] ?? '');
        $deckId = (int)($entryArr['deck'] ?? 0);
        $packId = (int)($entryArr['pack'] ?? 0);

        if (class_exists(OnlineVisitors::class)) {
            OnlineVisitors::record();
        }

        // Ensure our frontend CSS/JS is available on the route.
        $cssFile = DTAROT_PATH . 'assets/frontend.css';
        $cssVer = is_readable($cssFile) ? (string)@filemtime($cssFile) : DTAROT_VERSION;
        wp_enqueue_style('dtarot-frontend', DTAROT_URL.'assets/frontend.css', [], $cssVer);
        $jsFile = DTAROT_PATH . 'assets/frontend.js';
        $ver = is_readable($jsFile) ? (string)@filemtime($jsFile) : DTAROT_VERSION;
        wp_enqueue_script('dtarot-frontend-js', DTAROT_URL.'assets/frontend.js', [], $ver, true);

        $shareCss = DTAROT_PATH . 'assets/share-image.css';
        if (is_readable($shareCss)) {
            $shareVer = (string)@filemtime($shareCss);
            wp_enqueue_style('dtarot-share-image', DTAROT_URL.'assets/share-image.css', [], $shareVer);
        }
        $shareJs = DTAROT_PATH . 'assets/share-image.js';
        if (is_readable($shareJs)) {
            $shareVer = (string)@filemtime($shareJs);
            wp_enqueue_script('dtarot-share-image', DTAROT_URL.'assets/share-image.js', [], $shareVer, true);
        }

        wp_localize_script('dtarot-frontend-js', 'DTAROT_FRONTEND', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('dtarot_booking'),
            'shareNonce' => wp_create_nonce('dtarot_share'),
            'shareImage' => class_exists(ShareImageSettings::class) ? ShareImageSettings::get() : [],
            'analyticsEnabled' => (bool)apply_filters('dtarot_analytics_enabled', true),
            'analytics' => [
                'pageType' => 'readable',
                'date' => $date,
                'deckId' => $deckId,
                'cardId' => $cardId,
                'url' => ReadableRoutes::urlForDate($date),
            ],
            'i18n' => [
                'loading' => __('Loading...','daily-tarot'),
                'noSlots' => __('No available times for this date.','daily-tarot'),
                'selectTime' => __('Select a time','daily-tarot'),
            ],
        ]);

        $meaning = MeaningPackRepository::getMeaning($packId, $cardId);
        $content = (string)($entryArr['content'] ?? '');
        $dailyText = (string)($entryArr['daily_text'] ?? '');
        $fallback = ReadingComposer::applyMeaningFallback($content, $dailyText, $meaning);
        $content = $fallback['content'];
        $dailyText = $fallback['daily_text'];

        $prevDate = DayEntryService::previousPublishedDate($date);
        $nextDate = DayEntryService::nextPublishedDate($date);
        $prevUrl = $prevDate !== '' ? self::urlForDate($prevDate) : '';
        $nextUrl = $nextDate !== '' ? self::urlForDate($nextDate) : '';

        $cardName = Cards::name($cardId);
        $cardUrl = '';
        if ($deckId > 0 && $cardId !== '') {
            $system = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
            if ($system === '') $system = Cards::SYSTEM_TAROT;
            $cardUrl = self::urlForCard($system, $cardId);
        }

        $imgUrl = '';
        $back = '';
        $overrideUrl = isset($entryArr['image_override_url']) ? (string)$entryArr['image_override_url'] : '';
        if ($overrideUrl !== '') {
            $imgUrl = esc_url($overrideUrl);
            // Custom day image: do not show flip UI.
            $back = '';
        } elseif ($deckId > 0 && $cardId !== '') {
            $imgs = get_post_meta($deckId, '_dtarot_cards', true);
            if (is_array($imgs) && !empty($imgs[$cardId]) && is_string($imgs[$cardId])) {
                $imgUrl = esc_url((string)$imgs[$cardId]);
            }
            $back = (string)get_post_meta($deckId, '_dtarot_back', true);
        }

        $html = '<div class="dtarot-frontend dtarot-readable">';

        if ($prevUrl !== '' || $nextUrl !== '') {
            $html .= '<nav class="dtarot-readable-nav" aria-label="' . esc_attr__('Daily navigation','daily-tarot') . '">';
            if ($prevUrl !== '') {
                $html .= '<a class="dtarot-readable-nav-link dtarot-readable-nav-prev" href="' . esc_url($prevUrl) . '">← ' . esc_html__('Previous','daily-tarot') . '</a>';
            } else {
                $html .= '<span class="dtarot-readable-nav-spacer"></span>';
            }
            if ($nextUrl !== '') {
                $html .= '<a class="dtarot-readable-nav-link dtarot-readable-nav-next" href="' . esc_url($nextUrl) . '">' . esc_html__('Next','daily-tarot') . ' →</a>';
            }
            $html .= '</nav>';
        }

        $html .= '<div class="dtarot-readable-row">';
        if ($imgUrl !== '') {
            if ($back !== '') {
                $html .= '<div class="dtarot-readable-left">' . self::renderFlip($imgUrl, $back) . '</div>';
            } else {
                $img = '<img src="'.$imgUrl.'" alt="" />';
                if ($cardUrl !== '') {
                    $img = '<a class="dtarot-card-link" href="' . esc_url($cardUrl) . '">' . $img . '</a>';
                }
                $html .= '<div class="dtarot-readable-left">' . $img . '</div>';
            }
        }

        $html .= '<div class="dtarot-readable-right">';
        if ($cardUrl !== '') {
            $html .= '<h2 class="dtarot-readable-title"><a class="dtarot-card-link" href="' . esc_url($cardUrl) . '">' . esc_html($cardName) . '</a></h2>';
        } else {
            $html .= '<h2 class="dtarot-readable-title">'.esc_html($cardName).'</h2>';
        }

        $html .= self::renderLenormandMetaBlock($cardId);

        if ($content !== '') {
            $html .= '<div class="dtarot-readable-intro">'.wp_kses_post($content).'</div>';
        }
        if ($dailyText !== '') {
            $html .= '<div class="dtarot-readable-expanded">'.wp_kses_post($dailyText).'</div>';
        }

        if (class_exists(RelatedLinks::class)) {
            $html .= RelatedLinks::renderBlock($cardId, $cardName);
        }

        $shareStyle = 'text';
        if (class_exists(ShortcodeSettings::class)) {
            $s = ShortcodeSettings::get();
            $shareStyle = isset($s['share_style']) && is_string($s['share_style']) ? (string)$s['share_style'] : 'text';
        }
        if (!in_array($shareStyle, ['text','arcana-sigils','crystal-aura','tarot-card'], true)) {
            $shareStyle = 'text';
        }

        $shareSource = $content !== '' ? $content : $dailyText;
        $shareExcerpt = self::shareExcerpt($shareSource, 220);
        $html .= '<div class="dtarot-readable-share">';
        $html .= Shortcodes::renderShareLinks((string)get_permalink(), 'daily', [
            'date' => $date,
            'post_id' => (isset($GLOBALS['post']) && is_object($GLOBALS['post']) && isset($GLOBALS['post']->ID)) ? (int)$GLOBALS['post']->ID : 0,
            'deck_id' => $deckId,
            'card_id' => $cardId,
            'share_style' => $shareStyle,
            'title' => $cardName,
            'excerpt' => $shareExcerpt,
            'readable_url' => ReadableRoutes::urlForDate($date),
            'card_img' => $imgUrl,
        ]);
        $html .= '</div>';

        $html .= '</div>';
        $html .= '</div>';
        $html .= '</div>';

        // Store cached HTML for a short TTL.
        set_transient($cacheKey, $html, 2 * HOUR_IN_SECONDS);

        $etag = 'W/"dtarot-readable-' . md5($html) . '"';
        $lastmod = class_exists(CacheVersion::class) ? CacheVersion::getDayLastModified($date) : 0;
        self::maybeHandleConditionalGet($etag, $lastmod);

        status_header(200);
        nocache_headers();

        // If WP resolved this as a 404 (no underlying page), force a normal render.
        global $wp_query;
        if ($wp_query) {
            $wp_query->is_404 = false;
        }

        get_header();
        echo '<main id="primary" class="site-main">' . $html . '</main>';
        get_footer();
        exit;
    }

    public static function maybe_render_card(): void {
        if (!self::isCardRequest()) return;
        if (!self::isValidCardRequest()) return;

        [$system, $cardId] = self::getCardRequestVars();

        // Ensure our frontend CSS/JS is available on the route.
        wp_enqueue_style('dtarot-frontend', DTAROT_URL.'assets/frontend.css', [], DTAROT_VERSION);
        $jsFile = DTAROT_PATH . 'assets/frontend.js';
        $ver = is_readable($jsFile) ? (string)@filemtime($jsFile) : DTAROT_VERSION;
        wp_enqueue_script('dtarot-frontend-js', DTAROT_URL.'assets/frontend.js', [], $ver, true);

        // Ensure share tracking + analytics context are available.
        wp_localize_script('dtarot-frontend-js', 'DTAROT_FRONTEND', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('dtarot_booking'),
            'shareNonce' => wp_create_nonce('dtarot_share'),
            'shareImage' => class_exists(ShareImageSettings::class) ? ShareImageSettings::get() : [],
            'analyticsEnabled' => (bool)apply_filters('dtarot_analytics_enabled', true),
            'analytics' => [
                'pageType' => 'card',
                'system' => $system,
                'cardId' => $cardId,
                'deckId' => 0,
                'url' => self::urlForCard($system, $cardId),
            ],
            'i18n' => [
                'loading' => __('Loading...','daily-tarot'),
                'noSlots' => __('No available times for this date.','daily-tarot'),
                'selectTime' => __('Select a time','daily-tarot'),
            ],
        ]);

        [$deckId, $packId] = self::getCardOverrides($system);
        $hasDeckOverride = isset($_GET['deck_id']) && $deckId > 0;
        $hasPackOverride = isset($_GET['pack_id']) && $packId > 0;

        if ($deckId <= 0) $deckId = self::findFirstDeckForSystem($system);
        if ($packId <= 0) $packId = self::findFirstPackForSystem($system);

        $noteHtml = '';
        if ($hasDeckOverride || $hasPackOverride) {
            $items = [];
            if ($hasDeckOverride) {
                $items[] = sprintf(
                    /* translators: %s is a Deck title */
                    __('Deck override: %s','daily-tarot'),
                    get_the_title($deckId) ?: __('(no title)','daily-tarot')
                );
            }
            if ($hasPackOverride) {
                $items[] = sprintf(
                    /* translators: %s is a Meaning Pack title */
                    __('Meaning pack override: %s','daily-tarot'),
                    get_the_title($packId) ?: __('(no title)','daily-tarot')
                );
            }

            $noteHtml = '<div class="dtarot-card-overrides-note">' . esc_html(implode(' · ', $items)) . '</div>';
        }

        $entry = [
            'deck' => $deckId > 0 ? (string)$deckId : '0',
            'card' => $cardId,
            'pack' => $packId > 0 ? (string)$packId : '0',
            'content' => '',
            'daily_text' => '',
        ];

        $meaning = $packId > 0 ? MeaningPackRepository::getMeaning($packId, $cardId) : MeaningPackRepository::emptyMeaning();
        $fallback = ReadingComposer::applyMeaningFallback('', '', $meaning);

        status_header(200);
        nocache_headers();

        get_header();

        echo '<main id="primary" class="site-main">';
        echo self::renderCardDetailHtml($system, $cardId, $deckId, $packId, $meaning, $fallback['content'], $fallback['daily_text'], 'h1', $noteHtml);
        echo '</main>';

        get_footer();
        exit;
    }

    private static function isValidCardRequest(): bool {
        [$system, $cardId] = self::getCardRequestVars();
        if ($system === '' || $cardId === '') return false;
        $cards = Cards::forSystem($system);
        return isset($cards[$cardId]);
    }

    /** @return array{0:string,1:string} */
    private static function getCardRequestVars(): array {
        $systemRaw = get_query_var('dtarot_system');
        $cardRaw = get_query_var('dtarot_card');

        $system = is_string($systemRaw) ? Cards::normalizeSystem($systemRaw) : '';
        $cardId = is_string($cardRaw) ? sanitize_text_field($cardRaw) : '';

        // Legacy compatibility:
        // - Some older builds used system=kipper with card ids like gypsy_01.
        // - Some stored links may also have system=gypsy with ids like kipper_01.
        if ($system === Cards::SYSTEM_KIPPER && preg_match('/^gypsy_(\d{2})$/', $cardId)) {
            $system = Cards::SYSTEM_GYPSY;
        } elseif ($system === Cards::SYSTEM_GYPSY && preg_match('/^kipper_(\d{2})$/', $cardId, $m)) {
            $cardId = 'gypsy_' . $m[1];
        }

        return [$system, $cardId];
    }

    private static function findFirstDeckForSystem(string $system): int {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return 0;

        $defaultId = DefaultDecks::get($system);
        if ($defaultId > 0) return $defaultId;

        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => 50,
            'post_status' => ['publish','draft'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);
        foreach ((array)$decks as $p) {
            if (!isset($p->ID)) continue;
            $pid = (int)$p->ID;
            $ps = Cards::normalizeSystem((string)get_post_meta($pid, '_dtarot_system', true));
            if ($ps === $system) return $pid;
        }
        return 0;
    }

    private static function findFirstPackForSystem(string $system): int {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return 0;

        $defaultId = DefaultMeaningPacks::get($system);
        if ($defaultId > 0) return $defaultId;

        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => 50,
            'post_status' => ['publish','draft'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);
        foreach ((array)$packs as $p) {
            if (!isset($p->ID)) continue;
            $pid = (int)$p->ID;
            $ps = Cards::normalizeSystem((string)get_post_meta($pid, '_dtarot_system', true));
            if ($ps === $system) return $pid;
        }
        return 0;
    }

    /** @return array{0:int,1:int} [deckId, packId] */
    private static function getCardOverrides(string $system): array {
        $system = Cards::normalizeSystem($system);
        if ($system === '') return [0, 0];

        $deckId = 0;
        $packId = 0;

        if (isset($_GET['deck_id'])) {
            $deckId = (int)sanitize_text_field((string)wp_unslash($_GET['deck_id']));
        }
        if (isset($_GET['pack_id'])) {
            $packId = (int)sanitize_text_field((string)wp_unslash($_GET['pack_id']));
        }

        if ($deckId > 0 && !self::isDeckForSystem($deckId, $system)) {
            $deckId = 0;
        }
        if ($packId > 0 && !self::isPackForSystem($packId, $system)) {
            $packId = 0;
        }

        return [$deckId, $packId];
    }

    private static function isDeckForSystem(int $deckId, string $system): bool {
        if ($deckId <= 0) return false;
        $p = get_post($deckId);
        if (!$p || !isset($p->post_type) || !PostTypes::isDeckType((string)$p->post_type)) return false;
        $ps = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
        return $ps === $system;
    }

    private static function isPackForSystem(int $packId, string $system): bool {
        if ($packId <= 0) return false;
        $p = get_post($packId);
        if (!$p || !isset($p->post_type) || !PostTypes::isMeaningPackType((string)$p->post_type)) return false;
        $ps = Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true));
        return $ps === $system;
    }

    private static function renderFlip(string $frontUrl, string $backUrl): string {
        $frontUrl = esc_url(trim($frontUrl));
        $backUrl = esc_url(trim($backUrl));

        if ($frontUrl === '' || $backUrl === '') {
            return $frontUrl !== '' ? '<img src="' . $frontUrl . '" alt="" />' : '';
        }

        return '<div class="dtarot-card-flip" data-dtarot-flip role="button" tabindex="0" aria-label="'.esc_attr__('Flip card','daily-tarot').'">'
            . '<div class="dtarot-card-flip-inner">'
            . '<div class="dtarot-card-face dtarot-card-front"><img src="' . $frontUrl . '" alt="" /></div>'
            . '<div class="dtarot-card-face dtarot-card-back"><img src="' . $backUrl . '" alt="" /></div>'
            . '</div>'
            . '</div>';
    }

    private static function renderLenormandMetaBlock(string $cardId): string {
        $cardId = sanitize_text_field($cardId);
        if ($cardId === '' || !str_starts_with($cardId, 'lenormand_')) return '';

        $meta = Cards::meta($cardId);
        if (!$meta) return '';

        $subject = isset($meta['subject']) && is_string($meta['subject']) ? trim($meta['subject']) : '';
        $modifier = isset($meta['modifier']) && is_string($meta['modifier']) ? trim($meta['modifier']) : '';
        $extended = isset($meta['extended']) && is_string($meta['extended']) ? trim($meta['extended']) : '';

        $rank = '';
        $suit = '';
        if (isset($meta['playing']) && is_array($meta['playing'])) {
            $rank = isset($meta['playing']['rank']) && is_string($meta['playing']['rank']) ? trim($meta['playing']['rank']) : '';
            $suit = isset($meta['playing']['suit']) && is_string($meta['playing']['suit']) ? trim($meta['playing']['suit']) : '';
        }

        $isSignifier = !empty($meta['is_signifier']);

        if ($subject === '' && $modifier === '' && $extended === '' && $rank === '' && $suit === '' && !$isSignifier) {
            return '';
        }

        $suitLabels = [
            'hearts' => __('Hearts','daily-tarot'),
            'spades' => __('Spades','daily-tarot'),
            'diamonds' => __('Diamonds','daily-tarot'),
            'clubs' => __('Clubs','daily-tarot'),
        ];
        $suitLabel = $suit !== '' ? ($suitLabels[$suit] ?? $suit) : '';

        $html = '<div class="dtarot-lenormand-meta">';

        if ($isSignifier) {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Signifier','daily-tarot').'</span></div>';
        }

        if ($rank !== '' || $suitLabel !== '') {
            $inset = trim($rank . ' ' . $suitLabel);
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Playing card inset','daily-tarot').':</span> '.esc_html($inset).'</div>';
        }

        if ($subject !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Subject','daily-tarot').':</span> '.esc_html($subject).'</div>';
        }
        if ($modifier !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Modifier','daily-tarot').':</span> '.esc_html($modifier).'</div>';
        }
        if ($extended !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Extended','daily-tarot').':</span> '.esc_html($extended).'</div>';
        }

        $html .= '</div>';
        return $html;
    }

    /** @param array{upright?:string,reversed?:string,short?:string,long?:string,keywords?:string,correspondences?:string,symbols?:string} $meaning */
    private static function renderCardDetailHtml(string $system, string $cardId, int $deckId, int $packId, array $meaning, string $content, string $dailyText, string $headingTag, string $noteHtml = ''): string {
        $cardName = Cards::name($cardId);

        $system = Cards::normalizeSystem($system);
        $systems = Cards::systems();
        $systemLabel = $systems[$system] ?? $system;

        $deckTitle = '';
        if ($deckId > 0) {
            $p = get_post($deckId);
            $deckTitle = ($p && isset($p->post_title)) ? (string)$p->post_title : '';
        }

        $imgUrl = '';
        $imgResolvedKey = '';
        $back = '';
        if ($deckId > 0) {
            $imgs = get_post_meta($deckId, '_dtarot_cards', true);
            if (is_array($imgs) && $imgs) {
                foreach (Cards::kipperGypsyAliases($cardId) as $id) {
                    if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
                    $u = trim((string)$imgs[$id]);
                    if ($u === '') continue;
                    $imgUrl = esc_url($u);
                    $imgResolvedKey = (string)$id;
                    break;
                }
            }
            $back = (string)get_post_meta($deckId, '_dtarot_back', true);
        }

        $html = '<div class="dtarot-frontend dtarot-readable dtarot-card-detail">';

        // Top: same card across decks.
        // If we have at least one *other* deck, omit the active deck from the strip.
        // If there are no other decks, show the active deck so the section isn't empty.
        $deckVariants = self::getDeckVariantsForCard($system, $cardId, $deckId);
        if ($deckVariants) {
            $hasOtherDeck = false;
            if ($deckId > 0) {
                foreach ($deckVariants as $d) {
                    if (empty($d['is_active'])) {
                        $hasOtherDeck = true;
                        break;
                    }
                }
            }

            $deckVariantsToShow = $hasOtherDeck
                ? array_values(array_filter($deckVariants, fn(array $d): bool => empty($d['is_active'])))
                : $deckVariants;

            if (!$deckVariantsToShow) {
                $deckVariantsToShow = $deckVariants;
            }

            $showControls = count($deckVariantsToShow) > 1;
            $stripId = 'dtarot-deck-variants-' . substr(md5($system . '|' . $cardId), 0, 10);
            $html .= '<div class="dtarot-card-switcher" aria-label="' . esc_attr__('Other decks','daily-tarot') . '">';
            $html .= '<div class="dtarot-card-switcher-head">';
            $html .= '<div class="dtarot-card-switcher-title">' . esc_html__('This card in other decks','daily-tarot') . '</div>';
            $html .= '</div>';

            $html .= '<div class="dtarot-scroll-wrap">';
            if ($showControls) {
                $html .= '<button type="button" class="dtarot-scroll-btn dtarot-scroll-btn-prev" aria-label="' . esc_attr__('Scroll left','daily-tarot') . '" data-dtarot-scroll="prev" data-dtarot-scroll-target="' . esc_attr($stripId) . '">' . esc_html__('←','daily-tarot') . '</button>';
            }

            $stripCls = 'dtarot-scroll-strip dtarot-card-switcher-strip' . ($showControls ? '' : ' dtarot-scroll-strip--no-controls');
            $html .= '<div class="' . esc_attr($stripCls) . '" id="' . esc_attr($stripId) . '" data-dtarot-scroll-strip>';
            foreach ($deckVariantsToShow as $d) {
                $did = (int)($d['deck_id'] ?? 0);
                $title = isset($d['title']) && is_string($d['title']) ? trim($d['title']) : '';
                $img = isset($d['img_url']) && is_string($d['img_url']) ? (string)$d['img_url'] : '';
                $isActive = !empty($d['is_active']);

                $href = add_query_arg(['deck_id' => (string)$did], self::urlForCard($system, $cardId));
                $cls = 'dtarot-card-switcher-item' . ($isActive ? ' is-active' : '');

                $html .= '<a class="' . esc_attr($cls) . '" href="' . esc_url($href) . '">';
                $html .= '<span class="dtarot-card-switcher-thumb"><img src="' . esc_url($img) . '" alt="" loading="lazy" /></span>';
                if ($title !== '') {
                    $html .= '<span class="dtarot-card-switcher-caption">' . esc_html($title) . '</span>';
                }
                $html .= '</a>';
            }
            $html .= '</div>';
            if ($showControls) {
                $html .= '<button type="button" class="dtarot-scroll-btn dtarot-scroll-btn-next" aria-label="' . esc_attr__('Scroll right','daily-tarot') . '" data-dtarot-scroll="next" data-dtarot-scroll-target="' . esc_attr($stripId) . '">' . esc_html__('→','daily-tarot') . '</button>';
            }
            $html .= '</div>';
            $html .= '</div>';
        }

        $html .= '<div class="dtarot-readable-row">';

        if ($imgUrl !== '') {
            $html .= '<div class="dtarot-readable-left">' . (($back !== '') ? self::renderFlip($imgUrl, $back) : '<img src="'.$imgUrl.'" alt="" />') . '</div>';
        }

        $html .= '<div class="dtarot-readable-right">';

        if ($noteHtml !== '') {
            $html .= $noteHtml;
        }

        // Admin-only Troubleshoot panel (shown only when explicitly enabled).
        $showDebug = is_user_logged_in() && current_user_can('manage_options');
        $debugParam = isset($_GET['dtarot_debug']) ? sanitize_key((string)wp_unslash($_GET['dtarot_debug'])) : '';
        if ($showDebug && $debugParam === '1') {
            $deckEdit = $deckId > 0 ? get_edit_post_link($deckId, '') : '';
            $packEdit = $packId > 0 ? get_edit_post_link($packId, '') : '';

            $deckSystem = $deckId > 0 ? Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true)) : '';
            $packSystem = $packId > 0 ? Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true)) : '';

            $html .= '<details class="dtarot-admin-debug" style="margin:10px 0;padding:10px;border:1px solid #ddd;border-radius:8px;background:#fff;">';
            $html .= '<summary style="cursor:pointer;font-weight:600;">' . esc_html__('Troubleshoot (admin)','daily-tarot') . '</summary>';
            $html .= '<div style="margin-top:10px;">';
            $html .= '<p class="description" style="margin-top:0;">' . esc_html__('If something looks missing, check system matches and whether a default deck/pack is selected.','daily-tarot') . '</p>';
            $html .= '<table class="widefat striped" style="max-width:760px;">';
            $html .= '<tbody>';
            $html .= '<tr><td style="width:220px;"><strong>' . esc_html__('System (URL)','daily-tarot') . '</strong></td><td>' . esc_html($system) . '</td></tr>';
            $html .= '<tr><td><strong>' . esc_html__('Card ID','daily-tarot') . '</strong></td><td><code>' . esc_html($cardId) . '</code></td></tr>';
            $html .= '<tr><td><strong>' . esc_html__('Deck','daily-tarot') . '</strong></td><td>' . ($deckTitle !== '' ? esc_html($deckTitle) : esc_html__('(none)','daily-tarot')) . ($deckEdit ? ' <a href="' . esc_url($deckEdit) . '">' . esc_html__('Edit','daily-tarot') . '</a>' : '') . '</td></tr>';
            $html .= '<tr><td><strong>' . esc_html__('Deck system','daily-tarot') . '</strong></td><td>' . esc_html($deckSystem ?: '-') . '</td></tr>';
            $html .= '<tr><td><strong>' . esc_html__('Meaning pack','daily-tarot') . '</strong></td><td>' . ($packId > 0 ? esc_html(get_the_title($packId) ?: (string)$packId) : esc_html__('(none)','daily-tarot')) . ($packEdit ? ' <a href="' . esc_url($packEdit) . '">' . esc_html__('Edit','daily-tarot') . '</a>' : '') . '</td></tr>';
            $html .= '<tr><td><strong>' . esc_html__('Pack system','daily-tarot') . '</strong></td><td>' . esc_html($packSystem ?: '-') . '</td></tr>';
            $html .= '<tr><td><strong>' . esc_html__('Card image','daily-tarot') . '</strong></td><td>' . ($imgUrl !== '' ? esc_html__('Found','daily-tarot') : esc_html__('Missing','daily-tarot')) . '</td></tr>';
            $html .= '<tr><td><strong>' . esc_html__('Resolved image key','daily-tarot') . '</strong></td><td><code>' . esc_html($imgResolvedKey !== '' ? $imgResolvedKey : '-') . '</code></td></tr>';
            $html .= '</tbody>';
            $html .= '</table>';
            $html .= '<p style="margin:10px 0 0 0;">';
            $html .= '<a class="button" href="' . esc_url(admin_url('admin.php?page=daily-tarot-content&tab=cards&deck_id=' . (string)$deckId)) . '">' . esc_html__('Fix card images','daily-tarot') . '</a>';
            $html .= '</p>';
            $html .= '</div>';
            $html .= '</details>';
        }

        $tag = strtolower(trim($headingTag));
        if (!in_array($tag, ['h1','h2','h3','h4','h5','h6'], true)) $tag = 'h1';
        $html .= '<' . $tag . ' class="dtarot-readable-title">'.esc_html($cardName).'</' . $tag . '>';

        $metaBits = [];
        if ($systemLabel !== '') {
            $metaBits[] = $systemLabel;
        }
        if ($deckTitle !== '') {
            $metaBits[] = sprintf(
                /* translators: %s is a Deck title */
                __('Deck: %s','daily-tarot'),
                $deckTitle
            );
        }
        if ($metaBits) {
            $html .= '<div class="dtarot-readable-meta">' . esc_html(implode(' · ', $metaBits)) . '</div>';
        }

        $html .= self::renderLenormandMetaBlock($cardId);

        $upright = isset($meaning['upright']) && is_string($meaning['upright']) ? trim($meaning['upright']) : '';
        $reversed = isset($meaning['reversed']) && is_string($meaning['reversed']) ? trim($meaning['reversed']) : '';

        // Card detail pages should show the explicit Upright/Reversed text when present.
        // If a pack doesn't provide those fields, fall back to the existing Meaning/More blocks.
        $hasExplicit = ($upright !== '' || $reversed !== '');
        if ($hasExplicit) {
            $uprightToShow = $upright;
            if ($uprightToShow === '' && $content !== '') {
                $uprightToShow = $content;
            }

            if ($uprightToShow !== '') {
                $html .= '<h2 class="dtarot-readable-section-title">' . esc_html__('Upright','daily-tarot') . '</h2>';
                $html .= '<div class="dtarot-readable-intro">' . wp_kses_post(wpautop($uprightToShow)) . '</div>';
            }
            if ($reversed !== '') {
                $html .= '<h2 class="dtarot-readable-section-title">' . esc_html__('Reversed','daily-tarot') . '</h2>';
                $html .= '<div class="dtarot-readable-intro">' . wp_kses_post(wpautop($reversed)) . '</div>';
            }
        } else {
            if ($content !== '') {
                $html .= '<h2 class="dtarot-readable-section-title">' . esc_html__('Meaning','daily-tarot') . '</h2>';
                $html .= '<div class="dtarot-readable-intro">'.wp_kses_post($content).'</div>';
            }
            if ($dailyText !== '') {
                $html .= '<h2 class="dtarot-readable-section-title">' . esc_html__('More','daily-tarot') . '</h2>';
                $html .= '<div class="dtarot-readable-expanded">'.wp_kses_post($dailyText).'</div>';
            }
        }

        $html .= '</div>';
        $html .= '</div>';

        // Bottom: other cards from this deck in a horizontal scroller.
        $otherCards = self::getOtherCardsFromDeck($system, $deckId, $cardId);
        if ($otherCards) {
            $stripId = 'dtarot-deck-cards-' . substr(md5($system . '|' . $deckId . '|' . $cardId), 0, 10);
            $html .= '<div class="dtarot-card-deck-strip" aria-label="' . esc_attr__('More cards from this deck','daily-tarot') . '">';
            $html .= '<div class="dtarot-card-deck-strip-head">';
            $html .= '<div class="dtarot-card-deck-strip-title">' . esc_html__('More cards from this deck','daily-tarot') . '</div>';
            $html .= '</div>';

            $html .= '<div class="dtarot-scroll-wrap">';
            $html .= '<button type="button" class="dtarot-scroll-btn dtarot-scroll-btn-prev" aria-label="' . esc_attr__('Scroll left','daily-tarot') . '" data-dtarot-scroll="prev" data-dtarot-scroll-target="' . esc_attr($stripId) . '">' . esc_html__('←','daily-tarot') . '</button>';
            $html .= '<div class="dtarot-scroll-strip dtarot-card-deck-strip-row" id="' . esc_attr($stripId) . '" data-dtarot-scroll-strip>';
            foreach ($otherCards as $c) {
                $name = isset($c['name']) && is_string($c['name']) ? $c['name'] : '';
                $img = isset($c['img_url']) && is_string($c['img_url']) ? $c['img_url'] : '';
                $url = isset($c['url']) && is_string($c['url']) ? $c['url'] : '';

                if ($img === '' || $url === '') continue;
                $html .= '<a class="dtarot-card-strip-item" href="' . esc_url($url) . '">'
                    . '<span class="dtarot-card-strip-thumb"><img src="' . esc_url($img) . '" alt="" loading="lazy" /></span>'
                    . '<span class="dtarot-card-strip-caption">' . esc_html($name) . '</span>'
                    . '</a>';
            }
            $html .= '</div>';
                    $html .= '<button type="button" class="dtarot-scroll-btn dtarot-scroll-btn-next" aria-label="' . esc_attr__('Scroll right','daily-tarot') . '" data-dtarot-scroll="next" data-dtarot-scroll-target="' . esc_attr($stripId) . '">' . esc_html__('→','daily-tarot') . '</button>';
                    $html .= '</div>';
            $html .= '</div>';
        }

        $html .= '</div>';
        return $html;
    }
}
