<?php
declare(strict_types=1);


namespace DailyTarot\Frontend;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Support\BookingAvailability;
use DailyTarot\Support\BookingEmails;
use DailyTarot\Support\BookingSettings;
use DailyTarot\Support\PostTypes;
use DailyTarot\Support\RateLimit;
use DailyTarot\Support\Log;

final class Booking {

    public static function init(): void {
        add_action('wp_ajax_dtarot_booking_slots', [__CLASS__, 'slots']);
        add_action('wp_ajax_nopriv_dtarot_booking_slots', [__CLASS__, 'slots']);
        add_action('admin_post_dtarot_booking_submit', [__CLASS__, 'submit']);
        add_action('admin_post_nopriv_dtarot_booking_submit', [__CLASS__, 'submit']);
    }

    public static function slots(): void {
        if (class_exists(RateLimit::class) && !RateLimit::hit('booking_slots', 60, 60)) {
            if (class_exists(Log::class)) {
                Log::add('warn', 'rate_limited', 'Booking slots rate limited', ['action' => 'booking_slots']);
            }
            header('Retry-After: ' . (string)RateLimit::defaultRetryAfterSeconds(60));
            wp_send_json_error(['message' => 'rate_limited'], 429);
        }

        $nonce = isset($_POST['nonce']) ? sanitize_text_field((string)wp_unslash($_POST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'dtarot_booking')) {
            wp_send_json_error(['message' => 'bad_nonce'], 403);
        }

        $date = isset($_POST['date']) ? sanitize_text_field((string)wp_unslash($_POST['date'])) : '';
        $readingId = isset($_POST['reading_type']) ? (int)wp_unslash($_POST['reading_type']) : 0;
        $userTz = isset($_POST['timezone']) ? sanitize_text_field((string)wp_unslash($_POST['timezone'])) : '';

        if ($date === '' || $readingId <= 0) {
            wp_send_json_success(['slots' => []]);
        }
        $reading = get_post($readingId);
        if (!$reading || !isset($reading->post_type) || !PostTypes::isReadingTypeType((string)$reading->post_type)) {
            wp_send_json_success(['slots' => []]);
        }

        $slots = BookingAvailability::slotsForDate($date, $readingId);
        $out = [];
        foreach ($slots as $slot) {
            $startUtc = (int)$slot['start_utc'];
            $endUtc = (int)$slot['end_utc'];
            $label = $slot['start_admin'];
            $userDate = $slot['start_admin'];

            if ($userTz !== '') {
                try {
                    $tz = new \DateTimeZone($userTz);
                    $label = function_exists('wp_date') ? wp_date('Y-m-d H:i', $startUtc, $tz) : gmdate('Y-m-d H:i', $startUtc);
                    $userDate = $label;
                } catch (\Exception $e) {
                }
            }

            $out[] = [
                'start_utc' => $startUtc,
                'end_utc' => $endUtc,
                'label' => $label,
                'start_label' => $userDate,
            ];
        }

        wp_send_json_success(['slots' => $out]);
    }

    public static function submit(): void {
        $nonce = isset($_POST['dtarot_booking_nonce']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_booking_nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'dtarot_booking_submit')) {
            self::redirect('error');
        }

        $name = isset($_POST['booking_name']) ? sanitize_text_field((string)wp_unslash($_POST['booking_name'])) : '';
        $email = isset($_POST['booking_email']) ? sanitize_email((string)wp_unslash($_POST['booking_email'])) : '';
        $timezone = isset($_POST['booking_timezone']) ? sanitize_text_field((string)wp_unslash($_POST['booking_timezone'])) : '';
        if ($timezone !== '') {
            try {
                new \DateTimeZone($timezone);
            } catch (\Exception $e) {
                $timezone = '';
            }
        }
        $question = isset($_POST['booking_question']) ? sanitize_textarea_field((string)wp_unslash($_POST['booking_question'])) : '';
        $readingId = isset($_POST['reading_type']) ? (int)wp_unslash($_POST['reading_type']) : 0;
        $startUtc = isset($_POST['booking_start_utc']) ? (int)wp_unslash($_POST['booking_start_utc']) : 0;
        $consent = !empty($_POST['booking_consent']);
        $paymentProvider = isset($_POST['payment_provider']) ? sanitize_key((string)wp_unslash($_POST['payment_provider'])) : '';
        $paymentRef = isset($_POST['payment_ref']) ? sanitize_text_field((string)wp_unslash($_POST['payment_ref'])) : '';
        $paymentConfirm = !empty($_POST['payment_confirm']);

        if ($name === '' || $email === '' || $readingId <= 0 || $startUtc <= 0 || !$consent) {
            self::redirect('error');
        }
        if (!is_email($email)) self::redirect('error');

        $reading = get_post($readingId);
        if (!$reading || !isset($reading->post_type) || !PostTypes::isReadingTypeType((string)$reading->post_type)) {
            self::redirect('error');
        }

        if (!BookingAvailability::validateSlot($startUtc, $readingId)) {
            self::redirect('error');
        }

        $settings = BookingSettings::get();
        $paymentMode = isset($settings['payment_mode']) ? (string)$settings['payment_mode'] : 'none';
        if ($paymentMode === 'before' && !$paymentConfirm) {
            self::redirect('error');
        }
        if (!in_array($paymentProvider, ['paypal','stripe'], true)) {
            $paymentProvider = (string)($settings['payment_provider'] ?? 'paypal');
            if (!in_array($paymentProvider, ['paypal','stripe'], true)) {
                $paymentProvider = 'paypal';
            }
        }

        $duration = BookingAvailability::getReadingDuration($readingId);
        $endUtc = $startUtc + ($duration * 60);

        $adminTz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
        $adminDate = function_exists('wp_date') ? wp_date('Y-m-d', $startUtc, $adminTz) : gmdate('Y-m-d', $startUtc);
        $adminTime = function_exists('wp_date') ? wp_date('H:i', $startUtc, $adminTz) : gmdate('H:i', $startUtc);

        $postId = wp_insert_post([
            'post_type' => PostTypes::BOOKING,
            'post_status' => 'publish',
            'post_title' => sprintf('%s - %s %s', $name, $adminDate, $adminTime),
        ]);

        if (!$postId || is_wp_error($postId)) {
            self::redirect('error');
        }

        $status = (BookingSettings::getMode() === 'instant') ? 'approved' : 'pending';
        $paymentStatus = ($paymentMode === 'before' && $paymentConfirm) ? 'paid' : 'unpaid';

        update_post_meta($postId, '_dtarot_booking_name', $name);
        update_post_meta($postId, '_dtarot_booking_email', $email);
        update_post_meta($postId, '_dtarot_booking_timezone', $timezone);
        update_post_meta($postId, '_dtarot_booking_reading_type', $readingId);
        update_post_meta($postId, '_dtarot_booking_start_utc', $startUtc);
        update_post_meta($postId, '_dtarot_booking_end_utc', $endUtc);
        update_post_meta($postId, '_dtarot_booking_status', $status);
        update_post_meta($postId, '_dtarot_booking_question', $question);
        update_post_meta($postId, '_dtarot_booking_payment_status', $paymentStatus);
        update_post_meta($postId, '_dtarot_booking_payment_provider', $paymentProvider);
        update_post_meta($postId, '_dtarot_booking_payment_ref', $paymentRef);

        $payload = self::payload($postId, $status, '');
        BookingEmails::sendAdminNew($payload);
        if ($status === 'approved') {
            BookingEmails::sendUserConfirmed($payload);
        } else {
            BookingEmails::sendUserRequest($payload);
        }

        self::log($postId, 'created', 'Booking created via frontend');
        do_action('dtarot_booking_created', $postId, $payload);
        self::redirect('ok');
    }

    private static function payload(int $postId, string $status, string $message): array {
        $name = (string)get_post_meta($postId, '_dtarot_booking_name', true);
        $email = (string)get_post_meta($postId, '_dtarot_booking_email', true);
        $timezone = (string)get_post_meta($postId, '_dtarot_booking_timezone', true);
        $readingId = (int)get_post_meta($postId, '_dtarot_booking_reading_type', true);
        $question = (string)get_post_meta($postId, '_dtarot_booking_question', true);
        $startUtc = (int)get_post_meta($postId, '_dtarot_booking_start_utc', true);
        $paymentProvider = (string)get_post_meta($postId, '_dtarot_booking_payment_provider', true);
        $paymentUrl = BookingSettings::paymentUrlFor($paymentProvider);

        $adminTz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
        $adminDate = $startUtc ? wp_date('Y-m-d', $startUtc, $adminTz) : '';
        $adminTime = $startUtc ? wp_date('H:i', $startUtc, $adminTz) : '';

        $userDate = $adminDate;
        $userTime = $adminTime;
        if ($timezone !== '') {
            try {
                $userTz = new \DateTimeZone($timezone);
                $userDate = $startUtc ? wp_date('Y-m-d', $startUtc, $userTz) : $userDate;
                $userTime = $startUtc ? wp_date('H:i', $startUtc, $userTz) : $userTime;
            } catch (\Exception $e) {
            }
        }

        return [
            'name' => $name,
            'email' => $email,
            'reading' => $readingId > 0 ? (string)get_the_title($readingId) : '',
            'date' => $userDate,
            'time' => $userTime,
            'timezone' => $timezone,
            'status' => $status,
            'question' => $question,
            'message' => $message,
            'payment_provider' => $paymentProvider,
            'payment_url' => $paymentUrl,
            'admin_date' => $adminDate,
            'admin_time' => $adminTime,
            'admin_timezone' => (string)$adminTz->getName(),
        ];
    }

    private static function log(int $id, string $action, string $note): void {
        $log = get_post_meta($id, '_dtarot_booking_log', true);
        if (!is_array($log)) $log = [];
        $log[] = [
            'time' => gmdate('c'),
            'action' => $action,
            'note' => $note,
        ];
        update_post_meta($id, '_dtarot_booking_log', $log);
    }

    private static function redirect(string $state): void {
        $ref = wp_get_referer();
        if (!$ref) $ref = home_url('/');
        $url = add_query_arg('dtarot_booking', $state, $ref);
        wp_safe_redirect($url);
        exit;
    }
}
