<?php
declare(strict_types=1);


namespace DailyTarot\Calendar;
if (!defined('ABSPATH')) { exit; }


/**
 * Stores publish times for the daily automation job.
 *
 * - Default publish time applies to all dates.
 * - Optional per-date overrides can be set (Pro only UI).
 *
 * Storage is intentionally separate from DayEntry storage so it works for both
 * option-backed and table-backed calendar entry backends.
 */
final class PublishTimes {

    private const OPT_DEFAULT_TIME = 'dtarot_calendar_publish_time_default';
    private const OPT_OVERRIDES = 'dtarot_calendar_publish_time_overrides';

    public static function sanitizeTime(string $time): string {
        $time = trim($time);
        if ($time === '') return '';
        if (!preg_match('/^(\d{1,2}):(\d{2})$/', $time, $m)) return '';
        $hh = (int)$m[1];
        $mm = (int)$m[2];
        if ($hh < 0 || $hh > 23) return '';
        if ($mm < 0 || $mm > 59) return '';
        return sprintf('%02d:%02d', $hh, $mm);
    }

    public static function sanitizeDate(string $date): string {
        $date = trim($date);
        return preg_match('/^\d{4}-\d{2}-\d{2}$/', $date) ? $date : '';
    }

    /**
     * Returns the default publish time.
     *
     * If not set, returns $fallback (if valid) or empty string.
     */
    public static function getDefaultTime(string $fallback = ''): string {
        $t = (string)get_option(self::OPT_DEFAULT_TIME, '');
        $t = self::sanitizeTime($t);
        if ($t !== '') return $t;

        $fallback = self::sanitizeTime($fallback);
        return $fallback;
    }

    public static function setDefaultTime(string $time): void {
        $time = self::sanitizeTime($time);
        update_option(self::OPT_DEFAULT_TIME, $time, false);
    }

    /**
     * Returns the per-date override time (empty string if none).
     */
    public static function getOverride(string $date): string {
        $date = self::sanitizeDate($date);
        if ($date === '') return '';

        $all = get_option(self::OPT_OVERRIDES, []);
        if (!is_array($all)) $all = [];

        $t = isset($all[$date]) && !is_array($all[$date]) ? (string)$all[$date] : '';
        return self::sanitizeTime($t);
    }

    /**
     * Sets or clears a per-date override.
     *
     * @return bool true if saved, false on invalid date.
     */
    public static function setOverride(string $date, string $time): bool {
        $date = self::sanitizeDate($date);
        if ($date === '') return false;

        $time = self::sanitizeTime($time);
        $all = get_option(self::OPT_OVERRIDES, []);
        if (!is_array($all)) $all = [];

        if ($time === '') {
            if (isset($all[$date])) {
                unset($all[$date]);
                update_option(self::OPT_OVERRIDES, $all, false);
            }
            return true;
        }

        $all[$date] = $time;
        update_option(self::OPT_OVERRIDES, $all, false);
        return true;
    }

    /**
     * Returns the publish time for a date: override if present, else default/fallback.
     */
    public static function timeForDate(string $date, string $fallbackDefaultTime = '06:00'): string {
        $date = self::sanitizeDate($date);
        if ($date === '') return self::sanitizeTime($fallbackDefaultTime);

        $override = self::getOverride($date);
        if ($override !== '') return $override;

        $default = self::getDefaultTime($fallbackDefaultTime);
        return $default !== '' ? $default : self::sanitizeTime($fallbackDefaultTime);
    }

    public static function countOverridesForMonth(int $year, int $month): int {
        $year = max(1970, $year);
        $month = max(1, min(12, $month));
        $prefix = sprintf('%04d-%02d-', $year, $month);

        $all = get_option(self::OPT_OVERRIDES, []);
        if (!is_array($all)) $all = [];

        $count = 0;
        foreach ($all as $date => $time) {
            if (!is_string($date) || strpos($date, $prefix) !== 0) continue;
            if (self::sanitizeTime(is_array($time) ? '' : (string)$time) === '') continue;
            $count++;
        }
        return $count;
    }

    /**
     * Returns all per-date overrides for a month.
     *
     * @return array<string,string> Map of YYYY-MM-DD => HH:MM
     */
    public static function overridesForMonth(int $year, int $month): array {
        $year = max(1970, $year);
        $month = max(1, min(12, $month));
        $prefix = sprintf('%04d-%02d-', $year, $month);

        $all = get_option(self::OPT_OVERRIDES, []);
        if (!is_array($all)) $all = [];

        $out = [];
        foreach ($all as $date => $time) {
            if (!is_string($date) || strpos($date, $prefix) !== 0) continue;
            $t = self::sanitizeTime(is_array($time) ? '' : (string)$time);
            if ($t === '') continue;
            $out[$date] = $t;
        }

        return $out;
    }

    /**
     * Clears all per-date overrides within a month.
     *
     * @return int number removed
     */
    public static function clearOverridesForMonth(int $year, int $month): int {
        $year = max(1970, $year);
        $month = max(1, min(12, $month));
        $prefix = sprintf('%04d-%02d-', $year, $month);

        $all = get_option(self::OPT_OVERRIDES, []);
        if (!is_array($all)) $all = [];

        $removed = 0;
        foreach (array_keys($all) as $date) {
            if (!is_string($date) || strpos($date, $prefix) !== 0) continue;
            unset($all[$date]);
            $removed++;
        }

        if ($removed > 0) {
            update_option(self::OPT_OVERRIDES, $all, false);
        }

        return $removed;
    }

    /**
     * If automation is enabled, reschedule the next run so changes take effect.
     */
    public static function rescheduleAutomationIfEnabled(): void {
        if (!class_exists('DailyTarot\\Automation\\Scheduler')) return;
        if (!class_exists('DailyTarot\\Automation\\AutomationSettings')) return;

        if (!\DailyTarot\Automation\AutomationSettings::isEnabled()) return;
        \DailyTarot\Automation\Scheduler::ensureScheduled(true);
    }
}
