<?php
declare(strict_types=1);


namespace DailyTarot\Calendar;
if (!defined('ABSPATH')) { exit; }

use DailyTarot\Support\CachePurge;
use DailyTarot\Support\CacheVersion;


/**
 * Service layer around day entry storage.
 *
 * Purpose:
 * - centralize normalization/contract rules (DayEntry)
 * - keep callers away from raw option storage
 */
final class DayEntryService {

    private const CACHE_GROUP = 'dtarot';

    private static function cacheVer(): int {
        return class_exists(CacheVersion::class) ? CacheVersion::get() : 1;
    }

    /** @return mixed */
    private static function cacheGet(string $key) {
        $val = wp_cache_get($key, self::CACHE_GROUP);
        if ($val !== false) return $val;

        $t = get_transient($key);
        if ($t !== false) {
            wp_cache_set($key, $t, self::CACHE_GROUP, 300);
            return $t;
        }

        return false;
    }

    /** @param mixed $val */
    private static function cacheSet(string $key, $val, int $ttl): void {
        wp_cache_set($key, $val, self::CACHE_GROUP, min(300, max(1, $ttl)));
        set_transient($key, $val, $ttl);
    }

    public static function get(string $date): DayEntry {
        $date = DayEntry::normalizeDate($date);
        if ($date === '') return new DayEntry();

        $raw = DayEntryRepository::get($date);
        return DayEntry::fromArray($raw);
    }

    public static function getPublished(string $date): ?DayEntry {
        $date = DayEntry::normalizeDate($date);
        if ($date === '') return null;

        $key = 'dtarot_pub_' . self::cacheVer() . '_' . $date;
        $cached = self::cacheGet($key);
        if (is_array($cached) && array_key_exists('found', $cached)) {
            if ($cached['found'] !== true) return null;
            $raw = (isset($cached['entry']) && is_array($cached['entry'])) ? $cached['entry'] : [];
            $e = DayEntry::fromArray($raw);
            if (!$e->isPublished() || !$e->isPublishable() || !self::hasRequiredTextAndImage($e)) return null;
            return $e;
        }

        $e = self::get($date);
        $ok = $e->isPublished() && $e->isPublishable() && self::hasRequiredTextAndImage($e);
        self::cacheSet($key, $ok ? ['found' => true, 'entry' => $e->toArray()] : ['found' => false], 6 * HOUR_IN_SECONDS);
        return $ok ? $e : null;
    }

    /**
     * Returns up to $limit published entries going backwards from $fromDate (inclusive).
     *
     * @return array<string,DayEntry> Map of date => DayEntry
     */
    public static function latestPublished(int $limit, string $fromDate = ''): array {
        $limit = (int)$limit;
        if ($limit <= 0) return [];
        if ($limit > 365) $limit = 365;

        $fromDate = trim($fromDate);
        if ($fromDate === '') {
            $fromDate = function_exists('wp_date') ? (string)wp_date('Y-m-d') : (string)current_time('Y-m-d');
        }
        $fromDate = DayEntry::normalizeDate($fromDate);
        if ($fromDate === '') return [];

        $key = 'dtarot_latest_' . self::cacheVer() . '_' . $limit . '_' . $fromDate;
        $cached = self::cacheGet($key);
        if (is_array($cached)) {
            $out = [];
            foreach ($cached as $date => $raw) {
                if (!is_string($date) || !is_array($raw)) continue;
                $e = DayEntry::fromArray($raw);
                if (!$e->isPublished() || !$e->isPublishable() || !self::hasRequiredTextAndImage($e)) continue;
                $out[$date] = $e;
            }
            return $out;
        }

        $items = DayEntryRepository::latestPublished($limit, $fromDate);
        if (!$items) {
            self::cacheSet($key, [], 10 * MINUTE_IN_SECONDS);
            return [];
        }

        $out = [];
        $rawOut = [];
        foreach ($items as $date => $entry) {
            if (!is_string($date) || !is_array($entry)) continue;
            $e = DayEntry::fromArray($entry);
            if (!$e->isPublished() || !$e->isPublishable() || !self::hasRequiredTextAndImage($e)) continue;
            $out[$date] = $e;
            $rawOut[$date] = $e->toArray();
        }

        self::cacheSet($key, $rawOut, 6 * HOUR_IN_SECONDS);
        return $out;
    }

    /**
     * Fetch published+publishable entries for an inclusive date range.
     *
     * @return array<string,DayEntry> Map of date => DayEntry
     */
    public static function publishedRange(string $startDate, string $endDate): array {
        $startDate = DayEntry::normalizeDate($startDate);
        $endDate = DayEntry::normalizeDate($endDate);
        if ($startDate === '' || $endDate === '') return [];

        $key = 'dtarot_pub_range_' . self::cacheVer() . '_' . $startDate . '_' . $endDate;
        $cached = self::cacheGet($key);
        if (is_array($cached)) {
            $out = [];
            foreach ($cached as $date => $raw) {
                if (!is_string($date) || !is_array($raw)) continue;
                $e = DayEntry::fromArray($raw);
                if (!$e->isPublished() || !$e->isPublishable() || !self::hasRequiredTextAndImage($e)) continue;
                $out[$date] = $e;
            }
            return $out;
        }

        $items = DayEntryRepository::publishedRange($startDate, $endDate);
        if (!$items) {
            self::cacheSet($key, [], 10 * MINUTE_IN_SECONDS);
            return [];
        }

        $out = [];
        $rawOut = [];
        foreach ($items as $date => $entry) {
            if (!is_string($date) || !is_array($entry)) continue;
            $e = DayEntry::fromArray($entry);
            if (!$e->isPublished() || !$e->isPublishable() || !self::hasRequiredTextAndImage($e)) continue;
            $out[$date] = $e;
            $rawOut[$date] = $e->toArray();
        }

        self::cacheSet($key, $rawOut, 6 * HOUR_IN_SECONDS);
        return $out;
    }

    public static function set(string $date, DayEntry $entry): bool {
        $date = DayEntry::normalizeDate($date);
        if ($date === '') return false;
        $before = self::get($date);
        $ok = DayEntryRepository::set($date, $entry->toArray());
        if ($ok) {
            if (class_exists(CacheVersion::class)) {
                CacheVersion::setDayLastModified($date);
                CacheVersion::bump();
            }

            if ($entry->isPublished() && class_exists(CachePurge::class)) {
                CachePurge::purgeDate($date);
            }
            if ($before->isPublished() && !$entry->isPublished() && class_exists(CachePurge::class)) {
                CachePurge::purgeDate($date);
            }
        }
        return $ok;
    }

    /** @param array<string,mixed> $entry */
    public static function setFromArray(string $date, array $entry): bool {
        $date = DayEntry::normalizeDate($date);
        if ($date === '') return false;
        $before = self::get($date);
        $san = DayEntry::sanitizeArray($entry);
        $after = DayEntry::fromArray($san);

        $ok = DayEntryRepository::set($date, $san);
        if ($ok) {
            if (class_exists(CacheVersion::class)) {
                CacheVersion::setDayLastModified($date);
                CacheVersion::bump();
            }

            // Purge public caches for published days (including updates).
            if ($after->isPublished() && class_exists(CachePurge::class)) {
                CachePurge::purgeDate($date);
            }
            // If a day was just unpublished, also purge to avoid stale public caches.
            if ($before->isPublished() && !$after->isPublished() && class_exists(CachePurge::class)) {
                CachePurge::purgeDate($date);
            }
        }
        return $ok;
    }

    /**
     * Finds the nearest previous day with any saved text.
     *
     * @return array{from_date:string,content:string,daily_text:string}|null
     */
    public static function findPreviousText(string $date): ?array {
        $date = DayEntry::normalizeDate($date);
        if ($date === '') return null;
        return DayEntryRepository::findPreviousText($date);
    }

    public static function previousPublishedDate(string $date): string {
        $date = DayEntry::normalizeDate($date);
        if ($date === '') return '';

        $key = 'dtarot_prev_' . self::cacheVer() . '_' . $date;
        $cached = self::cacheGet($key);
        if (is_array($cached) && array_key_exists('val', $cached)) {
            return is_string($cached['val']) ? (string)$cached['val'] : '';
        }

        $val = DayEntryRepository::previousPublishedDate($date);
        self::cacheSet($key, ['val' => $val], 12 * HOUR_IN_SECONDS);
        return $val;
    }

    public static function nextPublishedDate(string $date): string {
        $date = DayEntry::normalizeDate($date);
        if ($date === '') return '';

        $key = 'dtarot_next_' . self::cacheVer() . '_' . $date;
        $cached = self::cacheGet($key);
        if (is_array($cached) && array_key_exists('val', $cached)) {
            return is_string($cached['val']) ? (string)$cached['val'] : '';
        }

        $val = DayEntryRepository::nextPublishedDate($date);
        self::cacheSet($key, ['val' => $val], 12 * HOUR_IN_SECONDS);
        return $val;
    }

    private static function hasText(DayEntry $e): bool {
        $c = trim((string)wp_strip_all_tags((string)$e->content));
        $d = trim((string)wp_strip_all_tags((string)$e->dailyText));
        return ($c !== '' || $d !== '');
    }

    private static function resolveCardImageUrl(int $deckId, string $cardId): string {
        if ($deckId <= 0 || trim($cardId) === '') return '';
        $imgs = get_post_meta($deckId, '_dtarot_cards', true);
        if (!is_array($imgs) || !$imgs) return '';

        foreach (Cards::kipperGypsyAliases($cardId) as $id) {
            if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
            $url = trim((string)$imgs[$id]);
            if ($url !== '') return $url;
        }

        return '';
    }

    private static function hasImage(DayEntry $e): bool {
        if (trim((string)$e->imageOverrideUrl) !== '') return true;
        return self::resolveCardImageUrl((int)$e->deckId, (string)$e->cardId) !== '';
    }

    /**
     * Published entries must have both text and an image.
     *
     * - Image: custom override URL OR deck card image.
     * - Text: at least one of content / daily_text.
     */
    private static function hasRequiredTextAndImage(DayEntry $e): bool {
        return self::hasText($e) && self::hasImage($e);
    }
}
