<?php
declare(strict_types=1);


namespace DailyTarot\Calendar;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.DateTime.RestrictedFunctions.date_date

// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter



/**
 * Thin wrapper around the current option-based calendar storage.
 *
 * This isolates storage decisions so we can later migrate to CPT/custom tables
 * without rewriting admin UI, shortcodes, automation, etc.
 */
final class DayEntryRepository {

    private const BACKEND_OPTION = 'option';
    private const BACKEND_TABLE = 'table';
    private const BACKEND_SETTING = 'dtarot_calendar_storage_backend';

    public static function backend(): string {
        $b = apply_filters('dtarot_calendar_storage_backend', (string)get_option(self::BACKEND_SETTING, self::BACKEND_OPTION));
        $b = is_string($b) ? strtolower(trim($b)) : self::BACKEND_OPTION;
        if ($b !== self::BACKEND_TABLE) return self::BACKEND_OPTION;
        if (!class_exists(DayEntryTable::class)) return self::BACKEND_OPTION;
        if (!DayEntryTable::exists()) return self::BACKEND_OPTION;
        return self::BACKEND_TABLE;
    }

    /** @return array<string,array<string,mixed>> */
    public static function allFromOption(): array {
        $entries = get_option('dtarot_calendar_entries', []);
        if (!is_array($entries)) $entries = [];
        return self::sanitizeMany($entries);
    }

    public static function countOptionEntries(): int {
        return count(self::allFromOption());
    }

    /** @return array<string,array<string,mixed>> */
    public static function all(): array {
        if (self::backend() === self::BACKEND_TABLE) {
            return self::allFromTable();
        }
        return self::allFromOption();
    }

    /** @return array<string,array<string,mixed>> */
    private static function allFromTable(): array {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return [];

        $rows = $wpdb->get_results(
            'SELECT date,status,deck_id,pack_id,card_id,content,daily_text,image_override_attachment_id,image_override_url FROM ' . $table,
            ARRAY_A
        );
        if (!is_array($rows)) return [];

        $out = [];
        foreach ($rows as $r) {
            if (!is_array($r)) continue;
            $date = isset($r['date']) ? (string)$r['date'] : '';
            $date = self::normalizeDate($date);
            if ($date === '') continue;

            $out[$date] = self::sanitizeEntry([
                'deck' => isset($r['deck_id']) ? (string)$r['deck_id'] : '',
                'card' => isset($r['card_id']) ? (string)$r['card_id'] : '',
                'pack' => isset($r['pack_id']) ? (string)$r['pack_id'] : '',
                'status' => isset($r['status']) ? (string)$r['status'] : 'draft',
                'content' => isset($r['content']) ? (string)$r['content'] : '',
                'daily_text' => isset($r['daily_text']) ? (string)$r['daily_text'] : '',
                'image_override_attachment_id' => isset($r['image_override_attachment_id']) ? (string)$r['image_override_attachment_id'] : '',
                'image_override_url' => isset($r['image_override_url']) ? (string)$r['image_override_url'] : '',
            ]);
        }

        ksort($out);
        return $out;
    }

    /**
     * Replace all stored calendar entries.
     *
     * @param array<string,mixed> $entries Map of date => entry
     */
    public static function replaceAll(array $entries): bool {
        $clean = self::sanitizeMany($entries);
        if (self::backend() === self::BACKEND_TABLE) {
            return self::replaceAllTable($clean);
        }
        update_option('dtarot_calendar_entries', $clean, false);
        return true;
    }

    /**
     * Merge entries into the stored calendar entries (incoming wins on key collisions).
     *
     * @param array<string,mixed> $entries Map of date => entry
     */
    public static function mergeAll(array $entries): bool {
        $clean = self::sanitizeMany($entries);
        if (self::backend() === self::BACKEND_TABLE) {
            return self::mergeAllTable($clean);
        }
        $existing = get_option('dtarot_calendar_entries', []);
        if (!is_array($existing)) $existing = [];
        update_option('dtarot_calendar_entries', array_merge(self::sanitizeMany($existing), $clean), false);
        return true;
    }

    /** @return array<string,mixed> */
    public static function get(string $date): array {
        $date = self::normalizeDate($date);
        if ($date === '') return self::emptyEntry();

        if (self::backend() === self::BACKEND_TABLE) {
            return self::getFromTable($date);
        }

        $entries = get_option('dtarot_calendar_entries', []);
        if (!is_array($entries)) $entries = [];

        $entry = $entries[$date] ?? self::emptyEntry();
        if (!is_array($entry)) $entry = self::emptyEntry();

        return self::sanitizeEntry($entry);
    }

    /** @return array<string,mixed> */
    private static function getFromTable(string $date): array {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return self::emptyEntry();

        $sql = $wpdb->prepare(
            'SELECT status,deck_id,pack_id,card_id,content,daily_text,image_override_attachment_id,image_override_url FROM ' . $table . ' WHERE date=%s',
            $date
        );
        $row = $wpdb->get_row($sql, ARRAY_A);
        if (!is_array($row)) return self::emptyEntry();

        return self::sanitizeEntry([
            'deck' => isset($row['deck_id']) ? (string)$row['deck_id'] : '',
            'card' => isset($row['card_id']) ? (string)$row['card_id'] : '',
            'pack' => isset($row['pack_id']) ? (string)$row['pack_id'] : '',
            'status' => isset($row['status']) ? (string)$row['status'] : 'draft',
            'content' => isset($row['content']) ? (string)$row['content'] : '',
            'daily_text' => isset($row['daily_text']) ? (string)$row['daily_text'] : '',
            'image_override_attachment_id' => isset($row['image_override_attachment_id']) ? (string)$row['image_override_attachment_id'] : '',
            'image_override_url' => isset($row['image_override_url']) ? (string)$row['image_override_url'] : '',
        ]);
    }

    /** @return array<string,mixed> */
    public static function getPublished(string $date): array {
        $entry = self::get($date);
        return (($entry['status'] ?? 'draft') === 'publish') ? $entry : self::emptyEntry();
    }

    /**
     * Fetch published entries for an inclusive date range.
     *
     * @return array<string,array<string,mixed>> Map of date => entry
     */
    public static function publishedRange(string $startDate, string $endDate): array {
        $startDate = self::normalizeDate($startDate);
        $endDate = self::normalizeDate($endDate);
        if ($startDate === '' || $endDate === '') return [];

        $startTs = strtotime($startDate . ' 00:00:00');
        $endTs = strtotime($endDate . ' 00:00:00');
        if ($startTs === false || $endTs === false) return [];

        if ($startTs > $endTs) {
            [$startDate, $endDate] = [$endDate, $startDate];
            [$startTs, $endTs] = [$endTs, $startTs];
        }

        // Guard against accidental huge ranges.
        $maxDays = 400;
        $days = (int)floor(($endTs - $startTs) / DAY_IN_SECONDS) + 1;
        if ($days <= 0) return [];
        if ($days > $maxDays) {
            $endTs = $startTs + (($maxDays - 1) * DAY_IN_SECONDS);
            $endDate = date('Y-m-d', $endTs);
        }

        if (self::backend() === self::BACKEND_TABLE) {
            return self::publishedRangeFromTable($startDate, $endDate);
        }

        $entries = get_option('dtarot_calendar_entries', []);
        if (!is_array($entries)) $entries = [];

        $out = [];
        for ($ts = $startTs; $ts <= $endTs; $ts += DAY_IN_SECONDS) {
            $d = date('Y-m-d', $ts);
            if (!isset($entries[$d]) || !is_array($entries[$d])) continue;
            $e = self::sanitizeEntry($entries[$d]);
            if (($e['status'] ?? 'draft') !== 'publish') continue;
            $out[$d] = $e;
        }

        return $out;
    }

    /** @return array<string,array<string,mixed>> */
    private static function publishedRangeFromTable(string $startDate, string $endDate): array {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return [];

        $sql = $wpdb->prepare(
            'SELECT date,status,deck_id,pack_id,card_id,content,daily_text,image_override_attachment_id,image_override_url FROM ' . $table . " WHERE date >= %s AND date <= %s AND status='publish' ORDER BY date ASC",
            $startDate,
            $endDate
        );
        $rows = $wpdb->get_results($sql, ARRAY_A);
        if (!is_array($rows)) return [];

        $out = [];
        foreach ($rows as $row) {
            if (!is_array($row)) continue;
            $d = isset($row['date']) ? self::normalizeDate((string)$row['date']) : '';
            if ($d === '') continue;
            $out[$d] = self::sanitizeEntry([
                'deck' => isset($row['deck_id']) ? (string)$row['deck_id'] : '',
                'card' => isset($row['card_id']) ? (string)$row['card_id'] : '',
                'pack' => isset($row['pack_id']) ? (string)$row['pack_id'] : '',
                'status' => isset($row['status']) ? (string)$row['status'] : 'draft',
                'content' => isset($row['content']) ? (string)$row['content'] : '',
                'daily_text' => isset($row['daily_text']) ? (string)$row['daily_text'] : '',
                'image_override_attachment_id' => isset($row['image_override_attachment_id']) ? (string)$row['image_override_attachment_id'] : '',
                'image_override_url' => isset($row['image_override_url']) ? (string)$row['image_override_url'] : '',
            ]);
        }

        return $out;
    }

    /**
     * Returns up to $limit published entries going backwards from $fromDate (inclusive).
     *
     * @return array<string,array<string,mixed>> Map of date => entry
     */
    public static function latestPublished(int $limit, string $fromDate = ''): array {
        if ($limit <= 0) return [];

        $fromDate = trim($fromDate);
        if ($fromDate === '') {
            $fromDate = function_exists('wp_date') ? (string)wp_date('Y-m-d') : (string)current_time('Y-m-d');
        }
        $fromDate = self::normalizeDate($fromDate);
        if ($fromDate === '') return [];

        if (self::backend() === self::BACKEND_TABLE) {
            return self::latestPublishedFromTable($limit, $fromDate);
        }

        $out = [];
        $t = strtotime($fromDate . ' 00:00:00');
        if ($t === false) return [];

        // Scan at most 365 days back to prevent heavy loops.
        $maxScan = 365;
        for ($i = 0; $i < $maxScan && count($out) < $limit; $i++) {
            $d = date('Y-m-d', strtotime("-$i day", $t));
            $e = self::getPublished($d);
            if (!empty($e['deck']) && !empty($e['card'])) {
                $out[$d] = $e;
            }
        }

        return $out;
    }

    /**
     * Finds the nearest previous published date (strictly before $date).
     *
     * Returns empty string if not found.
     */
    public static function previousPublishedDate(string $date): string {
        $date = self::normalizeDate($date);
        if ($date === '') return '';

        if (self::backend() === self::BACKEND_TABLE) {
            return self::previousPublishedDateFromTable($date);
        }

        $t = strtotime($date . ' 00:00:00');
        if ($t === false) return '';

        $maxScan = 365;
        for ($i = 1; $i <= $maxScan; $i++) {
            $d = date('Y-m-d', strtotime("-$i day", $t));
            $e = self::getPublished($d);
            if (!empty($e['deck']) && !empty($e['card'])) return $d;
        }

        return '';
    }

    /**
     * Finds the nearest next published date (strictly after $date).
     *
     * Returns empty string if not found.
     */
    public static function nextPublishedDate(string $date): string {
        $date = self::normalizeDate($date);
        if ($date === '') return '';

        if (self::backend() === self::BACKEND_TABLE) {
            return self::nextPublishedDateFromTable($date);
        }

        $t = strtotime($date . ' 00:00:00');
        if ($t === false) return '';

        $maxScan = 365;
        for ($i = 1; $i <= $maxScan; $i++) {
            $d = date('Y-m-d', strtotime("+$i day", $t));
            $e = self::getPublished($d);
            if (!empty($e['deck']) && !empty($e['card'])) return $d;
        }

        return '';
    }

    private static function previousPublishedDateFromTable(string $date): string {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return '';

        $sql = $wpdb->prepare(
            'SELECT date FROM ' . $table . " WHERE date < %s AND status='publish' AND deck_id > 0 AND card_id <> '' ORDER BY date DESC LIMIT 1",
            $date
        );
        $val = $wpdb->get_var($sql);
        return is_string($val) ? self::normalizeDate($val) : '';
    }

    private static function nextPublishedDateFromTable(string $date): string {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return '';

        $sql = $wpdb->prepare(
            'SELECT date FROM ' . $table . " WHERE date > %s AND status='publish' AND deck_id > 0 AND card_id <> '' ORDER BY date ASC LIMIT 1",
            $date
        );
        $val = $wpdb->get_var($sql);
        return is_string($val) ? self::normalizeDate($val) : '';
    }

    /** @return array<string,array<string,mixed>> */
    private static function latestPublishedFromTable(int $limit, string $fromDate): array {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return [];

        $limit = max(1, (int)$limit);
        if ($limit > 365) $limit = 365;

        $sql = $wpdb->prepare(
            "SELECT date,status,deck_id,pack_id,card_id,content,daily_text,image_override_attachment_id,image_override_url\n" .
            'FROM ' . $table . "\n" .
            "WHERE date <= %s AND status = 'publish' AND deck_id > 0 AND card_id <> ''\n" .
            "ORDER BY date DESC\n" .
            "LIMIT %d",
            $fromDate,
            $limit
        );
        $rows = $wpdb->get_results($sql, ARRAY_A);
        if (!is_array($rows)) return [];

        $out = [];
        foreach ($rows as $row) {
            if (!is_array($row)) continue;
            $d = isset($row['date']) ? self::normalizeDate((string)$row['date']) : '';
            if ($d === '') continue;
            $out[$d] = self::sanitizeEntry([
                'deck' => isset($row['deck_id']) ? (string)$row['deck_id'] : '',
                'card' => isset($row['card_id']) ? (string)$row['card_id'] : '',
                'pack' => isset($row['pack_id']) ? (string)$row['pack_id'] : '',
                'status' => isset($row['status']) ? (string)$row['status'] : 'draft',
                'content' => isset($row['content']) ? (string)$row['content'] : '',
                'daily_text' => isset($row['daily_text']) ? (string)$row['daily_text'] : '',
                'image_override_attachment_id' => isset($row['image_override_attachment_id']) ? (string)$row['image_override_attachment_id'] : '',
                'image_override_url' => isset($row['image_override_url']) ? (string)$row['image_override_url'] : '',
            ]);
        }

        return $out;
    }

    /**
     * Upserts a calendar entry for a date.
     *
     * @param array<string,mixed> $entry
     */
    public static function set(string $date, array $entry): bool {
        $date = self::normalizeDate($date);
        if ($date === '') return false;

        if (self::backend() === self::BACKEND_TABLE) {
            return self::setToTable($date, $entry);
        }

        $entries = get_option('dtarot_calendar_entries', []);
        if (!is_array($entries)) $entries = [];

        $entries[$date] = self::sanitizeEntry($entry);
        update_option('dtarot_calendar_entries', $entries, false);
        return true;
    }

    /** @param array<string,mixed> $entry */
    private static function setToTable(string $date, array $entry): bool {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return false;

        $e = self::sanitizeEntry($entry);
        $now = function_exists('current_time') ? (string)current_time('mysql') : gmdate('Y-m-d H:i:s');

        $existingCreated = $wpdb->get_var(
            $wpdb->prepare('SELECT created_at FROM ' . $table . ' WHERE date=%s', $date)
        );
        $createdAt = (is_string($existingCreated) && $existingCreated !== '' && $existingCreated !== '0000-00-00 00:00:00') ? $existingCreated : $now;

        $data = [
            'date' => $date,
            'status' => (string)$e['status'],
            'deck_id' => (int)($e['deck'] !== '' ? $e['deck'] : 0),
            'pack_id' => (int)($e['pack'] !== '' ? $e['pack'] : 0),
            'card_id' => (string)$e['card'],
            'content' => (string)$e['content'],
            'daily_text' => (string)$e['daily_text'],
            'image_override_attachment_id' => (int)($e['image_override_attachment_id'] !== '' ? $e['image_override_attachment_id'] : 0),
            'image_override_url' => (string)$e['image_override_url'],
            'created_at' => $createdAt,
            'updated_at' => $now,
        ];

        $ok = $wpdb->replace($table, $data, ['%s','%s','%d','%d','%s','%s','%s','%d','%s','%s','%s']);
        return $ok !== false;
    }

    /** @param array<string,array<string,mixed>> $entries */
    private static function replaceAllTable(array $entries): bool {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return false;
        $wpdb->query('TRUNCATE TABLE ' . $table);
        return self::mergeAllTable($entries);
    }

    /** @param array<string,array<string,mixed>> $entries */
    private static function mergeAllTable(array $entries): bool {
        foreach ($entries as $date => $entry) {
            if (!is_string($date) || !is_array($entry)) continue;
            self::setToTable($date, $entry);
        }
        return true;
    }

    /**
     * Finds the nearest previous day with any saved text.
     *
     * @return array{from_date:string,content:string,daily_text:string}|null
     */
    public static function findPreviousText(string $date): ?array {
        $date = self::normalizeDate($date);
        if ($date === '') return null;

        if (self::backend() === self::BACKEND_TABLE) {
            return self::findPreviousTextTable($date);
        }

        $t = strtotime($date . ' 00:00:00');
        if ($t === false) return null;
        $entries = self::allFromOption();

        for ($i = 1; $i <= 365; $i++) {
            $d = date('Y-m-d', strtotime("-$i day", $t));
            $e = $entries[$d] ?? null;
            if (!is_array($e)) continue;
            $entry = self::sanitizeEntry($e);

            $content = (string)($entry['content'] ?? '');
            $dailyText = (string)($entry['daily_text'] ?? '');
            $hasAny = (trim(wp_strip_all_tags($content)) !== '') || (trim(wp_strip_all_tags($dailyText)) !== '');
            if (!$hasAny) continue;

            return [
                'from_date' => $d,
                'content' => $content,
                'daily_text' => $dailyText,
            ];
        }

        return null;
    }

    /** @return array{from_date:string,content:string,daily_text:string}|null */
    private static function findPreviousTextTable(string $date): ?array {
        global $wpdb;
        $table = class_exists(DayEntryTable::class) ? DayEntryTable::name() : '';
        if ($table === '' || !preg_match('/^[A-Za-z0-9_]+$/', $table) || !isset($wpdb) || !is_object($wpdb)) return null;

        $sql = $wpdb->prepare(
            'SELECT date,content,daily_text FROM ' . $table . "\n" .
            "WHERE date < %s AND (content <> '' OR daily_text <> '')\n" .
            "ORDER BY date DESC\n" .
            "LIMIT 30",
            $date
        );
        $rows = $wpdb->get_results($sql, ARRAY_A);
        if (!is_array($rows)) return null;

        foreach ($rows as $row) {
            if (!is_array($row)) continue;
            $d = isset($row['date']) ? self::normalizeDate((string)$row['date']) : '';
            if ($d === '') continue;
            $content = isset($row['content']) ? (string)$row['content'] : '';
            $dailyText = isset($row['daily_text']) ? (string)$row['daily_text'] : '';
            $hasAny = (trim(wp_strip_all_tags($content)) !== '') || (trim(wp_strip_all_tags($dailyText)) !== '');
            if (!$hasAny) continue;

            return [
                'from_date' => $d,
                'content' => $content,
                'daily_text' => $dailyText,
            ];
        }

        return null;
    }

    /**
     * Migrates option-based calendar entries into the custom DB table.
     * Does not delete option storage (safe rollback).
     *
     * @return array{ok:bool,count:int,msg:string}
     */
    public static function migrateOptionToTable(bool $replace): array {
        if (!class_exists(DayEntryTable::class)) {
            return ['ok' => false, 'count' => 0, 'msg' => 'no_table_class'];
        }
        DayEntryTable::ensureSchema();
        if (!DayEntryTable::exists()) {
            return ['ok' => false, 'count' => 0, 'msg' => 'table_missing'];
        }

        $entries = self::allFromOption();
        if ($replace) {
            self::replaceAllTable($entries);
        } else {
            self::mergeAllTable($entries);
        }

        update_option(self::BACKEND_SETTING, self::BACKEND_TABLE, false);
        return ['ok' => true, 'count' => count($entries), 'msg' => 'migrated'];
    }

    /** @return array<string,mixed> */
    public static function emptyEntry(): array {
        return [
            'deck' => '',
            'card' => '',
            'pack' => '',
            'status' => 'draft',
            'content' => '',
            'daily_text' => '',
            'image_override_attachment_id' => '',
            'image_override_url' => '',
        ];
    }

    /** @param array<string,mixed> $entry */
    private static function sanitizeEntry(array $entry): array {
        $out = self::emptyEntry();
        foreach (array_keys($out) as $k) {
            if (isset($entry[$k]) && !is_array($entry[$k])) {
                $out[$k] = (string) $entry[$k];
            }
        }
        if ($out['status'] === '') $out['status'] = 'draft';
        return $out;
    }

    /**
     * @param array<string,mixed> $entries
     * @return array<string,array<string,mixed>>
     */
    private static function sanitizeMany(array $entries): array {
        $out = [];
        foreach ($entries as $date => $entry) {
            if (!is_string($date)) continue;
            $date = self::normalizeDate($date);
            if ($date === '') continue;
            if (!is_array($entry)) continue;
            $out[$date] = self::sanitizeEntry($entry);
        }
        return $out;
    }

    private static function normalizeDate(string $date): string {
        $date = trim($date);
        if ($date === '') return '';

        // Accept YYYY-MM-DD only.
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) return '';
        return $date;
    }
}
