<?php
declare(strict_types=1);


namespace DailyTarot\Calendar;
if (!defined('ABSPATH')) { exit; }


/**
 * Domain contract for a calendar day entry.
 *
 * Storage format remains the same as legacy option storage:
 * - deck, pack are stringified IDs
 * - card/status/content/daily_text are strings
 */
final class DayEntry {

    public const STATUS_DRAFT = 'draft';
    public const STATUS_PUBLISH = 'publish';

    public int $deckId;
    public int $packId;
    public string $cardId;
    public string $status;
    public string $content;
    public string $dailyText;

    public int $imageOverrideAttachmentId;
    public string $imageOverrideUrl;

    public function __construct(
        int $deckId = 0,
        int $packId = 0,
        string $cardId = '',
        string $status = self::STATUS_DRAFT,
        string $content = '',
        string $dailyText = '',
        int $imageOverrideAttachmentId = 0,
        string $imageOverrideUrl = ''
    ) {
        $this->deckId = $deckId;
        $this->packId = $packId;
        $this->cardId = $cardId;
        $this->status = $status;
        $this->content = $content;
        $this->dailyText = $dailyText;

        $this->imageOverrideAttachmentId = $imageOverrideAttachmentId;
        $this->imageOverrideUrl = $imageOverrideUrl;
    }

    public static function normalizeDate(string $date): string {
        $date = trim($date);
        if ($date === '') return '';
        return preg_match('/^\d{4}-\d{2}-\d{2}$/', $date) ? $date : '';
    }

    /** @param array<string,mixed> $entry */
    public static function fromArray(array $entry): self {
        $deck = isset($entry['deck']) && !is_array($entry['deck']) ? (string)$entry['deck'] : '';
        $pack = isset($entry['pack']) && !is_array($entry['pack']) ? (string)$entry['pack'] : '';

        $deckId = (is_string($deck) && ctype_digit($deck)) ? (int)$deck : (int)$deck;
        $packId = (is_string($pack) && ctype_digit($pack)) ? (int)$pack : (int)$pack;

        $cardId = isset($entry['card']) && !is_array($entry['card']) ? (string)$entry['card'] : '';
        $status = isset($entry['status']) && !is_array($entry['status']) ? (string)$entry['status'] : self::STATUS_DRAFT;
        $content = isset($entry['content']) && !is_array($entry['content']) ? (string)$entry['content'] : '';
        $dailyText = isset($entry['daily_text']) && !is_array($entry['daily_text']) ? (string)$entry['daily_text'] : '';

        $imgAtt = isset($entry['image_override_attachment_id']) && !is_array($entry['image_override_attachment_id'])
            ? (string)$entry['image_override_attachment_id']
            : '';
        $imgAttId = (is_string($imgAtt) && ctype_digit($imgAtt)) ? (int)$imgAtt : (int)$imgAtt;
        $imgUrl = isset($entry['image_override_url']) && !is_array($entry['image_override_url'])
            ? (string)$entry['image_override_url']
            : '';

        $status = self::normalizeStatus($status);

        return new self(
            $deckId > 0 ? $deckId : 0,
            $packId > 0 ? $packId : 0,
            (string)$cardId,
            $status,
            (string)$content,
            (string)$dailyText,
            $imgAttId > 0 ? $imgAttId : 0,
            (string)$imgUrl
        );
    }

    /** @return array{deck:string,card:string,pack:string,status:string,content:string,daily_text:string,image_override_attachment_id:string,image_override_url:string} */
    public function toArray(): array {
        return [
            'deck' => $this->deckId > 0 ? (string)$this->deckId : '',
            'card' => (string)$this->cardId,
            'pack' => $this->packId > 0 ? (string)$this->packId : '',
            'status' => self::normalizeStatus($this->status),
            'content' => (string)$this->content,
            'daily_text' => (string)$this->dailyText,
            'image_override_attachment_id' => $this->imageOverrideAttachmentId > 0 ? (string)$this->imageOverrideAttachmentId : '',
            'image_override_url' => (string)$this->imageOverrideUrl,
        ];
    }

    /** @param array<string,mixed> $entry
     *  @return array{deck:string,card:string,pack:string,status:string,content:string,daily_text:string,image_override_attachment_id:string,image_override_url:string}
     */
    public static function sanitizeArray(array $entry): array {
        return self::fromArray($entry)->toArray();
    }

    public static function normalizeStatus(string $status): string {
        $status = strtolower(trim($status));
        if ($status === self::STATUS_PUBLISH) return self::STATUS_PUBLISH;
        return self::STATUS_DRAFT;
    }

    public function isPublished(): bool {
        return self::normalizeStatus($this->status) === self::STATUS_PUBLISH;
    }

    public function isPublishable(): bool {
        return $this->deckId > 0 && trim($this->cardId) !== '';
    }

    public function isEmpty(): bool {
        return $this->deckId <= 0
            && $this->packId <= 0
            && trim($this->cardId) === ''
            && trim($this->content) === ''
            && trim($this->dailyText) === '';
    }
}
