<?php
declare(strict_types=1);


namespace DailyTarot\CPT;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText



use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;

final class MeaningPack {

    public static function register(): void {
        if (!post_type_exists(PostTypes::MEANING_PACK)) {
            self::registerPostType();
        }
        add_action('init', [__CLASS__, 'registerPostType'], 0);

        // Admin list table columns.
        add_filter('manage_' . PostTypes::MEANING_PACK . '_posts_columns', [__CLASS__, 'columns']);
        add_action('manage_' . PostTypes::MEANING_PACK . '_posts_custom_column', [__CLASS__, 'columnContent'], 10, 2);

        // Back-compat: legacy list table hooks.
        add_filter('manage_' . PostTypes::LEGACY_MEANING_PACK . '_posts_columns', static function (array $cols): array {
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core list-table hook name.
            return (array)apply_filters('manage_' . PostTypes::MEANING_PACK . '_posts_columns', $cols);
        });
        add_action('manage_' . PostTypes::LEGACY_MEANING_PACK . '_posts_custom_column', static function (string $column, int $postId): void {
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core list-table hook name.
            do_action('manage_' . PostTypes::MEANING_PACK . '_posts_custom_column', $column, $postId);
        }, 10, 2);
    }

    public static function registerPostType(): void {
        if (post_type_exists(PostTypes::MEANING_PACK)) return;

        $labels = [
            'name'          => did_action('init') ? __('Meaning Packs','daily-tarot') : 'Meaning Packs',
            'singular_name' => did_action('init') ? __('Meaning Pack','daily-tarot') : 'Meaning Pack',
            'add_new_item'  => did_action('init') ? __('Add Meaning Pack','daily-tarot') : 'Add Meaning Pack',
            'edit_item'     => did_action('init') ? __('Edit Meaning Pack','daily-tarot') : 'Edit Meaning Pack',
        ];

        register_post_type(PostTypes::MEANING_PACK, [
            'labels' => $labels,
            'public'       => false,
            'show_ui'      => true,
            'show_in_menu' => false,
            'supports'     => ['title'],
            'rewrite'      => false,
        ]);

        // Back-compat: keep legacy post type registered so old URLs/admin screens work.
        if (!post_type_exists(PostTypes::LEGACY_MEANING_PACK)) {
            register_post_type(PostTypes::LEGACY_MEANING_PACK, [
                'labels' => $labels,
                'public'       => false,
                'show_ui'      => true,
                'show_in_menu' => false,
                'supports'     => ['title'],
                'rewrite'      => false,
            ]);
        }
    }

    /**
     * @param array<string,string> $columns
     * @return array<string,string>
     */
    public static function columns(array $columns): array {
        $out = [];
        foreach ($columns as $key => $label) {
            $out[$key] = $label;
            if ($key === 'title') {
                $out['dtarot_progress'] = (string)__('Progress','daily-tarot');
            }
        }
        if (!isset($out['dtarot_progress'])) {
            $out['dtarot_progress'] = (string)__('Progress','daily-tarot');
        }
        return $out;
    }

    /**
     * @return array{complete:int,incomplete:int,total:int,percent:int}
     */
    private static function progressForPack(int $packId): array {
        $system = Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true));
        $registry = Cards::forSystem($system);
        $total = count((array)$registry);

        $meanings = get_post_meta($packId, '_dtarot_meanings', true);
        if (!is_array($meanings)) $meanings = [];

        $complete = 0;
        foreach (array_keys((array)$registry) as $cardId) {
            $m = isset($meanings[$cardId]) && is_array($meanings[$cardId]) ? $meanings[$cardId] : [];
            $hasUpright = !empty($m['upright']);
            if ($hasUpright) $complete++;
        }
        $incomplete = max(0, $total - $complete);
        $percent = $total > 0 ? (int)round(($complete / $total) * 100) : 0;

        return [
            'complete' => $complete,
            'incomplete' => $incomplete,
            'total' => $total,
            'percent' => max(0, min(100, $percent)),
        ];
    }

    public static function columnContent(string $column, int $postId): void {
        if ($column !== 'dtarot_progress') return;

        $p = self::progressForPack($postId);
        echo '<div class="dtarot-pack-progressbar" role="img" aria-label="' . esc_attr(sprintf(__('%d of %d complete','daily-tarot'), (int)$p['complete'], (int)$p['total'])) . '">';
        echo '<div class="dtarot-pack-progressbar-fill" style="width: ' . esc_attr((string)$p['percent']) . '%;"></div>';
        echo '</div>';
        echo '<div class="dtarot-pack-progressbar-meta">' . esc_html(sprintf(__('%d complete / %d incomplete','daily-tarot'), (int)$p['complete'], (int)$p['incomplete'])) . '</div>';
    }
}
