<?php
declare(strict_types=1);


namespace DailyTarot\CPT;
if (!defined('ABSPATH')) { exit; }

use DailyTarot\Support\PostTypes;


final class Booking {

    public static function register(): void {
        add_action('init', static function (): void {
            register_post_type(PostTypes::BOOKING, [
                'labels' => [
                    'name' => __('Bookings','daily-tarot'),
                    'singular_name' => __('Booking','daily-tarot'),
                    'add_new_item' => __('Add Booking','daily-tarot'),
                    'edit_item' => __('Edit Booking','daily-tarot'),
                ],
                'public' => false,
                'show_ui' => true,
                'show_in_menu' => false,
                'supports' => ['title'],
                'rewrite' => false,
            ]);

            // Back-compat.
            if (!post_type_exists(PostTypes::LEGACY_BOOKING)) {
                register_post_type(PostTypes::LEGACY_BOOKING, [
                    'labels' => [
                        'name' => __('Bookings','daily-tarot'),
                        'singular_name' => __('Booking','daily-tarot'),
                    ],
                    'public' => false,
                    'show_ui' => true,
                    'show_in_menu' => false,
                    'supports' => ['title'],
                    'rewrite' => false,
                ]);
            }
        });

        add_filter('manage_' . PostTypes::BOOKING . '_posts_columns', static function (array $cols): array {
            $out = [];
            foreach ($cols as $k => $v) {
                $out[$k] = $v;
                if ($k === 'title') {
                    $out['dtarot_name'] = __('Name','daily-tarot');
                    $out['dtarot_email'] = __('Email','daily-tarot');
                    $out['dtarot_reading'] = __('Reading','daily-tarot');
                    $out['dtarot_time'] = __('Date & Time','daily-tarot');
                    $out['dtarot_status'] = __('Status','daily-tarot');
                }
            }
            return $out;
        });

        add_action('manage_' . PostTypes::BOOKING . '_posts_custom_column', static function (string $column, int $postId): void {
            if ($column === 'dtarot_reading') {
                $rid = (int)get_post_meta($postId, '_dtarot_booking_reading_type', true);
                $title = $rid > 0 ? (string)get_the_title($rid) : '';
                echo esc_html($title !== '' ? $title : __('(none)','daily-tarot'));
            } elseif ($column === 'dtarot_name') {
                $name = (string)get_post_meta($postId, '_dtarot_booking_name', true);
                echo esc_html($name !== '' ? $name : __('(none)','daily-tarot'));
            } elseif ($column === 'dtarot_email') {
                $email = (string)get_post_meta($postId, '_dtarot_booking_email', true);
                echo esc_html($email !== '' ? $email : __('(none)','daily-tarot'));
            } elseif ($column === 'dtarot_time') {
                $start = (int)get_post_meta($postId, '_dtarot_booking_start_utc', true);
                $tz = (string)get_post_meta($postId, '_dtarot_booking_timezone', true);
                if ($start > 0) {
                    $admin = function_exists('wp_date') ? wp_date('Y-m-d H:i', $start, wp_timezone()) : gmdate('Y-m-d H:i', $start);
                    $label = $admin . ' (admin)';
                    if ($tz !== '') {
                        try {
                            $userTz = new \DateTimeZone($tz);
                            $user = function_exists('wp_date') ? wp_date('Y-m-d H:i', $start, $userTz) : gmdate('Y-m-d H:i', $start);
                            $label .= ' / ' . $user . ' (' . $tz . ')';
                        } catch (\Exception $e) {
                        }
                    }
                    echo esc_html($label);
                }
            } elseif ($column === 'dtarot_status') {
                $status = (string)get_post_meta($postId, '_dtarot_booking_status', true);
                if ($status === '') $status = 'pending';
                echo esc_html(ucfirst($status));
            }
        }, 10, 2);

        // Legacy list table hooks.
        add_filter('manage_' . PostTypes::LEGACY_BOOKING . '_posts_columns', static function (array $cols): array {
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core list-table hook name.
            return (array)apply_filters('manage_' . PostTypes::BOOKING . '_posts_columns', $cols);
        });
        add_action('manage_' . PostTypes::LEGACY_BOOKING . '_posts_custom_column', static function (string $column, int $postId): void {
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core list-table hook name.
            do_action('manage_' . PostTypes::BOOKING . '_posts_custom_column', $column, $postId);
        }, 10, 2);
    }
}
