<?php
declare(strict_types=1);


namespace DailyTarot\Blocks;
if (!defined('ABSPATH')) { exit; }


use DailyTarot\Frontend\ReadableRoutes;
use DailyTarot\Frontend\Shortcodes;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\PostTypes;
use DailyTarot\Support\SpreadMeaningPacks;
use DailyTarot\Support\SpreadPresets;

final class Blocks {

    public static function init(): void {
        add_action('init', [__CLASS__, 'register']);
        add_filter('block_categories_all', [__CLASS__, 'registerCategory'], 10, 2);
    }

    /** @param array<int,array<string,mixed>> $categories */
    public static function registerCategory(array $categories, $context): array {
        foreach ($categories as $cat) {
            if (isset($cat['slug']) && $cat['slug'] === 'daily-tarot') {
                return $categories;
            }
        }

        $categories[] = [
            'slug' => 'daily-tarot',
            'title' => __('Daily Tarot','daily-tarot'),
            'icon' => null,
        ];

        return $categories;
    }

    public static function register(): void {
        if (!function_exists('register_block_type')) return;

        $jsFile = DTAROT_PATH . 'assets/blocks.js';
        $ver = is_readable($jsFile) ? (string)@filemtime($jsFile) : DTAROT_VERSION;

        wp_register_script(
            'dtarot-blocks',
            DTAROT_URL . 'assets/blocks.js',
            ['wp-blocks','wp-element','wp-components','wp-block-editor','wp-server-side-render'],
            $ver,
            true
        );

        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);

        $deckOptions = [];
        foreach ((array)$decks as $d) {
            if (!isset($d->ID)) continue;
            $id = (int)$d->ID;
            $title = isset($d->post_title) && $d->post_title ? (string)$d->post_title : (string)__('(no title)','daily-tarot');
            $system = (string)get_post_meta($id, '_dtarot_system', true);
            $system = is_string($system) ? sanitize_key($system) : '';
            if ($system === '') $system = 'tarot';
            if (!in_array($system, ['tarot','lenormand','kipper','gypsy'], true)) $system = 'tarot';

            $deckOptions[] = ['id' => $id, 'title' => $title, 'system' => $system];
        }

        $cards = Cards::allMerged();
        $cardOptions = [];
        foreach ($cards as $id => $name) {
            $cardOptions[] = ['id' => (string)$id, 'title' => (string)$name];
        }

        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);
        $packOptions = [];
        foreach ((array)$packs as $p) {
            if (!isset($p->ID)) continue;
            $id = (int)$p->ID;
            $title = isset($p->post_title) && $p->post_title ? (string)$p->post_title : (string)__('(no title)','daily-tarot');
            $system = (string)get_post_meta($id, '_dtarot_system', true);
            $system = is_string($system) ? sanitize_key($system) : '';
            if ($system === '') $system = 'tarot';
            if (!in_array($system, ['tarot','lenormand','kipper','gypsy'], true)) $system = 'tarot';

            $packOptions[] = ['id' => $id, 'title' => $title, 'system' => $system];
        }

        $spreadPresetOptions = [];
        if (class_exists(SpreadPresets::class)) {
            $presets = SpreadPresets::all();
            foreach ($presets as $id => $preset) {
                $label = isset($preset['label']) && is_string($preset['label']) ? (string)$preset['label'] : (string)$id;
                $spreadPresetOptions[] = ['id' => (string)$id, 'title' => $label];
            }
        }

        $spreadPackOptions = [];
        if (class_exists(SpreadMeaningPacks::class)) {
            $packs = SpreadMeaningPacks::all();
            foreach ($packs as $id => $pack) {
                if (!is_array($pack)) continue;
                $title = isset($pack['title']) && is_string($pack['title']) ? (string)$pack['title'] : (string)$id;
                $spreadPackOptions[] = ['id' => (string)$id, 'title' => $title];
            }
        }

        wp_localize_script('dtarot-blocks', 'DTAROT_BLOCKS', [
            'decks' => $deckOptions,
            'cards' => $cardOptions,
            'packs' => $packOptions,
            'spreadPresets' => $spreadPresetOptions,
            'spreadPacks' => $spreadPackOptions,
            // WP.org directory assets (icons/banners) are not shipped in the plugin zip.
        ]);

        register_block_type('dtarot/daily-tarot', [
            'api_version' => 2,
            'editor_script' => 'dtarot-blocks',
            'category' => 'daily-tarot',
            'title' => __('Daily Tarot','daily-tarot'),
            'description' => __('Card of the day output (wraps the [daily_tarot] shortcode).','daily-tarot'),
            'supports' => ['html' => false],
            'attributes' => [
                'date' => ['type' => 'string', 'default' => ''],
                'mode' => ['type' => 'string', 'default' => 'latest'],
                'days' => ['type' => 'number', 'default' => 7],
                'showDate' => ['type' => 'boolean', 'default' => false],
                'layout' => ['type' => 'string', 'default' => 'minimal'],
                'showShare' => ['type' => 'boolean', 'default' => false],
                'readableUrl' => ['type' => 'string', 'default' => ''],
            ],
            'render_callback' => [__CLASS__, 'renderDailyTarot'],
        ]);

        register_block_type('dtarot/deck-gallery', [
            'api_version' => 2,
            'editor_script' => 'dtarot-blocks',
            'category' => 'daily-tarot',
            'title' => __('Deck / Cards Gallery','daily-tarot'),
            'description' => __('Shows all decks or a single deck grid (wraps [dtarot_decks] / [dtarot_deck]).','daily-tarot'),
            'supports' => ['html' => false],
            'attributes' => [
                'deckId' => ['type' => 'number', 'default' => 0],
                'columns' => ['type' => 'number', 'default' => 6],
            ],
            'render_callback' => [__CLASS__, 'renderDeckGallery'],
        ]);

        register_block_type('dtarot/card-details', [
            'api_version' => 2,
            'editor_script' => 'dtarot-blocks',
            'category' => 'daily-tarot',
            'title' => __('Single Card Details','daily-tarot'),
            'description' => __('Shows a single card image/title from a deck (wraps [dtarot_card]).','daily-tarot'),
            'supports' => ['html' => false],
            'attributes' => [
                'deckId' => ['type' => 'number', 'default' => 0],
                'cardId' => ['type' => 'string', 'default' => ''],
                'showTitle' => ['type' => 'boolean', 'default' => true],
            ],
            'render_callback' => [__CLASS__, 'renderCardDetails'],
        ]);

        register_block_type('dtarot/spread', [
            'api_version' => 2,
            'editor_script' => 'dtarot-blocks',
            'category' => 'daily-tarot',
            'title' => __('Spread','daily-tarot'),
            'description' => __('Shows a spread of multiple cards from a deck (wraps [dtarot_spread]).','daily-tarot'),
            'supports' => ['html' => false],
            'attributes' => [
                'deckId' => ['type' => 'number', 'default' => 0],
                'cards' => ['type' => 'string', 'default' => ''],
                'columns' => ['type' => 'number', 'default' => 0],
                'titles' => ['type' => 'boolean', 'default' => false],
                'preset' => ['type' => 'string', 'default' => ''],
                'spreadPack' => ['type' => 'string', 'default' => ''],
                'cardWidth' => ['type' => 'number', 'default' => 0],
                'cardGap' => ['type' => 'number', 'default' => 0],
                'cardWidthOffset' => ['type' => 'number', 'default' => 0],
                'cardGapOffset' => ['type' => 'number', 'default' => 0],
            ],
            'render_callback' => [__CLASS__, 'renderSpread'],
        ]);

        register_block_type('dtarot/card-page', [
            'api_version' => 2,
            'editor_script' => 'dtarot-blocks',
            'category' => 'daily-tarot',
            'title' => __('Card Page','daily-tarot'),
            'description' => __('Embeds the card detail page output (same renderer as /cards/{system}/{card_id}/).','daily-tarot'),
            'supports' => ['html' => false],
            'attributes' => [
                'system' => ['type' => 'string', 'default' => 'tarot'],
                'cardId' => ['type' => 'string', 'default' => ''],
                'deckId' => ['type' => 'number', 'default' => 0],
                'packId' => ['type' => 'number', 'default' => 0],
            ],
            'render_callback' => [__CLASS__, 'renderCardPage'],
        ]);

        register_block_type('dtarot/booking', [
            'api_version' => 2,
            'editor_script' => 'dtarot-blocks',
            'category' => 'daily-tarot',
            'title' => __('Booking','daily-tarot'),
            'description' => __('Booking form and popup variants (wraps booking shortcodes).','daily-tarot'),
            'supports' => ['html' => false],
            'attributes' => [
                'variant' => ['type' => 'string', 'default' => 'form'],
                'label' => ['type' => 'string', 'default' => ''],
                'title' => ['type' => 'string', 'default' => ''],
                'text' => ['type' => 'string', 'default' => ''],
                'style' => ['type' => 'string', 'default' => ''],
            ],
            'render_callback' => [__CLASS__, 'renderBooking'],
        ]);
    }

    /** @param array<string,mixed> $attrs */
    public static function renderDailyTarot(array $attrs): string {
        if (!class_exists(Shortcodes::class)) return '';

        $date = isset($attrs['date']) && is_string($attrs['date']) ? trim($attrs['date']) : '';
        $mode = isset($attrs['mode']) && is_string($attrs['mode']) ? trim($attrs['mode']) : 'latest';
        $layout = isset($attrs['layout']) && is_string($attrs['layout']) ? trim($attrs['layout']) : 'minimal';
        $days = isset($attrs['days']) ? (int)$attrs['days'] : 7;
        if ($days <= 0) $days = 7;
        if ($days > 90) $days = 90;

        $showDate = !empty($attrs['showDate']);
        $showShare = !empty($attrs['showShare']);
        $readableUrl = isset($attrs['readableUrl']) && is_string($attrs['readableUrl']) ? trim($attrs['readableUrl']) : '';

        $parts = ['[daily_tarot'];
        if ($date !== '') $parts[] = 'date="' . esc_attr($date) . '"';
        if ($mode !== '') $parts[] = 'mode="' . esc_attr($mode) . '"';
        if ($layout !== '') $parts[] = 'layout="' . esc_attr($layout) . '"';
        $parts[] = 'days="' . (int)$days . '"';
        $parts[] = 'show_date="' . ($showDate ? '1' : '0') . '"';
        if ($showShare) $parts[] = 'show_share="1"';
        if ($readableUrl !== '') $parts[] = 'readable_url="' . esc_attr($readableUrl) . '"';
        $parts[] = ']';

        return do_shortcode(implode(' ', $parts));
    }

    /** @param array<string,mixed> $attrs */
    public static function renderDeckGallery(array $attrs): string {
        if (!class_exists(Shortcodes::class)) return '';

        $deckId = isset($attrs['deckId']) ? (int)$attrs['deckId'] : 0;
        $cols = isset($attrs['columns']) ? (int)$attrs['columns'] : 6;
        if ($cols < 2) $cols = 2;
        if ($cols > 10) $cols = 10;

        if ($deckId > 0) {
            return do_shortcode('[dtarot_deck id="' . (int)$deckId . '" columns="' . (int)$cols . '"]');
        }

        return do_shortcode('[dtarot_decks]');
    }

    /** @param array<string,mixed> $attrs */
    public static function renderCardDetails(array $attrs): string {
        if (!class_exists(Shortcodes::class)) return '';

        $deckId = isset($attrs['deckId']) ? (int)$attrs['deckId'] : 0;
        $cardId = isset($attrs['cardId']) && is_string($attrs['cardId']) ? trim($attrs['cardId']) : '';
        $showTitle = !empty($attrs['showTitle']);

        if ($deckId <= 0 || $cardId === '') return '';

        return do_shortcode('[dtarot_card deck_id="' . (int)$deckId . '" card="' . esc_attr($cardId) . '" title="' . ($showTitle ? '1' : '0') . '"]');
    }

    /** @param array<string,mixed> $attrs */
    public static function renderSpread(array $attrs): string {
        if (!class_exists(Shortcodes::class)) return '';

        $deckId = isset($attrs['deckId']) ? (int)$attrs['deckId'] : 0;
        $cards = isset($attrs['cards']) && is_string($attrs['cards']) ? trim($attrs['cards']) : '';
        $cols = isset($attrs['columns']) ? (int)$attrs['columns'] : 0;
        $titles = !empty($attrs['titles']);
        $preset = isset($attrs['preset']) && is_string($attrs['preset']) ? sanitize_key($attrs['preset']) : '';
        $spreadPack = isset($attrs['spreadPack']) && is_string($attrs['spreadPack']) ? sanitize_key($attrs['spreadPack']) : '';
        $cardWidth = isset($attrs['cardWidth']) ? (int)$attrs['cardWidth'] : 0;
        $cardGap = isset($attrs['cardGap']) ? (int)$attrs['cardGap'] : 0;
        $cardWidthOffset = isset($attrs['cardWidthOffset']) ? (int)$attrs['cardWidthOffset'] : 0;
        $cardGapOffset = isset($attrs['cardGapOffset']) ? (int)$attrs['cardGapOffset'] : 0;

        if ($cols <= 0) {
            $autoCols = 0;
            if ($cards !== '') {
                $autoCols = count(array_filter(array_map('trim', explode(',', $cards)), static fn($v) => $v !== ''));
            }
            if ($autoCols <= 0 && $preset !== '' && class_exists(SpreadPresets::class)) {
                $presetDef = SpreadPresets::get($preset);
                if (is_array($presetDef) && isset($presetDef['slots']) && is_array($presetDef['slots'])) {
                    $autoCols = count($presetDef['slots']);
                }
            }
            if ($autoCols <= 0) $autoCols = 3;
            if ($autoCols > 10) $autoCols = 10;
            if ($autoCols < 1) $autoCols = 1;
            $cols = $autoCols;
        } else {
            if ($cols < 2) $cols = 2;
            if ($cols > 10) $cols = 10;
        }

        $parts = ['[dtarot_spread'];
        if ($deckId > 0) {
            $parts[] = 'deck_id="' . (int)$deckId . '"';
        }
        if ($cards !== '') $parts[] = 'cards="' . esc_attr($cards) . '"';
        $parts[] = 'columns="' . (int)$cols . '"';
        $parts[] = 'titles="' . ($titles ? '1' : '0') . '"';
        if ($preset !== '') $parts[] = 'preset="' . esc_attr($preset) . '"';
        if ($spreadPack !== '') $parts[] = 'spread_pack="' . esc_attr($spreadPack) . '"';
        if ($cardWidth > 0) $parts[] = 'card_width="' . (int)$cardWidth . '"';
        if ($cardGap > 0) $parts[] = 'card_gap="' . (int)$cardGap . '"';
        if ($cardWidthOffset !== 0) $parts[] = 'card_width_offset="' . (int)$cardWidthOffset . '"';
        if ($cardGapOffset !== 0) $parts[] = 'card_gap_offset="' . (int)$cardGapOffset . '"';
        $parts[] = ']';

        return do_shortcode(implode(' ', $parts));
    }

    /** @param array<string,mixed> $attrs */
    public static function renderCardPage(array $attrs): string {
        if (!class_exists(ReadableRoutes::class)) return '';

        $system = isset($attrs['system']) && is_string($attrs['system']) ? trim($attrs['system']) : 'tarot';
        $cardId = isset($attrs['cardId']) && is_string($attrs['cardId']) ? trim($attrs['cardId']) : '';
        $deckId = isset($attrs['deckId']) ? (int)$attrs['deckId'] : 0;
        $packId = isset($attrs['packId']) ? (int)$attrs['packId'] : 0;

        if ($cardId === '') return '';

        return ReadableRoutes::renderCardEmbed($system, $cardId, $deckId, $packId);
    }

    /** @param array<string,mixed> $attrs */
    public static function renderBooking(array $attrs): string {
        if (!class_exists(Shortcodes::class)) return '';

        $variant = isset($attrs['variant']) && is_string($attrs['variant']) ? trim($attrs['variant']) : 'form';
        if (!in_array($variant, ['form','button','teaser'], true)) $variant = 'form';

        $label = isset($attrs['label']) && is_string($attrs['label']) ? trim($attrs['label']) : '';
        $title = isset($attrs['title']) && is_string($attrs['title']) ? trim($attrs['title']) : '';
        $text = isset($attrs['text']) && is_string($attrs['text']) ? trim($attrs['text']) : '';
        $style = isset($attrs['style']) && is_string($attrs['style']) ? trim($attrs['style']) : '';

        if ($variant === 'button') {
            $parts = ['[dtarot_booking_button'];
            if ($label !== '') $parts[] = 'label="' . esc_attr($label) . '"';
            if ($style !== '') $parts[] = 'style="' . esc_attr($style) . '"';
            $parts[] = ']';
        } elseif ($variant === 'teaser') {
            $parts = ['[dtarot_booking_teaser'];
            if ($title !== '') $parts[] = 'title="' . esc_attr($title) . '"';
            if ($text !== '') $parts[] = 'text="' . esc_attr($text) . '"';
            if ($label !== '') $parts[] = 'label="' . esc_attr($label) . '"';
            if ($style !== '') $parts[] = 'style="' . esc_attr($style) . '"';
            $parts[] = ']';
        } else {
            $parts = ['[dtarot_booking'];
            if ($style !== '') $parts[] = 'style="' . esc_attr($style) . '"';
            $parts[] = ']';
        }

        return do_shortcode(implode(' ', $parts));
    }
}
