<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



final class UiSettings {

    private const OPT_SHOW_QUICK_ACTIONS = 'dtarot_show_quick_actions';

    /**
     * Registry of auto-save boolean settings.
     *
     * Add new toggles by adding one entry: 'setting_key' => 'setterMethodName'.
     */
    private static function boolSettingMap(): array {
        return [
            'show_quick_actions' => 'setShowQuickActions',
        ];
    }

    public static function saveBoolSetting(string $key, bool $value): bool {
        $key = sanitize_key($key);
        $map = self::boolSettingMap();
        if (!isset($map[$key])) return false;

        $method = (string)$map[$key];
        if (!method_exists(__CLASS__, $method)) return false;

        self::{$method}($value);
        return true;
    }

    public static function showQuickActions(): bool {
        $v = get_option(self::OPT_SHOW_QUICK_ACTIONS, null);
        if ($v === null) return true;
        if ($v === true || $v === 1 || $v === '1') return true;
        if ($v === false || $v === 0 || $v === '0' || $v === '') return false;
        return (bool)$v;
    }

    public static function setShowQuickActions(bool $show): void {
        update_option(self::OPT_SHOW_QUICK_ACTIONS, $show, false);

        // Be defensive: ensure next request reads the updated value even with a persistent object cache.
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::OPT_SHOW_QUICK_ACTIONS, 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }

    public static function handleSave(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_ui_settings');

        $raw = '';
        if (isset($_POST['show_quick_actions']) && !is_array($_POST['show_quick_actions'])) {
            $raw = sanitize_text_field((string)wp_unslash($_POST['show_quick_actions']));
        }
        $show = ($raw === '1');
        self::setShowQuickActions($show);

        wp_safe_redirect(admin_url('admin.php?page=daily-tarot-settings&tab=diagnostics&dtarot_ui=ok'));
        exit;
    }
}
