<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Packs\ManifestValidator;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\SpreadMeaningPacks;
use DailyTarot\Support\SpreadPresets;
use DailyTarot\Support\SpreadScanner;
use DailyTarot\Support\SpreadMappings;
use DailyTarot\Support\SpreadSettings;
use DailyTarot\Support\FeatureFlags;

final class SpreadAdmin {

    private static function filesystem(): ?\WP_Filesystem_Base {
        if (!function_exists('WP_Filesystem')) {
            $inc = ABSPATH . 'wp-admin/includes/file.php';
            if (is_readable($inc)) {
                require_once $inc;
            }
        }

        global $wp_filesystem;
        if (!is_object($wp_filesystem)) {
            WP_Filesystem();
        }

        return is_object($wp_filesystem) ? $wp_filesystem : null;
    }

    private const FORMAT_VERSION = 1;

    public static function handleScan(): void {
        self::check('dtarot_spreads_scan');
        SpreadScanner::scanAll();
        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-spreads',
            'tab' => 'dashboard',
            'msg' => 'scan_ok',
        ], admin_url('admin.php')));
        exit;
    }

    public static function handleMappingSave(): void {
        self::check('dtarot_spread_mapping_save');

        $postId = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        $index = isset($_POST['index']) ? absint(wp_unslash($_POST['index'])) : 0;
        $preset = isset($_POST['preset']) ? sanitize_key((string)wp_unslash($_POST['preset'])) : '';
        $pack = isset($_POST['pack']) ? sanitize_key((string)wp_unslash($_POST['pack'])) : '';
        $deckId = isset($_POST['deck_id']) ? absint(wp_unslash($_POST['deck_id'])) : 0;

        SpreadMappings::setMapping($postId, $index, [
            'preset' => $preset,
            'pack' => $pack,
            'deck_id' => $deckId,
        ]);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-spreads',
            'tab' => 'dashboard',
            'msg' => 'mapping_saved',
        ], admin_url('admin.php')));
        exit;
    }

    public static function handleOptionsSave(): void {
        self::check('dtarot_spread_options_save');

        SpreadSettings::set([
            'default_preset' => isset($_POST['default_preset']) ? sanitize_key((string)wp_unslash($_POST['default_preset'])) : '',
            'default_pack' => isset($_POST['default_pack']) ? sanitize_key((string)wp_unslash($_POST['default_pack'])) : '',
            'default_deck_id' => isset($_POST['default_deck_id']) ? absint(wp_unslash($_POST['default_deck_id'])) : 0,
            'show_titles' => isset($_POST['show_titles']) ? '1' : '0',
            'show_labels' => isset($_POST['show_labels']) ? '1' : '0',
            'show_meanings' => isset($_POST['show_meanings']) ? '1' : '0',
            'link_cards' => isset($_POST['link_cards']) ? '1' : '0',
        ]);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-spreads',
            'tab' => 'options',
            'msg' => 'options_saved',
        ], admin_url('admin.php')));
        exit;
    }

    public static function handlePackSave(): void {
        self::check('dtarot_spread_pack_save');
        $packId = isset($_POST['pack_id']) ? sanitize_key((string)wp_unslash($_POST['pack_id'])) : '';
        $presetId = isset($_POST['preset_id']) ? sanitize_key((string)wp_unslash($_POST['preset_id'])) : '';
        $meanings = isset($_POST['meanings']) && is_array($_POST['meanings']) ? wp_unslash($_POST['meanings']) : [];

        $packs = SpreadMeaningPacks::all();
        if (!isset($packs[$packId]) || !is_array($packs[$packId])) {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'options','msg'=>'pack_import_error'], admin_url('admin.php')));
            exit;
        }
        $isUserPack = !empty($packs[$packId]['user_created']);
        if (!self::canEditPacks() && !$isUserPack) {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'meanings','msg'=>'pack_import_error'], admin_url('admin.php')));
            exit;
        }
        if (empty($packs[$packId]['allow_edit'])) {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'options','msg'=>'pack_import_error'], admin_url('admin.php')));
            exit;
        }

        $preset = SpreadPresets::get($presetId);
        if (!$preset) {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'options','msg'=>'pack_import_error'], admin_url('admin.php')));
            exit;
        }

        $slots = [];
        $input = isset($meanings[$presetId]) && is_array($meanings[$presetId]) ? $meanings[$presetId] : [];
        foreach ($preset['slots'] as $idx => $slot) {
            $label = isset($slot['label']) && is_string($slot['label']) ? $slot['label'] : '';
            $text = isset($input[$idx]) ? (string)$input[$idx] : '';
            $slots[] = [
                'label' => $label,
                'meaning' => wp_kses_post($text),
            ];
        }

        if (!isset($packs[$packId]['spreads']) || !is_array($packs[$packId]['spreads'])) {
            $packs[$packId]['spreads'] = [];
        }
        $packs[$packId]['spreads'][$presetId] = [
            'slots' => $slots,
        ];

        SpreadMeaningPacks::saveAll($packs);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-spreads',
            'tab' => 'options',
            'msg' => 'pack_saved',
        ], admin_url('admin.php')));
        exit;
    }

    public static function handlePackImport(): void {
        self::check('dtarot_spread_pack_import');

        if (!isset($_FILES['dtarot_spread_pack_zip']) || !is_array($_FILES['dtarot_spread_pack_zip'])) {
            self::redirectImportError('pack_import_error');
        }

        $file = $_FILES['dtarot_spread_pack_zip'];
        $name = isset($file['name']) ? (string)$file['name'] : '';
        $tmp = isset($file['tmp_name']) ? (string)$file['tmp_name'] : '';
        $size = isset($file['size']) ? absint($file['size']) : 0;
        $err = isset($file['error']) ? (int)$file['error'] : UPLOAD_ERR_OK;

        // Guard (10MB)
        if ($size <= 0 || $size > 10 * 1024 * 1024) {
            self::redirectImportError('pack_import_error');
        }

        if ($err !== UPLOAD_ERR_OK) {
            self::redirectImportError('pack_import_error');
        }
        if ($tmp === '' || !is_uploaded_file($tmp)) {
            self::redirectImportError('pack_import_error');
        }

        $ft = wp_check_filetype_and_ext($tmp, $name);
        if (empty($ft['ext']) || strtolower((string)$ft['ext']) !== 'zip') {
            self::redirectImportError('pack_import_error');
        }

        self::importPackZipFromPath($tmp);
    }

    public static function handlePackCreate(): void {
        self::check('dtarot_spread_pack_create');
        $userCreated = !empty($_POST['user_created']);
        if (!self::canEditPacks() && !$userCreated) {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'meanings','msg'=>'pack_import_error'], admin_url('admin.php')));
            exit;
        }
        if ($userCreated && !self::canCreateUserPack()) {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'meanings','msg'=>'pack_limit'], admin_url('admin.php')));
            exit;
        }

        $title = isset($_POST['pack_title']) ? sanitize_text_field((string)wp_unslash($_POST['pack_title'])) : '';
        $allowEdit = $userCreated ? true : !empty($_POST['allow_edit']);
        $template = $userCreated ? true : !empty($_POST['template']);

        if ($title === '') {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'meanings','msg'=>'pack_import_error'], admin_url('admin.php')));
            exit;
        }

        $all = SpreadMeaningPacks::all();
        $id = $userCreated
            ? self::generatePackIdFromTitle($title, $all)
            : (isset($_POST['pack_id']) ? sanitize_key((string)wp_unslash($_POST['pack_id'])) : '');
        if ($id === '') {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'meanings','msg'=>'pack_import_error'], admin_url('admin.php')));
            exit;
        }
        if (isset($all[$id])) {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-spreads','tab'=>'meanings','msg'=>'pack_import_error'], admin_url('admin.php')));
            exit;
        }

        $spreads = [];
        foreach (SpreadPresets::all() as $presetId => $preset) {
            $slots = [];
            foreach ($preset['slots'] as $slot) {
                $label = isset($slot['label']) && is_string($slot['label']) ? $slot['label'] : '';
                $slots[] = [
                    'label' => $label,
                    'meaning' => '',
                ];
            }
            $spreads[$presetId] = ['slots' => $slots];
        }

        $all[$id] = [
            'id' => $id,
            'title' => $title,
            'version' => '1.0.0',
            'allow_edit' => $allowEdit,
            'template' => $template,
            'purchase_token' => '',
            'user_created' => $userCreated ? 1 : 0,
            'system' => isset($_POST['pack_system']) ? Cards::normalizeSystem(sanitize_text_field((string)wp_unslash($_POST['pack_system']))) : Cards::SYSTEM_TAROT,
            'spreads' => $spreads,
            'installed_at' => gmdate('c'),
        ];

        SpreadMeaningPacks::saveAll($all);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-spreads',
            'tab' => 'meanings',
            'msg' => 'pack_saved',
        ], admin_url('admin.php')));
        exit;
    }

    private static function importPackZipFromPath(string $zipPath): void {
        if (!class_exists('ZipArchive')) {
            self::redirectImportError('pack_import_error');
        }

        $workDir = self::createWorkDir();
        if ($workDir === '') {
            self::redirectImportError('pack_import_error');
        }

        $zip = new \ZipArchive();
        if ($zip->open($zipPath) !== true) {
            self::cleanupDir($workDir);
            self::redirectImportError('pack_import_error');
        }

        $extractErrors = self::safeZipExtract($zip, $workDir);
        $zip->close();

        if ($extractErrors) {
            self::cleanupDir($workDir);
            self::redirectImportError('pack_import_error');
        }

        $manifest = self::readManifest($workDir);
        if (!is_array($manifest)) {
            self::cleanupDir($workDir);
            self::redirectImportError('pack_import_error');
        }

        $commonErr = ManifestValidator::validateCommon($manifest, self::FORMAT_VERSION, defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '');
        if ($commonErr !== '') {
            self::cleanupDir($workDir);
            self::redirectImportError('pack_import_error');
        }

        $pack = isset($manifest['spread_pack']) && is_array($manifest['spread_pack']) ? $manifest['spread_pack'] : null;
        if (!$pack) {
            self::cleanupDir($workDir);
            self::redirectImportError('pack_import_error');
        }

        $id = isset($pack['slug']) && is_string($pack['slug']) ? sanitize_key($pack['slug']) : '';
        if ($id === '') {
            $id = isset($pack['id']) && is_string($pack['id']) ? sanitize_key($pack['id']) : '';
        }
        $title = isset($pack['title']) && is_string($pack['title']) ? sanitize_text_field($pack['title']) : '';
        if ($id === '' || $title === '') {
            self::cleanupDir($workDir);
            self::redirectImportError('pack_import_error');
        }

        $allowEdit = !empty($pack['allow_edit']);
        $template = !empty($pack['template']);
        $token = isset($pack['purchase_token']) && is_string($pack['purchase_token']) ? sanitize_text_field($pack['purchase_token']) : '';
        if ($token !== '' && SpreadMeaningPacks::tokenUsed($token)) {
            self::cleanupDir($workDir);
            self::redirectImportError('pack_import_error');
        }

        $spreads = isset($pack['spreads']) && is_array($pack['spreads']) ? $pack['spreads'] : [];
        $cleanSpreads = self::sanitizeSpreads($spreads);

        $packVersion = isset($manifest['pack_version']) && is_string($manifest['pack_version']) ? trim($manifest['pack_version']) : '0.0.0';

        $all = SpreadMeaningPacks::all();
        $all[$id] = [
            'id' => $id,
            'title' => $title,
            'version' => $packVersion,
            'allow_edit' => $allowEdit,
            'template' => $template,
            'purchase_token' => $token,
            'spreads' => $cleanSpreads,
            'installed_at' => gmdate('c'),
        ];

        SpreadMeaningPacks::saveAll($all);
        if ($token !== '') {
            SpreadMeaningPacks::markTokenUsed($token);
        }

        self::cleanupDir($workDir);

        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-spreads',
            'tab' => 'options',
            'msg' => 'pack_imported',
        ], admin_url('admin.php')));
        exit;
    }

    private static function sanitizeSpreads(array $spreads): array {
        $out = [];
        foreach ($spreads as $presetId => $row) {
            if (!is_string($presetId) || !is_array($row)) continue;
            $presetId = sanitize_key($presetId);
            $preset = SpreadPresets::get($presetId);
            if (!$preset) continue;

            $slots = isset($row['slots']) && is_array($row['slots']) ? $row['slots'] : [];
            $cleanSlots = [];
            foreach ($preset['slots'] as $idx => $slotDef) {
                $label = isset($slotDef['label']) && is_string($slotDef['label']) ? $slotDef['label'] : '';
                $slotRow = isset($slots[$idx]) && is_array($slots[$idx]) ? $slots[$idx] : [];
                $slotLabel = isset($slotRow['label']) && is_string($slotRow['label']) ? sanitize_text_field($slotRow['label']) : $label;
                $slotMeaning = isset($slotRow['meaning']) && !is_array($slotRow['meaning']) ? wp_kses_post((string)$slotRow['meaning']) : '';
                $cleanSlots[] = [
                    'label' => $slotLabel,
                    'meaning' => $slotMeaning,
                ];
            }

            $out[$presetId] = ['slots' => $cleanSlots];
        }
        return $out;
    }

    private static function check(string $action): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer($action);
    }

    private static function canEditPacks(): bool {
        $allowed = defined('DTAROT_ALLOW_SPREAD_PACK_EDIT') && (bool)DTAROT_ALLOW_SPREAD_PACK_EDIT;
        return (bool) apply_filters('dtarot_allow_spread_pack_edit', $allowed);
    }

    private static function canCreateUserPack(): bool {
        $isPro = class_exists(FeatureFlags::class) && FeatureFlags::enabled('pro');
        if ($isPro) return true;
        return self::countUserCreatedPacks() < 1;
    }

    private static function countUserCreatedPacks(): int {
        $all = SpreadMeaningPacks::all();
        $count = 0;
        foreach ($all as $pack) {
            if (is_array($pack) && !empty($pack['user_created'])) {
                $count++;
            }
        }
        return $count;
    }

    /**
     * @param array<string,mixed> $all
     */
    private static function generatePackIdFromTitle(string $title, array $all): string {
        $base = sanitize_key($title);
        if ($base === '') $base = 'custom_pack';
        $candidate = $base;
        $n = 2;
        while (isset($all[$candidate])) {
            $candidate = $base . '-' . $n;
            $n++;
        }
        return $candidate;
    }

    private static function redirectImportError(string $msg): void {
        wp_safe_redirect(add_query_arg([
            'page' => 'daily-tarot-spreads',
            'tab' => 'options',
            'msg' => $msg,
        ], admin_url('admin.php')));
        exit;
    }

    private static function createWorkDir(): string {
        $uploadDir = wp_upload_dir();
        $base = is_array($uploadDir) && isset($uploadDir['basedir']) ? (string)$uploadDir['basedir'] : '';
        if ($base === '') return '';

        $dir = rtrim($base, '/\\') . '/dtarot-spread-import-' . wp_generate_password(12, false, false);
        if (!wp_mkdir_p($dir)) return '';
        return $dir;
    }

    private static function readManifest(string $workDir): ?array {
        $manifestPath = $workDir . '/manifest.json';
        $raw = is_readable($manifestPath) ? file_get_contents($manifestPath) : false;
        if (!is_string($raw) || $raw === '') return null;
        $manifest = json_decode($raw, true);
        return is_array($manifest) ? $manifest : null;
    }

    private static function cleanupDir(string $dir): void {
        if ($dir === '' || !is_dir($dir)) return;
        $items = scandir($dir);
        if (!is_array($items)) return;
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') continue;
            $path = $dir . '/' . $item;
            if (is_dir($path)) {
                self::cleanupDir($path);
            } else {
                wp_delete_file($path);
            }
        }
        $fs = self::filesystem();
        if ($fs && method_exists($fs, 'rmdir')) {
            $fs->rmdir($dir, true);
        }
    }

    private static function safeZipExtract(\ZipArchive $zip, string $workDir): array {
        $errors = [];
        $count = $zip->numFiles;
        for ($i = 0; $i < $count; $i++) {
            $name = (string)$zip->getNameIndex($i);
            $name = str_replace('\\', '/', $name);
            if ($name === '' || $name === '/' || str_ends_with($name, '/')) continue;

            $target = $workDir . '/' . ltrim($name, '/');
            $target = str_replace('\\', '/', $target);
            if (str_contains($target, '../')) {
                $errors[] = 'bad_path: ' . $name;
                continue;
            }
            $dir = dirname($target);
            if (!is_dir($dir) && !wp_mkdir_p($dir)) {
                $errors[] = 'mkdir_failed: ' . $dir;
                continue;
            }

            $contents = $zip->getFromIndex($i);
            if ($contents === false) {
                $errors[] = 'read_failed: ' . $name;
                continue;
            }
            $fs = self::filesystem();
            if (!$fs || $fs->put_contents($target, $contents, FS_CHMOD_FILE) === false) {
                $errors[] = 'write_failed: ' . $name;
                continue;
            }
        }
        return $errors;
    }
}
