<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Support\PostTypes;

final class ReadingTypeMeta {

    public static function init(): void {
        add_action('add_meta_boxes_' . PostTypes::READING_TYPE, [__CLASS__, 'add']);
        add_action('save_post_' . PostTypes::READING_TYPE, [__CLASS__, 'save']);

        // Back-compat.
        add_action('add_meta_boxes_' . PostTypes::LEGACY_READING_TYPE, [__CLASS__, 'addLegacy']);
        add_action('save_post_' . PostTypes::LEGACY_READING_TYPE, [__CLASS__, 'save']);
    }

    public static function add(): void {
        add_meta_box('dtarot_reading_type_meta', __('Reading Settings','daily-tarot'), [__CLASS__, 'render'], PostTypes::READING_TYPE, 'normal', 'high');
    }

    public static function addLegacy(): void {
        add_meta_box('dtarot_reading_type_meta', __('Reading Settings','daily-tarot'), [__CLASS__, 'render'], PostTypes::LEGACY_READING_TYPE, 'normal', 'high');
    }

    public static function render($post): void {
        wp_nonce_field('dtarot_reading_type_save', 'dtarot_reading_type_nonce');
        $duration = (int)get_post_meta($post->ID, '_dtarot_reading_duration', true);
        $price = (string)get_post_meta($post->ID, '_dtarot_reading_price', true);
        $icon = (string)get_post_meta($post->ID, '_dtarot_reading_icon', true);
        ?>
        <p>
            <label for="dtarot-reading-duration"><strong><?php esc_html_e('Duration (minutes)','daily-tarot'); ?></strong></label><br>
            <input id="dtarot-reading-duration" type="number" min="5" step="5" name="dtarot_reading_duration" value="<?php echo esc_attr((string)$duration); ?>" class="small-text" />
        </p>

        <p>
            <label for="dtarot-reading-price"><strong><?php esc_html_e('Price (optional)','daily-tarot'); ?></strong></label><br>
            <input id="dtarot-reading-price" type="text" name="dtarot_reading_price" value="<?php echo esc_attr($price); ?>" class="regular-text" placeholder="€50" />
        </p>

        <p>
            <label for="dtarot-reading-icon"><strong><?php esc_html_e('Icon or emoji (optional)','daily-tarot'); ?></strong></label><br>
            <input id="dtarot-reading-icon" type="text" name="dtarot_reading_icon" value="<?php echo esc_attr($icon); ?>" class="regular-text" placeholder="🔮" />
        </p>
        <?php
    }

    public static function save(int $post_id): void {
        if (!current_user_can('edit_post', $post_id)) return;
        if (!isset($_POST['dtarot_reading_type_nonce']) || !wp_verify_nonce(sanitize_text_field((string)wp_unslash($_POST['dtarot_reading_type_nonce'])), 'dtarot_reading_type_save')) return;

        $duration = isset($_POST['dtarot_reading_duration']) ? (int)wp_unslash($_POST['dtarot_reading_duration']) : 0;
        if ($duration <= 0) $duration = 30;
        update_post_meta($post_id, '_dtarot_reading_duration', $duration);

        $price = isset($_POST['dtarot_reading_price']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_reading_price'])) : '';
        if ($price === '') {
            delete_post_meta($post_id, '_dtarot_reading_price');
        } else {
            update_post_meta($post_id, '_dtarot_reading_price', $price);
        }

        $icon = isset($_POST['dtarot_reading_icon']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_reading_icon'])) : '';
        if ($icon === '') {
            delete_post_meta($post_id, '_dtarot_reading_icon');
        } else {
            update_post_meta($post_id, '_dtarot_reading_icon', $icon);
        }
    }
}
