<?php
declare(strict_types=1);

namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.NonceVerification.Recommended

final class PluginsPage {

    private const FEEDBACK_OPTION = 'dtarot_uninstall_feedback_v1';

    public static function init(): void {
        if (!is_admin()) return;

        add_filter('plugin_row_meta', [__CLASS__, 'filterPluginRowMeta'], 10, 2);

        $base = self::pluginBasename();
        if ($base !== '') {
            add_filter('plugin_action_links_' . $base, [__CLASS__, 'filterActionLinks']);
            add_action('after_plugin_row_' . $base, [__CLASS__, 'renderFaqRow'], 10, 3);
        }

        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue']);
    }

    private static function pluginBasename(): string {
        if (!defined('DTAROT_PATH')) return '';
        if (!function_exists('plugin_basename')) return '';
        $file = DTAROT_PATH . 'daily-tarot.php';
        return is_string($file) ? (string)plugin_basename($file) : '';
    }

    /** @param array<int,string> $links */
    public static function filterPluginRowMeta(array $links, string $file): array {
        $base = self::pluginBasename();
        if ($base === '' || $file !== $base) return $links;

        $faqAnchor = '#dtarot-plugin-faq';
        $pluginsUrl = admin_url('plugins.php') . $faqAnchor;

        $links[] = '<a href="' . esc_url($pluginsUrl) . '">' . esc_html__('FAQ','daily-tarot') . '</a>';

        return $links;
    }

    /** @param array<int,string> $links */
    public static function filterActionLinks(array $links): array {
        $settings = admin_url('admin.php?page=daily-tarot-settings');

        $links[] = '<a href="' . esc_url($settings) . '">' . esc_html__('Settings','daily-tarot') . '</a>';

        return $links;
    }

    public static function enqueue(string $hook): void {
        if ($hook !== 'plugins.php') return;
        if (!current_user_can('activate_plugins')) return;

        $jsFile = DTAROT_PATH . 'assets/plugins-page.js';
        $cssFile = DTAROT_PATH . 'assets/plugins-page.css';
        $jsVer = is_readable($jsFile) ? (string)@filemtime($jsFile) : (defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '1');
        $cssVer = is_readable($cssFile) ? (string)@filemtime($cssFile) : (defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '1');

        wp_enqueue_style('dtarot-plugins-page', DTAROT_URL . 'assets/plugins-page.css', [], $cssVer);
        wp_enqueue_script('dtarot-plugins-page', DTAROT_URL . 'assets/plugins-page.js', ['jquery'], $jsVer, true);

        wp_localize_script('dtarot-plugins-page', 'DTAROT_PLUGINS', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('dtarot_admin'),
            'plugin' => self::pluginBasename(),
        ]);
    }

    /**
     * @param array<string,mixed> $plugin_data
     */
    public static function renderFaqRow(string $plugin_file, array $plugin_data, string $status): void {
        $base = self::pluginBasename();
        if ($base === '' || $plugin_file !== $base) return;

        // Column count varies across WP versions and screens.
        $colspan = 3;
        global $wp_list_table;
        if (isset($wp_list_table) && is_object($wp_list_table) && method_exists($wp_list_table, 'get_column_count')) {
            $n = (int)$wp_list_table->get_column_count();
            if ($n > 0) $colspan = $n;
        }

        $startHere = admin_url('admin.php?page=daily-tarot-start-here');
        $contentCards = admin_url('admin.php?page=daily-tarot-content&tab=cards');
        $contentMeanings = admin_url('admin.php?page=daily-tarot-content&tab=meanings');
        $calendar = admin_url('admin.php?page=daily-tarot-calendar');

        echo '<tr id="dtarot-plugin-faq" class="dtarot-plugin-faq-row">';
        echo '<td colspan="' . esc_attr((string)$colspan) . '">';

        echo '<div class="dtarot-plugin-faq">';
        echo '<details>';
        echo '<summary><strong>' . esc_html__('Daily Tarot FAQ (quick help)','daily-tarot') . '</strong> <span class="description">' . esc_html__('Common “stuck” moments and fixes.','daily-tarot') . '</span></summary>';

        echo '<div class="dtarot-plugin-faq-body">';
        echo '<p style="margin:8px 0 12px 0;">' . esc_html__('Short version: Decks = images. Meaning Packs = text. Calendar = publish. Systems must match.','daily-tarot') . '</p>';

        echo '<details class="dtarot-plugin-faq-item"><summary>' . esc_html__('My card image is missing','daily-tarot') . '</summary>';
        echo '<p>' . esc_html__('Pick a deck for the same system and upload the card image.','daily-tarot') . ' <a href="' . esc_url($contentCards) . '">' . esc_html__('Content → Cards','daily-tarot') . '</a></p>';
        echo '</details>';

        echo '<details class="dtarot-plugin-faq-item"><summary>' . esc_html__('Meanings are blank / wrong','daily-tarot') . '</summary>';
        echo '<p>' . esc_html__('Choose a meaning pack for the same system and fill meanings for the card IDs.','daily-tarot') . ' <a href="' . esc_url($contentMeanings) . '">' . esc_html__('Content → Meaning Packs','daily-tarot') . '</a></p>';
        echo '</details>';

        echo '<details class="dtarot-plugin-faq-item"><summary>' . esc_html__('I don’t know what to do first','daily-tarot') . '</summary>';
        echo '<p>' . esc_html__('Create a Deck (images), then a Meaning Pack (text), then use Calendar to publish. Check the Content tab for deck/pack setup.','daily-tarot') . ' <a href="' . esc_url($contentCards) . '">' . esc_html__('Content → Cards','daily-tarot') . '</a></p>';
        echo '</details>';

        echo '<details class="dtarot-plugin-faq-item"><summary>' . esc_html__('How do I publish the daily card?','daily-tarot') . '</summary>';
        echo '<p>' . esc_html__('Go to Calendar, pick a date, select deck + meaning pack + card, then publish.','daily-tarot') . ' <a href="' . esc_url($calendar) . '">' . esc_html__('Open Calendar','daily-tarot') . '</a></p>';
        echo '</details>';

        echo '<details class="dtarot-plugin-faq-item"><summary>' . esc_html__('What’s the difference between Kipper and Gypsy?','daily-tarot') . '</summary>';
        echo '<p>' . esc_html__('They are separate systems with different fixed card IDs. Make sure your deck and meaning pack are set to the system you want.','daily-tarot') . '</p>';
        echo '</details>';

        echo '<p style="margin:12px 0 0 0; display:flex; gap:8px; flex-wrap:wrap;">';
        echo '<a class="button" href="' . esc_url($contentCards) . '">' . esc_html__('Upload images','daily-tarot') . '</a>';
        echo '<a class="button" href="' . esc_url($contentMeanings) . '">' . esc_html__('Edit meanings','daily-tarot') . '</a>';
        echo '<a class="button" href="' . esc_url($calendar) . '">' . esc_html__('Calendar','daily-tarot') . '</a>';
        echo '</p>';

        echo '</div>';
        echo '</details>';
        echo '</div>';

        echo '</td>';
        echo '</tr>';
    }
}
