<?php
declare(strict_types=1);

namespace DailyTarot\Admin\Pages;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped

use DailyTarot\Calendar\DayEntryTable;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\DefaultDecks;
use DailyTarot\Support\DefaultMeaningPacks;
use DailyTarot\Support\PostTypes;

final class StartHere {

    public static function render(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));

        $systems = Cards::systems();
        $deckCount = self::countPosts(PostTypes::deckTypes());
        $packCount = self::countPosts(PostTypes::meaningPackTypes());
        $dayCounts = self::dayEntryCounts();

        $contentUrl = admin_url('admin.php?page=daily-tarot-content');
        $cardsUrl = admin_url('admin.php?page=daily-tarot-content&tab=cards');
        $decksUrl = admin_url('admin.php?page=daily-tarot-content&tab=decks');
        $packsUrl = admin_url('admin.php?page=daily-tarot-content&tab=meanings');
        $calendarUrl = admin_url('admin.php?page=daily-tarot-calendar');

        $newDeckUrl = admin_url('post-new.php?post_type=' . PostTypes::DECK);
        $newPackUrl = admin_url('post-new.php?post_type=' . PostTypes::MEANING_PACK);

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__('Start Here','daily-tarot') . '</h1>';

        echo '<p class="description">' . esc_html__('If you feel lost, this page connects the setup steps: Decks (images) → Meaning Packs (text) → Calendar (publish).','daily-tarot') . '</p>';

        echo '<div class="dtarot-start-grid" style="display:grid;grid-template-columns:1.1fr 0.9fr;gap:16px;align-items:start;">';

        // Left: checklist + wizard
        echo '<div>';
        self::renderChecklist($deckCount, $packCount, $dayCounts);
        self::renderWizard($newDeckUrl, $cardsUrl, $newPackUrl, $calendarUrl);
        echo '</div>';

        // Right: quick links + glossary
        echo '<div>';
        echo '<div class="postbox" style="padding:14px;">';
        echo '<h2 style="margin:0 0 10px 0;">' . esc_html__('Quick Links','daily-tarot') . '</h2>';
        echo '<p><a class="button button-primary" href="' . esc_url($calendarUrl) . '">' . esc_html__('Open Calendar','daily-tarot') . '</a></p>';
        echo '<p><a class="button" href="' . esc_url($contentUrl) . '">' . esc_html__('Open Content','daily-tarot') . '</a></p>';
        echo '<p><a class="button" href="' . esc_url($decksUrl) . '">' . esc_html__('Manage Decks','daily-tarot') . '</a> ';
        echo '<a class="button" href="' . esc_url($cardsUrl) . '">' . esc_html__('Upload Card Images','daily-tarot') . '</a></p>';
        echo '<p><a class="button" href="' . esc_url($packsUrl) . '">' . esc_html__('Manage Meaning Packs','daily-tarot') . '</a></p>';
        echo '</div>';

        echo '<div class="postbox" style="padding:14px;margin-top:16px;">';
        echo '<h2 style="margin:0 0 10px 0;">' . esc_html__('Glossary','daily-tarot') . '</h2>';
        echo '<ul style="margin:0 0 0 18px;">';
        echo '<li><strong>' . esc_html__('System','daily-tarot') . ':</strong> ' . esc_html__('The card registry + fixed IDs (Tarot / Lenormand / Kipper / Gypsy).','daily-tarot') . '</li>';
        echo '<li><strong>' . esc_html__('Deck','daily-tarot') . ':</strong> ' . esc_html__('Images for each fixed card ID (uploaded in Content → Cards).','daily-tarot') . '</li>';
        echo '<li><strong>' . esc_html__('Meaning Pack','daily-tarot') . ':</strong> ' . esc_html__('Text/interpretations for each fixed card ID.','daily-tarot') . '</li>';
        echo '<li><strong>' . esc_html__('Day Entry','daily-tarot') . ':</strong> ' . esc_html__('A published daily page that uses a deck + a meaning pack + a chosen card.','daily-tarot') . '</li>';
        echo '</ul>';
        echo '</div>';

        echo '<div class="postbox" style="padding:14px;margin-top:16px;">';
        echo '<h2 style="margin:0 0 10px 0;">' . esc_html__('Systems at a glance','daily-tarot') . '</h2>';
        echo '<p class="description" style="margin:0 0 8px 0;">' . esc_html__('Defaults are per system. If something is missing, it is usually because the selected deck/pack is for a different system.','daily-tarot') . '</p>';
        echo '<table class="widefat striped" style="margin:0;">';
        echo '<thead><tr><th>' . esc_html__('System','daily-tarot') . '</th><th>' . esc_html__('Default deck','daily-tarot') . '</th><th>' . esc_html__('Default meaning pack','daily-tarot') . '</th></tr></thead><tbody>';
        foreach ($systems as $sys => $label) {
            $sys = is_string($sys) ? $sys : '';
            if ($sys === '') continue;
            $deckId = DefaultDecks::get($sys);
            $packId = DefaultMeaningPacks::get($sys);
            echo '<tr>';
            echo '<td>' . esc_html((string)$label) . '</td>';
            echo '<td>' . ($deckId > 0 ? esc_html(get_the_title($deckId) ?: (string)$deckId) : '<span class="description">' . esc_html__('Not set','daily-tarot') . '</span>') . '</td>';
            echo '<td>' . ($packId > 0 ? esc_html(get_the_title($packId) ?: (string)$packId) : '<span class="description">' . esc_html__('Not set','daily-tarot') . '</span>') . '</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        echo '</div>';

        echo '</div>'; // right

        echo '</div>'; // grid
        echo '</div>'; // wrap
    }

    /** @param array{total:int,published:int} $dayCounts */
    private static function renderChecklist(int $deckCount, int $packCount, array $dayCounts): void {
        $steps = [];

        if ($deckCount <= 0) {
            $steps[] = [
                'title' => __('Create your first deck (images)','daily-tarot'),
                'done' => false,
                'desc' => __('A deck stores one image per fixed card ID (for a system).','daily-tarot'),
                'url' => admin_url('post-new.php?post_type=' . PostTypes::DECK),
                'cta' => __('Create deck','daily-tarot'),
            ];
        } else {
            $steps[] = [
                'title' => __('You have at least one deck','daily-tarot'),
                'done' => true,
                'desc' => __('Next: upload card images in Content → Cards.','daily-tarot'),
                'url' => admin_url('admin.php?page=daily-tarot-content&tab=cards'),
                'cta' => __('Upload card images','daily-tarot'),
            ];
        }

        $steps[] = [
            'title' => __('Create or choose a meaning pack (text)','daily-tarot'),
            'done' => ($packCount > 0),
            'desc' => __('Meaning packs store text/interpretations for each fixed card ID.','daily-tarot'),
            'url' => ($packCount > 0) ? admin_url('admin.php?page=daily-tarot-content&tab=meanings') : admin_url('post-new.php?post_type=' . PostTypes::MEANING_PACK),
            'cta' => ($packCount > 0) ? __('Manage meaning packs','daily-tarot') : __('Create meaning pack','daily-tarot'),
        ];

        $steps[] = [
            'title' => __('Publish your first day (Calendar)','daily-tarot'),
            'done' => ($dayCounts['published'] > 0),
            'desc' => __('A published day entry picks a card and uses a deck + meaning pack to render your page.','daily-tarot'),
            'url' => admin_url('admin.php?page=daily-tarot-calendar'),
            'cta' => __('Open calendar','daily-tarot'),
        ];

        echo '<div class="postbox" style="padding:14px;">';
        echo '<h2 style="margin:0 0 10px 0;">' . esc_html__('Setup Checklist','daily-tarot') . '</h2>';
        echo '<p class="description" style="margin:0 0 10px 0;">' . esc_html__('Complete these in order. Each item links to the right screen.','daily-tarot') . '</p>';

        foreach ($steps as $s) {
            $done = !empty($s['done']);
            $title = (string)($s['title'] ?? '');
            $desc = (string)($s['desc'] ?? '');
            $url = (string)($s['url'] ?? '');
            $cta = (string)($s['cta'] ?? '');

            echo '<div style="padding:10px 0;border-top:1px solid #eee;">';
            echo '<div style="display:flex;gap:10px;align-items:flex-start;">';
            echo '<div style="min-width:20px;">' . ($done ? '<span style="color:#1e8e3e;font-weight:700;">✓</span>' : '<span style="color:#b32d2e;font-weight:700;">•</span>') . '</div>';
            echo '<div style="flex:1;">';
            echo '<div style="font-weight:600;">' . esc_html($title) . '</div>';
            if ($desc !== '') {
                echo '<div class="description" style="margin-top:2px;">' . esc_html($desc) . '</div>';
            }
            if ($url !== '' && $cta !== '') {
                echo '<div style="margin-top:8px;"><a class="button" href="' . esc_url($url) . '">' . esc_html($cta) . '</a></div>';
            }
            echo '</div>';
            echo '</div>';
            echo '</div>';
        }

        echo '<div style="padding-top:10px;border-top:1px solid #eee;">';
        echo '<span class="description">' . esc_html(sprintf(__('Decks: %d · Meaning Packs: %d · Day entries: %d total (%d published)','daily-tarot'), $deckCount, $packCount, $dayCounts['total'], $dayCounts['published'])) . '</span>';
        echo '</div>';

        echo '</div>';
    }

    private static function renderWizard(string $newDeckUrl, string $cardsUrl, string $newPackUrl, string $calendarUrl): void {
        echo '<div class="postbox" style="padding:14px;margin-top:16px;">';
        echo '<h2 style="margin:0 0 10px 0;">' . esc_html__('Guided Setup (5 minutes)','daily-tarot') . '</h2>';
        echo '<ol style="margin:0 0 0 18px;">';
        echo '<li>' . esc_html__('Create a deck and choose its system.','daily-tarot') . ' <a href="' . esc_url($newDeckUrl) . '">' . esc_html__('Create deck','daily-tarot') . '</a></li>';
        echo '<li>' . esc_html__('Upload card images for that deck.','daily-tarot') . ' <a href="' . esc_url($cardsUrl) . '">' . esc_html__('Upload images','daily-tarot') . '</a></li>';
        echo '<li>' . esc_html__('Create a meaning pack for the same system.','daily-tarot') . ' <a href="' . esc_url($newPackUrl) . '">' . esc_html__('Create meaning pack','daily-tarot') . '</a></li>';
        echo '<li>' . esc_html__('Set defaults (so pages/shortcodes “just work”).','daily-tarot') . ' <a href="' . esc_url(admin_url('admin.php?page=daily-tarot-content&tab=status')) . '">' . esc_html__('Set defaults','daily-tarot') . '</a></li>';
        echo '<li>' . esc_html__('Open Calendar and publish your first day.','daily-tarot') . ' <a href="' . esc_url($calendarUrl) . '">' . esc_html__('Open calendar','daily-tarot') . '</a></li>';
        echo '</ol>';
        echo '<p class="description" style="margin-top:10px;">' . esc_html__('If something looks missing on the frontend, it’s almost always a system mismatch or missing defaults. Use the Troubleshoot panel on card pages (admin only).','daily-tarot') . '</p>';
        echo '</div>';
    }

    /** @param array<int,string> $postTypes */
    private static function countPosts(array $postTypes): int {
        $q = new \WP_Query([
            'post_type' => $postTypes,
            'post_status' => ['publish','draft','pending','private'],
            'posts_per_page' => 1,
            'fields' => 'ids',
            'no_found_rows' => false,
        ]);
        return is_numeric($q->found_posts) ? (int)$q->found_posts : 0;
    }

    /** @return array{total:int,published:int} */
    private static function dayEntryCounts(): array {
        $total = class_exists(DayEntryTable::class) ? DayEntryTable::countRows() : 0;
        $published = 0;

        if (class_exists(DayEntryTable::class) && DayEntryTable::exists()) {
            global $wpdb;
            $table = DayEntryTable::name();
            if (isset($wpdb) && is_object($wpdb) && $table !== '') {
                $n = $wpdb->get_var($wpdb->prepare('SELECT COUNT(*) FROM ' . $table . ' WHERE status=%s', 'publish'));
                $published = is_numeric($n) ? (int)$n : 0;
            }
        }

        return [
            'total' => max(0, (int)$total),
            'published' => max(0, (int)$published),
        ];
    }
}
