<?php
declare(strict_types=1);


namespace DailyTarot\Admin\Pages;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.DateTime.RestrictedFunctions.date_date

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Registry\Cards;
use DailyTarot\Support\SpreadPresets;
use DailyTarot\Support\SpreadMeaningPacks;
use DailyTarot\Support\SpreadScanner;
use DailyTarot\Support\SpreadMappings;
use DailyTarot\Support\SpreadSettings;
use DailyTarot\Support\FeatureFlags;
use DailyTarot\Support\PostTypes;

final class Spreads {

    public static function render(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));

        $tab = isset($_GET['tab']) ? sanitize_key((string)wp_unslash($_GET['tab'])) : 'dashboard';
        $tabs = [
            'dashboard' => __('Dashboard','daily-tarot'),
            'meanings' => __('Meanings','daily-tarot'),
            'options' => __('Options','daily-tarot'),
        ];
        if (!isset($tabs[$tab])) $tab = 'dashboard';

        ?>
        <div class="wrap dtarot-wrap dtarot-spreads-page">
            <h1><?php esc_html_e('Spreads','daily-tarot'); ?></h1>

            <h2 class="nav-tab-wrapper">
                <?php foreach ($tabs as $k => $label): ?>
                    <a class="nav-tab <?php echo $tab===$k?'nav-tab-active':''; ?>"
                       href="<?php echo esc_url(add_query_arg(['page'=>'daily-tarot-spreads','tab'=>$k], admin_url('admin.php'))); ?>">
                        <?php echo esc_html($label); ?>
                    </a>
                <?php endforeach; ?>
            </h2>

            <?php
            if ($tab === 'options') self::renderOptions();
            elseif ($tab === 'meanings') self::renderMeanings();
            else self::renderDashboard();
            ?>
        </div>
        <?php
    }

    private static function renderDashboard(): void {
        $msg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
        if ($msg === 'scan_ok') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Spread scan complete.','daily-tarot') . '</p></div>';
        } elseif ($msg === 'mapping_saved') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Spread settings saved.','daily-tarot') . '</p></div>';
        }

        $detected = SpreadScanner::getAll();
        $lastScan = SpreadScanner::lastScanTs();
        $lastScanText = $lastScan > 0 ? (function_exists('wp_date') ? (string)wp_date('Y-m-d H:i', $lastScan) : (string)date('Y-m-d H:i', $lastScan)) : __('Never','daily-tarot');

        $presets = SpreadPresets::all();
        $packs = SpreadMeaningPacks::all();
        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
        ]);

        ?>
        <div class="dtarot-card">
            <p class="description"><?php esc_html_e('Detected spread shortcodes on your site. Click a card to choose the spread type, meaning pack, and optional deck.','daily-tarot'); ?></p>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('dtarot_spreads_scan'); ?>
                <input type="hidden" name="action" value="dtarot_spreads_scan" />
                <button type="submit" class="button"><?php esc_html_e('Scan now','daily-tarot'); ?></button>
                <span class="description" style="margin-left:8px;"><?php echo esc_html(sprintf(__('Last scan: %s','daily-tarot'), $lastScanText)); ?></span>
            </form>
        </div>

        <div class="dtarot-spread-grid">
            <?php
            $foundAny = false;
            foreach ($detected as $postId => $items) {
                if (!is_numeric($postId)) continue;
                $postId = (int)$postId;
                if ($postId <= 0) continue;
                if (!is_array($items) || !$items) continue;
                $foundAny = true;
                $post = get_post($postId);
                $title = $post ? (string)($post->post_title ?: __('(no title)','daily-tarot')) : __('(missing post)','daily-tarot');
                $editUrl = $post ? get_edit_post_link($postId, '') : '';

                foreach ($items as $item) {
                    if (!is_array($item)) continue;
                    $index = isset($item['index']) ? (int)$item['index'] : 0;
                    $mapping = SpreadMappings::forPost($postId);
                    $saved = isset($mapping[$index]) && is_array($mapping[$index]) ? $mapping[$index] : [];
                    $preset = isset($saved['preset']) ? (string)$saved['preset'] : '';
                    $pack = isset($saved['pack']) ? (string)$saved['pack'] : '';
                    $deckId = isset($saved['deck_id']) ? (int)$saved['deck_id'] : 0;

                    $presetLabel = ($preset !== '' && isset($presets[$preset])) ? (string)$presets[$preset]['label'] : __('Not set','daily-tarot');
                    $packLabel = ($pack !== '' && isset($packs[$pack])) ? (string)($packs[$pack]['title'] ?? $pack) : __('Not set','daily-tarot');
                    $deckLabel = $deckId > 0 ? (string)(get_the_title($deckId) ?: __('(no title)','daily-tarot')) : __('Default deck','daily-tarot');

                    $data = [
                        'post' => $postId,
                        'index' => $index,
                        'preset' => $preset,
                        'pack' => $pack,
                        'deck' => $deckId,
                    ];
                    ?>
                    <div class="dtarot-card dtarot-spread-card-item">
                        <div class="dtarot-spread-card-title"><?php echo esc_html($title); ?></div>
                        <div class="dtarot-spread-card-meta"><?php echo esc_html(sprintf(__('Shortcode #%d','daily-tarot'), $index + 1)); ?></div>
                        <div class="dtarot-spread-card-meta"><?php echo esc_html(sprintf(__('Preset: %s','daily-tarot'), $presetLabel)); ?></div>
                        <div class="dtarot-spread-card-meta"><?php echo esc_html(sprintf(__('Meaning pack: %s','daily-tarot'), $packLabel)); ?></div>
                        <div class="dtarot-spread-card-meta"><?php echo esc_html(sprintf(__('Deck: %s','daily-tarot'), $deckLabel)); ?></div>
                        <?php if ($editUrl !== ''): ?>
                            <div class="dtarot-spread-card-link"><a href="<?php echo esc_url($editUrl); ?>"><?php esc_html_e('Edit post','daily-tarot'); ?></a></div>
                        <?php endif; ?>
                        <button
                            type="button"
                            class="button button-primary dtarot-spread-config-btn"
                            data-dtarot-spread='<?php echo esc_attr(wp_json_encode($data)); ?>'
                        ><?php esc_html_e('Configure','daily-tarot'); ?></button>
                    </div>
                    <?php
                }
            }
            if (!$foundAny): ?>
                <div class="dtarot-card"><?php esc_html_e('No spread shortcodes detected yet.','daily-tarot'); ?></div>
            <?php endif; ?>
        </div>

        <div class="dtarot-modal" id="dtarot-spread-modal" aria-hidden="true">
            <div class="dtarot-modal-backdrop" data-dtarot-modal-close></div>
            <div class="dtarot-modal-panel" role="dialog" aria-modal="true" aria-labelledby="dtarot-spread-modal-title">
                <div class="dtarot-modal-header">
                    <h2 id="dtarot-spread-modal-title"><?php esc_html_e('Configure Spread','daily-tarot'); ?></h2>
                    <button type="button" class="dtarot-modal-close" data-dtarot-modal-close aria-label="<?php esc_attr_e('Close','daily-tarot'); ?>">x</button>
                </div>

                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" id="dtarot-spread-config-form">
                    <?php wp_nonce_field('dtarot_spread_mapping_save'); ?>
                    <input type="hidden" name="action" value="dtarot_spread_mapping_save" />
                    <input type="hidden" name="post_id" id="dtarot_spread_post_id" value="0" />
                    <input type="hidden" name="index" id="dtarot_spread_index" value="0" />

                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><label for="dtarot_spread_preset"><?php esc_html_e('Spread preset','daily-tarot'); ?></label></th>
                            <td>
                                <select name="preset" id="dtarot_spread_preset">
                                    <option value=""><?php esc_html_e('Use default','daily-tarot'); ?></option>
                                    <?php foreach ($presets as $id => $preset): ?>
                                        <option value="<?php echo esc_attr($id); ?>"><?php echo esc_html($preset['label']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="dtarot_spread_pack"><?php esc_html_e('Meaning pack','daily-tarot'); ?></label></th>
                            <td>
                                <select name="pack" id="dtarot_spread_pack">
                                    <option value=""><?php esc_html_e('Use default','daily-tarot'); ?></option>
                                    <?php foreach ($packs as $id => $pack): ?>
                                        <option value="<?php echo esc_attr($id); ?>"><?php echo esc_html((string)($pack['title'] ?? $id)); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="dtarot_spread_deck"><?php esc_html_e('Deck (optional)','daily-tarot'); ?></label></th>
                            <td>
                                <select name="deck_id" id="dtarot_spread_deck">
                                    <option value="0"><?php esc_html_e('Use default','daily-tarot'); ?></option>
                                    <?php foreach ($decks as $deck): ?>
                                        <option value="<?php echo (int)$deck->ID; ?>"><?php echo esc_html((string)($deck->post_title ?: __('(no title)','daily-tarot'))); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                    </table>

                    <div class="dtarot-modal-preview">
                        <div class="dtarot-modal-preview-title"><?php esc_html_e('Preview','daily-tarot'); ?></div>
                        <div class="dtarot-modal-preview-body" id="dtarot-spread-preview">
                            <?php esc_html_e('Select options to preview this spread.','daily-tarot'); ?>
                        </div>
                    </div>

                    <div class="dtarot-modal-actions">
                        <button type="button" class="button" id="dtarot-spread-preview-btn"><?php esc_html_e('Preview','daily-tarot'); ?></button>
                        <button type="submit" class="button button-primary"><?php esc_html_e('Save','daily-tarot'); ?></button>
                    </div>
                </form>
            </div>
        </div>
        <?php
    }

    private static function renderOptions(): void {
        $msg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
        if ($msg === 'options_saved') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Spread options saved.','daily-tarot') . '</p></div>';
        } elseif ($msg === 'pack_saved') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Spread meaning pack updated.','daily-tarot') . '</p></div>';
        } elseif ($msg === 'pack_imported') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Spread meaning pack imported.','daily-tarot') . '</p></div>';
        } elseif ($msg === 'pack_import_error') {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Spread meaning pack import failed.','daily-tarot') . '</p></div>';
        }

        $presets = SpreadPresets::all();
        $packs = SpreadMeaningPacks::all();
        $settings = SpreadSettings::get();
        $editable = array_filter($packs, static fn($p) => is_array($p) && !empty($p['allow_edit']));

        ?>
        <div class="dtarot-card">
            <h2><?php esc_html_e('Defaults','daily-tarot'); ?></h2>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('dtarot_spread_options_save'); ?>
                <input type="hidden" name="action" value="dtarot_spread_options_save" />

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="dtarot_spread_default_preset"><?php esc_html_e('Default preset','daily-tarot'); ?></label></th>
                        <td>
                            <select id="dtarot_spread_default_preset" name="default_preset">
                                <?php foreach ($presets as $id => $preset): ?>
                                    <option value="<?php echo esc_attr($id); ?>" <?php selected((string)$settings['default_preset'], $id); ?>><?php echo esc_html($preset['label']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="dtarot_spread_default_pack"><?php esc_html_e('Default meaning pack','daily-tarot'); ?></label></th>
                        <td>
                            <select id="dtarot_spread_default_pack" name="default_pack">
                                <?php foreach ($packs as $id => $pack): ?>
                                    <option value="<?php echo esc_attr($id); ?>" <?php selected((string)$settings['default_pack'], $id); ?>><?php echo esc_html((string)($pack['title'] ?? $id)); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="dtarot_spread_default_deck"><?php esc_html_e('Default deck (optional)','daily-tarot'); ?></label></th>
                        <td>
                            <?php
                            $decks = get_posts([
                                'post_type' => PostTypes::deckTypes(),
                                'numberposts' => -1,
                                'post_status' => ['publish','draft','pending','private'],
                            ]);
                            ?>
                            <select id="dtarot_spread_default_deck" name="default_deck_id">
                                <option value="0"><?php esc_html_e('Use system default','daily-tarot'); ?></option>
                                <?php foreach ($decks as $deck): ?>
                                    <option value="<?php echo (int)$deck->ID; ?>" <?php selected((int)$settings['default_deck_id'], (int)$deck->ID); ?>><?php echo esc_html((string)($deck->post_title ?: __('(no title)','daily-tarot'))); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Default display','daily-tarot'); ?></th>
                        <td>
                            <label><input type="checkbox" name="show_titles" value="1" <?php checked((string)$settings['show_titles'], '1'); ?> /> <?php esc_html_e('Show card titles','daily-tarot'); ?></label><br />
                            <label><input type="checkbox" name="show_labels" value="1" <?php checked((string)$settings['show_labels'], '1'); ?> /> <?php esc_html_e('Show position labels','daily-tarot'); ?></label><br />
                            <label><input type="checkbox" name="show_meanings" value="1" <?php checked((string)$settings['show_meanings'], '1'); ?> /> <?php esc_html_e('Show position meanings','daily-tarot'); ?></label><br />
                            <label><input type="checkbox" name="link_cards" value="1" <?php checked((string)$settings['link_cards'], '1'); ?> /> <?php esc_html_e('Link cards to detail pages','daily-tarot'); ?></label>
                        </td>
                    </tr>
                </table>

                <p><button type="submit" class="button button-primary"><?php esc_html_e('Save options','daily-tarot'); ?></button></p>
            </form>
        </div>

        <div class="dtarot-card">
            <h2><?php esc_html_e('Import meaning pack','daily-tarot'); ?></h2>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
                <?php wp_nonce_field('dtarot_spread_pack_import'); ?>
                <input type="hidden" name="action" value="dtarot_spread_pack_import" />
                <input type="file" name="dtarot_spread_pack_zip" accept=".zip" required />
                <button type="submit" class="button"><?php esc_html_e('Import pack','daily-tarot'); ?></button>
            </form>
            <p class="description"><?php esc_html_e('Packs include position labels and meanings for each spread preset.','daily-tarot'); ?></p>
        </div>
        <?php
    }

    private static function renderMeanings(): void {
        $packs = SpreadMeaningPacks::all();
        $canEdit = (bool) apply_filters(
            'dtarot_allow_spread_pack_edit',
            defined('DTAROT_ALLOW_SPREAD_PACK_EDIT') && (bool)DTAROT_ALLOW_SPREAD_PACK_EDIT
        );
        $presets = SpreadPresets::all();
        $isPro = class_exists(FeatureFlags::class) && FeatureFlags::enabled('pro');
        $userCreated = array_filter($packs, static fn($p) => is_array($p) && !empty($p['user_created']));
        $editable = array_filter($packs, static fn($p) => is_array($p) && !empty($p['allow_edit']));
        $limitReached = !$isPro && count($userCreated) >= 1;
        $msg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
        if ($msg === 'pack_saved') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Spread meaning pack created.','daily-tarot') . '</p></div>';
        } elseif ($msg === 'pack_limit') {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Free users can create one spread meaning pack. Activate Pro to add more.','daily-tarot') . '</p></div>';
        }
        ?>
        <div class="dtarot-card">
            <h2><?php esc_html_e('Add new meaning pack','daily-tarot'); ?></h2>
            <p class="description"><?php esc_html_e('Create a custom spread meaning pack you can edit in the template editor.','daily-tarot'); ?></p>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('dtarot_spread_pack_create'); ?>
                <input type="hidden" name="action" value="dtarot_spread_pack_create" />
                <input type="hidden" name="user_created" value="1" />
                <input type="hidden" name="allow_edit" value="1" />
                <input type="hidden" name="template" value="1" />
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="dtarot_spread_user_pack_title"><?php esc_html_e('Title','daily-tarot'); ?></label></th>
                        <td>
                            <input type="text" id="dtarot_spread_user_pack_title" name="pack_title" required <?php echo $limitReached ? 'disabled' : ''; ?> />
                            <p class="description"><?php esc_html_e('Pack ID is generated automatically from the title.','daily-tarot'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="dtarot_spread_user_pack_system"><?php esc_html_e('System','daily-tarot'); ?></label></th>
                        <td>
                            <select id="dtarot_spread_user_pack_system" name="pack_system" <?php echo $limitReached ? 'disabled' : ''; ?>>
                                <?php foreach (Cards::systems() as $k => $label): ?>
                                    <option value="<?php echo esc_attr($k); ?>"><?php echo esc_html($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                </table>
                <p>
                    <button type="submit" class="button button-primary" <?php echo $limitReached ? 'disabled' : ''; ?>><?php esc_html_e('Create pack','daily-tarot'); ?></button>
                    <?php if ($limitReached): ?>
                        <span class="description" style="margin-left:8px;"><?php esc_html_e('Limit reached for free users.','daily-tarot'); ?></span>
                    <?php endif; ?>
                </p>
            </form>
        </div>

        <div class="dtarot-card">
            <h2><?php esc_html_e('Spread Meaning Packs','daily-tarot'); ?></h2>
            <p class="description"><?php esc_html_e('These packs define position labels and meanings for each spread preset.','daily-tarot'); ?></p>
            <?php if ($userCreated): ?>
                <p>
                    <button type="button" class="button dtarot-open-meanings-modal" data-dtarot-modal="#dtarot-spread-meanings-modal">
                        <?php esc_html_e('Edit meanings','daily-tarot'); ?>
                    </button>
                </p>
            <?php endif; ?>
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e('ID','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Title','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Version','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Editable','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Template','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Installed','daily-tarot'); ?></th>
                    </tr>
                </thead>
                <tbody>
                <?php if (!$packs): ?>
                    <tr><td colspan="6"><?php esc_html_e('No spread meaning packs installed.','daily-tarot'); ?></td></tr>
                <?php else: ?>
                    <?php foreach ($packs as $id => $pack): ?>
                        <?php
                        $title = is_array($pack) && isset($pack['title']) ? (string)$pack['title'] : (string)$id;
                        $version = is_array($pack) && isset($pack['version']) ? (string)$pack['version'] : '';
                        $editableLabel = is_array($pack) && !empty($pack['allow_edit']) ? __('Yes','daily-tarot') : __('No','daily-tarot');
                        $templateLabel = is_array($pack) && !empty($pack['template']) ? __('Yes','daily-tarot') : __('No','daily-tarot');
                        $installed = is_array($pack) && isset($pack['installed_at']) ? (string)$pack['installed_at'] : '';
                        ?>
                        <tr>
                            <td><?php echo esc_html((string)$id); ?></td>
                            <td><?php echo esc_html($title); ?></td>
                            <td><?php echo esc_html($version); ?></td>
                            <td><?php echo esc_html($editableLabel); ?></td>
                            <td><?php echo esc_html($templateLabel); ?></td>
                            <td><?php echo esc_html($installed); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php if (!$canEdit): ?>
            <p class="description"><?php esc_html_e('The default set is not editable. Create a new set to start editing.','daily-tarot'); ?></p>
        <?php endif; ?>
        <div class="dtarot-modal" id="dtarot-spread-meanings-modal" aria-hidden="true">
            <div class="dtarot-modal-backdrop" data-dtarot-modal-close></div>
            <div class="dtarot-modal-panel" role="dialog" aria-modal="true" aria-labelledby="dtarot-spread-meanings-title">
                <div class="dtarot-modal-header">
                    <h2 id="dtarot-spread-meanings-title"><?php esc_html_e('Edit template pack','daily-tarot'); ?></h2>
                    <button type="button" class="dtarot-modal-close" data-dtarot-modal-close aria-label="<?php esc_attr_e('Close','daily-tarot'); ?>">x</button>
                </div>
                <?php self::renderSpreadPackEditor($editable, $presets); ?>
            </div>
        </div>
                <?php
                $js = implode("\n", [
                    '(function(){',
                    '    function openModal(modal){',
                    '        if (!modal) return;',
                    "        modal.style.removeProperty('display');",
                    "        modal.classList.add('is-open');",
                    "        modal.setAttribute('aria-hidden', 'false');",
                    '    }',
                    '    function closeModal(modal){',
                    '        if (!modal) return;',
                    "        modal.classList.remove('is-open');",
                    "        modal.setAttribute('aria-hidden', 'true');",
                    '    }',
                    "    document.addEventListener('click', function(e){",
                    "        var btn = e.target && e.target.closest ? e.target.closest('.dtarot-open-meanings-modal') : null;",
                    '        if (btn) {',
                    '            e.preventDefault();',
                    "            var sel = btn.getAttribute('data-dtarot-modal') || '';",
                    '            var modal = sel ? document.querySelector(sel) : null;',
                    '            if (modal) openModal(modal);',
                    '            return;',
                    '        }',
                    "        var closeBtn = e.target && e.target.closest ? e.target.closest('[data-dtarot-modal-close]') : null;",
                    '        if (closeBtn) {',
                    '            e.preventDefault();',
                    "            var m = closeBtn.closest ? closeBtn.closest('.dtarot-modal') : null;",
                    '            if (m) closeModal(m);',
                    '        }',
                    '    }, true);',
                    '})();',
                    '',
                ]);
                wp_add_inline_script('dtarot-admin', $js, 'after');
                ?>
        <?php if ($canEdit): ?>
            <div class="dtarot-card">
                <h2><?php esc_html_e('Add new pack (developer)','daily-tarot'); ?></h2>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_spread_pack_create'); ?>
                    <input type="hidden" name="action" value="dtarot_spread_pack_create" />
                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><label for="dtarot_spread_pack_id"><?php esc_html_e('Pack ID','daily-tarot'); ?></label></th>
                            <td><input type="text" id="dtarot_spread_pack_id" name="pack_id" placeholder="custom_pack" required /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="dtarot_spread_pack_title"><?php esc_html_e('Title','daily-tarot'); ?></label></th>
                            <td><input type="text" id="dtarot_spread_pack_title" name="pack_title" required /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="dtarot_spread_pack_system"><?php esc_html_e('System','daily-tarot'); ?></label></th>
                            <td>
                                <select id="dtarot_spread_pack_system" name="pack_system">
                                    <?php foreach (Cards::systems() as $k => $label): ?>
                                        <option value="<?php echo esc_attr($k); ?>"><?php echo esc_html($label); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Flags','daily-tarot'); ?></th>
                            <td>
                                <label><input type="checkbox" name="allow_edit" value="1" /> <?php esc_html_e('Allow edit','daily-tarot'); ?></label><br />
                                <label><input type="checkbox" name="template" value="1" /> <?php esc_html_e('Template pack','daily-tarot'); ?></label>
                            </td>
                        </tr>
                    </table>
                    <p><button type="submit" class="button button-primary"><?php esc_html_e('Create pack','daily-tarot'); ?></button></p>
                </form>
            </div>
        <?php endif; ?>
        <?php
    }

    /**
     * @param array<string,array<string,mixed>> $editable
     * @param array<string,array<string,mixed>> $presets
     */
    private static function renderSpreadPackEditor(array $editable, array $presets): void {
        if (!$editable) {
            echo '<p class="description">' . esc_html__('No editable template packs installed.','daily-tarot') . '</p>';
            return;
        }
        ?>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('dtarot_spread_pack_save'); ?>
            <input type="hidden" name="action" value="dtarot_spread_pack_save" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="dtarot_spread_edit_pack"><?php esc_html_e('Pack','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_spread_edit_pack" name="pack_id">
                            <?php foreach ($editable as $id => $pack): ?>
                                <option value="<?php echo esc_attr($id); ?>"><?php echo esc_html((string)($pack['title'] ?? $id)); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_spread_edit_preset"><?php esc_html_e('Spread preset','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_spread_edit_preset" name="preset_id">
                            <?php foreach ($presets as $id => $preset): ?>
                                <option value="<?php echo esc_attr($id); ?>"><?php echo esc_html($preset['label']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>
            </table>

            <p class="description"><?php esc_html_e('Save to update meanings for the selected preset.','daily-tarot'); ?></p>

            <div class="dtarot-spread-editor" data-dtarot-spread-editor="1">
                <?php foreach ($presets as $id => $preset): ?>
                    <div class="dtarot-spread-editor-preset" data-preset="<?php echo esc_attr($id); ?>">
                        <?php foreach ($preset['slots'] as $idx => $slot): ?>
                            <label>
                                <span class="dtarot-spread-editor-label"><?php echo esc_html((string)($slot['label'] ?? '')); ?></span>
                                <textarea name="meanings[<?php echo esc_attr($id); ?>][<?php echo (int)$idx; ?>]" rows="3"></textarea>
                            </label>
                        <?php endforeach; ?>
                    </div>
                <?php endforeach; ?>
            </div>

            <p><button type="submit" class="button button-primary"><?php esc_html_e('Save meanings','daily-tarot'); ?></button></p>
        </form>
        <?php
        $editableData = [];
        foreach ($editable as $id => $pack) {
            if (!is_array($pack)) continue;
            $editableData[$id] = [
                'title' => (string)($pack['title'] ?? $id),
                'spreads' => isset($pack['spreads']) && is_array($pack['spreads']) ? $pack['spreads'] : [],
            ];
        }
        ?>
        <script type="application/json" id="dtarot-spread-pack-data"><?php echo wp_json_encode($editableData); ?></script>
        <?php
    }
}
