<?php
declare(strict_types=1);


namespace DailyTarot\Admin\Pages;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.DateTime.RestrictedFunctions.date_date

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Admin\Backup;
use DailyTarot\Automation\AutomationSettings;
use DailyTarot\Automation\Scheduler;
use DailyTarot\Calendar\DayEntryRepository;
use DailyTarot\Calendar\DayEntryTable;
use DailyTarot\Registry\Cards;
use DailyTarot\Admin\UiSettings;
use DailyTarot\Support\License;
use DailyTarot\Support\FeatureFlags;
use DailyTarot\Support\AiProviderSettings;
use DailyTarot\Support\EmailCtaSettings;
use DailyTarot\Support\EmailCtaStore;
use DailyTarot\Support\ProUrls;
use DailyTarot\Support\RelatedLinks;
use DailyTarot\Support\ShortcodeSettings;
use DailyTarot\Support\ShareImageSettings;
use DailyTarot\Support\BookingSettings;
use DailyTarot\Support\PostTypes;
use DailyTarot\Seo\Sitemap;
use DailyTarot\Support\Log;

final class Settings {

    private const DIAG_PUBLIC_TRANSIENT_PREFIX = 'dtarot_diag_public_fetch_';

    private const TAB_BACKUP = 'backup';
    private const TAB_PACKS = 'packs';
    private const TAB_CTA = 'cta';
    private const TAB_SHORTCODE = 'shortcode';
    private const TAB_BOOKINGS = 'bookings';
    private const TAB_HELP = 'help';
    private const TAB_DIAGNOSTICS = 'diagnostics';
    private const TAB_PRO = 'pro';

    private static function baseUrl(): string {
        return admin_url('admin.php?page=daily-tarot-settings');
    }

    private static function defaultTabForNotice(string $notice): string {
        $notice = sanitize_key($notice);

        // Heuristic: ZIP imports/exports and installed-pack actions live under Packs.
        $packsNotices = [
            'deck_zip_ok',
            'pack_zip_ok',
            'import_report',
            // import_error can be either JSON or ZIP; default to Packs if it looks ZIP-related.
            'import_error',
        ];

        if (in_array($notice, $packsNotices, true)) {
            $msg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
            $zipish = in_array($msg, [
                'zip_missing','upload_dir','workdir','bad_zip','missing_manifest','bad_manifest','format_version','min_version',
                'missing_files','missing_deck','deck_slug','deck_create','missing_pack','pack_slug','pack_create','exists',
                'downgrade_blocked','invalid_url','download_failed',
            ], true);

            // If we can't tell, Packs is still the better default for import_report.
            if ($notice === 'import_report' || $notice === 'deck_zip_ok' || $notice === 'pack_zip_ok' || $zipish) {
                return self::TAB_PACKS;
            }
        }

        return self::TAB_BACKUP;
    }

    private static function renderTabNav(string $active): void {
        $tabs = [
            self::TAB_BACKUP => __('Backup','daily-tarot'),
            self::TAB_PACKS => __('Packs','daily-tarot'),
            self::TAB_CTA => __('CTA','daily-tarot'),
            self::TAB_SHORTCODE => __('Shortcode','daily-tarot'),
            self::TAB_BOOKINGS => __('Bookings','daily-tarot'),
            self::TAB_HELP => __('Help','daily-tarot'),
            self::TAB_DIAGNOSTICS => __('Diagnostics','daily-tarot'),
            self::TAB_PRO => __('Pro','daily-tarot'),
        ];
        echo '<h2 class="nav-tab-wrapper">';
        foreach ($tabs as $id => $label) {
            $url = add_query_arg(['tab' => $id], self::baseUrl());
            $cls = 'nav-tab' . (($active === $id) ? ' nav-tab-active' : '');
            echo '<a class="' . esc_attr($cls) . '" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
        }
        echo '</h2>';
    }

    private static function resolveTab(string $tab, string $noticeForDefault): string {
        $tab = sanitize_key($tab);
        $noticeForDefault = sanitize_key($noticeForDefault);

        $allowed = [
            self::TAB_BACKUP,
            self::TAB_PACKS,
            self::TAB_CTA,
            self::TAB_SHORTCODE,
            self::TAB_BOOKINGS,
            self::TAB_HELP,
            self::TAB_DIAGNOSTICS,
            self::TAB_PRO,
        ];

        if (!in_array($tab, $allowed, true)) {
            return self::defaultTabForNotice($noticeForDefault);
        }

        return $tab;
    }

    public static function render_panel(string $tab): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));

        $noticeForDefault = isset($_GET['dtarot_backup']) ? sanitize_key((string)wp_unslash($_GET['dtarot_backup'])) : '';
        $tab = self::resolveTab($tab, $noticeForDefault);

        ?>
        <div id="dtarot-settings-panel" data-tab="<?php echo esc_attr($tab); ?>">
            <?php
            if ($tab === self::TAB_PACKS) {
                self::renderPacksTab();
            } elseif ($tab === self::TAB_CTA) {
                self::renderCtaTab();
            } elseif ($tab === self::TAB_SHORTCODE) {
                self::renderShortcodeTab();
            } elseif ($tab === self::TAB_BOOKINGS) {
                self::renderBookingsTab();
            } elseif ($tab === self::TAB_HELP) {
                self::renderHelpTab();
            } elseif ($tab === self::TAB_DIAGNOSTICS) {
                self::renderDiagnosticsTab();
            } elseif ($tab === self::TAB_PRO) {
                self::renderProTab();
            } else {
                self::renderBackupTab();
            }
            ?>
        </div>
        <?php
    }

    private static function renderProTab(): void {
        $isPro = FeatureFlags::enabled('pro');
        $fs = function_exists('dtarot_fs') ? dtarot_fs() : false;
        $usingFreemius = is_object($fs);
        $lic = class_exists(License::class) ? License::status() : ['status' => 'inactive', 'plan' => ''];

        $buyUrl = class_exists(ProUrls::class) ? trim((string)ProUrls::buyProUrl()) : '';

        $notice = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
        $aiProvider = class_exists(AiProviderSettings::class) ? AiProviderSettings::get() : ['url' => ''];
        $status = isset($lic['status']) ? (string)$lic['status'] : 'inactive';
        $plan = isset($lic['plan']) ? (string)$lic['plan'] : '';
        $accountUrl = '';
        if ($usingFreemius && method_exists($fs, 'get_account_url')) {
            $accountUrl = (string)$fs->get_account_url();
        }
        $uniqueAffix = '';
        if ($usingFreemius && method_exists($fs, 'get_unique_affix')) {
            $uniqueAffix = (string)$fs->get_unique_affix();
        }
        $openActivation = isset($_GET['dtarot_activate_license']) && (string)sanitize_text_field((string)wp_unslash($_GET['dtarot_activate_license'])) === '1';
        ?>
        <h2><?php esc_html_e('Pro (License)','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('License management is handled by Freemius for this plugin.','daily-tarot'); ?></p>

        <?php if ($notice === 'ai_provider_saved'): ?>
            <div class="notice notice-success is-dismissible"><p><?php esc_html_e('AI provider settings saved.','daily-tarot'); ?></p></div>
        <?php elseif ($notice === 'pro_required'): ?>
            <div class="notice notice-error is-dismissible"><p><?php esc_html_e('Activate Pro to change AI provider settings.','daily-tarot'); ?></p></div>
        <?php elseif ($notice === 'license_synced'): ?>
            <div class="notice notice-success is-dismissible"><p><?php esc_html_e('License status refreshed.','daily-tarot'); ?></p></div>
        <?php elseif ($notice === 'license_sync_unavailable'): ?>
            <div class="notice notice-warning is-dismissible"><p><?php esc_html_e('License sync is unavailable until the site is connected in Freemius. Please activate your license first.','daily-tarot'); ?></p></div>
        <?php endif; ?>

        <?php
        $fsDebug = [];
        if ($usingFreemius) {
            if (method_exists($fs, 'is_registered')) {
                $fsDebug[] = 'registered: ' . ($fs->is_registered() ? 'yes' : 'no');
            }
            if (method_exists($fs, 'has_active_valid_license')) {
                $fsDebug[] = 'active license: ' . ($fs->has_active_valid_license() ? 'yes' : 'no');
            }
            if (method_exists($fs, 'is_paying')) {
                $fsDebug[] = 'paying: ' . ($fs->is_paying() ? 'yes' : 'no');
            }
            if (method_exists($fs, 'get_site')) {
                $site = $fs->get_site();
                if (is_object($site) && isset($site->license_id)) {
                    $fsDebug[] = 'license id: ' . (string)$site->license_id;
                }
            }
        }
        ?>

        <table class="form-table" role="presentation" style="max-width: 980px;">
            <tr>
                <th scope="row"><?php esc_html_e('Status','daily-tarot'); ?></th>
                <td>
                    <div>
                        <strong><?php echo esc_html($isPro ? __('Active','daily-tarot') : __('Not active','daily-tarot')); ?></strong>
                        <span class="description">
                            <?php
                            $metaBits = [];
                            if ($status !== '') $metaBits[] = 'status: ' . $status;
                            if ($plan !== '') $metaBits[] = 'plan: ' . $plan;
                            echo esc_html($metaBits ? ('(' . implode(', ', $metaBits) . ')') : '');
                            ?>
                        </span>
                    </div>
                    <?php if (!$isPro && $uniqueAffix !== '' && $usingFreemius && method_exists($fs, '_add_license_activation_dialog_box')): ?>
                        <p style="margin-top:10px;">
                            <?php
                            $activateUrl = add_query_arg(
                                ['tab' => self::TAB_PRO, 'dtarot_activate_license' => '1'],
                                self::baseUrl()
                            );
                            ?>
                            <a class="button button-primary activate-license-trigger <?php echo esc_attr($uniqueAffix); ?>" href="<?php echo esc_url($activateUrl); ?>">
                                <?php esc_html_e('Activate license','daily-tarot'); ?>
                            </a>
                            <span class="description" style="margin-left:8px;"><?php esc_html_e('Enter your license key to enable Pro on this site.','daily-tarot'); ?></span>
                        </p>
                        <?php $fs->_add_license_activation_dialog_box(); ?>
                        <?php
                        $canOpenActivation = $openActivation && method_exists($fs, '_open_license_activation_dialog_box');
                        if ($canOpenActivation && method_exists($fs, 'get_site')) {
                            $site = $fs->get_site();
                            if (!is_object($site)) {
                                $canOpenActivation = false;
                            }
                        }
                        ?>
                        <?php if ($canOpenActivation): ?>
                            <?php $fs->_open_license_activation_dialog_box(); ?>
                        <?php endif; ?>
                    <?php endif; ?>
                    <?php if (!$isPro && $buyUrl !== ''): ?>
                        <p style="margin-top:10px;">
                            <a class="button button-primary" href="<?php echo esc_url($buyUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Get a license','daily-tarot'); ?></a>
                        </p>
                    <?php endif; ?>
                    <?php if ($isPro && $accountUrl !== ''): ?>
                        <p style="margin-top:10px;">
                            <a class="button" href="<?php echo esc_url($accountUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Manage subscription','daily-tarot'); ?></a>
                        </p>
                    <?php endif; ?>
                </td>
            </tr>
            <?php if ($usingFreemius): ?>
                <tr>
                    <th scope="row"><?php esc_html_e('License tools','daily-tarot'); ?></th>
                    <td>
                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                            <?php wp_nonce_field('dtarot_license_sync'); ?>
                            <input type="hidden" name="action" value="dtarot_license_sync" />
                            <button type="submit" class="button"><?php esc_html_e('Refresh license status','daily-tarot'); ?></button>
                        </form>
                        <?php if (!empty($fsDebug)): ?>
                            <p class="description"><?php echo esc_html('Freemius: ' . implode(', ', $fsDebug)); ?></p>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endif; ?>
        </table>

        <hr />

        <h3><?php esc_html_e('AI provider (first setting)','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('Optional: configure a remote AI provider endpoint. If set, the plugin will POST generation payloads as JSON and expects JSON response with {"content":"...","daily_text":"..."}.','daily-tarot'); ?></p>

        <?php
        $detectedProviderUrl = function_exists('rest_url') ? (string)rest_url('dtarot/v1/generate') : (string)home_url('/wp-json/dtarot/v1/generate');
        $detectedProviderUrl = trim($detectedProviderUrl);

        $providerSecretDefined = defined('DTAROT_AI_PROVIDER_SECRET') && trim((string)DTAROT_AI_PROVIDER_SECRET) !== '';
        $upstreamKeyDefined = defined('DTAROT_AI_API_KEY') && trim((string)DTAROT_AI_API_KEY) !== '';
        ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width: 980px;">
            <?php wp_nonce_field('dtarot_ai_provider_save'); ?>
            <input type="hidden" name="action" value="dtarot_ai_provider_save" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="dtarot_ai_provider_url"><?php esc_html_e('Provider endpoint URL','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="url"
                            id="dtarot_ai_provider_url"
                            name="dtarot_ai_provider_url"
                            value="<?php echo esc_attr((string)($aiProvider['url'] ?? '')); ?>"
                            style="width: 520px; max-width: 100%;"
                            placeholder="<?php echo esc_attr($detectedProviderUrl !== '' ? $detectedProviderUrl : 'https://example.com/wp-json/dtarot/v1/generate'); ?>"
                            <?php echo $isPro ? '' : 'disabled'; ?>
                        />
                        <?php if (!$isPro): ?>
                            <p class="description"><?php esc_html_e('Activate Pro to enable this setting.','daily-tarot'); ?></p>
                        <?php else: ?>
                            <p class="description">
                                <?php esc_html_e('Leave empty to use the default generator/filter fallback.','daily-tarot'); ?>
                                <?php if ($detectedProviderUrl !== ''): ?>
                                    <br /><?php esc_html_e('Detected same-site provider endpoint:','daily-tarot'); ?> <code><?php echo esc_html($detectedProviderUrl); ?></code>
                                <?php endif; ?>
                            </p>
                        <?php endif; ?>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_ai_output_language"><?php esc_html_e('AI output language','daily-tarot'); ?></label></th>
                    <td>
                        <?php $aiLang = is_array($aiProvider) && isset($aiProvider['output_language']) ? trim((string)$aiProvider['output_language']) : ''; ?>
                        <select id="dtarot_ai_output_language" name="dtarot_ai_output_language" style="width: 520px; max-width: 100%;" <?php echo $isPro ? '' : 'disabled'; ?>>
                            <option value="" <?php selected($aiLang, ''); ?>><?php esc_html_e('Use site language','daily-tarot'); ?></option>
                            <option value="en-US" <?php selected($aiLang, 'en-US'); ?>>English (en-US)</option>
                            <option value="es-ES" <?php selected($aiLang, 'es-ES'); ?>>Español (es-ES)</option>
                            <option value="fr-FR" <?php selected($aiLang, 'fr-FR'); ?>>Français (fr-FR)</option>
                            <option value="de-DE" <?php selected($aiLang, 'de-DE'); ?>>Deutsch (de-DE)</option>
                            <option value="it-IT" <?php selected($aiLang, 'it-IT'); ?>>Italiano (it-IT)</option>
                            <option value="pt-BR" <?php selected($aiLang, 'pt-BR'); ?>>Português (pt-BR)</option>
                            <option value="nl-NL" <?php selected($aiLang, 'nl-NL'); ?>>Nederlands (nl-NL)</option>
                            <option value="pl-PL" <?php selected($aiLang, 'pl-PL'); ?>>Polski (pl-PL)</option>
                        </select>
                        <p class="description">
                            <?php esc_html_e('Controls the language used by AI generation (provider/filter). UI translations follow your WordPress Site Language and installed translation files.','daily-tarot'); ?>
                        </p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="dtarot_ai_provider_secret"><?php esc_html_e('Provider secret','daily-tarot'); ?></label>
                        <button type="button" class="button-link dtarot-help-btn" data-dtarot-modal="#dtarot-help-provider-secret" aria-label="<?php echo esc_attr(__('Help','daily-tarot')); ?>" style="vertical-align:middle; margin-left:6px;">
                            <span class="dashicons dashicons-editor-help"></span>
                        </button>
                    </th>
                    <td>
                        <div style="display:flex; gap:8px; align-items:center; flex-wrap:wrap;">
                            <input
                                type="password"
                                id="dtarot_ai_provider_secret"
                                name="dtarot_ai_provider_secret"
                                value=""
                                autocomplete="new-password"
                                style="width: 520px; max-width: 100%;"
                                placeholder="<?php echo esc_attr(__('Enter a shared secret (stored, hidden)','daily-tarot')); ?>"
                                <?php echo $isPro ? '' : 'disabled'; ?>
                            />
                            <button type="button" class="button" id="dtarot_ai_provider_secret_generate" <?php echo $isPro ? '' : 'disabled'; ?>><?php esc_html_e('Generate','daily-tarot'); ?></button>
                        </div>
                        <?php if (!$isPro): ?>
                            <p class="description"><?php esc_html_e('Activate Pro to enable this setting.','daily-tarot'); ?></p>
                        <?php else: ?>
                            <p class="description">
                                <?php esc_html_e('Used as the X-DTAROT-Provider-Secret header when calling the provider (and required by the same-site provider). Leave empty to keep the existing value.','daily-tarot'); ?>
                                <br /><strong><?php esc_html_e('Status:','daily-tarot'); ?></strong>
                                <?php
                                if ($providerSecretDefined) {
                                    esc_html_e('Defined in wp-config.php (constant overrides settings).','daily-tarot');
                                } else {
                                    echo !empty($aiProvider['has_provider_secret']) ? esc_html__('Saved (hidden).','daily-tarot') : esc_html__('Not set.','daily-tarot');
                                }
                                ?>
                            </p>
                        <?php endif; ?>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="dtarot_ai_upstream_api_key"><?php esc_html_e('Upstream AI API key','daily-tarot'); ?></label>
                        <button type="button" class="button-link dtarot-help-btn" data-dtarot-modal="#dtarot-help-upstream-key" aria-label="<?php echo esc_attr(__('Help','daily-tarot')); ?>" style="vertical-align:middle; margin-left:6px;">
                            <span class="dashicons dashicons-editor-help"></span>
                        </button>
                    </th>
                    <td>
                        <input
                            type="password"
                            id="dtarot_ai_upstream_api_key"
                            name="dtarot_ai_upstream_api_key"
                            value=""
                            autocomplete="new-password"
                            style="width: 520px; max-width: 100%;"
                            placeholder="<?php echo esc_attr(__('Enter your upstream AI API key (stored, hidden)','daily-tarot')); ?>"
                            <?php echo $isPro ? '' : 'disabled'; ?>
                        />
                        <?php if (!$isPro): ?>
                            <p class="description"><?php esc_html_e('Activate Pro to enable this setting.','daily-tarot'); ?></p>
                        <?php else: ?>
                            <p class="description">
                                <?php esc_html_e('Used by the same-site provider to call your upstream AI API. Leave empty to keep the existing value.','daily-tarot'); ?>
                                <br /><strong><?php esc_html_e('Status:','daily-tarot'); ?></strong>
                                <?php
                                if ($upstreamKeyDefined) {
                                    esc_html_e('Defined in wp-config.php (constant overrides settings).','daily-tarot');
                                } else {
                                    echo !empty($aiProvider['has_upstream_api_key']) ? esc_html__('Saved (hidden).','daily-tarot') : esc_html__('Not set.','daily-tarot');
                                }
                                ?>
                            </p>
                        <?php endif; ?>
                    </td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-primary" <?php echo $isPro ? '' : 'disabled'; ?>><?php esc_html_e('Save AI provider','daily-tarot'); ?></button>
            </p>
        </form>

        <?php if ($isPro): ?>
            <h4 style="margin-top:18px;"><?php esc_html_e('Test provider','daily-tarot'); ?></h4>
            <p class="description"><?php esc_html_e('Sends a sample generation payload for the selected date to the configured provider URL and shows the response.','daily-tarot'); ?></p>

            <p style="display:flex; gap:10px; align-items:center; flex-wrap:wrap;">
                <label for="dtarot_ai_provider_test_date"><?php esc_html_e('Date','daily-tarot'); ?></label>
                <input type="date" id="dtarot_ai_provider_test_date" value="<?php echo esc_attr(function_exists('wp_date') ? (string)wp_date('Y-m-d') : (string)current_time('Y-m-d')); ?>" />
                <button type="button" class="button" id="dtarot_ai_provider_test_btn"><?php esc_html_e('Send test request','daily-tarot'); ?></button>
                <span class="description" id="dtarot_ai_provider_test_status"></span>
            </p>

            <div id="dtarot_ai_provider_test_output" style="display:none; max-width: 980px;">
                <p><strong><?php esc_html_e('Parsed result','daily-tarot'); ?></strong></p>
                <textarea readonly rows="5" style="width:100%;" id="dtarot_ai_provider_test_content" placeholder="content..."></textarea>
                <textarea readonly rows="5" style="width:100%; margin-top:8px;" id="dtarot_ai_provider_test_daily_text" placeholder="daily_text..."></textarea>
                <p style="margin-top:10px;"><strong><?php esc_html_e('HTTP response (snippet)','daily-tarot'); ?></strong></p>
                <pre style="white-space:pre-wrap; background:#f6f7f7; border:1px solid #dcdcde; border-radius:8px; padding:10px;" id="dtarot_ai_provider_test_body"></pre>
            </div>
        <?php endif; ?>

        <div class="dtarot-modal" id="dtarot-help-provider-secret" style="display:none;">
            <div class="dtarot-modal-backdrop"></div>
            <div class="dtarot-modal-panel" style="width: min(640px, calc(100% - 24px));">
                <div class="dtarot-modal-header">
                    <h3 style="margin:0;"><?php esc_html_e('Provider secret','daily-tarot'); ?></h3>
                    <button type="button" class="dtarot-modal-close" aria-label="<?php echo esc_attr(__('Close','daily-tarot')); ?>">×</button>
                </div>
                <div>
                    <p><?php esc_html_e('You create this yourself. It is a shared password between the plugin and the provider endpoint on your site.','daily-tarot'); ?></p>
                    <p><?php esc_html_e('Pick a long random string (recommended 32–64+ characters). This secret protects your /generate endpoint from random people.','daily-tarot'); ?></p>
                    <p><?php esc_html_e('Easiest: click the Generate button next to the Provider secret field.','daily-tarot'); ?></p>
                    <p>
                        <?php esc_html_e('PowerShell example:','daily-tarot'); ?>
                        <code>[guid]::NewGuid().ToString('N') + [guid]::NewGuid().ToString('N')</code>
                        <br />

                        <h2><?php esc_html_e('Self-check (rewrites + sitemap)','daily-tarot'); ?></h2>
                        <p class="description"><?php esc_html_e('Quick checks for readable routes and sitemap availability.','daily-tarot'); ?></p>

                        <?php
                            $permalinkStructure = (string)get_option('permalink_structure', '');
                            $permalinksOn = $permalinkStructure !== '';
                            $rules = get_option('rewrite_rules');
                            $rulesStr = is_array($rules) ? implode("\n", array_keys($rules)) : '';
                            $hasReadableRule = $rulesStr !== '' && (str_contains($rulesStr, 'dtarot_date') || str_contains($rulesStr, 'card-of-the-day'));
                            $hasSitemapRule = $rulesStr !== '' && str_contains($rulesStr, 'dtarot-sitemap');
                            $sitemapUrl = class_exists(\DailyTarot\Seo\Sitemap::class) ? \DailyTarot\Seo\Sitemap::url() : '';
                        ?>

                        <table class="widefat striped" style="max-width:1100px;">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e('Check','daily-tarot'); ?></th>
                                    <th><?php esc_html_e('Status','daily-tarot'); ?></th>
                                    <th><?php esc_html_e('Details','daily-tarot'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td><?php esc_html_e('Permalinks enabled','daily-tarot'); ?></td>
                                    <td><?php echo esc_html($fmtBool($permalinksOn)); ?></td>
                                    <td><?php echo esc_html($permalinksOn ? $permalinkStructure : __('Readable routes require permalinks (not Plain).','daily-tarot')); ?></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e('Readable route rewrite present','daily-tarot'); ?></td>
                                    <td><?php echo esc_html($fmtBool($hasReadableRule)); ?></td>
                                    <td><?php esc_html_e('If missing, click “Refresh permalinks” when prompted after updates.','daily-tarot'); ?></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e('Sitemap rewrite present','daily-tarot'); ?></td>
                                    <td><?php echo esc_html($fmtBool($hasSitemapRule)); ?></td>
                                    <td><?php echo esc_html($sitemapUrl !== '' ? $sitemapUrl : __('Unknown','daily-tarot')); ?></td>
                                </tr>
                            </tbody>
                        </table>
                        <?php esc_html_e('CMD example (uses PowerShell):','daily-tarot'); ?>
                        <code>powershell -NoProfile -Command "[guid]::NewGuid().ToString('N') + [guid]::NewGuid().ToString('N')"</code>
                    </p>
                </div>
            </div>
        </div>

        <div class="dtarot-modal" id="dtarot-help-upstream-key" style="display:none;">
            <div class="dtarot-modal-backdrop"></div>
            <div class="dtarot-modal-panel" style="width: min(640px, calc(100% - 24px));">
                <div class="dtarot-modal-header">
                    <h3 style="margin:0;"><?php esc_html_e('Upstream AI API key','daily-tarot'); ?></h3>
                    <button type="button" class="dtarot-modal-close" aria-label="<?php echo esc_attr(__('Close','daily-tarot')); ?>">×</button>
                </div>
                <div>
                    <p><?php esc_html_e('This key comes from your AI provider. In our setup we use OpenAI (OpenAI-compatible chat completions).','daily-tarot'); ?></p>
                    <p>
                        <?php esc_html_e('Create an API key in your OpenAI account and paste it here:','daily-tarot'); ?>
                        <a href="<?php echo esc_url('https://platform.openai.com/api-keys'); ?>" target="_blank" rel="noopener noreferrer">platform.openai.com/api-keys</a>
                    </p>
                    <p><?php esc_html_e('Important: this key is used by your WordPress server (not your laptop).','daily-tarot'); ?></p>
                </div>
            </div>
        </div>
        <?php
    }

    private static function renderBackupTab(): void {
        ?>
        <h2><?php esc_html_e('Backup (Import / Export)','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Export creates a JSON backup of decks, meaning packs, and calendar entries. Import restores/updates them on this site.','daily-tarot'); ?></p>

        <div style="display:flex; gap:24px; flex-wrap:wrap;">
            <div style="min-width:320px; max-width:520px; flex:1;">
                <h3><?php esc_html_e('Export','daily-tarot'); ?></h3>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_backup'); ?>
                    <input type="hidden" name="action" value="dtarot_export" />
                    <p>
                        <button type="submit" class="button button-primary"><?php esc_html_e('Download backup JSON','daily-tarot'); ?></button>
                    </p>
                </form>
                <p class="description">
                    <?php esc_html_e('Includes: Deck images (stored as URLs), Meaning Pack meanings, and all calendar entries.','daily-tarot'); ?>
                </p>
            </div>

            <div style="min-width:320px; max-width:520px; flex:1;">
                <h3><?php esc_html_e('Import','daily-tarot'); ?></h3>
                <form method="post" enctype="multipart/form-data" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_backup'); ?>
                    <input type="hidden" name="action" value="dtarot_import" />

                    <p>
                        <input type="file" name="dtarot_backup_file" accept="application/json,.json" required />
                    </p>

                    <p>
                        <label>
                            <input type="checkbox" name="replace_calendar" value="1" checked />
                            <?php esc_html_e('Replace calendar entries (otherwise merge by date)','daily-tarot'); ?>
                        </label>
                    </p>

                    <p>
                        <label>
                            <input type="checkbox" name="overwrite_existing" value="1" checked />
                            <?php esc_html_e('Overwrite existing decks/meaning packs (matched by slug)','daily-tarot'); ?>
                        </label>
                    </p>

                    <p>
                        <button type="submit" class="button"><?php esc_html_e('Import backup','daily-tarot'); ?></button>
                    </p>

                    <p class="description">
                        <?php esc_html_e('Decks and meaning packs are matched by slug and updated/created. Calendar deck/pack IDs are remapped during import.','daily-tarot'); ?>
                    </p>
                </form>
            </div>
        </div>
        <?php
    }

    private static function renderPacksTab(): void {
        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
        ]);
        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
        ]);
        ?>

        <h2><?php esc_html_e('Export ZIPs','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Create ZIP files (manifest.json + data/assets) for backups or distribution.','daily-tarot'); ?></p>

        <div style="display:flex; gap:24px; flex-wrap:wrap;">
            <div style="min-width:320px; max-width:520px; flex:1;">
                <h3><?php esc_html_e('Export deck ZIP (includes images)','daily-tarot'); ?></h3>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_backup'); ?>
                    <input type="hidden" name="action" value="dtarot_export_deck_zip" />
                    <p>
                        <select name="deck_id" required>
                            <option value=""><?php esc_html_e('Select deck…','daily-tarot'); ?></option>
                            <?php foreach ((array)$decks as $d) : ?>
                                <option value="<?php echo (int)$d->ID; ?>"><?php echo esc_html($d->post_title ?: __('(no title)','daily-tarot')); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <button type="submit" class="button"><?php esc_html_e('Download deck ZIP','daily-tarot'); ?></button>
                    </p>
                    <p class="description"><?php esc_html_e('Downloads all card images by URL and packages them with integrity hashes.','daily-tarot'); ?></p>
                </form>
            </div>

            <div style="min-width:320px; max-width:520px; flex:1;">
                <h3><?php esc_html_e('Export meaning pack ZIP','daily-tarot'); ?></h3>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_backup'); ?>
                    <input type="hidden" name="action" value="dtarot_export_pack_zip" />
                    <p>
                        <select name="pack_id" required>
                            <option value=""><?php esc_html_e('Select meaning pack…','daily-tarot'); ?></option>
                            <?php foreach ((array)$packs as $p) : ?>
                                <option value="<?php echo (int)$p->ID; ?>"><?php echo esc_html($p->post_title ?: __('(no title)','daily-tarot')); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <button type="submit" class="button"><?php esc_html_e('Download meaning pack ZIP','daily-tarot'); ?></button>
                    </p>
                </form>
            </div>
        </div>

        <hr />

        <h2><?php esc_html_e('Deck ZIP Import (with images)','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Upload a ZIP that contains manifest.json plus card images. Images will be imported into the Media Library and linked to the deck.','daily-tarot'); ?></p>

        <div style="display:flex; gap:24px; flex-wrap:wrap; align-items:flex-start;">
            <div style="min-width:320px; max-width:520px; flex:1;">
                <h3><?php esc_html_e('Import from file','daily-tarot'); ?></h3>
                <form method="post" enctype="multipart/form-data" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_backup'); ?>
                    <input type="hidden" name="action" value="dtarot_import_deck_zip" />

                    <p>
                        <input type="file" name="dtarot_deck_zip" accept="application/zip,.zip" required />
                        <button type="submit" class="button"><?php esc_html_e('Import deck ZIP','daily-tarot'); ?></button>
                    </p>

                    <p>
                        <label><input type="checkbox" name="overwrite" value="1" checked /> <?php esc_html_e('Overwrite existing deck with same slug','daily-tarot'); ?></label>
                        &nbsp;&nbsp;
                        <label><input type="checkbox" name="dry_run" value="1" /> <?php esc_html_e('Dry run (validate only)','daily-tarot'); ?></label>
                        &nbsp;&nbsp;
                        <label><input type="checkbox" name="allow_downgrade" value="1" /> <?php esc_html_e('Allow downgrade','daily-tarot'); ?></label>
                    </p>

                    <details style="max-width:900px;">
                        <summary><?php esc_html_e('ZIP format (manifest.json)','daily-tarot'); ?></summary>
                        <pre style="white-space:pre-wrap; background:#fff; padding:12px; border:1px solid #ccd0d4;">{
  "format_version": 1,
    "plugin_min_version": "3.1.2",
  "type": "deck",
  "files": [
    {"path": "back.jpg", "sha256": "..."},
    {"path": "cards/maj_00.jpg", "sha256": "..."}
  ],
  "deck": {
    "slug": "my-deck",
    "title": "My Deck",
    "status": "publish",
    "back": "back.jpg",
    "cards": {
      "maj_00": "cards/maj_00.jpg"
    }
  }
}</pre>
                        <p class="description"><?php esc_html_e('Card IDs must match the plugin registry (e.g. maj_00). Unknown IDs are skipped.','daily-tarot'); ?></p>
                    </details>
                </form>
            </div>

            <div style="min-width:320px; max-width:520px; flex:1;">
                <h3><?php esc_html_e('Import from URL','daily-tarot'); ?></h3>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_backup'); ?>
                    <input type="hidden" name="action" value="dtarot_import_deck_zip_url" />
                    <input type="hidden" name="redirect_to" value="<?php echo esc_attr(admin_url('admin.php?page=daily-tarot-settings')); ?>" />
                    <p>
                        <input type="url" name="zip_url" style="width:520px; max-width:100%;" placeholder="https://..." required />
                        <button type="submit" class="button"><?php esc_html_e('Download & import','daily-tarot'); ?></button>
                    </p>
                    <p>
                        <label><input type="checkbox" name="overwrite" value="1" checked /> <?php esc_html_e('Overwrite existing deck with same slug','daily-tarot'); ?></label>
                        &nbsp;&nbsp;
                        <label><input type="checkbox" name="dry_run" value="1" /> <?php esc_html_e('Dry run (validate only)','daily-tarot'); ?></label>
                        &nbsp;&nbsp;
                        <label><input type="checkbox" name="allow_downgrade" value="1" /> <?php esc_html_e('Allow downgrade','daily-tarot'); ?></label>
                    </p>
                </form>
            </div>
        </div>

        <hr />

        <h2><?php esc_html_e('Meaning Pack ZIP Import','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Upload a ZIP that contains manifest.json with a meaning_pack section. Use this format for backups or sharing packs.','daily-tarot'); ?></p>

        <div style="display:flex; gap:24px; flex-wrap:wrap; align-items:flex-start;">
            <div style="min-width:320px; max-width:520px; flex:1;">
                <h3><?php esc_html_e('Import from file','daily-tarot'); ?></h3>
                <form method="post" enctype="multipart/form-data" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_backup'); ?>
                    <input type="hidden" name="action" value="dtarot_import_pack_zip" />

                    <p>
                        <input type="file" name="dtarot_pack_zip" accept="application/zip,.zip" required />
                        <button type="submit" class="button"><?php esc_html_e('Import meaning pack ZIP','daily-tarot'); ?></button>
                    </p>

                    <p>
                        <label><input type="checkbox" name="overwrite" value="1" checked /> <?php esc_html_e('Overwrite existing meaning pack with same slug','daily-tarot'); ?></label>
                        &nbsp;&nbsp;
                        <label><input type="checkbox" name="dry_run" value="1" /> <?php esc_html_e('Dry run (validate only)','daily-tarot'); ?></label>
                        &nbsp;&nbsp;
                        <label><input type="checkbox" name="allow_downgrade" value="1" /> <?php esc_html_e('Allow downgrade','daily-tarot'); ?></label>
                    </p>

                    <details style="max-width:900px;">
                        <summary><?php esc_html_e('ZIP format (manifest.json)','daily-tarot'); ?></summary>
                        <pre style="white-space:pre-wrap; background:#fff; padding:12px; border:1px solid #ccd0d4;">{
  "format_version": 1,
    "plugin_min_version": "3.1.2",
  "type": "meaning_pack",
  "files": [],
  "meaning_pack": {
    "slug": "default-pack",
    "title": "Default Meaning Pack",
    "status": "publish",
    "meanings": {
      "maj_00": {
        "upright": "...",
        "reversed": "...",
        "keywords": "...",
        "short": "...",
        "long": "...",
        "correspondences": "...",
        "symbols": "..."
      }
    }
  }
}</pre>
                        <p class="description"><?php esc_html_e('Card IDs must match the plugin registry (e.g. maj_00). Unknown IDs are skipped.','daily-tarot'); ?></p>
                    </details>
                </form>
            </div>

            <div style="min-width:320px; max-width:520px; flex:1;">
                <h3><?php esc_html_e('Import from URL','daily-tarot'); ?></h3>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_backup'); ?>
                    <input type="hidden" name="action" value="dtarot_import_pack_zip_url" />
                    <input type="hidden" name="redirect_to" value="<?php echo esc_attr(admin_url('admin.php?page=daily-tarot-settings')); ?>" />
                    <p>
                        <input type="url" name="zip_url" style="width:520px; max-width:100%;" placeholder="https://..." required />
                        <button type="submit" class="button"><?php esc_html_e('Download & import','daily-tarot'); ?></button>
                    </p>
                    <p>
                        <label><input type="checkbox" name="overwrite" value="1" checked /> <?php esc_html_e('Overwrite existing meaning pack with same slug','daily-tarot'); ?></label>
                        &nbsp;&nbsp;
                        <label><input type="checkbox" name="dry_run" value="1" /> <?php esc_html_e('Dry run (validate only)','daily-tarot'); ?></label>
                        &nbsp;&nbsp;
                        <label><input type="checkbox" name="allow_downgrade" value="1" /> <?php esc_html_e('Allow downgrade','daily-tarot'); ?></label>
                    </p>
                </form>
            </div>
        </div>

        <hr />

        <h2><?php esc_html_e('Installed Packs','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Shows packs installed via ZIP import (upload or URL).','daily-tarot'); ?></p>

        <?php
        $installed = method_exists(Backup::class, 'installedPacks') ? Backup::installedPacks() : [];
        if (!is_array($installed) || empty($installed)) :
        ?>
            <p><?php esc_html_e('No installed packs recorded yet.','daily-tarot'); ?></p>
        <?php else : ?>
            <table class="widefat striped" style="max-width:1100px;">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Type','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Slug','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Title','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Version','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Installed','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Source','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Actions','daily-tarot'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($installed as $entry) :
                        if (!is_array($entry)) continue;
                        $type = isset($entry['type']) ? (string)$entry['type'] : '';
                        $slug = isset($entry['slug']) ? (string)$entry['slug'] : '';
                        $title = isset($entry['title']) ? (string)$entry['title'] : '';
                        $ver = isset($entry['pack_version']) ? (string)$entry['pack_version'] : '';
                        $at = isset($entry['installed_at']) ? (string)$entry['installed_at'] : '';
                        $source = isset($entry['source']) ? (string)$entry['source'] : '';
                        $url = isset($entry['source_url']) ? (string)$entry['source_url'] : '';
                        $src = $source;
                        if ($url !== '') $src .= ' – ' . $url;
                    ?>
                        <tr>
                            <td><?php echo esc_html($type); ?></td>
                            <td><?php echo esc_html($slug); ?></td>
                            <td><?php echo esc_html($title); ?></td>
                            <td><?php echo esc_html($ver); ?></td>
                            <td><?php echo esc_html($at); ?></td>
                            <td><?php echo esc_html($src); ?></td>
                            <td>
                                <?php if ($url !== ''): ?>
                                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline-block; margin-right:6px;">
                                        <?php wp_nonce_field('dtarot_backup'); ?>
                                        <input type="hidden" name="action" value="dtarot_installed_pack_update_url" />
                                        <input type="hidden" name="pack_type" value="<?php echo esc_attr($type); ?>" />
                                        <input type="hidden" name="pack_slug" value="<?php echo esc_attr($slug); ?>" />
                                        <button type="submit" class="button button-small"><?php esc_html_e('Update from URL','daily-tarot'); ?></button>
                                    </form>
                                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline-block;">
                                        <?php wp_nonce_field('dtarot_backup'); ?>
                                        <input type="hidden" name="action" value="dtarot_installed_pack_reinstall_url" />
                                        <input type="hidden" name="pack_type" value="<?php echo esc_attr($type); ?>" />
                                        <input type="hidden" name="pack_slug" value="<?php echo esc_attr($slug); ?>" />
                                        <button type="submit" class="button button-small"><?php esc_html_e('Reinstall','daily-tarot'); ?></button>
                                    </form>
                                <?php else: ?>
                                    <span class="description"><?php esc_html_e('No URL recorded','daily-tarot'); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
        <?php
    }

    private static function renderAutomationTab(): void {
        if (!class_exists(AutomationSettings::class)) {
            echo '<p>' . esc_html__('Automation is unavailable (missing plugin files).','daily-tarot') . '</p>';
            return;
        }

        $s = AutomationSettings::get();

        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);
        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);

        $systems = Cards::systems();

        $tomorrow = function_exists('wp_date') ? (string)wp_date('Y-m-d', strtotime('tomorrow')) : (string)date('Y-m-d', strtotime('tomorrow'));
        $meEmail = '';
        if (function_exists('wp_get_current_user')) {
            $u = wp_get_current_user();
            if (is_object($u) && isset($u->user_email)) {
                $meEmail = (string)$u->user_email;
            }
        }
        ?>
        <h2><?php esc_html_e('Automation (Daily publish + optional email)','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Uses WP-Cron to publish today\'s card and generate the Intro + Expanded text at the chosen time (site timezone).','daily-tarot'); ?></p>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width:900px;">
            <?php wp_nonce_field('dtarot_automation'); ?>
            <input type="hidden" name="action" value="dtarot_automation_save" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('Enable automation','daily-tarot'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="enabled" value="1" <?php checked($s['enabled']); ?> />
                            <?php esc_html_e('Publish the card of the day automatically','daily-tarot'); ?>
                        </label>
                        <p class="description"><?php esc_html_e('When enabled, the daily job publishes the Card of the Day and generates the Intro + Expanded text. If your publish time is soon, it may run shortly after saving. To publish immediately, use “Run job now” below.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Publish time','daily-tarot'); ?></th>
                    <td>
                        <input type="time" name="time" value="<?php echo esc_attr($s['time']); ?>" />
                        <p class="description"><?php esc_html_e('Time is in the site timezone (Settings → General).','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Deck','daily-tarot'); ?></th>
                    <td>
                        <select name="deck_id">
                            <option value="0"><?php esc_html_e('Auto (most recent)','daily-tarot'); ?></option>
                            <?php foreach ((array)$decks as $d) :
                                $system = Cards::normalizeSystem((string)get_post_meta((int)$d->ID, '_dtarot_system', true));
                                $label = ($systems[$system] ?? $system);
                                $title = $d->post_title ?: __('(no title)','daily-tarot');
                            ?>
                                <option value="<?php echo (int)$d->ID; ?>" <?php selected((int)$s['deck_id'], (int)$d->ID); ?>><?php echo esc_html('[' . $label . '] ' . $title); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php esc_html_e('If set to Auto, the most recent deck is used.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Meaning pack','daily-tarot'); ?></th>
                    <td>
                        <select name="pack_id">
                            <option value="0"><?php esc_html_e('Auto (system-matching)','daily-tarot'); ?></option>
                            <?php foreach ((array)$packs as $p) :
                                $system = Cards::normalizeSystem((string)get_post_meta((int)$p->ID, '_dtarot_system', true));
                                $label = ($systems[$system] ?? $system);
                                $title = $p->post_title ?: __('(no title)','daily-tarot');
                            ?>
                                <option value="<?php echo (int)$p->ID; ?>" <?php selected((int)$s['pack_id'], (int)$p->ID); ?>><?php echo esc_html('[' . $label . '] ' . $title); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php esc_html_e('If Auto, a meaning pack matching the deck\'s system is chosen.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Email','daily-tarot'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="send_email" value="1" <?php checked($s['send_email']); ?> />
                            <?php esc_html_e('Send an email after publishing','daily-tarot'); ?>
                        </label>
                        <p>
                            <label>
                                <input type="checkbox" name="email_html" value="1" <?php checked(!empty($s['email_html'])); ?> />
                                <?php esc_html_e('Send as HTML (recommended)','daily-tarot'); ?>
                            </label>
                        </p>
                        <p>
                            <textarea name="recipients" rows="3" style="width:520px; max-width:100%;" placeholder="name@example.com, other@example.com"><?php echo esc_textarea($s['recipients']); ?></textarea>
                        </p>
                        <p class="description"><?php esc_html_e('Comma/space-separated recipient emails. Uses WordPress email configuration.','daily-tarot'); ?></p>

                        <p>
                            <label for="dtarot-email-subject"><strong><?php esc_html_e('Email subject template','daily-tarot'); ?></strong></label><br />
                            <input id="dtarot-email-subject" name="email_subject" type="text" class="regular-text" style="width:520px; max-width:100%;" value="<?php echo esc_attr((string)$s['email_subject']); ?>" placeholder="Card of the Day — {date}: {card}" />
                        </p>

                        <p>
                            <label for="dtarot-email-body"><strong><?php esc_html_e('Email body template','daily-tarot'); ?></strong></label><br />
                            <textarea id="dtarot-email-body" name="email_body" rows="8" style="width:520px; max-width:100%;" placeholder="<p><strong>{site}</strong></p>
<p>{date} — {card}</p>
{content}
{daily_text}
<p>{url}</p>"><?php echo esc_textarea((string)$s['email_body']); ?></textarea>
                        </p>

                        <p class="description">
                            <?php esc_html_e('Placeholders: {site}, {date}, {card}, {card_id}, {deck}, {url}, {content}, {daily_text}.','daily-tarot'); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button(__('Save automation settings','daily-tarot')); ?>
        </form>

        <hr />

        <h3><?php esc_html_e('Run now','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('Publishes today if it is not already published.','daily-tarot'); ?></p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('dtarot_automation'); ?>
            <input type="hidden" name="action" value="dtarot_automation_run_now" />
            <button type="submit" class="button"><?php esc_html_e('Run job now','daily-tarot'); ?></button>
        </form>

        <h3><?php esc_html_e('Publish for date (QA)','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('Manually publishes the chosen date if it is not already published (does not send email).','daily-tarot'); ?></p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('dtarot_automation'); ?>
            <input type="hidden" name="action" value="dtarot_automation_publish_for_date" />
            <label>
                <?php esc_html_e('Date','daily-tarot'); ?>
                <input type="date" name="date" value="<?php echo esc_attr($tomorrow); ?>" />
            </label>
            <button type="submit" class="button"><?php esc_html_e('Publish for date','daily-tarot'); ?></button>
        </form>

        <h3><?php esc_html_e('Email preview','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('Renders a preview using the chosen date and your current settings (does not send).','daily-tarot'); ?></p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('dtarot_automation'); ?>
            <input type="hidden" name="action" value="dtarot_automation_email_preview" />
            <label style="margin-right:8px;">
                <?php esc_html_e('Date','daily-tarot'); ?>
                <input type="date" name="date" value="<?php echo esc_attr($tomorrow); ?>" />
            </label>
            <button type="submit" class="button"><?php esc_html_e('Generate preview','daily-tarot'); ?></button>
        </form>

        <h3><?php esc_html_e('Send test email','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('Sends a single email to the address below using the chosen date (uses your template settings).','daily-tarot'); ?></p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('dtarot_automation'); ?>
            <input type="hidden" name="action" value="dtarot_automation_test_email" />
            <label style="margin-right:8px;">
                <?php esc_html_e('Date','daily-tarot'); ?>
                <input type="date" name="date" value="<?php echo esc_attr($tomorrow); ?>" />
            </label>
            <label style="margin-right:8px;">
                <?php esc_html_e('To','daily-tarot'); ?>
                <input type="email" name="to" value="<?php echo esc_attr($meEmail); ?>" placeholder="name@example.com" style="min-width:260px;" />
            </label>
            <button type="submit" class="button"><?php esc_html_e('Send test email','daily-tarot'); ?></button>
        </form>

        <?php
            $previewKey = 'dtarot_email_preview_' . (function_exists('get_current_user_id') ? (string)get_current_user_id() : '0');
            $preview = get_transient($previewKey);
            if (is_array($preview) && isset($preview['subject'], $preview['html'], $preview['text'])) {
                echo '<div style="margin-top:12px; max-width: 1100px;">';
                echo '<p><strong>' . esc_html__('Subject','daily-tarot') . ':</strong> ' . esc_html((string)$preview['subject']) . '</p>';
                if (isset($preview['date'])) {
                    echo '<p class="description" style="margin-top:0;">' . esc_html(sprintf(__('Preview date: %s','daily-tarot'), (string)$preview['date'])) . '</p>';
                }
                echo '<div style="background:#fff; border:1px solid #ccd0d4; padding:12px;">' . wp_kses_post((string)$preview['html']) . '</div>';
                echo '<p style="margin-top:12px;"><strong>' . esc_html__('Plain text','daily-tarot') . '</strong></p>';
                echo '<pre style="white-space:pre-wrap; background:#fff; border:1px solid #ccd0d4; padding:12px;">' . esc_html((string)$preview['text']) . '</pre>';
                echo '</div>';
            }
        ?>

        <h3><?php esc_html_e('Schedule','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('If the job shows as “Not scheduled”, you can force a reschedule (uses your configured time).','daily-tarot'); ?></p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('dtarot_automation'); ?>
            <input type="hidden" name="action" value="dtarot_automation_reschedule" />
            <button type="submit" class="button"><?php esc_html_e('Reschedule now','daily-tarot'); ?></button>
        </form>
        <?php
    }

    private static function renderCtaTab(): void {
        if (!class_exists(EmailCtaSettings::class)) {
            echo '<p>' . esc_html__('Email CTA settings are unavailable (missing plugin files).','daily-tarot') . '</p>';
            return;
        }

        $s = EmailCtaSettings::get();
        $msg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
        ?>
        <h2><?php esc_html_e('Email CTA (Signup form)','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Configure a signup form provider. The shortcode [dtarot_email_cta] will use these settings automatically when its action attribute is not provided.','daily-tarot'); ?></p>

        <?php if ($msg === 'email_cta_saved'): ?>
            <div class="notice notice-success is-dismissible"><p><?php esc_html_e('Email CTA settings saved.','daily-tarot'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width:900px;">
            <?php wp_nonce_field('dtarot_email_cta_save'); ?>
            <input type="hidden" name="action" value="dtarot_email_cta_save" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="dtarot_email_cta_provider"><?php esc_html_e('Provider','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_email_cta_provider" name="dtarot_email_cta_provider">
                            <option value="" <?php selected((string)$s['provider'], ''); ?>><?php esc_html_e('None (disabled)','daily-tarot'); ?></option>
                            <option value="mailchimp" <?php selected((string)$s['provider'], 'mailchimp'); ?>><?php esc_html_e('Mailchimp','daily-tarot'); ?></option>
                            <option value="wp" <?php selected((string)$s['provider'], 'wp'); ?>><?php esc_html_e('WordPress (wp_mail)','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('This config is used as the default for the [dtarot_email_cta] shortcode.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_email_cta_action_url"><?php esc_html_e('Mailchimp form action URL','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="url"
                            id="dtarot_email_cta_action_url"
                            name="dtarot_email_cta_action_url"
                            value="<?php echo esc_attr((string)($s['action_url'] ?? '')); ?>"
                            style="width: 520px; max-width: 100%;"
                            placeholder="https://xxx.list-manage.com/subscribe/post?u=...&amp;id=..."
                        />
                        <p class="description"><?php esc_html_e('Only used when provider is Mailchimp. Paste the form action from Mailchimp embed code (it looks like https://xxx.list-manage.com/subscribe/post?u=...&id=...).','daily-tarot'); ?></p>
                        <p class="description"><?php esc_html_e('Automation emails can also use a Mailchimp audience if you provide an API key and Audience ID below.','daily-tarot'); ?></p>

                        <details style="max-width: 900px; margin: 8px 0;">
                            <summary><?php esc_html_e('Where do I find this URL?','daily-tarot'); ?></summary>
                            <div class="description" style="margin-top: 8px;">
                                <ol style="margin: 0 0 0 18px;">
                                    <li><?php esc_html_e('In Mailchimp, open your Audience, then go to Signup forms / Embedded forms (wording varies by account).','daily-tarot'); ?></li>
                                    <li><?php esc_html_e('Copy the embedded form HTML code.','daily-tarot'); ?></li>
                                    <li><?php esc_html_e('Find the <form> tag and copy the value of its action attribute.','daily-tarot'); ?></li>
                                </ol>
                            </div>
                        </details>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_email_cta_honeypot_name"><?php esc_html_e('Honeypot field name (optional)','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="text"
                            id="dtarot_email_cta_honeypot_name"
                            name="dtarot_email_cta_honeypot_name"
                            value="<?php echo esc_attr((string)($s['honeypot_name'] ?? '')); ?>"
                            style="width: 420px; max-width: 100%;"
                            placeholder="b_xxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"
                        />
                        <p class="description"><?php esc_html_e('Optional anti-bot field name from the Mailchimp embed code. For WordPress, this adds a hidden field to reduce spam.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_email_cta_mailchimp_api_key"><?php esc_html_e('Mailchimp API key','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="password"
                            id="dtarot_email_cta_mailchimp_api_key"
                            name="dtarot_email_cta_mailchimp_api_key"
                            value=""
                            autocomplete="new-password"
                            style="width: 420px; max-width: 100%;"
                            placeholder="<?php echo esc_attr(__('Enter API key (stored, hidden)','daily-tarot')); ?>"
                        />
                        <p class="description">
                            <?php
                                echo !empty($s['has_mailchimp_api_key']) ? esc_html__('Saved (hidden).','daily-tarot') : esc_html__('Not set.','daily-tarot');
                            ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_email_cta_mailchimp_list_id"><?php esc_html_e('Mailchimp Audience ID','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="text"
                            id="dtarot_email_cta_mailchimp_list_id"
                            name="dtarot_email_cta_mailchimp_list_id"
                            value="<?php echo esc_attr((string)($s['mailchimp_list_id'] ?? '')); ?>"
                            style="width: 420px; max-width: 100%;"
                            placeholder="<?php echo esc_attr(__('e.g. a1b2c3d4e5','daily-tarot')); ?>"
                        />
                        <p class="description"><?php esc_html_e('Used by Automation emails to pull Mailchimp subscribers.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_email_cta_recipient"><?php esc_html_e('Recipient email (WP)','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="email"
                            id="dtarot_email_cta_recipient"
                            name="dtarot_email_cta_recipient"
                            value="<?php echo esc_attr((string)($s['recipient'] ?? '')); ?>"
                            style="width: 420px; max-width: 100%;"
                            placeholder="<?php echo esc_attr((string)get_option('admin_email', '')); ?>"
                        />
                        <p class="description"><?php esc_html_e('Only used when provider is WordPress (wp_mail). Defaults to the admin email if left blank.','daily-tarot'); ?></p>
                    </td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-primary"><?php esc_html_e('Save Email CTA','daily-tarot'); ?></button>
            </p>

            <p class="description" style="max-width: 900px;">
                <?php esc_html_e('Note: Automation can optionally use the stored CTA list and Mailchimp audience if enabled in AI Integration -> Automation.','daily-tarot'); ?>
            </p>

            <?php if (class_exists(EmailCtaStore::class)) :
                $rows = EmailCtaStore::getAll();
                $recent = array_slice(array_values($rows), -10);
            ?>
            <hr style="margin: 20px 0;">
            <h3><?php esc_html_e('Stored signups (WP)','daily-tarot'); ?></h3>
            <p class="description"><?php echo esc_html(sprintf(__('Stored emails: %d (max %d).','daily-tarot'), count($rows), 500)); ?></p>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-bottom: 10px;">
                <?php wp_nonce_field('dtarot_email_cta_export'); ?>
                <input type="hidden" name="action" value="dtarot_email_cta_export" />
                <button type="submit" class="button"><?php esc_html_e('Export CSV','daily-tarot'); ?></button>
            </form>
            <?php if (!$recent): ?>
                <p class="description"><?php esc_html_e('No stored signups yet.','daily-tarot'); ?></p>
            <?php else: ?>
                <table class="widefat striped" style="max-width: 900px;">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Email','daily-tarot'); ?></th>
                            <th><?php esc_html_e('First seen','daily-tarot'); ?></th>
                            <th><?php esc_html_e('Last seen','daily-tarot'); ?></th>
                            <th><?php esc_html_e('Count','daily-tarot'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent as $row): ?>
                            <tr>
                                <td><?php echo esc_html((string)$row['email']); ?></td>
                                <td><?php echo esc_html((string)$row['first_at']); ?></td>
                                <td><?php echo esc_html((string)$row['last_at']); ?></td>
                                <td><?php echo esc_html((string)$row['count']); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
            <?php endif; ?>
        </form>
        <?php
    }

    private static function renderShortcodeTab(): void {
        if (!class_exists(ShortcodeSettings::class)) {
            echo '<p>' . esc_html__('Shortcode settings are unavailable (missing plugin files).','daily-tarot') . '</p>';
            return;
        }

        $s = ShortcodeSettings::get();
        $shareImage = class_exists(ShareImageSettings::class) ? ShareImageSettings::get() : [];
        $related = class_exists(RelatedLinks::class) ? RelatedLinks::get() : ['enabled' => '0', 'post_type' => 'page', 'base_path' => '', 'map' => []];
        $msg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'date',
            'order' => 'DESC',
        ]);
        $systems = Cards::systems();
        ?>
        <h2><?php esc_html_e('Daily Tarot shortcode','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Set defaults for [daily_tarot] so you can control card placement and how much text is shown without editing each shortcode.','daily-tarot'); ?></p>

        <?php if ($msg === 'shortcode_saved'): ?>
            <div class="notice notice-success is-dismissible"><p><?php esc_html_e('Shortcode settings saved.','daily-tarot'); ?></p></div>
        <?php elseif ($msg === 'share_image_saved'): ?>
            <div class="notice notice-success is-dismissible"><p><?php esc_html_e('Share image settings saved.','daily-tarot'); ?></p></div>
        <?php elseif ($msg === 'related_links_saved'): ?>
            <div class="notice notice-success is-dismissible"><p><?php esc_html_e('Related link settings saved.','daily-tarot'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width:900px;">
            <?php wp_nonce_field('dtarot_shortcode_save'); ?>
            <input type="hidden" name="action" value="dtarot_shortcode_save" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="dtarot_shortcode_card_position"><?php esc_html_e('Card position','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_shortcode_card_position" name="dtarot_shortcode_card_position">
                            <option value="left" <?php selected((string)$s['card_position'], 'left'); ?>><?php esc_html_e('Left','daily-tarot'); ?></option>
                            <option value="right" <?php selected((string)$s['card_position'], 'right'); ?>><?php esc_html_e('Right','daily-tarot'); ?></option>
                            <option value="center" <?php selected((string)$s['card_position'], 'center'); ?>><?php esc_html_e('Center (stacked)','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Controls where the card image appears in the [daily_tarot] minimal layout.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_layout_default"><?php esc_html_e('Default layout','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_shortcode_layout_default" name="dtarot_shortcode_layout_default">
                            <option value="minimal" <?php selected((string)$s['layout_default'], 'minimal'); ?>><?php esc_html_e('Minimal','daily-tarot'); ?></option>
                            <option value="hero" <?php selected((string)$s['layout_default'], 'hero'); ?>><?php esc_html_e('Hero','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Used when [daily_tarot] has no layout attribute.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_show_share"><?php esc_html_e('Show share links','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_shortcode_show_share" name="dtarot_shortcode_show_share">
                            <option value="0" <?php selected((string)$s['show_share'], '0'); ?>><?php esc_html_e('No','daily-tarot'); ?></option>
                            <option value="1" <?php selected((string)$s['show_share'], '1'); ?>><?php esc_html_e('Yes','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Used when [daily_tarot] has no show_share attribute.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_shortcode_share_style"><?php esc_html_e('Share style','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_shortcode_share_style" name="dtarot_shortcode_share_style">
                            <option value="text" <?php selected((string)$s['share_style'], 'text'); ?>><?php esc_html_e('Text links','daily-tarot'); ?></option>
                            <option value="arcana-sigils" <?php selected((string)$s['share_style'], 'arcana-sigils'); ?>><?php esc_html_e('Arcana sigils','daily-tarot'); ?></option>
                            <option value="crystal-aura" <?php selected((string)$s['share_style'], 'crystal-aura'); ?>><?php esc_html_e('Crystal aura','daily-tarot'); ?></option>
                            <option value="tarot-card" <?php selected((string)$s['share_style'], 'tarot-card'); ?>><?php esc_html_e('Tarot card','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Applies when share links are enabled.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_theme_style"><?php esc_html_e('Theme style','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_shortcode_theme_style" name="dtarot_shortcode_theme_style">
                            <option value="minimal" <?php selected((string)$s['theme_style'], 'minimal'); ?>><?php esc_html_e('Minimal','daily-tarot'); ?></option>
                            <option value="mystic" <?php selected((string)$s['theme_style'], 'mystic'); ?>><?php esc_html_e('Mystic','daily-tarot'); ?></option>
                            <option value="modern" <?php selected((string)$s['theme_style'], 'modern'); ?>><?php esc_html_e('Modern','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Applies to [daily_tarot] minimal and hero layouts.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_text_mode"><?php esc_html_e('Text display','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_shortcode_text_mode" name="dtarot_shortcode_text_mode">
                            <option value="full" <?php selected((string)$s['text_mode'], 'full'); ?>><?php esc_html_e('Full text','daily-tarot'); ?></option>
                            <option value="excerpt" <?php selected((string)$s['text_mode'], 'excerpt'); ?>><?php esc_html_e('Excerpt only','daily-tarot'); ?></option>
                            <option value="read_more" <?php selected((string)$s['text_mode'], 'read_more'); ?>><?php esc_html_e('Excerpt + Read more','daily-tarot'); ?></option>
                            <option value="none" <?php selected((string)$s['text_mode'], 'none'); ?>><?php esc_html_e('No text','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Full text uses Daily Text or meaning pack fallback. Excerpt trims the text by word count.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_excerpt_words"><?php esc_html_e('Excerpt length (words)','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="number"
                            id="dtarot_shortcode_excerpt_words"
                            name="dtarot_shortcode_excerpt_words"
                            value="<?php echo esc_attr((string)($s['excerpt_words'] ?? 40)); ?>"
                            min="10"
                            max="200"
                            step="1"
                        />
                        <p class="description"><?php esc_html_e('Used for Excerpt and Read more modes.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_read_more_label"><?php esc_html_e('Read more label','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="text"
                            id="dtarot_shortcode_read_more_label"
                            name="dtarot_shortcode_read_more_label"
                            value="<?php echo esc_attr((string)($s['read_more_label'] ?? 'Read more')); ?>"
                            style="width: 260px; max-width: 100%;"
                        />
                        <p class="description"><?php esc_html_e('Shown when Text display is set to Excerpt + Read more.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_empty_card_mode"><?php esc_html_e('Empty day card back','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_shortcode_empty_card_mode" name="dtarot_shortcode_empty_card_mode">
                            <option value="random" <?php selected((string)($s['empty_card_mode'] ?? 'random'), 'random'); ?>><?php esc_html_e('Random deck','daily-tarot'); ?></option>
                            <option value="default" <?php selected((string)($s['empty_card_mode'] ?? 'random'), 'default'); ?>><?php esc_html_e('Default deck','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Shows a card back when no card is published for the day. Random picks a deck at random; Default uses the deck below.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_empty_deck_id"><?php esc_html_e('Default empty deck','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_shortcode_empty_deck_id" name="dtarot_shortcode_empty_deck_id">
                            <option value="0"><?php esc_html_e('Auto (most recent)','daily-tarot'); ?></option>
                            <?php foreach ((array)$decks as $d) :
                                $system = Cards::normalizeSystem((string)get_post_meta((int)$d->ID, '_dtarot_system', true));
                                $label = ($systems[$system] ?? $system);
                                $title = $d->post_title ?: __('(no title)','daily-tarot');
                            ?>
                                <option value="<?php echo (int)$d->ID; ?>" <?php selected((int)($s['empty_deck_id'] ?? 0), (int)$d->ID); ?>><?php echo esc_html('[' . $label . '] ' . $title); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php esc_html_e('Used when Empty day card back is set to Default.','daily-tarot'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="dtarot_shortcode_empty_text"><?php esc_html_e('Empty day text','daily-tarot'); ?></label></th>
                    <td>
                        <textarea
                            id="dtarot_shortcode_empty_text"
                            name="dtarot_shortcode_empty_text"
                            rows="4"
                            style="width: 100%; max-width: 520px;"
                        ><?php echo esc_textarea((string)($s['empty_text'] ?? '')); ?></textarea>
                        <p class="description"><?php esc_html_e('Shown when no card is published for the selected day.','daily-tarot'); ?></p>
                    </td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-primary"><?php esc_html_e('Save Shortcode Settings','daily-tarot'); ?></button>
            </p>
        </form>

        <hr />

        <h3><?php esc_html_e('Related links (internal linking)','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('Optionally link each card to a WordPress post/page (great for SEO). You can auto-detect candidates by slug and then choose the right one, or set a link manually.','daily-tarot'); ?></p>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width:900px;">
            <?php wp_nonce_field('dtarot_related_links_save'); ?>
            <input type="hidden" name="action" value="dtarot_related_links_save" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('Enable related links','daily-tarot'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="dtarot_related_links_enabled" value="1" <?php checked((string)($related['enabled'] ?? '0'), '1'); ?> />
                            <?php esc_html_e('Show a “Learn more” link under the daily reading when a match is found.','daily-tarot'); ?>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_related_links_post_type"><?php esc_html_e('Link to','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_related_links_post_type" name="dtarot_related_links_post_type">
                            <option value="page" <?php selected((string)($related['post_type'] ?? 'page'), 'page'); ?>><?php esc_html_e('Pages','daily-tarot'); ?></option>
                            <option value="post" <?php selected((string)($related['post_type'] ?? 'page'), 'post'); ?>><?php esc_html_e('Posts','daily-tarot'); ?></option>
                            <option value="any" <?php selected((string)($related['post_type'] ?? 'page'), 'any'); ?>><?php esc_html_e('Pages or Posts','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Used for auto-detection and search. Manual URL links always work.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_related_links_base_path"><?php esc_html_e('Auto-detect base path (optional)','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="text"
                            id="dtarot_related_links_base_path"
                            name="dtarot_related_links_base_path"
                            value="<?php echo esc_attr((string)($related['base_path'] ?? '')); ?>"
                            style="width: 320px; max-width: 100%;"
                            placeholder="tarot-cards"
                        />
                        <p class="description"><?php esc_html_e('If set, we will first try to match pages like /base-path/the-fool/. Leave blank to match by slug anywhere.','daily-tarot'); ?></p>
                    </td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-secondary"><?php esc_html_e('Save Related Link Settings','daily-tarot'); ?></button>
            </p>
        </form>

        <div id="dtarot-related-links-manager" style="max-width:900px; margin-top: 10px;">
            <h4 style="margin: 14px 0 6px; "><?php esc_html_e('Choose a link for a card','daily-tarot'); ?></h4>

            <div style="display:flex; gap:10px; flex-wrap:wrap; align-items:flex-end;">
                <div>
                    <label for="dtarot_related_links_card"><strong><?php esc_html_e('Card','daily-tarot'); ?></strong></label><br />
                    <select id="dtarot_related_links_card" style="min-width: 280px;">
                        <?php foreach (Cards::systems() as $sysKey => $sysLabel): ?>
                            <optgroup label="<?php echo esc_attr($sysLabel); ?>">
                                <?php foreach (Cards::forSystem((string)$sysKey) as $cid => $cname): ?>
                                    <option value="<?php echo esc_attr((string)$cid); ?>"><?php echo esc_html((string)$cname); ?></option>
                                <?php endforeach; ?>
                            </optgroup>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <button type="button" class="button" data-dtarot-related-action="suggest"><?php esc_html_e('Auto-detect options','daily-tarot'); ?></button>
                </div>

                <div>
                    <button type="button" class="button" data-dtarot-related-action="clear"><?php esc_html_e('Clear','daily-tarot'); ?></button>
                </div>
            </div>

            <div id="dtarot_related_links_status" style="margin-top: 10px;"></div>

            <div style="margin-top: 10px;">
                <strong><?php esc_html_e('Auto-detected matches','daily-tarot'); ?></strong>
                <div id="dtarot_related_links_suggestions" class="dtarot-related-links-list"></div>
            </div>

            <div style="margin-top: 14px;">
                <strong><?php esc_html_e('Search posts/pages','daily-tarot'); ?></strong>
                <div style="display:flex; gap:8px; align-items:center; flex-wrap:wrap; margin-top:6px;">
                    <input type="text" id="dtarot_related_links_search" placeholder="<?php echo esc_attr__('Search…','daily-tarot'); ?>" style="width: 320px; max-width: 100%;" />
                    <button type="button" class="button" data-dtarot-related-action="search"><?php esc_html_e('Search','daily-tarot'); ?></button>
                </div>
                <div id="dtarot_related_links_search_results" class="dtarot-related-links-list"></div>
            </div>

            <div style="margin-top: 14px;">
                <strong><?php esc_html_e('Manual URL','daily-tarot'); ?></strong>
                <div style="display:flex; gap:8px; align-items:center; flex-wrap:wrap; margin-top:6px;">
                    <input type="url" id="dtarot_related_links_url" placeholder="https://example.com/the-fool" style="width: 420px; max-width: 100%;" />
                    <button type="button" class="button" data-dtarot-related-action="set-url"><?php esc_html_e('Use URL','daily-tarot'); ?></button>
                </div>
                <p class="description" style="margin-top:6px;">
                    <?php esc_html_e('Tip: use this if your meaning pages live in a custom structure or another post type.','daily-tarot'); ?>
                </p>
            </div>
        </div>

        <h3><?php esc_html_e('Instagram share image','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('Configure the popup image generator for Instagram sharing. This creates a downloadable square image on the frontend.','daily-tarot'); ?></p>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width:900px;">
            <?php wp_nonce_field('dtarot_share_image_save'); ?>
            <input type="hidden" name="action" value="dtarot_share_image_save" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('Enable Instagram popup','daily-tarot'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="dtarot_share_image_enabled" value="1" <?php checked(!empty($shareImage['enabled'])); ?> />
                            <?php esc_html_e('Enable image popup for Instagram share button','daily-tarot'); ?>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_style"><?php esc_html_e('Default theme style','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_share_image_style" name="dtarot_share_image_style">
                            <option value="minimal" <?php selected((string)($shareImage['theme']['default_style'] ?? ''), 'minimal'); ?>><?php esc_html_e('Minimal','daily-tarot'); ?></option>
                            <option value="mystic" <?php selected((string)($shareImage['theme']['default_style'] ?? ''), 'mystic'); ?>><?php esc_html_e('Mystic','daily-tarot'); ?></option>
                            <option value="modern" <?php selected((string)($shareImage['theme']['default_style'] ?? ''), 'modern'); ?>><?php esc_html_e('Modern','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Controls the default color palette used in the generated image.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_layout"><?php esc_html_e('Popup layout','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_share_image_layout" name="dtarot_share_image_layout">
                            <option value="left_preview" <?php selected((string)($shareImage['popup']['layout'] ?? ''), 'left_preview'); ?>><?php esc_html_e('Preview left','daily-tarot'); ?></option>
                            <option value="top_preview" <?php selected((string)($shareImage['popup']['layout'] ?? ''), 'top_preview'); ?>><?php esc_html_e('Preview top','daily-tarot'); ?></option>
                            <option value="full_preview" <?php selected((string)($shareImage['popup']['layout'] ?? ''), 'full_preview'); ?>><?php esc_html_e('Preview only (controls below)','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Choose how the preview and controls are arranged in the popup.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_size"><?php esc_html_e('Popup size','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_share_image_size" name="dtarot_share_image_size">
                            <option value="sm" <?php selected((string)($shareImage['popup']['size'] ?? ''), 'sm'); ?>><?php esc_html_e('Small','daily-tarot'); ?></option>
                            <option value="md" <?php selected((string)($shareImage['popup']['size'] ?? ''), 'md'); ?>><?php esc_html_e('Medium','daily-tarot'); ?></option>
                            <option value="lg" <?php selected((string)($shareImage['popup']['size'] ?? ''), 'lg'); ?>><?php esc_html_e('Large','daily-tarot'); ?></option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Caption visibility','daily-tarot'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="dtarot_share_image_show_caption" value="1" <?php checked(!empty($shareImage['popup']['show_caption'])); ?> />
                            <?php esc_html_e('Show caption copy box by default','daily-tarot'); ?>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_branding_mode"><?php esc_html_e('Branding mode','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_share_image_branding_mode" name="dtarot_share_image_branding_mode">
                            <option value="optional" <?php selected((string)($shareImage['branding']['mode'] ?? ''), 'optional'); ?>><?php esc_html_e('Optional','daily-tarot'); ?></option>
                            <option value="required" <?php selected((string)($shareImage['branding']['mode'] ?? ''), 'required'); ?>><?php esc_html_e('Required','daily-tarot'); ?></option>
                            <option value="disabled" <?php selected((string)($shareImage['branding']['mode'] ?? ''), 'disabled'); ?>><?php esc_html_e('Disabled','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Controls whether branding can be toggled in the popup.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_branding_text"><?php esc_html_e('Branding text','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="text"
                            id="dtarot_share_image_branding_text"
                            name="dtarot_share_image_branding_text"
                            value="<?php echo esc_attr((string)($shareImage['branding']['text'] ?? '')); ?>"
                            style="width: 420px; max-width: 100%;"
                        />
                        <p class="description"><?php esc_html_e('Displayed in the image footer when branding is enabled.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_branding_logo"><?php esc_html_e('Branding logo URL','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="url"
                            id="dtarot_share_image_branding_logo"
                            name="dtarot_share_image_branding_logo"
                            value="<?php echo esc_attr((string)($shareImage['branding']['logo_url'] ?? '')); ?>"
                            style="width: 520px; max-width: 100%;"
                            placeholder="https://example.com/logo.png"
                        />
                        <p class="description"><?php esc_html_e('Optional logo to show next to branding text.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('QR code','daily-tarot'); ?></th>
                    <td>
                        <label style="margin-right:16px;">
                            <input type="checkbox" name="dtarot_share_image_qr_enabled" value="1" <?php checked(!empty($shareImage['qr']['enabled'])); ?> />
                            <?php esc_html_e('Enable QR in image','daily-tarot'); ?>
                        </label>
                        <label>
                            <input type="checkbox" name="dtarot_share_image_qr_toggle" value="1" <?php checked(!empty($shareImage['qr']['allow_toggle'])); ?> />
                            <?php esc_html_e('Allow users to toggle QR','daily-tarot'); ?>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_qr_dest"><?php esc_html_e('QR destination','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_share_image_qr_dest" name="dtarot_share_image_qr_dest">
                            <option value="daily_readable" <?php selected((string)($shareImage['qr']['dest_rule'] ?? ''), 'daily_readable'); ?>><?php esc_html_e('Daily readable URL','daily-tarot'); ?></option>
                            <option value="current_page" <?php selected((string)($shareImage['qr']['dest_rule'] ?? ''), 'current_page'); ?>><?php esc_html_e('Current page URL','daily-tarot'); ?></option>
                            <option value="custom" <?php selected((string)($shareImage['qr']['dest_rule'] ?? ''), 'custom'); ?>><?php esc_html_e('Custom URL','daily-tarot'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Controls the URL encoded in the QR.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_qr_custom"><?php esc_html_e('Custom QR URL','daily-tarot'); ?></label></th>
                    <td>
                        <input
                            type="url"
                            id="dtarot_share_image_qr_custom"
                            name="dtarot_share_image_qr_custom"
                            value="<?php echo esc_attr((string)($shareImage['qr']['custom_url'] ?? '')); ?>"
                            style="width: 520px; max-width: 100%;"
                            placeholder="https://example.com"
                        />
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Text layer','daily-tarot'); ?></th>
                    <td>
                        <label style="margin-right:16px;">
                            <input type="checkbox" name="dtarot_share_image_text_toggle" value="1" <?php checked(!empty($shareImage['text']['allow_toggle'])); ?> />
                            <?php esc_html_e('Allow users to toggle text','daily-tarot'); ?>
                        </label>
                        <label>
                            <?php esc_html_e('Max characters:','daily-tarot'); ?>
                            <input type="number" name="dtarot_share_image_text_max" value="<?php echo esc_attr((string)($shareImage['text']['max_chars'] ?? 220)); ?>" min="40" max="500" style="width:80px;" />
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_resolution"><?php esc_html_e('Export resolution','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_share_image_resolution" name="dtarot_share_image_resolution">
                            <option value="1080" <?php selected((string)($shareImage['export']['resolution'] ?? ''), '1080'); ?>>1080</option>
                            <option value="1440" <?php selected((string)($shareImage['export']['resolution'] ?? ''), '1440'); ?>>1440</option>
                            <option value="2048" <?php selected((string)($shareImage['export']['resolution'] ?? ''), '2048'); ?>>2048</option>
                        </select>
                        <p class="description"><?php esc_html_e('Square output size in pixels.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="dtarot_share_image_format"><?php esc_html_e('Export format','daily-tarot'); ?></label></th>
                    <td>
                        <select id="dtarot_share_image_format" name="dtarot_share_image_format">
                            <option value="png" <?php selected((string)($shareImage['export']['format'] ?? ''), 'png'); ?>>PNG</option>
                            <option value="webp" <?php selected((string)($shareImage['export']['format'] ?? ''), 'webp'); ?>>WebP</option>
                        </select>
                        <label style="margin-left:16px;">
                            <?php esc_html_e('WebP quality:','daily-tarot'); ?>
                            <input type="number" name="dtarot_share_image_quality" value="<?php echo esc_attr((string)($shareImage['export']['quality'] ?? 92)); ?>" min="40" max="100" style="width:80px;" />
                        </label>
                    </td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-primary"><?php esc_html_e('Save Share Image Settings','daily-tarot'); ?></button>
            </p>
        </form>
        <?php
    }

    private static function renderBookingsTab(): void {
        if (!class_exists(BookingSettings::class)) {
            echo '<p>' . esc_html__('Booking settings are unavailable (missing plugin files).','daily-tarot') . '</p>';
            return;
        }

        $s = BookingSettings::get();
        $notice = isset($_GET['dtarot_booking']) ? sanitize_key((string)wp_unslash($_GET['dtarot_booking'])) : '';
        if ($notice === 'saved') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Booking settings saved.','daily-tarot') . '</p></div>';
        }

        $weekdays = isset($s['weekdays']) && is_array($s['weekdays']) ? $s['weekdays'] : [];
        $tz = isset($s['timezone']) ? (string)$s['timezone'] : '';
        $templates = isset($s['templates']) && is_array($s['templates']) ? $s['templates'] : [];
        ?>
        <h2><?php esc_html_e('Bookings','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Control how bookings are confirmed and when you are available.','daily-tarot'); ?></p>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width:980px;">
            <?php wp_nonce_field('dtarot_booking_settings'); ?>
            <input type="hidden" name="action" value="dtarot_booking_settings_save" />

            <h3><?php esc_html_e('Booking mode','daily-tarot'); ?></h3>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('Mode','daily-tarot'); ?></th>
                    <td>
                        <select name="booking_mode">
                            <option value="request" <?php selected((string)$s['mode'], 'request'); ?>><?php esc_html_e('Request Only (manual confirmation)','daily-tarot'); ?></option>
                            <option value="instant" <?php selected((string)$s['mode'], 'instant'); ?>><?php esc_html_e('Instant Booking (auto-confirm)','daily-tarot'); ?></option>
                        </select>
                    </td>
                </tr>
            </table>

        <h3><?php esc_html_e('Availability','daily-tarot'); ?></h3>
        <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('Available days','daily-tarot'); ?></th>
                    <td>
                        <?php
                        $days = [
                            'mon' => __('Mon','daily-tarot'),
                            'tue' => __('Tue','daily-tarot'),
                            'wed' => __('Wed','daily-tarot'),
                            'thu' => __('Thu','daily-tarot'),
                            'fri' => __('Fri','daily-tarot'),
                            'sat' => __('Sat','daily-tarot'),
                            'sun' => __('Sun','daily-tarot'),
                        ];
                        foreach ($days as $key => $label) {
                            $checked = in_array($key, $weekdays, true);
                            echo '<label style="margin-right:12px;"><input type="checkbox" name="weekdays[]" value="' . esc_attr($key) . '" ' . checked($checked, true, false) . ' /> ' . esc_html($label) . '</label>';
                        }
                        ?>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Daily window','daily-tarot'); ?></th>
                    <td>
                        <input type="time" name="start_time" value="<?php echo esc_attr((string)$s['start_time']); ?>" />
                        <span style="margin:0 6px;"><?php esc_html_e('to','daily-tarot'); ?></span>
                        <input type="time" name="end_time" value="<?php echo esc_attr((string)$s['end_time']); ?>" />
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Buffer time (minutes)','daily-tarot'); ?></th>
                    <td>
                        <label><?php esc_html_e('Before','daily-tarot'); ?>
                            <input type="number" min="0" step="5" name="buffer_before" value="<?php echo esc_attr((string)$s['buffer_before']); ?>" class="small-text" />
                        </label>
                        <span style="margin:0 10px;"></span>
                        <label><?php esc_html_e('After','daily-tarot'); ?>
                            <input type="number" min="0" step="5" name="buffer_after" value="<?php echo esc_attr((string)$s['buffer_after']); ?>" class="small-text" />
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Timezone','daily-tarot'); ?></th>
                    <td>
                        <input type="text" name="timezone" value="<?php echo esc_attr($tz); ?>" class="regular-text" placeholder="<?php echo esc_attr((string)get_option('timezone_string', 'UTC')); ?>" />
                        <p class="description"><?php esc_html_e('Leave empty to use the site timezone.','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Blackout dates','daily-tarot'); ?></th>
                    <td>
                        <textarea name="blackout_dates" rows="3" class="large-text" placeholder="2025-12-24"><?php echo esc_textarea((string)$s['blackout_dates']); ?></textarea>
                        <p class="description"><?php esc_html_e('One date per line (YYYY-MM-DD).','daily-tarot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Slot interval (minutes)','daily-tarot'); ?></th>
                    <td>
                        <input type="number" min="5" step="5" name="slot_interval" value="<?php echo esc_attr((string)$s['slot_interval']); ?>" class="small-text" />
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Booking form style','daily-tarot'); ?></th>
                    <td>
                        <select name="style">
                            <option value="modern" <?php selected((string)$s['style'], 'modern'); ?>><?php esc_html_e('Modern','daily-tarot'); ?></option>
                            <option value="mystic" <?php selected((string)$s['style'], 'mystic'); ?>><?php esc_html_e('Mystic','daily-tarot'); ?></option>
                            <option value="minimal" <?php selected((string)$s['style'], 'minimal'); ?>><?php esc_html_e('Minimal','daily-tarot'); ?></option>
                        </select>
                    </td>
                </tr>
            </table>

            <h3><?php esc_html_e('Email defaults','daily-tarot'); ?></h3>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('Admin email','daily-tarot'); ?></th>
                    <td><input type="email" name="admin_email" value="<?php echo esc_attr((string)$s['admin_email']); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('From name','daily-tarot'); ?></th>
                    <td><input type="text" name="from_name" value="<?php echo esc_attr((string)$s['from_name']); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('From email','daily-tarot'); ?></th>
                    <td><input type="email" name="from_email" value="<?php echo esc_attr((string)$s['from_email']); ?>" class="regular-text" /></td>
                </tr>
            </table>

            <h3><?php esc_html_e('Email templates','daily-tarot'); ?></h3>
            <p class="description"><?php esc_html_e('Placeholders: {name}, {email}, {reading}, {date}, {time}, {timezone}, {status}, {question}, {admin_date}, {admin_time}, {admin_timezone}, {site}.','daily-tarot'); ?></p>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('User request subject','daily-tarot'); ?></th>
                    <td><input type="text" name="user_request_subject" value="<?php echo esc_attr((string)($templates['user_request_subject'] ?? '')); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('User request body','daily-tarot'); ?></th>
                    <td><textarea name="user_request_body" rows="4" class="large-text"><?php echo esc_textarea((string)($templates['user_request_body'] ?? '')); ?></textarea></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('User confirmed subject','daily-tarot'); ?></th>
                    <td><input type="text" name="user_confirmed_subject" value="<?php echo esc_attr((string)($templates['user_confirmed_subject'] ?? '')); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('User confirmed body','daily-tarot'); ?></th>
                    <td><textarea name="user_confirmed_body" rows="4" class="large-text"><?php echo esc_textarea((string)($templates['user_confirmed_body'] ?? '')); ?></textarea></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('User declined subject','daily-tarot'); ?></th>
                    <td><input type="text" name="user_declined_subject" value="<?php echo esc_attr((string)($templates['user_declined_subject'] ?? '')); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('User declined body','daily-tarot'); ?></th>
                    <td><textarea name="user_declined_body" rows="4" class="large-text"><?php echo esc_textarea((string)($templates['user_declined_body'] ?? '')); ?></textarea></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Admin notification subject','daily-tarot'); ?></th>
                    <td><input type="text" name="admin_new_subject" value="<?php echo esc_attr((string)($templates['admin_new_subject'] ?? '')); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Admin notification body','daily-tarot'); ?></th>
                    <td><textarea name="admin_new_body" rows="4" class="large-text"><?php echo esc_textarea((string)($templates['admin_new_body'] ?? '')); ?></textarea></td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-primary"><?php esc_html_e('Save Booking Settings','daily-tarot'); ?></button>
            </p>
        </form>
        <?php
    }

    private static function renderHelpTab(): void {
        $paths = [
            DTAROT_PATH . 'ADDON_README.md',
            DTAROT_PATH . 'README.md',
        ];

        $path = '';
        foreach ($paths as $candidate) {
            if (is_readable($candidate)) {
                $path = $candidate;
                break;
            }
        }

        if ($path === '') {
            echo '<p>' . esc_html__('Readme file not found.','daily-tarot') . '</p>';
            return;
        }

        $raw = (string)file_get_contents($path);
        $html = self::renderMarkdownToHtml($raw);
        ?>
        <h2><?php esc_html_e('Help & Readme','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('This is the bundled user guide for the Daily Tarot plugin.','daily-tarot'); ?></p>
        <div class="dtarot-readme-intro">
            <h3><?php esc_html_e('What this plugin does','daily-tarot'); ?></h3>
            <p><?php esc_html_e('Daily Tarot helps you publish a daily card with consistent meanings, images, and optional automation. Use the calendar to schedule days, attach decks and meaning packs, and display the result anywhere with shortcodes or readable routes.','daily-tarot'); ?></p>
            <ul>
                <li><?php esc_html_e('Daily Calendar: set the card, deck, and meaning pack per day.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Meaning Packs: manage structured meanings for each card.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Decks: upload images and backs for each card system.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Shortcodes and Embeds: show the daily card or a single card anywhere.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Automation and Analytics: optional helpers for workflows and insights.','daily-tarot'); ?></li>
            </ul>
            <h3><?php esc_html_e('Quick start','daily-tarot'); ?></h3>
            <ol>
                <li><?php esc_html_e('Add or import a deck and confirm each card has an image and a back.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Open the Calendar and choose a card for each day you want to publish.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Pick a meaning pack (or keep the default) for consistent interpretations.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Place the shortcode on a page and publish.','daily-tarot'); ?></li>
            </ol>
            <h3><?php esc_html_e('Where to edit content','daily-tarot'); ?></h3>
            <ul>
                <li><?php esc_html_e('Calendar: daily selections and status.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Meaning Packs: meanings, keywords, and long text.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Decks: images, backs, and card systems.','daily-tarot'); ?></li>
                <li><?php esc_html_e('Settings: shortcode options and display rules.','daily-tarot'); ?></li>
            </ul>
        </div>
        <div class="dtarot-readme">
            <?php echo wp_kses_post($html); ?>
        </div>
        <?php
    }

    private static function renderMarkdownToHtml(string $md): string {
        $md = str_replace("\r\n", "\n", $md);
        $lines = explode("\n", $md);

        $html = '';
        $inCode = false;
        $listOpen = false;
        $para = '';

        $flushPara = static function() use (&$para, &$html): void {
            $text = trim($para);
            if ($text !== '') {
                $html .= '<p>' . esc_html($text) . '</p>';
            }
            $para = '';
        };

        $closeList = static function() use (&$listOpen, &$html): void {
            if ($listOpen) {
                $html .= '</ul>';
                $listOpen = false;
            }
        };

        foreach ($lines as $line) {
            if (preg_match('/^```/', $line)) {
                $flushPara();
                $closeList();
                if (!$inCode) {
                    $inCode = true;
                    $html .= '<pre class="dtarot-readme-code"><code>';
                } else {
                    $html .= '</code></pre>';
                    $inCode = false;
                }
                continue;
            }

            if ($inCode) {
                $html .= esc_html($line) . "\n";
                continue;
            }

            if (preg_match('/^(#{1,3})\s*(.+)$/', $line, $m)) {
                $flushPara();
                $closeList();
                $level = strlen($m[1]) + 1;
                $level = max(2, min(4, $level));
                $html .= '<h' . (int)$level . '>' . esc_html(trim($m[2])) . '</h' . (int)$level . '>';
                continue;
            }

            if (preg_match('/^\s*-\s+(.+)$/', $line, $m)) {
                $flushPara();
                if (!$listOpen) {
                    $html .= '<ul>';
                    $listOpen = true;
                }
                $html .= '<li>' . esc_html(trim($m[1])) . '</li>';
                continue;
            }

            if (trim($line) === '') {
                $flushPara();
                $closeList();
                continue;
            }

            $para .= ($para === '' ? '' : ' ') . trim($line);
        }

        $flushPara();
        $closeList();

        if ($inCode) {
            $html .= '</code></pre>';
        }

        return $html;
    }

    private static function renderDiagnosticsTab(): void {
        $uiNotice = isset($_GET['dtarot_ui']) ? sanitize_key((string)wp_unslash($_GET['dtarot_ui'])) : '';
        $showQuickActions = class_exists(UiSettings::class) ? UiSettings::showQuickActions() : true;

        $publicNotice = isset($_GET['dtarot_diag_public']) ? sanitize_key((string)wp_unslash($_GET['dtarot_diag_public'])) : '';
        $publicDiag = get_transient(self::DIAG_PUBLIC_TRANSIENT_PREFIX . (string)get_current_user_id());

        $phpVersion = PHP_VERSION;
        $wpVersion = function_exists('get_bloginfo') ? (string)get_bloginfo('version') : '';

        $zipOk = class_exists('ZipArchive');

        $uploadMax = ini_get('upload_max_filesize');
        $postMax = ini_get('post_max_size');
        $memoryLimit = ini_get('memory_limit');
        $maxExec = ini_get('max_execution_time');

        $uploads = wp_upload_dir();
        $uploadPath = is_array($uploads) && isset($uploads['basedir']) ? (string)$uploads['basedir'] : '';
        $uploadWritable = $uploadPath !== '' ? wp_is_writable($uploadPath) : false;

        $tmpDir = function_exists('get_temp_dir') ? (string)get_temp_dir() : (string)sys_get_temp_dir();
        $tmpWritable = $tmpDir !== '' ? wp_is_writable($tmpDir) : false;

        $tmpCreateOk = false;
        if ($tmpWritable) {
            $p = trailingslashit($tmpDir) . 'dtarot_diag_' . wp_generate_password(12, false, false) . '.tmp';
            $tmpCreateOk = (bool)@file_put_contents($p, 'ok');
            if ($tmpCreateOk) {
                wp_delete_file($p);
            }
        }

        $fmtBool = static function(bool $ok): string {
            return $ok ? __('OK','daily-tarot') : __('Missing / Disabled','daily-tarot');
        };

        ?>
        <h2><?php esc_html_e('Admin UI','daily-tarot'); ?></h2>

        <?php if ($uiNotice === 'ok'): ?>
            <div class="notice notice-success is-dismissible"><p><?php esc_html_e('UI settings saved.','daily-tarot'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width:900px;">
            <?php wp_nonce_field('dtarot_ui_settings'); ?>
            <input type="hidden" name="action" value="dtarot_ui_save" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('Quick actions bar','daily-tarot'); ?></th>
                    <td>
                        <input type="hidden" name="show_quick_actions" id="dtarot_show_quick_actions_hidden" value="<?php echo $showQuickActions ? '1' : '0'; ?>" />
                        <label class="dtarot-toggle">
                            <span class="dtarot-switch">
                                <input type="checkbox" name="show_quick_actions_toggle" value="1" <?php checked($showQuickActions); ?> role="switch" data-dtarot-autosave="1" data-dtarot-setting="show_quick_actions" data-dtarot-target=".dtarot-quick-actions" data-dtarot-hidden-target="#dtarot_show_quick_actions_hidden" />
                                <span class="dtarot-switch-ui" aria-hidden="true"></span>
                            </span>
                            <span><?php esc_html_e('Show the Quick actions bar on admin pages','daily-tarot'); ?></span>
                        </label>
                        <span class="dtarot-auto-save-status" aria-live="polite" style="margin-left:10px;"></span>
                        <p class="description" style="margin-top:6px;">
                            <?php echo esc_html(sprintf(__('Current value: %s','daily-tarot'), $showQuickActions ? __('On','daily-tarot') : __('Off','daily-tarot'))); ?>
                        </p>
                        <p class="description"><?php esc_html_e('If disabled, the Quick actions card will be hidden across Daily Tarot admin pages.','daily-tarot'); ?></p>
                    </td>
                </tr>
            </table>

        </form>

        <hr />

        <h2><?php esc_html_e('Host diagnostics','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Use this to quickly confirm the server supports ZIP import/export.','daily-tarot'); ?></p>

        <table class="widefat striped" style="max-width:1100px;">
            <thead>
                <tr>
                    <th><?php esc_html_e('Check','daily-tarot'); ?></th>
                    <th><?php esc_html_e('Status','daily-tarot'); ?></th>
                    <th><?php esc_html_e('Details','daily-tarot'); ?></th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td><?php esc_html_e('PHP version','daily-tarot'); ?></td>
                    <td><?php echo esc_html($phpVersion); ?></td>
                    <td><?php esc_html_e('Plugin requires PHP 8.1+.','daily-tarot'); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('WordPress version','daily-tarot'); ?></td>
                    <td><?php echo esc_html($wpVersion); ?></td>
                    <td><?php esc_html_e('Plugin requires WordPress 6.0+.','daily-tarot'); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('ZipArchive extension','daily-tarot'); ?></td>
                    <td><?php echo esc_html($fmtBool($zipOk)); ?></td>
                    <td><?php esc_html_e('Required for ZIP import/export (packs).','daily-tarot'); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('Uploads directory writable','daily-tarot'); ?></td>
                    <td><?php echo esc_html($fmtBool($uploadWritable)); ?></td>
                    <td><?php echo esc_html($uploadPath !== '' ? $uploadPath : __('Unknown','daily-tarot')); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('Temp directory writable','daily-tarot'); ?></td>
                    <td><?php echo esc_html($fmtBool($tmpWritable)); ?></td>
                    <td><?php echo esc_html($tmpDir !== '' ? $tmpDir : __('Unknown','daily-tarot')); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('Can create temp files','daily-tarot'); ?></td>
                    <td><?php echo esc_html($fmtBool($tmpCreateOk)); ?></td>
                    <td><?php esc_html_e('Used for URL downloads and intermediate ZIP handling.','daily-tarot'); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('PHP limits','daily-tarot'); ?></td>
                    <td><?php esc_html_e('Info','daily-tarot'); ?></td>
                    <td>
                        <?php
                        echo esc_html(sprintf(
                            __('upload_max_filesize=%s; post_max_size=%s; memory_limit=%s; max_execution_time=%s','daily-tarot'),
                            (string)$uploadMax,
                            (string)$postMax,
                            (string)$memoryLimit,
                            (string)$maxExec
                        ));
                        ?>
                    </td>
                </tr>
            </tbody>
        </table>

        <p class="description" style="max-width:1100px;">
            <?php esc_html_e('If ZIP imports fail, confirm ZipArchive is enabled and that upload/post size limits are large enough for your pack files. If URL imports fail, confirm the temp directory is writable and that your host allows outbound HTTP requests.','daily-tarot'); ?>
        </p>

        <hr />

        <h2><?php esc_html_e('Self-check (rewrites + sitemap)','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Quick checks for readable routes and sitemap availability.','daily-tarot'); ?></p>

        <?php
            $permalinkStructure = (string)get_option('permalink_structure', '');
            $permalinksOn = $permalinkStructure !== '';
            $rules = get_option('rewrite_rules');
            $rulesStr = is_array($rules) ? implode("\n", array_keys($rules)) : '';
            $hasReadableRule = $rulesStr !== '' && (str_contains($rulesStr, 'dtarot_date') || str_contains($rulesStr, 'card-of-the-day'));
            $hasSitemapRule = $rulesStr !== '' && str_contains($rulesStr, 'dtarot-sitemap');
            $sitemapUrl = class_exists(Sitemap::class) ? Sitemap::url() : '';
        ?>

        <table class="widefat striped" style="max-width:1100px;">
            <thead>
                <tr>
                    <th><?php esc_html_e('Check','daily-tarot'); ?></th>
                    <th><?php esc_html_e('Status','daily-tarot'); ?></th>
                    <th><?php esc_html_e('Details','daily-tarot'); ?></th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td><?php esc_html_e('Permalinks enabled','daily-tarot'); ?></td>
                    <td><?php echo esc_html($fmtBool($permalinksOn)); ?></td>
                    <td><?php echo esc_html($permalinksOn ? $permalinkStructure : __('Readable routes require permalinks (not Plain).','daily-tarot')); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('Readable route rewrite present','daily-tarot'); ?></td>
                    <td><?php echo esc_html($fmtBool($hasReadableRule)); ?></td>
                    <td><?php esc_html_e('If missing, click “Refresh permalinks” when prompted after updates.','daily-tarot'); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('Sitemap rewrite present','daily-tarot'); ?></td>
                    <td><?php echo esc_html($fmtBool($hasSitemapRule)); ?></td>
                    <td><?php echo esc_html($sitemapUrl !== '' ? $sitemapUrl : __('Unknown','daily-tarot')); ?></td>
                </tr>
            </tbody>
        </table>

        <h2 style="margin-top:18px;"><?php esc_html_e('Fetch public endpoints','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Fetches robots.txt and the Daily Tarot sitemap through the public frontend and checks for common issues.','daily-tarot'); ?></p>

        <?php if ($publicNotice === 'ok'): ?>
            <div class="notice notice-success is-dismissible" style="max-width:1100px;"><p><?php esc_html_e('Public endpoint check completed.','daily-tarot'); ?></p></div>
        <?php elseif ($publicNotice === 'error'): ?>
            <div class="notice notice-error is-dismissible" style="max-width:1100px;"><p><?php esc_html_e('Public endpoint check failed.','daily-tarot'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin: 10px 0 14px;">
            <?php wp_nonce_field('dtarot_diagnostics_fetch_public'); ?>
            <input type="hidden" name="action" value="dtarot_diagnostics_fetch_public" />
            <button type="submit" class="button"><?php esc_html_e('Fetch robots.txt and sitemap','daily-tarot'); ?></button>
        </form>

        <?php if (is_array($publicDiag) && !empty($publicDiag['time'])):
            $robotsUrl = isset($publicDiag['robots_url']) ? (string)$publicDiag['robots_url'] : '';
            $robotsCode = isset($publicDiag['robots_code']) ? (int)$publicDiag['robots_code'] : 0;
            $robotsHasSitemap = !empty($publicDiag['robots_has_sitemap']);
            $sitemapUrl2 = isset($publicDiag['sitemap_url']) ? (string)$publicDiag['sitemap_url'] : '';
            $sitemapCode = isset($publicDiag['sitemap_code']) ? (int)$publicDiag['sitemap_code'] : 0;
            $sitemapLooksXml = !empty($publicDiag['sitemap_looks_xml']);
            $err = isset($publicDiag['error']) ? (string)$publicDiag['error'] : '';
        ?>
            <table class="widefat striped" style="max-width:1100px;">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Endpoint','daily-tarot'); ?></th>
                        <th><?php esc_html_e('HTTP','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Status','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Details','daily-tarot'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><?php esc_html_e('robots.txt','daily-tarot'); ?></td>
                        <td><?php echo esc_html((string)$robotsCode); ?></td>
                        <td><?php echo esc_html($fmtBool($robotsCode >= 200 && $robotsCode < 400)); ?></td>
                        <td>
                            <?php if ($robotsUrl !== ''): ?>
                                <a href="<?php echo esc_url($robotsUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Open','daily-tarot'); ?></a>
                            <?php endif; ?>
                            <?php echo ' — ' . esc_html($robotsHasSitemap ? __('Sitemap line present','daily-tarot') : __('Sitemap line missing','daily-tarot')); ?>
                        </td>
                    </tr>
                    <tr>
                        <td><?php esc_html_e('dtarot-sitemap.xml','daily-tarot'); ?></td>
                        <td><?php echo esc_html((string)$sitemapCode); ?></td>
                        <td><?php echo esc_html($fmtBool($sitemapCode >= 200 && $sitemapCode < 400 && $sitemapLooksXml)); ?></td>
                        <td>
                            <?php if ($sitemapUrl2 !== ''): ?>
                                <a href="<?php echo esc_url($sitemapUrl2); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Open','daily-tarot'); ?></a>
                            <?php endif; ?>
                            <?php echo ' — ' . esc_html($sitemapLooksXml ? __('Looks like XML','daily-tarot') : __('Unexpected body','daily-tarot')); ?>
                        </td>
                    </tr>
                </tbody>
            </table>
            <?php if ($err !== ''): ?>
                <p class="description" style="max-width:1100px;"><?php echo esc_html($err); ?></p>
            <?php endif; ?>
            <p class="description" style="max-width:1100px;"><?php echo esc_html(sprintf(__('Last checked: %s','daily-tarot'), (string)$publicDiag['time'])); ?></p>
        <?php endif; ?>

        <hr />

        <h2><?php esc_html_e('Content health check','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Quick scan of recent published days to catch missing images or text that can cause blank or low-quality public pages.','daily-tarot'); ?></p>

        <h2 style="margin-top:18px;"><?php esc_html_e('Plugin log','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Lightweight diagnostic log (rate limits, provider auth failures, etc).','daily-tarot'); ?></p>

        <p style="display:flex; gap:10px; flex-wrap:wrap; align-items:center;">
            <a class="button" href="<?php echo esc_url(wp_nonce_url(admin_url('admin-post.php?action=dtarot_log_export'), 'dtarot_log_export')); ?>"><?php esc_html_e('Download log JSON','daily-tarot'); ?></a>
            <a class="button" href="<?php echo esc_url(wp_nonce_url(admin_url('admin-post.php?action=dtarot_log_clear'), 'dtarot_log_clear')); ?>"><?php esc_html_e('Clear log','daily-tarot'); ?></a>
        </p>

        <?php
            // Optional: show last remote AI provider error (if any).
            $lastAiError = null;
            if (class_exists(\DailyTarot\Ai\RemoteProvider::class) && method_exists(\DailyTarot\Ai\RemoteProvider::class, 'getLastError')) {
                $lastAiError = \DailyTarot\Ai\RemoteProvider::getLastError();
            }
        ?>

        <?php if (is_array($lastAiError) && !empty($lastAiError['message'])): ?>
            <div class="notice notice-warning" style="max-width:1100px;">
                <p style="margin:10px 12px;"><strong><?php esc_html_e('Last AI provider error:','daily-tarot'); ?></strong> <?php echo esc_html((string)$lastAiError['message']); ?></p>
                <?php if (!empty($lastAiError['time'])): ?>
                    <p style="margin:0 12px 10px;"><span class="description"><?php echo esc_html((string)$lastAiError['time']); ?></span></p>
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <?php
            $healthLimit = (int)apply_filters('dtarot_diagnostics_health_limit', 30);
            if ($healthLimit <= 0) $healthLimit = 30;
            if ($healthLimit > 365) $healthLimit = 365;

            $healthItems = class_exists(DayEntryService::class) ? DayEntryService::latestPublished($healthLimit) : [];

            $issues = [];
            foreach ($healthItems as $d => $e) {
                if (!is_string($d) || !$e || !is_object($e)) continue;
                $arr = method_exists($e, 'toArray') ? (array)$e->toArray() : [];

                $content = isset($arr['content']) ? (string)$arr['content'] : '';
                $dailyText = isset($arr['daily_text']) ? (string)$arr['daily_text'] : '';
                $cardId = isset($arr['card']) ? (string)$arr['card'] : '';
                $deckId = isset($arr['deck']) ? (int)$arr['deck'] : 0;

                $hasText = trim(wp_strip_all_tags($content)) !== '' || trim(wp_strip_all_tags($dailyText)) !== '';

                $imgOverride = isset($arr['image_override_url']) ? (string)$arr['image_override_url'] : '';
                $hasImg = trim($imgOverride) !== '';
                if (!$hasImg && $deckId > 0 && $cardId !== '') {
                    $imgs = get_post_meta($deckId, '_dtarot_cards', true);
                    $hasImg = is_array($imgs) && !empty($imgs[$cardId]) && is_string($imgs[$cardId]) && trim((string)$imgs[$cardId]) !== '';
                }

                if (!$hasText || !$hasImg) {
                    $issues[] = [
                        'date' => $d,
                        'text' => $hasText,
                        'img' => $hasImg,
                    ];
                }
            }
        ?>

        <?php if (!$issues): ?>
            <p><?php esc_html_e('OK: No issues found in the scanned days.','daily-tarot'); ?></p>
        <?php else: ?>
            <table class="widefat striped" style="max-width:1100px;">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Date','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Missing text','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Missing image','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Public URL','daily-tarot'); ?></th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($issues as $row):
                    $d = (string)$row['date'];
                    $missingText = empty($row['text']);
                    $missingImg = empty($row['img']);
                    $u = class_exists(ReadableRoutes::class) ? ReadableRoutes::urlForDate($d) : '';
                ?>
                    <tr>
                        <td><?php echo esc_html($d); ?></td>
                        <td><?php echo esc_html($missingText ? __('Yes','daily-tarot') : __('No','daily-tarot')); ?></td>
                        <td><?php echo esc_html($missingImg ? __('Yes','daily-tarot') : __('No','daily-tarot')); ?></td>
                        <td>
                            <?php if ($u !== ''): ?>
                                <a href="<?php echo esc_url($u); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Open','daily-tarot'); ?></a>
                            <?php else: ?>
                                <?php esc_html_e('Unknown','daily-tarot'); ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>

        <h2><?php esc_html_e('Calendar storage','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('You can store daily calendar entries in the WordPress options table (legacy) or in a dedicated database table (recommended for performance).','daily-tarot'); ?></p>
        <p class="description"><?php esc_html_e('If things feel fast and you don’t have a large history, you can skip migration. Use it when the calendar grows or feels slow.','daily-tarot'); ?></p>

        <?php
            $backend = class_exists(DayEntryRepository::class) ? DayEntryRepository::backend() : 'option';
            $tableExists = class_exists(DayEntryTable::class) ? DayEntryTable::exists() : false;
            $tableCount = ($tableExists && class_exists(DayEntryTable::class)) ? DayEntryTable::countRows() : 0;
            $optionCount = class_exists(DayEntryRepository::class) ? DayEntryRepository::countOptionEntries() : 0;

            $migrateNotice = isset($_GET['dtarot_calendar_migrate']) ? sanitize_key((string)wp_unslash($_GET['dtarot_calendar_migrate'])) : '';
            $migrateCount = isset($_GET['count']) ? (int)wp_unslash($_GET['count']): 0;
            $migrateMsg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
        ?>

        <?php if ($migrateNotice === 'ok'): ?>
            <div class="notice notice-success is-dismissible"><p>
                <?php echo esc_html(sprintf(__('Migration finished. Entries copied: %d.','daily-tarot'), $migrateCount)); ?>
            </p></div>
        <?php elseif ($migrateNotice === 'error'): ?>
            <div class="notice notice-error is-dismissible"><p>
                <?php echo esc_html(sprintf(__('Migration failed (%s).','daily-tarot'), $migrateMsg !== '' ? $migrateMsg : 'error')); ?>
            </p></div>
        <?php endif; ?>

        <table class="widefat striped" style="max-width:1100px;">
            <tbody>
                <tr>
                    <td style="width:260px;"><strong><?php esc_html_e('Active backend','daily-tarot'); ?></strong></td>
                    <td><?php echo esc_html($backend); ?></td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Option entries','daily-tarot'); ?></strong></td>
                    <td><?php echo esc_html((string)$optionCount); ?></td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Table exists','daily-tarot'); ?></strong></td>
                    <td><?php echo esc_html($tableExists ? __('Yes','daily-tarot') : __('No','daily-tarot')); ?></td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Table rows','daily-tarot'); ?></strong></td>
                    <td><?php echo esc_html((string)$tableCount); ?></td>
                </tr>
            </tbody>
        </table>

        <h3><?php esc_html_e('Migrate to database table','daily-tarot'); ?></h3>
        <p class="description"><?php esc_html_e('Copies your existing calendar entries into the database table and switches the plugin to use it. Option storage is kept as a safe fallback.','daily-tarot'); ?></p>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('dtarot_calendar_storage'); ?>
            <input type="hidden" name="action" value="dtarot_calendar_migrate_table" />
            <p>
                <label>
                    <input type="checkbox" name="replace" value="1" />
                    <?php esc_html_e('Replace table contents before migrating','daily-tarot'); ?>
                </label>
            </p>
            <p>
                <button type="submit" class="button button-primary"><?php esc_html_e('Migrate now','daily-tarot'); ?></button>
            </p>
        </form>

        <hr />

        <h2><?php esc_html_e('Automation','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Runtime info for the Daily Tarot automation job (WP-Cron).','daily-tarot'); ?></p>

        <?php
            $enabled = false;
            $settings = class_exists(AutomationSettings::class) ? AutomationSettings::get() : [];
            if (is_array($settings) && !empty($settings['enabled'])) {
                $enabled = true;
            }

            $cronDisabled = (defined('DISABLE_WP_CRON') && (bool)DISABLE_WP_CRON);
            $alternateCron = (defined('ALTERNATE_WP_CRON') && (bool)ALTERNATE_WP_CRON);

            $nextTs = function_exists('wp_next_scheduled') ? wp_next_scheduled(Scheduler::HOOK) : false;
            $nextStr = $nextTs ? (function_exists('wp_date') ? (string)wp_date('Y-m-d H:i:s', (int)$nextTs) : gmdate('Y-m-d H:i:s', (int)$nextTs)) : '';

            $lastAt = (string)get_option('dtarot_automation_last_run_at', '');
            $last = get_option('dtarot_automation_last_run', null);
            $lastMsg = '';
            $lastOk = null;
            $lastManual = null;
            if (is_array($last)) {
                $lastMsg = isset($last['msg']) ? (string)$last['msg'] : '';
                $lastOk = isset($last['ok']) ? (bool)$last['ok'] : null;
                $lastManual = isset($last['manual']) ? (bool)$last['manual'] : null;
            }
        ?>

        <table class="widefat striped" style="max-width:1100px;">
            <tbody>
                <tr>
                    <td style="width:260px;"><strong><?php esc_html_e('Enabled','daily-tarot'); ?></strong></td>
                    <td><?php echo esc_html($enabled ? __('Yes','daily-tarot') : __('No','daily-tarot')); ?></td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Next scheduled run','daily-tarot'); ?></strong></td>
                    <td>
                        <?php
                            if ($nextTs) {
                                echo esc_html($nextStr);
                            } else {
                                echo esc_html__('Not scheduled','daily-tarot');
                            }
                        ?>
                    </td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('WP-Cron status','daily-tarot'); ?></strong></td>
                    <td>
                        <?php
                            if ($cronDisabled) {
                                echo esc_html__('Disabled (DISABLE_WP_CRON is true)','daily-tarot');
                            } elseif ($alternateCron) {
                                echo esc_html__('Alternate cron enabled (ALTERNATE_WP_CRON)','daily-tarot');
                            } else {
                                echo esc_html__('Normal','daily-tarot');
                            }
                        ?>
                    </td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Not scheduled reason','daily-tarot'); ?></strong></td>
                    <td>
                        <?php
                            if ($nextTs) {
                                echo esc_html__('N/A','daily-tarot');
                            } elseif (!$enabled) {
                                echo esc_html__('Automation is disabled in Settings → Automation.','daily-tarot');
                            } elseif ($cronDisabled) {
                                echo esc_html__('WP-Cron is disabled, so the job will not run automatically.','daily-tarot');
                            } else {
                                echo esc_html__('No scheduled event found. Click “Reschedule now” in Settings → Automation.','daily-tarot');
                            }
                        ?>
                    </td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Last run','daily-tarot'); ?></strong></td>
                    <td><?php echo esc_html($lastAt !== '' ? $lastAt : __('Never','daily-tarot')); ?></td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Last result','daily-tarot'); ?></strong></td>
                    <td>
                        <?php
                            if ($lastMsg !== '') {
                                $bits = [$lastMsg];
                                if ($lastOk !== null) {
                                    $bits[] = $lastOk ? 'ok' : 'error';
                                }
                                if ($lastManual !== null) {
                                    $bits[] = $lastManual ? 'manual' : 'scheduled';
                                }
                                echo esc_html(implode(' / ', $bits));
                            } else {
                                echo esc_html__('N/A','daily-tarot');
                            }
                        ?>
                    </td>
                </tr>
            </tbody>
        </table>
        <?php
    }

    public static function handleFetchPublicDiagnostics(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_diagnostics_fetch_public');

        $userId = (int)get_current_user_id();
        $key = self::DIAG_PUBLIC_TRANSIENT_PREFIX . (string)$userId;

        $robotsUrl = home_url('/robots.txt');
        $sitemapUrl = class_exists(Sitemap::class) ? Sitemap::url() : '';

        $result = [
            'time' => gmdate('c'),
            'robots_url' => $robotsUrl,
            'robots_code' => 0,
            'robots_has_sitemap' => false,
            'sitemap_url' => $sitemapUrl,
            'sitemap_code' => 0,
            'sitemap_looks_xml' => false,
            'error' => '',
        ];

        $args = [
            'timeout' => 10,
            'redirection' => 3,
            'headers' => [
                'Accept' => 'text/plain, application/xml, text/xml;q=0.9, */*;q=0.8',
            ],
        ];

        try {
            $robotsResp = wp_remote_get($robotsUrl, $args);
            if (is_wp_error($robotsResp)) {
                $result['error'] .= 'robots.txt: ' . $robotsResp->get_error_message() . "\n";
            } else {
                $code = (int)wp_remote_retrieve_response_code($robotsResp);
                $body = (string)wp_remote_retrieve_body($robotsResp);
                $result['robots_code'] = $code;

                if ($sitemapUrl !== '' && $body !== '') {
                    $result['robots_has_sitemap'] = (stripos($body, 'sitemap:') !== false) && (stripos($body, $sitemapUrl) !== false);
                }
            }

            if ($sitemapUrl !== '') {
                $sitemapResp = wp_remote_get($sitemapUrl, $args);
                if (is_wp_error($sitemapResp)) {
                    $result['error'] .= 'sitemap: ' . $sitemapResp->get_error_message() . "\n";
                } else {
                    $code = (int)wp_remote_retrieve_response_code($sitemapResp);
                    $body = (string)wp_remote_retrieve_body($sitemapResp);
                    $result['sitemap_code'] = $code;
                    if ($body !== '') {
                        $result['sitemap_looks_xml'] = (stripos($body, '<urlset') !== false) || (stripos($body, '<sitemapindex') !== false);
                    }
                }
            }
        } catch (\Throwable $e) {
            $result['error'] .= $e->getMessage();
        }

        set_transient($key, $result, 10 * MINUTE_IN_SECONDS);

        $to = admin_url('admin.php?page=daily-tarot-settings&tab=diagnostics');
        $to = add_query_arg(['dtarot_diag_public' => 'ok'], $to);
        wp_safe_redirect($to);
        exit;
    }

    public static function handleLogExport(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_log_export');

        $items = class_exists(Log::class) ? Log::all() : [];
        $payload = [
            'generated_at' => gmdate('c'),
            'count' => is_array($items) ? count($items) : 0,
            'items' => $items,
        ];

        $name = 'dtarot-log-' . gmdate('Ymd-His') . '.json';
        nocache_headers();
        header('Content-Type: application/json; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . $name . '"');
        echo wp_json_encode($payload, JSON_PRETTY_PRINT);
        exit;
    }

    public static function handleLogClear(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_log_clear');
        if (class_exists(Log::class)) {
            Log::clear();
        }
        $to = admin_url('admin.php?page=daily-tarot-settings&tab=diagnostics');
        wp_safe_redirect($to);
        exit;
    }

    public static function render(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));
        ?>
        <div class="wrap dtarot-wrap" data-dtarot-settings-page="1">
            <h1><?php esc_html_e('Daily Tarot Settings','daily-tarot'); ?></h1>
            <p class="description"><?php esc_html_e('Settings will expand in future versions (Email, Pro, AI, Analytics).','daily-tarot'); ?></p>

            <?php if (class_exists(Ui::class)) { Ui::renderQuickActions('settings'); } ?>

            <?php
            $notice = isset($_GET['dtarot_backup']) ? sanitize_key((string)wp_unslash($_GET['dtarot_backup'])) : '';
            if ($notice === 'import_ok') {
                $decks = isset($_GET['decks']) ? (int)wp_unslash($_GET['decks']): 0;
                $packs = isset($_GET['packs']) ? (int)wp_unslash($_GET['packs']): 0;
                $decksSkipped = isset($_GET['decks_skipped']) ? (int)wp_unslash($_GET['decks_skipped']): 0;
                $packsSkipped = isset($_GET['packs_skipped']) ? (int)wp_unslash($_GET['packs_skipped']): 0;
                $days = isset($_GET['days']) ? (int)wp_unslash($_GET['days']): 0;
                $msg = sprintf(__('Import complete. Decks: %d (skipped %d), Meaning Packs: %d (skipped %d), Days: %d.','daily-tarot'), $decks, $decksSkipped, $packs, $packsSkipped, $days);
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($msg) . '</p></div>';
            } elseif ($notice === 'deck_zip_ok') {
                $deckId = isset($_GET['deck_id']) ? (int)wp_unslash($_GET['deck_id']): 0;
                $imported = isset($_GET['imported']) ? (int)wp_unslash($_GET['imported']): 0;
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html(sprintf(__('Deck ZIP imported. Deck ID: %d. Images imported: %d.','daily-tarot'), $deckId, $imported)) . '</p></div>';
            } elseif ($notice === 'pack_zip_ok') {
                $packId = isset($_GET['pack_id']) ? (int)wp_unslash($_GET['pack_id']): 0;
                $cards = isset($_GET['cards']) ? (int)wp_unslash($_GET['cards']): 0;
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html(sprintf(__('Meaning Pack ZIP imported. Pack ID: %d. Cards imported: %d.','daily-tarot'), $packId, $cards)) . '</p></div>';
            } elseif ($notice === 'import_report') {
                echo '<div class="notice notice-info is-dismissible"><p>' . esc_html__('Import report is available below.','daily-tarot') . '</p></div>';
            } elseif ($notice === 'import_error') {
                $msg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : 'unknown';
                $map = [
                    'missing_file' => __('No file uploaded.','daily-tarot'),
                    'bad_upload' => __('Upload failed. Please try again.','daily-tarot'),
                    'file_too_large' => __('File is too large (max 10MB).','daily-tarot'),
                    'empty_file' => __('File was empty.','daily-tarot'),
                    'bad_json' => __('Invalid JSON backup file.','daily-tarot'),
                    'zip_missing' => __('ZIP import requires PHP ZipArchive support on the server.','daily-tarot'),
                    'upload_dir' => __('Could not access uploads directory.','daily-tarot'),
                    'workdir' => __('Could not create a temporary import directory.','daily-tarot'),
                    'bad_zip' => __('Invalid ZIP file.','daily-tarot'),
                    'missing_manifest' => __('ZIP is missing manifest.json.','daily-tarot'),
                    'bad_manifest' => __('manifest.json is invalid.','daily-tarot'),
                    'format_version' => __('Unsupported pack format version.','daily-tarot'),
                    'min_version' => __('Pack requires a newer plugin version.','daily-tarot'),
                    'missing_files' => __('manifest.json is missing the files list (integrity data).','daily-tarot'),
                    'missing_deck' => __('manifest.json is missing the deck section.','daily-tarot'),
                    'deck_slug' => __('Deck manifest must include a valid slug or title.','daily-tarot'),
                    'deck_create' => __('Could not create/update the deck post.','daily-tarot'),
                    'missing_pack' => __('manifest.json is missing the meaning_pack section.','daily-tarot'),
                    'pack_slug' => __('Meaning pack manifest must include a valid slug or title.','daily-tarot'),
                    'pack_create' => __('Could not create/update the meaning pack post.','daily-tarot'),
                    'exists' => __('An item with this slug already exists (enable Overwrite to update it).','daily-tarot'),
                    'downgrade_blocked' => __('Downgrade blocked (enable Allow downgrade to install an older version).','daily-tarot'),
                    'invalid_url' => __('Invalid URL.','daily-tarot'),
                    'download_failed' => __('Could not download the ZIP from the URL.','daily-tarot'),
                ];
                $text = $map[$msg] ?? __('Import failed.','daily-tarot');
                echo '<div class="notice notice-error is-dismissible"><p>' . esc_html($text) . '</p></div>';
            }

            $autoNotice = isset($_GET['dtarot_automation']) ? sanitize_key((string)wp_unslash($_GET['dtarot_automation'])) : '';
            if ($autoNotice === 'ok') {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Automation settings saved.','daily-tarot') . '</p></div>';
            } elseif ($autoNotice === 'ran') {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Automation job ran.','daily-tarot') . '</p></div>';
            } elseif ($autoNotice === 'error') {
                $msg = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : 'error';
                $map = [
                    'disabled' => __('Automation is disabled.','daily-tarot'),
                    'already_published' => __('Today is already published.','daily-tarot'),
                    'no_deck' => __('No deck found. Create a deck first.','daily-tarot'),
                    'no_pack' => __('No meaning pack found for the selected system.','daily-tarot'),
                    'no_card' => __('Could not select a card.','daily-tarot'),
                    'no_job' => __('Automation job is unavailable (missing plugin files).','daily-tarot'),
                    'error' => __('Automation failed.','daily-tarot'),
                ];
                echo '<div class="notice notice-error is-dismissible"><p>' . esc_html($map[$msg] ?? $map['error']) . '</p></div>';
            }

            $reportToken = isset($_GET['report']) ? sanitize_text_field((string)wp_unslash($_GET['report'])) : '';
            $report = $reportToken !== '' && class_exists(Backup::class) ? Backup::readReport($reportToken) : null;
            ?>

            <?php if (is_array($report)) :
                $ok = !empty($report['ok']);
                $kind = isset($report['kind']) ? (string)$report['kind'] : '';
                $dry = !empty($report['dry_run']);
                $lines = [];
                $lines[] = 'kind: ' . $kind;
                $lines[] = 'ok: ' . ($ok ? 'true' : 'false');
                if ($dry) $lines[] = 'dry_run: true';
                foreach (['slug','title','deck_id','pack_id','pack_version','expected_images','imported_images','skipped_images','cards'] as $k) {
                    if (isset($report[$k])) {
                        $lines[] = $k . ': ' . (is_scalar($report[$k]) ? (string)$report[$k] : '[complex]');
                    }
                }
                if (!empty($report['missing']) && is_array($report['missing'])) {
                    $lines[] = 'missing:';
                    foreach (array_slice($report['missing'], 0, 25) as $m) {
                        $lines[] = '  - ' . (string)$m;
                    }
                }
                if (!empty($report['errors']) && is_array($report['errors'])) {
                    $lines[] = 'errors:';
                    foreach (array_slice($report['errors'], 0, 25) as $e) {
                        $lines[] = '  - ' . (string)$e;
                    }
                }
                $text = implode("\n", $lines);
            ?>
                <details style="max-width:1100px; margin: 16px 0;">
                    <summary><?php esc_html_e('Last import report','daily-tarot'); ?></summary>
                    <pre style="white-space:pre-wrap; background:#fff; padding:12px; border:1px solid #ccd0d4;"><?php echo esc_html($text); ?></pre>
                </details>
            <?php endif; ?>

            <hr />

            <?php
            $noticeForDefault = isset($_GET['dtarot_backup']) ? sanitize_key((string)wp_unslash($_GET['dtarot_backup'])) : '';
            $tab = isset($_GET['tab']) ? sanitize_key((string)wp_unslash($_GET['tab'])) : '';
            $tab = self::resolveTab($tab, $noticeForDefault);

            self::renderTabNav($tab);

            self::render_panel($tab);
            ?>
        </div>
        <?php
    }
}
