<?php
declare(strict_types=1);


namespace DailyTarot\Admin\Pages;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



final class Feedback {

    private const TO_EMAIL = 'dailytarothelp@gmail.com';

    private static function baseUrl(): string {
        return admin_url('admin.php?page=daily-tarot-feedback');
    }

    public static function render(): void {
        if (!current_user_can('manage_options')) {
            echo '<div class="wrap"><h1>' . esc_html__('Feedback','daily-tarot') . '</h1><p>' . esc_html__('Forbidden','daily-tarot') . '</p></div>';
            return;
        }

        $notice = isset($_GET['msg']) ? sanitize_key((string)wp_unslash($_GET['msg'])) : '';
        ?>
        <div class="wrap dtarot-wrap">
            <h1><?php esc_html_e('Feedback','daily-tarot'); ?></h1>

            <?php if ($notice === 'ok'): ?>
                <div class="notice notice-success is-dismissible"><p><?php esc_html_e('Thanks! Your feedback was sent.','daily-tarot'); ?></p></div>
            <?php elseif ($notice === 'error'): ?>
                <div class="notice notice-error is-dismissible"><p><?php esc_html_e('Feedback could not be sent. Please try again.','daily-tarot'); ?></p></div>
            <?php elseif ($notice === 'missing'): ?>
                <div class="notice notice-error is-dismissible"><p><?php esc_html_e('Please include a message.','daily-tarot'); ?></p></div>
            <?php endif; ?>

            <div class="dtarot-card" style="max-width: 900px;">
                <h2 style="margin-top:0;"><?php esc_html_e('Send feedback','daily-tarot'); ?></h2>
                <p class="description"><?php esc_html_e('Share a complaint, wish, or bug report. This goes directly to the Daily Tarot team.','daily-tarot'); ?></p>

                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('dtarot_feedback_submit'); ?>
                    <input type="hidden" name="action" value="dtarot_feedback_submit" />

                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><label for="dtarot_feedback_type"><?php esc_html_e('Type','daily-tarot'); ?></label></th>
                            <td>
                                <select id="dtarot_feedback_type" name="dtarot_feedback_type">
                                    <option value="wish"><?php esc_html_e('Feature request','daily-tarot'); ?></option>
                                    <option value="bug"><?php esc_html_e('Bug report','daily-tarot'); ?></option>
                                    <option value="complaint"><?php esc_html_e('Complaint','daily-tarot'); ?></option>
                                    <option value="other"><?php esc_html_e('Other','daily-tarot'); ?></option>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="dtarot_feedback_email"><?php esc_html_e('Your email (optional)','daily-tarot'); ?></label></th>
                            <td>
                                <input type="email" id="dtarot_feedback_email" name="dtarot_feedback_email" style="width: 520px; max-width: 100%;" placeholder="name@example.com" />
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="dtarot_feedback_message"><?php esc_html_e('Message','daily-tarot'); ?></label></th>
                            <td>
                                <textarea id="dtarot_feedback_message" name="dtarot_feedback_message" rows="6" style="width: 520px; max-width: 100%;" required></textarea>
                            </td>
                        </tr>
                    </table>

                    <p>
                        <button type="submit" class="button button-primary"><?php esc_html_e('Send feedback','daily-tarot'); ?></button>
                    </p>
                </form>
            </div>
        </div>
        <?php
    }

    public static function handleSubmit(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer('dtarot_feedback_submit');

        $type = isset($_POST['dtarot_feedback_type']) ? sanitize_key((string)wp_unslash($_POST['dtarot_feedback_type'])) : 'other';
        if (!in_array($type, ['wish','bug','complaint','other'], true)) {
            $type = 'other';
        }

        $email = isset($_POST['dtarot_feedback_email']) ? sanitize_email((string)wp_unslash($_POST['dtarot_feedback_email'])) : '';
        $message = isset($_POST['dtarot_feedback_message']) ? trim((string)wp_unslash($_POST['dtarot_feedback_message'])) : '';

        if ($message === '') {
            wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-feedback', 'msg' => 'missing'], admin_url('admin.php')));
            exit;
        }

        $siteUrl = function_exists('home_url') ? (string)home_url('/') : '';
        $pluginVer = defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '';
        $wpVer = function_exists('get_bloginfo') ? (string)get_bloginfo('version') : '';
        $userId = function_exists('get_current_user_id') ? (string)get_current_user_id() : '';
        $createdAt = gmdate('c');

        $subject = sprintf(__('Daily Tarot feedback (%s)','daily-tarot'), $type);
        $body = "Type: {$type}\n";
        if ($email !== '') {
            $body .= "Email: {$email}\n";
        }
        if ($siteUrl !== '') {
            $body .= "Site: {$siteUrl}\n";
        }
        if ($pluginVer !== '') {
            $body .= "Plugin: {$pluginVer}\n";
        }
        if ($wpVer !== '') {
            $body .= "WP: {$wpVer}\n";
        }
        $body .= "\nMessage:\n{$message}\n";

        $headers = [];
        if ($email !== '') {
            $headers[] = 'Reply-To: ' . $email;
        }

        $to = sanitize_email(self::TO_EMAIL);
        $ok = $to !== '' ? wp_mail($to, $subject, $body, $headers) : false;

        wp_safe_redirect(add_query_arg(['page' => 'daily-tarot-feedback', 'msg' => $ok ? 'ok' : 'error'], admin_url('admin.php')));
        exit;
    }
}
