<?php
declare(strict_types=1);


namespace DailyTarot\Admin\Pages;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.DateTime.RestrictedFunctions.date_date

// phpcs:disable WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Automation\AutomationSettings;
use DailyTarot\Automation\Scheduler;
use DailyTarot\Calendar\DayEntryRepository;
use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Frontend\ReadableRoutes;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\FeatureFlags;
use DailyTarot\Support\ProUrls;
use DailyTarot\Support\PostTypes;
use DailyTarot\Support\OnlineVisitors;
use DailyTarot\Support\SpreadMappings;
use DailyTarot\Support\SpreadPresets;

final class Dashboard {

    private static function todayYmd(): string {
        if (function_exists('wp_date')) return (string)wp_date('Y-m-d');
        if (function_exists('current_time')) return (string)current_time('Y-m-d');
        return (string)gmdate('Y-m-d');
    }

    /** @return array<int,string> */
    private static function missingForEntryArray(array $arr): array {
        $deckId = isset($arr['deck']) ? (int)$arr['deck'] : 0;
        $cardId = isset($arr['card']) && is_string($arr['card']) ? trim((string)$arr['card']) : '';
        $packId = isset($arr['pack']) ? (int)$arr['pack'] : 0;

        $content = isset($arr['content']) && is_string($arr['content']) ? trim(wp_strip_all_tags((string)$arr['content'])) : '';
        $dailyText = isset($arr['daily_text']) && is_string($arr['daily_text']) ? trim(wp_strip_all_tags((string)$arr['daily_text'])) : '';

        $missing = [];
        if ($deckId <= 0) $missing[] = __('Deck','daily-tarot');
        if ($cardId === '') $missing[] = __('Card','daily-tarot');
        if ($packId <= 0) $missing[] = __('Meaning Pack','daily-tarot');
        if ($content === '' && $dailyText === '') $missing[] = __('Text','daily-tarot');
        return $missing;
    }

    private static function calendarEditUrl(string $date): string {
        $y = (int)substr($date, 0, 4);
        $m = (int)substr($date, 5, 2);
        return (string)add_query_arg([
            'page' => 'daily-tarot-calendar',
            'year' => (string)$y,
            'month' => (string)$m,
            'dtarot_edit_date' => $date,
        ], admin_url('admin.php'));
    }

    public static function render(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));

        $today = self::todayYmd();
        $todayEntry = class_exists(DayEntryService::class) ? DayEntryService::get($today) : null;
        $todayArr = ($todayEntry && method_exists($todayEntry, 'toArray')) ? (array)$todayEntry->toArray() : [];
        $todayStatus = ($todayEntry && property_exists($todayEntry, 'status')) ? (string)$todayEntry->status : '';
        $todayMissing = self::missingForEntryArray($todayArr);

        $todayDeckId = isset($todayArr['deck']) ? (int)$todayArr['deck'] : 0;
        $todayCardId = isset($todayArr['card']) && is_string($todayArr['card']) ? trim((string)$todayArr['card']) : '';
        $todayCardName = $todayCardId !== '' ? Cards::name($todayCardId) : '';
        $todayDeckTitle = $todayDeckId > 0 ? (string)get_the_title($todayDeckId) : '';
        if ($todayDeckId > 0 && $todayDeckTitle === '') {
            $todayDeckTitle = __('(no title)','daily-tarot');
        }

        $todayPretty = $today;
        $todayTs = strtotime($today . ' 00:00:00');
        if ($todayTs !== false) {
            $todayPretty = function_exists('wp_date') ? (string)wp_date('M j, Y', $todayTs) : (string)date_i18n('M j, Y', $todayTs);
        }

        $todayImageUrl = '';
        $overrideUrl = isset($todayArr['image_override_url']) && is_string($todayArr['image_override_url'])
            ? trim((string)$todayArr['image_override_url'])
            : '';
        $overrideAttId = isset($todayArr['image_override_attachment_id']) ? (int)$todayArr['image_override_attachment_id'] : 0;
        if ($overrideUrl !== '') {
            $todayImageUrl = esc_url($overrideUrl);
        } elseif ($overrideAttId > 0 && function_exists('wp_get_attachment_url')) {
            $attUrl = wp_get_attachment_url($overrideAttId);
            if (is_string($attUrl) && trim($attUrl) !== '') {
                $todayImageUrl = esc_url($attUrl);
            }
        } elseif ($todayDeckId > 0 && $todayCardId !== '') {
            $imgs = get_post_meta($todayDeckId, '_dtarot_cards', true);
            if (is_array($imgs)) {
                $url = $imgs[$todayCardId] ?? '';
                if (is_string($url) && trim($url) !== '') {
                    $todayImageUrl = esc_url((string)$url);
                }
            }
        }

        $todayReadableUrl = class_exists(ReadableRoutes::class) ? (string)ReadableRoutes::urlForDate($today) : '';
        $todayExcerpt = '';
        $rawExcerpt = isset($todayArr['content']) && is_string($todayArr['content']) ? trim(wp_strip_all_tags((string)$todayArr['content'])) : '';
        if ($rawExcerpt === '') {
            $rawExcerpt = isset($todayArr['daily_text']) && is_string($todayArr['daily_text']) ? trim(wp_strip_all_tags((string)$todayArr['daily_text'])) : '';
        }
        if ($rawExcerpt !== '') {
            $todayExcerpt = (mb_strlen($rawExcerpt) > 200) ? (mb_substr($rawExcerpt, 0, 197) . '...') : $rawExcerpt;
        }

        $calendarTodayUrl = self::calendarEditUrl($today);
        $calendarMonthUrl = '';
        $statusUrl = admin_url('admin.php?page=daily-tarot-content&tab=status');
        $contentUrl = admin_url('admin.php?page=daily-tarot-content');
        $settingsAutomationUrl = admin_url('admin.php?page=daily-tarot-ai&tab=automation');
        $helpUrl = admin_url('admin.php?page=daily-tarot-settings&tab=help');

        $nowTs = function_exists('current_time') ? (int)current_time('timestamp') : time();

        $monthLabel = function_exists('wp_date') ? (string)wp_date('F Y', $nowTs) : (string)date('F Y', $nowTs);
        $monthYear = (int)(function_exists('wp_date') ? wp_date('Y', $nowTs) : date('Y', $nowTs));
        $monthNum = (int)(function_exists('wp_date') ? wp_date('n', $nowTs) : date('n', $nowTs));
        $daysInMonth = (int)(function_exists('wp_date') ? wp_date('t', $nowTs) : date('t', $nowTs));

        $calendarMonthUrl = (string)add_query_arg([
            'page' => 'daily-tarot-calendar',
            'year' => (string)$monthYear,
            'month' => (string)$monthNum,
        ], admin_url('admin.php'));

        $firstDayTs = strtotime(sprintf('%04d-%02d-01', $monthYear, $monthNum) . ' 00:00:00');
        $firstWeekday = 0;
        if (is_int($firstDayTs) && $firstDayTs > 0) {
            $firstWeekday = (int)(function_exists('wp_date') ? wp_date('w', $firstDayTs) : date('w', $firstDayTs));
        }
        $monthStart = function_exists('wp_date') ? (string)wp_date('Y-m-01', $nowTs) : (string)date('Y-m-01', $nowTs);
        $monthEnd = function_exists('wp_date') ? (string)wp_date('Y-m-t', $nowTs) : (string)date('Y-m-t', $nowTs);
        $entriesThisMonth = 0;
        if (class_exists(DayEntryRepository::class)) {
            $all = DayEntryRepository::all();
            foreach ((array)$all as $date => $_entry) {
                if (!is_string($date)) continue;
                if ($date >= $monthStart && $date <= $monthEnd) {
                    $entriesThisMonth++;
                }
            }
        }

        $activeDecks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
            'fields' => 'ids',
        ]);
        $activeDeckCount = is_array($activeDecks) ? count($activeDecks) : 0;

        $activePacks = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
            'fields' => 'ids',
        ]);
        $activePackCount = is_array($activePacks) ? count($activePacks) : 0;

        $rangeDays = 7;

        $unpublishedCount = 0;
        $readinessIssues = [];
        for ($i = 0; $i < $rangeDays; $i++) {
            $ts = strtotime('+' . $i . ' day', $nowTs);
            if (!is_int($ts) || $ts <= 0) continue;
            $date = function_exists('wp_date') ? (string)wp_date('Y-m-d', $ts) : (string)date('Y-m-d', $ts);

            if (!class_exists(DayEntryService::class)) continue;
            $e = DayEntryService::get($date);
            $status = property_exists($e, 'status') ? (string)$e->status : '';
            if ($status !== 'publish') {
                $unpublishedCount++;
            }
            $arr = (array)$e->toArray();
            $missing = self::missingForEntryArray($arr);
            if ($missing) {
                $readinessIssues[] = [
                    'date' => $date,
                    'status' => $status,
                    'missing' => $missing,
                ];
            }
        }

        $autoEnabled = class_exists(AutomationSettings::class) ? (bool)AutomationSettings::isEnabled() : false;
        $autoSettings = class_exists(AutomationSettings::class) ? (array)AutomationSettings::get() : ['time' => ''];
        $autoTime = isset($autoSettings['time']) ? (string)$autoSettings['time'] : '';
        $nextRun = class_exists(Scheduler::class) ? wp_next_scheduled(Scheduler::HOOK) : false;
        $nextRunText = $nextRun ? (function_exists('wp_date') ? (string)wp_date('Y-m-d H:i', (int)$nextRun) : (string)date('Y-m-d H:i', (int)$nextRun)) : __('Not scheduled','daily-tarot');

        $lastRunAt = (string)get_option('dtarot_automation_last_run_at', '');
        $lastRun = get_option('dtarot_automation_last_run', []);
        if (!is_array($lastRun)) $lastRun = [];
        $lastRunOk = isset($lastRun['ok']) ? (bool)$lastRun['ok'] : null;
        $lastRunMsg = isset($lastRun['msg']) && is_string($lastRun['msg']) ? (string)$lastRun['msg'] : '';

        $onlineCount = class_exists(OnlineVisitors::class) ? OnlineVisitors::countOnline() : 0;
        $onlineWindow = class_exists(OnlineVisitors::class) ? OnlineVisitors::windowSeconds() : 300;
        $onlineMinutes = (int)max(1, (int)round($onlineWindow / 60));

        $nowUtc = time();
        $bookingHorizon = $nowUtc + (int)(DAY_IN_SECONDS * 7);
        $upcomingBookings = [];
        $upcomingCount = 0;
        if (class_exists('WP_Query')) {
            $query = new \WP_Query([
                'post_type' => PostTypes::bookingTypes(),
                'post_status' => 'publish',
                'posts_per_page' => 6,
                'fields' => 'ids',
                'meta_key' => '_dtarot_booking_start_utc', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Limited admin dashboard query.
                'orderby' => 'meta_value_num',
                'order' => 'ASC',
                'meta_query' => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Limited admin dashboard query.
                    [
                        'key' => '_dtarot_booking_start_utc',
                        'value' => $nowUtc,
                        'compare' => '>=',
                        'type' => 'NUMERIC',
                    ],
                    [
                        'key' => '_dtarot_booking_status',
                        'value' => ['pending','approved'],
                        'compare' => 'IN',
                    ],
                ],
            ]);

            if ($query->have_posts()) {
                foreach ((array)$query->posts as $bookingId) {
                    $startUtc = (int)get_post_meta((int)$bookingId, '_dtarot_booking_start_utc', true);
                    if ($startUtc <= 0) continue;
                    $status = (string)get_post_meta((int)$bookingId, '_dtarot_booking_status', true);
                    $readingId = (int)get_post_meta((int)$bookingId, '_dtarot_booking_reading_type', true);
                    $readingTitle = $readingId > 0 ? (string)get_the_title($readingId) : '';
                    if ($readingTitle === '') $readingTitle = __('(reading)','daily-tarot');

                    if ($startUtc <= $bookingHorizon) {
                        $upcomingCount++;
                    }

                    $upcomingBookings[] = [
                        'id' => (int)$bookingId,
                        'start' => $startUtc,
                        'status' => $status,
                        'reading' => $readingTitle,
                    ];
                }
            }
            wp_reset_postdata();
        }

        $spreadCounts = [];
        if (class_exists(SpreadMappings::class)) {
            $mappings = SpreadMappings::all();
            foreach ($mappings as $postId => $rows) {
                if (!is_array($rows)) continue;
                foreach ($rows as $row) {
                    if (!is_array($row)) continue;
                    $preset = isset($row['preset']) ? sanitize_key((string)$row['preset']) : '';
                    if ($preset === '') continue;
                    if (!isset($spreadCounts[$preset])) $spreadCounts[$preset] = 0;
                    $spreadCounts[$preset]++;
                }
            }
        }
        arsort($spreadCounts);
        $topSpreads = array_slice($spreadCounts, 0, 3, true);
        $presetLabels = class_exists(SpreadPresets::class) ? SpreadPresets::all() : [];

        $userId = function_exists('get_current_user_id') ? (int)get_current_user_id() : 0;
        $onboardDismissed = $userId > 0 ? (bool)get_user_meta($userId, 'dtarot_onboard_dismissed', true) : false;
        $onboardTourDone = $userId > 0 ? (bool)get_user_meta($userId, 'dtarot_onboard_tour_done', true) : false;

        $hasCard = ($todayDeckId > 0 && $todayCardId !== '');
        $hasSpread = !empty($spreadCounts);
        $readingTypes = get_posts([
            'post_type' => PostTypes::readingTypeTypes(),
            'numberposts' => 1,
            'post_status' => ['publish','draft','pending','private'],
            'fields' => 'ids',
        ]);
        $hasBooking = is_array($readingTypes) && count($readingTypes) > 0;
        $onboardComplete = ($hasCard && $hasSpread && $hasBooking);
        $showOnboardBanner = (!$onboardDismissed && !$onboardComplete);
        $showOnboardCard = (!$onboardDismissed && !$onboardComplete);



        $buyUrl = class_exists(ProUrls::class) ? trim((string)ProUrls::buyProUrl()) : '';
        $accountUrl = class_exists(ProUrls::class) ? trim((string)ProUrls::accountUrl()) : '';
        ?>
        <div class="wrap dtarot-wrap">
            <div class="dtarot-dashboard-hero dtarot-dashboard-hero-new">
                <div>
                    <h1 class="dtarot-dashboard-title"><?php esc_html_e('Welcome to Daily Tarot','daily-tarot'); ?></h1>
                    <p class="dtarot-dashboard-subtitle"><?php esc_html_e('Your daily tarot insights at a glance.','daily-tarot'); ?></p>
                </div>
                <div class="dtarot-dashboard-hero-actions">
                    <a class="button button-primary" href="<?php echo esc_url($helpUrl); ?>"><?php esc_html_e('Help / Readme','daily-tarot'); ?></a>
                </div>
            </div>

            <?php if ($activeDeckCount <= 0 || $activePackCount <= 0): ?>
                <div class="notice notice-info" style="padding:10px 12px;">
                    <p style="margin:0;">
                        <strong><?php esc_html_e('Not sure what to do next?','daily-tarot'); ?></strong>
                        <?php esc_html_e('Decks store images → Meaning Packs store text → Calendar publishes your daily entries.','daily-tarot'); ?>
                    </p>
                    <p style="margin:8px 0 0 0; display:flex; gap:8px; flex-wrap:wrap;">
                        <?php if ($activeDeckCount <= 0): ?>
                            <a class="button" href="<?php echo esc_url(admin_url('post-new.php?post_type=' . PostTypes::DECK)); ?>"><?php esc_html_e('Create deck','daily-tarot'); ?></a>
                        <?php else: ?>
                            <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-content&tab=cards')); ?>"><?php esc_html_e('Upload card images','daily-tarot'); ?></a>
                        <?php endif; ?>

                        <?php if ($activePackCount <= 0): ?>
                            <a class="button" href="<?php echo esc_url(admin_url('post-new.php?post_type=' . PostTypes::MEANING_PACK)); ?>"><?php esc_html_e('Create meaning pack','daily-tarot'); ?></a>
                        <?php else: ?>
                            <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-content&tab=meanings')); ?>"><?php esc_html_e('Edit meaning packs','daily-tarot'); ?></a>
                        <?php endif; ?>

                        <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-calendar')); ?>"><?php esc_html_e('Open calendar','daily-tarot'); ?></a>
                    </p>
                </div>
            <?php endif; ?>

            <div class="dtarot-dashboard-welcome">
                <span class="dtarot-dashboard-welcome-icon">*</span>
                <?php esc_html_e('Welcome back - your tarot energy is flowing smoothly.','daily-tarot'); ?>
            </div>

            <?php if ($showOnboardBanner): ?>
                <div class="dtarot-onboard-banner" data-dtarot-onboard-banner="1">
                    <div class="dtarot-onboard-banner-text">
                        <?php esc_html_e('New here? Set your first card of the day, then explore spreads and bookings.','daily-tarot'); ?>
                    </div>
                    <div class="dtarot-onboard-banner-actions">
                        <button type="button" class="button dtarot-onboard-start" data-dtarot-onboard-start="1"><?php esc_html_e('Start tour','daily-tarot'); ?></button>
                        <button type="button" class="button-link dtarot-onboard-dismiss" data-dtarot-onboard-dismiss="1"><?php esc_html_e('Dismiss','daily-tarot'); ?></button>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (!FeatureFlags::enabled('pro')): ?>
                <div class="dtarot-card dtarot-pro-card">
                    <h2 style="margin-top:0;"><?php esc_html_e('Upgrade to Pro','daily-tarot'); ?></h2>
                    <p class="description" style="margin: 6px 0 12px;">
                        <?php esc_html_e('Pro unlocks AI tools, Analytics, and editing any date in the Calendar.','daily-tarot'); ?>
                    </p>
                    <p style="display:flex; gap:10px; align-items:center; flex-wrap:wrap; margin:0;">
                        <?php if ($buyUrl !== ''): ?>
                            <a class="button button-primary" href="<?php echo esc_url($buyUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Get a license','daily-tarot'); ?></a>
                        <?php endif; ?>
                        <?php if ($accountUrl !== ''): ?>
                            <a class="button" href="<?php echo esc_url($accountUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Manage subscription','daily-tarot'); ?></a>
                        <?php else: ?>
                            <a class="button" href="<?php echo esc_url(ProUrls::settingsProUrl()); ?>"><?php esc_html_e('Pro settings','daily-tarot'); ?></a>
                        <?php endif; ?>
                    </p>
                </div>
            <?php endif; ?>

            <?php if (class_exists(Ui::class)) { Ui::renderQuickActions('dashboard'); } ?>

            <div class="dtarot-dashboard-layout">
                <div class="dtarot-dashboard-left">
                    <div class="dtarot-card dtarot-today-card" data-dtarot-tour="today">
                        <div class="dtarot-card-header-row">
                            <div class="dtarot-card-title"><?php esc_html_e("Today's Energy",'daily-tarot'); ?></div>
                            <div class="dtarot-card-meta"><?php echo esc_html($todayPretty); ?></div>
                        </div>
                        <div class="dtarot-today-body">
                            <div class="dtarot-today-media">
                                <?php if ($todayImageUrl !== ''): ?>
                                    <img src="<?php echo esc_url($todayImageUrl); ?>" alt="" loading="lazy" />
                                <?php else: ?>
                                    <div class="dtarot-today-media-empty">*</div>
                                <?php endif; ?>
                            </div>
                            <div class="dtarot-today-content">
                                <h3 class="dtarot-today-title"><?php echo esc_html($todayCardName !== '' ? $todayCardName : __('(not set)','daily-tarot')); ?></h3>
                                <div class="dtarot-today-sub"><?php echo esc_html($todayDeckTitle !== '' ? $todayDeckTitle : __('Deck not selected','daily-tarot')); ?></div>
                                <div class="dtarot-today-status">
                                    <?php echo esc_html(sprintf(__('Status: %s','daily-tarot'), $todayStatus !== '' ? $todayStatus : __('(unknown)','daily-tarot'))); ?>
                                    <?php if ($todayMissing): ?>
                                        <span class="dtarot-today-warn"><?php echo esc_html(sprintf(__('Missing: %s','daily-tarot'), implode(', ', $todayMissing))); ?></span>
                                    <?php endif; ?>
                                </div>
                                <?php if ($todayExcerpt !== ''): ?>
                                    <div class="dtarot-today-excerpt"><?php echo esc_html($todayExcerpt); ?></div>
                                <?php endif; ?>
                                <div class="dtarot-actions">
                                    <?php if ($todayReadableUrl !== ''): ?>
                                        <a class="button button-primary" href="<?php echo esc_url($todayReadableUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('View Reading','daily-tarot'); ?></a>
                                    <?php endif; ?>
                                    <a class="button" href="<?php echo esc_url($calendarTodayUrl); ?>"><?php esc_html_e('Edit','daily-tarot'); ?></a>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="dtarot-card dtarot-schedule-card">
                        <div class="dtarot-card-header-row">
                            <div class="dtarot-card-title"><?php esc_html_e('Your Tarot Schedule','daily-tarot'); ?></div>
                            <div class="dtarot-card-meta"><?php echo esc_html($monthLabel); ?></div>
                        </div>
                        <div class="dtarot-schedule-actions">
                            <a class="dtarot-schedule-pill" href="<?php echo esc_url($calendarTodayUrl); ?>"><span class="dashicons dashicons-plus"></span><?php esc_html_e('Create New Reading','daily-tarot'); ?></a>
                            <a class="dtarot-schedule-pill" href="<?php echo esc_url($statusUrl); ?>"><span class="dashicons dashicons-archive"></span><?php esc_html_e('Manage Decks','daily-tarot'); ?></a>
                            <a class="dtarot-schedule-pill" href="<?php echo esc_url($calendarMonthUrl); ?>"><span class="dashicons dashicons-calendar-alt"></span><?php esc_html_e('Edit Schedule','daily-tarot'); ?></a>
                            <a class="dtarot-schedule-pill" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-spreads')); ?>"><span class="dashicons dashicons-grid-view"></span><?php esc_html_e('Spread Presets','daily-tarot'); ?></a>
                            <a class="dtarot-schedule-pill" href="<?php echo esc_url($helpUrl); ?>"><span class="dashicons dashicons-admin-generic"></span><?php esc_html_e('Settings','daily-tarot'); ?></a>
                        </div>
                    </div>

                    <div class="dtarot-card">
                        <div class="dtarot-dashboard-panel-header">
                            <div>
                                <h2 style="margin-top:0; margin-bottom:4px;"><?php esc_html_e('Calendar overview','daily-tarot'); ?></h2>
                                <p class="description" style="margin:0;"><?php echo esc_html($monthLabel); ?></p>
                            </div>
                            <a class="button" href="<?php echo esc_url($calendarMonthUrl); ?>"><?php esc_html_e('Manage Calendar','daily-tarot'); ?></a>
                        </div>

                        <div class="dtarot-calendar-overview">
                            <div class="dtarot-mini-cal" data-dtarot-tour="calendar" aria-label="<?php echo esc_attr($monthLabel); ?>">
                                <div class="dtarot-mini-cal-head" aria-hidden="true">
                                    <div><?php esc_html_e('Sun','daily-tarot'); ?></div>
                                    <div><?php esc_html_e('Mon','daily-tarot'); ?></div>
                                    <div><?php esc_html_e('Tue','daily-tarot'); ?></div>
                                    <div><?php esc_html_e('Wed','daily-tarot'); ?></div>
                                    <div><?php esc_html_e('Thu','daily-tarot'); ?></div>
                                    <div><?php esc_html_e('Fri','daily-tarot'); ?></div>
                                    <div><?php esc_html_e('Sat','daily-tarot'); ?></div>
                                </div>
                                <div class="dtarot-mini-cal-grid">
                                    <?php
                                    $cell = 0;
                                    for ($i = 0; $i < $firstWeekday; $i++) {
                                        echo '<div class="dtarot-mini-day dtarot-mini-day-empty" aria-hidden="true"></div>';
                                        $cell++;
                                    }

                                    for ($day = 1; $day <= $daysInMonth; $day++) {
                                        $date = sprintf('%04d-%02d-%02d', $monthYear, $monthNum, $day);
                                        $url = self::calendarEditUrl($date);

                                        $status = '';
                                        $missing = [];
                                        if (class_exists(DayEntryService::class)) {
                                            $e = DayEntryService::get($date);
                                            $status = property_exists($e, 'status') ? (string)$e->status : '';
                                            $missing = self::missingForEntryArray((array)$e->toArray());
                                        }

                                        $hasAny = false;
                                        if (is_array($missing) && count($missing) < 4) {
                                            $hasAny = true;
                                        }

                                        $cls = 'dtarot-mini-day';
                                        if ($date === $today) $cls .= ' dtarot-mini-day-today';
                                        if ($status === 'publish' && !$missing) $cls .= ' dtarot-mini-day-published';
                                        elseif ($missing && $hasAny) $cls .= ' dtarot-mini-day-issue';
                                        elseif ($hasAny) $cls .= ' dtarot-mini-day-draft';
                                        else $cls .= ' dtarot-mini-day-empty';

                                        echo '<a class="' . esc_attr($cls) . '" href="' . esc_url($url) . '">';
                                        echo '<span class="dtarot-mini-day-num">' . (int)$day . '</span>';
                                        echo '<span class="dtarot-mini-day-dot" aria-hidden="true"></span>';
                                        echo '</a>';
                                        $cell++;
                                    }

                                    while (($cell % 7) !== 0) {
                                        echo '<div class="dtarot-mini-day dtarot-mini-day-empty" aria-hidden="true"></div>';
                                        $cell++;
                                    }
                                    ?>
                                </div>
                            </div>

                            <div class="dtarot-calendar-overview-side">
                                <div class="dtarot-mini-legend" style="margin-top:0;">
                                    <div class="dtarot-mini-legend-item"><span class="dtarot-mini-legend-dot dtarot-mini-legend-dot-published" aria-hidden="true"></span> <?php esc_html_e('Published','daily-tarot'); ?></div>
                                    <div class="dtarot-mini-legend-item"><span class="dtarot-mini-legend-dot dtarot-mini-legend-dot-draft" aria-hidden="true"></span> <?php esc_html_e('Draft','daily-tarot'); ?></div>
                                    <div class="dtarot-mini-legend-item"><span class="dtarot-mini-legend-dot dtarot-mini-legend-dot-issue" aria-hidden="true"></span> <?php esc_html_e('Needs work','daily-tarot'); ?></div>
                                </div>
                                <div class="dtarot-actions" style="margin-top:10px;">
                                    <a class="button button-primary" href="<?php echo esc_url($calendarTodayUrl); ?>"><?php esc_html_e('Edit today','daily-tarot'); ?></a>
                                    <a class="button" href="<?php echo esc_url($statusUrl); ?>"><?php esc_html_e('Status','daily-tarot'); ?></a>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="dtarot-dashboard-right">
                    <?php if ($showOnboardCard): ?>
                        <div class="dtarot-card dtarot-onboard-card">
                            <div class="dtarot-card-header-row">
                                <div class="dtarot-card-title"><?php esc_html_e('Getting started','daily-tarot'); ?></div>
                                <div class="dtarot-card-meta"><?php echo esc_html($onboardComplete ? __('All set','daily-tarot') : __('Quick checklist','daily-tarot')); ?></div>
                            </div>
                            <div class="dtarot-onboard-checklist">
                                <div class="dtarot-onboard-item <?php echo $hasCard ? 'is-done' : ''; ?>" data-dtarot-tour-target="today">
                                    <span class="dtarot-onboard-dot"><?php echo $hasCard ? '&#10003;' : '&#8226;'; ?></span>
                                    <span class="dtarot-onboard-text"><?php esc_html_e('Set today\'s card','daily-tarot'); ?></span>
                                    <a class="dtarot-onboard-link" href="<?php echo esc_url($calendarTodayUrl); ?>"><?php esc_html_e('Edit','daily-tarot'); ?></a>
                                </div>
                                <div class="dtarot-onboard-item <?php echo $hasSpread ? 'is-done' : ''; ?>" data-dtarot-tour-target="spreads">
                                    <span class="dtarot-onboard-dot"><?php echo $hasSpread ? '&#10003;' : '&#8226;'; ?></span>
                                    <span class="dtarot-onboard-text"><?php esc_html_e('Add a spread preset','daily-tarot'); ?></span>
                                    <a class="dtarot-onboard-link" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-spreads')); ?>"><?php esc_html_e('Spreads','daily-tarot'); ?></a>
                                </div>
                                <div class="dtarot-onboard-item <?php echo $hasBooking ? 'is-done' : ''; ?>" data-dtarot-tour-target="bookings">
                                    <span class="dtarot-onboard-dot"><?php echo $hasBooking ? '&#10003;' : '&#8226;'; ?></span>
                                    <span class="dtarot-onboard-text"><?php esc_html_e('Set up bookings','daily-tarot'); ?></span>
                                    <a class="dtarot-onboard-link" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-bookings')); ?>"><?php esc_html_e('Bookings','daily-tarot'); ?></a>
                                </div>
                            </div>
                            <div class="dtarot-actions" style="margin-top:10px;">
                                <button type="button" class="button button-primary dtarot-onboard-start" data-dtarot-onboard-start="1"><?php esc_html_e('Start 1-minute tour','daily-tarot'); ?></button>
                                <?php if (!$onboardDismissed): ?>
                                    <button type="button" class="button-link dtarot-onboard-dismiss" data-dtarot-onboard-dismiss="1"><?php esc_html_e('Dismiss','daily-tarot'); ?></button>
                                <?php endif; ?>
                                <?php if ($onboardTourDone): ?>
                                    <span class="dtarot-onboard-meta"><?php esc_html_e('Tour completed','daily-tarot'); ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>

                    <div class="dtarot-card dtarot-attention-card">
                        <div class="dtarot-card-header-row">
                            <div class="dtarot-card-title"><?php esc_html_e('Your attention is needed','daily-tarot'); ?></div>
                        </div>
                        <div class="dtarot-attention-list">
                            <?php if (count($readinessIssues) > 0): ?>
                                <div class="dtarot-attention-item"><?php echo esc_html(sprintf(__('%d upcoming day(s) missing fields','daily-tarot'), count($readinessIssues))); ?></div>
                            <?php endif; ?>
                            <?php if ($unpublishedCount > 0): ?>
                                <div class="dtarot-attention-item"><?php echo esc_html(sprintf(__('%d unpublished day(s) in next 7 days','daily-tarot'), $unpublishedCount)); ?></div>
                            <?php endif; ?>
                            <?php if (!$autoEnabled): ?>
                                <div class="dtarot-attention-item"><?php esc_html_e('Automation is disabled','daily-tarot'); ?></div>
                            <?php endif; ?>
                            <?php if (count($readinessIssues) === 0 && $unpublishedCount === 0 && $autoEnabled): ?>
                                <div class="dtarot-attention-item dtarot-attention-ok"><?php esc_html_e('All clear - your schedule looks good.','daily-tarot'); ?></div>
                            <?php endif; ?>
                        </div>
                        <div class="dtarot-actions" style="margin-top:10px;">
                            <a class="button" href="<?php echo esc_url($settingsAutomationUrl); ?>"><?php esc_html_e('Manage Automation','daily-tarot'); ?></a>
                        </div>
                    </div>

                    <div class="dtarot-card dtarot-insights-card">
                        <div class="dtarot-card-header-row">
                            <div class="dtarot-card-title"><?php esc_html_e('Activity & Insights','daily-tarot'); ?></div>
                        </div>
                        <div class="dtarot-dashboard-list">
                            <div class="dtarot-dashboard-list-item">
                                <div class="dtarot-dashboard-list-title"><?php esc_html_e('Entries This Month','daily-tarot'); ?></div>
                                <div class="dtarot-dashboard-list-meta"><?php echo esc_html(sprintf(__('%d scheduled - %s to %s','daily-tarot'), (int)$entriesThisMonth, $monthStart, $monthEnd)); ?></div>
                            </div>
                            <div class="dtarot-dashboard-list-item">
                                <div class="dtarot-dashboard-list-title"><?php esc_html_e('Active Decks','daily-tarot'); ?></div>
                                <div class="dtarot-dashboard-list-meta"><?php echo esc_html(sprintf(__('%d decks available','daily-tarot'), (int)$activeDeckCount)); ?></div>
                            </div>
                            <div class="dtarot-dashboard-list-item">
                                <div class="dtarot-dashboard-list-title"><?php esc_html_e('Visitors Online','daily-tarot'); ?></div>
                                <div class="dtarot-dashboard-list-meta"><?php echo esc_html(sprintf(__('%d in the last %d minutes','daily-tarot'), (int)$onlineCount, $onlineMinutes)); ?></div>
                            </div>
                        </div>
                    </div>

                    <div class="dtarot-card dtarot-automation-card">
                        <div class="dtarot-card-header-row">
                            <div class="dtarot-card-title"><?php esc_html_e('Automation Status','daily-tarot'); ?></div>
                        </div>
                        <div class="dtarot-dashboard-list">
                            <div class="dtarot-dashboard-list-item">
                                <div class="dtarot-dashboard-list-title"><?php echo esc_html(sprintf(__('Enabled: %s','daily-tarot'), $autoEnabled ? __('Yes','daily-tarot') : __('No','daily-tarot'))); ?></div>
                                <div class="dtarot-dashboard-list-meta"><?php echo esc_html(sprintf(__('Next run: %s','daily-tarot'), $nextRunText)); ?></div>
                                <?php if ($autoTime !== ''): ?>
                                    <div class="dtarot-dashboard-list-meta"><?php echo esc_html(sprintf(__('Scheduled time: %s','daily-tarot'), $autoTime)); ?></div>
                                <?php endif; ?>
                            </div>
                            <?php if ($lastRunAt !== '' || $lastRunMsg !== ''): ?>
                                <div class="dtarot-dashboard-list-item">
                                    <div class="dtarot-dashboard-list-title"><?php esc_html_e('Last run','daily-tarot'); ?></div>
                                    <div class="dtarot-dashboard-list-meta">
                                        <?php
                                        $parts = [];
                                        if ($lastRunAt !== '') $parts[] = sprintf(__('At: %s','daily-tarot'), $lastRunAt);
                                        if ($lastRunMsg !== '') $parts[] = sprintf(__('Result: %s','daily-tarot'), $lastRunMsg);
                                        if ($lastRunOk === true) $parts[] = __('OK','daily-tarot');
                                        elseif ($lastRunOk === false) $parts[] = __('Error','daily-tarot');
                                        echo esc_html(implode(' - ', $parts));
                                        ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="dtarot-actions" style="margin-top:10px;">
                            <a class="button button-primary" href="<?php echo esc_url($settingsAutomationUrl); ?>"><?php esc_html_e('Enable / Edit Settings','daily-tarot'); ?></a>
                        </div>
                    </div>

                    <div class="dtarot-card" data-dtarot-tour="bookings">
                        <h2 style="margin-top:0;"><?php esc_html_e('Upcoming bookings','daily-tarot'); ?></h2>
                        <p class="description"><?php echo esc_html(sprintf(__('Next 7 days: %d','daily-tarot'), (int)$upcomingCount)); ?></p>
                        <?php if ($upcomingBookings): ?>
                            <div class="dtarot-dashboard-list">
                                <?php
                                $shown = 0;
                                foreach ($upcomingBookings as $booking) :
                                    if ($shown >= 3) break;
                                    $shown++;
                                    $when = function_exists('wp_date')
                                        ? (string)wp_date('M j, g:ia', (int)$booking['start'])
                                        : (string)date_i18n('M j, g:ia', (int)$booking['start']);
                                    $status = $booking['status'] !== '' ? $booking['status'] : 'pending';
                                    $editUrl = (string)add_query_arg([
                                        'post' => (int)$booking['id'],
                                        'action' => 'edit',
                                    ], admin_url('post.php'));
                                ?>
                                    <div class="dtarot-dashboard-list-item">
                                        <div class="dtarot-dashboard-list-title"><?php echo esc_html($booking['reading']); ?></div>
                                        <div class="dtarot-dashboard-list-meta"><?php echo esc_html($when); ?> - <?php echo esc_html(ucfirst($status)); ?></div>
                                        <div class="dtarot-dashboard-list-link"><a href="<?php echo esc_url($editUrl); ?>"><?php esc_html_e('Open','daily-tarot'); ?></a></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php else: ?>
                            <p class="description"><?php esc_html_e('No upcoming bookings found.','daily-tarot'); ?></p>
                        <?php endif; ?>
                        <div class="dtarot-actions" style="margin-top:10px;">
                            <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-bookings')); ?>"><?php esc_html_e('Bookings','daily-tarot'); ?></a>
                        </div>
                    </div>

                    <div class="dtarot-card" data-dtarot-tour="spreads">
                        <h2 style="margin-top:0;"><?php esc_html_e('Top spreads','daily-tarot'); ?></h2>
                        <p class="description"><?php esc_html_e('Most configured spread presets on your site.','daily-tarot'); ?></p>
                        <?php if ($topSpreads): ?>
                            <div class="dtarot-dashboard-list">
                                <?php foreach ($topSpreads as $presetId => $count) :
                                    $label = isset($presetLabels[$presetId]['label']) ? (string)$presetLabels[$presetId]['label'] : $presetId;
                                ?>
                                    <div class="dtarot-dashboard-list-item">
                                        <div class="dtarot-dashboard-list-title"><?php echo esc_html($label); ?></div>
                                        <div class="dtarot-dashboard-list-meta"><?php echo esc_html(sprintf(__('%d placements','daily-tarot'), (int)$count)); ?></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php else: ?>
                            <p class="description"><?php esc_html_e('No spreads configured yet.','daily-tarot'); ?></p>
                        <?php endif; ?>
                        <div class="dtarot-actions" style="margin-top:10px;">
                            <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-spreads')); ?>"><?php esc_html_e('Spreads','daily-tarot'); ?></a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
<?php
    }
}













