<?php
declare(strict_types=1);


namespace DailyTarot\Admin\Pages;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Registry\Cards;
use DailyTarot\Support\DefaultDecks;
use DailyTarot\Support\DefaultMeaningPacks;
use DailyTarot\Support\PostTypes;
use DailyTarot\Support\StarterDecks;
use DailyTarot\Admin\StarterDecksPrompt;
use DailyTarot\Admin\Backup;

use DailyTarot\Calendar\DayEntryService;

final class Content {

    /**
     * @return array{complete:int,incomplete:int,total:int,percent:int}
     */
    private static function meaningPackProgress(int $packId): array {
        $system = Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true));
        $registry = Cards::forSystem($system);
        $total = count((array)$registry);

        $meanings = get_post_meta($packId, '_dtarot_meanings', true);
        if (!is_array($meanings)) $meanings = [];

        $complete = 0;
        foreach (array_keys((array)$registry) as $cardId) {
            $m = isset($meanings[$cardId]) && is_array($meanings[$cardId]) ? $meanings[$cardId] : [];
            $hasUpright = !empty($m['upright']);
            if ($hasUpright) {
                $complete++;
            }
        }

        $incomplete = max(0, $total - $complete);
        $percent = $total > 0 ? (int)round(($complete / $total) * 100) : 0;

        return [
            'complete' => $complete,
            'incomplete' => $incomplete,
            'total' => $total,
            'percent' => max(0, min(100, $percent)),
        ];
    }

    private static function paginateLinks(string $tab, string $pageKey, int $current, int $totalPages, array $extraArgs = []): string {
        if ($totalPages <= 1) return '';
        $baseArgs = array_merge([
            'page' => 'daily-tarot-content',
            'tab' => $tab,
        ], $extraArgs);

        $base = add_query_arg(array_merge($baseArgs, [$pageKey => '%#%']), admin_url('admin.php'));
        return (string)paginate_links([
            'base' => $base,
            'format' => '',
            'current' => max(1, $current),
            'total' => max(1, $totalPages),
            'type' => 'plain',
        ]);
    }

    public static function render(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));

        if (class_exists(StarterDecksPrompt::class)) {
            StarterDecksPrompt::markSeen();
        }

        // Import notices (deck/pack ZIP import reports).
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only query args used only for showing an admin notice.
        $backupStatus = isset($_GET['dtarot_backup']) ? sanitize_key((string)wp_unslash($_GET['dtarot_backup'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only query args used only for showing an admin notice.
        $reportToken = isset($_GET['report']) ? sanitize_text_field((string)wp_unslash($_GET['report'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only query args used only for showing an admin notice.
        $deckId = isset($_GET['deck_id']) ? absint(wp_unslash($_GET['deck_id'])) : 0;

        $tab = isset($_GET['tab']) ? sanitize_key((string)wp_unslash($_GET['tab'])) : 'status';
        $tabs = [
            'status'   => __('Status','daily-tarot'),
            'cards'    => __('Cards','daily-tarot'),
            'decks'    => __('Decks','daily-tarot'),
            'meanings' => __('Meaning Packs','daily-tarot'),
        ];

        if (!isset($tabs[$tab])) {
            $tab = 'status';
        }
        ?>
        <div class="wrap dtarot-wrap" data-dtarot-content-page="1">
            <h1><?php esc_html_e('Content','daily-tarot'); ?></h1>

            <div class="notice notice-info" style="padding:10px 12px;">
                <p style="margin:0;">
                    <strong><?php esc_html_e('How it fits together:','daily-tarot'); ?></strong>
                    <?php esc_html_e('Decks store images → Cards assigns images to fixed IDs → Meaning Packs → Calendar publishes daily entries.','daily-tarot'); ?>
                </p>
            </div>

            <?php
            $userId = function_exists('get_current_user_id') ? (int)get_current_user_id() : 0;
            $onboardDismissed = $userId > 0 ? (bool)get_user_meta($userId, 'dtarot_onboard_dismissed', true) : false;
            if (!$onboardDismissed):
            ?>
                <div class="dtarot-onboard-banner" data-dtarot-onboard-banner="1">
                    <div class="dtarot-onboard-banner-text">
                        <?php esc_html_e('New here? Set your first card of the day, then explore spreads and bookings.','daily-tarot'); ?>
                    </div>
                    <div class="dtarot-onboard-banner-actions">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-calendar')); ?>" class="button"><?php esc_html_e('Go to Calendar','daily-tarot'); ?></a>
                        <button type="button" class="button-link dtarot-onboard-dismiss" data-dtarot-onboard-dismiss="1"><?php esc_html_e('Dismiss','daily-tarot'); ?></button>
                    </div>
                </div>
            <?php endif; ?>

            <?php
            $report = null;
            if ($reportToken !== '' && class_exists(Backup::class)) {
                $report = Backup::readReport($reportToken);
            }

            if ($backupStatus !== ''):
                $ok = is_array($report) ? !empty($report['ok']) : ($backupStatus === 'deck_zip_ok' || $backupStatus === 'pack_zip_ok');
                if ($ok):
            ?>
                    <div class="notice notice-success is-dismissible">
                        <p>
                            <?php
                            if (is_array($report) && isset($report['title'])) {
                                echo esc_html(sprintf(__('Imported: %s','daily-tarot'), (string)$report['title']));
                            } else {
                                esc_html_e('Import completed.','daily-tarot');
                            }

                            $did = 0;
                            if (is_array($report) && !empty($report['deck_id'])) {
                                $did = absint($report['deck_id']);
                            } elseif ($deckId > 0) {
                                $did = $deckId;
                            }
                            if ($did > 0 && function_exists('get_edit_post_link')) {
                                $edit = (string)get_edit_post_link($did, '');
                                if ($edit !== '') {
                                    echo ' ';
                                    echo '<a href="' . esc_url($edit) . '">' . esc_html__('Edit deck','daily-tarot') . '</a>';
                                }
                            }

                            if (is_array($report) && !empty($report['default_set'])) {
                                echo ' ';
                                echo esc_html__('Set as default deck.','daily-tarot');
                            }
                            ?>
                        </p>

                        <?php if (is_array($report) && isset($report['imported_images'], $report['expected_images'])):
                            $imported = absint($report['imported_images']);
                            $expected = absint($report['expected_images']);
                        ?>
                            <p class="description">
                                <?php
                                echo esc_html(sprintf(
                                    __('Images imported: %d / %d','daily-tarot'),
                                    $imported,
                                    $expected
                                ));
                                ?>
                            </p>
                        <?php endif; ?>

                        <?php if (is_array($report) && !empty($report['errors']) && is_array($report['errors'])):
                            $errs = array_slice($report['errors'], 0, 5);
                        ?>
                            <p class="description">
                                <?php echo esc_html(sprintf(__('Warnings: %d','daily-tarot'), count($errs))); ?>
                            </p>
                        <?php endif; ?>
                        <p>
                            <a class="button button-primary" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-calendar')); ?>"><?php esc_html_e('Go to Calendar','daily-tarot'); ?></a>
                            <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-content&tab=decks')); ?>"><?php esc_html_e('View Decks','daily-tarot'); ?></a>
                        </p>
                    </div>
                <?php else: ?>
                    <div class="notice notice-error is-dismissible">
                        <p><?php esc_html_e('Import failed. Please try again or check the import report in Settings → Backup.','daily-tarot'); ?></p>
                    </div>
                <?php endif; ?>
            <?php endif; ?>

            <?php if (class_exists(Ui::class)) { Ui::renderQuickActions($tab === 'status' ? 'status' : 'content'); } ?>

            <h2 class="nav-tab-wrapper">
                <?php foreach ($tabs as $k => $label): ?>
                    <a class="nav-tab <?php echo $tab===$k?'nav-tab-active':''; ?>"
                       href="<?php echo esc_url(add_query_arg(['page'=>'daily-tarot-content','tab'=>$k], admin_url('admin.php'))); ?>">
                        <?php echo esc_html($label); ?>
                    </a>
                <?php endforeach; ?>
            </h2>

            <?php self::render_panel($tab); ?>
        </div>
        <?php
    }

    public static function render_panel(string $tab): void {
        $tab = sanitize_key($tab);
        $allowed = ['status','cards','decks','meanings'];
        if (!in_array($tab, $allowed, true)) {
            $tab = 'status';
        }

        ?>
        <div id="dtarot-content-panel" data-tab="<?php echo esc_attr($tab); ?>">
            <?php if ($tab !== 'cards'): ?>
                <div class="dtarot-card">
                    <?php if ($tab === 'status'): ?>
                        <p><?php esc_html_e('Quick health check for decks (missing images/back) and meaning packs (missing meanings).','daily-tarot'); ?></p>
                        <p>
                            <?php
                            $hasDecks = false;
                            foreach (PostTypes::deckTypes() as $deckType) {
                                $counts = wp_count_posts((string)$deckType);
                                if (is_object($counts)) {
                                    $p = isset($counts->publish) ? (int)$counts->publish : 0;
                                    $d = isset($counts->draft) ? (int)$counts->draft : 0;
                                    $hasDecks = ($p + $d) > 0;
                                }
                                if ($hasDecks) break;
                            }
                            $showStarter = !$hasDecks && class_exists(StarterDecksPrompt::class) && !StarterDecksPrompt::isDismissed();
                            if ($showStarter):
                                $dismissUrl = wp_nonce_url(
                                    admin_url('admin-post.php?action=dtarot_starter_decks_dismiss'),
                                    'dtarot_starter_decks_dismiss'
                                );
                                $decks = class_exists(StarterDecks::class) ? StarterDecks::all() : [];
                            ?>
                                <div style="display:flex; align-items:center; justify-content:space-between; gap:12px; flex-wrap:wrap;">
                                    <strong><?php esc_html_e('Free starter decks','daily-tarot'); ?></strong>
                                    <a class="button button-small" href="<?php echo esc_url($dismissUrl); ?>"><?php esc_html_e('Dismiss','daily-tarot'); ?></a>
                                </div>
                                <p class="description" style="margin:8px 0 10px;">
                                    <?php esc_html_e('Import a starter deck hosted on WordPress.org to get up and running in minutes.','daily-tarot'); ?>
                                </p>
                                <div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap:12px;">
                                    <?php foreach ((array)$decks as $deck):
                                        if (!is_array($deck)) continue;
                                        $title = isset($deck['title']) ? (string)$deck['title'] : '';
                                        $desc = isset($deck['description']) ? (string)$deck['description'] : '';
                                        $slug = isset($deck['slug']) ? (string)$deck['slug'] : '';
                                        $cards = isset($deck['cards']) ? (int)$deck['cards'] : 0;
                                        $sizeBytes = isset($deck['size_bytes']) ? (int)$deck['size_bytes'] : 0;
                                        $zipUrl = StarterDecks::zipUrl($deck);
                                        $imgUrl = StarterDecks::backPreviewUrl($deck);
                                        if ($zipUrl === '') continue;

                                        $installed = false;
                                        $installedDeckId = 0;
                                        if (class_exists(Backup::class)) {
                                            $packs = Backup::installedPacks();
                                            $k = 'deck:' . sanitize_title($slug);
                                            if (isset($packs[$k]) && is_array($packs[$k])) {
                                                $installed = true;
                                                $installedDeckId = isset($packs[$k]['post_id']) ? absint($packs[$k]['post_id']) : 0;
                                            }
                                        }
                                        if ($installedDeckId <= 0 && $slug !== '') {
                                            $installedDeckId = StarterDecks::findDeckIdBySlug($slug);
                                        }

                                        $metaBits = [];
                                        if ($cards > 0) $metaBits[] = sprintf(_n('%d card','%d cards',$cards,'daily-tarot'), $cards);
                                        if ($sizeBytes > 0 && function_exists('size_format')) $metaBits[] = size_format($sizeBytes);
                                        $metaLine = $metaBits ? implode(' • ', $metaBits) : '';
                                    ?>
                                        <div style="border:1px solid #e5e7eb; background:#fff; border-radius:10px; padding:12px; display:flex; gap:12px; align-items:center;">
                                            <div style="width:56px; height:78px; border-radius:8px; overflow:hidden; background:#f3f4f6; flex:0 0 auto; display:flex; align-items:center; justify-content:center;">
                                                <?php if ($imgUrl !== ''): ?>
                                                    <img src="<?php echo esc_url($imgUrl); ?>" alt="" loading="lazy" decoding="async" width="56" height="78" style="width:56px; height:78px; object-fit:cover; display:block;" />
                                                <?php else: ?>
                                                    <span style="color:#6b7280;">—</span>
                                                <?php endif; ?>
                                            </div>
                                            <div style="min-width:0; flex:1;">
                                                <div style="font-weight:600; margin-bottom:2px;">
                                                    <?php echo esc_html($title !== '' ? $title : __('Starter Deck','daily-tarot')); ?>
                                                </div>
                                                <?php if ($installed): ?>
                                                    <div class="description" style="margin-bottom:6px; color:#0f766e;">
                                                        <?php esc_html_e('Installed','daily-tarot'); ?>
                                                    </div>
                                                <?php endif; ?>
                                                <?php if ($metaLine !== ''): ?>
                                                    <div class="description" style="margin-bottom:6px;">
                                                        <?php echo esc_html($metaLine); ?>
                                                    </div>
                                                <?php endif; ?>
                                                <?php if ($desc !== ''): ?>
                                                    <div class="description" style="margin-bottom:8px;">
                                                        <?php echo esc_html($desc); ?>
                                                    </div>
                                                <?php endif; ?>

                                                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin:0;">
                                                    <input type="hidden" name="action" value="dtarot_import_deck_zip_url" />
                                                    <?php wp_nonce_field('dtarot_backup'); ?>
                                                    <input type="hidden" name="redirect_to" value="<?php echo esc_attr(admin_url('admin.php?page=daily-tarot-content&tab=status')); ?>" />
                                                    <input type="hidden" name="starter" value="1" />
                                                    <input type="hidden" name="zip_url" value="<?php echo esc_attr($zipUrl); ?>" />
                                                    <input type="hidden" name="overwrite" value="1" />
                                                    <input type="hidden" name="set_default" value="1" />
                                                    <button type="submit" class="button button-primary">
                                                        <?php echo esc_html($installed ? __('Reinstall','daily-tarot') : __('Import free deck','daily-tarot')); ?>
                                                    </button>

                                                    <?php if ($installedDeckId > 0 && function_exists('get_edit_post_link')):
                                                        $editUrl = (string)get_edit_post_link($installedDeckId, '');
                                                        if ($editUrl !== ''):
                                                    ?>
                                                        <a class="button" style="margin-left:6px;" href="<?php echo esc_url($editUrl); ?>"><?php esc_html_e('Edit','daily-tarot'); ?></a>
                                                    <?php endif; endif; ?>
                                                </form>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </p>
                    <?php elseif ($tab === 'decks'): ?>
                        <p><?php esc_html_e('Decks are visual skins. Each deck can have a card back image and per-card images.','daily-tarot'); ?></p>
                        <p>
                            <a class="button button-primary" href="<?php echo esc_url(admin_url('post-new.php?post_type=' . PostTypes::DECK)); ?>"><?php esc_html_e('Add Deck','daily-tarot'); ?></a>
                        </p>
                    <?php else: ?>
                        <p><?php esc_html_e('Meaning Packs store interpretations for each fixed card identity.','daily-tarot'); ?></p>
                        <p>
                            <a class="button button-primary" href="<?php echo esc_url(admin_url('post-new.php?post_type=' . PostTypes::MEANING_PACK)); ?>"><?php esc_html_e('Add Meaning Pack','daily-tarot'); ?></a>
                        </p>
                    <?php endif; ?>
                </div>
            <?php endif; ?>

            <?php
            if ($tab === 'status') self::render_status();
            elseif ($tab === 'decks') self::render_decks();
            elseif ($tab === 'meanings') self::render_meanings();
            else self::render_cards();
            ?>
        </div>
        <?php
    }

    private static function render_status(): void {
        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
        ]);

        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
        ]);

        $deckIssues = [];
        foreach ((array)$decks as $deck) {
            if (!is_object($deck) || empty($deck->ID)) continue;
            $deckId = (int)$deck->ID;

            $system = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));
            $registry = Cards::forSystem($system);

            $imgs = get_post_meta($deckId, '_dtarot_cards', true);
            if (!is_array($imgs)) $imgs = [];

            $missing = 0;
            foreach (array_keys((array)$registry) as $cardId) {
                $has = false;
                foreach (Cards::kipperGypsyAliases((string)$cardId) as $id) {
                    $url = $imgs[$id] ?? '';
                    if (is_string($url) && trim($url) !== '') {
                        $has = true;
                        break;
                    }
                }
                if (!$has) $missing++;
            }

            $back = (string)get_post_meta($deckId, '_dtarot_back', true);
            $missingBack = trim($back) === '';

            if ($missing > 0 || $missingBack) {
                $deckIssues[] = [
                    'id' => $deckId,
                    'title' => (string)($deck->post_title ?: __('(no title)','daily-tarot')),
                    'system' => $system,
                    'missing_cards' => $missing,
                    'total_cards' => count((array)$registry),
                    'missing_back' => $missingBack,
                ];
            }
        }

        $packIssues = [];
        foreach ((array)$packs as $pack) {
            if (!is_object($pack) || empty($pack->ID)) continue;
            $packId = (int)$pack->ID;

            $system = Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true));
            $registry = Cards::forSystem($system);

            $meanings = get_post_meta($packId, '_dtarot_meanings', true);
            if (!is_array($meanings)) $meanings = [];

            $missing = 0;
            foreach (array_keys((array)$registry) as $cardId) {
                $m = isset($meanings[$cardId]) && is_array($meanings[$cardId]) ? $meanings[$cardId] : [];
                $hasUpright = !empty($m['upright']);
                if (!$hasUpright) {
                    $missing++;
                }
            }

            if ($missing > 0) {
                $packIssues[] = [
                    'id' => $packId,
                    'title' => (string)($pack->post_title ?: __('(no title)','daily-tarot')),
                    'system' => $system,
                    'missing_cards' => $missing,
                    'total_cards' => count((array)$registry),
                ];
            }
        }

        ?>
        <div class="dtarot-card">
            <h2 style="margin-top:0;"><?php esc_html_e('Deck images','daily-tarot'); ?></h2>
            <p class="description"><?php echo esc_html(sprintf(__('Decks with missing card images (or missing back). Found: %d.','daily-tarot'), count($deckIssues))); ?></p>

            <?php if (!$deckIssues): ?>
                <p><?php esc_html_e('All decks look complete.','daily-tarot'); ?></p>
            <?php else: ?>
                <table class="widefat striped">
                    <thead><tr>
                        <th><?php esc_html_e('Deck','daily-tarot'); ?></th>
                        <th><?php esc_html_e('System','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Missing card images','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Back image','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Action','daily-tarot'); ?></th>
                    </tr></thead>
                    <tbody>
                    <?php foreach ($deckIssues as $row):
                        $editUrl = function_exists('get_edit_post_link') ? (string)get_edit_post_link((int)$row['id'], '') : '';
                    ?>
                        <tr>
                            <td><?php echo esc_html((string)$row['title']); ?></td>
                            <td><?php echo esc_html((string)(Cards::systems()[(string)$row['system']] ?? (string)$row['system'])); ?></td>
                            <td><?php echo esc_html((string)$row['missing_cards'] . '/' . (string)$row['total_cards']); ?></td>
                            <td>
                                <?php if (!empty($row['missing_back'])): ?>
                                    <span style="color:#b45309;"><?php esc_html_e('Missing','daily-tarot'); ?></span>
                                <?php else: ?>
                                    <span style="color:green;"><?php esc_html_e('OK','daily-tarot'); ?></span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($editUrl !== ''): ?>
                                    <a class="button button-small" href="<?php echo esc_url($editUrl); ?>"><?php esc_html_e('Edit deck','daily-tarot'); ?></a>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>

        <div class="dtarot-card">
            <h2 style="margin-top:0;"><?php esc_html_e('Meaning packs','daily-tarot'); ?></h2>
            <p class="description"><?php echo esc_html(sprintf(__('Packs with missing required meanings (primary meaning). Found: %d.','daily-tarot'), count($packIssues))); ?></p>

            <?php if (!$packIssues): ?>
                <p><?php esc_html_e('All meaning packs look complete.','daily-tarot'); ?></p>
            <?php else: ?>
                <table class="widefat striped">
                    <thead><tr>
                        <th><?php esc_html_e('Meaning pack','daily-tarot'); ?></th>
                        <th><?php esc_html_e('System','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Missing cards','daily-tarot'); ?></th>
                        <th><?php esc_html_e('Action','daily-tarot'); ?></th>
                    </tr></thead>
                    <tbody>
                    <?php foreach ($packIssues as $row):
                        $editUrl = function_exists('get_edit_post_link') ? (string)get_edit_post_link((int)$row['id'], '') : '';
                    ?>
                        <tr>
                            <td><?php echo esc_html((string)$row['title']); ?></td>
                            <td><?php echo esc_html((string)(Cards::systems()[(string)$row['system']] ?? (string)$row['system'])); ?></td>
                            <td><?php echo esc_html((string)$row['missing_cards'] . '/' . (string)$row['total_cards']); ?></td>
                            <td>
                                <?php if ($editUrl !== ''): ?>
                                    <a class="button button-small" href="<?php echo esc_url($editUrl); ?>"><?php esc_html_e('Edit pack','daily-tarot'); ?></a>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php
    }

    private static function render_decks(): void {
        $pageKey = 'decks_page';
        $paged = isset($_GET[$pageKey]) ? max(1, absint(wp_unslash($_GET[$pageKey]))) : 1;
        $perPage = 20;
        $q = new \WP_Query([
            'post_type' => PostTypes::deckTypes(),
            'post_status' => ['publish','draft'],
            'posts_per_page' => $perPage,
            'paged' => $paged,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);
        $decks = $q->posts;
        $pager = self::paginateLinks('decks', $pageKey, $paged, (int)$q->max_num_pages);
        ?>
        <?php if ($pager): ?>
            <p class="tablenav-pages" style="margin:10px 0;"><?php echo $pager; ?></p>
        <?php endif; ?>

        <table class="widefat striped">
            <thead><tr>
                <th><?php esc_html_e('Deck','daily-tarot'); ?></th>
                <th style="width:92px;"><?php esc_html_e('Preview','daily-tarot'); ?></th>
                <th><?php esc_html_e('Image URL','daily-tarot'); ?></th>
                <th style="width:160px;"><?php esc_html_e('Default','daily-tarot'); ?></th>
                <th><?php esc_html_e('Actions','daily-tarot'); ?></th>
            </tr></thead>
            <tbody>
            <?php if (!$decks): ?>
                <tr><td colspan="5"><?php esc_html_e('No decks yet. Create one.','daily-tarot'); ?></td></tr>
            <?php else: foreach ($decks as $d):
                $back = (string)get_post_meta($d->ID, '_dtarot_back', true);
                $system = Cards::normalizeSystem((string)get_post_meta($d->ID, '_dtarot_system', true));
                $systems = Cards::systems();
                $systemLabel = ($system !== '' && isset($systems[$system])) ? (string)$systems[$system] : '';
                $defaultId = $system !== '' ? DefaultDecks::get($system) : 0;
                $isDefault = ($defaultId > 0 && $defaultId === (int)$d->ID);

                // Preview: back image, fallback to first available card image.
                $preview = $back;
                if ($preview === '') {
                    $imgs = get_post_meta($d->ID, '_dtarot_cards', true);
                    if (is_array($imgs)) {
                        foreach ($imgs as $u) {
                            if (is_string($u) && trim($u) !== '') {
                                $preview = $u;
                                break;
                            }
                            if (is_numeric($u) && trim((string)$u) !== '') {
                                $preview = (string)$u;
                                break;
                            }
                        }
                    }
                }
                ?>
                <tr>
                    <td><strong><?php echo esc_html($d->post_title ?: __('(no title)','daily-tarot')); ?></strong></td>
                    <td>
                        <div class="dtarot-card-preview">
                            <?php if ($preview): ?>
                                <img class="dtarot-card-preview-img" src="<?php echo esc_url($preview); ?>" alt="" />
                            <?php else: ?>
                                <div class="dtarot-card-preview-empty">—</div>
                            <?php endif; ?>
                        </div>
                    </td>
                    <td>
                        <div class="dtarot-row">
                            <input type="text" class="dtarot-image-url" value="<?php echo esc_attr($back); ?>"
                                   placeholder="https://"
                                   data-deck-id="<?php echo esc_attr((string)$d->ID); ?>"
                                   data-card-id="__back" />
                            <button type="button" class="button dtarot-pick" data-deck-id="<?php echo esc_attr((string)$d->ID); ?>" data-card-id="__back">
                                <?php esc_html_e('Choose','daily-tarot'); ?>
                            </button>
                            <button type="button" class="button button-primary dtarot-save" data-deck-id="<?php echo esc_attr((string)$d->ID); ?>" data-card-id="__back">
                                <?php esc_html_e('Save','daily-tarot'); ?>
                            </button>
                        </div>
                        <div class="description dtarot-row-msg"></div>
                    </td>
                    <td>
                        <div class="dtarot-admin-default-deck">
                            <?php if ($systemLabel !== ''): ?>
                                <div class="dtarot-admin-default-deck-system"><?php echo esc_html($systemLabel); ?></div>
                            <?php endif; ?>

                            <?php if ($isDefault): ?>
                                <span class="dashicons dashicons-yes" aria-hidden="true"></span>
                                <span><?php esc_html_e('Selected','daily-tarot'); ?></span>
                            <?php else: ?>
                                <?php if ($system !== ''):
                                    $url = add_query_arg(
                                        [
                                            'action' => 'dtarot_set_default_deck',
                                            'deck_id' => (string)$d->ID,
                                        ],
                                        admin_url('admin-post.php')
                                    );
                                    $url = wp_nonce_url($url, 'dtarot_set_default_deck_' . (int)$d->ID);
                                ?>
                                    <a class="button button-small" href="<?php echo esc_url($url); ?>"><?php esc_html_e('Set Default','daily-tarot'); ?></a>
                                <?php else: ?>
                                    <span class="description"><?php esc_html_e('Set system first','daily-tarot'); ?></span>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </td>
                    <td><a class="button button-small" href="<?php echo esc_url(admin_url('post.php?post='.$d->ID.'&action=edit')); ?>"><?php esc_html_e('Edit','daily-tarot'); ?></a></td>
                </tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <?php if ($pager): ?>
            <p class="tablenav-pages" style="margin:10px 0;"><?php echo $pager; ?></p>
        <?php endif; ?>
        <?php
    }

    private static function render_meanings(): void {
        $pageKey = 'packs_page';
        $paged = isset($_GET[$pageKey]) ? max(1, absint(wp_unslash($_GET[$pageKey]))) : 1;
        $perPage = 20;
        $q = new \WP_Query([
            'post_type' => PostTypes::meaningPackTypes(),
            'post_status' => ['publish','draft'],
            'posts_per_page' => $perPage,
            'paged' => $paged,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);
        $packs = $q->posts;
        $pager = self::paginateLinks('meanings', $pageKey, $paged, (int)$q->max_num_pages);
        ?>
        <?php if ($pager): ?>
            <p class="tablenav-pages" style="margin:10px 0;"><?php echo $pager; ?></p>
        <?php endif; ?>

        <table class="widefat striped">
            <thead><tr>
                <th><?php esc_html_e('Pack','daily-tarot'); ?></th>
                <th style="width:160px;"><?php esc_html_e('Default','daily-tarot'); ?></th>
                <th><?php esc_html_e('Progress','daily-tarot'); ?></th>
                <th><?php esc_html_e('Status','daily-tarot'); ?></th>
                <th><?php esc_html_e('Actions','daily-tarot'); ?></th>
            </tr></thead>
            <tbody>
            <?php if (!$packs): ?>
                <tr><td colspan="5"><?php esc_html_e('No meaning packs yet. Create one.','daily-tarot'); ?></td></tr>
            <?php else: foreach ($packs as $p): ?>
                <?php $progress = self::meaningPackProgress((int)$p->ID); ?>
                <?php
                    $system = Cards::normalizeSystem((string)get_post_meta((int)$p->ID, '_dtarot_system', true));
                    $systems = Cards::systems();
                    $systemLabel = ($system !== '' && isset($systems[$system])) ? (string)$systems[$system] : '';
                    $defaultId = $system !== '' ? DefaultMeaningPacks::get($system) : 0;
                    $isDefault = ($defaultId > 0 && $defaultId === (int)$p->ID);
                ?>
                <tr>
                    <td><strong><?php echo esc_html($p->post_title ?: __('(no title)','daily-tarot')); ?></strong></td>
                    <td>
                        <div class="dtarot-admin-default-pack">
                            <?php if ($systemLabel !== ''): ?>
                                <div class="dtarot-admin-default-pack-system"><?php echo esc_html($systemLabel); ?></div>
                            <?php endif; ?>

                            <?php if ($isDefault): ?>
                                <span class="dashicons dashicons-yes" aria-hidden="true"></span>
                                <span><?php esc_html_e('Selected','daily-tarot'); ?></span>
                            <?php else: ?>
                                <?php if ($system !== ''):
                                    $url = add_query_arg(
                                        [
                                            'action' => 'dtarot_set_default_pack',
                                            'pack_id' => (string)$p->ID,
                                        ],
                                        admin_url('admin-post.php')
                                    );
                                    $url = wp_nonce_url($url, 'dtarot_set_default_pack_' . (int)$p->ID);
                                ?>
                                    <a class="button button-small" href="<?php echo esc_url($url); ?>"><?php esc_html_e('Set Default','daily-tarot'); ?></a>
                                <?php else: ?>
                                    <span class="description"><?php esc_html_e('Set system first','daily-tarot'); ?></span>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </td>
                    <td>
                        <div class="dtarot-pack-progressbar" role="img" aria-label="<?php echo esc_attr(sprintf(__('%d of %d complete','daily-tarot'), (int)$progress['complete'], (int)$progress['total'])); ?>">
                            <div class="dtarot-pack-progressbar-fill" style="width: <?php echo esc_attr((string)$progress['percent']); ?>%;"></div>
                        </div>
                        <div class="dtarot-pack-progressbar-meta">
                            <?php echo esc_html(sprintf(__('%d complete / %d incomplete','daily-tarot'), (int)$progress['complete'], (int)$progress['incomplete'])); ?>
                        </div>
                    </td>
                    <td><?php echo esc_html(ucfirst((string)$p->post_status)); ?></td>
                    <td><a class="button button-small" href="<?php echo esc_url(admin_url('post.php?post='.$p->ID.'&action=edit')); ?>"><?php esc_html_e('Edit','daily-tarot'); ?></a></td>
                </tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <?php if ($pager): ?>
            <p class="tablenav-pages" style="margin:10px 0;"><?php echo $pager; ?></p>
        <?php endif; ?>
        <?php
    }

    private static function render_cards(): void {
        $deckPageKey = 'deck_page';
        $deckPage = isset($_GET[$deckPageKey]) ? max(1, absint(wp_unslash($_GET[$deckPageKey]))) : 1;
        $deckPerPage = 50;

        $q = new \WP_Query([
            'post_type' => PostTypes::deckTypes(),
            'post_status' => ['publish','draft'],
            'posts_per_page' => $deckPerPage,
            'paged' => $deckPage,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);
        $decks = $q->posts;
        $deckPager = self::paginateLinks('cards', $deckPageKey, $deckPage, (int)$q->max_num_pages);

        $deck_id = isset($_GET['deck_id']) ? absint(wp_unslash($_GET['deck_id'])) : (int)($decks[0]->ID ?? 0);

        if (!$deck_id) {
            echo '<div class="notice notice-warning"><p>';
            echo esc_html__('Create a deck first (a deck stores your card images).','daily-tarot') . ' ';
            echo '<a href="' . esc_url(admin_url('post-new.php?post_type=' . PostTypes::DECK)) . '">' . esc_html__('Create deck','daily-tarot') . '</a>';
            echo '</p></div>';
            return;
        }

        $system = Cards::normalizeSystem((string)get_post_meta($deck_id, '_dtarot_system', true));
        $cards = Cards::forSystem($system);

        $images = get_post_meta($deck_id, '_dtarot_cards', true);
        if (!is_array($images)) $images = [];

        ?>
        <p class="description"><?php esc_html_e('Select a deck and upload a custom image for each fixed card.','daily-tarot'); ?></p>

        <div class="dtarot-toolbar">
            <label class="dtarot-search">
                <span><?php esc_html_e('Search','daily-tarot'); ?></span>
                <input type="search" id="dtarot-card-search" placeholder="<?php echo esc_attr__('Type to filter cards…','daily-tarot'); ?>" />
            </label>

            <div class="dtarot-row" style="gap:8px; align-items:center; margin-left:auto;">
                <button type="button" class="button" id="dtarot-bulk-upload" data-deck-id="<?php echo esc_attr((string)$deck_id); ?>">
                    <?php esc_html_e('Bulk upload','daily-tarot'); ?>
                </button>
                <span class="description" style="max-width:520px;">
                    <?php esc_html_e('After selecting images, you can map each image to a card (filenames do not need to match).','daily-tarot'); ?>
                </span>
                <span class="description dtarot-row-msg" id="dtarot-bulk-msg"></span>
            </div>
        </div>

        <form method="get" style="margin:12px 0;">
            <input type="hidden" name="page" value="daily-tarot-content">
            <input type="hidden" name="tab" value="cards">
            <input type="hidden" name="<?php echo esc_attr($deckPageKey); ?>" value="<?php echo esc_attr((string)$deckPage); ?>">
            <label>
                <?php esc_html_e('Deck','daily-tarot'); ?>
                <select name="deck_id" onchange="this.form.submit()">
                    <?php
                        // If a deck is selected via URL but isn't on this page, include it.
                        $hasSelected = false;
                        foreach ($decks as $d) {
                            if ((int)$d->ID === (int)$deck_id) { $hasSelected = true; break; }
                        }
                        if (!$hasSelected && $deck_id > 0) {
                            $p = get_post($deck_id);
                            if ($p && is_object($p) && isset($p->post_type) && PostTypes::isDeckType((string)$p->post_type)) {
                                echo '<option value="' . esc_attr((string)$deck_id) . '" selected>' . esc_html($p->post_title ?: __('(no title)','daily-tarot')) . '</option>';
                            }
                        }
                    ?>
                    <?php foreach ($decks as $d): ?>
                        <option value="<?php echo esc_attr((string)$d->ID); ?>" <?php selected($deck_id, $d->ID); ?>>
                            <?php echo esc_html($d->post_title ?: __('(no title)','daily-tarot')); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </label>
        </form>

        <?php if ($deckPager): ?>
            <p class="tablenav-pages" style="margin:10px 0;"><?php echo $deckPager; ?></p>
        <?php endif; ?>

        <table class="widefat striped dtarot-cards-table" id="dtarot-cards-table">
            <thead>
                <tr>
                    <th style="width:320px;"><?php esc_html_e('Card','daily-tarot'); ?></th>
                    <th style="width:92px;"><?php esc_html_e('Preview','daily-tarot'); ?></th>
                    <th><?php esc_html_e('Image','daily-tarot'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($cards as $id => $name):
                    $url = '';
                    foreach (Cards::kipperGypsyAliases((string)$id) as $aid) {
                        $u = $images[$aid] ?? '';
                        if (is_string($u) && trim($u) !== '') {
                            $url = (string)$u;
                            break;
                        }
                    }
                ?>
                <tr class="dtarot-card-row" data-card-name="<?php echo esc_attr(strtolower($name)); ?>" data-card-id="<?php echo esc_attr(strtolower($id)); ?>">
                    <td><strong><?php echo esc_html($name); ?></strong><br><code><?php echo esc_html($id); ?></code></td>
                    <td>
                        <div class="dtarot-card-preview">
                            <?php if ($url): ?>
                                <img class="dtarot-card-preview-img" src="<?php echo esc_url($url); ?>" alt="" />
                            <?php else: ?>
                                <div class="dtarot-card-preview-empty">—</div>
                            <?php endif; ?>
                        </div>
                    </td>
                    <td>
                        <div class="dtarot-row">
                            <input type="text" class="dtarot-image-url" value="<?php echo esc_attr($url); ?>"
                                   placeholder="https://"
                                   data-deck-id="<?php echo esc_attr((string)$deck_id); ?>"
                                   data-card-id="<?php echo esc_attr($id); ?>"/>
                            <button type="button" class="button dtarot-pick" data-deck-id="<?php echo esc_attr((string)$deck_id); ?>" data-card-id="<?php echo esc_attr($id); ?>">
                                <?php esc_html_e('Choose','daily-tarot'); ?>
                            </button>
                            <button type="button" class="button button-primary dtarot-save" data-deck-id="<?php echo esc_attr((string)$deck_id); ?>" data-card-id="<?php echo esc_attr($id); ?>">
                                <?php esc_html_e('Save','daily-tarot'); ?>
                            </button>
                        </div>
                        <div class="description dtarot-row-msg"></div>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <div class="dtarot-modal" id="dtarot-bulk-map-modal" style="display:none;">
            <div class="dtarot-modal-backdrop"></div>
            <div class="dtarot-modal-panel" style="width: min(980px, calc(100% - 24px));">
                <div class="dtarot-modal-header">
                    <h3 style="margin:0;"><?php esc_html_e('Bulk map images to cards','daily-tarot'); ?></h3>
                    <button type="button" class="dtarot-modal-close" aria-label="<?php echo esc_attr(__('Close','daily-tarot')); ?>">×</button>
                </div>
                <p class="description" style="margin-top:0;">
                    <?php esc_html_e('For each selected image, choose which card it belongs to. Then click Save.','daily-tarot'); ?>
                </p>

                <div id="dtarot-bulk-map-list"></div>

                <div class="dtarot-modal-footer">
                    <button type="button" class="button" id="dtarot-bulk-map-cancel"><?php esc_html_e('Cancel','daily-tarot'); ?></button>
                    <button type="button" class="button button-primary" id="dtarot-bulk-map-save"><?php esc_html_e('Save mappings','daily-tarot'); ?></button>
                    <span class="description dtarot-row-msg" id="dtarot-bulk-map-status"></span>
                </div>
            </div>
        </div>

        <div class="dtarot-modal" id="dtarot-bulk-zoom-modal" style="display:none;">
            <div class="dtarot-modal-backdrop"></div>
            <div class="dtarot-modal-panel" style="width: min(980px, calc(100% - 24px));">
                <div class="dtarot-modal-header">
                    <h3 style="margin:0;"><?php esc_html_e('Preview','daily-tarot'); ?></h3>
                    <button type="button" class="dtarot-modal-close" aria-label="<?php echo esc_attr(__('Close','daily-tarot')); ?>">×</button>
                </div>
                <div style="display:flex; align-items:center; justify-content:center;">
                    <img id="dtarot-bulk-zoom-img" src="" alt="" style="max-width:100%; height:auto;" />
                </div>
            </div>
        </div>
        <?php
    }
}
