<?php
declare(strict_types=1);


namespace DailyTarot\Admin\Pages;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.DateTime.RestrictedFunctions.date_date

// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Calendar\PublishTimes;
use DailyTarot\Registry\Cards;
use DailyTarot\Meaning\DefaultMeaningPack;
use DailyTarot\Support\FeatureFlags;
use DailyTarot\Support\PlanLimits;
use DailyTarot\Support\PostTypes;
use DailyTarot\Support\ProUrls;

final class Calendar {

    public static function render(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));

        // If a jump date is provided, redirect to the correct month and deep-link the modal.
        $jump = isset($_GET['dtarot_jump_date']) ? sanitize_text_field((string)wp_unslash($_GET['dtarot_jump_date'])) : '';
        if ($jump !== '' && preg_match('/^\d{4}-\d{2}-\d{2}$/', $jump)) {
            $jy = (int)substr($jump, 0, 4);
            $jm = (int)substr($jump, 5, 2);
            if ($jy > 1970 && $jm >= 1 && $jm <= 12) {
                wp_safe_redirect(add_query_arg([
                    'page' => 'daily-tarot-calendar',
                    'year' => (string)$jy,
                    'month' => (string)$jm,
                    'dtarot_edit_date' => $jump,
                ], admin_url('admin.php')));
                exit;
            }
        }

        $year  = isset($_GET['year']) ? (int)wp_unslash($_GET['year']): (int)date('Y');
        $month = isset($_GET['month']) ? (int)wp_unslash($_GET['month']): (int)date('n');
        $prev = strtotime('-1 month', strtotime("$year-$month-01"));
        $next = strtotime('+1 month', strtotime("$year-$month-01"));

        $first = mktime(0,0,0,$month,1,$year);
        $days  = (int)date('t',$first);
        $start = (int)date('N',$first);

        // Use WP timezone-aware formatting.
        $today = function_exists('wp_date') ? (string)wp_date('Y-m-d') : (string)current_time('Y-m-d');

        $isPro = class_exists(FeatureFlags::class) && FeatureFlags::enabled('pro');
        $allowedFreeDate = class_exists(PlanLimits::class) ? PlanLimits::calendarAllowedFreeEditDate() : $today;

        $buyUrl = class_exists(ProUrls::class) ? trim((string)ProUrls::buyProUrl()) : '';
        $accountUrl = class_exists(ProUrls::class) ? trim((string)ProUrls::accountUrl()) : '';

        $editPrimaryDate = $today;
        $editPrimaryUrl = add_query_arg(['dtarot_edit_date' => $editPrimaryDate], admin_url('admin.php?page=daily-tarot-calendar'));

        $decks = get_posts(['post_type'=>PostTypes::deckTypes(),'numberposts'=>-1,'post_status'=>['publish','draft']]);
        $packs = get_posts(['post_type'=>PostTypes::meaningPackTypes(),'numberposts'=>-1,'post_status'=>['publish','draft']]);

        $missingDeck = (!is_array($decks) || count($decks) === 0);
        $missingPack = (!is_array($packs) || count($packs) === 0);
        $missingSetup = ($missingDeck || $missingPack);

        $monthPublishTimeOverrides = [];
        if ($isPro && class_exists(PublishTimes::class)) {
            $monthPublishTimeOverrides = PublishTimes::overridesForMonth($year, $month);
        }

        ?>
        <div class="wrap dtarot-wrap">
            <h1><?php esc_html_e('Daily Tarot Calendar','daily-tarot'); ?></h1>

            <div class="notice notice-info" style="padding:10px 12px;">
                <p style="margin:0;">
                    <strong><?php esc_html_e('How publishing works:','daily-tarot'); ?></strong>
                    <?php esc_html_e('Each day uses a Deck (images) + Meaning Pack (text) + a chosen Card.','daily-tarot'); ?>
                </p>
                <p style="margin:8px 0 0 0; display:flex; gap:8px; flex-wrap:wrap;">
                    <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-content&tab=cards')); ?>"><?php esc_html_e('Upload card images','daily-tarot'); ?></a>
                    <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-content&tab=meanings')); ?>"><?php esc_html_e('Edit meaning packs','daily-tarot'); ?></a>
                </p>
            </div>

            <?php if (class_exists(Ui::class)) { Ui::renderQuickActions('calendar'); } ?>

            <?php if ($missingSetup): ?>
                <div class="notice notice-warning" style="padding:10px 12px;">
                    <p style="margin:0;">
                        <?php esc_html_e('Before you can publish days, create at least one Deck and one Meaning Pack (for the same system).','daily-tarot'); ?>
                    </p>
                    <p style="margin:8px 0 0 0; display:flex; gap:8px; flex-wrap:wrap;">
                        <?php if ($missingDeck): ?>
                            <a class="button" href="<?php echo esc_url(admin_url('post-new.php?post_type=' . PostTypes::DECK)); ?>"><?php esc_html_e('Create deck','daily-tarot'); ?></a>
                        <?php endif; ?>
                        <?php if ($missingPack): ?>
                            <a class="button" href="<?php echo esc_url(admin_url('post-new.php?post_type=' . PostTypes::MEANING_PACK)); ?>"><?php esc_html_e('Create meaning pack','daily-tarot'); ?></a>
                        <?php endif; ?>
                    </p>
                </div>
            <?php endif; ?>

            <?php
            $userId = function_exists('get_current_user_id') ? (int)get_current_user_id() : 0;
            $onboardDismissed = $userId > 0 ? (bool)get_user_meta($userId, 'dtarot_onboard_dismissed', true) : false;
            if (!$onboardDismissed && !$missingSetup):
            ?>
                <div class="dtarot-onboard-banner" data-dtarot-onboard-banner="1">
                    <div class="dtarot-onboard-banner-text">
                        <?php esc_html_e('New here? Set your first card of the day, then explore spreads and bookings.','daily-tarot'); ?>
                    </div>
                    <div class="dtarot-onboard-banner-actions">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-spreads')); ?>" class="button"><?php esc_html_e('Explore Spreads','daily-tarot'); ?></a>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=daily-tarot-bookings')); ?>" class="button"><?php esc_html_e('Set up Bookings','daily-tarot'); ?></a>
                        <button type="button" class="button-link dtarot-onboard-dismiss" data-dtarot-onboard-dismiss="1"><?php esc_html_e('Dismiss','daily-tarot'); ?></button>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (!$isPro): ?>
                <div class="notice notice-warning">
                    <p>
                        <?php echo esc_html(sprintf(__('Free plan: you can edit days up to and including tomorrow (%s).','daily-tarot'), $allowedFreeDate)); ?>
                        <?php if ($buyUrl !== ''): ?>
                            <a href="<?php echo esc_url($buyUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Get a license','daily-tarot'); ?></a>
                        <?php endif; ?>
                        <?php if ($accountUrl !== ''): ?>
                            <a href="<?php echo esc_url($accountUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Manage subscription','daily-tarot'); ?></a>
                        <?php else: ?>
                            <a href="<?php echo esc_url(ProUrls::settingsProUrl()); ?>"><?php esc_html_e('Pro settings','daily-tarot'); ?></a>
                        <?php endif; ?>
                    </p>
                </div>
            <?php endif; ?>

            <?php
                $ptNotice = isset($_GET['dtarot_publish_time']) ? sanitize_key((string)wp_unslash($_GET['dtarot_publish_time'])) : '';
                if ($ptNotice === 'ok') {
                    echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Publish time settings saved.','daily-tarot') . '</p></div>';
                }
            ?>

            <div class="dtarot-cal-nav">
                <a class="button" href="<?php echo esc_url(add_query_arg(['year'=>date('Y',$prev),'month'=>date('n',$prev)])); ?>">← <?php esc_html_e('Prev','daily-tarot'); ?></a>
                <strong><?php echo esc_html(date_i18n('F Y', $first)); ?></strong>
                <a class="button" href="<?php echo esc_url(add_query_arg(['year'=>date('Y',$next),'month'=>date('n',$next)])); ?>"><?php esc_html_e('Next','daily-tarot'); ?> →</a>

                <span style="flex:1;"></span>

                <a class="button button-primary" href="<?php echo esc_url($editPrimaryUrl); ?>"><?php echo esc_html(__('Edit today','daily-tarot')); ?></a>

                <form method="get" style="display:flex; gap:8px; align-items:center; margin:0;">
                    <input type="hidden" name="page" value="daily-tarot-calendar" />
                    <label style="display:flex; gap:8px; align-items:center;">
                        <span class="screen-reader-text"><?php esc_html_e('Jump to date','daily-tarot'); ?></span>
                        <input type="date" name="dtarot_jump_date" value="" />
                    </label>
                    <button type="submit" class="button"><?php esc_html_e('Jump','daily-tarot'); ?></button>
                </form>
            </div>

            <div class="dtarot-calendar" data-max-date="<?php echo esc_attr($allowedFreeDate); ?>" data-allowed-date="<?php echo esc_attr($allowedFreeDate); ?>" data-is-pro="<?php echo esc_attr($isPro ? '1' : '0'); ?>">
                <?php for ($i=1; $i<$start; $i++) echo '<div class="dtarot-empty"></div>'; ?>

                <?php for ($d=1; $d<=$days; $d++):
                    $date = sprintf('%04d-%02d-%02d',$year,$month,$d);
                    $e = DayEntryService::get($date);
                    $isSet = !$e->isEmpty();
                    $status = (string)$e->status;
                    $cardId = (string)$e->cardId;
                    $isToday = ($date === $today);
                    $isLocked = (!$isPro && $date > $allowedFreeDate);
                    $thumb  = '';
                    // Prefer per-day custom image override (does not affect deck mapping).
                    if (!empty($e->imageOverrideUrl) && is_string($e->imageOverrideUrl)) {
                        $thumb = esc_url((string)$e->imageOverrideUrl);
                    } elseif ($cardId && $e->deckId > 0) {
                        $imgs = get_post_meta((int)$e->deckId, '_dtarot_cards', true);
                        if (is_array($imgs) && $imgs) {
                            foreach (Cards::kipperGypsyAliases($cardId) as $id) {
                                if (empty($imgs[$id]) || !is_string($imgs[$id])) continue;
                                $u = trim((string)$imgs[$id]);
                                if ($u === '') continue;
                                $thumb = esc_url($u);
                                break;
                            }
                        }
                    }
                ?>
                    <div class="dtarot-day dtarot-day-cell <?php echo $status==='publish'?'dtarot-filled':''; ?> <?php echo $isToday?'dtarot-today':''; ?> <?php echo $isSet?'dtarot-is-set':''; ?> <?php echo $isLocked?'dtarot-locked':''; ?>" 
                         data-date="<?php echo esc_attr($date); ?>" data-locked="<?php echo esc_attr($isLocked ? '1' : '0'); ?>">
                        <strong><?php echo esc_html((string)$d); ?></strong>

                        <?php if ($isPro && !empty($monthPublishTimeOverrides[$date])): ?>
                            <span class="dtarot-publish-time-badge" title="<?php echo esc_attr(__('Publish time override','daily-tarot')); ?>">
                                <?php echo esc_html((string)$monthPublishTimeOverrides[$date]); ?>
                            </span>
                        <?php endif; ?>

                        <?php if ($thumb): ?>
                            <div class="dtarot-thumb">
                                <img src="<?php echo $thumb; ?>" alt="" />
                            </div>
                        <?php endif; ?>

                        <span class="dtarot-status"><?php echo esc_html($isSet ? ucfirst($status) : __('Not set','daily-tarot')); ?></span>

                        <?php if ($cardId !== ''): ?>
                            <div class="dtarot-card-name"><?php echo esc_html(Cards::name($cardId)); ?></div>
                        <?php endif; ?>
                    </div>
                <?php endfor; ?>
            </div>

            <?php if ($isPro): ?>
                <?php
                    $fallbackTime = '06:00';
                    if (class_exists('DailyTarot\\Automation\\AutomationSettings')) {
                        $auto = \DailyTarot\Automation\AutomationSettings::get();
                        $fallbackTime = isset($auto['time']) ? (string)$auto['time'] : $fallbackTime;
                    }
                    $defaultTime = class_exists(PublishTimes::class) ? PublishTimes::getDefaultTime($fallbackTime) : $fallbackTime;
                    $overrideCount = class_exists(PublishTimes::class) ? PublishTimes::countOverridesForMonth($year, $month) : 0;

                    $automationEnabled = false;
                    if (class_exists('DailyTarot\\Automation\\AutomationSettings')) {
                        $automationEnabled = (bool)\DailyTarot\Automation\AutomationSettings::isEnabled();
                    }
                    $automationUrl = admin_url('admin.php?page=daily-tarot-ai&tab=automation');
                ?>
                <div class="dtarot-card" style="max-width: 980px; margin-top: 14px;">
                    <h2 style="margin-top:0; margin-bottom:6px;"><?php esc_html_e('Publish time (Pro)','daily-tarot'); ?></h2>
                    <p class="description" style="margin-top:0;">
                        <?php esc_html_e('Sets the time the daily automation job publishes the Card of the Day. Default applies to all days; you can override a specific day in its modal.','daily-tarot'); ?>
                    </p>
                    <p class="description" style="margin-top:0;">
                        <strong><?php esc_html_e('Automation status','daily-tarot'); ?>:</strong>
                        <?php echo $automationEnabled ? esc_html__('Enabled','daily-tarot') : esc_html__('Disabled','daily-tarot'); ?>
                        &nbsp;|&nbsp;
                        <a href="<?php echo esc_url($automationUrl); ?>"><?php esc_html_e('Open Automation settings','daily-tarot'); ?></a>
                    </p>

                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:flex; gap:10px; align-items:flex-end; flex-wrap:wrap;">
                        <?php wp_nonce_field('dtarot_calendar_publish_times'); ?>
                        <input type="hidden" name="action" value="dtarot_calendar_publish_times_save" />
                        <input type="hidden" name="year" value="<?php echo esc_attr((string)$year); ?>" />
                        <input type="hidden" name="month" value="<?php echo esc_attr((string)$month); ?>" />

                        <label>
                            <div style="font-weight:600; margin-bottom:4px;"><?php esc_html_e('Default publish time','daily-tarot'); ?></div>
                            <input type="time" name="default_time" value="<?php echo esc_attr($defaultTime); ?>" />
                        </label>

                        <button type="submit" class="button button-primary"><?php esc_html_e('Save publish time','daily-tarot'); ?></button>

                        <label style="margin-left:10px;">
                            <input type="checkbox" name="clear_month_overrides" value="1" />
                            <?php
                                /* translators: %d is number of per-day overrides in the current month */
                                echo esc_html(sprintf(__('Clear per-day overrides for this month (%d)','daily-tarot'), (int)$overrideCount));
                            ?>
                        </label>
                    </form>

                    <p class="description" style="margin-bottom:0;">
                        <?php esc_html_e('Per-day override: click a day, then set “Publish time (override)” in the modal. Leave it blank to use the default.','daily-tarot'); ?>
                    </p>
                </div>
            <?php else: ?>
                <div class="dtarot-card" style="max-width: 980px; margin-top: 14px;">
                    <h2 style="margin-top:0; margin-bottom:6px;"><?php esc_html_e('Publish time (Oracle / Pro)','daily-tarot'); ?></h2>
                    <p class="description" style="margin-top:0;">
                        <?php esc_html_e('Setting a default publish time and per-day publish time overrides is available for Oracle (Pro) users.','daily-tarot'); ?>
                    </p>
                    <p style="display:flex; gap:10px; align-items:center; flex-wrap:wrap; margin:0;">
                        <?php if ($buyUrl !== ''): ?>
                            <a class="button button-primary" href="<?php echo esc_url($buyUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Get a license','daily-tarot'); ?></a>
                        <?php endif; ?>
                        <?php if ($accountUrl !== ''): ?>
                            <a class="button" href="<?php echo esc_url($accountUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Manage subscription','daily-tarot'); ?></a>
                        <?php else: ?>
                            <a class="button" href="<?php echo esc_url(ProUrls::settingsProUrl()); ?>"><?php esc_html_e('Pro settings','daily-tarot'); ?></a>
                        <?php endif; ?>
                    </p>
                </div>
            <?php endif; ?>
        </div>

        <?php self::modal(); ?>
        <?php
    }

    private static function modal(): void {
        $isPro = class_exists(FeatureFlags::class) && FeatureFlags::enabled('pro');
        $decks = get_posts(['post_type'=>PostTypes::deckTypes(),'numberposts'=>-1,'post_status'=>['publish','draft']]);
        $packs = get_posts(['post_type'=>PostTypes::meaningPackTypes(),'numberposts'=>-1,'post_status'=>['publish','draft']]);
        $systems = Cards::systems();
        ?>
        <div id="dtarot-day-modal" class="dtarot-modal" style="display:none;">
            <div class="dtarot-modal-backdrop"></div>
            <div class="dtarot-modal-panel">
                <div class="dtarot-modal-header">
                    <strong id="dtarot-modal-title"></strong>
                    <button type="button" class="dtarot-modal-close">×</button>
                </div>
                <div class="dtarot-modal-body">
                    <div class="dtarot-modal-layout">
                        <div class="dtarot-modal-left">
                            <div class="dtarot-form-stack">
                                <p><label><?php esc_html_e('Deck','daily-tarot'); ?></label>
                                    <select id="dtarot-deck"><option value=""><?php esc_html_e('— Select —','daily-tarot'); ?></option>
                                        <?php foreach ($decks as $d): ?>
                                            <?php $sys = Cards::normalizeSystem((string)get_post_meta((int)$d->ID, '_dtarot_system', true)); ?>
                                            <option value="<?php echo esc_attr((string)$d->ID); ?>" data-system="<?php echo esc_attr($sys); ?>"><?php echo esc_html($d->post_title ?: __('(no title)','daily-tarot')); ?></option>
                                        <?php endforeach; ?>
                                    </select></p>

                                <p><label><?php esc_html_e('Card','daily-tarot'); ?></label>
                                    <select id="dtarot-card"><option value=""><?php esc_html_e('— Select —','daily-tarot'); ?></option>
                                        <?php foreach ($systems as $systemId => $systemLabel): ?>
                                            <?php foreach (Cards::forSystem($systemId) as $id=>$name): ?>
                                                <option value="<?php echo esc_attr($id); ?>" data-system="<?php echo esc_attr($systemId); ?>"><?php echo esc_html($name); ?></option>
                                            <?php endforeach; ?>
                                        <?php endforeach; ?>
                                    </select></p>

                                <p><label><?php esc_html_e('Meaning Pack','daily-tarot'); ?></label>
                                    <select id="dtarot-pack"><option value=""><?php esc_html_e('— Select —','daily-tarot'); ?></option>
                                        <?php foreach ($packs as $p): ?>
                                            <?php
                                                $label = $p->post_title ?: __('(no title)','daily-tarot');
                                                $sys = Cards::normalizeSystem((string)get_post_meta((int)$p->ID, '_dtarot_system', true));
                                                if (class_exists(DefaultMeaningPack::class) && DefaultMeaningPack::isDefault((int)$p->ID)) {
                                                    $label .= ' ' . __('(Default)','daily-tarot');
                                                }
                                            ?>
                                            <option value="<?php echo esc_attr((string)$p->ID); ?>" data-system="<?php echo esc_attr($sys); ?>"><?php echo esc_html($label); ?></option>
                                        <?php endforeach; ?>
                                    </select></p>

                                <p><label><?php esc_html_e('Status','daily-tarot'); ?></label>
                                    <select id="dtarot-status">
                                        <option value="draft"><?php esc_html_e('Draft','daily-tarot'); ?></option>
                                        <option value="publish"><?php esc_html_e('Publish','daily-tarot'); ?></option>
                                    </select></p>

                                <?php if ($isPro): ?>
                                    <p>
                                        <label><?php esc_html_e('Publish time (override)','daily-tarot'); ?></label>
                                        <input type="time" id="dtarot-publish-time" value="" />
                                        <span class="description"><?php esc_html_e('Optional. Overrides the default publish time for this date only.','daily-tarot'); ?></span>
                                    </p>
                                <?php else: ?>
                                    <p>
                                        <label><?php esc_html_e('Publish time (Oracle / Pro)','daily-tarot'); ?></label>
                                        <input type="time" id="dtarot-publish-time" value="" disabled />
                                        <span class="description">
                                            <?php esc_html_e('Per-day publish time overrides are available for Oracle (Pro) users.','daily-tarot'); ?>
                                            <?php if (!empty($buyUrl)) : ?>
                                                <a href="<?php echo esc_url($buyUrl); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e('Get a license','daily-tarot'); ?></a>
                                            <?php endif; ?>
                                        </span>
                                    </p>
                                <?php endif; ?>
                            </div>

                            <div class="dtarot-editor-field">
                                <div class="dtarot-field-label"><?php esc_html_e('Content','daily-tarot'); ?></div>
                                <?php wp_editor('', 'dtarot-content', ['textarea_rows'=>7, 'media_buttons' => false]); ?>
                            </div>

                            <div class="dtarot-editor-field">
                                <div class="dtarot-field-label"><?php esc_html_e('Daily Text (expanded)','daily-tarot'); ?></div>
                                <?php wp_editor('', 'dtarot_daily_text', ['textarea_rows'=>7, 'media_buttons' => false]); ?>
                            </div>
                        </div>

                        <div class="dtarot-modal-right">
                            <div class="dtarot-day-preview" aria-live="polite">
                                <div class="dtarot-day-preview-box">
                                    <img id="dtarot-day-preview-img" alt="" style="display:none;" />
                                    <div id="dtarot-day-preview-empty" class="description"><?php esc_html_e('Select a deck and card to preview the image.','daily-tarot'); ?></div>
                                </div>

                                <div class="dtarot-day-preview-actions" style="margin-top:10px; display:flex; gap:10px; align-items:center; flex-wrap:wrap;">
                                    <button type="button" class="button" id="dtarot-day-upload-image"><?php esc_html_e('Upload custom image','daily-tarot'); ?></button>
                                    <span id="dtarot-day-upload-note" class="description" style="display:none;"></span>
                                </div>
                            </div>

                            <div class="dtarot-day-link-helper" id="dtarot-day-link-helper">
                                <div class="dtarot-field-label"><?php esc_html_e('Internal links','daily-tarot'); ?></div>
                                <div class="dtarot-day-link-helper-controls">
                                    <button type="button" class="button" id="dtarot-day-link-refresh"><?php esc_html_e('Auto-find posts','daily-tarot'); ?></button>
                                </div>

                                <div id="dtarot-day-link-status" class="description" style="margin-top:8px;"></div>
                                <div id="dtarot-day-link-suggestions" class="dtarot-day-link-list"></div>

                                <div class="dtarot-day-link-search" style="margin-top:12px;">
                                    <div class="dtarot-field-label" style="margin-bottom:6px;"><?php esc_html_e('Search posts/pages','daily-tarot'); ?></div>
                                    <div style="display:flex; gap:8px; align-items:center; flex-wrap:wrap;">
                                        <input type="text" id="dtarot-day-link-q" placeholder="<?php echo esc_attr__('Search…','daily-tarot'); ?>" style="width: 100%;" />
                                        <button type="button" class="button" id="dtarot-day-link-search"><?php esc_html_e('Search','daily-tarot'); ?></button>
                                    </div>
                                    <div id="dtarot-day-link-results" class="dtarot-day-link-list"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="dtarot-modal-footer">
                    <button type="button" class="button" id="dtarot-reuse-old-text"><?php esc_html_e('Reuse old text','daily-tarot'); ?></button>
                    <button type="button" class="button" id="dtarot-copy-prev"><?php esc_html_e('Copy previous day','daily-tarot'); ?></button>
                    <button type="button" class="button button-primary" id="dtarot-save-day"><?php esc_html_e('Save','daily-tarot'); ?></button>
                    <span id="dtarot-ready-note" class="dtarot-ready-note" aria-live="polite"></span>
                    <span id="dtarot-save-note" class="dtarot-save-note" aria-live="polite"></span>
                </div>
            </div>
        </div>
        <?php
    }
}
