<?php
declare(strict_types=1);


namespace DailyTarot\Admin\Pages;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Support\PostTypes;

final class Bookings {

    private const TAB_REQUESTS = 'requests';
    private const TAB_TYPES = 'types';
    private const TAB_OPTIONS = 'options';

    public static function render(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }

        $tab = isset($_GET['tab']) ? sanitize_key((string)wp_unslash($_GET['tab'])) : self::TAB_REQUESTS;
        if (!in_array($tab, [self::TAB_REQUESTS, self::TAB_TYPES, self::TAB_OPTIONS], true)) {
            $tab = self::TAB_REQUESTS;
        }

        $title = __('Bookings','daily-tarot');
        $postType = ($tab === self::TAB_TYPES) ? PostTypes::READING_TYPE : PostTypes::BOOKING;
        ?>
        <div class="wrap dtarot-wrap dtarot-bookings-wrap">
            <h1><?php echo esc_html($title); ?></h1>
            <h2 class="nav-tab-wrapper">
                <?php
                $tabs = [
                    self::TAB_REQUESTS => __('Requests','daily-tarot'),
                    self::TAB_TYPES => __('Reading Types','daily-tarot'),
                    self::TAB_OPTIONS => __('Options','daily-tarot'),
                ];
                foreach ($tabs as $id => $label) {
                    $url = add_query_arg(['page' => 'daily-tarot-bookings', 'tab' => $id], admin_url('admin.php'));
                    $cls = 'nav-tab' . (($tab === $id) ? ' nav-tab-active' : '');
                    echo '<a class="' . esc_attr($cls) . '" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
                }
                ?>
            </h2>

            <div class="dtarot-bookings-summary">
                <div class="dtarot-bookings-card">
                    <div class="dtarot-bookings-card-title"><?php esc_html_e('Requests','daily-tarot'); ?></div>
                    <div class="dtarot-bookings-card-text"><?php esc_html_e('Review, approve, and manage incoming booking requests.','daily-tarot'); ?></div>
                </div>
                <div class="dtarot-bookings-card">
                    <div class="dtarot-bookings-card-title"><?php esc_html_e('Reading Types','daily-tarot'); ?></div>
                    <div class="dtarot-bookings-card-text"><?php esc_html_e('Create your reading packages with duration, price, and description.','daily-tarot'); ?></div>
                </div>
            </div>

            <?php
            if ($tab === self::TAB_OPTIONS) {
                self::renderOptions();
            } else {
                echo '<div class="dtarot-bookings-help">';
                echo '<strong>' . esc_html__('Quick help:','daily-tarot') . '</strong> ';
                echo '<code>[dtarot_booking]</code> ';
                echo '<span class="description">' . esc_html__('Full booking form (readings + times + contact). Supports `style="modern|mystic|minimal"`.','daily-tarot') . '</span>';
                echo '<br />';
                echo '<code>[dtarot_booking_button]</code> ';
                echo '<span class="description">' . esc_html__('Popup button that opens the booking form. Supports `style="modern|mystic|minimal"`.','daily-tarot') . '</span>';
                echo '<br />';
                echo '<code>[dtarot_booking_teaser]</code> ';
                echo '<span class="description">' . esc_html__('Compact teaser card that opens the booking form. Supports `title`, `text`, `label`, and `style`.','daily-tarot') . '</span>';
                echo '<div class="description" style="margin-top:6px;">' . esc_html__('Tip: You can also insert these via the Booking Gutenberg block with a variant + style selector.','daily-tarot') . '</div>';
                echo '</div>';
                self::renderListTable($postType, $tab);
            }
            ?>
        </div>
        <?php
    }

    private static function renderListTable(string $postType, string $tab): void {
        if (!function_exists('get_current_screen')) {
            require_once ABSPATH . 'wp-admin/includes/screen.php';
        }
        require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-posts-list-table.php';

        $screenId = 'edit-' . $postType;
        $screen = function_exists('convert_to_screen') ? convert_to_screen($screenId) : null;

        $oldPostType = isset($_GET['post_type']) ? sanitize_key((string)wp_unslash($_GET['post_type'])) : '';
        $_GET['post_type'] = $postType;

        $listTable = new \WP_Posts_List_Table([
            'screen' => $screen,
        ]);
        $listTable->prepare_items();

        $addUrl = admin_url('post-new.php?post_type=' . $postType);
        $addLabel = ($postType === PostTypes::READING_TYPE) ? __('Add Reading Type','daily-tarot') : __('Add Booking','daily-tarot');
        ?>
        <div style="margin-top:12px;">
            <a href="<?php echo esc_url($addUrl); ?>" class="page-title-action"><?php echo esc_html($addLabel); ?></a>
        </div>

        <form method="get">
            <input type="hidden" name="page" value="daily-tarot-bookings" />
            <input type="hidden" name="tab" value="<?php echo esc_attr($tab); ?>" />
            <?php $listTable->search_box(__('Search','daily-tarot'), 'dtarot-bookings'); ?>
            <?php $listTable->views(); ?>
            <?php $listTable->display(); ?>
        </form>
        <?php

        if ($oldPostType === '') {
            unset($_GET['post_type']);
        } else {
            $_GET['post_type'] = $oldPostType;
        }
    }

    private static function renderOptions(): void {
        if (!class_exists(\DailyTarot\Support\BookingSettings::class)) {
            echo '<p>' . esc_html__('Booking settings are unavailable (missing plugin files).','daily-tarot') . '</p>';
            return;
        }

        $s = \DailyTarot\Support\BookingSettings::get();
        $isPro = class_exists(\DailyTarot\Support\FeatureFlags::class) && \DailyTarot\Support\FeatureFlags::enabled('pro');
        $notice = isset($_GET['dtarot_booking']) ? sanitize_key((string)wp_unslash($_GET['dtarot_booking'])) : '';
        if ($notice === 'saved') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Booking options saved.','daily-tarot') . '</p></div>';
        }
        ?>
        <h2><?php esc_html_e('Payment options','daily-tarot'); ?></h2>
        <p class="description"><?php esc_html_e('Choose how payments are handled for bookings.','daily-tarot'); ?></p>
        <?php if (!$isPro): ?>
            <div class="notice notice-warning"><p><?php esc_html_e('Payment configuration is available in Oracle (Pro). Free mode can take bookings without payment.','daily-tarot'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="max-width:900px;">
            <?php wp_nonce_field('dtarot_booking_settings'); ?>
            <input type="hidden" name="action" value="dtarot_booking_settings_save" />
            <input type="hidden" name="redirect_page" value="daily-tarot-bookings" />
            <input type="hidden" name="redirect_tab" value="options" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php esc_html_e('Payment timing','daily-tarot'); ?></th>
                    <td>
                        <select name="payment_mode" <?php echo $isPro ? '' : 'disabled'; ?>>
                            <option value="none" <?php selected((string)$s['payment_mode'], 'none'); ?>><?php esc_html_e('No payment','daily-tarot'); ?></option>
                            <option value="before" <?php selected((string)$s['payment_mode'], 'before'); ?>><?php esc_html_e('Pay before booking','daily-tarot'); ?></option>
                            <option value="after" <?php selected((string)$s['payment_mode'], 'after'); ?>><?php esc_html_e('Pay after confirmation','daily-tarot'); ?></option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Payment provider','daily-tarot'); ?></th>
                    <td>
                        <select name="payment_provider" <?php echo $isPro ? '' : 'disabled'; ?>>
                            <option value="paypal" <?php selected((string)$s['payment_provider'], 'paypal'); ?>><?php esc_html_e('PayPal','daily-tarot'); ?></option>
                            <option value="stripe" <?php selected((string)$s['payment_provider'], 'stripe'); ?>><?php esc_html_e('Stripe','daily-tarot'); ?></option>
                            <option value="both" <?php selected((string)$s['payment_provider'], 'both'); ?>><?php esc_html_e('Both (let user choose)','daily-tarot'); ?></option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('PayPal link','daily-tarot'); ?></th>
                    <td>
                        <input type="url" name="paypal_url" value="<?php echo esc_attr((string)$s['paypal_url']); ?>" class="regular-text" placeholder="https://paypal.me/yourname" <?php echo $isPro ? '' : 'disabled'; ?> />
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Stripe link','daily-tarot'); ?></th>
                    <td>
                        <input type="url" name="stripe_url" value="<?php echo esc_attr((string)$s['stripe_url']); ?>" class="regular-text" placeholder="https://buy.stripe.com/..." <?php echo $isPro ? '' : 'disabled'; ?> />
                    </td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-primary"><?php esc_html_e('Save options','daily-tarot'); ?></button>
            </p>
        </form>
        <?php
    }
}
