<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText



use DailyTarot\Support\FeatureFlags;
use DailyTarot\Support\ProUrls;

use DailyTarot\Support\PostTypes;

final class Menu {

    public static function init(): void {
        add_action('admin_menu', [__CLASS__, 'register']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue']);
        add_filter('parent_file', [__CLASS__, 'parent_file']);
        add_filter('submenu_file', [__CLASS__, 'submenu_file']);
    }

    public static function register(): void {
        $parent = 'daily-tarot';
        $iconSvg = '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20" aria-hidden="true"><rect x="3.5" y="5.5" width="7" height="11" rx="1" transform="rotate(-8 7 11)" fill="none" stroke="currentColor" stroke-width="1.2"/><rect x="6.5" y="4.5" width="7" height="11.5" rx="1.2" fill="none" stroke="currentColor" stroke-width="1.4"/><rect x="9.5" y="5.5" width="7" height="11" rx="1" transform="rotate(8 13 11)" fill="none" stroke="currentColor" stroke-width="1.2"/></svg>';
        $iconData = 'data:image/svg+xml;base64,' . base64_encode($iconSvg);

        add_menu_page(
            __('Daily Tarot','daily-tarot'),
            __('Daily Tarot','daily-tarot'),
            'manage_options',
            $parent,
            [Pages\Dashboard::class,'render'],
            $iconData,
            58
        );

        add_submenu_page($parent, __('Dashboard','daily-tarot'), __('Dashboard','daily-tarot'), 'manage_options', $parent, [Pages\Dashboard::class,'render']);
        add_submenu_page($parent, __('Calendar','daily-tarot'), __('Calendar','daily-tarot'), 'manage_options', 'daily-tarot-calendar', [Pages\Calendar::class,'render']);
        add_submenu_page($parent, __('Content','daily-tarot'), __('Content','daily-tarot'), 'manage_options', 'daily-tarot-content', [Pages\Content::class,'render']);
        add_submenu_page($parent, __('Spreads','daily-tarot'), __('Spreads','daily-tarot'), 'manage_options', 'daily-tarot-spreads', [Pages\Spreads::class,'render']);
        add_submenu_page($parent, __('Bookings','daily-tarot'), __('Bookings','daily-tarot'), 'manage_options', 'daily-tarot-bookings', [Pages\Bookings::class,'render']);

        // Pro menus stay visible; in the free build the premium page classes are stripped.
        // These callbacks safely fall back to an Upgrade screen.
        add_submenu_page($parent, __('AI Integration','daily-tarot'), __('AI Integration (Experimental)','daily-tarot'), 'manage_options', 'daily-tarot-ai', [__CLASS__, 'renderAi']);
        add_submenu_page($parent, __('Analytics','daily-tarot'), __('Analytics','daily-tarot'), 'manage_options', 'daily-tarot-analytics', [__CLASS__, 'renderAnalytics']);

        add_submenu_page($parent, __('Settings','daily-tarot'), __('Settings','daily-tarot'), 'manage_options', 'daily-tarot-settings', [Pages\Settings::class,'render']);
        
        // Show custom Feedback menu only for free users (Pro users see Freemius Contact Us)
        $isPro = class_exists(FeatureFlags::class) && FeatureFlags::enabled('pro');
        if (!$isPro) {
            add_submenu_page($parent, __('Feedback','daily-tarot'), __('Feedback','daily-tarot'), 'manage_options', 'daily-tarot-feedback', [Pages\Feedback::class,'render']);
        }
    }

    public static function renderAi(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));
        $cls = 'DailyTarot\\Admin\\Pages\\Ai';
        if (class_exists($cls)) {
            $cls::render();
            return;
        }
        self::renderUpgrade(__('AI Integration','daily-tarot'));
    }

    public static function renderAnalytics(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));
        $cls = 'DailyTarot\\Admin\\Pages\\Analytics';
        if (class_exists($cls)) {
            $cls::render();
            return;
        }
        self::renderUpgrade(__('Analytics','daily-tarot'));
    }

    private static function renderUpgrade(string $featureLabel): void {
        $buyUrl = class_exists(ProUrls::class) ? ProUrls::buyProUrl() : '';

        echo '<div class="wrap">';
        echo '<h1>' . esc_html($featureLabel) . '</h1>';
        echo '<p class="description">' . esc_html__('This feature is available in Daily Tarot Pro.','daily-tarot') . '</p>';

        if ($buyUrl !== '') {
            echo '<p><a class="button button-primary" href="' . esc_url($buyUrl) . '" target="_blank" rel="noopener">' . esc_html__('Upgrade to Pro','daily-tarot') . '</a></p>';
        } else {
            echo '<p><a class="button" href="' . esc_url(ProUrls::settingsProUrl()) . '">' . esc_html__('View Pro options','daily-tarot') . '</a></p>';
        }

        echo '</div>';
    }

    public static function enqueue($hook): void {
        if (!is_string($hook)) return;

        $load = (strpos($hook, 'daily-tarot') !== false);
        if (!$load && function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if ($screen && is_object($screen)) {
                $pt = (string)($screen->post_type ?? '');
                $load = in_array($pt, array_merge(
                    PostTypes::deckTypes(),
                    PostTypes::meaningPackTypes(),
                    PostTypes::bookingTypes(),
                    PostTypes::readingTypeTypes()
                ), true);
            }
        }
        if (!$load) return;

        $cssFile = DTAROT_PATH . 'assets/admin.css';
        $jsFile  = DTAROT_PATH . 'assets/admin.js';
        $cssVer = is_readable($cssFile) ? (string)@filemtime($cssFile) : DTAROT_VERSION;
        $jsVer  = is_readable($jsFile) ? (string)@filemtime($jsFile) : DTAROT_VERSION;

        wp_enqueue_style('dtarot-admin', DTAROT_URL.'assets/admin.css', [], $cssVer);
        wp_enqueue_media();
        wp_enqueue_script('dtarot-admin', DTAROT_URL.'assets/admin.js', ['jquery'], $jsVer, true);

        if (strpos($hook, 'daily-tarot-spreads') !== false) {
            $frontendCss = DTAROT_PATH . 'assets/frontend.css';
            $frontendJs = DTAROT_PATH . 'assets/frontend.js';
            $frontendCssVer = is_readable($frontendCss) ? (string)@filemtime($frontendCss) : DTAROT_VERSION;
            $frontendJsVer = is_readable($frontendJs) ? (string)@filemtime($frontendJs) : DTAROT_VERSION;

            wp_enqueue_style('dtarot-frontend', DTAROT_URL.'assets/frontend.css', [], $frontendCssVer);
            wp_enqueue_script('dtarot-frontend-js', DTAROT_URL.'assets/frontend.js', [], $frontendJsVer, true);
        }
        wp_localize_script('dtarot-admin', 'DTAROT', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('dtarot_admin'),
            'i18n'    => [
                'mediaBackTitle' => __('Select card back','daily-tarot'),
                'mediaCardTitle' => __('Select card image','daily-tarot'),
                'mediaUseImage'  => __('Use this image','daily-tarot'),
                'mediaMissing'   => __('Media library not available','daily-tarot'),

                'selectedClickSave' => __('Selected. Click Save.','daily-tarot'),
                'savingEllipsis'    => __('Saving…','daily-tarot'),
                'save'              => __('Save','daily-tarot'),
                'saveFailed'        => __('Save failed','daily-tarot'),
                'saved'             => __('Saved.','daily-tarot'),

                'loadingEllipsis'   => __('Loading…','daily-tarot'),
                'generatingEllipsis'=> __('Generating…','daily-tarot'),
                'pickDateFirst'     => __('Pick a date first.','daily-tarot'),

                'aiDisabled'        => __('AI is disabled.','daily-tarot'),
                'aiNotEnoughCredits'=> __('Not enough credits to generate.','daily-tarot'),
                'aiNeedDeckCard'    => __('Select a deck + card for this date first (Calendar).','daily-tarot'),
                'aiNoGenerator'     => __('No AI generator is configured yet.','daily-tarot'),
                'aiGenerationFailed'=> __('Generation failed.','daily-tarot'),
                'aiGenerated'       => __('Generated.','daily-tarot'),
                'aiGeneratedSaved'  => __('Generated + saved.','daily-tarot'),

                'aiEnterCreditAmount' => __('Enter a valid credit amount.','daily-tarot'),
                'aiAddingCredits'     => __('Adding credits…','daily-tarot'),
                'aiCreditsAdded'      => __('Credits added.','daily-tarot'),
                'aiFailedAddCredits'  => __('Failed to add credits.','daily-tarot'),
                'reuseOldText'      => __('Reuse old text','daily-tarot'),
                'noOlderTextFound'  => __('No older text found','daily-tarot'),
                'loadedTextFrom'    => __('Loaded text from %s.','daily-tarot'),
                'loadedText'        => __('Loaded text.','daily-tarot'),

                'readyToPublish'    => __('Ready to publish','daily-tarot'),
                'draftReady'        => __('Draft (ready)','daily-tarot'),
                'draftIncomplete'   => __('Draft (missing deck/card)','daily-tarot'),
                'publishNeedsDeckCard' => __('To publish, select a deck and a card.','daily-tarot'),

                'calendarLimit'     => __('Free plan can edit days up to and including %s. Activate Pro to edit any date.','daily-tarot'),

                'onboardTodayTitle' => __('Set today card','daily-tarot'),
                'onboardTodayText'  => __('Edit your card of the day and preview it.','daily-tarot'),
                'onboardCalendarTitle' => __('Plan your calendar','daily-tarot'),
                'onboardCalendarText'  => __('Fill upcoming days in the calendar overview.','daily-tarot'),
                'onboardSpreadsTitle'  => __('Configure spreads','daily-tarot'),
                'onboardSpreadsText'   => __('Set spread presets and meaning packs.','daily-tarot'),
                'onboardBookingsTitle' => __('Enable bookings','daily-tarot'),
                'onboardBookingsText'  => __('Create reading types and accept bookings.','daily-tarot'),
                'onboardBack'          => __('Back','daily-tarot'),
                'onboardNext'          => __('Next','daily-tarot'),
                'onboardFinish'        => __('Finish','daily-tarot'),
                'onboardClose'         => __('Close','daily-tarot'),
            ],
        ]);
    }

    public static function parent_file(string $parent_file): string {
        if (!function_exists('get_current_screen')) return $parent_file;
        $screen = get_current_screen();
        if ($screen && is_object($screen)) {
            $pt = (string)($screen->post_type ?? '');
            if (in_array($pt, array_merge(PostTypes::deckTypes(), PostTypes::meaningPackTypes()), true)) return 'daily-tarot';
            if (in_array($pt, array_merge(PostTypes::bookingTypes(), PostTypes::readingTypeTypes()), true)) return 'daily-tarot';
        }
        return $parent_file;
    }

    public static function submenu_file(?string $submenu_file): ?string {
        if (!function_exists('get_current_screen')) return $submenu_file;
        $screen = get_current_screen();
        if ($screen && is_object($screen)) {
            $pt = (string)($screen->post_type ?? '');
            if (in_array($pt, array_merge(PostTypes::deckTypes(), PostTypes::meaningPackTypes()), true)) return 'daily-tarot-content';
            if (in_array($pt, array_merge(PostTypes::bookingTypes(), PostTypes::readingTypeTypes()), true)) return 'daily-tarot-bookings';
        }
        return $submenu_file;
    }
}
