<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Registry\Cards;
use DailyTarot\Packs\PackVersion;
use DailyTarot\Meaning\DefaultMeaningPack;
use DailyTarot\Meaning\MeaningPackRepository;
use DailyTarot\Support\PostTypes;

final class MeaningPackMeta {
    public const META_KEY = '_dtarot_meanings';
    public const META_SYSTEM = '_dtarot_system';
    public const META_DECK = '_dtarot_deck';

    public static function init(): void {
        add_action('add_meta_boxes_' . PostTypes::MEANING_PACK, [__CLASS__, 'add']);
        add_action('save_post_' . PostTypes::MEANING_PACK, [__CLASS__, 'save']);

        // Back-compat.
        add_action('add_meta_boxes_' . PostTypes::LEGACY_MEANING_PACK, [__CLASS__, 'addLegacy']);
        add_action('save_post_' . PostTypes::LEGACY_MEANING_PACK, [__CLASS__, 'save']);

        // Toolbar area under the (hidden) title field.
        add_action('edit_form_after_title', [__CLASS__, 'renderTopToolbar']);

        // Allow JS-driven “save then switch” by redirecting after save.
        add_filter('redirect_post_location', [__CLASS__, 'filterRedirectAfterSave'], 10, 2);
    }

    public static function renderTopToolbar($post): void {
        if (!$post || !is_object($post) || !isset($post->post_type) || !PostTypes::isMeaningPackType((string)$post->post_type)) return;
        if (!isset($post->ID) || (int)$post->ID <= 0) return;

        $backUrl = admin_url('edit.php?post_type=' . PostTypes::MEANING_PACK);

        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'post_status' => ['publish','draft','pending','private'],
            'posts_per_page' => 500,
            'orderby' => 'title',
            'order' => 'ASC',
            'fields' => 'ids',
        ]);

        $options = [];
        foreach ((array)$packs as $pid) {
            $pid = (int)$pid;
            if ($pid <= 0 || $pid === (int)$post->ID) continue;

            $title = (string)get_the_title($pid);
            if (trim($title) === '') $title = sprintf(__('(no title) #%d','daily-tarot'), $pid);

            $system = Cards::normalizeSystem((string)get_post_meta($pid, self::META_SYSTEM, true));
            $systemLabel = '';
            $systems = Cards::systems();
            if ($system !== '' && isset($systems[$system])) {
                $systemLabel = (string)$systems[$system];
            }

            $label = $systemLabel !== '' ? ($title . ' — ' . $systemLabel) : $title;
            $url = get_edit_post_link($pid, '');
            if (!$url) continue;

            $options[] = [
                'id' => $pid,
                'label' => $label,
                'url' => $url,
            ];
        }

        echo '<div class="dtarot-meaning-pack-topbar" data-dtarot-meaning-pack-topbar="1">';
        echo '<a id="dtarot-pack-back" class="button" href="' . esc_url($backUrl) . '">' . esc_html__('Back to meanings','daily-tarot') . '</a>';

        echo '<label class="dtarot-pack-switch">';
        echo '<span class="screen-reader-text">' . esc_html__('Switch meaning pack','daily-tarot') . '</span>';
        echo '<select id="dtarot-pack-switch" aria-label="' . esc_attr__('Switch meaning pack','daily-tarot') . '">';
        echo '<option value="">' . esc_html__('Switch meaning pack…','daily-tarot') . '</option>';
        foreach ($options as $opt) {
            echo '<option value="' . esc_attr((string)$opt['url']) . '">' . esc_html((string)$opt['label']) . '</option>';
        }
        echo '</select>';
        echo '</label>';

        echo '</div>';
    }

    public static function filterRedirectAfterSave(string $location, int $post_id): string {
        $post = get_post($post_id);
        if (!$post || !isset($post->post_type) || !PostTypes::isMeaningPackType((string)$post->post_type)) return $location;
        if (!current_user_can('edit_post', $post_id)) return $location;

        // Only honor the redirect parameter when the metabox save nonce is valid.
        // This prevents a crafted POST from influencing admin redirects.
        if (!isset($_POST['dtarot_meaning_pack_nonce']) || !wp_verify_nonce(sanitize_text_field((string)wp_unslash($_POST['dtarot_meaning_pack_nonce'])), 'dtarot_meaning_pack_save')) {
            return $location;
        }
        if (!isset($_POST['dtarot_after_save'])) return $location;

        $target = esc_url_raw((string)wp_unslash($_POST['dtarot_after_save']));
        if ($target === '') return $location;

        // Only allow redirects to wp-admin.
        $validated = wp_validate_redirect($target, '');
        if ($validated === '') return $location;
        $adminBase = admin_url();
        if (!str_starts_with($validated, $adminBase)) return $location;

        return $validated;
    }

    public static function add(): void {
        add_meta_box('dtarot_meaning_pack', __('Meaning Pack — Card Meanings','daily-tarot'), [__CLASS__, 'render'], PostTypes::MEANING_PACK, 'normal', 'high');
    }

    public static function addLegacy(): void {
        add_meta_box('dtarot_meaning_pack', __('Meaning Pack — Card Meanings','daily-tarot'), [__CLASS__, 'render'], PostTypes::LEGACY_MEANING_PACK, 'normal', 'high');
    }

    private static function renderLenormandMetaBox(string $cardId): string {
        $cardId = sanitize_text_field($cardId);
        if ($cardId === '' || !str_starts_with($cardId, 'lenormand_')) return '';

        $meta = Cards::meta($cardId);
        if (!$meta) return '';

        $subject = isset($meta['subject']) && is_string($meta['subject']) ? trim($meta['subject']) : '';
        $modifier = isset($meta['modifier']) && is_string($meta['modifier']) ? trim($meta['modifier']) : '';
        $extended = isset($meta['extended']) && is_string($meta['extended']) ? trim($meta['extended']) : '';

        $rank = '';
        $suit = '';
        if (isset($meta['playing']) && is_array($meta['playing'])) {
            $rank = isset($meta['playing']['rank']) && is_string($meta['playing']['rank']) ? trim($meta['playing']['rank']) : '';
            $suit = isset($meta['playing']['suit']) && is_string($meta['playing']['suit']) ? trim($meta['playing']['suit']) : '';
        }

        $isSignifier = !empty($meta['is_signifier']);

        if ($subject === '' && $modifier === '' && $extended === '' && $rank === '' && $suit === '' && !$isSignifier) {
            return '';
        }

        $suitLabels = [
            'hearts' => __('Hearts','daily-tarot'),
            'spades' => __('Spades','daily-tarot'),
            'diamonds' => __('Diamonds','daily-tarot'),
            'clubs' => __('Clubs','daily-tarot'),
        ];
        $suitLabel = $suit !== '' ? ($suitLabels[$suit] ?? $suit) : '';

        $html = '<div class="dtarot-lenormand-meta-box">';
        $html .= '<h4 class="dtarot-lenormand-meta-title">'.esc_html__('Lenormand reference','daily-tarot').'</h4>';

        if ($isSignifier) {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Signifier','daily-tarot').'</span></div>';
        }

        if ($rank !== '' || $suitLabel !== '') {
            $inset = trim($rank . ' ' . $suitLabel);
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Playing card inset','daily-tarot').':</span> '.esc_html($inset).'</div>';
        }

        if ($subject !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Subject','daily-tarot').':</span> '.esc_html($subject).'</div>';
        }
        if ($modifier !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Modifier','daily-tarot').':</span> '.esc_html($modifier).'</div>';
        }
        if ($extended !== '') {
            $html .= '<div class="dtarot-lenormand-meta-row"><span class="dtarot-lenormand-meta-label">'.esc_html__('Extended','daily-tarot').':</span> '.esc_html($extended).'</div>';
        }

        $html .= '</div>';
        return $html;
    }

    /**
     * System-aware optional field configuration.
     *
     * Upright/Reversed are always shown separately (WYSIWYG).
     *
     * @return array<string,array{label:string,type:'text'|'textarea',rows?:int,help?:string,placeholder?:string}>
     */
    private static function optionalFieldsForSystem(string $system): array {
        $system = Cards::normalizeSystem($system);

        if ($system === Cards::SYSTEM_TAROT) {
            return [
                'keywords' => [
                    'label' => (string)__('Quick keywords','daily-tarot'),
                    'type' => 'text',
                    'placeholder' => (string)__('e.g. courage, patience, resilience','daily-tarot'),
                    'help' => (string)__('Comma-separated. Used for completeness and quick summaries.','daily-tarot'),
                ],
                'short' => [
                    'label' => (string)__('Short summary','daily-tarot'),
                    'type' => 'textarea',
                    'rows' => 2,
                    'help' => (string)__('One or two sentences. Used when a shorter meaning is needed.','daily-tarot'),
                ],
                'long' => [
                    'label' => (string)__('Detailed meaning','daily-tarot'),
                    'type' => 'textarea',
                    'rows' => 6,
                    'help' => (string)__('Optional longer explanation. HTML is allowed.','daily-tarot'),
                ],
                'symbols' => [
                    'label' => (string)__('Symbols / imagery','daily-tarot'),
                    'type' => 'textarea',
                    'rows' => 3,
                    'help' => (string)__('Optional. List key symbols as a comma-separated list or short phrases.','daily-tarot'),
                ],
                'correspondences' => [
                    'label' => (string)__('Correspondences (element/astrology)','daily-tarot'),
                    'type' => 'textarea',
                    'rows' => 3,
                    'help' => (string)__('Optional. Keep this concise (e.g., planet/zodiac/element). Correspondences vary by tradition.','daily-tarot'),
                ],
            ];
        }

        // Lenormand / Kipper: keep it focused.
        return [
            'keywords' => [
                'label' => (string)__('Quick keywords','daily-tarot'),
                'type' => 'text',
                'placeholder' => (string)__('e.g. message, movement, opportunity','daily-tarot'),
                'help' => (string)__('Comma-separated. Used for completeness and quick summaries.','daily-tarot'),
            ],
            'short' => [
                'label' => (string)__('Short summary','daily-tarot'),
                'type' => 'textarea',
                'rows' => 2,
            ],
            'long' => [
                'label' => (string)__('Detailed meaning','daily-tarot'),
                'type' => 'textarea',
                'rows' => 6,
                'help' => (string)__('Optional longer explanation. HTML is allowed.','daily-tarot'),
            ],
        ];
    }

    /**
     * Prefill Tarot fields from the Default Meaning Pack when available.
     * Does not overwrite existing values.
     *
     * @param array<string,mixed> $m
     * @return array<string,mixed>
     */
    private static function prefillTarotFromDefaultPack(array $m, string $cardId): array {
        $cardId = sanitize_text_field($cardId);
        if ($cardId === '') return $m;
        if (!class_exists(DefaultMeaningPack::class) || !class_exists(MeaningPackRepository::class)) return $m;

        $defaultId = DefaultMeaningPack::getId();
        if ($defaultId <= 0) return $m;

        $d = MeaningPackRepository::getMeaning($defaultId, $cardId);
        if (!is_array($d)) return $m;

        foreach (['upright','reversed','keywords','correspondences','symbols','short','long'] as $k) {
            $cur = isset($m[$k]) && is_string($m[$k]) ? (string)$m[$k] : '';
            $val = isset($d[$k]) && is_string($d[$k]) ? (string)$d[$k] : '';
            if (trim($cur) === '' && trim($val) !== '') {
                $m[$k] = $val;
            }
        }

        return $m;
    }

    private static function isMeaningComplete(array $m): bool {
        $upright = isset($m['upright']) && is_string($m['upright']) ? trim($m['upright']) : '';
        return $upright !== '';
    }

    /**
     * @return array{positive:string,negative:string}
     */
    private static function primaryMeaningLabels(string $system): array {
        $system = Cards::normalizeSystem($system);

        if ($system === Cards::SYSTEM_TAROT) {
            return [
                'positive' => (string)__('Upright','daily-tarot'),
                'negative' => (string)__('Reversed','daily-tarot'),
            ];
        }

        if ($system === Cards::SYSTEM_LENORMAND) {
            return [
                'positive' => (string)__('Meaning Positive','daily-tarot'),
                'negative' => (string)__('Meaning Negative','daily-tarot'),
            ];
        }

        // Kipper + Kipper fin de siecle
        return [
            'positive' => (string)__('Positive Expression','daily-tarot'),
            'negative' => (string)__('Negative Expression','daily-tarot'),
        ];
    }

    private static function completenessHelpText(string $system): string {
        $labels = self::primaryMeaningLabels($system);
        return (string)sprintf(
            /* translators: %s is the required primary meaning label */
            __('A card meaning is marked complete when “%s” is filled in. Other fields are optional.','daily-tarot'),
            $labels['positive']
        );
    }

    /**
     * Adds a human-friendly card number prefix to each card label for the editor UI.
     * Keeps IDs unchanged (meanings/images are keyed by card ID).
     *
     * @param array<string,string> $cards
     * @return array<string,string>
     */
    private static function withCardNumbers(string $system, array $cards): array {
        $system = Cards::normalizeSystem($system);

        $out = [];
        foreach ($cards as $id => $name) {
            if (!is_string($id)) continue;
            $label = is_string($name) ? $name : '';
            $prefix = self::cardNumberPrefix($system, $id);
            if ($prefix !== '') {
                $label = $prefix . ' — ' . $label;
            }
            $out[$id] = $label;
        }
        return $out;
    }

    private static function cardNumberPrefix(string $system, string $cardId): string {
        $system = Cards::normalizeSystem($system);
        $cardId = sanitize_text_field($cardId);
        if ($cardId === '') return '';

        // Lenormand: lenormand_01..36
        if ($system === Cards::SYSTEM_LENORMAND && preg_match('/^lenormand_(\d{1,2})$/', $cardId, $m)) {
            return (string)((int)$m[1]);
        }

        // Kipper: kipper_01..36
        if ($system === Cards::SYSTEM_KIPPER && preg_match('/^kipper_(\d{1,2})$/', $cardId, $m)) {
            return (string)((int)$m[1]);
        }

        // Gypsy (Tarot Gypsy): gypsy_01..36
        if ($system === Cards::SYSTEM_GYPSY && preg_match('/^gypsy_(\d{1,2})$/', $cardId, $m)) {
            return (string)((int)$m[1]);
        }

        // Kipper Fin de Siècle: kipper_fds_01..39
        if ($system === Cards::SYSTEM_KIPPER_FIN_DE_SIECLE && preg_match('/^kipper_fds_(\d{1,2})$/', $cardId, $m)) {
            return (string)((int)$m[1]);
        }

        // Tarot majors: tarot_major_0..21
        if ($system === Cards::SYSTEM_TAROT && preg_match('/^tarot_major_(\d{1,2})$/', $cardId, $m)) {
            return (string)((int)$m[1]);
        }

        // Tarot minors: tarot_{suit}_01..14 (numbers repeat per suit; still useful as quick cue)
        if ($system === Cards::SYSTEM_TAROT && preg_match('/^tarot_(wands|cups|swords|pentacles)_(\d{1,2})$/', $cardId, $m)) {
            $suit = (string)$m[1];
            $num = (int)$m[2];
            return strtoupper(substr($suit, 0, 1)) . (string)$num;
        }

        return '';
    }

    public static function render($post): void {
        wp_nonce_field('dtarot_meaning_pack_save', 'dtarot_meaning_pack_nonce');

        $savedSystemRaw = (string)get_post_meta($post->ID, self::META_SYSTEM, true);
        $savedSystem = $savedSystemRaw !== '' ? Cards::normalizeSystem($savedSystemRaw) : '';
        $systemIsLocked = $savedSystemRaw !== '' && metadata_exists('post', $post->ID, self::META_SYSTEM);
        $system = $savedSystem;
        // Allow a temporary “preview” system only before the pack has a saved system.
        if ($system === '') {
            $previewSystem = isset($_GET['dtarot_system_preview']) ? Cards::normalizeSystem((string)sanitize_text_field((string)wp_unslash($_GET['dtarot_system_preview']))) : '';
            if ($previewSystem !== '' && array_key_exists($previewSystem, Cards::systems())) {
                $system = $previewSystem;
            }
        }
        if ($system === '') $system = Cards::SYSTEM_TAROT;

        $packVersion = (string)get_post_meta($post->ID, PackVersion::META_KEY, true);
        $cards = self::withCardNumbers($system, Cards::forSystem($system));
        $all = get_post_meta($post->ID, self::META_KEY, true);
        if (!is_array($all)) $all = [];

        $active = isset($_GET['card']) ? sanitize_text_field((string)wp_unslash($_GET['card'])) : '';
        if ($active === '' || !isset($cards[$active])) {
            $first = array_key_first($cards);
            $active = is_string($first) ? $first : '';
        }

        // Build a JSON map of meanings for JS-driven card switching.
        $meaningData = [];
        foreach ($cards as $id => $_name) {
            $m = is_array($all[$id] ?? null) ? $all[$id] : [];
            if ($system === Cards::SYSTEM_TAROT) {
                $m = self::prefillTarotFromDefaultPack($m, $id);
            }
            $meaningData[$id] = [
                'upright' => isset($m['upright']) && is_string($m['upright']) ? (string)$m['upright'] : '',
                'reversed' => isset($m['reversed']) && is_string($m['reversed']) ? (string)$m['reversed'] : '',
                'keywords' => isset($m['keywords']) && is_string($m['keywords']) ? (string)$m['keywords'] : '',
                'correspondences' => isset($m['correspondences']) && is_string($m['correspondences']) ? (string)$m['correspondences'] : '',
                'symbols' => isset($m['symbols']) && is_string($m['symbols']) ? (string)$m['symbols'] : '',
                'short' => isset($m['short']) && is_string($m['short']) ? (string)$m['short'] : '',
                'long' => isset($m['long']) && is_string($m['long']) ? (string)$m['long'] : '',
            ];
        }

        $completeCount = 0;
        foreach ($cards as $id => $_name) {
            $m = is_array($all[$id] ?? null) ? $all[$id] : [];
            if (self::isMeaningComplete($m)) $completeCount++;
        }
        $totalCount = count($cards);
        ?>
        <p>
            <label for="dtarot-pack-version"><strong><?php esc_html_e('Pack version','daily-tarot'); ?></strong></label><br>
            <input id="dtarot-pack-version" type="text" name="dtarot_pack_version" value="<?php echo esc_attr($packVersion); ?>" placeholder="0.0.0" class="regular-text" />
            <span class="description"><?php esc_html_e('Used for ZIP export/import versioning and downgrade protection.','daily-tarot'); ?></span>
        </p>

        <p>
            <label for="dtarot-pack-name"><strong><?php esc_html_e('Name','daily-tarot'); ?></strong></label><br>
            <input id="dtarot-pack-name" type="text" name="dtarot_pack_name" value="<?php echo esc_attr((string)$post->post_title); ?>" class="regular-text" placeholder="<?php echo esc_attr__('Meaning pack name…','daily-tarot'); ?>" />
        </p>

        <p>
            <label for="dtarot-pack-system"><strong><?php esc_html_e('System','daily-tarot'); ?></strong></label><br>
            <select id="dtarot-pack-system" name="dtarot_pack_system" <?php echo $systemIsLocked ? 'disabled' : ''; ?>>
                <?php foreach (Cards::systems() as $k => $label): ?>
                    <option value="<?php echo esc_attr($k); ?>" <?php selected($system, $k); ?>><?php echo esc_html($label); ?></option>
                <?php endforeach; ?>
            </select>
            <span class="description"><?php esc_html_e('Pick a system for this pack. After you save the pack, the system cannot be changed.','daily-tarot'); ?></span>
        </p>

                <?php
                $js = implode("\n", [
                    '(function(){',
                    "    var KEY = 'dtarot_meaning_pack_system_preview_v1';",
                    "    var sel = document.getElementById('dtarot-pack-system');",
                    '    if (!sel || sel.disabled) return;',
                    '',
                    '    function getUrlSys(){',
                    '        try {',
                    "            return (new URL(window.location.href)).searchParams.get('dtarot_system_preview') || '';",
                    "        } catch (e) { return ''; }",
                    '    }',
                    '',
                    '    function reloadWithSys(sys){',
                    '        try {',
                    '            if (!sys) return;',
                    '            var url = new URL(window.location.href);',
                    "            url.searchParams.set('dtarot_system_preview', sys);",
                    "            url.searchParams.delete('card');",
                    '            window.location.href = url.toString();',
                    '        } catch (e) {}',
                    '    }',
                    '',
                    "    // WP's post-new.php flow creates an auto-draft and redirects to post.php,",
                    '    // often dropping custom query params. Persist the preview selection.',
                    '    try {',
                    '        var urlSys = getUrlSys();',
                    '        if (urlSys) {',
                    '            sessionStorage.setItem(KEY, urlSys);',
                    '        } else {',
                    "            var stored = sessionStorage.getItem(KEY) || '';",
                    '            // If we have a stored choice and no param, re-apply it.',
                    '            if (stored && stored !== sel.value) {',
                    '                reloadWithSys(stored);',
                    '                return;',
                    '            }',
                    '        }',
                    '    } catch (e) {}',
                    '',
                    "    sel.addEventListener('change', function(){",
                    '        try {',
                    "            var sys = (sel.value || '').toString();",
                    '            if (!sys) return;',
                    '            sessionStorage.setItem(KEY, sys);',
                    '            reloadWithSys(sys);',
                    '        } catch (e) {}',
                    '    });',
                    '})();',
                    '',
                ]);

                wp_add_inline_script('dtarot-admin', $js, 'after');
                ?>

        <p>
            <label for="dtarot-pack-deck"><strong><?php esc_html_e('Associated Deck (optional)','daily-tarot'); ?></strong></label><br>
            <select id="dtarot-pack-deck" name="dtarot_pack_deck">
                <option value=""><?php esc_html_e('None - use with any deck','daily-tarot'); ?></option>
                <?php 
                $decks = get_posts([
                    'post_type' => PostTypes::deckTypes(),
                    'numberposts' => -1,
                    'post_status' => ['publish','draft','pending','private'],
                    'orderby' => 'title',
                    'order' => 'ASC',
                ]);
                $savedDeckId = (int)get_post_meta($post->ID, self::META_DECK, true);
                foreach ((array)$decks as $deck): 
                    $deckId = (int)$deck->ID;
                    $deckTitle = $deck->post_title ?: sprintf(__('(no title) #%d','daily-tarot'), $deckId);
                    ?>
                    <option value="<?php echo esc_attr((string)$deckId); ?>" <?php selected($savedDeckId, $deckId); ?>><?php echo esc_html($deckTitle); ?></option>
                <?php endforeach; ?>
            </select>
            <span class="description"><?php esc_html_e('Optional: link this pack to a deck (useful when you have multiple decks in the same system). The pack still stores meanings; the deck still stores images.','daily-tarot'); ?></span>
        </p>

        <p class="description"><?php esc_html_e('Meaning packs store text/interpretations for each fixed card ID in a system. Select a card row to edit its meanings on the right.','daily-tarot'); ?></p>

        <script type="application/json" id="dtarot-meaning-pack-json"><?php
            echo wp_json_encode([
                'postId' => (int)$post->ID,
                'system' => (string)$system,
                'cards' => $cards,
                'meanings' => $meaningData,
                'optionalKeys' => array_keys(self::optionalFieldsForSystem($system)),
            ], JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_AMP | JSON_HEX_QUOT);
        ?></script>

        <div class="dtarot-toolbar">
            <div class="dtarot-pack-progress">
                <?php echo esc_html(sprintf(__('%d/%d complete','daily-tarot'), $completeCount, $totalCount)); ?>
            </div>
            <label class="dtarot-search">
                <span><?php esc_html_e('Search','daily-tarot'); ?></span>
                <input type="search" id="dtarot-meaning-search" placeholder="<?php echo esc_attr__('Type to filter cards…','daily-tarot'); ?>" />
            </label>
        </div>

        <p class="description" style="margin-top:8px;"><?php echo esc_html(self::completenessHelpText($system)); ?></p>

        <div class="dtarot-meaning-layout">
            <div class="dtarot-meaning-left">

        <?php $primaryLabels = self::primaryMeaningLabels($system); ?>
        <table class="widefat striped dtarot-meaning-table" id="dtarot-meaning-table">
            <thead><tr>
                <th><?php esc_html_e('Card','daily-tarot'); ?></th>
                <th><?php echo esc_html($primaryLabels['positive']); ?></th>
                <th><?php echo esc_html($primaryLabels['negative']); ?></th>
                <th><?php esc_html_e('Status','daily-tarot'); ?></th>
            </tr></thead>
            <tbody>
            <?php foreach ($cards as $id=>$name):
                $m = is_array($all[$id] ?? null) ? $all[$id] : [];
                $has_u = !empty($m['upright']);
                $has_r = !empty($m['reversed']);
                $complete = self::isMeaningComplete($m);
            ?>
                <tr class="dtarot-meaning-row <?php echo $complete ? 'dtarot-meaning-complete' : 'dtarot-meaning-incomplete'; ?> <?php echo ($active===$id) ? 'dtarot-meaning-active' : ''; ?>"
                    data-card-name="<?php echo esc_attr(strtolower($name)); ?>"
                    data-card-id="<?php echo esc_attr(strtolower($id)); ?>">
                    <td><strong><?php echo esc_html($name); ?></strong><br><code><?php echo esc_html($id); ?></code></td>
                    <td><?php echo $has_u ? '✔️' : '—'; ?></td>
                    <td><?php echo $has_r ? '✔️' : '—'; ?></td>
                    <td>
                        <?php if ($complete): ?><span style="color:green;"><?php esc_html_e('Complete','daily-tarot'); ?></span>
                        <?php else: ?><span style="color:#b45309;"><?php esc_html_e('Incomplete','daily-tarot'); ?></span><?php endif; ?>
                        <div style="margin-top:6px;">
                            <button type="button" class="button button-small dtarot-meaning-edit-btn" data-dtarot-card="<?php echo esc_attr($id); ?>"><?php esc_html_e('Edit','daily-tarot'); ?></button>
                        </div>
                    </td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>

            </div>

            <div class="dtarot-meaning-right">

        <?php if ($active && isset($cards[$active])):
            $m = $meaningData[$active] ?? [];
            if (!is_array($m)) $m = [];

            $optionalFields = self::optionalFieldsForSystem($system);
        ?>
            <div class="dtarot-meaning-editor">
                <h3><?php echo esc_html__('Edit Meaning —','daily-tarot'); ?> <span data-dtarot-card-title><?php echo esc_html($cards[$active]); ?></span></h3>
                <input type="hidden" name="dtarot_card_id" value="<?php echo esc_attr($active); ?>" data-dtarot-card-id>

                <div data-dtarot-lenormand-slot><?php echo self::renderLenormandMetaBox($active); ?></div>

                <div class="dtarot-lenormand-meta-templates" hidden>
                    <?php foreach ($cards as $id => $_name):
                        $html = self::renderLenormandMetaBox($id);
                        if ($html === '') continue;
                    ?>
                        <div data-dtarot-lenormand-template="<?php echo esc_attr($id); ?>"><?php echo $html; ?></div>
                    <?php endforeach; ?>
                </div>

                <div class="dtarot-meaning-editors">
                    <div class="dtarot-meaning-editor-block">
                        <div class="dtarot-meaning-editor-label"><?php echo esc_html($primaryLabels['positive']); ?></div>
                        <?php
                            $upright = isset($m['upright']) && is_string($m['upright']) ? (string)$m['upright'] : '';
                            wp_editor($upright, 'dtarot_upright_editor', [
                                'textarea_name' => 'dtarot_upright',
                                'textarea_rows' => 10,
                                'media_buttons' => false,
                                'teeny' => true,
                                'quicktags' => true,
                            ]);
                        ?>
                    </div>

                    <div class="dtarot-meaning-editor-block">
                        <div class="dtarot-meaning-editor-label"><?php echo esc_html($primaryLabels['negative']); ?></div>
                        <?php
                            $reversed = isset($m['reversed']) && is_string($m['reversed']) ? (string)$m['reversed'] : '';
                            wp_editor($reversed, 'dtarot_reversed_editor', [
                                'textarea_name' => 'dtarot_reversed',
                                'textarea_rows' => 10,
                                'media_buttons' => false,
                                'teeny' => true,
                                'quicktags' => true,
                            ]);
                        ?>
                    </div>
                </div>

                <div class="dtarot-meaning-optional" data-dtarot-meaning-optional>
                    <?php foreach ($optionalFields as $fieldKey => $cfg):
                        $val = isset($m[$fieldKey]) && is_string($m[$fieldKey]) ? (string)$m[$fieldKey] : '';
                        $preview = trim(wp_strip_all_tags($val));
                        if (strlen($preview) > 70) $preview = substr($preview, 0, 67) . '…';

                        $label = isset($cfg['label']) ? (string)$cfg['label'] : $fieldKey;
                        $type = isset($cfg['type']) ? (string)$cfg['type'] : 'text';
                        $rows = isset($cfg['rows']) ? (int)$cfg['rows'] : 3;
                        $help = isset($cfg['help']) ? (string)$cfg['help'] : '';
                        $placeholder = isset($cfg['placeholder']) ? (string)$cfg['placeholder'] : '';
                    ?>
                        <div class="dtarot-inline-field" data-dtarot-inline-field="1">
                            <button type="button" class="dtarot-inline-toggle" aria-expanded="false">
                                <span class="dtarot-inline-label"><?php echo esc_html($label); ?></span>
                                <span class="dtarot-inline-preview"><?php echo esc_html($preview !== '' ? $preview : __('(click to add)','daily-tarot')); ?></span>
                            </button>
                            <div class="dtarot-inline-body" hidden>
                                <?php if ($type === 'textarea'): ?>
                                    <textarea name="<?php echo esc_attr('dtarot_' . $fieldKey); ?>" rows="<?php echo esc_attr((string)$rows); ?>" style="width:100%;" placeholder="<?php echo esc_attr($placeholder); ?>"><?php echo esc_textarea($val); ?></textarea>
                                <?php else: ?>
                                    <input type="text" name="<?php echo esc_attr('dtarot_' . $fieldKey); ?>" value="<?php echo esc_attr($val); ?>" style="width:100%;" placeholder="<?php echo esc_attr($placeholder); ?>" />
                                <?php endif; ?>
                                <?php if ($help !== ''): ?>
                                    <p class="description"><?php echo esc_html($help); ?></p>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <p><button type="submit" class="button button-primary"><?php esc_html_e('Save Meaning','daily-tarot'); ?></button></p>
            </div>
        <?php endif; ?>

            </div>
        </div>
        <?php
    }

    public static function save(int $post_id): void {
        if (!current_user_can('edit_post', $post_id)) return;
        if (!isset($_POST['dtarot_meaning_pack_nonce']) || !wp_verify_nonce(sanitize_text_field((string)wp_unslash($_POST['dtarot_meaning_pack_nonce'])), 'dtarot_meaning_pack_save')) return;

        // Avoid recursion if we update post title.
        static $inSave = false;
        if ($inSave) return;
        $inSave = true;

        // Save/update title from metabox “Name”.
        if (isset($_POST['dtarot_pack_name'])) {
            $name = sanitize_text_field((string)wp_unslash($_POST['dtarot_pack_name']));
            $cur = (string)get_post_field('post_title', $post_id);
            if ($name !== '' && $name !== $cur) {
                remove_action('save_post_' . PostTypes::MEANING_PACK, [__CLASS__, 'save']);
                remove_action('save_post_' . PostTypes::LEGACY_MEANING_PACK, [__CLASS__, 'save']);
                wp_update_post([
                    'ID' => $post_id,
                    'post_title' => $name,
                ]);
                add_action('save_post_' . PostTypes::MEANING_PACK, [__CLASS__, 'save']);
                add_action('save_post_' . PostTypes::LEGACY_MEANING_PACK, [__CLASS__, 'save']);
            }
        }

        // System is immutable after first save.
        // IMPORTANT: normalizeSystem('') returns Tarot, so we must treat empty meta as "unset".
        $existingSystemRaw = (string)get_post_meta($post_id, self::META_SYSTEM, true);
        if ($existingSystemRaw !== '') {
            $system = Cards::normalizeSystem($existingSystemRaw);
        } else {
            $system = isset($_POST['dtarot_pack_system'])
                ? Cards::normalizeSystem(sanitize_text_field((string)wp_unslash($_POST['dtarot_pack_system'])))
                : Cards::SYSTEM_TAROT;
            update_post_meta($post_id, self::META_SYSTEM, $system);
        }

        $packVersion = isset($_POST['dtarot_pack_version']) ? PackVersion::sanitize((string)wp_unslash($_POST['dtarot_pack_version'])) : '';
        if ($packVersion === '') {
            delete_post_meta($post_id, PackVersion::META_KEY);
        } else {
            update_post_meta($post_id, PackVersion::META_KEY, $packVersion);
        }

        // Save deck association
        $deckId = isset($_POST['dtarot_pack_deck']) ? absint(wp_unslash($_POST['dtarot_pack_deck'])) : 0;
        if ($deckId > 0) {
            update_post_meta($post_id, self::META_DECK, $deckId);
        } else {
            delete_post_meta($post_id, self::META_DECK);
        }

        $card_id = isset($_POST['dtarot_card_id']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_card_id'])) : '';
        if ($card_id === '') {
            $inSave = false;
            return;
        }

        $cards = Cards::forSystem($system);
        if (!isset($cards[$card_id])) return;

        $all = get_post_meta($post_id, self::META_KEY, true);
        if (!is_array($all)) $all = [];

        $all[$card_id] = [
            'upright'  => wp_kses_post((string)($_POST['dtarot_upright'] ?? '')),
            'reversed' => wp_kses_post((string)($_POST['dtarot_reversed'] ?? '')),
            'keywords' => sanitize_text_field((string)($_POST['dtarot_keywords'] ?? '')),
            'correspondences' => wp_kses_post((string)($_POST['dtarot_correspondences'] ?? '')),
            'symbols' => wp_kses_post((string)($_POST['dtarot_symbols'] ?? '')),
            'short'    => wp_kses_post((string)($_POST['dtarot_short'] ?? '')),
            'long'     => wp_kses_post((string)($_POST['dtarot_long'] ?? '')),
        ];

        update_post_meta($post_id, self::META_KEY, $all);

        $inSave = false;
    }
}
