<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Registry\Cards;
use DailyTarot\Packs\PackVersion;
use DailyTarot\Support\PostTypes;

final class DeckMeta {

    public const META_SYSTEM = '_dtarot_system';

    public static function init(): void {
        add_action('add_meta_boxes_' . PostTypes::DECK, [__CLASS__, 'add']);
        add_action('save_post_' . PostTypes::DECK, [__CLASS__, 'save']);

        // Back-compat.
        add_action('add_meta_boxes_' . PostTypes::LEGACY_DECK, [__CLASS__, 'addLegacy']);
        add_action('save_post_' . PostTypes::LEGACY_DECK, [__CLASS__, 'save']);
    }

    public static function add(): void {
        add_meta_box('dtarot_deck_meta', __('Deck Settings','daily-tarot'), [__CLASS__, 'render'], PostTypes::DECK, 'normal', 'high');
    }

    public static function addLegacy(): void {
        add_meta_box('dtarot_deck_meta', __('Deck Settings','daily-tarot'), [__CLASS__, 'render'], PostTypes::LEGACY_DECK, 'normal', 'high');
    }

    public static function render($post): void {
        wp_nonce_field('dtarot_deck_save', 'dtarot_deck_nonce');
        $system = Cards::normalizeSystem((string)get_post_meta($post->ID, self::META_SYSTEM, true));
        $back = (string)get_post_meta($post->ID, '_dtarot_back', true);
        $packVersion = (string)get_post_meta($post->ID, PackVersion::META_KEY, true);
        ?>
        <p>
            <label for="dtarot-deck-system"><strong><?php esc_html_e('System','daily-tarot'); ?></strong></label><br>
            <select id="dtarot-deck-system" name="dtarot_system">
                <?php foreach (Cards::systems() as $k => $label): ?>
                    <option value="<?php echo esc_attr($k); ?>" <?php selected($system, $k); ?>><?php echo esc_html($label); ?></option>
                <?php endforeach; ?>
            </select>
            <span class="description"><?php esc_html_e('Choose which card registry this deck uses (Tarot / Lenormand / Kipper / Gypsy).','daily-tarot'); ?></span>
        </p>

        <p class="description"><?php esc_html_e('Set the card back image for this deck. Card face images are managed in Content → Cards.','daily-tarot'); ?></p>

        <div class="dtarot-row">
            <input type="text" id="dtarot-back-url" name="dtarot_back_url" value="<?php echo esc_attr($back); ?>" placeholder="https://" style="width:100%;" />
            <button type="button" class="button" id="dtarot-back-pick"><?php esc_html_e('Choose','daily-tarot'); ?></button>
        </div>
        <?php if ($back): ?>
            <p><img src="<?php echo esc_url($back); ?>" alt="" style="max-width:160px;height:auto;border:1px solid #ddd;border-radius:6px;"></p>
        <?php endif; ?>

        <hr />

        <p>
            <label for="dtarot-pack-version"><strong><?php esc_html_e('Pack version','daily-tarot'); ?></strong></label><br>
            <input id="dtarot-pack-version" type="text" name="dtarot_pack_version" value="<?php echo esc_attr($packVersion); ?>" placeholder="0.0.0" class="regular-text" />
            <span class="description"><?php esc_html_e('Used for ZIP export/import versioning and downgrade protection.','daily-tarot'); ?></span>
        </p>
        <?php
    }

    public static function save(int $post_id): void {
        if (!current_user_can('edit_post', $post_id)) return;
        if (!isset($_POST['dtarot_deck_nonce']) || !wp_verify_nonce(sanitize_text_field((string)wp_unslash($_POST['dtarot_deck_nonce'])), 'dtarot_deck_save')) return;

        $system = isset($_POST['dtarot_system']) ? Cards::normalizeSystem(sanitize_text_field((string)wp_unslash($_POST['dtarot_system']))) : Cards::SYSTEM_TAROT;
        update_post_meta($post_id, self::META_SYSTEM, $system);

        $url = isset($_POST['dtarot_back_url']) ? esc_url_raw((string)wp_unslash($_POST['dtarot_back_url'])) : '';

        // If this is a Media Library attachment, try to optimize/convert to WebP.
        if ($url !== '' && function_exists('attachment_url_to_postid')) {
            $aid = (int)attachment_url_to_postid($url);
            if ($aid > 0 && class_exists(Ajax::class)) {
                $r = Ajax::maybe_convert_attachment_to_webp($aid);
                if (!empty($r['url']) && is_string($r['url'])) {
                    $url = (string)$r['url'];
                }
            }
        }
        update_post_meta($post_id, '_dtarot_back', $url);

        $packVersion = isset($_POST['dtarot_pack_version']) ? PackVersion::sanitize((string)wp_unslash($_POST['dtarot_pack_version'])) : '';
        if ($packVersion === '') {
            delete_post_meta($post_id, PackVersion::META_KEY);
        } else {
            update_post_meta($post_id, PackVersion::META_KEY, $packVersion);
        }
    }
}
