<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.WP.I18n.MissingTranslatorsComment, WordPress.WP.I18n.UnorderedPlaceholdersText

// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.EscapeOutput.UnsafePrintingFunction



use DailyTarot\Support\BookingEmails;
use DailyTarot\Support\BookingAvailability;
use DailyTarot\Support\PostTypes;

final class BookingMeta {

    private const STATUSES = ['pending','approved','denied','completed','cancelled'];

    public static function init(): void {
        add_action('add_meta_boxes_' . PostTypes::BOOKING, [__CLASS__, 'add']);
        add_action('save_post_' . PostTypes::BOOKING, [__CLASS__, 'save']);

        // Back-compat.
        add_action('add_meta_boxes_' . PostTypes::LEGACY_BOOKING, [__CLASS__, 'addLegacy']);
        add_action('save_post_' . PostTypes::LEGACY_BOOKING, [__CLASS__, 'save']);
    }

    public static function add(): void {
        add_meta_box('dtarot_booking_meta', __('Booking Details','daily-tarot'), [__CLASS__, 'render'], PostTypes::BOOKING, 'normal', 'high');
    }

    public static function addLegacy(): void {
        add_meta_box('dtarot_booking_meta', __('Booking Details','daily-tarot'), [__CLASS__, 'render'], PostTypes::LEGACY_BOOKING, 'normal', 'high');
    }

    public static function render($post): void {
        wp_nonce_field('dtarot_booking_save', 'dtarot_booking_nonce');
        $name = (string)get_post_meta($post->ID, '_dtarot_booking_name', true);
        $email = (string)get_post_meta($post->ID, '_dtarot_booking_email', true);
        $timezone = (string)get_post_meta($post->ID, '_dtarot_booking_timezone', true);
        $readingId = (int)get_post_meta($post->ID, '_dtarot_booking_reading_type', true);
        $status = (string)get_post_meta($post->ID, '_dtarot_booking_status', true);
        $status = $status !== '' ? $status : 'pending';
        $question = (string)get_post_meta($post->ID, '_dtarot_booking_question', true);
        $notes = (string)get_post_meta($post->ID, '_dtarot_booking_notes', true);
        $startUtc = (int)get_post_meta($post->ID, '_dtarot_booking_start_utc', true);
        $paymentStatus = (string)get_post_meta($post->ID, '_dtarot_booking_payment_status', true);
        $paymentProvider = (string)get_post_meta($post->ID, '_dtarot_booking_payment_provider', true);
        $paymentRef = (string)get_post_meta($post->ID, '_dtarot_booking_payment_ref', true);

        $adminTz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
        $startDate = $startUtc > 0 ? wp_date('Y-m-d', $startUtc, $adminTz) : '';
        $startTime = $startUtc > 0 ? wp_date('H:i', $startUtc, $adminTz) : '';

        $types = get_posts([
            'post_type' => PostTypes::readingTypeTypes(),
            'numberposts' => 200,
            'post_status' => ['publish','draft','pending','private'],
            'orderby' => 'title',
            'order' => 'ASC',
        ]);
        ?>
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row"><?php esc_html_e('Name','daily-tarot'); ?></th>
                <td><input type="text" name="dtarot_booking_name" value="<?php echo esc_attr($name); ?>" class="regular-text" /></td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Email','daily-tarot'); ?></th>
                <td><input type="email" name="dtarot_booking_email" value="<?php echo esc_attr($email); ?>" class="regular-text" /></td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('User timezone','daily-tarot'); ?></th>
                <td><input type="text" name="dtarot_booking_timezone" value="<?php echo esc_attr($timezone); ?>" class="regular-text" /></td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Reading type','daily-tarot'); ?></th>
                <td>
                    <select name="dtarot_booking_reading_type">
                        <option value="0"><?php esc_html_e('Select reading','daily-tarot'); ?></option>
                        <?php foreach ($types as $t) : ?>
                            <option value="<?php echo (int)$t->ID; ?>" <?php selected($readingId, (int)$t->ID); ?>>
                                <?php echo esc_html($t->post_title ?: __('(no title)','daily-tarot')); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Appointment (admin time)','daily-tarot'); ?></th>
                <td>
                    <input type="date" name="dtarot_booking_date" value="<?php echo esc_attr($startDate); ?>" />
                    <input type="time" name="dtarot_booking_time" value="<?php echo esc_attr($startTime); ?>" />
                    <p class="description"><?php esc_html_e('Editing this will update the appointment time.','daily-tarot'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Status','daily-tarot'); ?></th>
                <td>
                    <select name="dtarot_booking_status">
                        <?php foreach (self::STATUSES as $opt) : ?>
                            <option value="<?php echo esc_attr($opt); ?>" <?php selected($status, $opt); ?>><?php echo esc_html(ucfirst($opt)); ?></option>
                        <?php endforeach; ?>
                    </select>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Payment status','daily-tarot'); ?></th>
                <td>
                    <select name="dtarot_booking_payment_status">
                        <?php foreach (['unpaid','paid','refunded'] as $opt) : ?>
                            <option value="<?php echo esc_attr($opt); ?>" <?php selected($paymentStatus, $opt); ?>><?php echo esc_html(ucfirst($opt)); ?></option>
                        <?php endforeach; ?>
                    </select>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Payment provider','daily-tarot'); ?></th>
                <td>
                    <select name="dtarot_booking_payment_provider">
                        <option value="paypal" <?php selected($paymentProvider, 'paypal'); ?>><?php esc_html_e('PayPal','daily-tarot'); ?></option>
                        <option value="stripe" <?php selected($paymentProvider, 'stripe'); ?>><?php esc_html_e('Stripe','daily-tarot'); ?></option>
                    </select>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Payment reference','daily-tarot'); ?></th>
                <td>
                    <input type="text" name="dtarot_booking_payment_ref" value="<?php echo esc_attr($paymentRef); ?>" class="regular-text" />
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Question / Topic','daily-tarot'); ?></th>
                <td><textarea name="dtarot_booking_question" rows="4" class="large-text"><?php echo esc_textarea($question); ?></textarea></td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Internal notes','daily-tarot'); ?></th>
                <td><textarea name="dtarot_booking_notes" rows="4" class="large-text"><?php echo esc_textarea($notes); ?></textarea></td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Message to customer','daily-tarot'); ?></th>
                <td>
                    <textarea name="dtarot_booking_message" rows="4" class="large-text"></textarea>
                    <label><input type="checkbox" name="dtarot_booking_send_message" value="1" /> <?php esc_html_e('Send this message on save','daily-tarot'); ?></label>
                </td>
            </tr>
        </table>
        <?php
    }

    public static function save(int $post_id): void {
        if (!current_user_can('edit_post', $post_id)) return;
        if (!isset($_POST['dtarot_booking_nonce']) || !wp_verify_nonce(sanitize_text_field((string)wp_unslash($_POST['dtarot_booking_nonce'])), 'dtarot_booking_save')) return;

        $prevStatus = (string)get_post_meta($post_id, '_dtarot_booking_status', true);

        $name = isset($_POST['dtarot_booking_name']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_booking_name'])) : '';
        $email = isset($_POST['dtarot_booking_email']) ? sanitize_email((string)wp_unslash($_POST['dtarot_booking_email'])) : '';
        $timezone = isset($_POST['dtarot_booking_timezone']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_booking_timezone'])) : '';
        $readingId = isset($_POST['dtarot_booking_reading_type']) ? (int)wp_unslash($_POST['dtarot_booking_reading_type']) : 0;
        $status = isset($_POST['dtarot_booking_status']) ? sanitize_key((string)wp_unslash($_POST['dtarot_booking_status'])) : 'pending';
        if (!in_array($status, self::STATUSES, true)) $status = 'pending';
        $question = isset($_POST['dtarot_booking_question']) ? sanitize_textarea_field((string)wp_unslash($_POST['dtarot_booking_question'])) : '';
        $notes = isset($_POST['dtarot_booking_notes']) ? sanitize_textarea_field((string)wp_unslash($_POST['dtarot_booking_notes'])) : '';
        $paymentStatus = isset($_POST['dtarot_booking_payment_status']) ? sanitize_key((string)wp_unslash($_POST['dtarot_booking_payment_status'])) : 'unpaid';
        if (!in_array($paymentStatus, ['unpaid','paid','refunded'], true)) $paymentStatus = 'unpaid';
        $paymentProvider = isset($_POST['dtarot_booking_payment_provider']) ? sanitize_key((string)wp_unslash($_POST['dtarot_booking_payment_provider'])) : 'paypal';
        if (!in_array($paymentProvider, ['paypal','stripe'], true)) $paymentProvider = 'paypal';
        $paymentRef = isset($_POST['dtarot_booking_payment_ref']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_booking_payment_ref'])) : '';

        update_post_meta($post_id, '_dtarot_booking_name', $name);
        update_post_meta($post_id, '_dtarot_booking_email', $email);
        update_post_meta($post_id, '_dtarot_booking_timezone', $timezone);
        update_post_meta($post_id, '_dtarot_booking_reading_type', $readingId);
        update_post_meta($post_id, '_dtarot_booking_status', $status);
        update_post_meta($post_id, '_dtarot_booking_question', $question);
        update_post_meta($post_id, '_dtarot_booking_notes', $notes);
        update_post_meta($post_id, '_dtarot_booking_payment_status', $paymentStatus);
        update_post_meta($post_id, '_dtarot_booking_payment_provider', $paymentProvider);
        update_post_meta($post_id, '_dtarot_booking_payment_ref', $paymentRef);

        $date = isset($_POST['dtarot_booking_date']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_booking_date'])) : '';
        $time = isset($_POST['dtarot_booking_time']) ? sanitize_text_field((string)wp_unslash($_POST['dtarot_booking_time'])) : '';
        if ($date !== '' && $time !== '') {
            $tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
            $dt = \DateTimeImmutable::createFromFormat('Y-m-d H:i', $date . ' ' . $time, $tz);
            if ($dt) {
                $startUtc = (int)$dt->setTimezone(new \DateTimeZone('UTC'))->format('U');
                update_post_meta($post_id, '_dtarot_booking_start_utc', $startUtc);
                $duration = BookingAvailability::getReadingDuration($readingId);
                $endUtc = $startUtc + ($duration * 60);
                update_post_meta($post_id, '_dtarot_booking_end_utc', $endUtc);
                wp_update_post([
                    'ID' => $post_id,
                    'post_title' => sprintf('%s - %s %s', $name, $date, $time),
                ]);
            }
        } else {
            $startUtc = (int)get_post_meta($post_id, '_dtarot_booking_start_utc', true);
            if ($startUtc > 0) {
                $duration = BookingAvailability::getReadingDuration($readingId);
                $endUtc = $startUtc + ($duration * 60);
                update_post_meta($post_id, '_dtarot_booking_end_utc', $endUtc);
            }
        }

        $message = isset($_POST['dtarot_booking_message']) ? sanitize_textarea_field((string)wp_unslash($_POST['dtarot_booking_message'])) : '';
        $sendMessage = !empty($_POST['dtarot_booking_send_message']);
        if ($sendMessage && $message !== '' && $email !== '') {
            $subject = sprintf(__('Message about your reading (%s)','daily-tarot'), get_bloginfo('name'));
            $body = $message;
            wp_mail($email, $subject, $body, BookingEmails::headers());
        }

        if ($prevStatus !== $status && $email !== '') {
            $payload = self::bookingPayload($post_id, $status, $message);
            if ($status === 'approved') {
                BookingEmails::sendUserConfirmed($payload);
            } elseif ($status === 'denied') {
                BookingEmails::sendUserDeclined($payload);
            }
            self::log($post_id, 'status', 'Status changed to ' . $status);
        }
    }

    public static function bookingPayload(int $postId, string $status, string $message): array {
        $name = (string)get_post_meta($postId, '_dtarot_booking_name', true);
        $email = (string)get_post_meta($postId, '_dtarot_booking_email', true);
        $timezone = (string)get_post_meta($postId, '_dtarot_booking_timezone', true);
        $readingId = (int)get_post_meta($postId, '_dtarot_booking_reading_type', true);
        $question = (string)get_post_meta($postId, '_dtarot_booking_question', true);
        $startUtc = (int)get_post_meta($postId, '_dtarot_booking_start_utc', true);
        $paymentProvider = (string)get_post_meta($postId, '_dtarot_booking_payment_provider', true);
        $paymentUrl = \DailyTarot\Support\BookingSettings::paymentUrlFor($paymentProvider);

        $adminTz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
        $adminDate = $startUtc ? wp_date('Y-m-d', $startUtc, $adminTz) : '';
        $adminTime = $startUtc ? wp_date('H:i', $startUtc, $adminTz) : '';

        $userDate = $adminDate;
        $userTime = $adminTime;
        if ($timezone !== '') {
            try {
                $userTz = new \DateTimeZone($timezone);
                $userDate = $startUtc ? wp_date('Y-m-d', $startUtc, $userTz) : $userDate;
                $userTime = $startUtc ? wp_date('H:i', $startUtc, $userTz) : $userTime;
            } catch (\Exception $e) {
            }
        }

        return [
            'name' => $name,
            'email' => $email,
            'reading' => $readingId > 0 ? (string)get_the_title($readingId) : '',
            'date' => $userDate,
            'time' => $userTime,
            'timezone' => $timezone,
            'status' => $status,
            'question' => $question,
            'message' => $message,
            'payment_provider' => $paymentProvider,
            'payment_url' => $paymentUrl,
            'admin_date' => $adminDate,
            'admin_time' => $adminTime,
            'admin_timezone' => (string)$adminTz->getName(),
        ];
    }

    private static function log(int $id, string $action, string $note): void {
        $log = get_post_meta($id, '_dtarot_booking_log', true);
        if (!is_array($log)) $log = [];
        $log[] = [
            'time' => gmdate('c'),
            'action' => $action,
            'note' => $note,
        ];
        update_post_meta($id, '_dtarot_booking_log', $log);
    }
}
