<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Support\BookingEmails;
use DailyTarot\Support\PostTypes;

final class BookingActions {

    public static function init(): void {
        add_filter('post_row_actions', [__CLASS__, 'rowActions'], 10, 2);
        add_action('admin_post_dtarot_booking_action', [__CLASS__, 'handleAction']);
        add_action('admin_post_dtarot_booking_resend', [__CLASS__, 'handleResend']);
        add_action('admin_post_dtarot_booking_erase', [__CLASS__, 'handleErase']);
    }

    public static function rowActions(array $actions, \WP_Post $post): array {
        if (!isset($post->post_type) || !PostTypes::isBookingType((string)$post->post_type)) return $actions;

        $status = (string)get_post_meta($post->ID, '_dtarot_booking_status', true);
        if ($status === '') $status = 'pending';

        $base = admin_url('admin-post.php');
        $nonce = wp_create_nonce('dtarot_booking_action_' . $post->ID);

        $make = function (string $label, string $status) use ($base, $nonce, $post): string {
            $url = add_query_arg([
                'action' => 'dtarot_booking_action',
                'booking_id' => $post->ID,
                'status' => $status,
                '_wpnonce' => $nonce,
            ], $base);
            return '<a href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
        };

        if ($status === 'pending') {
            $actions['dtarot_approve'] = $make(__('Approve','daily-tarot'), 'approved');
            $actions['dtarot_deny'] = $make(__('Deny','daily-tarot'), 'denied');
        } elseif ($status === 'approved') {
            $actions['dtarot_complete'] = $make(__('Complete','daily-tarot'), 'completed');
            $actions['dtarot_cancel'] = $make(__('Cancel','daily-tarot'), 'cancelled');
        }

        $resendUrl = add_query_arg([
            'action' => 'dtarot_booking_resend',
            'booking_id' => $post->ID,
            '_wpnonce' => $nonce,
        ], $base);
        $actions['dtarot_resend'] = '<a href="' . esc_url($resendUrl) . '">' . esc_html__('Resend email','daily-tarot') . '</a>';

        $eraseUrl = add_query_arg([
            'action' => 'dtarot_booking_erase',
            'booking_id' => $post->ID,
            '_wpnonce' => $nonce,
        ], $base);
        $actions['dtarot_erase'] = '<a href="' . esc_url($eraseUrl) . '">' . esc_html__('Erase personal data','daily-tarot') . '</a>';

        return $actions;
    }

    public static function handleAction(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));
        $id = isset($_GET['booking_id']) ? absint(wp_unslash($_GET['booking_id'])) : 0;
        $status = isset($_GET['status']) ? sanitize_key((string)wp_unslash($_GET['status'])) : '';
        $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field((string)wp_unslash($_GET['_wpnonce'])) : '';
        if ($id <= 0 || $status === '') wp_die(esc_html__('Missing booking.','daily-tarot'));
        if (!wp_verify_nonce($nonce, 'dtarot_booking_action_' . $id)) wp_die(esc_html__('Invalid request.','daily-tarot'));

        $allowed = ['approved','denied','completed','cancelled'];
        if (!in_array($status, $allowed, true)) {
            wp_die(esc_html__('Invalid status.','daily-tarot'));
        }

        update_post_meta($id, '_dtarot_booking_status', $status);
        self::log($id, 'status', 'Status set to ' . $status);

        $payload = BookingMeta::bookingPayload($id, $status, '');
        if ($status === 'approved') {
            BookingEmails::sendUserConfirmed($payload);
        } elseif ($status === 'denied') {
            BookingEmails::sendUserDeclined($payload);
        }

        do_action('dtarot_booking_status_changed', $id, $status, $payload);

        wp_safe_redirect(wp_get_referer() ? wp_get_referer() : admin_url('edit.php?post_type=' . PostTypes::BOOKING));
        exit;
    }

    public static function handleResend(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));
        $id = isset($_GET['booking_id']) ? absint(wp_unslash($_GET['booking_id'])) : 0;
        $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field((string)wp_unslash($_GET['_wpnonce'])) : '';
        if ($id <= 0) wp_die(esc_html__('Missing booking.','daily-tarot'));
        if (!wp_verify_nonce($nonce, 'dtarot_booking_action_' . $id)) wp_die(esc_html__('Invalid request.','daily-tarot'));

        $status = (string)get_post_meta($id, '_dtarot_booking_status', true);
        if ($status === '') $status = 'pending';
        $payload = BookingMeta::bookingPayload($id, $status, '');

        if ($status === 'approved') {
            BookingEmails::sendUserConfirmed($payload);
        } elseif ($status === 'denied') {
            BookingEmails::sendUserDeclined($payload);
        } else {
            BookingEmails::sendUserRequest($payload);
        }

        self::log($id, 'email', 'Resent email');
        wp_safe_redirect(wp_get_referer() ? wp_get_referer() : admin_url('edit.php?post_type=' . PostTypes::BOOKING));
        exit;
    }

    public static function handleErase(): void {
        if (!current_user_can('manage_options')) wp_die(esc_html__('Forbidden','daily-tarot'));
        $id = isset($_GET['booking_id']) ? absint(wp_unslash($_GET['booking_id'])) : 0;
        $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field((string)wp_unslash($_GET['_wpnonce'])) : '';
        if ($id <= 0) wp_die(esc_html__('Missing booking.','daily-tarot'));
        if (!wp_verify_nonce($nonce, 'dtarot_booking_action_' . $id)) wp_die(esc_html__('Invalid request.','daily-tarot'));

        update_post_meta($id, '_dtarot_booking_name', '');
        update_post_meta($id, '_dtarot_booking_email', '');
        update_post_meta($id, '_dtarot_booking_question', '');
        self::log($id, 'erase', 'Personal data erased');

        wp_safe_redirect(wp_get_referer() ? wp_get_referer() : admin_url('edit.php?post_type=' . PostTypes::BOOKING));
        exit;
    }

    private static function log(int $id, string $action, string $note): void {
        $log = get_post_meta($id, '_dtarot_booking_log', true);
        if (!is_array($log)) $log = [];
        $log[] = [
            'time' => gmdate('c'),
            'action' => $action,
            'note' => $note,
        ];
        update_post_meta($id, '_dtarot_booking_log', $log);
    }
}
