<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped

// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Calendar\DayEntry;
use DailyTarot\Calendar\DayEntryRepository;
use DailyTarot\Registry\Cards;
use DailyTarot\Packs\ManifestValidator;
use DailyTarot\Support\DefaultDecks;
use DailyTarot\Support\PostTypes;
use DailyTarot\Support\StarterDecks;

final class Backup {

    private const FORMAT_VERSION = 1;
    private const REPORT_TTL = 600;
    private const INSTALLED_OPTION = 'dtarot_installed_packs';
    private const LENORMAND_META_OPTION = 'dtarot_card_meta_lenormand';

    /**
     * Optional redirect target for this request.
     *
     * Used to send import flows back to the calling admin page (e.g., Content)
     * while preserving existing Settings redirects by default.
     */
    private static ?string $redirectOverride = null;

    private static function filesystem(): ?\WP_Filesystem_Base {
        if (!function_exists('WP_Filesystem')) {
            $inc = ABSPATH . 'wp-admin/includes/file.php';
            if (is_readable($inc)) {
                require_once $inc;
            }
        }

        global $wp_filesystem;
        if (!is_object($wp_filesystem)) {
            WP_Filesystem();
        }

        return is_object($wp_filesystem) ? $wp_filesystem : null;
    }

    /** @return array<string,array<string,mixed>> */
    private static function exportLenormandMetaMap(): array {
        $out = [];
        foreach (Cards::forSystem(Cards::SYSTEM_LENORMAND) as $id => $_name) {
            $meta = Cards::meta($id);
            if (is_array($meta) && $meta) {
                $out[$id] = $meta;
            }
        }
        return $out;
    }

    /** @return array<string,array<string,mixed>> */
    private static function sanitizeLenormandMetaMap($in): array {
        if (!is_array($in)) return [];

        $registry = Cards::forSystem(Cards::SYSTEM_LENORMAND);
        $allowedRanks = ['A','6','7','8','9','10','J','Q','K'];
        $allowedSuits = ['hearts','spades','diamonds','clubs'];

        $out = [];
        foreach ($in as $cardId => $row) {
            if (!is_string($cardId) || !isset($registry[$cardId]) || !is_array($row)) continue;
            $cardId = sanitize_text_field($cardId);
            if ($cardId === '') continue;

            $clean = [];

            $subject = isset($row['subject']) && !is_array($row['subject']) ? sanitize_text_field((string)$row['subject']) : '';
            $modifier = isset($row['modifier']) && !is_array($row['modifier']) ? sanitize_text_field((string)$row['modifier']) : '';
            $extended = isset($row['extended']) && !is_array($row['extended']) ? sanitize_text_field((string)$row['extended']) : '';
            if ($subject !== '') $clean['subject'] = $subject;
            if ($modifier !== '') $clean['modifier'] = $modifier;
            if ($extended !== '') $clean['extended'] = $extended;

            $isSignifier = !empty($row['is_signifier']);
            if ($isSignifier) $clean['is_signifier'] = true;

            if (isset($row['playing']) && is_array($row['playing'])) {
                $rank = isset($row['playing']['rank']) && !is_array($row['playing']['rank']) ? trim((string)$row['playing']['rank']) : '';
                $suit = isset($row['playing']['suit']) && !is_array($row['playing']['suit']) ? trim((string)$row['playing']['suit']) : '';

                if ($rank !== '' && !in_array($rank, $allowedRanks, true)) $rank = '';
                if ($suit !== '' && !in_array($suit, $allowedSuits, true)) $suit = '';

                if ($rank !== '' || $suit !== '') {
                    $clean['playing'] = [
                        'rank' => $rank,
                        'suit' => $suit,
                    ];
                }
            }

            if ($clean) {
                $out[$cardId] = $clean;
            }
        }

        return $out;
    }

    public static function init(): void {
        add_action('admin_post_dtarot_export', [__CLASS__, 'export']);
        add_action('admin_post_dtarot_import', [__CLASS__, 'import']);

        // ZIP pack export (sellable-pack format)
        add_action('admin_post_dtarot_export_deck_zip', [__CLASS__, 'export_deck_zip']);
        add_action('admin_post_dtarot_export_pack_zip', [__CLASS__, 'export_pack_zip']);

        // ZIP pack import (upload)
        add_action('admin_post_dtarot_import_deck_zip', [__CLASS__, 'import_deck_zip']);
        add_action('admin_post_dtarot_import_pack_zip', [__CLASS__, 'import_pack_zip']);

        // ZIP pack import (URL) - intended for EDD/Woo downloads
        add_action('admin_post_dtarot_import_deck_zip_url', [__CLASS__, 'import_deck_zip_url']);
        add_action('admin_post_dtarot_import_pack_zip_url', [__CLASS__, 'import_pack_zip_url']);

        // Installed pack actions (one-click updates)
        add_action('admin_post_dtarot_installed_pack_update_url', [__CLASS__, 'installed_pack_update_url']);
        add_action('admin_post_dtarot_installed_pack_reinstall_url', [__CLASS__, 'installed_pack_reinstall_url']);
    }

    public static function installed_pack_update_url(): void {
        self::runInstalledPackUrlAction(false);
    }

    public static function installed_pack_reinstall_url(): void {
        self::runInstalledPackUrlAction(true);
    }

    private static function runInstalledPackUrlAction(bool $allowDowngrade): void {
        self::check('dtarot_backup');

        $type = isset($_POST['pack_type']) ? sanitize_key((string)wp_unslash($_POST['pack_type'])) : '';
        $slug = isset($_POST['pack_slug']) ? sanitize_title((string)wp_unslash($_POST['pack_slug'])) : '';
        if (!in_array($type, ['deck', 'meaning_pack'], true) || $slug === '') {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'invalid_url']));
            exit;
        }

        $installed = self::installedPacks();
        $k = $type . ':' . $slug;
        $entry = isset($installed[$k]) && is_array($installed[$k]) ? $installed[$k] : null;
        if (!$entry) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'invalid_url']));
            exit;
        }

        $url = isset($entry['source_url']) && is_string($entry['source_url']) ? trim($entry['source_url']) : '';
        if ($url === '' || !preg_match('#^https?://#i', $url)) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'invalid_url']));
            exit;
        }

        $max = ($type === 'deck') ? (50 * 1024 * 1024) : (10 * 1024 * 1024);
        $tmp = self::downloadZipToTemp($url, $max);
        if ($tmp === '') {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'download_failed']));
            exit;
        }

        $opts = [
            'dry_run' => false,
            'overwrite' => true,
            'allow_downgrade' => $allowDowngrade,
            'source_url' => $url,
        ];

        // These methods will redirect+exit; they will also delete the downloaded ZIP when source_url is set.
        if ($type === 'deck') {
            self::importDeckZipFromPath($tmp, $opts);
        }
        self::importPackZipFromPath($tmp, $opts);
    }

    /** @return array<string,array<string,mixed>> */
    public static function installedPacks(): array {
        $v = get_option(self::INSTALLED_OPTION, []);
        return is_array($v) ? $v : [];
    }

    /** @param array<string,array<string,mixed>> $packs */
    private static function saveInstalledPacks(array $packs): void {
        update_option(self::INSTALLED_OPTION, $packs, false);
    }

    /** @param array<string,mixed> $entry */
    private static function recordInstalledPack(string $type, string $slug, array $entry): void {
        $type = sanitize_key($type);
        $slug = sanitize_title($slug);
        if ($type === '' || $slug === '') return;

        $packs = self::installedPacks();
        $key = $type . ':' . $slug;
        $packs[$key] = $entry;
        self::saveInstalledPacks($packs);
    }

    private static function check(string $action = 'dtarot_backup'): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Forbidden','daily-tarot'));
        }
        check_admin_referer($action);
    }

    private static function settingsUrl(array $args = []): string {
        $url = is_string(self::$redirectOverride) && self::$redirectOverride !== ''
            ? self::$redirectOverride
            : admin_url('admin.php?page=daily-tarot-settings');
        if (!$args) return $url;
        return (string)add_query_arg($args, $url);
    }

    /**
     * Set redirect override from POST[redirect_to] if it's a safe admin URL.
     *
     * Must be called after nonce/cap checks.
     */
    private static function maybeSetRedirectOverrideFromPost(): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified by check() in caller.
        $raw = isset($_POST['redirect_to']) ? (string)wp_unslash($_POST['redirect_to']) : '';
        $raw = trim($raw);
        if ($raw === '') return;

        $candidate = esc_url_raw($raw);
        if ($candidate === '') return;

        // Only allow redirects back into wp-admin on this site.
        $adminBase = (string)admin_url();
        if ($adminBase === '' || stripos($candidate, $adminBase) !== 0) return;

        self::$redirectOverride = $candidate;
    }

    /** Public so Settings can display detailed import reports. */
    public static function readReport(string $token): ?array {
        $token = preg_replace('/[^a-zA-Z0-9_-]/', '', $token);
        if (!is_string($token) || $token === '') return null;
        $key = self::reportKey($token);
        $report = get_transient($key);
        return is_array($report) ? $report : null;
    }

    private static function storeReport(array $report): string {
        $token = wp_generate_password(18, false, false);
        $key = self::reportKey($token);
        set_transient($key, $report, self::REPORT_TTL);
        return $token;
    }

    private static function reportKey(string $token): string {
        $uid = function_exists('get_current_user_id') ? (int)get_current_user_id() : 0;
        return 'dtarot_report_' . $uid . '_' . $token;
    }

    private static function redirectReport(string $status, array $params, array $report): void {
        $token = self::storeReport($report);
        $params = array_merge($params, [
            'dtarot_backup' => $status,
            'report' => $token,
        ]);
        wp_safe_redirect(self::settingsUrl($params));
        exit;
    }

    public static function export(): void {
        self::check('dtarot_backup');

        $decks = get_posts([
            'post_type' => PostTypes::deckTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
        ]);

        $deckData = [];
        foreach ($decks as $deck) {
            $cards = get_post_meta($deck->ID, '_dtarot_cards', true);
            if (!is_array($cards)) $cards = [];

            $back = (string)get_post_meta($deck->ID, '_dtarot_back', true);
            $system = Cards::normalizeSystem((string)get_post_meta($deck->ID, '_dtarot_system', true));

            $deckData[] = [
                'id' => (int)$deck->ID,
                'slug' => (string)$deck->post_name,
                'title' => (string)$deck->post_title,
                'status' => (string)$deck->post_status,
                'meta' => [
                    'system' => $system,
                    'back' => $back,
                    'cards' => $cards,
                ],
            ];
        }

        $packs = get_posts([
            'post_type' => PostTypes::meaningPackTypes(),
            'numberposts' => -1,
            'post_status' => ['publish','draft','pending','private'],
        ]);

        $packData = [];
        foreach ($packs as $pack) {
            $meanings = get_post_meta($pack->ID, '_dtarot_meanings', true);
            if (!is_array($meanings)) $meanings = [];

            $system = Cards::normalizeSystem((string)get_post_meta($pack->ID, '_dtarot_system', true));

            $packData[] = [
                'id' => (int)$pack->ID,
                'slug' => (string)$pack->post_name,
                'title' => (string)$pack->post_title,
                'status' => (string)$pack->post_status,
                'meta' => [
                    'system' => $system,
                    'meanings' => $meanings,
                ],
            ];
        }

        $calendar = [];
        foreach (DayEntryRepository::all() as $date => $entry) {
            if (!is_string($date) || !is_array($entry)) continue;
            // Ensure legacy keys exist and values are strings.
            $calendar[$date] = DayEntry::sanitizeArray($entry);
        }

        $payload = [
            'meta' => [
                'plugin' => 'daily-tarot',
                'version' => defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '',
                'exported_at' => gmdate('c'),
                'site' => function_exists('home_url') ? (string)home_url('/') : '',
            ],
            // Structured, system-specific card metadata (portable; currently Lenormand only).
            'card_meta' => [
                Cards::SYSTEM_LENORMAND => self::exportLenormandMetaMap(),
            ],
            'decks' => $deckData,
            'meaning_packs' => $packData,
            'calendar_entries' => $calendar,
        ];

        $json = wp_json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        if (!is_string($json)) {
            wp_die(esc_html__('Export failed (could not encode JSON).','daily-tarot'));
        }

        $filename = 'daily-tarot-backup-' . gmdate('Ymd-His') . '.json';

        nocache_headers();
        header('Content-Type: application/json; charset=' . get_option('blog_charset'));
        header('Content-Disposition: attachment; filename=' . $filename);
        header('X-Content-Type-Options: nosniff');
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON file download.
        echo $json;
        exit;
    }

    public static function import(): void {
        self::check('dtarot_backup');

        if (!isset($_FILES['dtarot_backup_file']) || !is_array($_FILES['dtarot_backup_file'])) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'missing_file']));
            exit;
        }

        $file = $_FILES['dtarot_backup_file'];
        $tmp = isset($file['tmp_name']) ? (string)$file['tmp_name'] : '';
        $size = isset($file['size']) ? absint($file['size']) : 0;
        $name = isset($file['name']) ? (string)$file['name'] : '';
        $err = isset($file['error']) ? (int)$file['error'] : UPLOAD_ERR_OK;

        if ($err !== UPLOAD_ERR_OK || $tmp === '' || !is_uploaded_file($tmp)) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_upload']));
            exit;
        }

        $ft = wp_check_filetype_and_ext($tmp, $name);
        if (empty($ft['ext']) || strtolower((string)$ft['ext']) !== 'json') {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_json']));
            exit;
        }

        // Simple size guard (10MB).
        if ($size <= 0 || $size > 10 * 1024 * 1024) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'file_too_large']));
            exit;
        }

        $raw = file_get_contents($tmp);
        if (!is_string($raw) || $raw === '') {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'empty_file']));
            exit;
        }

        $data = json_decode($raw, true);
        if (!is_array($data)) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_json']));
            exit;
        }

        $decks = isset($data['decks']) && is_array($data['decks']) ? $data['decks'] : [];
        $packs = isset($data['meaning_packs']) && is_array($data['meaning_packs']) ? $data['meaning_packs'] : [];
        $calendarIn = isset($data['calendar_entries']) && is_array($data['calendar_entries']) ? $data['calendar_entries'] : [];

        // Optional: import portable card meta.
        $metaIn = isset($data['card_meta']) && is_array($data['card_meta']) ? $data['card_meta'] : [];
        $lenormandMetaIn = isset($metaIn[Cards::SYSTEM_LENORMAND]) ? $metaIn[Cards::SYSTEM_LENORMAND] : [];

        $overwriteExisting = isset($_POST['overwrite_existing']) && (sanitize_text_field((string)wp_unslash($_POST['overwrite_existing'])) === '1');

        // Store/merge Lenormand metadata into an option so it survives round-trips.
        $cleanLenormandMeta = self::sanitizeLenormandMetaMap($lenormandMetaIn);
        if ($cleanLenormandMeta) {
            $existingLenormandMeta = get_option(self::LENORMAND_META_OPTION, []);
            if (!is_array($existingLenormandMeta)) $existingLenormandMeta = [];

            // If overwrite_existing is set, prefer incoming; otherwise preserve existing.
            $merged = $overwriteExisting
                ? array_merge($existingLenormandMeta, $cleanLenormandMeta)
                : array_merge($cleanLenormandMeta, $existingLenormandMeta);

            update_option(self::LENORMAND_META_OPTION, $merged, false);
        }

        $deckIdMap = [];
        $packIdMap = [];

        $deckCount = 0;
        $deckSkipped = 0;
        foreach ($decks as $deck) {
            if (!is_array($deck)) continue;

            $oldId = isset($deck['id']) ? (int)$deck['id'] : 0;
            $slug = isset($deck['slug']) && is_string($deck['slug']) ? sanitize_title($deck['slug']) : '';
            $title = isset($deck['title']) && is_string($deck['title']) ? sanitize_text_field($deck['title']) : '';
            $status = isset($deck['status']) && is_string($deck['status']) ? sanitize_key($deck['status']) : 'draft';
            if (!in_array($status, ['publish','draft','pending','private'], true)) $status = 'draft';

            $existing = null;
            if ($slug !== '') {
                $existing = get_page_by_path($slug, OBJECT, PostTypes::DECK);
                if (!$existing) {
                    $existing = get_page_by_path($slug, OBJECT, PostTypes::LEGACY_DECK);
                }
            }

            if ($existing && isset($existing->ID) && !$overwriteExisting) {
                $deckSkipped++;
                continue;
            }

            $postarr = [
                'post_type' => PostTypes::DECK,
                'post_title' => $title !== '' ? $title : $slug,
                'post_status' => $status,
            ];

            if ($existing && isset($existing->ID)) {
                $postarr['ID'] = (int)$existing->ID;
                $newId = wp_update_post($postarr, true);
            } else {
                if ($slug !== '') {
                    $postarr['post_name'] = $slug;
                }
                $newId = wp_insert_post($postarr, true);
            }

            if (is_wp_error($newId)) {
                continue;
            }

            $newId = (int)$newId;
            if ($oldId > 0) $deckIdMap[$oldId] = $newId;

            $meta = isset($deck['meta']) && is_array($deck['meta']) ? $deck['meta'] : [];
            $system = isset($meta['system']) && is_string($meta['system']) ? Cards::normalizeSystem($meta['system']) : Cards::SYSTEM_TAROT;
            $back = isset($meta['back']) && is_string($meta['back']) ? esc_url_raw($meta['back']) : '';
            $cards = isset($meta['cards']) && is_array($meta['cards']) ? $meta['cards'] : [];

            $cleanCards = [];
            foreach ($cards as $cardId => $url) {
                if (!is_string($cardId) || (!is_string($url) && !is_numeric($url))) continue;
                $cardId = sanitize_text_field($cardId);
                if ($cardId === '') continue;
                $cleanCards[$cardId] = esc_url_raw((string)$url);
            }

            update_post_meta($newId, '_dtarot_system', $system);
            update_post_meta($newId, '_dtarot_back', $back);
            update_post_meta($newId, '_dtarot_cards', $cleanCards);
            $deckCount++;
        }

        $packCount = 0;
        $packSkipped = 0;
        foreach ($packs as $pack) {
            if (!is_array($pack)) continue;

            $oldId = isset($pack['id']) ? (int)$pack['id'] : 0;
            $slug = isset($pack['slug']) && is_string($pack['slug']) ? sanitize_title($pack['slug']) : '';
            $title = isset($pack['title']) && is_string($pack['title']) ? sanitize_text_field($pack['title']) : '';
            $status = isset($pack['status']) && is_string($pack['status']) ? sanitize_key($pack['status']) : 'draft';
            if (!in_array($status, ['publish','draft','pending','private'], true)) $status = 'draft';

            $existing = null;
            if ($slug !== '') {
                $existing = get_page_by_path($slug, OBJECT, PostTypes::MEANING_PACK);
                if (!$existing) {
                    $existing = get_page_by_path($slug, OBJECT, PostTypes::LEGACY_MEANING_PACK);
                }
            }

            if ($existing && isset($existing->ID) && !$overwriteExisting) {
                $packSkipped++;
                continue;
            }

            $postarr = [
                'post_type' => PostTypes::MEANING_PACK,
                'post_title' => $title !== '' ? $title : $slug,
                'post_status' => $status,
            ];

            if ($existing && isset($existing->ID)) {
                $postarr['ID'] = (int)$existing->ID;
                $newId = wp_update_post($postarr, true);
            } else {
                if ($slug !== '') {
                    $postarr['post_name'] = $slug;
                }
                $newId = wp_insert_post($postarr, true);
            }

            if (is_wp_error($newId)) {
                continue;
            }

            $newId = (int)$newId;
            if ($oldId > 0) $packIdMap[$oldId] = $newId;

            $meta = isset($pack['meta']) && is_array($pack['meta']) ? $pack['meta'] : [];
            $system = isset($meta['system']) && is_string($meta['system']) ? Cards::normalizeSystem($meta['system']) : Cards::SYSTEM_TAROT;
            $meanings = isset($meta['meanings']) && is_array($meta['meanings']) ? $meta['meanings'] : [];

            $allowedFields = ['upright','reversed','keywords','short','long','correspondences','symbols'];
            $cleanMeanings = [];

            foreach ($meanings as $cardId => $meaning) {
                if (!is_string($cardId) || !is_array($meaning)) continue;
                $cardId = sanitize_text_field($cardId);
                if ($cardId === '') continue;

                $clean = [];
                foreach ($allowedFields as $field) {
                    if (!isset($meaning[$field]) || is_array($meaning[$field])) {
                        $clean[$field] = '';
                        continue;
                    }
                    $clean[$field] = wp_kses_post((string)$meaning[$field]);
                }
                $cleanMeanings[$cardId] = $clean;
            }

            update_post_meta($newId, '_dtarot_system', $system);
            update_post_meta($newId, '_dtarot_meanings', $cleanMeanings);
            $packCount++;
        }

        // Calendar entries
        $replaceCalendar = isset($_POST['replace_calendar']) && ((string)wp_unslash($_POST['replace_calendar']) === '1');

        $cleanCalendar = [];
        foreach ($calendarIn as $date => $entry) {
            if (!is_string($date) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) continue;
            if (!is_array($entry)) continue;

            $deck = isset($entry['deck']) && !is_array($entry['deck']) ? (string)$entry['deck'] : '';
            $pack = isset($entry['pack']) && !is_array($entry['pack']) ? (string)$entry['pack'] : '';

            // Remap IDs if they look numeric.
            $deckInt = (int)$deck;
            if ($deckInt > 0 && isset($deckIdMap[$deckInt])) {
                $deck = (string)$deckIdMap[$deckInt];
            }
            $packInt = (int)$pack;
            if ($packInt > 0 && isset($packIdMap[$packInt])) {
                $pack = (string)$packIdMap[$packInt];
            }

            $status = isset($entry['status']) && !is_array($entry['status']) ? sanitize_key((string)$entry['status']) : 'draft';
            if (!in_array($status, ['publish','draft'], true)) $status = 'draft';

            $cleanCalendar[$date] = [
                'deck' => sanitize_text_field($deck),
                'card' => isset($entry['card']) && !is_array($entry['card']) ? sanitize_text_field((string)$entry['card']) : '',
                'pack' => sanitize_text_field($pack),
                'status' => $status,
                'content' => isset($entry['content']) && !is_array($entry['content']) ? wp_kses_post((string)$entry['content']) : '',
                'daily_text' => isset($entry['daily_text']) && !is_array($entry['daily_text']) ? wp_kses_post((string)$entry['daily_text']) : '',
            ];
        }

        if ($replaceCalendar) {
            DayEntryRepository::replaceAll($cleanCalendar);
        } else {
            DayEntryRepository::mergeAll($cleanCalendar);
        }

        wp_safe_redirect(self::settingsUrl([
            'dtarot_backup' => 'import_ok',
            'decks' => (string)$deckCount,
            'packs' => (string)$packCount,
            'decks_skipped' => (string)$deckSkipped,
            'packs_skipped' => (string)$packSkipped,
            'days' => (string)count($cleanCalendar),
        ]));
        exit;
    }

    public static function export_deck_zip(): void {
        self::check('dtarot_backup');

        if (!class_exists('ZipArchive')) {
            wp_die(esc_html__('ZIP export requires PHP ZipArchive support on the server.','daily-tarot'));
        }

        $deckId = isset($_POST['deck_id']) ? (int)wp_unslash($_POST['deck_id']): 0;
        if ($deckId <= 0) {
            wp_die(esc_html__('Missing deck ID.','daily-tarot'));
        }

        $deck = get_post($deckId);
        if (!$deck || !isset($deck->ID) || !isset($deck->post_type) || !PostTypes::isDeckType((string)$deck->post_type)) {
            wp_die(esc_html__('Deck not found.','daily-tarot'));
        }

        $slug = $deck->post_name ? (string)$deck->post_name : sanitize_title((string)$deck->post_title);
        if ($slug === '') $slug = 'deck';

        $packVersion = (string)get_post_meta($deckId, '_dtarot_pack_version', true);
        $packVersion = trim($packVersion);
        if ($packVersion === '') {
            $packVersion = '1.0.0';
        }

        $cards = get_post_meta($deckId, '_dtarot_cards', true);
        if (!is_array($cards)) $cards = [];
        $back = (string)get_post_meta($deckId, '_dtarot_back', true);
        $system = Cards::normalizeSystem((string)get_post_meta($deckId, '_dtarot_system', true));

        $zipPath = wp_tempnam('dtarot-deck.zip');
        if (!is_string($zipPath) || $zipPath === '') {
            wp_die(esc_html__('Could not create temporary ZIP.','daily-tarot'));
        }

        $zip = new \ZipArchive();
        if ($zip->open($zipPath, \ZipArchive::OVERWRITE) !== true) {
            wp_delete_file($zipPath);
            wp_die(esc_html__('Could not open ZIP for writing.','daily-tarot'));
        }

        $files = [];
        $errors = [];
        $included = 0;
        $tmpFiles = [];

        // Back image
        $manifestCards = [];
        $backRel = '';
        if ($back !== '') {
            $tmp = self::downloadToTempFile($back, 25 * 1024 * 1024);
            if ($tmp !== '') {
                $ext = self::guessExt($back, '', 'jpg');
                $backRel = 'back.' . $ext;
                $zip->addFile($tmp, $backRel);
                $files[] = ['path' => $backRel, 'sha256' => hash_file('sha256', $tmp) ?: ''];
                $tmpFiles[] = $tmp;
                $included++;
            } else {
                $errors[] = 'Back: download failed';
            }
        }

        // Card images
        $registry = Cards::forSystem($system);
        foreach ($cards as $cardId => $url) {
            if (!is_string($cardId) || (!is_string($url) && !is_numeric($url))) continue;
            $cardId = sanitize_text_field($cardId);
            if ($cardId === '' || !isset($registry[$cardId])) continue;

            $url = (string)$url;
            if (trim($url) === '') continue;

            $tmp = self::downloadToTempFile($url, 25 * 1024 * 1024);
            if ($tmp === '') {
                $errors[] = $cardId . ': download failed';
                continue;
            }

            $ext = self::guessExt($url, '', 'jpg');
            $rel = 'cards/' . $cardId . '.' . $ext;
            $zip->addFile($tmp, $rel);
            $files[] = ['path' => $rel, 'sha256' => hash_file('sha256', $tmp) ?: ''];
            $tmpFiles[] = $tmp;
            $manifestCards[$cardId] = $rel;
            $included++;
        }

        $manifest = [
            'format_version' => self::FORMAT_VERSION,
            'plugin_min_version' => defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '',
            'exported_at' => gmdate('c'),
            'type' => 'deck',
            'pack_version' => $packVersion,
            'files' => $files,
            'deck' => [
                'slug' => $slug,
                'title' => (string)$deck->post_title,
                'status' => (string)$deck->post_status,
                'system' => $system,
                'back' => $backRel,
                'cards' => $manifestCards,
            ],
        ];

        if ($errors) {
            $manifest['warnings'] = array_slice($errors, 0, 50);
        }

        $json = wp_json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        if (!is_string($json)) {
            $zip->close();
            wp_delete_file($zipPath);
            foreach ($tmpFiles as $p) {
                if (is_string($p) && $p !== '') wp_delete_file($p);
            }
            wp_die(esc_html__('Could not encode manifest JSON.','daily-tarot'));
        }
        $zip->addFromString('manifest.json', $json);
        $zip->close();

        foreach ($tmpFiles as $p) {
            if (is_string($p) && $p !== '') wp_delete_file($p);
        }

        $filename = 'dtarot-deck-' . $slug . '-' . gmdate('Ymd-His') . '.zip';
        self::sendFile($zipPath, $filename, 'application/zip');
    }

    public static function export_pack_zip(): void {
        self::check('dtarot_backup');

        if (!class_exists('ZipArchive')) {
            wp_die(esc_html__('ZIP export requires PHP ZipArchive support on the server.','daily-tarot'));
        }

        $packId = isset($_POST['pack_id']) ? (int)wp_unslash($_POST['pack_id']): 0;
        if ($packId <= 0) {
            wp_die(esc_html__('Missing meaning pack ID.','daily-tarot'));
        }

        $pack = get_post($packId);
        if (!$pack || !isset($pack->ID) || !isset($pack->post_type) || !PostTypes::isMeaningPackType((string)$pack->post_type)) {
            wp_die(esc_html__('Meaning pack not found.','daily-tarot'));
        }

        $slug = $pack->post_name ? (string)$pack->post_name : sanitize_title((string)$pack->post_title);
        if ($slug === '') $slug = 'meaning-pack';

        $packVersion = (string)get_post_meta($packId, '_dtarot_pack_version', true);
        $packVersion = trim($packVersion);
        if ($packVersion === '') {
            $packVersion = '1.0.0';
        }

        $meanings = get_post_meta($packId, '_dtarot_meanings', true);
        if (!is_array($meanings)) $meanings = [];

        $system = Cards::normalizeSystem((string)get_post_meta($packId, '_dtarot_system', true));
        $registry = Cards::forSystem($system);

        // Export only meanings for cards that exist in this system.
        $filteredMeanings = [];
        foreach ($meanings as $cardId => $meaning) {
            if (!is_string($cardId) || !is_array($meaning)) continue;
            $cardId = sanitize_text_field($cardId);
            if ($cardId === '' || !isset($registry[$cardId])) continue;
            $filteredMeanings[$cardId] = $meaning;
        }

        $zipPath = wp_tempnam('dtarot-pack.zip');
        if (!is_string($zipPath) || $zipPath === '') {
            wp_die(esc_html__('Could not create temporary ZIP.','daily-tarot'));
        }

        $zip = new \ZipArchive();
        if ($zip->open($zipPath, \ZipArchive::OVERWRITE) !== true) {
            wp_delete_file($zipPath);
            wp_die(esc_html__('Could not open ZIP for writing.','daily-tarot'));
        }

        $manifest = [
            'format_version' => self::FORMAT_VERSION,
            'plugin_min_version' => defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '',
            'exported_at' => gmdate('c'),
            'type' => 'meaning_pack',
            'pack_version' => $packVersion,
            'files' => [],
            'meaning_pack' => [
                'slug' => $slug,
                'title' => (string)$pack->post_title,
                'status' => (string)$pack->post_status,
                'system' => $system,
                // For Lenormand packs, include structured card metadata for portability.
                'card_meta' => $system === Cards::SYSTEM_LENORMAND ? self::exportLenormandMetaMap() : [],
                'meanings' => $filteredMeanings,
            ],
        ];

        $json = wp_json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        if (!is_string($json)) {
            $zip->close();
            wp_delete_file($zipPath);
            wp_die(esc_html__('Could not encode manifest JSON.','daily-tarot'));
        }

        $zip->addFromString('manifest.json', $json);
        $zip->close();

        $filename = 'dtarot-meaning-pack-' . $slug . '-' . gmdate('Ymd-His') . '.zip';
        self::sendFile($zipPath, $filename, 'application/zip');
    }

    public static function import_deck_zip(): void {
        self::check('dtarot_backup');

        if (!class_exists('ZipArchive')) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'zip_missing']));
            exit;
        }

        if (!isset($_FILES['dtarot_deck_zip']) || !is_array($_FILES['dtarot_deck_zip'])) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'missing_file']));
            exit;
        }

        $file = $_FILES['dtarot_deck_zip'];
        $tmp = isset($file['tmp_name']) ? (string)$file['tmp_name'] : '';
        $size = isset($file['size']) ? absint($file['size']) : 0;
        $name = isset($file['name']) ? (string)$file['name'] : '';
        $err = isset($file['error']) ? (int)$file['error'] : UPLOAD_ERR_OK;

        if ($err !== UPLOAD_ERR_OK || $tmp === '' || !is_uploaded_file($tmp)) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_upload']));
            exit;
        }

        $ft = wp_check_filetype_and_ext($tmp, $name);
        if (empty($ft['ext']) || strtolower((string)$ft['ext']) !== 'zip') {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'pack_import_error']));
            exit;
        }

        // Guard (50MB)
        if ($size <= 0 || $size > 50 * 1024 * 1024) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'file_too_large']));
            exit;
        }

        $opts = [
            'dry_run' => isset($_POST['dry_run']) && (sanitize_text_field((string)wp_unslash($_POST['dry_run'])) === '1'),
            'overwrite' => isset($_POST['overwrite']) && (sanitize_text_field((string)wp_unslash($_POST['overwrite'])) === '1'),
            'allow_downgrade' => isset($_POST['allow_downgrade']) && (sanitize_text_field((string)wp_unslash($_POST['allow_downgrade'])) === '1'),
        ];

        self::importDeckZipFromPath($tmp, $opts);
    }

    public static function import_deck_zip_url(): void {
        self::check('dtarot_backup');
        self::maybeSetRedirectOverrideFromPost();

        // Optional: restrict to known starter deck URLs.
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified by check() above.
        $isStarter = !empty($_POST['starter']) && (sanitize_text_field((string)wp_unslash($_POST['starter'])) === '1');

        $url = isset($_POST['zip_url']) ? esc_url_raw((string)wp_unslash($_POST['zip_url'])) : '';
        $url = trim($url);
        if ($url === '' || !preg_match('#^https?://#i', $url)) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'invalid_url']));
            exit;
        }

        if ($isStarter && class_exists(StarterDecks::class) && !StarterDecks::isAllowedZipUrl($url)) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'invalid_url']));
            exit;
        }

        $tmp = self::downloadZipToTemp($url, 50 * 1024 * 1024);
        if ($tmp === '') {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'download_failed']));
            exit;
        }

        $opts = [
            'dry_run' => isset($_POST['dry_run']) && (sanitize_text_field((string)wp_unslash($_POST['dry_run'])) === '1'),
            'overwrite' => isset($_POST['overwrite']) && (sanitize_text_field((string)wp_unslash($_POST['overwrite'])) === '1'),
            'allow_downgrade' => isset($_POST['allow_downgrade']) && (sanitize_text_field((string)wp_unslash($_POST['allow_downgrade'])) === '1'),
            'source_url' => $url,
            'set_default' => isset($_POST['set_default']) && (sanitize_text_field((string)wp_unslash($_POST['set_default'])) === '1'),
        ];
        self::importDeckZipFromPath($tmp, $opts);
    }

    /**
     * Import a bundled deck ZIP (e.g., shipped with the plugin).
     *
     * @param array{dry_run?:bool,overwrite?:bool,allow_downgrade?:bool} $opts
     */
    public static function importBundledDeckZip(string $zipPath, array $opts = []): void {
        if ($zipPath === '' || !is_readable($zipPath)) return;

        $opts = array_merge([
            'dry_run' => false,
            'overwrite' => false,
            'allow_downgrade' => false,
        ], $opts);

        self::importDeckZipFromPath($zipPath, $opts);
    }

    public static function import_pack_zip(): void {
        self::check('dtarot_backup');

        if (!class_exists('ZipArchive')) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'zip_missing']));
            exit;
        }

        if (!isset($_FILES['dtarot_pack_zip']) || !is_array($_FILES['dtarot_pack_zip'])) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'missing_file']));
            exit;
        }

        $file = $_FILES['dtarot_pack_zip'];
        $tmp = isset($file['tmp_name']) ? (string)$file['tmp_name'] : '';
        $size = isset($file['size']) ? absint($file['size']) : 0;
        $name = isset($file['name']) ? (string)$file['name'] : '';
        $err = isset($file['error']) ? (int)$file['error'] : UPLOAD_ERR_OK;

        if ($err !== UPLOAD_ERR_OK || $tmp === '' || !is_uploaded_file($tmp)) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_upload']));
            exit;
        }

        $ft = wp_check_filetype_and_ext($tmp, $name);
        if (empty($ft['ext']) || strtolower((string)$ft['ext']) !== 'zip') {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'pack_import_error']));
            exit;
        }

        // Guard (10MB)
        if ($size <= 0 || $size > 10 * 1024 * 1024) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'file_too_large']));
            exit;
        }

        $opts = [
            'dry_run' => isset($_POST['dry_run']) && (sanitize_text_field((string)wp_unslash($_POST['dry_run'])) === '1'),
            'overwrite' => isset($_POST['overwrite']) && (sanitize_text_field((string)wp_unslash($_POST['overwrite'])) === '1'),
            'allow_downgrade' => isset($_POST['allow_downgrade']) && (sanitize_text_field((string)wp_unslash($_POST['allow_downgrade'])) === '1'),
        ];
        self::importPackZipFromPath($tmp, $opts);
    }

    public static function import_pack_zip_url(): void {
        self::check('dtarot_backup');
        self::maybeSetRedirectOverrideFromPost();

        $url = isset($_POST['zip_url']) ? esc_url_raw((string)wp_unslash($_POST['zip_url'])) : '';
        $url = trim($url);
        if ($url === '' || !preg_match('#^https?://#i', $url)) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'invalid_url']));
            exit;
        }

        $tmp = self::downloadZipToTemp($url, 10 * 1024 * 1024);
        if ($tmp === '') {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'download_failed']));
            exit;
        }

        $opts = [
            'dry_run' => isset($_POST['dry_run']) && (sanitize_text_field((string)wp_unslash($_POST['dry_run'])) === '1'),
            'overwrite' => isset($_POST['overwrite']) && (sanitize_text_field((string)wp_unslash($_POST['overwrite'])) === '1'),
            'allow_downgrade' => isset($_POST['allow_downgrade']) && (sanitize_text_field((string)wp_unslash($_POST['allow_downgrade'])) === '1'),
            'source_url' => $url,
        ];
        self::importPackZipFromPath($tmp, $opts);
    }

    /** @param array{dry_run:bool,overwrite:bool,allow_downgrade?:bool,source_url?:string,set_default?:bool} $opts */
    private static function importDeckZipFromPath(string $zipPath, array $opts): void {
        if (!class_exists('ZipArchive')) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'zip_missing']));
            exit;
        }

        $workDir = self::createWorkDir();
        if ($workDir === '') {
            if (!empty($opts['source_url'])) wp_delete_file($zipPath);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'workdir']));
            exit;
        }

        $zip = new \ZipArchive();
        if ($zip->open($zipPath) !== true) {
            self::cleanupDir($workDir);
            if (!empty($opts['source_url'])) wp_delete_file($zipPath);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_zip']));
            exit;
        }

        $extractErrors = self::safeZipExtract($zip, $workDir);
        $zip->close();

        if (!empty($opts['source_url'])) {
            wp_delete_file($zipPath);
        }

        if ($extractErrors) {
            self::cleanupDir($workDir);
            self::redirectReport('import_report', [], [
                'kind' => 'deck_zip',
                'ok' => false,
                'errors' => $extractErrors,
            ]);
        }

        $manifest = self::readManifest($workDir);
        if (!is_array($manifest)) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_manifest']));
            exit;
        }

        $commonErr = self::validateManifestCommon($manifest);
        if ($commonErr !== '') {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => $commonErr]));
            exit;
        }

        if (!isset($manifest['files']) || !is_array($manifest['files'])) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'missing_files']));
            exit;
        }

        $hashErrors = self::validateFileHashes($workDir, $manifest['files']);
        if ($hashErrors) {
            self::cleanupDir($workDir);
            self::redirectReport('import_report', [], [
                'kind' => 'deck_zip',
                'ok' => false,
                'errors' => $hashErrors,
            ]);
        }

        $deck = isset($manifest['deck']) && is_array($manifest['deck']) ? $manifest['deck'] : null;
        if (!$deck) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'missing_deck']));
            exit;
        }

        $slug = isset($deck['slug']) && is_string($deck['slug']) ? sanitize_title($deck['slug']) : '';
        $title = isset($deck['title']) && is_string($deck['title']) ? sanitize_text_field($deck['title']) : '';
        if ($title === '' && $slug !== '') $title = $slug;
        if ($slug === '' && $title !== '') $slug = sanitize_title($title);
        if ($slug === '') {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'deck_slug']));
            exit;
        }

        $status = isset($deck['status']) && is_string($deck['status']) ? sanitize_key($deck['status']) : 'draft';
        if (!in_array($status, ['publish','draft','pending','private'], true)) $status = 'draft';

        $system = isset($deck['system']) && is_string($deck['system']) ? Cards::normalizeSystem($deck['system']) : Cards::SYSTEM_TAROT;

        $cards = isset($deck['cards']) && is_array($deck['cards']) ? $deck['cards'] : [];
        $backRel = isset($deck['back']) && is_string($deck['back']) ? (string)$deck['back'] : '';

        $packVersion = isset($manifest['pack_version']) && is_string($manifest['pack_version']) ? trim($manifest['pack_version']) : '';
        if ($packVersion === '') $packVersion = '0.0.0';

        $registry = Cards::forSystem($system);
        $expectedImages = 0;
        foreach ($cards as $cardId => $rel) {
            if (is_string($cardId) && isset($registry[sanitize_text_field($cardId)]) && is_string($rel) && $rel !== '') {
                $expectedImages++;
            }
        }
        if ($backRel !== '') $expectedImages++;

        if ($opts['dry_run']) {
            $missing = [];
            if ($backRel !== '') {
                $p = self::safeJoin($workDir, $backRel);
                if ($p === '' || !is_readable($p)) $missing[] = 'back: ' . $backRel;
            }
            foreach ($cards as $cardId => $relPath) {
                if (!is_string($cardId) || !is_string($relPath)) continue;
                $cardId = sanitize_text_field($cardId);
                if ($cardId === '' || !isset($registry[$cardId]) || $relPath === '') continue;
                $p = self::safeJoin($workDir, $relPath);
                if ($p === '' || !is_readable($p)) $missing[] = $cardId . ': ' . $relPath;
            }

            self::cleanupDir($workDir);
            self::redirectReport('import_report', [], [
                'kind' => 'deck_zip',
                'ok' => true,
                'dry_run' => true,
                'slug' => $slug,
                'title' => $title,
                'pack_version' => $packVersion,
                'expected_images' => $expectedImages,
                'missing' => array_slice($missing, 0, 50),
            ]);
        }

        // Create/update deck post
        $existing = get_page_by_path($slug, OBJECT, PostTypes::DECK);
        if (!$existing) {
            $existing = get_page_by_path($slug, OBJECT, PostTypes::LEGACY_DECK);
        }
        if ($existing && isset($existing->ID) && !$opts['overwrite']) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'exists']));
            exit;
        }

        // Version rule: block downgrade unless allow_downgrade.
        $installed = self::installedPacks();
        $k = 'deck:' . $slug;
        if (isset($installed[$k]) && is_array($installed[$k])) {
            $cur = isset($installed[$k]['pack_version']) && is_string($installed[$k]['pack_version']) ? $installed[$k]['pack_version'] : '';
            if ($cur !== '' && version_compare($packVersion, $cur, '<') && empty($opts['allow_downgrade'])) {
                self::cleanupDir($workDir);
                wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'downgrade_blocked']));
                exit;
            }
        }

        $postarr = [
            'post_type' => PostTypes::DECK,
            'post_title' => $title !== '' ? $title : $slug,
            'post_status' => $status,
        ];
        if ($existing && isset($existing->ID)) {
            $postarr['ID'] = (int)$existing->ID;
            $deckId = wp_update_post($postarr, true);
        } else {
            $postarr['post_name'] = $slug;
            $deckId = wp_insert_post($postarr, true);
        }

        if (is_wp_error($deckId)) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'deck_create']));
            exit;
        }
        $deckId = (int)$deckId;

        require_once ABSPATH . 'wp-admin/includes/image.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';

        $imported = 0;
        $skipped = 0;
        $errors = [];

        $backUrl = '';
        if ($backRel !== '') {
            $path = self::safeJoin($workDir, $backRel);
            if ($path !== '' && is_readable($path)) {
                $aid = self::attachFile($path, $deckId);
                if ($aid > 0) {
                    $backUrl = (string)wp_get_attachment_url($aid);
                    $imported++;
                } else {
                    $errors[] = 'back: upload failed';
                }
            } else {
                $skipped++;
                $errors[] = 'back: missing ' . $backRel;
            }
        }

        $outCards = [];
        foreach ($cards as $cardId => $relPath) {
            if (!is_string($cardId) || !is_string($relPath)) continue;
            $cardId = sanitize_text_field($cardId);
            if ($cardId === '' || !isset($registry[$cardId]) || $relPath === '') continue;

            $path = self::safeJoin($workDir, $relPath);
            if ($path === '' || !is_readable($path)) {
                $skipped++;
                $errors[] = $cardId . ': missing ' . $relPath;
                continue;
            }

            $aid = self::attachFile($path, $deckId);
            if ($aid <= 0) {
                $errors[] = $cardId . ': upload failed';
                continue;
            }

            $url = (string)wp_get_attachment_url($aid);
            if ($url === '') {
                $errors[] = $cardId . ': no URL';
                continue;
            }
            $outCards[$cardId] = esc_url_raw($url);
            $imported++;
        }

        if ($backUrl !== '') {
            update_post_meta($deckId, '_dtarot_back', esc_url_raw($backUrl));
        }
        if ($outCards) {
            update_post_meta($deckId, '_dtarot_cards', $outCards);
        }

        update_post_meta($deckId, '_dtarot_system', $system);

        update_post_meta($deckId, '_dtarot_pack_version', $packVersion);

        self::recordInstalledPack('deck', $slug, [
            'type' => 'deck',
            'slug' => $slug,
            'title' => $title,
            'pack_version' => $packVersion,
            'installed_at' => gmdate('c'),
            'source' => !empty($opts['source_url']) ? 'url' : 'upload',
            'source_url' => !empty($opts['source_url']) ? (string)$opts['source_url'] : '',
            'post_id' => $deckId,
        ]);

        $defaultSet = false;
        if (!empty($opts['set_default']) && class_exists(DefaultDecks::class)) {
            $defaultSet = DefaultDecks::set($system, $deckId);
        }

        self::cleanupDir($workDir);

        self::redirectReport('deck_zip_ok', [
            'deck_id' => (string)$deckId,
            'imported' => (string)$imported,
            'default_set' => $defaultSet ? '1' : '0',
        ], [
            'kind' => 'deck_zip',
            'ok' => true,
            'dry_run' => false,
            'deck_id' => $deckId,
            'slug' => $slug,
            'title' => $title,
            'pack_version' => $packVersion,
            'default_set' => $defaultSet,
            'expected_images' => $expectedImages,
            'imported_images' => $imported,
            'skipped_images' => $skipped,
            'errors' => array_slice($errors, 0, 50),
        ]);
    }

    /** @param array{dry_run:bool,overwrite:bool,allow_downgrade?:bool,source_url?:string} $opts */
    private static function importPackZipFromPath(string $zipPath, array $opts): void {
        if (!class_exists('ZipArchive')) {
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'zip_missing']));
            exit;
        }

        $workDir = self::createWorkDir();
        if ($workDir === '') {
            if (!empty($opts['source_url'])) wp_delete_file($zipPath);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'workdir']));
            exit;
        }

        $zip = new \ZipArchive();
        if ($zip->open($zipPath) !== true) {
            self::cleanupDir($workDir);
            if (!empty($opts['source_url'])) wp_delete_file($zipPath);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_zip']));
            exit;
        }

        $extractErrors = self::safeZipExtract($zip, $workDir);
        $zip->close();

        if (!empty($opts['source_url'])) {
            wp_delete_file($zipPath);
        }

        if ($extractErrors) {
            self::cleanupDir($workDir);
            self::redirectReport('import_report', [], [
                'kind' => 'pack_zip',
                'ok' => false,
                'errors' => $extractErrors,
            ]);
        }

        $manifest = self::readManifest($workDir);
        if (!is_array($manifest)) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'bad_manifest']));
            exit;
        }

        $commonErr = self::validateManifestCommon($manifest);
        if ($commonErr !== '') {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => $commonErr]));
            exit;
        }

        // meaning pack zips have no files, but still require files key for format consistency.
        if (!isset($manifest['files']) || !is_array($manifest['files'])) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'missing_files']));
            exit;
        }

        // If there are files (future), validate them.
        $hashErrors = self::validateFileHashes($workDir, $manifest['files']);
        if ($hashErrors) {
            self::cleanupDir($workDir);
            self::redirectReport('import_report', [], [
                'kind' => 'pack_zip',
                'ok' => false,
                'errors' => $hashErrors,
            ]);
        }

        $pack = isset($manifest['meaning_pack']) && is_array($manifest['meaning_pack']) ? $manifest['meaning_pack'] : null;
        if (!$pack) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'missing_pack']));
            exit;
        }

        $slug = isset($pack['slug']) && is_string($pack['slug']) ? sanitize_title($pack['slug']) : '';
        $title = isset($pack['title']) && is_string($pack['title']) ? sanitize_text_field($pack['title']) : '';
        if ($title === '' && $slug !== '') $title = $slug;
        if ($slug === '' && $title !== '') $slug = sanitize_title($title);
        if ($slug === '') {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'pack_slug']));
            exit;
        }

        $status = isset($pack['status']) && is_string($pack['status']) ? sanitize_key($pack['status']) : 'draft';
        if (!in_array($status, ['publish','draft','pending','private'], true)) $status = 'draft';

        $system = isset($pack['system']) && is_string($pack['system']) ? Cards::normalizeSystem($pack['system']) : Cards::SYSTEM_TAROT;

        $meanings = isset($pack['meanings']) && is_array($pack['meanings']) ? $pack['meanings'] : [];

        // Optional portable Lenormand metadata.
        $incomingCardMeta = isset($pack['card_meta']) && is_array($pack['card_meta']) ? $pack['card_meta'] : [];
        $cleanLenormandMeta = ($system === Cards::SYSTEM_LENORMAND) ? self::sanitizeLenormandMetaMap($incomingCardMeta) : [];

        $packVersion = isset($manifest['pack_version']) && is_string($manifest['pack_version']) ? trim($manifest['pack_version']) : '';
        if ($packVersion === '') $packVersion = '0.0.0';

        $registry = Cards::forSystem($system);
        $allowedFields = ['upright','reversed','keywords','short','long','correspondences','symbols'];

        $cleanMeanings = [];
        $count = 0;
        foreach ($meanings as $cardId => $meaning) {
            if (!is_string($cardId) || !is_array($meaning)) continue;
            $cardId = sanitize_text_field($cardId);
            if ($cardId === '' || !isset($registry[$cardId])) continue;

            $clean = [];
            foreach ($allowedFields as $field) {
                if (!isset($meaning[$field]) || is_array($meaning[$field])) {
                    $clean[$field] = '';
                    continue;
                }
                $clean[$field] = wp_kses_post((string)$meaning[$field]);
            }
            $cleanMeanings[$cardId] = $clean;
            $count++;
        }

        if ($opts['dry_run']) {
            self::cleanupDir($workDir);
            self::redirectReport('import_report', [], [
                'kind' => 'pack_zip',
                'ok' => true,
                'dry_run' => true,
                'slug' => $slug,
                'title' => $title,
                'pack_version' => $packVersion,
                'cards' => $count,
            ]);
        }

        // Persist Lenormand metadata for portability.
        if ($cleanLenormandMeta) {
            $existingLenormandMeta = get_option(self::LENORMAND_META_OPTION, []);
            if (!is_array($existingLenormandMeta)) $existingLenormandMeta = [];
            $merged = $opts['overwrite']
                ? array_merge($existingLenormandMeta, $cleanLenormandMeta)
                : array_merge($cleanLenormandMeta, $existingLenormandMeta);
            update_option(self::LENORMAND_META_OPTION, $merged, false);
        }

        $existing = get_page_by_path($slug, OBJECT, PostTypes::MEANING_PACK);
        if (!$existing) {
            $existing = get_page_by_path($slug, OBJECT, PostTypes::LEGACY_MEANING_PACK);
        }
        if ($existing && isset($existing->ID) && !$opts['overwrite']) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'exists']));
            exit;
        }

        // Version rule: block downgrade unless allow_downgrade.
        $installed = self::installedPacks();
        $k = 'meaning_pack:' . $slug;
        if (isset($installed[$k]) && is_array($installed[$k])) {
            $cur = isset($installed[$k]['pack_version']) && is_string($installed[$k]['pack_version']) ? $installed[$k]['pack_version'] : '';
            if ($cur !== '' && version_compare($packVersion, $cur, '<') && empty($opts['allow_downgrade'])) {
                self::cleanupDir($workDir);
                wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'downgrade_blocked']));
                exit;
            }
        }

        $postarr = [
            'post_type' => PostTypes::MEANING_PACK,
            'post_title' => $title !== '' ? $title : $slug,
            'post_status' => $status,
        ];
        if ($existing && isset($existing->ID)) {
            $postarr['ID'] = (int)$existing->ID;
            $packId = wp_update_post($postarr, true);
        } else {
            $postarr['post_name'] = $slug;
            $packId = wp_insert_post($postarr, true);
        }

        if (is_wp_error($packId)) {
            self::cleanupDir($workDir);
            wp_safe_redirect(self::settingsUrl(['dtarot_backup' => 'import_error', 'msg' => 'pack_create']));
            exit;
        }
        $packId = (int)$packId;

        update_post_meta($packId, '_dtarot_system', $system);
        update_post_meta($packId, '_dtarot_meanings', $cleanMeanings);
        update_post_meta($packId, '_dtarot_pack_version', $packVersion);

        self::recordInstalledPack('meaning_pack', $slug, [
            'type' => 'meaning_pack',
            'slug' => $slug,
            'title' => $title,
            'pack_version' => $packVersion,
            'installed_at' => gmdate('c'),
            'source' => !empty($opts['source_url']) ? 'url' : 'upload',
            'source_url' => !empty($opts['source_url']) ? (string)$opts['source_url'] : '',
            'post_id' => $packId,
        ]);

        self::cleanupDir($workDir);

        self::redirectReport('pack_zip_ok', [
            'pack_id' => (string)$packId,
            'cards' => (string)$count,
        ], [
            'kind' => 'pack_zip',
            'ok' => true,
            'dry_run' => false,
            'pack_id' => $packId,
            'slug' => $slug,
            'title' => $title,
            'pack_version' => $packVersion,
            'cards' => $count,
        ]);
    }

    private static function validateManifestCommon(array $manifest): string {
        $pluginVersion = defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '';
        return ManifestValidator::validateCommon($manifest, self::FORMAT_VERSION, $pluginVersion);
    }

    private static function readManifest(string $workDir): ?array {
        $manifestPath = $workDir . '/manifest.json';
        $raw = is_readable($manifestPath) ? file_get_contents($manifestPath) : false;
        if (!is_string($raw) || $raw === '') {
            return null;
        }
        $manifest = json_decode($raw, true);
        return is_array($manifest) ? $manifest : null;
    }

    /** @param array<int,array<string,mixed>> $files */
    private static function validateFileHashes(string $workDir, array $files): array {
        $errors = [];
        foreach ($files as $row) {
            if (!is_array($row)) continue;
            $path = isset($row['path']) && is_string($row['path']) ? (string)$row['path'] : '';
            $sha = isset($row['sha256']) && is_string($row['sha256']) ? strtolower(trim((string)$row['sha256'])) : '';
            if ($path === '' || $sha === '') continue;

            $p = self::safeJoin($workDir, $path);
            if ($p === '' || !is_readable($p)) {
                $errors[] = 'missing: ' . $path;
                continue;
            }

            $actual = hash_file('sha256', $p);
            if (!is_string($actual) || $actual === '') {
                $errors[] = 'hash_failed: ' . $path;
                continue;
            }
            if (strtolower($actual) !== $sha) {
                $errors[] = 'hash_mismatch: ' . $path;
            }
        }
        return $errors;
    }

    private static function createWorkDir(): string {
        $uploadDir = wp_upload_dir();
        $base = is_array($uploadDir) && isset($uploadDir['basedir']) ? (string)$uploadDir['basedir'] : '';
        if ($base === '') return '';

        $dir = rtrim($base, '/\\') . '/dtarot-import-' . wp_generate_password(12, false, false);
        if (!wp_mkdir_p($dir)) return '';
        return $dir;
    }

    private static function safeZipExtract(\ZipArchive $zip, string $workDir): array {
        $errors = [];
        $count = $zip->numFiles;
        for ($i = 0; $i < $count; $i++) {
            $name = (string)$zip->getNameIndex($i);
            $name = str_replace('\\', '/', $name);
            if ($name === '' || $name === '/' || str_ends_with($name, '/')) {
                continue;
            }
            if ($name[0] === '/' || str_contains($name, '..')) {
                $errors[] = 'invalid path: ' . $name;
                continue;
            }

            $dest = rtrim($workDir, '/\\') . '/' . $name;
            $destDir = dirname($dest);
            if (!wp_mkdir_p($destDir)) {
                $errors[] = 'mkdir failed: ' . $name;
                continue;
            }

            $stream = $zip->getStream($name);
            if (!$stream) {
                $errors[] = 'read failed: ' . $name;
                continue;
            }
            $fs = self::filesystem();
            if (!$fs) {
                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing ZipArchive stream handle.
                fclose($stream);
                $errors[] = 'filesystem unavailable: ' . $name;
                continue;
            }
            $contents = stream_get_contents($stream);
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing ZipArchive stream handle.
            fclose($stream);
            if (!is_string($contents)) {
                $errors[] = 'read failed: ' . $name;
                continue;
            }
            if (!$fs->put_contents($dest, $contents, FS_CHMOD_FILE)) {
                $errors[] = 'write failed: ' . $name;
                continue;
            }
        }
        return $errors;
    }

    private static function downloadZipToTemp(string $url, int $maxBytes): string {
        $tmp = wp_tempnam('dtarot-remote.zip');
        if (!is_string($tmp) || $tmp === '') {
            return '';
        }

        $resp = wp_remote_get($url, [
            'timeout' => 60,
            'redirection' => 8,
            'stream' => true,
            'filename' => $tmp,
            'user-agent' => 'DailyTarot/' . (defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '1.0') . '; ' . (function_exists('home_url') ? (string)home_url('/') : ''),
        ]);

        if (is_wp_error($resp)) {
            wp_delete_file($tmp);
            return '';
        }

        $code = (int)wp_remote_retrieve_response_code($resp);
        if ($code < 200 || $code >= 300) {
            wp_delete_file($tmp);
            return '';
        }

        // Size guard (post-download)
        clearstatcache(true, $tmp);
        $size = file_exists($tmp) ? (int)filesize($tmp) : 0;
        if ($size <= 0 || $size > $maxBytes) {
            wp_delete_file($tmp);
            return '';
        }

        return $tmp;
    }

    private static function downloadToTempFile(string $url, int $maxBytes): string {
        $tmp = wp_tempnam('dtarot-asset');
        if (!is_string($tmp) || $tmp === '') {
            return '';
        }

        $resp = wp_remote_get($url, [
            'timeout' => 40,
            'redirection' => 5,
            'stream' => true,
            'filename' => $tmp,
        ]);
        if (is_wp_error($resp)) {
            wp_delete_file($tmp);
            return '';
        }
        $code = (int)wp_remote_retrieve_response_code($resp);
        if ($code < 200 || $code >= 300) {
            wp_delete_file($tmp);
            return '';
        }

        clearstatcache(true, $tmp);
        $size = file_exists($tmp) ? (int)filesize($tmp) : 0;
        if ($size <= 0 || $size > $maxBytes) {
            wp_delete_file($tmp);
            return '';
        }

        return $tmp;
    }

    private static function guessExt(string $url, string $mime, string $fallback): string {
        $parsed = wp_parse_url($url);
        $path = is_array($parsed) && isset($parsed['path']) ? (string)$parsed['path'] : '';
        $ext = $path !== '' ? strtolower(pathinfo($path, PATHINFO_EXTENSION)) : '';
        if ($ext !== '' && preg_match('/^[a-z0-9]{2,5}$/', $ext)) {
            return $ext;
        }
        return match ($mime) {
            'image/png' => 'png',
            'image/webp' => 'webp',
            'image/gif' => 'gif',
            'image/jpeg', 'image/jpg' => 'jpg',
            default => $fallback,
        };
    }

    private static function sendFile(string $path, string $filename, string $mime): void {
        if (!is_readable($path)) {
            wp_die(esc_html__('File not found.','daily-tarot'));
        }
        $fs = self::filesystem();
        if (!$fs) {
            wp_die(esc_html__('File access unavailable.','daily-tarot'));
        }
        $contents = $fs->get_contents($path);
        if (!is_string($contents)) {
            wp_die(esc_html__('File not found.','daily-tarot'));
        }
        nocache_headers();
        header('Content-Type: ' . $mime);
        header('Content-Disposition: attachment; filename=' . $filename);
        header('X-Content-Type-Options: nosniff');
        header('Content-Length: ' . (string)strlen($contents));
        echo $contents;
        wp_delete_file($path);
        exit;
    }

    private static function attachFile(string $path, int $parentPostId): int {
        $filename = basename($path);
        $bits = wp_upload_bits($filename, null, (string)file_get_contents($path));
        if (!is_array($bits) || !empty($bits['error']) || empty($bits['file']) || empty($bits['url'])) {
            return 0;
        }

        $file = (string)$bits['file'];
        $type = wp_check_filetype($filename, null);
        $mime = isset($type['type']) ? (string)$type['type'] : '';

        $attachmentId = wp_insert_attachment([
            'post_mime_type' => $mime,
            'post_title' => preg_replace('/\.[^.]+$/', '', $filename),
            'post_status' => 'inherit',
            'post_parent' => $parentPostId,
        ], $file, $parentPostId);

        if (is_wp_error($attachmentId) || !$attachmentId) {
            return 0;
        }

        $meta = wp_generate_attachment_metadata((int)$attachmentId, $file);
        if (is_array($meta)) {
            wp_update_attachment_metadata((int)$attachmentId, $meta);
        }
        return (int)$attachmentId;
    }

    private static function safeJoin(string $baseDir, string $rel): string {
        $rel = str_replace('\\', '/', $rel);
        $rel = ltrim($rel, "/ \t\n\r\0\x0B");
        if ($rel === '' || str_contains($rel, '..')) return '';

        $full = rtrim($baseDir, '/\\') . '/' . $rel;
        $realBase = realpath($baseDir);
        $realFull = realpath($full);
        if ($realBase === false || $realFull === false) return '';

        $realBase = rtrim(str_replace('\\', '/', $realBase), '/') . '/';
        $realFull = str_replace('\\', '/', $realFull);
        if (strpos($realFull, $realBase) !== 0) return '';

        return $realFull;
    }

    private static function cleanupDir(string $dir): void {
        if (!is_dir($dir)) return;
        $items = scandir($dir);
        if (!is_array($items)) return;
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') continue;
            $path = $dir . DIRECTORY_SEPARATOR . $item;
            if (is_dir($path)) {
                self::cleanupDir($path);
            } else {
                wp_delete_file($path);
            }
        }
        $fs = self::filesystem();
        if ($fs && method_exists($fs, 'rmdir')) {
            $fs->rmdir($dir, true);
        }
    }
}
