<?php
declare(strict_types=1);


namespace DailyTarot\Admin;
if (!defined('ABSPATH')) { exit; }
// phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash



use DailyTarot\Calendar\DayEntry;
use DailyTarot\Calendar\DayEntryService;
use DailyTarot\Calendar\PublishTimes;
use DailyTarot\Ai\Generator;
use DailyTarot\Meaning\DefaultMeaningPack;
use DailyTarot\Meaning\MeaningPackRepository;
use DailyTarot\Registry\Cards;
use DailyTarot\Support\AiCredits;
use DailyTarot\Support\AiProviderSettings;
use DailyTarot\Support\FeatureFlags;
use DailyTarot\Support\PlanLimits;
use DailyTarot\Support\RelatedLinks;
use DailyTarot\Support\SpreadPreview;

final class Ajax {

    public static function init(): void {
        add_action('wp_ajax_dtarot_save_card_image', [__CLASS__, 'save_card_image']);
        add_action('wp_ajax_dtarot_get_card_image', [__CLASS__, 'get_card_image']);
        add_action('wp_ajax_dtarot_optimize_attachment', [__CLASS__, 'optimize_attachment']);
        add_action('wp_ajax_dtarot_get_content_panel', [__CLASS__, 'get_content_panel']);
        add_action('wp_ajax_dtarot_get_settings_panel', [__CLASS__, 'get_settings_panel']);
        add_action('wp_ajax_dtarot_get_day', [__CLASS__, 'get_day']);
        add_action('wp_ajax_dtarot_save_day', [__CLASS__, 'save_day']);
        add_action('wp_ajax_dtarot_reuse_old_text', [__CLASS__, 'reuse_old_text']);
        add_action('wp_ajax_dtarot_save_ui_setting', [__CLASS__, 'save_ui_setting']);
        add_action('wp_ajax_dtarot_ai_generate', [__CLASS__, 'ai_generate']);
        add_action('wp_ajax_dtarot_ai_save', [__CLASS__, 'ai_save']);
        add_action('wp_ajax_dtarot_ai_add_credits', [__CLASS__, 'ai_add_credits']);
        add_action('wp_ajax_dtarot_ai_provider_test', [__CLASS__, 'ai_provider_test']);
        add_action('wp_ajax_dtarot_spread_preview', [__CLASS__, 'spread_preview']);
        add_action('wp_ajax_dtarot_onboard_dismiss', [__CLASS__, 'onboard_dismiss']);
        add_action('wp_ajax_dtarot_onboard_done', [__CLASS__, 'onboard_done']);

        add_action('wp_ajax_dtarot_uninstall_feedback', [__CLASS__, 'uninstall_feedback']);

        add_action('wp_ajax_dtarot_related_link_get', [__CLASS__, 'related_link_get']);
        add_action('wp_ajax_dtarot_related_link_suggest', [__CLASS__, 'related_link_suggest']);
        add_action('wp_ajax_dtarot_related_link_suggest_from_text', [__CLASS__, 'related_link_suggest_from_text']);
        add_action('wp_ajax_dtarot_related_link_search', [__CLASS__, 'related_link_search']);
        add_action('wp_ajax_dtarot_related_link_set', [__CLASS__, 'related_link_set']);
        add_action('wp_ajax_dtarot_related_link_clear', [__CLASS__, 'related_link_clear']);
    }

    private static function check(): void {
        if (!current_user_can('manage_options')) wp_send_json_error(['message'=>'forbidden'], 403);
        $nonce = isset($_POST['nonce']) ? sanitize_text_field((string)wp_unslash($_POST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'dtarot_admin')) wp_send_json_error(['message'=>'bad nonce'], 403);
    }

    public static function save_card_image(): void {
        self::check();

        $deck_id = isset($_POST['deck_id']) ? absint(wp_unslash($_POST['deck_id'])) : 0;
        $card_id = isset($_POST['card_id']) ? sanitize_text_field((string)wp_unslash($_POST['card_id'])) : '';
        $url     = isset($_POST['url']) ? esc_url_raw((string)wp_unslash($_POST['url'])) : '';
        $attachment_id = isset($_POST['attachment_id']) ? absint(wp_unslash($_POST['attachment_id'])) : 0;

        if (!$deck_id || $card_id === '') wp_send_json_error(['message'=>'missing data'], 400);

        // Special case: deck back image.
        if ($card_id === '__back') {
            $converted = false;
            if ($attachment_id > 0) {
                $r = self::maybe_convert_attachment_to_webp($attachment_id);
                if (!empty($r['url']) && is_string($r['url'])) {
                    $url = (string)$r['url'];
                    $converted = !empty($r['converted']);
                }
            }

            update_post_meta($deck_id, '_dtarot_back', $url);
            wp_send_json_success(['message' => 'saved', 'url' => $url, 'converted' => $converted ? 1 : 0]);
        }

        $converted = false;
        if ($attachment_id > 0) {
            $r = self::maybe_convert_attachment_to_webp($attachment_id);
            if (!empty($r['url']) && is_string($r['url'])) {
                $url = (string)$r['url'];
                $converted = !empty($r['converted']);
            }
        }

        $images = get_post_meta($deck_id, '_dtarot_cards', true);
        if (!is_array($images)) $images = [];
        $images[$card_id] = $url;
        update_post_meta($deck_id, '_dtarot_cards', $images);

        wp_send_json_success(['message'=>'saved', 'url' => $url, 'converted' => $converted ? 1 : 0]);
    }

    public static function spread_preview(): void {
        self::check();

        $preset = isset($_POST['preset']) ? sanitize_key((string)wp_unslash($_POST['preset'])) : '';
        $pack = isset($_POST['pack']) ? sanitize_key((string)wp_unslash($_POST['pack'])) : '';
        $deckId = isset($_POST['deck_id']) ? absint(wp_unslash($_POST['deck_id'])) : 0;

        $html = class_exists(SpreadPreview::class) ? SpreadPreview::render($preset, $pack, $deckId) : '';
        wp_send_json_success(['html' => $html]);
    }

    public static function onboard_dismiss(): void {
        self::check();

        $userId = function_exists('get_current_user_id') ? (int)get_current_user_id() : 0;
        if ($userId <= 0) wp_send_json_error(['message' => 'no user'], 400);
        update_user_meta($userId, 'dtarot_onboard_dismissed', 1);
        wp_send_json_success(['ok' => 1]);
    }

    public static function onboard_done(): void {
        self::check();

        $userId = function_exists('get_current_user_id') ? (int)get_current_user_id() : 0;
        if ($userId <= 0) wp_send_json_error(['message' => 'no user'], 400);
        update_user_meta($userId, 'dtarot_onboard_tour_done', 1);
        wp_send_json_success(['ok' => 1]);
    }

    public static function uninstall_feedback(): void {
        self::check();

        $context = isset($_POST['context']) ? sanitize_key((string)wp_unslash($_POST['context'])) : 'deactivate';
        if (!in_array($context, ['deactivate','uninstall'], true)) {
            $context = 'deactivate';
        }

        $reason = isset($_POST['reason']) ? sanitize_key((string)wp_unslash($_POST['reason'])) : '';
        $details = isset($_POST['details']) ? sanitize_textarea_field((string)wp_unslash($_POST['details'])) : '';
        if (mb_strlen($details) > 800) {
            $details = mb_substr($details, 0, 800);
        }

        $userId = function_exists('get_current_user_id') ? (int)get_current_user_id() : 0;

        $entry = [
            'ts' => time(),
            'user_id' => $userId,
            'context' => $context,
            'reason' => $reason,
            'details' => $details,
            'ver' => defined('DTAROT_VERSION') ? (string)DTAROT_VERSION : '',
        ];

        $opt = get_option('dtarot_uninstall_feedback_v1', []);
        if (!is_array($opt)) $opt = [];
        $opt[] = $entry;
        if (count($opt) > 50) {
            $opt = array_slice($opt, -50);
        }
        update_option('dtarot_uninstall_feedback_v1', $opt, false);

        wp_send_json_success(['ok' => 1]);
    }

    public static function related_link_get(): void {
        self::check();

        $cardId = isset($_POST['card_id']) ? sanitize_text_field((string)wp_unslash($_POST['card_id'])) : '';
        $cardId = sanitize_key($cardId);
        if ($cardId === '') {
            wp_send_json_error(['message' => 'missing card_id'], 400);
        }

        $settings = class_exists(RelatedLinks::class) ? RelatedLinks::get() : ['enabled' => '0', 'post_type' => 'page', 'base_path' => '', 'map' => []];
        $selected = class_exists(RelatedLinks::class) ? RelatedLinks::selectedLink($cardId) : ['type' => 'none', 'id' => 0, 'title' => '', 'url' => ''];
        $resolved = class_exists(RelatedLinks::class) ? RelatedLinks::resolvedUrl($cardId) : '';

        wp_send_json_success([
            'settings' => [
                'enabled' => isset($settings['enabled']) ? (string)$settings['enabled'] : '0',
                'post_type' => isset($settings['post_type']) ? (string)$settings['post_type'] : 'page',
                'base_path' => isset($settings['base_path']) ? (string)$settings['base_path'] : '',
            ],
            'selected' => $selected,
            'resolvedUrl' => $resolved,
        ]);
    }

    public static function related_link_suggest(): void {
        self::check();
        $cardId = isset($_POST['card_id']) ? sanitize_text_field((string)wp_unslash($_POST['card_id'])) : '';
        $cardId = sanitize_key($cardId);
        if ($cardId === '') {
            wp_send_json_error(['message' => 'missing card_id'], 400);
        }

        $items = class_exists(RelatedLinks::class) ? RelatedLinks::suggest($cardId) : [];
        wp_send_json_success(['items' => $items]);
    }

    public static function related_link_suggest_from_text(): void {
        self::check();

        $content = isset($_POST['content']) ? sanitize_textarea_field((string)wp_unslash($_POST['content'])) : '';
        $dailyText = isset($_POST['daily_text']) ? sanitize_textarea_field((string)wp_unslash($_POST['daily_text'])) : '';
        $excludePostId = isset($_POST['exclude_post_id']) ? absint(wp_unslash($_POST['exclude_post_id'])) : 0;
        $limit = isset($_POST['limit']) ? absint(wp_unslash($_POST['limit'])) : 10;
        if ($limit < 1) $limit = 1;
        if ($limit > 50) $limit = 50;
        $content = wp_strip_all_tags($content);
        $dailyText = wp_strip_all_tags($dailyText);

        $items = class_exists(RelatedLinks::class) ? RelatedLinks::suggestFromText($content, $dailyText, $excludePostId, $limit) : [];

        $fallback = true;
        foreach ((array)$items as $it) {
            if (isset($it['anchor']) && is_string($it['anchor']) && trim($it['anchor']) !== '') {
                $fallback = false;
                break;
            }
        }

        wp_send_json_success([
            'items' => $items,
            'fallback' => $fallback ? 1 : 0,
        ]);
    }

    public static function related_link_search(): void {
        self::check();
        $q = isset($_POST['q']) ? sanitize_text_field((string)wp_unslash($_POST['q'])) : '';
        $q = trim($q);
        if ($q === '') {
            wp_send_json_success(['items' => []]);
        }

        $settings = class_exists(RelatedLinks::class) ? RelatedLinks::get() : ['post_type' => 'page'];
        $postType = isset($settings['post_type']) && is_string($settings['post_type']) ? (string)$settings['post_type'] : 'page';
        $types = $postType === 'any' ? ['page','post'] : [$postType];
        $types = array_values(array_filter($types, function($t){ return in_array($t, ['page','post'], true); }));
        if (!$types) $types = ['page'];

        $posts = get_posts([
            'post_type' => $types,
            'post_status' => 'publish',
            'numberposts' => 10,
            'no_found_rows' => true,
            's' => $q,
        ]);

        $items = [];
        foreach ((array)$posts as $p) {
            if (!is_object($p) || !isset($p->ID)) continue;
            $id = (int)$p->ID;
            if ($id <= 0) continue;
            $url = get_permalink($id);
            $url = is_string($url) ? (string)$url : '';
            if ($url === '') continue;
            $items[] = [
                'id' => $id,
                'title' => (string)get_the_title($id),
                'type' => (string)get_post_type($id),
                'url' => $url,
            ];
        }

        wp_send_json_success(['items' => $items]);
    }

    public static function related_link_set(): void {
        self::check();

        $cardId = isset($_POST['card_id']) ? sanitize_text_field((string)wp_unslash($_POST['card_id'])) : '';
        $cardId = sanitize_key($cardId);
        if ($cardId === '') {
            wp_send_json_error(['message' => 'missing card_id'], 400);
        }

        $postId = isset($_POST['post_id']) ? absint(wp_unslash($_POST['post_id'])) : 0;
        $url = isset($_POST['url']) ? trim((string)wp_unslash($_POST['url'])) : '';
        $url = $url !== '' ? (string)esc_url_raw($url) : '';

        if ($postId > 0 && class_exists(RelatedLinks::class)) {
            RelatedLinks::setMappingPost($cardId, $postId);
        } elseif ($url !== '' && class_exists(RelatedLinks::class)) {
            RelatedLinks::setMappingUrl($cardId, $url);
        } else {
            wp_send_json_error(['message' => 'missing post_id or url'], 400);
        }

        $selected = class_exists(RelatedLinks::class) ? RelatedLinks::selectedLink($cardId) : ['type' => 'none', 'id' => 0, 'title' => '', 'url' => ''];
        $resolved = class_exists(RelatedLinks::class) ? RelatedLinks::resolvedUrl($cardId) : '';
        wp_send_json_success(['selected' => $selected, 'resolvedUrl' => $resolved]);
    }

    public static function related_link_clear(): void {
        self::check();
        $cardId = isset($_POST['card_id']) ? sanitize_text_field((string)wp_unslash($_POST['card_id'])) : '';
        $cardId = sanitize_key($cardId);
        if ($cardId === '') {
            wp_send_json_error(['message' => 'missing card_id'], 400);
        }

        if (class_exists(RelatedLinks::class)) {
            RelatedLinks::clearMapping($cardId);
        }
        $resolved = class_exists(RelatedLinks::class) ? RelatedLinks::resolvedUrl($cardId) : '';
        wp_send_json_success(['ok' => 1, 'resolvedUrl' => $resolved]);
    }

    public static function maybe_convert_attachment_to_webp(int $attachmentId): array {
        if ($attachmentId <= 0) return ['url' => '', 'converted' => false];

        $mime = (string)get_post_mime_type($attachmentId);
        if (!in_array($mime, ['image/jpeg','image/png','image/webp'], true)) {
            return ['url' => '', 'converted' => false];
        }

        $file = (string)get_attached_file($attachmentId);
        if ($file === '' || !file_exists($file)) {
            return ['url' => '', 'converted' => false];
        }

        $ext = strtolower((string)pathinfo($file, PATHINFO_EXTENSION));
        if ($ext === 'webp' || $mime === 'image/webp') {
            $u = wp_get_attachment_url($attachmentId);
            return ['url' => is_string($u) ? (string)$u : '', 'converted' => false];
        }

        if (!function_exists('wp_get_image_editor')) {
            $inc = ABSPATH . 'wp-admin/includes/image.php';
            if (file_exists($inc)) {
                require_once $inc;
            }
        }

        if (!function_exists('wp_get_image_editor')) {
            return ['url' => '', 'converted' => false];
        }

        $editor = wp_get_image_editor($file);
        if (is_wp_error($editor) || !$editor) {
            return ['url' => '', 'converted' => false];
        }

        if (method_exists($editor, 'supports_mime_type') && !$editor->supports_mime_type('image/webp')) {
            return ['url' => '', 'converted' => false];
        }

        if (method_exists($editor, 'set_quality')) {
            // Basic optimization: trade size for quality.
            $editor->set_quality(82);
        }

        $dest = preg_replace('/\.[^.]+$/', '.webp', $file);
        if (!is_string($dest) || $dest === '') {
            return ['url' => '', 'converted' => false];
        }

        $saved = $editor->save($dest, 'image/webp');
        if (is_wp_error($saved) || !is_array($saved) || empty($saved['path']) || !is_string($saved['path'])) {
            return ['url' => '', 'converted' => false];
        }

        $upload = wp_upload_dir();
        $basedir = isset($upload['basedir']) ? (string)$upload['basedir'] : '';
        $baseurl = isset($upload['baseurl']) ? (string)$upload['baseurl'] : '';

        $webpPath = (string)$saved['path'];
        if ($basedir !== '' && $baseurl !== '' && strpos($webpPath, $basedir) === 0) {
            $rel = substr($webpPath, strlen($basedir));
            $rel = str_replace('\\', '/', (string)$rel);
            if ($rel !== '' && $rel[0] !== '/') $rel = '/' . $rel;
            return ['url' => $baseurl . $rel, 'converted' => true];
        }

        // Fallback: we created the file, but can't confidently map it to a URL.
        return ['url' => '', 'converted' => false];
    }

    public static function optimize_attachment(): void {
        self::check();

        $attachmentId = isset($_POST['attachment_id']) ? absint(wp_unslash($_POST['attachment_id'])) : 0;
        if ($attachmentId <= 0) {
            wp_send_json_error(['message' => 'missing attachment_id'], 400);
        }

        $fallbackUrl = wp_get_attachment_url($attachmentId);
        $fallbackUrl = is_string($fallbackUrl) ? (string)$fallbackUrl : '';

        $converted = false;
        $url = $fallbackUrl;

        $r = self::maybe_convert_attachment_to_webp($attachmentId);
        if (!empty($r['url']) && is_string($r['url'])) {
            $url = (string)$r['url'];
            $converted = !empty($r['converted']);
        }

        wp_send_json_success([
            'attachment_id' => $attachmentId,
            'url' => esc_url($url),
            'converted' => $converted ? 1 : 0,
        ]);
    }

    public static function get_content_panel(): void {
        self::check();

        $href = isset($_POST['href']) ? esc_url_raw((string)wp_unslash($_POST['href'])) : '';
        $href = trim($href);
        if ($href === '') {
            wp_send_json_error(['message' => 'missing href'], 400);
        }

        // Only allow loading the Daily Tarot Content admin page.
        $parts = wp_parse_url($href);
        if (!is_array($parts)) {
            wp_send_json_error(['message' => 'bad href'], 400);
        }

        $queryStr = isset($parts['query']) && is_string($parts['query']) ? (string)$parts['query'] : '';
        $query = [];
        if ($queryStr !== '') {
            parse_str($queryStr, $query);
        }

        $page = isset($query['page']) ? sanitize_key((string)$query['page']) : '';
        if ($page !== 'daily-tarot-content') {
            wp_send_json_error(['message' => 'forbidden'], 403);
        }

        // Temporarily apply the requested query args so existing render logic works.
        $oldGet = $_GET;
        try {
            foreach ($query as $k => $v) {
                // Keep things simple and safe: only scalar values.
                if (is_array($v) || is_object($v)) continue;
                $_GET[(string)$k] = (string)$v;
            }

            $tab = isset($_GET['tab']) ? sanitize_key((string)wp_unslash($_GET['tab'])) : 'status';

            ob_start();
            if (class_exists(\DailyTarot\Admin\Pages\Content::class)) {
                \DailyTarot\Admin\Pages\Content::render_panel($tab);
            }
            $html = (string)ob_get_clean();

            wp_send_json_success([
                'tab' => $tab,
                'html' => $html,
            ]);
        } finally {
            $_GET = $oldGet;
        }
    }

    public static function get_settings_panel(): void {
        self::check();

        $href = isset($_POST['href']) ? esc_url_raw((string)wp_unslash($_POST['href'])) : '';
        $href = trim($href);
        if ($href === '') {
            wp_send_json_error(['message' => 'missing href'], 400);
        }

        // Only allow loading the Daily Tarot Settings admin page.
        $parts = wp_parse_url($href);
        if (!is_array($parts)) {
            wp_send_json_error(['message' => 'bad href'], 400);
        }

        $queryStr = isset($parts['query']) && is_string($parts['query']) ? (string)$parts['query'] : '';
        $query = [];
        if ($queryStr !== '') {
            parse_str($queryStr, $query);
        }

        $page = isset($query['page']) ? sanitize_key((string)$query['page']) : '';
        if ($page !== 'daily-tarot-settings') {
            wp_send_json_error(['message' => 'forbidden'], 403);
        }

        $oldGet = $_GET;
        try {
            foreach ($query as $k => $v) {
                if (is_array($v) || is_object($v)) continue;
                $_GET[(string)$k] = (string)$v;
            }

            // Let Settings::render_panel() resolve invalid tabs the same way render() does.
            $tab = isset($_GET['tab']) ? sanitize_key((string)wp_unslash($_GET['tab'])) : '';

            ob_start();
            if (class_exists(\DailyTarot\Admin\Pages\Settings::class)) {
                \DailyTarot\Admin\Pages\Settings::render_panel($tab);
            }
            $html = (string)ob_get_clean();

            wp_send_json_success([
                'tab' => $tab,
                'html' => $html,
            ]);
        } finally {
            $_GET = $oldGet;
        }
    }

    public static function get_card_image(): void {
        self::check();

        $deck_id = isset($_POST['deck_id']) ? (int)wp_unslash($_POST['deck_id']): 0;
        $card_id = isset($_POST['card_id']) ? sanitize_text_field((string)wp_unslash($_POST['card_id'])) : '';
        if (!$deck_id || $card_id === '') {
            wp_send_json_success(['url' => '']);
        }

        $images = get_post_meta($deck_id, '_dtarot_cards', true);
        if (!is_array($images)) $images = [];

        $url = '';
        if (!empty($images[$card_id]) && is_string($images[$card_id])) {
            $url = esc_url((string)$images[$card_id]);
        }

        wp_send_json_success(['url' => $url]);
    }

    public static function get_day(): void {
        self::check();
        $date = isset($_POST['date']) ? sanitize_text_field((string)wp_unslash($_POST['date'])) : '';
        if ($date === '') wp_send_json_error(['message'=>'missing date'], 400);

        $entry = DayEntryService::get($date)->toArray();

        // Pro-only: include per-day publish time override (empty string means "use default").
        $isPro = class_exists(FeatureFlags::class) && FeatureFlags::enabled('pro');
        if ($isPro && class_exists(PublishTimes::class)) {
            $entry['publish_time'] = PublishTimes::getOverride($date);
        }

        if (($entry['pack'] ?? '') === '' && class_exists(DefaultMeaningPack::class)) {
            $defaultPack = DefaultMeaningPack::getId();
            if ($defaultPack > 0) {
                $entry['pack'] = (string)$defaultPack;
            }
        }

        wp_send_json_success($entry);
    }

    public static function save_day(): void {
        self::check();

        $date = isset($_POST['date']) ? sanitize_text_field((string)wp_unslash($_POST['date'])) : '';
        if ($date === '') wp_send_json_error(['message'=>'missing date'], 400);

        // Free-tier calendar limit: only allow editing up to N days ahead.
        if (class_exists(PlanLimits::class) && !PlanLimits::canEditCalendarDate($date)) {
            $max = PlanLimits::calendarMaxEditableDate();
            wp_send_json_error([
                'code' => 'calendar_limit',
                /* translators: %s is the max editable date for free plans (YYYY-MM-DD). */
                'message' => sprintf((string)__('Free plan can edit up to %s. Activate Pro to edit further.','daily-tarot'), $max),
                'max_date' => $max,
            ], 403);
        }

        $pack = isset($_POST['pack']) ? sanitize_text_field((string)wp_unslash($_POST['pack'])) : '';
        if ($pack === '' && class_exists(DefaultMeaningPack::class)) {
            $defaultPack = DefaultMeaningPack::getId();
            if ($defaultPack > 0) {
                $pack = (string)$defaultPack;
            }
        }

        $existing = DayEntryService::get($date)->toArray();

        $incoming = [
            'deck' => isset($_POST['deck']) ? sanitize_text_field((string)wp_unslash($_POST['deck'])) : '',
            'card' => isset($_POST['card']) ? sanitize_text_field((string)wp_unslash($_POST['card'])) : '',
            'pack' => $pack,
            'status' => isset($_POST['status']) ? sanitize_text_field((string)wp_unslash($_POST['status'])) : DayEntry::STATUS_DRAFT,
            'content' => isset($_POST['content']) ? wp_kses_post((string)wp_unslash($_POST['content'])) : '',
            'daily_text' => isset($_POST['daily_text']) ? wp_kses_post((string)wp_unslash($_POST['daily_text'])) : '',
        ];

        // Pro-only: save per-day publish time override separately.
        $isPro = class_exists(FeatureFlags::class) && FeatureFlags::enabled('pro');
        if ($isPro && class_exists(PublishTimes::class) && array_key_exists('publish_time', $_POST)) {
            $publishTime = PublishTimes::sanitizeTime(sanitize_text_field((string)wp_unslash($_POST['publish_time'])));
            PublishTimes::setOverride($date, $publishTime);

            // If automation is enabled, updating today's/tomorrow's time can affect the next run.
            try {
                $tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
                $now = new \DateTimeImmutable('now', $tz);
                $today = $now->format('Y-m-d');
                $tomorrow = $now->modify('+1 day')->format('Y-m-d');
                if ($date === $today || $date === $tomorrow) {
                    PublishTimes::rescheduleAutomationIfEnabled();
                }
            } catch (\Throwable $e) {
                // Ignore reschedule errors in AJAX.
            }
        }

        // Only update image override fields if explicitly provided.
        if (array_key_exists('image_override_attachment_id', $_POST)) {
            $incoming['image_override_attachment_id'] = absint(wp_unslash($_POST['image_override_attachment_id']));
        }
        if (array_key_exists('image_override_url', $_POST)) {
            $incoming['image_override_url'] = esc_url_raw((string)wp_unslash($_POST['image_override_url']));
        }

        $ok = DayEntryService::setFromArray($date, array_merge($existing, $incoming));

        if (!$ok) wp_send_json_error(['message' => 'bad date'], 400);
        wp_send_json_success(['message'=>'saved']);
    }

    public static function reuse_old_text(): void {
        self::check();

        $date = isset($_POST['date']) ? sanitize_text_field((string)wp_unslash($_POST['date'])) : '';
        if ($date === '') wp_send_json_error(['message' => 'missing date'], 400);
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) wp_send_json_error(['message' => 'bad date'], 400);

        $found = DayEntryService::findPreviousText($date);
        if (is_array($found)) {
            wp_send_json_success($found);
        }

        wp_send_json_error(['message' => 'no text found'], 404);
    }

    public static function save_ui_setting(): void {
        self::check();

        $setting = isset($_POST['setting']) ? sanitize_key((string)wp_unslash($_POST['setting'])) : '';
        $rawValue = isset($_POST['value']) ? (string)wp_unslash($_POST['value']) : '';
        $value = ($rawValue === '1');

        if ($setting === '') {
            wp_send_json_error(['message' => 'missing setting'], 400);
        }

        if (!class_exists(UiSettings::class)) {
            wp_send_json_error(['message' => 'settings unavailable'], 500);
        }

        // Whitelist via UiSettings registry.
        if (!UiSettings::saveBoolSetting($setting, $value)) {
            wp_send_json_error(['message' => 'unknown setting'], 400);
        }

        wp_send_json_success([
            'setting' => $setting,
            'value' => $value ? '1' : '0',
        ]);
    }

    public static function ai_generate(): void {
        self::check();

        if (!FeatureFlags::enabled('ai')) {
            wp_send_json_error(['message' => 'ai_disabled'], 403);
        }

        $date = isset($_POST['date']) ? sanitize_text_field((string)wp_unslash($_POST['date'])) : '';
        if ($date === '' || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            wp_send_json_error(['message' => 'bad_date'], 400);
        }

        $kind = isset($_POST['kind']) ? sanitize_key((string)wp_unslash($_POST['kind'])) : 'both';
        if (!in_array($kind, ['content','daily_text','both'], true)) {
            $kind = 'both';
        }

        $introMin = 450;
        $introMax = 562;
        $introTarget = isset($_POST['intro_len']) ? (int)wp_unslash($_POST['intro_len']): 0;
        if ($introTarget <= 0) {
            $introTarget = 500;
        }
        if ($introTarget < $introMin) $introTarget = $introMin;
        if ($introTarget > $introMax) $introTarget = $introMax;

        $dailyMin = $introMin * 2;
        $dailyMax = $introMax * 2;
        $dailyTarget = isset($_POST['daily_len']) ? (int)wp_unslash($_POST['daily_len']): 0;
        if ($dailyTarget <= 0) {
            $dailyTarget = $introTarget * 2;
        }
        if ($dailyTarget < $dailyMin) $dailyTarget = $dailyMin;
        if ($dailyTarget > $dailyMax) $dailyTarget = $dailyMax;

        $cost = AiCredits::costForKind($kind);
        if (!AiCredits::canConsume($cost)) {
            wp_send_json_error([
                'message' => 'insufficient_credits',
                'balance' => AiCredits::getBalance(),
                'required' => $cost,
            ], 402);
        }

        $save = isset($_POST['save']) && ((string)wp_unslash($_POST['save']) === '1');

        $entry = DayEntryService::get($date);

        // If pack missing, try default pack.
        if ((int)$entry->packId <= 0 && class_exists(DefaultMeaningPack::class)) {
            $defaultPack = DefaultMeaningPack::getId();
            if ($defaultPack > 0) {
                $entry->packId = (int)$defaultPack;
            }
        }

        if ((int)$entry->deckId <= 0 || (string)$entry->cardId === '') {
            wp_send_json_error(['message' => 'missing_deck_card'], 400);
        }

        $gen = Generator::generateForEntry($entry, $date, [
            'intro_min_chars' => $introMin,
            'intro_max_chars' => $introMax,
            'intro_target_chars' => $introTarget,
            'daily_min_chars' => $dailyMin,
            'daily_max_chars' => $dailyMax,
            'daily_target_chars' => $dailyTarget,
        ]);
        $content = isset($gen['content']) ? (string)$gen['content'] : '';
        $dailyText = isset($gen['daily_text']) ? (string)$gen['daily_text'] : '';

        if ($kind === 'content') {
            $dailyText = '';
        } elseif ($kind === 'daily_text') {
            $content = '';
        }

        if ($content === '' && $dailyText === '') {
            wp_send_json_error(['message' => 'no_result'], 500);
        }

        // Charge credits after generation succeeds.
        $charge = AiCredits::consume($cost, [
            'date' => $date,
            'kind' => $kind,
            'save' => $save ? 1 : 0,
            'deck_id' => (int)$entry->deckId,
            'card_id' => (string)$entry->cardId,
            'pack_id' => (int)$entry->packId,
        ]);
        if (!$charge['ok']) {
            wp_send_json_error([
                'message' => 'insufficient_credits',
                'balance' => (int)$charge['balance'],
                'required' => (int)$charge['cost'],
            ], 402);
        }

        $saved = false;
        if ($save) {
            $arr = (array)$entry->toArray();
            if ($content !== '') {
                $arr['content'] = $content;
            }
            if ($dailyText !== '') {
                $arr['daily_text'] = $dailyText;
            }

            $ok = DayEntryService::setFromArray($date, [
                'deck' => isset($arr['deck']) ? sanitize_text_field((string)$arr['deck']) : '',
                'card' => isset($arr['card']) ? sanitize_text_field((string)$arr['card']) : '',
                'pack' => isset($arr['pack']) ? sanitize_text_field((string)$arr['pack']) : '',
                'status' => isset($arr['status']) ? sanitize_text_field((string)$arr['status']) : DayEntry::STATUS_DRAFT,
                'content' => isset($arr['content']) ? wp_kses_post((string)$arr['content']) : '',
                'daily_text' => isset($arr['daily_text']) ? wp_kses_post((string)$arr['daily_text']) : '',
            ]);
            if (!$ok) {
                // Refund if saving fails (generation already succeeded).
                if ($cost > 0) {
                    AiCredits::add($cost, 'refund_save_failed', [
                        'date' => $date,
                        'kind' => $kind,
                    ]);
                }
                wp_send_json_error(['message' => 'save_failed'], 500);
            }
            $saved = true;
        }

        wp_send_json_success([
            'date' => $date,
            'content' => $content,
            'daily_text' => $dailyText,
            'saved' => $saved,
            'credits_left' => (int)$charge['balance'],
        ]);
    }

    public static function ai_save(): void {
        self::check();

        if (!FeatureFlags::enabled('ai')) {
            wp_send_json_error(['message' => 'ai_disabled'], 403);
        }

        $date = isset($_POST['date']) ? sanitize_text_field((string)wp_unslash($_POST['date'])) : '';
        if ($date === '' || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            wp_send_json_error(['message' => 'bad_date'], 400);
        }

        if (class_exists(PlanLimits::class) && !PlanLimits::canEditCalendarDate($date)) {
            $max = PlanLimits::calendarMaxEditableDate();
            wp_send_json_error([
                'code' => 'calendar_limit',
                /* translators: %s is the max editable date for free plans (YYYY-MM-DD). */
                'message' => sprintf((string)__('Free plan can edit up to %s. Activate Pro to edit further.','daily-tarot'), $max),
                'max_date' => $max,
            ], 403);
        }

        $content = isset($_POST['content']) ? wp_kses_post((string)wp_unslash($_POST['content'])) : '';
        $dailyText = isset($_POST['daily_text']) ? wp_kses_post((string)wp_unslash($_POST['daily_text'])) : '';

        $existing = DayEntryService::get($date)->toArray();
        $ok = DayEntryService::setFromArray($date, array_merge($existing, [
            'content' => $content,
            'daily_text' => $dailyText,
        ]));

        if (!$ok) wp_send_json_error(['message' => 'bad_date'], 400);
        wp_send_json_success(['message' => 'saved']);
    }

    public static function ai_add_credits(): void {
        self::check();

        $amount = isset($_POST['amount']) ? (int)wp_unslash($_POST['amount']): 0;
        if ($amount <= 0) {
            wp_send_json_error(['message' => 'bad_amount'], 400);
        }

        $reason = isset($_POST['reason']) ? sanitize_key((string)wp_unslash($_POST['reason'])) : 'manual';

        $state = AiCredits::add($amount, $reason, [
            'user_id' => (int) get_current_user_id(),
        ]);

        wp_send_json_success([
            'balance' => (int) ($state['balance'] ?? 0),
            'month' => (string) ($state['month'] ?? ''),
        ]);
    }

    public static function ai_provider_test(): void {
        self::check();

        if (!FeatureFlags::enabled('ai_provider')) {
            wp_send_json_error(['message' => 'ai_provider_disabled'], 403);
        }

        if (!class_exists(AiProviderSettings::class)) {
            wp_send_json_error(['message' => 'ai_provider_settings_unavailable'], 500);
        }

        $url = AiProviderSettings::getUrl();
        if ($url === '') {
            wp_send_json_error(['message' => 'missing_provider_url'], 400);
        }

        $date = isset($_POST['date']) ? sanitize_text_field((string)wp_unslash($_POST['date'])) : '';
        if ($date === '') {
            $date = function_exists('wp_date') ? (string)wp_date('Y-m-d') : (string)current_time('Y-m-d');
        }
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            wp_send_json_error(['message' => 'bad_date'], 400);
        }

        $entry = DayEntryService::get($date);

        // If pack missing, try default pack.
        if ((int)$entry->packId <= 0 && class_exists(DefaultMeaningPack::class)) {
            $defaultPack = DefaultMeaningPack::getId();
            if ($defaultPack > 0) {
                $entry->packId = (int)$defaultPack;
            }
        }

        $deckId = (int)$entry->deckId;
        $packId = (int)$entry->packId;
        $cardId = (string)$entry->cardId;

        $payload = [
            'date' => $date,
            'deck_id' => $deckId,
            'pack_id' => $packId,
            'card_id' => $cardId,
            'card_name' => Cards::name($cardId),
            'meaning' => $packId > 0 ? MeaningPackRepository::getMeaning($packId, $cardId) : MeaningPackRepository::emptyMeaning(),
            'site_name' => (string)get_bloginfo('name'),
            'site_url' => (string)home_url('/'),
            'locale' => function_exists('get_locale') ? (string)get_locale() : '',
            'language' => (class_exists(AiProviderSettings::class) && method_exists(AiProviderSettings::class, 'getOutputLanguage') && trim((string)AiProviderSettings::getOutputLanguage()) !== '')
                ? (string)AiProviderSettings::getOutputLanguage()
                : (function_exists('get_locale') ? str_replace('_', '-', (string)get_locale()) : ''),
            'timezone' => function_exists('wp_timezone_string') ? (string)wp_timezone_string() : (string)get_option('timezone_string', ''),
            'kind' => 'both',
            'source' => 'admin_provider_test',
        ];

        $headers = [
            'Content-Type' => 'application/json; charset=utf-8',
            'Accept' => 'application/json',
        ];

        $secret = '';
        if (class_exists(AiProviderSettings::class)) {
            $secret = (string)AiProviderSettings::getProviderSecret();
        } elseif (defined('DTAROT_AI_PROVIDER_SECRET')) {
            $secret = trim((string)DTAROT_AI_PROVIDER_SECRET);
        }

        $secret = trim((string)$secret);
        if ($secret !== '') {
            $headers['X-DTAROT-Provider-Secret'] = $secret;
        }

        $resp = wp_remote_post($url, [
            'timeout' => 25,
            'headers' => $headers,
            'body' => wp_json_encode($payload),
        ]);

        if (is_wp_error($resp)) {
            wp_send_json_error([
                'message' => 'request_failed',
                'error' => (string)$resp->get_error_message(),
            ], 502);
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        $body = (string) wp_remote_retrieve_body($resp);

        $bodySnippet = $body;
        $truncated = false;
        if (strlen($bodySnippet) > 4000) {
            $bodySnippet = substr($bodySnippet, 0, 4000);
            $truncated = true;
        }

        $decoded = json_decode($body, true);
        $content = '';
        $dailyText = '';
        $parsedOk = false;
        if (is_array($decoded)) {
            $content = isset($decoded['content']) && is_string($decoded['content']) ? $decoded['content'] : '';
            $dailyText = isset($decoded['daily_text']) && is_string($decoded['daily_text']) ? $decoded['daily_text'] : '';
            $parsedOk = ($content !== '' || $dailyText !== '');
        }

        wp_send_json_success([
            'url' => $url,
            'date' => $date,
            'http_code' => $code,
            'parsed_ok' => $parsedOk,
            'content' => $content,
            'daily_text' => $dailyText,
            'body_snippet' => $bodySnippet,
            'body_truncated' => $truncated,
        ]);
    }
}
