(function($){
  'use strict';

  function cfg(){
    return window.DTAROT_POST_LINKS || { ajaxUrl: (window.ajaxurl || ''), nonce: '', postId: 0 };
  }

  function postAjax(data, cb){
    data = data || {};
    data.nonce = cfg().nonce;
    return $.post(cfg().ajaxUrl || window.ajaxurl, data, cb);
  }

  function escapeAttr(s){
    return String(s || '')
      .replace(/&/g, '&amp;')
      .replace(/"/g, '&quot;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;');
  }
  function escapeHtml(s){
    return String(s || '')
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  function stripHtml(html){
    try {
      const div = document.createElement('div');
      div.innerHTML = String(html || '');
      return (div.textContent || div.innerText || '').toString();
    } catch (e) {
      return String(html || '');
    }
  }

  function getClassicContentHtml(){
    if (window.tinymce) {
      const ed = tinymce.get('content');
      if (ed) return ed.getContent();
    }
    const ta = document.getElementById('content');
    return ta ? (ta.value || '') : '';
  }

  function getBlockEditorContentHtml(){
    try {
      if (!(window.wp && wp.data && wp.data.select)) return '';
      const sel = wp.data.select('core/editor');
      if (!sel || !sel.getEditedPostContent) return '';
      return sel.getEditedPostContent() || '';
    } catch (e) {
      return '';
    }
  }

  function getPostTitle(){
    try {
      if (window.wp && wp.data && wp.data.select) {
        const sel = wp.data.select('core/editor');
        if (sel && sel.getEditedPostAttribute) {
          return (sel.getEditedPostAttribute('title') || '').toString();
        }
      }
    } catch (e) {}

    const el = document.getElementById('title');
    return el ? (el.value || '') : '';
  }

  function isBlockEditor(){
    try {
      if (!(window.wp && wp.data && wp.data.select)) return false;
      const sel = wp.data.select('core/editor');
      return !!(sel && sel.getEditedPostContent);
    } catch (e) {
      return false;
    }
  }

  function linkFirstOccurrenceHtml(html, anchor, url){
    html = String(html || '');
    anchor = String(anchor || '').trim();
    url = String(url || '').trim();
    if (!html || !anchor || !url) return { html: html, did: false };

    const parts = html.split(/(<[^>]+>)/g);
    const needle = anchor.toLowerCase();
    const disallowed = new Set(['h1','h2','a']);
    const stack = [];
    let did = false;
    for (let i = 0; i < parts.length; i++) {
      const part = parts[i];
      if (!part) continue;

      if (part[0] === '<') {
        const m = part.match(/^<\s*(\/)?\s*([a-zA-Z0-9:_-]+)/);
        if (m) {
          const isClose = !!m[1];
          const tag = String(m[2] || '').toLowerCase();
          const selfClosing = /\/\s*>$/.test(part) || /^<\s*!(?:--|doctype)/i.test(part) || /^<\s*\?/.test(part);
          if (!selfClosing && tag) {
            if (isClose) {
              for (let j = stack.length - 1; j >= 0; j--) {
                if (stack[j] === tag) {
                  stack.length = j;
                  break;
                }
              }
            } else {
              stack.push(tag);
            }
          }
        }
        continue;
      }

      if (stack.some(t => disallowed.has(t))) continue;
      const hay = part.toLowerCase();
      const idx = hay.indexOf(needle);
      if (idx === -1) continue;

      const before = part.slice(0, idx);
      const match = part.slice(idx, idx + anchor.length);
      const after = part.slice(idx + anchor.length);
      parts[i] = before + '<a href="' + escapeAttr(url) + '">' + escapeHtml(match) + '</a>' + after;
      did = true;
      break;
    }

    return { html: parts.join(''), did: did };
  }

  function autoLinkPhrase(url, anchor, title){
    url = String(url || '');
    anchor = String(anchor || '');
    title = String(title || '');
    if (!url) return false;

    if (isBlockEditor()) {
      const html = getBlockEditorContentHtml();
      const a = anchor || '';
      const res = (a && html && html.toLowerCase().indexOf(a.toLowerCase()) !== -1) ? linkFirstOccurrenceHtml(html, a, url) : { html: html, did: false };
      if (res.did) {
        try {
          wp.data.dispatch('core/editor').editPost({ content: res.html });
          return true;
        } catch (e) {}
      }
      // Fallback: append a small link paragraph at the end.
      const text = anchor || title || url;
      const appended = (html || '') + "\n<p><a href=\"" + escapeAttr(url) + "\">" + escapeHtml(text) + "</a></p>";
      try {
        wp.data.dispatch('core/editor').editPost({ content: appended });
        return true;
      } catch (e) {
        return false;
      }
    }

    // Classic editor: selection preferred.
    if (window.tinymce) {
      const ed = tinymce.get('content');
      if (ed) {
        try {
          const selHtml = ed.selection.getContent({ format: 'html' });
          const hasSel = (selHtml || '').replace(/\s+/g, '').length > 0;
          if (hasSel) {
            ed.selection.setContent('<a href="' + escapeAttr(url) + '">' + selHtml + '</a>');
            ed.focus();
            return true;
          }

          const cur = ed.getContent({ format: 'raw' });
          const a = anchor || '';
          if (a && cur && cur.toLowerCase().indexOf(a.toLowerCase()) !== -1) {
            const res = linkFirstOccurrenceHtml(cur, a, url);
            if (res.did) {
              ed.setContent(res.html);
              ed.focus();
              return true;
            }
          }

          const text = anchor || title || url;
          ed.insertContent('<a href="' + escapeAttr(url) + '">' + escapeHtml(text) + '</a>');
          ed.focus();
          return true;
        } catch (e) {}
      }
    }

    return false;
  }

  function renderItems(container, items, showFallbackNote){
    if (!container) return;
    const list = Array.isArray(items) ? items : [];
    if (!list.length) {
      container.innerHTML = '<p class="description" style="margin:8px 0;">No matches found.</p>';
      return;
    }

    let html = '';
    if (showFallbackNote) {
      html += '<p class="description" style="margin:8px 0;">No strong connections found — showing random posts/pages.</p>';
    }

    html += '<ul>';
    list.forEach(function(it){
      const title = it && it.title ? String(it.title) : '';
      const url = it && it.url ? String(it.url) : '';
      const type = it && it.type ? String(it.type) : '';
      const id = it && (typeof it.id === 'number' ? it.id : (parseInt(it.id, 10) || 0));
      const anchor = it && it.anchor ? String(it.anchor) : '';
      if (!url) return;

      html += '<li>';
      html += '<a href="' + escapeAttr(url) + '" target="_blank" rel="noopener noreferrer">' + escapeHtml(title || url) + '</a>';
      if (type) html += ' <span class="description">(' + escapeHtml(type) + (id ? (' #' + id) : '') + ')</span>';
      if (anchor) html += '<span class="description dtarot-post-links-anchor">Suggested: “' + escapeHtml(anchor) + '”</span>';
      html += '<span class="dtarot-post-links-actions">'
        + '<button type="button" class="button button-small" data-dtarot-pl="open" data-url="' + escapeAttr(url) + '">Open</button>'
        + '<button type="button" class="button button-small" data-dtarot-pl="link" data-url="' + escapeAttr(url) + '" data-title="' + escapeAttr(title) + '" data-anchor="' + escapeAttr(anchor) + '">Link phrase</button>'
        + '</span>';
      html += '</li>';
    });
    html += '</ul>';

    container.innerHTML = html;
  }

  function collectTextForSuggest(){
    const title = getPostTitle();
    const html = isBlockEditor() ? getBlockEditorContentHtml() : getClassicContentHtml();
    const text = stripHtml(html);
    // Include title so suggestions are better.
    const combined = (title ? (title + "\n") : '') + text;
    return { title: title, html: html, text: text, combined: combined };
  }

  function init(){
    const wrap = document.getElementById('dtarot-post-link-suggestions');
    if (!wrap) return;

    const statusEl = document.getElementById('dtarot-post-links-status');
    const listEl = document.getElementById('dtarot-post-links-list');
    const qEl = document.getElementById('dtarot-post-links-q');
    const resultsEl = document.getElementById('dtarot-post-links-results');

    function setStatus(msg){
      if (statusEl) statusEl.textContent = msg || '';
    }

    function suggest(){
      const p = collectTextForSuggest();
      if ((p.combined || '').trim().length < 8) {
        renderItems(listEl, []);
        setStatus('Type a bit more to get suggestions.');
        return;
      }
      setStatus('Loading…');
      postAjax({
        action: 'dtarot_related_link_suggest_from_text',
        content: p.combined,
        daily_text: '',
        exclude_post_id: String(cfg().postId || 0)
      }, function(res){
        setStatus('');
        if (!res || !res.success || !res.data) {
          renderItems(listEl, []);
          return;
        }
        renderItems(listEl, res.data.items || [], !!res.data.fallback);
      });
    }

    function search(){
      const q = (qEl && qEl.value ? qEl.value : '').toString().trim();
      if (!q) {
        renderItems(resultsEl, []);
        return;
      }
      resultsEl.innerHTML = '<p class="description" style="margin:8px 0;">Loading…</p>';
      postAjax({ action: 'dtarot_related_link_search', q: q }, function(res){
        if (!res || !res.success || !res.data) {
          renderItems(resultsEl, []);
          return;
        }
        renderItems(resultsEl, res.data.items || []);
      });
    }

    $(document).on('click', '#dtarot-post-links-refresh', function(e){ e.preventDefault(); suggest(); });
    $(document).on('click', '#dtarot-post-links-search', function(e){ e.preventDefault(); search(); });

    if (qEl) {
      qEl.addEventListener('keydown', function(e){
        if (e.key === 'Enter') {
          e.preventDefault();
          search();
        }
      });
    }

    wrap.addEventListener('click', function(e){
      const btn = e.target && e.target.closest ? e.target.closest('button[data-dtarot-pl]') : null;
      if (!btn) return;
      const act = (btn.getAttribute('data-dtarot-pl') || '').toString();
      const url = (btn.getAttribute('data-url') || '').toString();
      const title = (btn.getAttribute('data-title') || '').toString();
      const anchor = (btn.getAttribute('data-anchor') || '').toString();
      e.preventDefault();

      if (act === 'open') {
        if (url) window.open(url, '_blank', 'noopener,noreferrer');
        return;
      }
      if (act === 'link') {
        autoLinkPhrase(url, anchor, title);
        return;
      }
    }, true);

    // Auto-suggest as you type (block editor: subscribe; classic: keyup)
    let timer = null;
    function queue(){
      if (timer) clearTimeout(timer);
      timer = setTimeout(suggest, 900);
    }

    if (isBlockEditor() && window.wp && wp.data && wp.data.subscribe) {
      let last = '';
      wp.data.subscribe(function(){
        const now = (collectTextForSuggest().combined || '').slice(0, 5000);
        if (now !== last) {
          last = now;
          queue();
        }
      });
    } else {
      $(document).on('keyup', '#title, #content', function(){ queue(); });
      if (window.tinymce) {
        // Try to bind after init
        let tries = 0;
        const iv = setInterval(function(){
          tries++;
          const ed = tinymce.get('content');
          if (ed && !ed._dtarotPostLinksBound) {
            ed._dtarotPostLinksBound = true;
            ed.on('keyup change setcontent', function(){ queue(); });
          }
          if (tries >= 10) clearInterval(iv);
        }, 500);
      }
    }

    // Initial suggestions
    suggest();
  }

  $(init);

})(jQuery);
