(function ($) {
    'use strict';

    function el(tag, attrs, children) {
        var node = document.createElement(tag);
        if (attrs) {
            Object.keys(attrs).forEach(function (k) {
                if (k === 'class') node.className = attrs[k];
                else if (k === 'html') node.innerHTML = attrs[k];
                else node.setAttribute(k, attrs[k]);
            });
        }
        if (children && children.length) {
            children.forEach(function (c) {
                if (typeof c === 'string') node.appendChild(document.createTextNode(c));
                else if (c) node.appendChild(c);
            });
        }
        return node;
    }

    function openModal(deactivateHref) {
        var overlay = el('div', { class: 'dtarot-uninstall-overlay', role: 'dialog', 'aria-modal': 'true' });
        var modal = el('div', { class: 'dtarot-uninstall-modal' });

        var title = el('div', { class: 'dtarot-uninstall-title' }, ['Quick question before you deactivate']);
        var desc = el('div', { class: 'dtarot-uninstall-desc' }, [
            'This is optional, but it helps us improve. You can skip and deactivate immediately.'
        ]);

        var reasons = [
            { v: 'testing', label: 'Just testing' },
            { v: 'too_complicated', label: 'Too complicated / hard to set up' },
            { v: 'missing_feature', label: 'Missing a feature I need' },
            { v: 'not_working', label: 'Not working as expected' },
            { v: 'performance', label: 'Performance / speed issues' },
            { v: 'switched', label: 'Switched to a different plugin' },
            { v: 'other', label: 'Other' }
        ];

        var form = el('div', { class: 'dtarot-uninstall-form' });
        var reasonWrap = el('div', { class: 'dtarot-uninstall-reasons' });

        reasons.forEach(function (r, idx) {
            var id = 'dtarot-uninstall-reason-' + idx;
            var input = el('input', { type: 'radio', name: 'dtarot_uninstall_reason', id: id, value: r.v });
            var label = el('label', { for: id }, [r.label]);
            var row = el('div', { class: 'dtarot-uninstall-reason' }, [input, label]);
            reasonWrap.appendChild(row);
        });

        var details = el('textarea', {
            class: 'dtarot-uninstall-details',
            rows: '3',
            placeholder: 'Optional details (what went wrong / what you needed)…'
        });

        var actions = el('div', { class: 'dtarot-uninstall-actions' });
        var btnCancel = el('button', { type: 'button', class: 'button' }, ['Cancel']);
        var btnSkip = el('button', { type: 'button', class: 'button' }, ['Skip & Deactivate']);
        var btnSubmit = el('button', { type: 'button', class: 'button button-primary' }, ['Submit & Deactivate']);

        actions.appendChild(btnCancel);
        actions.appendChild(btnSkip);
        actions.appendChild(btnSubmit);

        form.appendChild(reasonWrap);
        form.appendChild(details);

        modal.appendChild(title);
        modal.appendChild(desc);
        modal.appendChild(form);
        modal.appendChild(actions);
        overlay.appendChild(modal);
        document.body.appendChild(overlay);

        function close() {
            if (overlay && overlay.parentNode) overlay.parentNode.removeChild(overlay);
        }

        function selectedReason() {
            var checked = overlay.querySelector('input[name="dtarot_uninstall_reason"]:checked');
            return checked ? checked.value : '';
        }

        function goDeactivate() {
            window.location.href = deactivateHref;
        }

        btnCancel.addEventListener('click', function () {
            close();
        });

        btnSkip.addEventListener('click', function () {
            goDeactivate();
        });

        btnSubmit.addEventListener('click', function () {
            var reason = selectedReason();
            var note = (details && details.value) ? String(details.value).slice(0, 800) : '';

            // Always allow deactivation even if feedback fails.
            if (!window.DTAROT_PLUGINS || !DTAROT_PLUGINS.ajaxUrl || !DTAROT_PLUGINS.nonce) {
                goDeactivate();
                return;
            }

            $.post(DTAROT_PLUGINS.ajaxUrl, {
                action: 'dtarot_uninstall_feedback',
                nonce: DTAROT_PLUGINS.nonce,
                context: 'deactivate',
                reason: reason,
                details: note
            }).always(function () {
                goDeactivate();
            });
        });

        // ESC closes
        overlay.addEventListener('keydown', function (e) {
            if (e.key === 'Escape') close();
        });

        // Click outside closes
        overlay.addEventListener('click', function (e) {
            if (e.target === overlay) close();
        });

        // Focus first radio
        var first = overlay.querySelector('input[name="dtarot_uninstall_reason"]');
        if (first) first.focus();
    }

    function hookDeactivateLinks() {
        if (!window.DTAROT_PLUGINS || !DTAROT_PLUGINS.plugin) return;

        var row = document.querySelector('tr[data-plugin="' + DTAROT_PLUGINS.plugin + '"]');
        if (!row) return;

        var links = row.querySelectorAll('.deactivate a');
        if (!links || !links.length) return;

        links.forEach(function (a) {
            a.addEventListener('click', function (e) {
                try {
                    e.preventDefault();
                    openModal(a.getAttribute('href'));
                } catch (err) {
                    // Fail open
                    window.location.href = a.getAttribute('href');
                }
            });
        });
    }

    document.addEventListener('DOMContentLoaded', function () {
        hookDeactivateLinks();
    });

})(jQuery);
