(function () {
    function uuid4() {
        // Not cryptographic; sufficient for analytics correlation.
        try {
            return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
                var r = Math.random() * 16 | 0;
                var v = c === 'x' ? r : (r & 0x3 | 0x8);
                return v.toString(16);
            });
        } catch (err) {
            return String(Date.now()) + '-' + String(Math.random()).slice(2);
        }
    }

    function getSessionId() {
        try {
            var k = 'dtarot_session_id_v1';
            var sid = '';
            if (window.sessionStorage) {
                sid = window.sessionStorage.getItem(k) || '';
                if (!sid) {
                    sid = uuid4();
                    window.sessionStorage.setItem(k, sid);
                }
            }
            return sid || '';
        } catch (err) {
            return '';
        }
    }
    function analyticsEnabled() {
        function hasStatisticsConsent() {
            try {
                // Cookiebot
                if (window.Cookiebot && Cookiebot.consent) {
                    return !!Cookiebot.consent.statistics;
                }
            } catch (err) {}

            try {
                // Complianz
                if (typeof window.cmplz_has_consent === 'function') {
                    return !!window.cmplz_has_consent('statistics');
                }
            } catch (err) {}

            try {
                // Borlabs Cookie (best-effort; API differs across versions)
                if (window.BorlabsCookie) {
                    if (typeof window.BorlabsCookie.checkCookieConsent === 'function') {
                        return !!window.BorlabsCookie.checkCookieConsent('statistics');
                    }
                    if (window.BorlabsCookie.Consents && typeof window.BorlabsCookie.Consents.statistics !== 'undefined') {
                        return !!window.BorlabsCookie.Consents.statistics;
                    }
                    // If Borlabs is present but we can't determine, be conservative.
                    return false;
                }
            } catch (err) {}

            // No known CMP detected.
            return true;
        }

        // If a CMP is present, require statistics consent.
        try {
            var cmpPresent = !!(window.Cookiebot || window.cmplz_has_consent || window.BorlabsCookie);
            if (cmpPresent && !hasStatisticsConsent()) {
                return false;
            }
        } catch (err) {}

        try {
            if (window.DTAROT_FRONTEND && typeof DTAROT_FRONTEND.analyticsEnabled !== 'undefined') {
                return !!DTAROT_FRONTEND.analyticsEnabled;
            }
        } catch (err) {}
        return true;
    }

    function pushDataLayerEvent(name, payload) {
        if (!analyticsEnabled()) return;
        try {
            window.dataLayer = window.dataLayer || [];
            var obj = payload || {};
            obj.event = name;
            obj.dtarot_event_id = uuid4();
            obj.dtarot_session_id = getSessionId();
            window.dataLayer.push(obj);
        } catch (err) {}
    }

    function trackViewOnce(root) {
        try {
            if (window.__dtarotViewTracked) return;
            root = root || document;
            var hasWidget = root.querySelector && root.querySelector('.dtarot-frontend');
            if (!hasWidget) return;

            window.__dtarotViewTracked = true;

            var ctx = (window.DTAROT_FRONTEND && DTAROT_FRONTEND.analytics) ? DTAROT_FRONTEND.analytics : {};

            pushDataLayerEvent('dtarot_view', {
                dtarot_page_type: ctx.pageType || '',
                dtarot_date: ctx.date || '',
                dtarot_deck_id: ctx.deckId || '',
                dtarot_card_id: ctx.cardId || '',
                dtarot_system: ctx.system || '',
                dtarot_url: ctx.url || (window.location.href || '')
            });
        } catch (err) {}
    }

    function toggle(el) {
        if (!el) return;
        if (el.classList.contains('is-flipped')) {
            el.classList.remove('is-flipped');
        } else {
            el.classList.add('is-flipped');
        }

        // Track flip events.
        try {
            if (el.classList.contains('is-flipped')) {
                var ctx = (window.DTAROT_FRONTEND && DTAROT_FRONTEND.analytics) ? DTAROT_FRONTEND.analytics : {};
                pushDataLayerEvent('dtarot_card_flip', {
                    dtarot_page_type: ctx.pageType || '',
                    dtarot_date: ctx.date || '',
                    dtarot_deck_id: ctx.deckId || '',
                    dtarot_card_id: ctx.cardId || '',
                    dtarot_system: ctx.system || '',
                    dtarot_url: ctx.url || (window.location.href || '')
                });
            }
        } catch (err) {}

        var card = el.closest ? el.closest('.dtarot-spread-card') : null;
        if (card) {
            card.classList.toggle('is-flipped', el.classList.contains('is-flipped'));
            if (el.classList.contains('is-flipped')) {
                card.classList.add('is-revealed');
            }
        }

        var overlap = el.closest ? el.closest('.dtarot-spread-overlap-card') : null;
        if (overlap && overlap.getAttribute) {
            var slotId = overlap.getAttribute('data-dtarot-slot');
            if (slotId) {
                var selector = '.dtarot-spread-card[data-dtarot-slot="' + slotId + '"]';
                var slot = document.querySelector(selector);
                if (slot) {
                    slot.classList.toggle('is-flipped', el.classList.contains('is-flipped'));
                    if (el.classList.contains('is-flipped')) {
                        slot.classList.add('is-revealed');
                    }
                }
            }
            overlap.classList.toggle('is-boosted', el.classList.contains('is-flipped'));
        }
    }

    function maybeNavigate(el) {
        if (!el) return false;
        var href = el.getAttribute('data-dtarot-href');
        if (!href) return false;

        // If already flipped, a second activation navigates to details.
        if (el.classList.contains('is-flipped')) {
            window.location.href = href;
            return true;
        }

        return false;
    }

    function init(root) {
        root = root || document;

        // Fire a single view event if Daily Tarot output is present.
        trackViewOnce(root);

        var els = root.querySelectorAll('[data-dtarot-flip]');
        for (var i = 0; i < els.length; i++) {
            (function (el) {
                if (el.__dtarotFlipBound) return;
                el.__dtarotFlipBound = true;

                el.addEventListener('click', function (e) {
                    // Don’t flip when selecting text.
                    if (window.getSelection && window.getSelection().toString()) return;

                    if (maybeNavigate(el)) {
                        e.preventDefault();
                        return;
                    }

                    e.preventDefault();
                    toggle(el);
                });

                el.addEventListener('keydown', function (e) {
                    var key = e.key || e.keyCode;
                    if (key === 'Enter' || key === ' ' || key === 13 || key === 32) {
                        if (maybeNavigate(el)) {
                            e.preventDefault();
                            return;
                        }

                        e.preventDefault();
                        toggle(el);
                    }
                });
            })(els[i]);
        }

        var mediaBlocks = root.querySelectorAll('.dtarot-spread-card-media');
        for (var m = 0; m < mediaBlocks.length; m++) {
            (function (media) {
                if (media.__dtarotMediaBound) return;
                media.__dtarotMediaBound = true;

                media.addEventListener('click', function (e) {
                    var flip = e.target && e.target.closest ? e.target.closest('[data-dtarot-flip]') : null;
                    if (flip) return;
                    var slot = media.closest ? media.closest('.dtarot-spread-card') : null;
                    if (!slot) return;
                    var flipped = slot.classList.toggle('is-flipped');
                    if (flipped) {
                        slot.classList.add('is-revealed');
                    }
                });
            })(mediaBlocks[m]);
        }

        var overlapCards = root.querySelectorAll('.dtarot-spread-overlap-card');
        for (var k = 0; k < overlapCards.length; k++) {
            (function (card) {
                if (card.__dtarotOverlapBound) return;
                card.__dtarotOverlapBound = true;

                card.addEventListener('click', function (e) {
                    var flip = e.target && e.target.closest ? e.target.closest('[data-dtarot-flip]') : null;
                    if (flip) return;
                    var slotId = card.getAttribute ? card.getAttribute('data-dtarot-slot') : '';
                    if (!slotId) return;
                    var selector = '.dtarot-spread-card[data-dtarot-slot="' + slotId + '"]';
                    var slot = document.querySelector(selector);
                    if (slot) {
                        var flipped = slot.classList.toggle('is-flipped');
                        if (flipped) {
                            slot.classList.add('is-revealed');
                        }
                    }
                });

                card.addEventListener('mouseenter', function () {
                    card.classList.add('is-boosted');
                });
                card.addEventListener('mouseleave', function () {
                    card.classList.remove('is-boosted');
                });
                card.addEventListener('focusin', function () {
                    card.classList.add('is-boosted');
                });
                card.addEventListener('focusout', function () {
                    card.classList.remove('is-boosted');
                });
            })(overlapCards[k]);
        }

        var shareBoxes = root.querySelectorAll('[data-dtarot-share="1"]');
        for (var s = 0; s < shareBoxes.length; s++) {
            (function (box) {
                if (box.__dtarotShareBound) return;
                box.__dtarotShareBound = true;

                box.addEventListener('click', function (e) {
                    var target = e.target && e.target.closest ? e.target.closest('[data-dtarot-share-link]') : null;
                    if (!target) return;

                    var platform = target.getAttribute('data-dtarot-share-link') || '';
                    if (platform === 'instagram' && window.DTAROT_FRONTEND && DTAROT_FRONTEND.shareImage && DTAROT_FRONTEND.shareImage.enabled) {
                        if (window.DTAROT_SHARE_IMAGE && typeof window.DTAROT_SHARE_IMAGE.openFromShareBox === 'function') {
                            e.preventDefault();
                            window.DTAROT_SHARE_IMAGE.openFromShareBox(box);
                            return;
                        }
                    }

                    if (platform === 'copy') {
                        e.preventDefault();
                        var url = box.getAttribute('data-dtarot-url') || window.location.href;
                        try {
                            if (navigator.clipboard && navigator.clipboard.writeText) {
                                navigator.clipboard.writeText(url);
                            } else {
                                var tmp = document.createElement('input');
                                tmp.value = url;
                                document.body.appendChild(tmp);
                                tmp.select();
                                document.execCommand('copy');
                                document.body.removeChild(tmp);
                            }
                        } catch (err) {}
                    }

                    trackShare(box, platform);
                });
            })(shareBoxes[s]);
        }

        // Horizontal scrollers (deck switcher + deck card strip)
        var btns = root.querySelectorAll('[data-dtarot-scroll][data-dtarot-scroll-target]');
        for (var j = 0; j < btns.length; j++) {
            (function (btn) {
                if (btn.__dtarotScrollBound) return;
                btn.__dtarotScrollBound = true;

                btn.addEventListener('click', function (e) {
                    var dir = btn.getAttribute('data-dtarot-scroll');
                    var targetId = btn.getAttribute('data-dtarot-scroll-target');
                    if (!targetId) return;
                    var strip = document.getElementById(targetId);
                    if (!strip) return;

                    e.preventDefault();

                    var delta = Math.max(120, Math.floor(strip.clientWidth * 0.85));
                    if (dir === 'prev') delta = -delta;

                    try {
                        strip.scrollBy({ left: delta, top: 0, behavior: 'smooth' });
                    } catch (err) {
                        strip.scrollLeft += delta;
                    }
                });
            })(btns[j]);
        }

        var bookingForms = root.querySelectorAll('.dtarot-booking-form');
        for (var b = 0; b < bookingForms.length; b++) {
            (function (form) {
                if (form.__dtarotBookingBound) return;
                form.__dtarotBookingBound = true;

                var dateInput = form.querySelector('input[name="booking_date"]');
                var timeSelect = form.querySelector('select[name="booking_time"]');
                var tzInput = form.querySelector('input[name="booking_timezone"]');
                var startUtcInput = form.querySelector('input[name="booking_start_utc"]');
                var readingInputs = form.querySelectorAll('input[name="reading_type"]');

                if (tzInput && !tzInput.value) {
                    try {
                        var tz = Intl.DateTimeFormat().resolvedOptions().timeZone;
                        if (tz) tzInput.value = tz;
                    } catch (err) {}
                }

                function setLoading() {
                    if (!timeSelect) return;
                    var label = (window.DTAROT_FRONTEND && DTAROT_FRONTEND.i18n && DTAROT_FRONTEND.i18n.loading) ? DTAROT_FRONTEND.i18n.loading : 'Loading...';
                    timeSelect.innerHTML = '<option value="">' + label + '</option>';
                }

                function setNoSlots() {
                    if (!timeSelect) return;
                    var label = (window.DTAROT_FRONTEND && DTAROT_FRONTEND.i18n && DTAROT_FRONTEND.i18n.noSlots) ? DTAROT_FRONTEND.i18n.noSlots : 'No available times';
                    timeSelect.innerHTML = '<option value="">' + label + '</option>';
                }

                function loadSlots() {
                    if (!window.DTAROT_FRONTEND || !DTAROT_FRONTEND.ajaxUrl) return;
                    if (!dateInput || !timeSelect) return;
                    var reading = form.querySelector('input[name="reading_type"]:checked');
                    var date = dateInput.value || '';
                    if (!reading || !date) return;

                    if (startUtcInput) startUtcInput.value = '';
                    setLoading();

                    var fd = new FormData();
                    fd.append('action', 'dtarot_booking_slots');
                    fd.append('nonce', DTAROT_FRONTEND.nonce || '');
                    fd.append('date', date);
                    fd.append('reading_type', reading.value || '');
                    fd.append('timezone', tzInput ? tzInput.value : '');

                    fetch(DTAROT_FRONTEND.ajaxUrl, {
                        method: 'POST',
                        body: fd
                    }).then(function (resp) { return resp.json(); }).then(function (data) {
                        if (!data || !data.success || !data.data || !data.data.slots) {
                            setNoSlots();
                            return;
                        }
                        var slots = data.data.slots || [];
                        if (!slots.length) {
                            setNoSlots();
                            return;
                        }
                        var html = '<option value="">' + (DTAROT_FRONTEND.i18n && DTAROT_FRONTEND.i18n.selectTime ? DTAROT_FRONTEND.i18n.selectTime : 'Select a time') + '</option>';
                        for (var i = 0; i < slots.length; i++) {
                            var slot = slots[i];
                            var val = slot.start_utc ? String(slot.start_utc) : '';
                            var label = slot.label || val;
                            html += '<option value="' + val + '">' + label + '</option>';
                        }
                        timeSelect.innerHTML = html;
                    }).catch(function () {
                        setNoSlots();
                    });
                }

                // Booking analytics.
                try {
                    var started = false;
                    function markStart() {
                        if (started) return;
                        started = true;
                        var ctx = (window.DTAROT_FRONTEND && DTAROT_FRONTEND.analytics) ? DTAROT_FRONTEND.analytics : {};
                        pushDataLayerEvent('dtarot_booking_start', {
                            dtarot_page_type: ctx.pageType || '',
                            dtarot_date: (dateInput && dateInput.value) ? dateInput.value : (ctx.date || ''),
                            dtarot_url: ctx.url || (window.location.href || '')
                        });
                    }

                    if (dateInput) dateInput.addEventListener('change', markStart);
                    for (var rr = 0; rr < readingInputs.length; rr++) {
                        readingInputs[rr].addEventListener('change', markStart);
                    }

                    form.addEventListener('submit', function () {
                        var ctx = (window.DTAROT_FRONTEND && DTAROT_FRONTEND.analytics) ? DTAROT_FRONTEND.analytics : {};
                        pushDataLayerEvent('dtarot_booking_submit', {
                            dtarot_page_type: ctx.pageType || '',
                            dtarot_date: (dateInput && dateInput.value) ? dateInput.value : (ctx.date || ''),
                            dtarot_url: ctx.url || (window.location.href || '')
                        });
                    });
                } catch (err) {}

                if (timeSelect) {
                    timeSelect.addEventListener('change', function () {
                        if (startUtcInput) {
                            startUtcInput.value = timeSelect.value || '';
                        }
                    });
                }

                if (dateInput) {
                    dateInput.addEventListener('change', loadSlots);
                }
                for (var r = 0; r < readingInputs.length; r++) {
                    readingInputs[r].addEventListener('change', loadSlots);
                }
            })(bookingForms[b]);
        }

        var modalOpens = root.querySelectorAll('[data-dtarot-modal-open]');
        for (var p = 0; p < modalOpens.length; p++) {
            (function (btn) {
                if (btn.__dtarotModalOpenBound) return;
                btn.__dtarotModalOpenBound = true;

                btn.addEventListener('click', function () {
                    var id = btn.getAttribute('data-dtarot-modal-open');
                    if (!id) return;
                    var modal = document.getElementById(id);
                    if (!modal) return;
                    modal.setAttribute('aria-hidden', 'false');
                    modal.classList.add('is-open');
                    document.body.classList.add('dtarot-modal-open');
                });
            })(modalOpens[p]);
        }

        var modalCloses = root.querySelectorAll('[data-dtarot-modal-close]');
        for (var q = 0; q < modalCloses.length; q++) {
            (function (btn) {
                if (btn.__dtarotModalCloseBound) return;
                btn.__dtarotModalCloseBound = true;

                btn.addEventListener('click', function () {
                    var id = btn.getAttribute('data-dtarot-modal-close');
                    if (!id) return;
                    var modal = document.getElementById(id);
                    if (!modal) return;
                    modal.setAttribute('aria-hidden', 'true');
                    modal.classList.remove('is-open');
                    document.body.classList.remove('dtarot-modal-open');
                });
            })(modalCloses[q]);
        }
    }

    function trackShare(box, platform) {
        if (!box || !platform) return;
        if (!window.DTAROT_FRONTEND || !DTAROT_FRONTEND.ajaxUrl || !DTAROT_FRONTEND.shareNonce) return;

        // GA4/Site Kit friendly event.
        try {
            var ctx = (window.DTAROT_FRONTEND && DTAROT_FRONTEND.analytics) ? DTAROT_FRONTEND.analytics : {};
            pushDataLayerEvent('dtarot_share', {
                dtarot_platform: platform,
                dtarot_context: box.getAttribute('data-dtarot-context') || '',
                dtarot_page_type: ctx.pageType || '',
                dtarot_date: box.getAttribute('data-dtarot-date') || (ctx.date || ''),
                dtarot_deck_id: box.getAttribute('data-dtarot-deck') || (ctx.deckId || ''),
                dtarot_card_id: box.getAttribute('data-dtarot-card') || (ctx.cardId || ''),
                dtarot_system: ctx.system || '',
                dtarot_url: box.getAttribute('data-dtarot-url') || (ctx.url || window.location.href || '')
            });
        } catch (err) {}

        var fd = new FormData();
        fd.append('action', 'dtarot_share_track');
        fd.append('nonce', DTAROT_FRONTEND.shareNonce);
        fd.append('platform', platform);
        fd.append('context', box.getAttribute('data-dtarot-context') || '');
        fd.append('url', box.getAttribute('data-dtarot-url') || window.location.href);
        fd.append('page', window.location.pathname || '');
        fd.append('date', box.getAttribute('data-dtarot-date') || '');
        fd.append('post_id', box.getAttribute('data-dtarot-post-id') || '');
        fd.append('deck_id', box.getAttribute('data-dtarot-deck') || '');
        fd.append('card_id', box.getAttribute('data-dtarot-card') || '');
        fd.append('preset', box.getAttribute('data-dtarot-preset') || '');
        fd.append('pack', box.getAttribute('data-dtarot-pack') || '');

        try {
            if (navigator.sendBeacon) {
                navigator.sendBeacon(DTAROT_FRONTEND.ajaxUrl, fd);
                return;
            }
        } catch (err) {}

        try {
            fetch(DTAROT_FRONTEND.ajaxUrl, { method: 'POST', body: fd, credentials: 'same-origin' });
        } catch (err) {}
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function () { init(document); });
    } else {
        init(document);
    }
})();
