(function($){
  'use strict';
  if (typeof window.DTAROT === 'undefined') return;

  function t(key, fallback){
    try {
      if (DTAROT && DTAROT.i18n && typeof DTAROT.i18n[key] === 'string' && DTAROT.i18n[key]) {
        return DTAROT.i18n[key];
      }
    } catch (e) {}
    return fallback;
  }

  function pad2(n){
    return (n < 10) ? ('0' + n) : String(n);
  }

  function localYmd(){
    const d = new Date();
    return d.getFullYear() + '-' + pad2(d.getMonth() + 1) + '-' + pad2(d.getDate());
  }

  function calendarMaxDate(){
    try {
      const cal = document.querySelector('.dtarot-calendar');
      if (!cal) return '';
      return (cal.getAttribute('data-max-date') || '').toString();
    } catch (e) {
      return '';
    }
  }

  function calendarAllowedDate(){
    try {
      const cal = document.querySelector('.dtarot-calendar');
      if (!cal) return '';
      return (cal.getAttribute('data-allowed-date') || '').toString();
    } catch (e) {
      return '';
    }
  }

  function calendarIsPro(){
    try {
      const cal = document.querySelector('.dtarot-calendar');
      if (!cal) return true;
      return (cal.getAttribute('data-is-pro') || '') === '1';
    } catch (e) {
      return true;
    }
  }

  function isDateBeyondMax(date, maxDate){
    if (!date || !maxDate) return false;
    // Lexicographic compare works for YYYY-MM-DD.
    return String(date) > String(maxDate);
  }

  function syncTodayHighlight(){
    const ymd = localYmd();
    // Remove any server-rendered highlight that might be stale.
    document.querySelectorAll('.dtarot-day-cell.dtarot-today').forEach(function(el){
      el.classList.remove('dtarot-today');
    });
    const todayCell = document.querySelector('.dtarot-day-cell[data-date="' + ymd + '"]');
    if (todayCell) todayCell.classList.add('dtarot-today');
  }

  function postAjax(data, cb){
    data = data || {};
    data.nonce = DTAROT.nonce;
    return $.post(DTAROT.ajaxUrl, data, cb);
  }

  function setAutoSaveStatus(inputEl, msg, ok){
    try {
      const td = inputEl.closest('td');
      if (!td) return;
      const el = td.querySelector('.dtarot-auto-save-status');
      if (!el) return;
      el.textContent = msg || '';
      el.style.color = ok === true ? '' : (ok === false ? '#b32d2e' : '');
    } catch (e) {}
  }

  function applyToggleEffect(inputEl, isOn){
    const targetSelector = inputEl.getAttribute('data-dtarot-target') || '';
    if (!targetSelector) return;
    document.querySelectorAll(targetSelector).forEach(function(node){
      node.style.display = isOn ? '' : 'none';
    });
  }

  $(function(){
    syncTodayHighlight();

    // Content page: fast tab switching (AJAX panel swap + History API)
    (function(){
      try {
        const wrap = document.querySelector('.dtarot-wrap[data-dtarot-content-page="1"]');
        if (!wrap) return;

        function isContentHref(href){
          if (!href) return false;
          let u;
          try { u = new URL(href, window.location.href); } catch (e) { return false; }
          if (!u.searchParams) return false;
          return (u.searchParams.get('page') || '') === 'daily-tarot-content';
        }

        function setActiveTab(tab){
          const links = wrap.querySelectorAll('.nav-tab-wrapper a.nav-tab');
          links.forEach(function(a){ a.classList.remove('nav-tab-active'); });
          links.forEach(function(a){
            try {
              const u = new URL(a.href, window.location.href);
              const t = (u.searchParams.get('tab') || 'status');
              if (t === tab) a.classList.add('nav-tab-active');
            } catch (e) {}
          });
        }

        let inFlight = false;

        function setBusy(isBusy){
          const panel = document.getElementById('dtarot-content-panel');
          if (!panel) return;
          panel.setAttribute('aria-busy', isBusy ? 'true' : 'false');
          if (isBusy) panel.classList.add('dtarot-loading');
          else panel.classList.remove('dtarot-loading');
        }

        function replacePanel(html){
          if (!html) return;
          const tmp = document.createElement('div');
          tmp.innerHTML = String(html);
          const next = tmp.querySelector('#dtarot-content-panel');
          const cur = document.getElementById('dtarot-content-panel');
          if (!next || !cur) return;
          cur.replaceWith(next);
          const tab = (next.getAttribute('data-tab') || 'status').toString();
          setActiveTab(tab);
          try {
            document.dispatchEvent(new CustomEvent('dtarot:contentPanelLoaded', { detail: { tab: tab } }));
          } catch (e) {}
        }

        function loadHref(href, pushState){
          if (!href || inFlight) return;
          if (!isContentHref(href)) {
            window.location.href = href;
            return;
          }

          inFlight = true;
          setBusy(true);

          postAjax({ action: 'dtarot_get_content_panel', href: href }, function(res){
            inFlight = false;
            setBusy(false);

            if (!res || !res.success || !res.data || typeof res.data.html !== 'string') {
              window.location.href = href;
              return;
            }

            replacePanel(res.data.html);

            if (pushState !== false) {
              try {
                window.history.pushState({ dtarotContent: 1 }, '', href);
              } catch (e) {}
            }
          });
        }

        // Intercept tab clicks + pagination links.
        wrap.addEventListener('click', function(e){
          const a = e.target && e.target.closest ? e.target.closest('a') : null;
          if (!a) return;
          if (a.target && a.target === '_blank') return;
          const href = (a.href || '').toString();
          if (!href) return;
          if (!isContentHref(href)) return;

          // Only intercept navigation within the content page.
          e.preventDefault();
          loadHref(href, true);
        }, true);

        // Intercept GET forms (e.g. Cards tab deck selector).
        wrap.addEventListener('submit', function(e){
          const form = e.target;
          if (!form || !(form instanceof HTMLFormElement)) return;
          const method = (form.getAttribute('method') || 'get').toLowerCase();
          if (method !== 'get') return;

          // Only handle forms that target the Daily Tarot content page.
          const action = (form.getAttribute('action') || window.location.href).toString();
          let u;
          try { u = new URL(action, window.location.href); } catch (e2) { return; }

          const fd = new FormData(form);
          fd.forEach(function(v, k){
            if (typeof v === 'string') u.searchParams.set(k, v);
          });

          if ((u.searchParams.get('page') || '') !== 'daily-tarot-content') return;

          e.preventDefault();
          loadHref(u.toString(), true);
        }, true);

        // Back/forward navigation
        window.addEventListener('popstate', function(){
          try {
            if (!isContentHref(window.location.href)) return;
            loadHref(window.location.href, false);
          } catch (e) {}
        });

      } catch (e) {}
    })();

    // Spreads page: modal config + preview + template editor.
    (function(){
      const page = document.querySelector('.dtarot-spreads-page');
      if (!page) return;

      const modal = document.getElementById('dtarot-spread-modal');
      const form = document.getElementById('dtarot-spread-config-form');
      const previewBtn = document.getElementById('dtarot-spread-preview-btn');
      const previewBox = document.getElementById('dtarot-spread-preview');

      function runPreview(){
        if (!form || !previewBox) return;
        const preset = form.querySelector('#dtarot_spread_preset');
        const pack = form.querySelector('#dtarot_spread_pack');
        const deck = form.querySelector('#dtarot_spread_deck');
        previewBox.textContent = t('loadingEllipsis', 'Loading...');
        postAjax({
          action: 'dtarot_spread_preview',
          preset: preset ? preset.value : '',
          pack: pack ? pack.value : '',
          deck_id: deck ? deck.value : 0,
        }, function(res){
          if (!previewBox) return;
          if (!res || !res.success || !res.data || typeof res.data.html !== 'string') {
            previewBox.textContent = t('saveFailed', 'Preview failed');
            return;
          }
          previewBox.innerHTML = res.data.html;
        });
      }

      function openModal(){
        if (!modal) return;
        modal.style.removeProperty('display');
        modal.classList.add('is-open');
        modal.setAttribute('aria-hidden', 'false');
        runPreview();
      }

      function closeModal(){
        if (!modal) return;
        modal.classList.remove('is-open');
        modal.setAttribute('aria-hidden', 'true');
      }

      function openGenericModal(target){
        if (!target) return;
        target.style.removeProperty('display');
        target.classList.add('is-open');
        target.setAttribute('aria-hidden', 'false');
      }

      function closeGenericModal(target){
        if (!target) return;
        target.classList.remove('is-open');
        target.setAttribute('aria-hidden', 'true');
      }

      page.addEventListener('click', function(e){
        const btn = e.target && e.target.closest ? e.target.closest('.dtarot-spread-config-btn') : null;
        if (btn) {
          e.preventDefault();
          const dataRaw = btn.getAttribute('data-dtarot-spread') || '{}';
          let data;
          try { data = JSON.parse(dataRaw); } catch (err) { data = {}; }
          const postId = document.getElementById('dtarot_spread_post_id');
          const index = document.getElementById('dtarot_spread_index');
          const preset = document.getElementById('dtarot_spread_preset');
          const pack = document.getElementById('dtarot_spread_pack');
          const deck = document.getElementById('dtarot_spread_deck');
          if (postId) postId.value = data.post || 0;
          if (index) index.value = data.index || 0;
          if (preset) preset.value = data.preset || '';
          if (pack) pack.value = data.pack || '';
          if (deck) deck.value = data.deck || 0;
          if (previewBox) previewBox.textContent = 'Loading...';
          openModal();
          return;
        }

        const openBtn = e.target && e.target.closest ? e.target.closest('.dtarot-open-modal') : null;
        if (openBtn) {
          e.preventDefault();
          const sel = (openBtn.getAttribute('data-dtarot-modal') || '').toString();
          if (!sel) return;
          const target = document.querySelector(sel);
          if (target) openGenericModal(target);
          return;
        }

        if (e.target && e.target.matches && e.target.matches('[data-dtarot-modal-close]')) {
          e.preventDefault();
          e.stopPropagation();
          const target = e.target.closest ? e.target.closest('.dtarot-modal') : null;
          if (target && target !== modal) {
            closeGenericModal(target);
            return;
          }
          closeModal();
        }
      });

      if (previewBtn) {
        previewBtn.addEventListener('click', function(e){
          e.preventDefault();
          runPreview();
        });
      }

      if (form) {
        form.addEventListener('change', function(e){
          const target = e.target;
          if (!target || !target.matches) return;
          if (target.matches('#dtarot_spread_preset, #dtarot_spread_pack, #dtarot_spread_deck')) {
            runPreview();
          }
        });
      }

      if (previewBox) {
        previewBox.addEventListener('click', function(e){
          const flip = e.target && e.target.closest ? e.target.closest('.dtarot-card-flip') : null;
          if (!flip) return;
          e.preventDefault();
          flip.classList.toggle('is-flipped');
        });
      }

      // Template pack editor
      const editor = page.querySelector('[data-dtarot-spread-editor="1"]');
      const packSelect = document.getElementById('dtarot_spread_edit_pack');
      const presetSelect = document.getElementById('dtarot_spread_edit_preset');
      const dataEl = document.getElementById('dtarot-spread-pack-data');
      let packData = {};
      if (dataEl && dataEl.textContent) {
        try { packData = JSON.parse(dataEl.textContent); } catch (e) { packData = {}; }
      }

      function refreshEditor(){
        if (!editor || !packSelect || !presetSelect) return;
        const packId = packSelect.value || '';
        const presetId = presetSelect.value || '';
        editor.querySelectorAll('.dtarot-spread-editor-preset').forEach(function(el){
          el.classList.toggle('is-active', el.getAttribute('data-preset') === presetId);
        });

        const presetBlock = editor.querySelector('.dtarot-spread-editor-preset[data-preset="' + presetId + '"]');
        editor.querySelectorAll('textarea').forEach(function(field){
          field.disabled = true;
        });
        if (!presetBlock) return;
        presetBlock.querySelectorAll('textarea').forEach(function(field){
          field.disabled = false;
        });
        const spreads = packData && packData[packId] && packData[packId].spreads ? packData[packId].spreads : {};
        const slots = spreads && spreads[presetId] && Array.isArray(spreads[presetId].slots) ? spreads[presetId].slots : [];
        const fields = presetBlock.querySelectorAll('textarea');
        fields.forEach(function(field, idx){
          const row = slots[idx] || {};
          field.value = (row.meaning || '');
        });
      }

      if (packSelect && presetSelect) {
        packSelect.addEventListener('change', refreshEditor);
        presetSelect.addEventListener('change', refreshEditor);
        refreshEditor();
      }
    })();

    // Settings page: fast tab switching (AJAX panel swap + History API)
    (function(){
      try {
        const wrap = document.querySelector('.dtarot-wrap[data-dtarot-settings-page="1"]');
        if (!wrap) return;

        function isSettingsHref(href){
          if (!href) return false;
          let u;
          try { u = new URL(href, window.location.href); } catch (e) { return false; }
          if (!u.searchParams) return false;
          return (u.searchParams.get('page') || '') === 'daily-tarot-settings';
        }

        function setActiveTab(tab){
          const links = wrap.querySelectorAll('.nav-tab-wrapper a.nav-tab');
          links.forEach(function(a){ a.classList.remove('nav-tab-active'); });
          links.forEach(function(a){
            try {
              const u = new URL(a.href, window.location.href);
              const t = (u.searchParams.get('tab') || '').toString();
              if (t === tab) a.classList.add('nav-tab-active');
            } catch (e) {}
          });
        }

        let inFlight = false;
        function setBusy(isBusy){
          const panel = document.getElementById('dtarot-settings-panel');
          if (!panel) return;
          panel.setAttribute('aria-busy', isBusy ? 'true' : 'false');
          if (isBusy) panel.classList.add('dtarot-loading');
          else panel.classList.remove('dtarot-loading');
        }

        function replacePanel(html){
          if (!html) return;
          const tmp = document.createElement('div');
          tmp.innerHTML = String(html);
          const next = tmp.querySelector('#dtarot-settings-panel');
          const cur = document.getElementById('dtarot-settings-panel');
          if (!next || !cur) return;
          cur.replaceWith(next);
          const tab = (next.getAttribute('data-tab') || '').toString();
          if (tab) setActiveTab(tab);
        }

        function loadHref(href, pushState){
          if (!href || inFlight) return;
          if (!isSettingsHref(href)) {
            window.location.href = href;
            return;
          }

          inFlight = true;
          setBusy(true);

          const req = postAjax({ action: 'dtarot_get_settings_panel', href: href }, function(res){
            inFlight = false;
            setBusy(false);

            if (!res || !res.success || !res.data || typeof res.data.html !== 'string') {
              window.location.href = href;
              return;
            }

            replacePanel(res.data.html);

            if (pushState !== false) {
              try { window.history.pushState({ dtarotSettings: 1 }, '', href); } catch (e) {}
            }
          });

          // If the request fails (e.g. 500/fatal error), fall back to a full navigation.
          if (req && typeof req.fail === 'function') {
            req.fail(function(){
              inFlight = false;
              setBusy(false);
              window.location.href = href;
            });
          }
        }

        // Only intercept *tab* clicks (avoid hijacking import/download/admin-post links).
        wrap.addEventListener('click', function(e){
          const a = e.target && e.target.closest ? e.target.closest('a.nav-tab') : null;
          if (!a) return;
          if (!a.closest('.nav-tab-wrapper')) return;
          if (a.target && a.target === '_blank') return;
          const href = (a.href || '').toString();
          if (!href) return;
          if (!isSettingsHref(href)) return;

          e.preventDefault();
          loadHref(href, true);
        }, true);

        window.addEventListener('popstate', function(){
          try {
            if (!isSettingsHref(window.location.href)) return;
            loadHref(window.location.href, false);
          } catch (e) {}
        });
      } catch (e) {}
    })();

    // Auto-save toggles (reusable)
    $(document).on('change', 'input[data-dtarot-autosave="1"]', function(){
      const inputEl = this;
      const setting = (inputEl.getAttribute('data-dtarot-setting') || '').toString();
      if (!setting) return;

      const newIsOn = !!inputEl.checked;
      const hiddenSelector = (inputEl.getAttribute('data-dtarot-hidden-target') || '').toString();
      if (hiddenSelector) {
        const hiddenEl = document.querySelector(hiddenSelector);
        if (hiddenEl) hiddenEl.value = newIsOn ? '1' : '0';
      }
      const prevIsOn = !newIsOn;

      // Apply impact immediately (optimistic).
      applyToggleEffect(inputEl, newIsOn);

      inputEl.disabled = true;
      setAutoSaveStatus(inputEl, t('savingEllipsis', 'Saving…'), null);

      postAjax({ action: 'dtarot_save_ui_setting', setting: setting, value: newIsOn ? '1' : '0' }, function(res){
        inputEl.disabled = false;

        if (!res || !res.success) {
          // Revert on failure.
          inputEl.checked = prevIsOn;
          applyToggleEffect(inputEl, prevIsOn);

          const msg = (res && res.data && res.data.message) ? res.data.message : t('saveFailed','Save failed');
          setAutoSaveStatus(inputEl, msg, false);
          return;
        }

        setAutoSaveStatus(inputEl, t('saved', 'Saved.'), true);
        window.setTimeout(function(){
          setAutoSaveStatus(inputEl, '', true);
        }, 1200);
      });
    });

    // Meaning Pack editor: changing system should refresh card list immediately.
    $(document).on('change', '#dtarot-pack-system', function(){
      try {
        const sys = (this.value || '').toString();
        if (!sys) return;

        const url = new URL(window.location.href);
        url.searchParams.set('dtarot_system_preview', sys);
        // Active card may not exist in the new system.
        url.searchParams.delete('card');
        window.location.href = url.toString();
      } catch (e) {}
    });

    // Settings → Shortcode: related links manager
    (function(){
      function el(id){ return document.getElementById(id); }
      const wrap = el('dtarot-related-links-manager');
      if (!wrap) return;

      const cardSelect = el('dtarot_related_links_card');
      const statusEl = el('dtarot_related_links_status');
      const sugEl = el('dtarot_related_links_suggestions');
      const searchInput = el('dtarot_related_links_search');
      const searchEl = el('dtarot_related_links_search_results');
      const urlInput = el('dtarot_related_links_url');

      function curCard(){ return cardSelect ? (cardSelect.value || '').toString() : ''; }

      function setStatus(html){
        if (!statusEl) return;
        statusEl.innerHTML = html || '';
      }

      function escapeHtml(s){
        return String(s)
          .replace(/&/g, '&amp;')
          .replace(/</g, '&lt;')
          .replace(/>/g, '&gt;')
          .replace(/"/g, '&quot;')
          .replace(/'/g, '&#039;');
      }

      function renderItems(container, items){
        if (!container) return;
        const list = Array.isArray(items) ? items : [];
        if (!list.length) {
          container.innerHTML = '<p class="description" style="margin:6px 0;">' + escapeHtml(t('noneFound','No matches found.')) + '</p>';
          return;
        }
        let html = '<ul style="margin:6px 0; padding-left:18px;">';
        list.forEach(function(it){
          const title = it && typeof it.title === 'string' ? it.title : '';
          const url = it && typeof it.url === 'string' ? it.url : '';
          const type = it && typeof it.type === 'string' ? it.type : '';
          const id = it && typeof it.id === 'number' ? it.id : (parseInt(it.id, 10) || 0);
          if (!url) return;
          html += '<li style="margin:6px 0;">'
            + '<a href="' + escapeHtml(url) + '" target="_blank" rel="noopener noreferrer">' + escapeHtml(title || url) + '</a>'
            + (type ? ' <span class="description">(' + escapeHtml(type) + (id ? (' #' + id) : '') + ')</span>' : '')
            + (id ? ' <button type="button" class="button button-small" data-dtarot-related-action="use" data-post-id="' + id + '">' + escapeHtml(t('use','Use')) + '</button>' : '')
            + '</li>';
        });
        html += '</ul>';
        container.innerHTML = html;
      }

      function refreshSelected(){
        const cid = curCard();
        if (!cid) return;
        setStatus('<p class="description">' + escapeHtml(t('loadingEllipsis','Loading…')) + '</p>');
        if (sugEl) sugEl.innerHTML = '';
        if (searchEl) searchEl.innerHTML = '';

        postAjax({ action: 'dtarot_related_link_get', card_id: cid }, function(res){
          if (!res || !res.success || !res.data) {
            setStatus('<p style="color:#b32d2e;">' + escapeHtml(t('loadFailed','Failed to load current link.')) + '</p>');
            return;
          }
          const sel = res.data.selected || {};
          const resolvedUrl = (res.data.resolvedUrl || '').toString();

          let html = '<div style="padding:10px; border:1px solid #ccd0d4; background:#fff;">';
          html += '<div><strong>' + escapeHtml(t('current','Current')) + ':</strong> ';
          if (sel && sel.url) {
            html += '<a href="' + escapeHtml(sel.url) + '" target="_blank" rel="noopener noreferrer">' + escapeHtml(sel.title || sel.url) + '</a>';
          } else {
            html += '<span class="description">' + escapeHtml(t('none','None')) + '</span>';
          }
          html += '</div>';
          html += '<div style="margin-top:6px;"><strong>' + escapeHtml(t('resolved','Resolved')) + ':</strong> ';
          if (resolvedUrl) {
            html += '<a href="' + escapeHtml(resolvedUrl) + '" target="_blank" rel="noopener noreferrer">' + escapeHtml(resolvedUrl) + '</a>';
          } else {
            html += '<span class="description">' + escapeHtml(t('none','None')) + '</span>';
          }
          html += '</div>';
          html += '</div>';
          setStatus(html);
        });
      }

      function doSuggest(){
        const cid = curCard();
        if (!cid) return;
        if (sugEl) sugEl.innerHTML = '<p class="description">' + escapeHtml(t('loadingEllipsis','Loading…')) + '</p>';
        postAjax({ action: 'dtarot_related_link_suggest', card_id: cid }, function(res){
          if (!res || !res.success || !res.data) {
            renderItems(sugEl, []);
            return;
          }
          renderItems(sugEl, res.data.items || []);
        });
      }

      function doSearch(){
        const q = searchInput ? (searchInput.value || '').toString().trim() : '';
        if (!q) {
          renderItems(searchEl, []);
          return;
        }
        if (searchEl) searchEl.innerHTML = '<p class="description">' + escapeHtml(t('loadingEllipsis','Loading…')) + '</p>';
        postAjax({ action: 'dtarot_related_link_search', q: q }, function(res){
          if (!res || !res.success || !res.data) {
            renderItems(searchEl, []);
            return;
          }
          renderItems(searchEl, res.data.items || []);
        });
      }

      function doSetPost(postId){
        const cid = curCard();
        if (!cid || !postId) return;
        postAjax({ action: 'dtarot_related_link_set', card_id: cid, post_id: String(postId) }, function(){
          refreshSelected();
        });
      }

      function doSetUrl(){
        const cid = curCard();
        const url = urlInput ? (urlInput.value || '').toString().trim() : '';
        if (!cid || !url) return;
        postAjax({ action: 'dtarot_related_link_set', card_id: cid, url: url }, function(){
          refreshSelected();
        });
      }

      function doClear(){
        const cid = curCard();
        if (!cid) return;
        postAjax({ action: 'dtarot_related_link_clear', card_id: cid }, function(){
          refreshSelected();
        });
      }

      if (cardSelect) {
        cardSelect.addEventListener('change', function(){
          refreshSelected();
        });
      }

      // Action buttons + "Use" buttons in lists.
      wrap.addEventListener('click', function(e){
        const btn = e.target && e.target.closest ? e.target.closest('button[data-dtarot-related-action]') : null;
        if (!btn) return;
        const act = (btn.getAttribute('data-dtarot-related-action') || '').toString();
        if (!act) return;
        e.preventDefault();

        if (act === 'suggest') return doSuggest();
        if (act === 'search') return doSearch();
        if (act === 'set-url') return doSetUrl();
        if (act === 'clear') return doClear();
        if (act === 'use') {
          const pid = parseInt(btn.getAttribute('data-post-id') || '0', 10) || 0;
          return doSetPost(pid);
        }
      }, true);

      if (searchInput) {
        searchInput.addEventListener('keydown', function(e){
          if (e.key === 'Enter') {
            e.preventDefault();
            doSearch();
          }
        });
      }

      refreshSelected();
    })();

    // Meaning Pack editor: top toolbar (back + switch pack) with unsaved-changes prompt.
    (function(){
      const topbar = document.querySelector('[data-dtarot-meaning-pack-topbar="1"]');
      if (!topbar) return;

      let dirty = false;
      function markDirty(){ dirty = true; }

      // Mark dirty on name/version changes.
      $(document).on('input change', '#dtarot-pack-name, #dtarot-pack-version', function(){
        markDirty();
      });

      // Mark dirty on optional fields and editor inputs.
      document.addEventListener('input', function(e){
        const t = e.target;
        if (!t || !(t instanceof HTMLElement)) return;
        if (!t.closest('.dtarot-meaning-right')) return;
        if (t.matches('textarea, input, select')) markDirty();
      }, true);

      // TinyMCE (Visual mode) changes.
      function bindTinyMce(){
        try {
          if (!window.tinymce || !window.tinymce.editors) return;
          window.tinymce.editors.forEach(function(ed){
            if (!ed || !ed.on) return;
            const id = (ed.id || '').toString();
            if (id !== 'dtarot_upright_editor' && id !== 'dtarot_reversed_editor') return;
            ed.on('change keyup', function(){ markDirty(); });
          });
        } catch (e) {}
      }
      bindTinyMce();
      setTimeout(bindTinyMce, 700);

      function submitAndRedirect(url){
        try {
          if (window.tinymce && window.tinymce.triggerSave) window.tinymce.triggerSave();
        } catch (e) {}

        const form = document.getElementById('post');
        if (!form) {
          window.location.href = url;
          return;
        }

        let hidden = form.querySelector('input[name="dtarot_after_save"]');
        if (!hidden) {
          hidden = document.createElement('input');
          hidden.type = 'hidden';
          hidden.name = 'dtarot_after_save';
          form.appendChild(hidden);
        }
        hidden.value = url;

        form.submit();
      }

      function navigateWithPrompt(url){
        if (!url) return;
        if (!dirty) {
          window.location.href = url;
          return;
        }

        const saveFirst = window.confirm('You have unsaved changes. Save before switching?');
        if (saveFirst) {
          submitAndRedirect(url);
          return;
        }

        const discard = window.confirm('Discard changes and switch?');
        if (discard) {
          window.location.href = url;
        }
      }

      const back = document.getElementById('dtarot-pack-back');
      if (back) {
        back.addEventListener('click', function(e){
          const href = (back.getAttribute('href') || '').toString();
          if (!href) return;
          if (!dirty) return;
          e.preventDefault();
          navigateWithPrompt(href);
        });
      }

      const sel = document.getElementById('dtarot-pack-switch');
      if (sel) {
        sel.addEventListener('change', function(){
          const url = (sel.value || '').toString();
          if (!url) return;
          sel.value = '';
          navigateWithPrompt(url);
        });
      }
    })();

    // AI page
    function aiStatus(msg, ok){
      const el = document.getElementById('dtarot_ai_status');
      if (!el) return;
      el.textContent = msg || '';
      el.style.color = ok === true ? '' : (ok === false ? '#b32d2e' : '');
    }

    function aiUpdateCalendarLink(date){
      const a = document.getElementById('dtarot_ai_open_calendar');
      if (!a) return;
      try {
        const url = new URL(a.getAttribute('href') || '', window.location.origin);
        if (date) url.searchParams.set('dtarot_edit_date', date);
        a.setAttribute('href', url.toString());
      } catch (e) {}
    }

    $(document).on('change', '#dtarot_ai_date', function(){
      aiUpdateCalendarLink((this.value || '').toString());
    });
    aiUpdateCalendarLink(($('#dtarot_ai_date').val() || '').toString());

    // AI Prefill: auto-submit enable toggle (admin-post)
    $(document).on('change', '#dtarot_ai_prefill_enabled[data-dtarot-submit-on-change="1"]', function(){
      try {
        const form = this.closest ? this.closest('form') : null;
        if (!form) return;
        if (form.getAttribute('data-dtarot-ai-prefill-form') !== '1') return;
        form.submit();
      } catch (e) {}
    });

    $(document).on('click', '[data-dtarot-ai] [data-dtarot-ai-action]', function(e){
      e.preventDefault();

      const btn = this;
      const action = (btn.getAttribute('data-dtarot-ai-action') || '').toString();
      const kind = (btn.getAttribute('data-dtarot-ai-kind') || 'both').toString();
      const date = ($('#dtarot_ai_date').val() || '').toString();
      const introLen = ($('#dtarot_ai_intro_len').val() || '').toString();
      let dailyLen = ($('#dtarot_ai_daily_len').val() || '').toString();

      if (!dailyLen) {
        const n = parseInt(introLen, 10);
        if (!isNaN(n) && n > 0) dailyLen = String(n * 2);
      }

      if (!date) {
        aiStatus(t('pickDateFirst', 'Pick a date first.'), false);
        return;
      }

      if (action === 'save') {
        btn.disabled = true;
        aiStatus(t('savingEllipsis', 'Saving…'), null);

        postAjax({
          action: 'dtarot_ai_save',
          date: date,
          content: ($('#dtarot_ai_content').val() || '').toString(),
          daily_text: ($('#dtarot_ai_daily_text').val() || '').toString()
        }, function(res){
          btn.disabled = false;
          if (!res || !res.success) {
            const msg = (res && res.data && res.data.message) ? res.data.message : 'Save failed.';
            aiStatus(msg, false);
            return;
          }
          aiStatus(t('saved', 'Saved.'), true);
          window.setTimeout(function(){ aiStatus('', true); }, 1400);
        });
        return;
      }

      const save = (action === 'generate_save');

      btn.disabled = true;
      aiStatus(t('generatingEllipsis', 'Generating…'), null);

      postAjax({ action: 'dtarot_ai_generate', date: date, kind: kind, save: save ? '1' : '0', intro_len: introLen, daily_len: dailyLen }, function(res){
        btn.disabled = false;

        if (!res || !res.success) {
          const code = (res && res.data && res.data.message) ? res.data.message : '';
          if (code === 'ai_disabled') {
            aiStatus(t('aiDisabled', 'AI is disabled.'), false);
          } else if (code === 'insufficient_credits') {
            const bal = (res && res.data && typeof res.data.balance !== 'undefined') ? res.data.balance : null;
            const req = (res && res.data && typeof res.data.required !== 'undefined') ? res.data.required : null;
            if (bal !== null && req !== null) {
              aiStatus('Not enough credits (have ' + String(bal) + ', need ' + String(req) + ').', false);
            } else {
              aiStatus(t('aiNotEnoughCredits', 'Not enough credits to generate.'), false);
            }
          } else if (code === 'missing_deck_card') {
            aiStatus(t('aiNeedDeckCard', 'Select a deck + card for this date first (Calendar).'), false);
          } else if (code === 'no_result') {
            aiStatus(t('aiNoGenerator', 'No AI generator is configured yet.'), false);
          } else {
            aiStatus(t('aiGenerationFailed', 'Generation failed.'), false);
          }
          return;
        }

        const data = res.data || {};
        if (typeof data.content === 'string' && data.content) {
          $('#dtarot_ai_content').val(data.content);
        }
        if (typeof data.daily_text === 'string' && data.daily_text) {
          $('#dtarot_ai_daily_text').val(data.daily_text);
        }

        if (typeof data.credits_left !== 'undefined') {
          $('#dtarot_ai_credits_balance').text(String(data.credits_left));
        }

        const leftMsg = (typeof data.credits_left !== 'undefined') ? (' Credits left: ' + String(data.credits_left) + '.') : '';
        aiStatus((data.saved ? t('aiGeneratedSaved', 'Generated + saved.') : t('aiGenerated', 'Generated.')) + leftMsg, true);
        window.setTimeout(function(){ aiStatus('', true); }, 1400);
      });
    });

    // AI credits: manual add
    $(document).on('click', '#dtarot_ai_credits_add_btn', function(e){
      e.preventDefault();

      const $amount = $('#dtarot_ai_credits_add_amount');
      const raw = ($amount.val() || '').toString();
      const amount = parseInt(raw, 10);
      if (!amount || amount <= 0) {
        aiStatus(t('aiEnterCreditAmount', 'Enter a valid credit amount.'), false);
        return;
      }

      const btn = this;
      btn.disabled = true;
      aiStatus(t('aiAddingCredits', 'Adding credits…'), null);

      postAjax({ action: 'dtarot_ai_add_credits', amount: String(amount), reason: 'manual' }, function(res){
        btn.disabled = false;
        if (!res || !res.success) {
          aiStatus(t('aiFailedAddCredits', 'Failed to add credits.'), false);
          return;
        }

        if (res.data && typeof res.data.balance !== 'undefined') {
          $('#dtarot_ai_credits_balance').text(String(res.data.balance));
        }
        aiStatus(t('aiCreditsAdded', 'Credits added.'), true);
        window.setTimeout(function(){ aiStatus('', true); }, 1200);
      });
    });

    // AI Provider test (Settings -> Pro)
    $(document).on('click', '#dtarot_ai_provider_test_btn', function(e){
      e.preventDefault();

      const btn = this;
      const statusEl = document.getElementById('dtarot_ai_provider_test_status');
      const outWrap = document.getElementById('dtarot_ai_provider_test_output');
      const contentEl = document.getElementById('dtarot_ai_provider_test_content');
      const dailyEl = document.getElementById('dtarot_ai_provider_test_daily_text');
      const bodyEl = document.getElementById('dtarot_ai_provider_test_body');

      const date = ($('#dtarot_ai_provider_test_date').val() || '').toString();

      btn.disabled = true;
      if (statusEl) statusEl.textContent = t('loadingEllipsis', 'Loading…');
      if (outWrap) outWrap.style.display = 'none';

      postAjax({ action: 'dtarot_ai_provider_test', date: date }, function(res){
        btn.disabled = false;

        if (!res || !res.success) {
          const msg = (res && res.data && res.data.message) ? String(res.data.message) : 'Request failed.';
          if (statusEl) statusEl.textContent = msg;
          if (contentEl) contentEl.value = '';
          if (dailyEl) dailyEl.value = '';
          if (bodyEl) bodyEl.textContent = '';
          return;
        }

        const data = res.data || {};
        const httpCode = (typeof data.http_code !== 'undefined') ? String(data.http_code) : '';
        const parsedOk = !!data.parsed_ok;
        if (statusEl) statusEl.textContent = parsedOk ? ('OK (HTTP ' + httpCode + ')') : ('No parsed result (HTTP ' + httpCode + ')');

        if (contentEl) contentEl.value = data.content ? String(data.content) : '';
        if (dailyEl) dailyEl.value = data.daily_text ? String(data.daily_text) : '';
        if (bodyEl) {
          const snippet = data.body_snippet ? String(data.body_snippet) : '';
          bodyEl.textContent = snippet + (data.body_truncated ? "\n\n[truncated]" : '');
        }

        if (outWrap) outWrap.style.display = '';
      });
    });

    // Provider secret generator (Settings -> Pro)
    $(document).on('click', '#dtarot_ai_provider_secret_generate', function(e){
      e.preventDefault();
      const input = document.getElementById('dtarot_ai_provider_secret');
      if (!input || input.disabled) return;

      try {
        const bytes = new Uint8Array(32); // 256-bit secret
        if (window.crypto && typeof window.crypto.getRandomValues === 'function') {
          window.crypto.getRandomValues(bytes);
        } else {
          // Weak fallback (should rarely happen in WP admin).
          for (let i = 0; i < bytes.length; i++) bytes[i] = Math.floor(Math.random() * 256);
        }

        let hex = '';
        for (let i = 0; i < bytes.length; i++) {
          hex += ('0' + bytes[i].toString(16)).slice(-2);
        }

        input.value = hex;
        input.focus();
      } catch (err) {
        try { alert('Could not generate a secret in this browser.'); } catch (e2) {}
      }
    });

    // Help modals (Settings -> Pro)
    $(document).on('click', '.dtarot-help-btn', function(e){
      e.preventDefault();
      const sel = (this.getAttribute('data-dtarot-modal') || '').toString();
      if (!sel) return;
      const $m = $(sel);
      if ($m.length) $m.show();
    });

    $(document).on('keydown', function(e){
      // Escape closes any open dtarot modals.
      if (e && (e.key === 'Escape' || e.key === 'Esc')) {
        $('.dtarot-modal:visible').hide();
      }
    });
  });

  // --- Deck Back image (deck edit screen) ---
  $(document).on('click', '#dtarot-back-pick', function(e){
    e.preventDefault();
    if (typeof wp === 'undefined' || !wp.media) return;

    const frame = wp.media({ title: t('mediaBackTitle', 'Select card back'), button:{ text: t('mediaUseImage', 'Use this image') }, multiple:false });
    frame.on('select', function(){
      const a = frame.state().get('selection').first().toJSON();
      if (a && a.url) $('#dtarot-back-url').val(a.url);
    });
    frame.open();
  });

  // --- Content -> Cards tab: per-row media pick & save ---
  function pickMedia(onSelect){
    if (typeof wp === 'undefined' || !wp.media) { alert(t('mediaMissing','Media library not available')); return; }
    const frame = wp.media({ title: t('mediaCardTitle','Select card image'), button:{ text: t('mediaUseImage','Use this image') }, multiple:false });
    frame.on('select', function(){
      const a = frame.state().get('selection').first().toJSON();
      onSelect({
        url: (a && a.url) ? String(a.url) : '',
        id: (a && a.id) ? String(a.id) : '',
        filename: (a && a.filename) ? String(a.filename) : ''
      });
    });
    frame.open();
  }

  function normalizeCardIdFromFilename(filename){
    if (!filename) return '';
    let base = String(filename);
    base = base.replace(/\?.*$/, '');
    base = base.replace(/^.*\//, '');
    base = base.replace(/\.[^.]+$/, '');
    base = base.toLowerCase();
    base = base.replace(/\s+/g, '_');
    base = base.replace(/[^a-z0-9_-]/g, '');
    return base;
  }

  function setCardRowUrl(deckId, cardId, url, attachmentId){
    const $input = $('.dtarot-image-url[data-deck-id="'+deckId+'"][data-card-id="'+cardId+'"]');
    if (!$input.length) return false;

    $input.val(url || '');
    if (attachmentId) {
      try { $input.attr('data-attachment-id', String(attachmentId)); } catch (e2) {}
    }
    // trigger preview update
    $input.trigger('input');

    const $msg = $input.closest('td').find('.dtarot-row-msg');
    $msg.removeClass('dtarot-msg-ok dtarot-msg-err').text(t('selectedClickSave','Selected. Click Save.'));
    return true;
  }

  function getCardOptionsFromTable(){
    const rows = document.querySelectorAll('tr.dtarot-card-row');
    const options = [];
    const ids = new Set();

    rows.forEach(function(row){
      const id = (row.getAttribute('data-card-id') || '').toString();
      if (!id) return;
      const strong = row.querySelector('td strong');
      const name = strong ? (strong.textContent || '').toString().trim() : id;
      ids.add(id);
      // Prefix label with the numeric suffix when present (keeps ordering clear).
      let label = name + ' (' + id + ')';
      const m = id.match(/_(\d+)$/);
      if (m && m[1]) {
        const n = parseInt(m[1], 10);
        if (!isNaN(n)) {
          const num = (n < 10 ? ('0' + String(n)) : String(n));
          label = num + ' — ' + name + ' (' + id + ')';
        }
      }
      // Preserve the current table order (which is already numeric per system).
      options.push({ id: id, label: label });
    });

    return { options: options, ids: ids };
  }

  function updateBulkMapUniqueness(){
    const selects = Array.prototype.slice.call(document.querySelectorAll('#dtarot-bulk-map-list select.dtarot-bulk-card-select'));
    if (!selects.length) return;

    // First pass: clear duplicate selections (keep the first occurrence).
    const used = new Set();
    selects.forEach(function(sel){
      const val = (sel.value || '').toString();
      if (!val) return;
      if (used.has(val)) {
        sel.value = '';
      } else {
        used.add(val);
      }
    });

    // Second pass: disable/hide already-used options in other selects.
    const selected = new Set();
    selects.forEach(function(sel){
      const val = (sel.value || '').toString();
      if (val) selected.add(val);
    });

    selects.forEach(function(sel){
      const current = (sel.value || '').toString();
      const opts = sel.querySelectorAll('option');
      opts.forEach(function(opt){
        const v = (opt.value || '').toString();
        if (!v) {
          opt.disabled = false;
          opt.hidden = false;
          return;
        }

        const takenElsewhere = selected.has(v) && v !== current;
        opt.disabled = takenElsewhere;
        opt.hidden = takenElsewhere;
      });
    });
  }

  function openBulkMapModal(deckId, items){
    const modalEl = document.getElementById('dtarot-bulk-map-modal');
    const listEl = document.getElementById('dtarot-bulk-map-list');
    const statusEl = document.getElementById('dtarot-bulk-map-status');
    const saveBtn = document.getElementById('dtarot-bulk-map-save');

    if (!modalEl || !listEl || !saveBtn) {
      alert('Bulk mapping UI is missing on this page.');
      return;
    }

    const cards = getCardOptionsFromTable();
    const options = cards.options;
    const ids = cards.ids;

    let optionHtml = '<option value="">' + t('selectCard','— Select card —') + '</option>';
    options.forEach(function(o){
      optionHtml += '<option value="' + String(o.id).replace(/"/g,'&quot;') + '">' + String(o.label) + '</option>';
    });

    listEl.innerHTML = '';
    if (statusEl) {
      statusEl.className = 'description dtarot-row-msg';
      statusEl.textContent = '';
    }

    items.forEach(function(it, idx){
      const url = (it && it.url) ? String(it.url) : '';
      const attachmentId = (it && typeof it.id !== 'undefined' && it.id !== null) ? String(it.id) : '';
      const filename = (it && it.filename) ? String(it.filename) : (url || ('image_' + String(idx+1)));
      const guess = normalizeCardIdFromFilename(filename);
      const preselect = (guess && ids.has(guess)) ? guess : '';

      const wrap = document.createElement('div');
      wrap.className = 'dtarot-bulk-map-item';
      wrap.setAttribute('data-url', url);

      wrap.innerHTML =
        '<div>' +
          '<div class="dtarot-card-preview">' +
            (url ? ('<img class="dtarot-card-preview-img" src="' + url.replace(/"/g,'&quot;') + '" alt="" />') : '<div class="dtarot-card-preview-empty">—</div>') +
          '</div>' +
        '</div>' +
        '<div class="dtarot-bulk-map-meta">' +
          '<select class="dtarot-bulk-card-select" data-url="' + url.replace(/"/g,'&quot;') + '" data-attachment-id="' + String(attachmentId).replace(/"/g,'&quot;') + '">' + optionHtml + '</select>' +
          '<div class="description dtarot-bulk-map-filename">' + String(filename) + '</div>' +
        '</div>';

      listEl.appendChild(wrap);

      const sel = wrap.querySelector('select.dtarot-bulk-card-select');
      if (sel && preselect) sel.value = preselect;
    });

    // Enforce uniqueness right away (handles preselect guesses).
    updateBulkMapUniqueness();

    modalEl.setAttribute('data-deck-id', String(deckId || ''));
    modalEl.style.display = 'flex';
  }

  // Bulk map modal: prevent selecting the same card twice.
  $(document).on('change', '#dtarot-bulk-map-list select.dtarot-bulk-card-select', function(){
    updateBulkMapUniqueness();
  });

  // Bulk map modal: click image to zoom.
  $(document).on('click', '.dtarot-bulk-map-item img.dtarot-card-preview-img', function(e){
    e.preventDefault();
    const src = (this.getAttribute('src') || '').toString();
    if (!src) return;

    const imgEl = document.getElementById('dtarot-bulk-zoom-img');
    const modalEl = document.getElementById('dtarot-bulk-zoom-modal');
    if (!imgEl || !modalEl) {
      try { window.open(src, '_blank', 'noopener'); } catch (e2) {}
      return;
    }

    imgEl.setAttribute('src', src);
    modalEl.style.display = 'flex';
  });

  // Bulk map modal: hover shows a 2x preview (no new window).
  let bulkHoverEl = null;
  let bulkHoverImg = null;

  function ensureBulkHoverZoom(){
    if (bulkHoverEl && bulkHoverImg) return;
    bulkHoverEl = document.getElementById('dtarot-bulk-hover-zoom');
    if (!bulkHoverEl) {
      bulkHoverEl = document.createElement('div');
      bulkHoverEl.id = 'dtarot-bulk-hover-zoom';
      bulkHoverEl.innerHTML = '<img alt="" />';
      document.body.appendChild(bulkHoverEl);
    }
    bulkHoverImg = bulkHoverEl.querySelector('img');
  }

  function positionBulkHoverZoom(clientX, clientY){
    if (!bulkHoverEl) return;
    const pad = 14;
    const vw = window.innerWidth || 0;
    const vh = window.innerHeight || 0;
    const rect = bulkHoverEl.getBoundingClientRect();
    const w = rect.width || 160;
    const h = rect.height || 210;

    let x = clientX + pad;
    let y = clientY + pad;
    if (x + w + pad > vw) x = Math.max(pad, clientX - w - pad);
    if (y + h + pad > vh) y = Math.max(pad, clientY - h - pad);

    bulkHoverEl.style.left = String(x) + 'px';
    bulkHoverEl.style.top = String(y) + 'px';
  }

  function hideBulkHoverZoom(){
    if (!bulkHoverEl) return;
    bulkHoverEl.style.display = 'none';
  }

  $(document).on('mouseenter', '.dtarot-bulk-map-item img.dtarot-card-preview-img', function(e){
    const src = (this.getAttribute('src') || '').toString();
    if (!src) return;
    ensureBulkHoverZoom();
    if (!bulkHoverEl || !bulkHoverImg) return;

    bulkHoverImg.setAttribute('src', src);
    bulkHoverEl.style.display = 'block';
    if (e && typeof e.clientX === 'number' && typeof e.clientY === 'number') {
      positionBulkHoverZoom(e.clientX, e.clientY);
    }
  });

  $(document).on('mousemove', '.dtarot-bulk-map-item img.dtarot-card-preview-img', function(e){
    if (!bulkHoverEl || bulkHoverEl.style.display === 'none') return;
    if (e && typeof e.clientX === 'number' && typeof e.clientY === 'number') {
      positionBulkHoverZoom(e.clientX, e.clientY);
    }
  });

  $(document).on('mouseleave', '.dtarot-bulk-map-item img.dtarot-card-preview-img', function(){
    hideBulkHoverZoom();
  });

  // Hide on scroll (prevents the preview floating in odd places).
  $(document).on('scroll', function(){
    hideBulkHoverZoom();
  });

  // Bulk upload/select card images and auto-map by filename.
  $(document).on('click', '#dtarot-bulk-upload', function(e){
    e.preventDefault();
    if (typeof wp === 'undefined' || !wp.media) { alert(t('mediaMissing','Media library not available')); return; }

    const deckId = (this.getAttribute('data-deck-id') || '').toString();
    const $bulkMsg = $('#dtarot-bulk-msg');
    $bulkMsg.removeClass('dtarot-msg-ok dtarot-msg-err').text('');

    const frame = wp.media({
      title: t('mediaBulkTitle', 'Select card images'),
      button: { text: t('mediaBulkUse', 'Use these images') },
      multiple: true
    });

    frame.on('select', function(){
      const selection = frame.state().get('selection');
      const items = selection ? selection.toJSON() : [];
      if (!items || !items.length) return;

      openBulkMapModal(deckId, items);
      $bulkMsg.removeClass('dtarot-msg-err').addClass('dtarot-msg-ok').text('Selected ' + items.length + ' images. Map and save in the window.');
    });

    frame.open();
  });

  // Bulk map modal: cancel
  $(document).on('click', '#dtarot-bulk-map-cancel', function(e){
    e.preventDefault();
    $('#dtarot-bulk-map-modal').hide();
  });

  // Bulk map modal: save
  $(document).on('click', '#dtarot-bulk-map-save', function(e){
    e.preventDefault();

    const $modal = $('#dtarot-bulk-map-modal');
    const deckId = ($modal.attr('data-deck-id') || '').toString();
    const $status = $('#dtarot-bulk-map-status');
    const $saveBtn = $('#dtarot-bulk-map-save');

    $status.removeClass('dtarot-msg-ok dtarot-msg-err').text('');

    const selects = document.querySelectorAll('#dtarot-bulk-map-list select.dtarot-bulk-card-select');
    const mappings = [];
    selects.forEach(function(sel){
      const cardId = (sel.value || '').toString();
      const url = (sel.getAttribute('data-url') || '').toString();
      const attachmentId = (sel.getAttribute('data-attachment-id') || '').toString();
      if (!cardId || !url) return;
      mappings.push({ card_id: cardId, url: url, attachment_id: attachmentId });
    });

    if (!mappings.length) {
      $status.addClass('dtarot-msg-err').text(t('bulkPickAtLeastOne','Pick at least one card to save.'));
      return;
    }

    // Optimistically set inputs/previews so the table reflects choices immediately.
    mappings.forEach(function(m){
      setCardRowUrl(deckId, m.card_id, m.url, m.attachment_id);
    });

    let i = 0;
    const total = mappings.length;
    $saveBtn.prop('disabled', true);

    function step(){
      if (i >= total) {
        $saveBtn.prop('disabled', false);
        $status.removeClass('dtarot-msg-err').addClass('dtarot-msg-ok').text('Saved ' + total + ' images.');

        // Reflect success in the main page toolbar message.
        try {
          $('#dtarot-bulk-msg').removeClass('dtarot-msg-err').addClass('dtarot-msg-ok').text('Saved ' + total + ' images.');
        } catch (e2) {}

        // Close the mapping window after completion.
        window.setTimeout(function(){
          try {
            $('#dtarot-bulk-zoom-modal').hide();
            $('#dtarot-bulk-map-modal').hide();
          } catch (e2) {}
        }, 500);
        return;
      }

      const m = mappings[i];
      i++;
      $status.removeClass('dtarot-msg-ok dtarot-msg-err').text('Saving ' + i + '/' + total + '…');
      postAjax({ action:'dtarot_save_card_image', deck_id:deckId, card_id:m.card_id, url:m.url, attachment_id: m.attachment_id || '' }, function(res){
        if (!res || !res.success) {
          $saveBtn.prop('disabled', false);
          $status.addClass('dtarot-msg-err').text('Bulk save failed on ' + m.card_id);
          return;
        }

        // If the server converted to WebP, update the row URL/preview.
        try {
          if (res && res.data && typeof res.data.url === 'string' && res.data.url) {
            setCardRowUrl(deckId, m.card_id, String(res.data.url), m.attachment_id);
          }
        } catch (e2) {}

        step();
      });
    }

    step();
  });

  $(document).on('click', '.dtarot-pick', function(){
    const deckId = this.getAttribute('data-deck-id');
    const cardId = this.getAttribute('data-card-id');
    const $input = $('.dtarot-image-url[data-deck-id="'+deckId+'"][data-card-id="'+cardId+'"]');
    const $msg = $input.closest('td').find('.dtarot-row-msg');
    pickMedia(function(sel){
      const url = sel && sel.url ? String(sel.url) : '';
      const attachmentId = sel && sel.id ? String(sel.id) : '';

      $input.val(url);
      if (attachmentId) {
        try { $input.attr('data-attachment-id', attachmentId); } catch (e2) {}
      }
      // Update preview in the same row.
      const $row = $input.closest('tr');
      const $prev = $row.find('img.dtarot-card-preview-img');
      const $empty = $row.find('.dtarot-card-preview-empty');
      if (url) {
        if ($prev.length) {
          $prev.attr('src', url);
        } else {
          $row.find('.dtarot-card-preview').html('<img class="dtarot-card-preview-img" src="'+url+'" alt="" />');
        }
      } else {
        if ($prev.length) $prev.remove();
        if ($empty.length) $empty.text('—');
      }

      $msg.removeClass('dtarot-msg-ok dtarot-msg-err').text(t('selectedClickSave','Selected. Click Save.'));
    });
  });

  // Live update preview when URL is pasted.
  $(document).on('input', '.dtarot-image-url', function(){
    const url = (this.value || '').trim();
    const $row = $(this).closest('tr');
    const $prev = $row.find('img.dtarot-card-preview-img');
    const $empty = $row.find('.dtarot-card-preview-empty');

    if (!url) {
      if ($prev.length) $prev.remove();
      if ($empty.length) $empty.text('—');
      return;
    }

    if ($prev.length) {
      $prev.attr('src', url);
    } else {
      $row.find('.dtarot-card-preview').html('<img class="dtarot-card-preview-img" src="'+url+'" alt="" />');
    }
  });

  $(document).on('click', '.dtarot-save', function(){
    const deckId = this.getAttribute('data-deck-id');
    const cardId = this.getAttribute('data-card-id');
    const $input = $('.dtarot-image-url[data-deck-id="'+deckId+'"][data-card-id="'+cardId+'"]');
    const $msg = $input.closest('td').find('.dtarot-row-msg');
    const url = ($input.val() || '').trim();
    const attachmentId = ($input.attr('data-attachment-id') || '').toString();

    const $row = $input.closest('tr');
    const $btnSave = $row.find('.dtarot-save');
    const $btnPick = $row.find('.dtarot-pick');
    $btnSave.prop('disabled', true);
    $btnPick.prop('disabled', true);

    $msg.removeClass('dtarot-msg-ok dtarot-msg-err').text(t('savingEllipsis','Saving…'));
    postAjax({ action:'dtarot_save_card_image', deck_id:deckId, card_id:cardId, url:url, attachment_id: attachmentId }, function(res){
      $btnSave.prop('disabled', false);
      $btnPick.prop('disabled', false);
      if (!res || !res.success) {
        $msg.addClass('dtarot-msg-err').text((res && res.data && res.data.message) ? res.data.message : t('saveFailed','Save failed'));
        return;
      }

      try {
        if (res && res.data && typeof res.data.url === 'string' && res.data.url) {
          $input.val(String(res.data.url));
          $input.trigger('input');
        }
      } catch (e2) {}

      $msg.addClass('dtarot-msg-ok').text(t('saved','Saved.'));
    });
  });

  // Cards table search filter
  $(document).on('input', '#dtarot-card-search', function(){
    const q = (this.value || '').trim().toLowerCase();
    const rows = document.querySelectorAll('tr.dtarot-card-row');
    rows.forEach(function(row){
      const name = (row.getAttribute('data-card-name') || '');
      const id = (row.getAttribute('data-card-id') || '');
      const show = (q === '') || name.includes(q) || id.includes(q);
      row.style.display = show ? '' : 'none';
    });
  });

  // Meaning pack search filter (meaning pack edit screen)
  $(document).on('input', '#dtarot-meaning-search', function(){
    const q = (this.value || '').trim().toLowerCase();
    const rows = document.querySelectorAll('tr.dtarot-meaning-row');
    rows.forEach(function(row){
      const name = (row.getAttribute('data-card-name') || '');
      const id = (row.getAttribute('data-card-id') || '');
      const show = (q === '') || name.includes(q) || id.includes(q);
      row.style.display = show ? '' : 'none';
    });
  });

  // Meaning pack: optional fields open-on-click (no popups).
  $(document).on('click', '.dtarot-inline-toggle', function(e){
    e.preventDefault();
    const btn = this;
    const wrap = btn.closest('.dtarot-inline-field');
    if (!wrap) return;
    const body = wrap.querySelector('.dtarot-inline-body');
    if (!body) return;

    const isOpen = btn.getAttribute('aria-expanded') === 'true';
    if (isOpen) {
      btn.setAttribute('aria-expanded', 'false');
      body.hidden = true;
      return;
    }

    btn.setAttribute('aria-expanded', 'true');
    body.hidden = false;

    // Focus the first input.
    try {
      const input = body.querySelector('input, textarea, select');
      if (input && input.focus) input.focus();
    } catch (e2) {}
  });

  // Meaning pack: switch cards without navigation.
  function dtarotStripPreview(s) {
    if (!s) return '';
    try {
      const tmp = document.createElement('div');
      tmp.innerHTML = String(s);
      const t = (tmp.textContent || tmp.innerText || '').trim();
      return t;
    } catch (e) {
      return String(s).trim();
    }
  }

  function dtarotSetWpEditorContent(editorId, html) {
    try {
      if (window.tinymce && tinymce.get(editorId)) {
        tinymce.get(editorId).setContent(html || '');
      }
    } catch (e) {}

    const textarea = document.getElementById(editorId);
    if (textarea) textarea.value = html || '';
  }

  function dtarotSelectMeaningCard(cardId, data) {
    if (!data || !data.cards || !data.meanings) return;
    if (!data.cards[cardId]) return;

    const meaning = data.meanings[cardId] || {};

    // Highlight row
    $('.dtarot-meaning-row').removeClass('dtarot-meaning-active');
    $('.dtarot-meaning-row[data-card-id="' + String(cardId).toLowerCase() + '"]').addClass('dtarot-meaning-active');

    // Update heading
    const titleEl = document.querySelector('[data-dtarot-card-title]');
    if (titleEl) titleEl.textContent = data.cards[cardId];

    // Update hidden card id
    const cardInput = document.querySelector('input[data-dtarot-card-id]');
    if (cardInput) cardInput.value = cardId;

    // Lenormand meta swap
    const slot = document.querySelector('[data-dtarot-lenormand-slot]');
    if (slot) {
      const tpl = document.querySelector('[data-dtarot-lenormand-template="' + cardId.replace(/"/g, '&quot;') + '"]');
      slot.innerHTML = tpl ? tpl.innerHTML : '';
    }

    // Upright/Reversed editors
    dtarotSetWpEditorContent('dtarot_upright_editor', meaning.upright || '');
    dtarotSetWpEditorContent('dtarot_reversed_editor', meaning.reversed || '');

    // Optional fields
    const keys = Array.isArray(data.optionalKeys) ? data.optionalKeys : ['keywords','short','long','symbols','correspondences'];
    keys.forEach(function(k){
      const name = 'dtarot_' + k;
      const el = document.querySelector('[name="' + name + '"]');
      if (el) el.value = meaning[k] || '';

      // Update preview text for the toggle
      const wrap = el ? el.closest('.dtarot-inline-field') : null;
      const previewEl = wrap ? wrap.querySelector('.dtarot-inline-preview') : null;
      if (previewEl) {
        let p = dtarotStripPreview(meaning[k] || '');
        if (p.length > 70) p = p.substring(0, 67) + '…';
        previewEl.textContent = p !== '' ? p : '(click to add)';
      }

      // Collapse bodies when switching
      if (wrap) {
        const btn = wrap.querySelector('.dtarot-inline-toggle');
        const body = wrap.querySelector('.dtarot-inline-body');
        if (btn && body) {
          btn.setAttribute('aria-expanded', 'false');
          body.hidden = true;
        }
      }
    });

    // Remember selection
    try {
      if (data.postId) {
        localStorage.setItem('dtarot_meaning_pack_last_' + String(data.postId), String(cardId));
      }
    } catch (e) {}
  }

  $(function(){
    const jsonEl = document.getElementById('dtarot-meaning-pack-json');
    if (!jsonEl) return;
    let data = null;
    try { data = JSON.parse(jsonEl.textContent || '{}'); } catch (e) { data = null; }
    if (!data) return;

    // Initial selection: last selected or current hidden input.
    let initial = null;
    try {
      if (data.postId) initial = localStorage.getItem('dtarot_meaning_pack_last_' + String(data.postId));
    } catch (e) {}
    if (!initial) {
      const cur = document.querySelector('input[data-dtarot-card-id]');
      initial = cur ? cur.value : null;
    }
    if (initial && data.cards && data.cards[initial]) {
      dtarotSelectMeaningCard(initial, data);
    }

    // Row click selects
    $(document).on('click', '.dtarot-meaning-row', function(e){
      if (e.target && (e.target.tagName === 'A' || e.target.tagName === 'BUTTON' || e.target.closest('button') || e.target.closest('a'))) {
        return;
      }
      const id = $(this).data('card-id');
      if (!id) return;
      // data-card-id is lowercased in markup; we need original key
      // Find matching key by case-insensitive compare.
      const key = Object.keys(data.cards || {}).find(k => String(k).toLowerCase() === String(id).toLowerCase());
      if (key) dtarotSelectMeaningCard(key, data);
    });

    // Edit button selects
    $(document).on('click', '.dtarot-meaning-edit-btn', function(e){
      e.preventDefault();
      const id = $(this).data('dtarot-card');
      if (!id) return;
      dtarotSelectMeaningCard(String(id), data);
    });
  });

  // --- Calendar modal ---
  function getEditorContent(id){
    if (window.tinymce) {
      const ed = tinymce.get(id);
      if (ed) return ed.getContent();
    }
    return $('#'+id).val() || '';
  }
  function setEditorContent(id, html){
    if (window.tinymce) {
      const ed = tinymce.get(id);
      if (ed) { ed.setContent(html || ''); return; }
    }
    $('#'+id).val(html || '');
  }

  function stripHtml(html){
    try {
      const div = document.createElement('div');
      div.innerHTML = String(html || '');
      return (div.textContent || div.innerText || '').toString();
    } catch (e) {
      return String(html || '');
    }
  }

  function escapeAttr(s){
    return String(s || '')
      .replace(/&/g, '&amp;')
      .replace(/"/g, '&quot;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;');
  }

  function escapeHtml(s){
    return String(s || '')
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  function insertLinkIntoEditor(editorId, url, title){
    editorId = (editorId || '').toString();
    url = (url || '').toString();
    title = (title || '').toString();
    if (!editorId || !url) return;

    // TinyMCE path
    if (window.tinymce) {
      const ed = tinymce.get(editorId);
      if (ed) {
        try {
          const selHtml = ed.selection.getContent({ format: 'html' });
          const hasSel = (selHtml || '').replace(/\s+/g, '').length > 0;
          if (hasSel) {
            ed.selection.setContent('<a href="' + escapeAttr(url) + '">' + selHtml + '</a>');
          } else {
            const text = title || url;
            ed.insertContent('<a href="' + escapeAttr(url) + '">' + escapeHtml(text) + '</a>');
          }
          ed.focus();
          return;
        } catch (e) {}
      }
    }

    // Textarea fallback
    const ta = document.getElementById(editorId);
    if (!ta) return;
    const start = ta.selectionStart || 0;
    const end = ta.selectionEnd || 0;
    const val = ta.value || '';
    const selected = val.substring(start, end);
    const linkText = selected || (title || url);
    const replacement = '<a href="' + url + '">' + linkText + '</a>';
    ta.value = val.substring(0, start) + replacement + val.substring(end);
    try {
      ta.selectionStart = start;
      ta.selectionEnd = start + replacement.length;
      ta.focus();
    } catch (e) {}
  }

  function linkFirstOccurrenceHtml(html, anchor, url){
    html = String(html || '');
    anchor = String(anchor || '').trim();
    url = String(url || '').trim();
    if (!html || !anchor || !url) return { html: html, did: false };

    // Split into tags vs text parts so we don't replace inside attributes.
    // Also track whether we're inside disallowed blocks (like H1/H2).
    const parts = html.split(/(<[^>]+>)/g);
    const needle = anchor.toLowerCase();
    const disallowed = new Set(['h1','h2','a']);
    const stack = [];
    let did = false;
    for (let i = 0; i < parts.length; i++) {
      const part = parts[i];
      if (!part) continue;

      if (part[0] === '<') {
        // Very small tag parser to maintain a stack.
        // Supports: <h1>, <h1 ...>, </h1>, <br/>, <!-- -->
        const m = part.match(/^<\s*(\/)?\s*([a-zA-Z0-9:_-]+)/);
        if (m) {
          const isClose = !!m[1];
          const tag = String(m[2] || '').toLowerCase();
          const selfClosing = /\/\s*>$/.test(part) || /^<\s*!(?:--|doctype)/i.test(part) || /^<\s*\?/.test(part);
          if (!selfClosing && tag) {
            if (isClose) {
              for (let j = stack.length - 1; j >= 0; j--) {
                if (stack[j] === tag) {
                  stack.length = j;
                  break;
                }
              }
            } else {
              stack.push(tag);
            }
          }
        }
        continue;
      }

      if (stack.some(t => disallowed.has(t))) continue;
      const hay = part.toLowerCase();
      const idx = hay.indexOf(needle);
      if (idx === -1) continue;

      const before = part.slice(0, idx);
      const match = part.slice(idx, idx + anchor.length);
      const after = part.slice(idx + anchor.length);
      parts[i] = before + '<a href="' + escapeAttr(url) + '">' + escapeHtml(match) + '</a>' + after;
      did = true;
      break;
    }
    return { html: parts.join(''), did: did };
  }

  function autoLinkPhrase(editorId, url, anchor, title){
    editorId = (editorId || '').toString();
    url = (url || '').toString();
    anchor = (anchor || '').toString();
    title = (title || '').toString();
    if (!editorId || !url) return false;

    // If user has a selection, respect it.
    if (window.tinymce) {
      const ed = tinymce.get(editorId);
      if (ed) {
        try {
          const selHtml = ed.selection.getContent({ format: 'html' });
          const hasSel = (selHtml || '').replace(/\s+/g, '').length > 0;
          if (hasSel) {
            ed.selection.setContent('<a href="' + escapeAttr(url) + '">' + selHtml + '</a>');
            ed.focus();
            return true;
          }

          const cur = ed.getContent({ format: 'raw' });
          if (anchor && cur && cur.toLowerCase().indexOf(anchor.toLowerCase()) !== -1) {
            const res = linkFirstOccurrenceHtml(cur, anchor, url);
            if (res.did) {
              ed.setContent(res.html);
              ed.focus();
              return true;
            }
          }

          // Fallback: insert a link at the cursor.
          const linkText = anchor || title || url;
          ed.insertContent('<a href="' + escapeAttr(url) + '">' + escapeHtml(linkText) + '</a>');
          ed.focus();
          return true;
        } catch (e) {}
      }
    }

    // Textarea fallback: insert at cursor.
    insertLinkIntoEditor(editorId, url, anchor || title || url);
    return true;
  }

  // Calendar helper: only link an existing match; never insert new linked text.
  function tryLinkExistingPhrase(editorId, url, anchor){
    editorId = (editorId || '').toString();
    url = (url || '').toString();
    anchor = (anchor || '').toString();
    if (!editorId || !url || !anchor) return false;

    // TinyMCE
    if (window.tinymce) {
      const ed = tinymce.get(editorId);
      if (ed) {
        try {
          const cur = ed.getContent({ format: 'raw' });
          if (cur && cur.toLowerCase().indexOf(anchor.toLowerCase()) !== -1) {
            const res = linkFirstOccurrenceHtml(cur, anchor, url);
            if (res.did) {
              ed.setContent(res.html);
              ed.focus();
              return true;
            }
          }
        } catch (e) {}
        return false;
      }
    }

    // Textarea fallback
    const ta = document.getElementById(editorId);
    if (!ta) return false;
    const cur = (ta.value || '').toString();
    if (!cur || cur.toLowerCase().indexOf(anchor.toLowerCase()) === -1) return false;
    const res = linkFirstOccurrenceHtml(cur, anchor, url);
    if (res.did) {
      ta.value = res.html;
      return true;
    }
    return false;
  }

  function renderDayLinkItems(containerEl, items){
    if (!containerEl) return;
    const list = Array.isArray(items) ? items : [];
    if (!list.length) {
      containerEl.innerHTML = '<p class="description" style="margin:8px 0;">' + escapeHtml(t('noneFound','No matches found.')) + '</p>';
      return;
    }
    let html = '<ul>';
    list.forEach(function(it){
      const id = it && (typeof it.id === 'number' ? it.id : (parseInt(it.id, 10) || 0));
      const title = it && it.title ? String(it.title) : '';
      const url = it && it.url ? String(it.url) : '';
      const type = it && it.type ? String(it.type) : '';
      const anchor = it && it.anchor ? String(it.anchor) : '';
      const source = it && it.source ? String(it.source) : '';
      if (!url) return;
      html += '<li>'
        + '<a href="' + escapeAttr(url) + '" target="_blank" rel="noopener noreferrer">' + escapeHtml(title || url) + '</a>'
        + (type ? ' <span class="description">(' + escapeHtml(type) + (id ? (' #' + id) : '') + ')</span>' : '')
        + (anchor ? ' <span class="description">' + escapeHtml(t('suggestedAnchor','Suggested:')) + ' “' + escapeHtml(anchor) + '”' + (source && source !== 'either' ? (' · ' + escapeHtml(source)) : '') + '</span>' : '')
        + '<span class="dtarot-day-link-actions">'
          + '<button type="button" class="button button-small" data-dtarot-day-link="open" data-url="' + escapeAttr(url) + '">' + escapeHtml(t('open','Open')) + '</button>'
          + '<button type="button" class="button button-small" data-dtarot-day-link="insert" data-url="' + escapeAttr(url) + '" data-title="' + escapeAttr(title) + '" data-anchor="' + escapeAttr(anchor) + '" data-source="' + escapeAttr(source) + '">' + escapeHtml(t('autoLinkPhrase','Link text')) + '</button>'
        + '</span>'
      + '</li>';
    });
    html += '</ul>';
    containerEl.innerHTML = html;
  }

  function setPreview(url){
    const $img = $('#dtarot-day-preview-img');
    const $empty = $('#dtarot-day-preview-empty');

    if (url) {
      $img.attr('src', url).show();
      $empty.hide();
    } else {
      $img.removeAttr('src').hide();
      $empty.show();
    }
  }

  let dayImageOverrideUrl = '';
  let dayImageOverrideAttachmentId = '';
  function setDayImageOverride(url, attachmentId){
    dayImageOverrideUrl = url || '';
    dayImageOverrideAttachmentId = attachmentId || '';

    const $note = $('#dtarot-day-upload-note');
    if (dayImageOverrideUrl) {
      $note.text(t('customImageSet','Custom image set.')).show();
    } else {
      $note.text('').hide();
    }
  }

  function refreshPreview(){
    if (dayImageOverrideUrl) {
      setPreview(dayImageOverrideUrl);
      return;
    }

    const deckId = $('#dtarot-deck').val();
    const cardId = $('#dtarot-card').val();
    if (!deckId || !cardId) {
      setPreview('');
      return;
    }

    postAjax({ action:'dtarot_get_card_image', deck_id:deckId, card_id:cardId }, function(res){
      if (!res || !res.success) return;
      const url = (res.data && res.data.url) ? res.data.url : '';
      setPreview(url);
    });
  }

  $(document).on('click', '.dtarot-day-cell', function(e){
    e.preventDefault();
    const date = this.getAttribute('data-date');
    if (!date) return;

    const maxDate = calendarMaxDate();
    const pro = calendarIsPro();
    const locked = !pro && maxDate && isDateBeyondMax(String(date), String(maxDate));

    $('#dtarot-modal-title').text(date);
    $('#dtarot-day-modal').show();

    // Pre-disable actions for locked dates (still allow viewing).
    if (locked) {
      $('#dtarot-save-note').removeClass('dtarot-msg-ok').addClass('dtarot-msg-err')
        .text(t('calendarLimit','Free plan can edit days up to and including %s. Activate Pro to edit any date.').replace('%s', (maxDate || '')));
      $('#dtarot-save-day').prop('disabled', true);
      $('#dtarot-copy-prev').prop('disabled', true);
      $('#dtarot-reuse-old-text').prop('disabled', true);
      $('#dtarot-day-upload-image').prop('disabled', true);
      $('#dtarot-publish-time').prop('disabled', true);
    } else {
      $('#dtarot-save-note').removeClass('dtarot-msg-ok dtarot-msg-err').text('');
      updateReadyNote();
      $('#dtarot-copy-prev').prop('disabled', false);
      $('#dtarot-reuse-old-text').prop('disabled', false);
      $('#dtarot-day-upload-image').prop('disabled', false);
      $('#dtarot-publish-time').prop('disabled', false);
    }

    postAjax({ action:'dtarot_get_day', date:date }, function(res){
      if (!res || !res.success) return;
      const d = res.data || {};
      $('#dtarot-deck').val(d.deck || '');
      applySystemFilters();
      $('#dtarot-card').val(d.card || '');
      $('#dtarot-pack').val(d.pack || '');
      applySystemFilters();
      $('#dtarot-status').val(d.status || 'draft');
      setEditorContent('dtarot-content', d.content || '');
      setEditorContent('dtarot_daily_text', d.daily_text || '');

      // Pro-only: per-day publish time override.
      if (typeof d.publish_time !== 'undefined') {
        $('#dtarot-publish-time').val(d.publish_time || '');
      }

      setDayImageOverride(d.image_override_url || '', d.image_override_attachment_id || '');
      refreshPreview();

      if (!locked) updateReadyNote();

      if (window.DTAROT_AUTO_COPY_PREV) {
        window.DTAROT_AUTO_COPY_PREV = false;
        copyPreviousDaySettings(date);
      }

      // Refresh auto link suggestions when the modal is populated.
      try {
        if (window.dtarotRefreshDayLinkSuggestions) {
          window.dtarotRefreshDayLinkSuggestions(true);
        }
      } catch (e) {}
    });
  });

  // Calendar modal: internal link helper
  (function(){
    const helper = document.getElementById('dtarot-day-link-helper');
    if (!helper) return;

    const statusEl = document.getElementById('dtarot-day-link-status');
    const sugEl = document.getElementById('dtarot-day-link-suggestions');
    const resEl = document.getElementById('dtarot-day-link-results');
    const qEl = document.getElementById('dtarot-day-link-q');

    let timer = null;

    function setStatus(msg){
      if (!statusEl) return;
      statusEl.textContent = msg || '';
    }

    function flashStatus(msg){
      setStatus(msg);
      try {
        setTimeout(function(){
          if (statusEl && statusEl.textContent === msg) setStatus('');
        }, 2600);
      } catch (e) {}
    }

    function currentTextPayload(){
      const contentHtml = getEditorContent('dtarot-content');
      const dailyHtml = getEditorContent('dtarot_daily_text');
      return {
        content: stripHtml(contentHtml),
        daily_text: stripHtml(dailyHtml)
      };
    }

    function doAutoSuggest(){
      if (!sugEl) return;
      const p = currentTextPayload();
      const combined = (p.content + ' ' + p.daily_text).trim();
      if (combined.length < 8) {
        renderDayLinkItems(sugEl, []);
        setStatus(t('typeMoreForSuggestions','Type a bit more text to get link suggestions.'));
        return;
      }
      setStatus(t('loadingEllipsis','Loading…'));
      sugEl.innerHTML = '';
      postAjax({ action: 'dtarot_related_link_suggest_from_text', content: p.content, daily_text: p.daily_text, limit: 3 }, function(res){
        setStatus('');
        if (!res || !res.success || !res.data) {
          renderDayLinkItems(sugEl, []);
          return;
        }
        if (res.data.fallback) {
          setStatus(t('noConnectionsFallback','No strong connections found — showing 3 random posts/pages you can link.'));
        }
        const items = Array.isArray(res.data.items) ? res.data.items.slice(0, 3) : [];
        renderDayLinkItems(sugEl, items);
      });
    }

    function queueAutoSuggest(){
      if (timer) clearTimeout(timer);
      timer = setTimeout(doAutoSuggest, 650);
    }

    window.dtarotRefreshDayLinkSuggestions = function(immediate){
      if (immediate) {
        if (timer) clearTimeout(timer);
        doAutoSuggest();
      } else {
        queueAutoSuggest();
      }
    };

    // Listen to TinyMCE changes (when available)
    function attachEditorListeners(){
      if (window.tinymce) {
        const ed1 = tinymce.get('dtarot-content');
        const ed2 = tinymce.get('dtarot_daily_text');
        [ed1, ed2].forEach(function(ed){
          if (!ed || ed._dtarotLinkHelperBound) return;
          ed._dtarotLinkHelperBound = true;
          ed.on('keyup change setcontent', function(){ queueAutoSuggest(); });
        });
      }

      // Textarea fallback (Text tab)
      ['dtarot-content','dtarot_daily_text'].forEach(function(id){
        const el = document.getElementById(id);
        if (!el || el._dtarotLinkHelperBound) return;
        el._dtarotLinkHelperBound = true;
        el.addEventListener('input', function(){ queueAutoSuggest(); });
      });
    }

    // Try a few times because editors can initialize late.
    let tries = 0;
    const interval = setInterval(function(){
      attachEditorListeners();
      tries++;
      if (tries >= 10) clearInterval(interval);
    }, 500);

    $(document).on('click', '#dtarot-day-link-refresh', function(){
      doAutoSuggest();
    });

    $(document).on('click', '#dtarot-day-link-search', function(){
      const q = (qEl && qEl.value ? qEl.value : '').toString().trim();
      if (!q) {
        renderDayLinkItems(resEl, []);
        return;
      }
      if (resEl) resEl.innerHTML = '<p class="description" style="margin:8px 0;">' + escapeHtml(t('loadingEllipsis','Loading…')) + '</p>';
      postAjax({ action: 'dtarot_related_link_search', q: q }, function(res){
        if (!res || !res.success || !res.data) {
          renderDayLinkItems(resEl, []);
          return;
        }
        renderDayLinkItems(resEl, res.data.items || []);
      });
    });

    if (qEl) {
      qEl.addEventListener('keydown', function(e){
        if (e.key === 'Enter') {
          e.preventDefault();
          $('#dtarot-day-link-search').trigger('click');
        }
      });
    }

    helper.addEventListener('click', function(e){
      const btn = e.target && e.target.closest ? e.target.closest('button[data-dtarot-day-link]') : null;
      if (!btn) return;
      const act = (btn.getAttribute('data-dtarot-day-link') || '').toString();
      const url = (btn.getAttribute('data-url') || '').toString();
      const title = (btn.getAttribute('data-title') || '').toString();
      const anchor = (btn.getAttribute('data-anchor') || '').toString();
      const source = (btn.getAttribute('data-source') || '').toString();
      if (!act) return;
      e.preventDefault();
      if (act === 'open') {
        if (url) window.open(url, '_blank', 'noopener,noreferrer');
        return;
      }
      if (act === 'insert') {
        // Never insert into the focused editor. Only wrap an existing phrase in-place.
        // Prefer the editor where the backend says the anchor was found.
        let first = 'dtarot-content';
        let second = 'dtarot_daily_text';
        if (source === 'daily_text') {
          first = 'dtarot_daily_text';
          second = 'dtarot-content';
        }

        let ok = false;
        if (anchor) {
          ok = tryLinkExistingPhrase(first, url, anchor) || tryLinkExistingPhrase(second, url, anchor);
        }

        if (!ok) {
          flashStatus(t('phraseNotFound','Phrase not found in Content or Daily Text — no link inserted.'));
        }
        return;
      }
    }, true);

  })();

  function dateMinusOne(ymd){
    try {
      const d = new Date(ymd + 'T00:00:00');
      if (isNaN(d.getTime())) return '';
      d.setDate(d.getDate() - 1);
      const y = d.getFullYear();
      const m = String(d.getMonth() + 1).padStart(2, '0');
      const day = String(d.getDate()).padStart(2, '0');
      return y + '-' + m + '-' + day;
    } catch (e) {
      return '';
    }
  }

  function copyPreviousDaySettings(date){
    const prev = dateMinusOne(date);
    if (!prev) return;

    postAjax({ action:'dtarot_get_day', date:prev }, function(res){
      if (!res || !res.success) return;
      const d = res.data || {};

      // Copy deck + pack. Do NOT copy the card by default (avoid duplicate card).
      if (d.deck) {
        $('#dtarot-deck').val(d.deck || '');
        applySystemFilters();
      }
      if (d.pack) {
        $('#dtarot-pack').val(d.pack || '');
        applySystemFilters();
      }

      // If text is empty, fill from previous day.
      const curContent = (getEditorContent('dtarot-content') || '').trim();
      const curDaily = (getEditorContent('dtarot_daily_text') || '').trim();
      if (!curContent && d.content) {
        setEditorContent('dtarot-content', d.content || '');
      }
      if (!curDaily && d.daily_text) {
        setEditorContent('dtarot_daily_text', d.daily_text || '');
      }

      refreshPreview();
      updateReadyNote();
    });
  }

  $(document).on('click', '#dtarot-copy-prev', function(){
    const date = ($('#dtarot-modal-title').text() || '').trim();
    if (!date) return;
    copyPreviousDaySettings(date);
  });

  // Deep-link: open modal directly.
  try {
    const params = new URLSearchParams(window.location.search || '');
    const editDate = (params.get('dtarot_edit_date') || '').trim();
    const copyPrev = (params.get('dtarot_copy_prev') || '').trim();
    if (editDate) {
      window.DTAROT_AUTO_COPY_PREV = (copyPrev === '1' || copyPrev === 'true');
      const cell = document.querySelector('.dtarot-day-cell[data-date="' + editDate + '"]');
      if (cell) cell.click();
    }
  } catch (e) {}

  function updateReadyNote(){
    const deckId = ($('#dtarot-deck').val() || '').trim();
    const cardId = ($('#dtarot-card').val() || '').trim();
    const status = ($('#dtarot-status').val() || 'draft').trim();
    const $note = $('#dtarot-ready-note');
    const $save = $('#dtarot-save-day');

    $note.removeClass('dtarot-ready-ok dtarot-ready-warn');

    const hasDeckCard = !!deckId && !!cardId;
    if (status === 'publish') {
      if (!hasDeckCard) {
        $note.addClass('dtarot-ready-warn').text(t('publishNeedsDeckCard', 'To publish, select a deck and a card.'));
        $save.prop('disabled', true);
        return;
      }
      $note.addClass('dtarot-ready-ok').text(t('readyToPublish', 'Ready to publish'));
      $save.prop('disabled', false);
      return;
    }

    // Draft
    $save.prop('disabled', false);
    if (hasDeckCard) {
      $note.addClass('dtarot-ready-ok').text(t('draftReady', 'Draft (ready)'));
    } else {
      $note.addClass('dtarot-ready-warn').text(t('draftIncomplete', 'Draft (missing deck/card)'));
    }
  }

  function getSelectedDeckSystem(){
    const sel = document.querySelector('#dtarot-deck option:checked');
    if (!sel) return 'tarot';
    return (sel.getAttribute('data-system') || 'tarot');
  }

  function filterSelectBySystem(selectEl, system){
    if (!selectEl) return;
    const opts = selectEl.querySelectorAll('option');
    opts.forEach(function(opt, idx){
      if (idx === 0) {
        opt.hidden = false;
        opt.disabled = false;
        return;
      }
      const s = opt.getAttribute('data-system') || '';
      const show = (s === '') || (s === system);
      opt.hidden = !show;
      opt.disabled = !show;
    });

    const selected = selectEl.options[selectEl.selectedIndex];
    if (selected && selected.hidden) {
      selectEl.value = '';
    }
  }

  function applySystemFilters(){
    const system = getSelectedDeckSystem();
    filterSelectBySystem(document.getElementById('dtarot-card'), system);
    filterSelectBySystem(document.getElementById('dtarot-pack'), system);
    updateReadyNote();
  }

  $(document).on('change', '#dtarot-deck,#dtarot-card,#dtarot-pack,#dtarot-status', function(){
    updateReadyNote();
  });

  $(document).on('change', '#dtarot-deck', function(){
    // Card images are per-deck; changing deck should reset card selection.
    $('#dtarot-card').val('');
    $('#dtarot-pack').val('');
    setDayImageOverride('', '');
    applySystemFilters();
    refreshPreview();
  });

  $(document).on('change', '#dtarot-card', function(){
    setDayImageOverride('', '');
    refreshPreview();
  });

  $(document).on('click', '#dtarot-day-upload-image', function(){
    const deckId = $('#dtarot-deck').val();
    const cardId = $('#dtarot-card').val();
    if (!deckId || !cardId) {
      alert(t('publishNeedsDeckCard', 'To publish, select a deck and a card.'));
      return;
    }

    if (!(window.wp && wp.media)) {
      alert(t('mediaMissing','Media library not available'));
      return;
    }

    const frame = wp.media({
      title: t('mediaCardTitle','Select card image'),
      button: { text: t('mediaUseImage','Use this image') },
      library: { type: 'image' },
      multiple: false
    });

    frame.on('select', function(){
      const sel = frame.state().get('selection');
      const first = sel && sel.first ? sel.first() : null;
      if (!first) return;

      const a = first.toJSON ? first.toJSON() : {};
      const attachmentId = a.id ? String(a.id) : '';
      const fallbackUrl = a.url ? String(a.url) : '';
      if (!attachmentId) return;

      const $note = $('#dtarot-day-upload-note');
      $note.text(t('loadingEllipsis','Loading…')).show();

      postAjax({ action:'dtarot_optimize_attachment', attachment_id: attachmentId }, function(res){
        if (!res || !res.success) {
          setDayImageOverride(fallbackUrl, attachmentId);
          refreshPreview();
          return;
        }
        const url = (res.data && res.data.url) ? String(res.data.url) : fallbackUrl;
        setDayImageOverride(url, attachmentId);
        refreshPreview();
      });
    });

    frame.open();
  });

  $(document).on('click', '.dtarot-modal-close, .dtarot-modal-backdrop', function(){
    const $m = $(this).closest('.dtarot-modal');
    if ($m.length) {
      $m.hide();
      return;
    }
    $('#dtarot-day-modal').hide();
  });

  $(document).on('click', '#dtarot-save-day', function(){
    const $btn = $('#dtarot-save-day');
    const $note = $('#dtarot-save-note');
    const date = ($('#dtarot-modal-title').text() || '').trim();
    if (!date) return;

    const maxDate = calendarMaxDate();
    const pro = calendarIsPro();
    if (!pro && maxDate && isDateBeyondMax(String(date), String(maxDate))) {
      $note.removeClass('dtarot-msg-ok').addClass('dtarot-msg-err')
        .text(t('calendarLimit','Free plan can edit days up to and including %s. Activate Pro to edit any date.').replace('%s', (maxDate || '')));
      return;
    }

    $btn.prop('disabled', true).text(t('savingEllipsis','Saving…'));
    $note.removeClass('dtarot-msg-ok dtarot-msg-err').text('');

    postAjax({
      action:'dtarot_save_day',
      date: date,
      deck: $('#dtarot-deck').val(),
      card: $('#dtarot-card').val(),
      pack: $('#dtarot-pack').val(),
      status: $('#dtarot-status').val(),
      publish_time: ($('#dtarot-publish-time').length ? $('#dtarot-publish-time').val() : ''),
      content: getEditorContent('dtarot-content'),
      daily_text: getEditorContent('dtarot_daily_text'),
      image_override_attachment_id: dayImageOverrideAttachmentId,
      image_override_url: dayImageOverrideUrl
    }, function(res){
      $btn.prop('disabled', false).text(t('save','Save'));

      if (!res || !res.success) {
        $note.addClass('dtarot-msg-err').text((res && res.data && res.data.message) ? res.data.message : t('saveFailed','Save failed'));
        return;
      }

      const status = $('#dtarot-status').val() || 'draft';
      const cellEl = document.querySelector('.dtarot-day-cell[data-date="'+date+'"]');
      if (cellEl) {
        cellEl.classList.add('dtarot-is-set');
        cellEl.classList.toggle('dtarot-filled', status === 'publish');
        const st = cellEl.querySelector('.dtarot-status');
        if (st) st.textContent = status.charAt(0).toUpperCase() + status.slice(1);
      }

      $note.addClass('dtarot-msg-ok').text(t('saved','Saved.'));
      // Close after a short delay so the user sees confirmation.
      setTimeout(function(){
        $('#dtarot-day-modal').hide();
      }, 350);

      const deck = $('#dtarot-deck').val();
      const card = $('#dtarot-card').val();
      if (deck && card) {
        // Keep the user on the same month after save.
        const parts = String(date).split('-');
        if (parts.length === 3) {
          const year = parts[0];
          const monthNum = parseInt(parts[1], 10);
          const month = Number.isFinite(monthNum) ? String(monthNum) : parts[1];
          try {
            const url = new URL(window.location.href);
            url.searchParams.set('page', 'daily-tarot-calendar');
            url.searchParams.set('year', year);
            url.searchParams.set('month', month);
            url.searchParams.delete('dtarot_edit_date');
            window.location.href = url.toString();
          } catch (err) {
            location.reload();
          }
        } else {
          location.reload();
        }
      }
    });
  });

  $(document).on('click', '#dtarot-reuse-old-text', function(){
    const date = ($('#dtarot-modal-title').text() || '').trim();
    if (!date) return;

    const $btn = $('#dtarot-reuse-old-text');
    const $note = $('#dtarot-save-note');
    $btn.prop('disabled', true).text(t('loadingEllipsis','Loading…'));
    $note.removeClass('dtarot-msg-ok dtarot-msg-err').text('');

    postAjax({
      action: 'dtarot_reuse_old_text',
      date: date
    }, function(res){
      $btn.prop('disabled', false).text(t('reuseOldText','Reuse old text'));

      if (!res || !res.success) {
        $note.addClass('dtarot-msg-err').text((res && res.data && res.data.message) ? res.data.message : t('noOlderTextFound','No older text found'));
        return;
      }

      const d = res.data || {};
      setEditorContent('dtarot-content', d.content || '');
      setEditorContent('dtarot_daily_text', d.daily_text || '');

      if (d.from_date) {
        $note.addClass('dtarot-msg-ok').text(t('loadedTextFrom','Loaded text from %s.').replace('%s', d.from_date));
      } else {
        $note.addClass('dtarot-msg-ok').text(t('loadedText','Loaded text.'));
      }
    });
  });

  function initOnboarding(){
    const dismissButtons = document.querySelectorAll('[data-dtarot-onboard-dismiss]');
    dismissButtons.forEach(function(btn){
      btn.addEventListener('click', function(e){
        e.preventDefault();
        document.querySelectorAll('[data-dtarot-onboard-banner]').forEach(function(el){ el.remove(); });
        document.querySelectorAll('.dtarot-onboard-card').forEach(function(el){ el.style.display = 'none'; });
        postAjax({ action: 'dtarot_onboard_dismiss' }, function(){});
      });
    });

    const tourButtons = document.querySelectorAll('[data-dtarot-onboard-start]');
    if (!tourButtons.length) return;

    let overlay = null;
    let tooltip = null;
    let currentIndex = 0;
    let currentTarget = null;

    const steps = [
      {
        id: 'today',
        title: t('onboardTodayTitle','Set today card'),
        text: t('onboardTodayText','Edit your card of the day and preview it.'),
        selector: '[data-dtarot-tour=\"today\"]'
      },
      {
        id: 'calendar',
        title: t('onboardCalendarTitle','Plan your calendar'),
        text: t('onboardCalendarText','Fill upcoming days in the calendar overview.'),
        selector: '[data-dtarot-tour=\"calendar\"]'
      },
      {
        id: 'spreads',
        title: t('onboardSpreadsTitle','Configure spreads'),
        text: t('onboardSpreadsText','Set spread presets and meaning packs.'),
        selector: '[data-dtarot-tour=\"spreads\"]'
      },
      {
        id: 'bookings',
        title: t('onboardBookingsTitle','Enable bookings'),
        text: t('onboardBookingsText','Create reading types and accept bookings.'),
        selector: '[data-dtarot-tour=\"bookings\"]'
      }
    ];

    function ensureTourUI(){
      if (overlay && tooltip) return;
      overlay = document.createElement('div');
      overlay.className = 'dtarot-onboard-tour-overlay';
      tooltip = document.createElement('div');
      tooltip.className = 'dtarot-onboard-tour';
      tooltip.innerHTML = '' +
        '<div class=\"dtarot-onboard-tour-title\"></div>' +
        '<div class=\"dtarot-onboard-tour-text\"></div>' +
        '<div class=\"dtarot-onboard-tour-actions\">' +
          '<button type=\"button\" class=\"button dtarot-onboard-prev\">' + t('onboardBack','Back') + '</button>' +
          '<button type=\"button\" class=\"button button-primary dtarot-onboard-next\">' + t('onboardNext','Next') + '</button>' +
          '<button type=\"button\" class=\"button-link dtarot-onboard-skip\">' + t('onboardClose','Close') + '</button>' +
        '</div>';
      document.body.appendChild(overlay);
      document.body.appendChild(tooltip);

      overlay.addEventListener('click', endTour);
      tooltip.querySelector('.dtarot-onboard-prev').addEventListener('click', function(){ stepTo(currentIndex - 1); });
      tooltip.querySelector('.dtarot-onboard-next').addEventListener('click', function(){ stepTo(currentIndex + 1); });
      tooltip.querySelector('.dtarot-onboard-skip').addEventListener('click', endTour);
    }

    function positionTooltip(target){
      if (!tooltip || !target) return;
      const rect = target.getBoundingClientRect();
      const tipRect = tooltip.getBoundingClientRect();
      let top = rect.bottom + 12;
      let left = rect.left;
      if (top + tipRect.height > window.innerHeight - 12) {
        top = rect.top - tipRect.height - 12;
      }
      if (top < 12) top = 12;
      if (left + tipRect.width > window.innerWidth - 12) {
        left = window.innerWidth - tipRect.width - 12;
      }
      if (left < 12) left = 12;
      tooltip.style.top = top + 'px';
      tooltip.style.left = left + 'px';
    }

    function stepTo(idx){
      if (!steps.length) return;
      if (idx < 0) idx = 0;
      if (idx >= steps.length) {
        endTour();
        return;
      }
      currentIndex = idx;
      const step = steps[idx];
      const target = document.querySelector(step.selector);
      if (!target) {
        stepTo(idx + 1);
        return;
      }
      if (currentTarget) currentTarget.classList.remove('dtarot-onboard-highlight');
      currentTarget = target;
      currentTarget.classList.add('dtarot-onboard-highlight');
      try {
        currentTarget.scrollIntoView({ behavior: 'smooth', block: 'center' });
      } catch (e) {}
      tooltip.querySelector('.dtarot-onboard-tour-title').textContent = step.title;
      tooltip.querySelector('.dtarot-onboard-tour-text').textContent = step.text;
      tooltip.querySelector('.dtarot-onboard-prev').disabled = (idx === 0);
      tooltip.querySelector('.dtarot-onboard-next').textContent = (idx === steps.length - 1) ? t('onboardFinish','Finish') : t('onboardNext','Next');
      positionTooltip(target);
    }

    function startTour(){
      ensureTourUI();
      overlay.classList.add('is-active');
      tooltip.classList.add('is-active');
      stepTo(0);
      window.addEventListener('resize', function(){ if (currentTarget) positionTooltip(currentTarget); }, { passive: true });
    }

    function endTour(){
      if (overlay) overlay.classList.remove('is-active');
      if (tooltip) tooltip.classList.remove('is-active');
      if (currentTarget) currentTarget.classList.remove('dtarot-onboard-highlight');
      currentTarget = null;
      postAjax({ action: 'dtarot_onboard_done' }, function(){});
    }

    tourButtons.forEach(function(btn){
      btn.addEventListener('click', function(e){
        e.preventDefault();
        startTour();
      });
    });
  }

  initOnboarding();

})(jQuery);
