<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://charlenesweb.ca
 * @since      1.0.0
 *
 * @package    CWSI_Invites
 * @subpackage CWSI_Invites/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    CWSI_Invites
 * @subpackage CWSI_Invites/public
 * @author     Charlene Copeland <charlene@charlenesweb.ca>
 */
#[AllowDynamicProperties]   
class cwsi_invites_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $cwsi_invites    The ID of this plugin.
	 */
	private $cwsi_invites;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $cwsi_invites       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $cwsi_invites, $version ) {
		$this->plugin_name = $cwsi_invites;
		$this->version = $version;
		add_action( 'init', array( $this, 'init_shortcodes' ), 20 ); // run on priority 20 as Shortcodes are registered at priority 10
	}
	
	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/cwsi-invites-public.css', array(), $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
		wp_enqueue_script($this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/cwsi-invites-public.js', array( 'jquery' ), $this->version, false );
	}
	
	
	/**
	 * Register public Shortcodes
	 *
	 * @since 1.0.0
	 */
	public function init_shortcodes() {
		add_shortcode( 'cwsiInviteAnswerForm', array($this, 'cwsiinviteanswerform_func') );
		add_shortcode( 'cwsiInviteShowResponses', array($this, 'cwsiinviteshowresponses_func') );
	}

	/**
	 * SHORTCODE FUNCTIONS
	 * Show the add numbers form for the public - [cwsiInviteAnswerForm]
	 * @since 1.0.0
	 */
	public function cwsiinviteanswerform_func($_atts) {
		$defaults = array(
			'event_id' => "1",
		);
		$atts = shortcode_atts( $defaults, $_atts );
		$event_id = $atts['event_id'];
		
		global $wp;
		$ct = "";
		$current_url  = home_url( $wp->request );
		$msg = "";
		$warn = "";
		// validate event id
		$message = cwsiPubIsEventForID($event_id);
		if ($message != "") {
			$ct .= $message;
		} else {
			// Was saveinviteeresponse form submitted?
			if (isset($_POST['saveinviteeresponse']) && isset($_POST['rsvp_answer_form_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['rsvp_answer_form_nonce'])), 'rsvp-answer')) {
				$results_add = cwsiPubProcessAddRsvpForm($_POST);
			} else {
				$results_add = array();
			}
			$results_get_guest = array();
			if (isset($_GET['email'])) {
				if (is_email(wp_unslash($_GET['email']))) {
					$results_get_guest = cwsiPubGetRsvpByEmail(sanitize_email(wp_unslash($_GET['email'])), $event_id);
				}
			}
	
			require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-public-display.php';
			$ct .= cwsiShowInvitesAddRsvpPage($current_url, $results_get_guest, $results_add, $event_id);
		}
		return $ct;
	}
	/**
	 * Show the responses so far for the public - [cwsiInviteShowResponses]
	 * @since 1.0.0
	 */
	public function cwsiinviteshowresponses_func($_atts) {
		$defaults = array(
			'event_id' => "1",
		);
		$atts = shortcode_atts( $defaults, $_atts );
		$event_id = $atts['event_id'];
		global $wp;
		$ct = "";
		$results = cwsiPubInviteGetResponses($event_id);
		require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-public-display.php';
		$ct .= cwsiPubInviteShowResponses($results);
		return $ct;
	}
} // END class cwsi_invites_Public

//////////////////////////////////////////////////
// HELPER FUNCTIONS
/////////////////////////////////////////////////

// Retrieve all invitees
function cwsiPubGetCurrentRsvps() {
	global $wpdb;
	$values = $wpdb->get_results('SELECT * FROM '.$wpdb->prefix.'cwsi_invites ORDER BY full_name ASC' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	
	if (!is_object($values) && !is_array($values)) {
		$results = array("status"=>0, "message"=>'Failed to find invitees. Error is '.$wpdb->last_error.'. ');
	} else {
		$results = array("status"=>1, "message"=>"Success", "data"=>$values);
	}
	return $results;
}

function cwsiPubInviteGetResponses($event_id) {
	global $wpdb;
	// MUST CHANGE TO ORDER BY RSVP ANSWER DESC ONCE JOINED
	$values = $wpdb->get_results($wpdb->prepare('SELECT a.event_id, a.invitee_id, a.num_guests, a.rsvp_answer, a.comments, a.personal_note, a.admin_notes, a.date_email_sent, a.date_responded, b.ID, b.full_name, b.email, b.email2, b.date_updated FROM '.$wpdb->prefix.'cwsi_events_invitees as a JOIN '.$wpdb->prefix.'cwsi_invites as b ON a.invitee_id=b.ID WHERE event_id=%d ORDER BY a.rsvp_answer DESC, b.full_name', array(sanitize_text_field($event_id))));  // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	if (!is_object($values) && !is_array($values)) {
		$results = array("status"=>0, "message"=>'Failed to find invitees. Error is '.$wpdb->last_error.'. ');
	} else {
		$results = array("status"=>1, "message"=>"Success", "data"=>$values);
	}
	return $results;
}

// Retrieve invitee by ID
function cwsiPubGetRsvpByID($cwsinvitee_id) {
	global $wpdb;
	$values = $wpdb->get_results($wpdb->prepare('SELECT * FROM '.$wpdb->prefix.'cwsi_invites WHERE ID=%d', array($cwsinvitee_id))); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	if (!is_object($values) && !is_array($values)) {
		$results = array("status"=>0, "message"=>'Failed to find invitee for ID '.esc_html($cwsinvitee_id).'. Error is '.$wpdb->last_error.'. ');
	} else {
		$results = array("status"=>1, "message"=>"Success", "data"=>$values);
	}
	return $results;
}

function cwsiPubGetRsvpByEmail($email, $event_id) {
	global $wpdb;
	
	$values = $wpdb->get_results($wpdb->prepare('SELECT a.event_id, a.invitee_id, a.num_guests, a.rsvp_answer, a.comments, a.personal_note, a.admin_notes, a.date_email_sent, a.date_responded, b.ID, b.full_name, b.email, b.email2, b.date_updated FROM '.$wpdb->prefix.'cwsi_events_invitees as a JOIN '.$wpdb->prefix.'cwsi_invites as b ON a.invitee_id=b.ID WHERE event_id=%d AND email=%s', array(sanitize_text_field($event_id), sanitize_email($email))));  // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	
	if ((!is_object($values) && !is_array($values)) || count($values)<1) {
		$results = array("status"=>0, "message"=>'Sorry could not find an invitation for the email '.esc_html($email).'. Please check your invitation and enter the same email address. If you think there is an error contact the organizer. ');
	} else {
		$results = array("status"=>1, "message"=>"Success", "data"=>$values);
	}
	return $results;
}

// validate form and if ok, add
function cwsiPubProcessAddRsvpForm($values) {
	$ok = 1;	
	$msg = "";
	$results = array("status"=>1, "message"=>"Thank you for your response. Check for a confirmation email. It may be in your spam!");

	if (!is_email($values['email'])) {
		$results = array("status"=>0, "message"=>"That is not a valid email.");
		return $results;
	}

	// Retrieve invitee by email
	$results_get_invitee = cwsiPubGetRsvpByEmail($values['email'], $values['event_id']);
	if (!isset($results_get_invitee['status']) || $results_get_invitee['status'] == 0 || !isset($results_get_invitee['data']) || !is_array($results_get_invitee['data']) || count($results_get_invitee['data']) < 1) {
		if (isset($results_get_invitee['message']) && $results_get_invitee['message'] != "") {
			$results = array("status"=>0, "message"=>esc_html($results_get_invitee['message']));
		} else {
			$results = array("status"=>0, "message"=>"Could not retrieve invitation for that email. Are you sure you have the right email? Please try again. ");
		}
		return $results;
	}

	// exists, so now update
	$invitees = $results_get_invitee['data'];
	if (isset($invitees[0])) {
		$invitee = $invitees[0];
		$invitee_id = $invitees[0]->ID;
	}
	$values['invitee_id'] = $invitee_id;
	
	$results_update = cwsiPubUpdateInvites($values);
	if (!isset($results_update['status']) || $results_update['status'] == 0) {
		if (isset($results_update['message']) && $results_update['message'] != "") {
			$results = array("status"=>0, "message"=>esc_html($results_update['message']));
		} else {
			$results = array("status"=>0, "message"=>"Could not update your response. Please try again. ");
		}
		return $results;
	} 

	// If all is still good then send an email
	$results = cwsiPubInvitesSendEmail($values);
	if (isset($results['status']) && $results['status'] == 1) {
		$results['message'] = "Thank you for your response! Look for a confirmation email. It may go into your spam!";
	}
	return $results;
}

// Add a new invitee
function cwsiPubAddInvitee($values) {
	global $wpdb;
	$now = current_datetime()->format('Y-m-d');
	// validate
	if (!isset($values['full_name']) || $values['full_name'] == "" ||
		!isset($values['email']) || $values['email'] == "") {
			$results = array("status"=>0, "message"=>"Please enter your name and email.");
			return $results;
	}
	if (!is_email($values['email'])) {
			$results = array("status"=>0, "message"=>"That is not a valid email.");
			return $results;
	}

	$query = $wpdb->insert($wpdb->prefix.'cwsi_invites', array( 'full_name' => sanitize_text_field($values['full_name']), 'email' => sanitize_text_field($values['email']), 'num_guests' => sanitize_text_field($values['num_guests']), 'rsvp_answer' => sanitize_text_field($values['rsvp_answer']), 'comments' => sanitize_textarea_field($values['comments']), 'date_responded' => sanitize_text_field($now) ), array('%s', '%s', '%d', '%s', '%s', '%s') ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	if ($wpdb->insert_id < 0) {
		$results = array("status"=>0, "message"=>'Could not add new invitee. Error is '.$wpdb->last_error.'. ');
	} else {
		$results = array("status"=>1, "message"=>"Invitee added successfully.", "insert_id"=>$wpdb->insert_id);
	}
	return $results;
}

// Update rsvp
function cwsiPubUpdateInvites($values) {
	global $wpdb;
	$now = current_datetime()->format('Y-m-d');
	
	// validate
	if (!isset($values['full_name']) || $values['full_name'] == "" ||
		!isset($values['email']) || $values['email'] == "") {
			$results = array("status"=>0, "message"=>"Please enter your name and email.");
			return $results;
	}
	if (!isset($values['invitee_id']) || $values['invitee_id'] == "") {
			$results = array("status"=>0, "message"=>"Could not find your entry. Please try again.");
			return $results;
	}
	if (!is_email($values['email'])) {
			$results = array("status"=>0, "message"=>"That is not a valid email.");
			return $results;
	}
	// invites table first
	$query = $wpdb->update ( $wpdb->prefix.'cwsi_invites', array('full_name' => sanitize_text_field($values['full_name'])), array('ID' => sanitize_text_field($values['invitee_id'])), array('%s'), array( '%d' )); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	if (!$query) {
		if ($wpdb->last_error != "") {
			$results = array("status"=>0, "message"=>'Could not update your response. Error is '.$wpdb->last_error.'. ');
		} else {
			$results = array("status"=>1, "message"=>"Nothing to update.");
		}
	} else {
		$results = array("status"=>1, "message"=>"Your response was saved!");
	}
	// events_invite table next
	$query = $wpdb->update ( $wpdb->prefix.'cwsi_events_invitees', array('num_guests' => sanitize_text_field($values['num_guests']), 'rsvp_answer' => sanitize_text_field($values['rsvp_answer']), 'comments' => sanitize_textarea_field($values['comments']), 'date_responded' => sanitize_text_field($now)), array('event_id'=>sanitize_text_field($values['event_id']), 'invitee_id' => sanitize_text_field($values['invitee_id'])), array('%d', '%s', '%s', '%s'), array( '%d' )); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	if (!$query) {
		if ($wpdb->last_error != "") {
			$results = array("status"=>0, "message"=>'Could not update your response. Error is '.$wpdb->last_error.'. ');
		} else {
			$results = array("status"=>1, "message"=>"Nothing to update.");
		}
	} else {
		$results = array("status"=>1, "message"=>"Your response was saved!");
	}
	return $results;
}

function cwsiPubIsEventForID($event_id) {
	global $wpdb;
	$values = $wpdb->get_results($wpdb->prepare('SELECT ID FROM '.$wpdb->prefix.'cwsi_events WHERE ID=%d', array(sanitize_text_field($event_id))));  // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	if ((!is_object($values) && !is_array($values)) || count($values) < 1) {
		$message = "There is no event with that ID. Please check in the admin area. ";
	} else {
		$message = "";
	}
	return $message;
}
//////////////////////////////////
// Email FUNCTIONS
//////////////////////////////////

// send email to Charlene and Steve and then a confirmation one to the invitee
function cwsiPubInvitesSendEmail($values) {
	$results = array("status"=>1, "message"=>"Email sent.");
	// first the invitee
	$results_settings = cwsiPubInviteGetEmailSettings("confirmation");
	if (isset($results_settings) && isset($results_settings['status']) && $results_settings['status'] == 1 && isset($results_settings['data']) && count($results_settings['data']) > 0) {
		$settings = $results_settings['data'][0];
	} else {
		return $results_settings;
	}
	$required = array('email_fromname'=>"Email From Name", 'email_from'=>"Email From", 'email_subject'=>"Subject", 'email_body'=>"Body of Email if YES", 'email_body_2'=>"Body of Email if NO");
	foreach ($required as $field => $desc) {
		if (!isset($settings->$field) || $settings->$field == "") {
			$results = array('status'=>0, 'message'=>$desc." is a required field. Cannot send confirmation.");
			return $results;
		}
	}
	$email_from = $settings->email_from;
	if (!is_email($email_from)) {
		$results = array('status'=>0, 'message'=>"The FROM email $email_from is invalid. Cannot send confirmation.");
		return $results;
	}
	
	$headers = array("Content-Type: text/html; charset=UTF-8\r\n");
	$headers[] = "From: ".esc_html($settings->email_fromname).' <'.esc_html($email_from).">\r\n";

	if (isset($settings->email_replyto) && $settings->email_replyto != "" && is_email($settings->email_replyto)) {
		$headers[] = "Reply-to: ".$settings->email_replyto."\r\n";
	}
	if (isset($settings->email_cc) && $settings->email_cc != "" && is_email($settings->email_cc)) {
		$headers[] = "Cc: ".$settings->email_cc."\r\n";
	}
	if (isset($settings->email_bcc) && $settings->email_bcc != "" && is_email($settings->email_bcc)) {
		$headers[] = "Bcc: ".$settings->email_bcc."\r\n";
	}
	$subject = wp_unslash($settings->email_subject);	
	if (isset($values['email']) && is_email($values['email']) && $values['email'] != "") {
		$to = $values['email'];
	} else {
		$results = array('status'=>0, 'message'=>"That is not a valid email. Cannot send confirmation.");
		return $results;
	}
	if (isset($values['rsvp_answer']) && $values['rsvp_answer'] == "Yes") {
		$body = wp_unslash(nl2br($settings->email_body));
	} else {
		$body = wp_unslash(nl2br($settings->email_body_2));
	}
	$body = str_replace("[email]", $values['email'], $body);
	if (isset($values['full_name']) && $values['full_name'] != "") {
		$body = str_replace("[full_name]", $values['full_name'], $body);
	}
	if (isset($values['num_guests']) && $values['num_guests'] >= 1) {
		$body = str_replace("[num_guests]", $values['num_guests'], $body);
	}
	if (isset($values['comments']) && $values['comments'] != "") {
		$body = str_replace("[comments]", esc_html(wp_unslash($values['comments'])), $body);
	} else {
		$body = str_replace("[comments]", "", $body);
	}
	if (isset($values['rsvp_answer']) && $values['rsvp_answer'] != "") {
		$body = str_replace("[rsvp_answer]", esc_html($values['rsvp_answer']), $body);
	}
	$sent = @wp_mail($to, $subject, $body, $headers);
	if (!$sent) {
		$results = array("status"=>0, "message"=>"Could not send the confirmation email. Please contact the host. ");
	}
	return $results;
}

///////////////////////////////////////////////////////////////////////////
// SETTINGS FUNCTIONS
//////////////////////////////////////////////////////////////////////////
// retrieve the email settings
function cwsiPubInviteGetEmailSettings($message_type) {
	global $wpdb;
	if (!isset($message_type) || ($message_type != "invite" && $message_type != "confirmation")) {
		$results = array("status"=>0, "message"=>'Could not retrieve email settings for invalid message type '.$message_type.'. ');
	}
	$values = $wpdb->get_results($wpdb->prepare('SELECT * FROM '.$wpdb->prefix.'cwsi_invitesettings WHERE message_type=%s ORDER BY date_updated DESC LIMIT 1', array($message_type)) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	if (!is_object($values) && !is_array($values)) {
		$results = array("status"=>0, "message"=>'Failed to find email settings. Error is '.$wpdb->last_error.'. ');
	} else {
		$results = array("status"=>1, "message"=>"Success", "data"=>$values);
	}
	return $results;
}