<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://charlenesweb.ca
 * @since      1.0.0
 *
 * @package    cwsi_invites
 * @subpackage cwsi_invites/admin
 */
/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    cwsi_invites
 * @subpackage cwsi_invites/admin
 * @author     Charlene Copeland <charlene@charlenesweb.ca>
 */
#[AllowDynamicProperties]
class cwsi_invites_Admin {
    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $cwsi_invites    The ID of this plugin.
     */
    private $cwsi_invites;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $cwsi_invites       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct( $cwsi_invites, $version ) {
        $this->plugin_name = $cwsi_invites;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in cwsi_invites_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The cwsi_invites_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */
        wp_enqueue_style(
            $this->plugin_name,
            plugin_dir_url( __FILE__ ) . 'css/cwsi-invites-admin.css',
            array(),
            $this->version,
            'all'
        );
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        wp_enqueue_script(
            $this->plugin_name,
            plugin_dir_url( __FILE__ ) . 'js/cwsi-invites-admin.js',
            array('jquery'),
            $this->version,
            false
        );
    }

    /**
     * create menu structure
     */
    public function add_menu_pages() {
        // top_level_menu
        add_menu_page(
            'Invitations and RSVPs',
            'Invitations and RSVPs',
            'edit_posts',
            'cwsi_invites_top_level',
            null,
            'dashicons-edit',
            '10.842015'
        );
        // Rename first
        $parent_slug = 'cwsi_invites_top_level';
        $page_app_suffix = add_submenu_page(
            $parent_slug,
            __( 'Invitations Intro', 'cwsi-invites' ),
            __( 'Invitations Intro', 'cwsi-invites' ),
            'edit_posts',
            'cwsi_invites_top_level',
            array($this, 'top_level_invites')
        );
        // Events
        $page_worker_suffix = add_submenu_page(
            'cwsi_invites_top_level',
            __( 'Events', 'cwsi-invites' ),
            '' . __( 'Events', 'cwsi-invites' ),
            'manage_options',
            'cwsi_invites_events',
            array($this, 'cwsiEvents_page')
        );
        // Invitees
        $page_worker_suffix = add_submenu_page(
            'cwsi_invites_top_level',
            __( 'Invitees', 'cwsi-invites' ),
            '' . __( 'Invitees', 'cwsi-invites' ),
            'manage_options',
            'cwsi_invites_responses',
            array($this, 'cwsiInvites_page')
        );
        // initial upload
        $page_worker_suffix = add_submenu_page(
            'cwsi_invites_top_level',
            __( 'Upload Invitees', 'cwsi-invites' ),
            '' . __( 'Upload Invitees', 'cwsi-invites' ),
            'manage_options',
            'cwsi_invites_upload',
            array($this, 'cwsiinvitesuploads_page')
        );
        // Design Invitation Email
        $page_worker_suffix = add_submenu_page(
            'cwsi_invites_top_level',
            __( 'Design Invitation Email', 'cwsi-invites' ),
            '' . __( 'Design Invitation Email', 'cwsi-invites' ),
            'manage_options',
            'cwsi_invites_design_invite_email',
            array($this, 'cwsiinvitedesigninviteemail_page')
        );
        // Design Confirmation Email
        $page_worker_suffix = add_submenu_page(
            'cwsi_invites_top_level',
            __( 'Design Confirmation Email', 'cwsi-invites' ),
            '' . __( 'Design Confirmation Email', 'cwsi-invites' ),
            'manage_options',
            'cwsi_invites_confirmation_invite_email',
            array($this, 'cwsiinvitedesignconfirmationemail_page')
        );
        // Send emails
        $page_worker_suffix = add_submenu_page(
            'cwsi_invites_top_level',
            __( 'Send Invitations', 'cwsi-invites' ),
            '' . __( 'Send Invitations', 'cwsi-invites' ),
            'manage_options',
            'cwsi_invites_sendemails',
            array($this, 'cwsiinvitessendemails_page')
        );
        // View Responses
        $page_worker_suffix = add_submenu_page(
            'cwsi_invites_top_level',
            __( 'View Responses', 'cwsi-invites' ),
            '' . __( 'View Responses', 'cwsi-invites' ),
            'manage_options',
            'cwsi_invites_viewresponses',
            array($this, 'cwsiinvitesviewresponses_page')
        );
        // END PREMIUM ONLY
    }

    public function top_level_invites() {
        if ( is_user_logged_in() ) {
            echo '<h1>Invitations and RSVPs</h1>
			<p>Welcome! Create your event, then add a list of invitees, set up an email invitations, set up a confirmation email and then track responses. </p>
			<h3>The Setup</h3>
			<ol>
				<li>Add a new event or view, edit and delete existing events at <a href="/wp-admin/admin.php?page=cwsi_invites_events">Events</a>. ';
            if ( ci_fs()->is_not_paying() ) {
                echo 'With the Free Plan you are limited to 1 event. ';
                echo sprintf( '<a href="%s"><small>Unlock Premium Plan</small></a>', ci_fs()->checkout_url() );
            } else {
                echo 'With the Premium plan, you may add more than 1 event. ';
            }
            echo '
				</li>
				<li>Add a new invitee or view, edit and delete existing invitees at <a href="/wp-admin/admin.php?page=cwsi_invites_responses">Invitees</a>. ';
            if ( ci_fs()->is_not_paying() ) {
                echo 'If you upgrade to the Premium Plan, you may export to CSV. ';
            } else {
                echo 'With your Premium Plan, you may export the invitees to a CSV. ';
            }
            echo '
				</li>
				<li>Add invitees through a bulk upload at <a href="/wp-admin/admin.php?page=cwsi_invites_upload">Upload Invitees</a>. </li>
				<li>Design the invitation email and send a test at <a href="/wp-admin/admin.php?page=cwsi_invites_design_invite_email">Design Invitation Email</a></li>
				<li>Design the confirmation email for a "Yes" and a "No" response, and send a test at <a href="/wp-admin/admin.php?page=cwsi_invites_confirmation_invite_email">Design Confirmation Email</a>. Set up the RSVP button as a custom link by using /?email=[email] in the URL. </li>
				<li>Send invitations to anyone who has not been sent one yet at <a href="/wp-admin/admin.php?page=cwsi_invites_sendemails">Send Invitations</a></li>
				<li>View the Yes, No, and No Answers at <a href="/wp-admin/admin.php?page=cwsi_invites_viewresponses">View Responses</a></li>';
            if ( ci_fs()->is_not_paying() ) {
                echo '
					<li>Upgrade to the Premium Plan to be able to send Reminder Emails and to send Save the Date emails. ';
                echo ' ' . sprintf( '<a href="%s"><small>Unlock Premium Plan</small></a>', ci_fs()->checkout_url() ) . ' ';
            } else {
                echo '<li>With your Premium Plan, you can send reminders to anyone who has received an invitation at <a href="/wp-admin/admin.php?page=cwsi_invites_viewresponses">Send Reminders</a> and send SAVE THE DATE emails to anyone on the invitee list. You can filter out any invitees that have already received an email, select individuals, or choose to send to all invitees. <a href="/wp-admin/admin.php?page=cwsi_invites_savethedate">Save The Date</a>. ';
            }
            echo '
				</li>
			</ol>
			<h3>On the Website</h3>
			<ol>
				<li>Create an RSVP page and use shortcode [cwsiInviteAnswerForm event_id="<em>put your event id here. Get from the Events page</em>"] to include the response form. You will need one page per event. </li>
				<li>Optionally include the shortcode [cwsiInviteShowResponses event_id="<em>put id number here</em>"] to display responses so far to the website visitor. </li>
			</ol>';
            if ( ci_fs()->is_not_paying() ) {
                // Adds a marketing sections with a link to in-dashboard pricing page.
                echo '<section><h1>Awesome Features in The Premium Version</h1>
				<p>Unlock the Premium Plan to: </p>
				<ol>
					<li>Have more than 1 event</li>
					<li>Export invitees to CSV</li>
					<li>Send Reminder emails to anyone that has not responded.</li>
					<li>Send a Save the Date email to anyone on the invitee list. </li>
				</ol>
				<p>';
                echo sprintf( '<a href="%s">Upgrade Now!</a>', ci_fs()->get_upgrade_url() );
                echo '</p></section>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    public function cwsiEvents_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>Events</h1>';
            // initialize vars
            $menu_slug = "cwsi_invites_events";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // Display functions
                require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-admin-display.php';
                // handle response from add
                if ( isset( $_POST['add_event'] ) ) {
                    check_admin_referer( 'show-add-event-responses', 'showaddeventresponsesform' );
                    $results_add_response = cwsiAdminAddEvent( $_POST );
                } else {
                    $results_add_response = array();
                }
                // handle response from save
                $results_save_response = array();
                if ( isset( $_POST['cwsevent_id'] ) ) {
                    check_admin_referer( 'show-event-responses-' . sanitize_text_field( wp_unslash( $_POST['cwsevent_id'] ) ), 'showeventform-' . sanitize_text_field( wp_unslash( $_POST['cwsevent_id'] ) ) );
                    if ( isset( $_POST['saveevent'] ) ) {
                        $results_save_response = cwsiAdminSaveEvent( $_POST );
                    } elseif ( isset( $_POST['deleteevent'] ) ) {
                        $results_save_response = cwsiAdminDeleteEvent( sanitize_text_field( wp_unslash( $_POST['cwsevent_id'] ) ) );
                    }
                }
                $message = cwsiIsEventsTable();
                // check if table exists since added in later version
                $ctr_events = 0;
                if ( $message == "" ) {
                    // Get current Events
                    $results = cwsiGetEvents();
                    if ( isset( $results['data'] ) && is_array( $results['data'] ) && count( $results['data'] ) > 0 ) {
                        $ctr_events = count( $results['data'] );
                    }
                    // Display forms
                    cwsiShowAddEventPage( $menu_slug, $results_add_response, $ctr_events );
                    cwsiShowEventsPage( $menu_slug, $results, $results_save_response );
                } else {
                    echo wp_kses( wp_unslash( nl2br( $message ) ), cwsi_invites_allowed_html() );
                }
            } else {
                echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    public function cwsiInvites_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>Invitations</h1>
			<h2>See Invitees</h2>';
            // initialize vars
            $menu_slug = "cwsi_invites_responses";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // Display functions
                require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-admin-display.php';
                // handle submission from the filter form
                if ( isset( $_POST['search_event_id'] ) ) {
                    check_admin_referer( 'show-filter-events', 'showfiltereventsform' );
                    $search_event_id = sanitize_text_field( wp_unslash( $_POST['search_event_id'] ) );
                } elseif ( isset( $_POST['cwsevent_id'] ) ) {
                    $search_event_id = sanitize_text_field( wp_unslash( $_POST['cwsevent_id'] ) );
                } else {
                    $search_event_id = "";
                }
                // show add invitee
                if ( isset( $_POST['add_invitee'] ) ) {
                    check_admin_referer( 'show-add-invitee-responses', 'showaddinviteeresponsesform' );
                    $results_add_response = cwsiAdminAddInvitee( $_POST );
                    if ( isset( $_POST['event_id'] ) ) {
                        $search_event_id = sanitize_text_field( wp_unslash( $_POST['event_id'] ) );
                    }
                } else {
                    $results_add_response = array();
                }
                if ( isset( $_POST['cwsinvitee_id'] ) ) {
                    check_admin_referer( 'show-invitee-responses-' . sanitize_text_field( wp_unslash( $_POST['cwsinvitee_id'] ) ), 'showinviteeresponsesform-' . sanitize_text_field( wp_unslash( $_POST['cwsinvitee_id'] ) ) );
                    $search_event_id = sanitize_text_field( wp_unslash( $_POST['event_id'] ) );
                    if ( isset( $_POST['saveinvite'] ) ) {
                        $results_save_response = cwsiAdminSaveInvitee( $_POST );
                    } elseif ( isset( $_POST['deleteinvite'] ) ) {
                        $results_save_response = cwsiAdminDeleteEventInvitee( sanitize_text_field( wp_unslash( $_POST['event_id'] ) ), sanitize_text_field( wp_unslash( $_POST['cwsinvitee_id'] ) ) );
                    }
                } else {
                    $results_save_response = array();
                }
                // Get all events and then set search id
                $results_events = cwsiGetEvents();
                $search_event_name = "";
                if ( isset( $results_events['status'] ) && $results_events['status'] == 1 && isset( $results_events['data'] ) ) {
                    $events = $results_events['data'];
                    if ( is_array( $events ) && count( $events ) >= 1 ) {
                        if ( $search_event_id == "" ) {
                            $search_event_id = $events[0]->ID;
                            $search_event_name = $events[0]->event_name;
                        } else {
                            foreach ( $events as $event ) {
                                if ( $event->ID == $search_event_id ) {
                                    $search_event_name = $event->event_name;
                                    break;
                                }
                            }
                        }
                    }
                }
                // Get current invitees and their responses for selected event
                if ( $search_event_id != "" ) {
                    $results = cwsiGetInvitesResponses( $search_event_id );
                } else {
                    $results = array(
                        'status'  => 0,
                        'message' => "First select an event. ",
                    );
                }
                // END PREMIUM ONLY
                // Display form
                cwsiShowFilterEvents( $menu_slug, $results_events, $search_event_id );
                cwsiShowAddInvitesPage(
                    $menu_slug,
                    $results_add_response,
                    $search_event_id,
                    $search_event_name
                );
                cwsiShowInvitesResponsesPage(
                    $menu_slug,
                    $results,
                    $results_save_response,
                    $search_event_id,
                    $search_event_name
                );
            } else {
                echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    public function cwsiinvitesuploads_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>Upload Invites</h1>';
            // initialize vars
            $menu_slug = "cwsi_invites_upload";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // Handle the upload
                if ( isset( $_POST['cwsi_file_name'] ) ) {
                    check_admin_referer( 'show-invites-upload', 'showinvitesuploadform' );
                    $results_save_import = cwsiSaveInvitesImport( $_POST, $_FILES );
                } else {
                    $results_save_import = array();
                }
                // Display functions
                // Get all events
                $results_events = cwsiGetEvents();
                // Display form
                require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-admin-display.php';
                cwsiShowInvitesUpload( $menu_slug, $results_save_import, $results_events );
            } else {
                echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    // design the invitation email
    public function cwsiinvitedesigninviteemail_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>Design the Invitation</h1>
			<h2>Design the email that will be sent out by email</h2>';
            $menu_slug = "cwsi_invites_design_invite_email";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // handle save if form was submitted
                if ( isset( $_POST['settings_id'] ) ) {
                    check_admin_referer( 'save-design-invite-settings', 'savedesigninvitesettingsform' );
                    $results_save = cwsiAdminInviteProcessEmailSettings( $_POST );
                } else {
                    $results_save = array();
                }
                $message = cwsiIsSettingsTableUpdated();
                // check if table exists since added in later version
                if ( $message == "" ) {
                    // get all events
                    $results_events = cwsiGetEvents();
                    $search_event_id = 1;
                    // default
                    if ( isset( $results_events['status'] ) && $results_events['status'] == 1 && isset( $results_events['data'] ) ) {
                        $events = $results_events['data'];
                        if ( isset( $_POST['search_event_id'] ) ) {
                            // search form
                            $search_event_id = sanitize_text_field( wp_unslash( $_POST['search_event_id'] ) );
                        } elseif ( isset( $_POST['event_id'] ) ) {
                            // email form
                            $search_event_id = sanitize_text_field( wp_unslash( $_POST['event_id'] ) );
                        } elseif ( is_array( $events ) && count( $events ) >= 1 ) {
                            $search_event_id = $events[0]->ID;
                        }
                    }
                    // get current settings
                    $results = cwsiAdminInviteGetEmailSettings( "invite", $search_event_id );
                    // do they want to send a test email?
                    if ( isset( $_POST['send_email_test'] ) && isset( $results['status'] ) && $results['status'] == 1 ) {
                        check_admin_referer( 'send-test-email', 'sendtestemailform' );
                        $results_send = cwsiAdminInviteSendTestEmail( $_POST, $results );
                    } else {
                        $results_send = array();
                    }
                    // Display functions
                    require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-admin-display.php';
                    cwsiShowFilterEvents( $menu_slug, $results_events, $search_event_id );
                    cwsiAdminInviteShowInviteSettings(
                        $menu_slug,
                        $results,
                        $results_save,
                        $results_send,
                        $search_event_id
                    );
                    cwsiAdminShowSendTestInviteEmail( $menu_slug, $results, "invite" );
                } else {
                    echo wp_kses( wp_unslash( nl2br( $message ) ), cwsi_invites_allowed_html() );
                }
            } else {
                echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    // design the confirmation email
    public function cwsiinvitedesignconfirmationemail_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>Design the Confirmation Email</h1>
			<h2>Design the email that will be sent out by email once the invitee has RSVPd</h2>';
            $menu_slug = "cwsi_invites_confirmation_invite_email";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // handle save if form was submitted
                if ( isset( $_POST['settings_id'] ) ) {
                    check_admin_referer( 'save-design-confirmation-settings', 'savedesignconfirmationsettingsform' );
                    $results_save = cwsiAdminInviteProcessEmailSettings( $_POST );
                } else {
                    $results_save = array();
                }
                $message = cwsiIsSettingsTableUpdated();
                // check if table exists since added in later version
                if ( $message == "" ) {
                    // get all events
                    $results_events = cwsiGetEvents();
                    $search_event_id = 1;
                    //default
                    if ( isset( $results_events['status'] ) && $results_events['status'] == 1 && isset( $results_events['data'] ) ) {
                        $events = $results_events['data'];
                        if ( isset( $_POST['search_event_id'] ) ) {
                            // search form
                            $search_event_id = sanitize_text_field( wp_unslash( $_POST['search_event_id'] ) );
                        } elseif ( isset( $_POST['event_id'] ) ) {
                            // email form
                            $search_event_id = sanitize_text_field( wp_unslash( $_POST['event_id'] ) );
                        } elseif ( is_array( $events ) && count( $events ) >= 1 ) {
                            $search_event_id = $events[0]->ID;
                        }
                    }
                    // get current settings
                    $results = cwsiAdminInviteGetEmailSettings( "confirmation", $search_event_id );
                    // do they want to send a test email?
                    if ( isset( $_POST['send_email_test'] ) && isset( $results['status'] ) && $results['status'] == 1 ) {
                        check_admin_referer( 'send-test-email', 'sendtestemailform' );
                        $results_send = cwsiAdminInviteSendTestEmail( $_POST, $results );
                    } else {
                        $results_send = array();
                    }
                    // Display functions
                    require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-admin-display.php';
                    cwsiShowFilterEvents( $menu_slug, $results_events, $search_event_id );
                    cwsiAdminInviteShowConfirmationSettings(
                        $menu_slug,
                        $results,
                        $results_save,
                        $results_send,
                        $search_event_id
                    );
                    cwsiAdminShowSendTestInviteEmail( $menu_slug, $results, "confirmation" );
                } else {
                    echo wp_kses( wp_unslash( nl2br( $message ) ), cwsi_invites_allowed_html() );
                }
            } else {
                echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    // send invitations by email
    public function cwsiinvitessendemails_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>Send Invitations by Email</h1>
			<h2>Loop through and send invitations to people who haven&rsquo;t had them sent yet. </h2>';
            // initialize vars
            $menu_slug = "cwsi_invites_sendemails";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // Display functions
                require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-admin-display.php';
                // get all events
                $results_events = cwsiGetEvents();
                $search_event_id = 1;
                //default
                if ( isset( $results_events['status'] ) && $results_events['status'] == 1 && isset( $results_events['data'] ) ) {
                    $events = $results_events['data'];
                    if ( isset( $_POST['search_event_id'] ) ) {
                        // search form
                        $search_event_id = sanitize_text_field( wp_unslash( $_POST['search_event_id'] ) );
                    } elseif ( isset( $_POST['event_id'] ) ) {
                        // email form
                        $search_event_id = sanitize_text_field( wp_unslash( $_POST['event_id'] ) );
                    } elseif ( is_array( $events ) && count( $events ) >= 1 ) {
                        $search_event_id = $events[0]->ID;
                    }
                }
                $results_settings = cwsiAdminInviteGetEmailSettings( "invite", sanitize_text_field( $search_event_id ) );
                // Handle the submit
                if ( isset( $_POST['cws_send_email'] ) ) {
                    check_admin_referer( 'show-invites-send-emails', 'showinvitessendemailsform' );
                    $results = cwsiAdminInvitesSendEmail( $results_settings );
                } else {
                    $results = array();
                }
                $results_recips = cwsiGetInvitesNotEmailed( $search_event_id );
                // Display form
                cwsiShowFilterEvents( $menu_slug, $results_events, $search_event_id );
                cwsiShowInvitesSendEmails(
                    $menu_slug,
                    $results,
                    $results_settings,
                    $results_recips
                );
            } else {
                echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    // View Responses
    public function cwsiinvitesviewresponses_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>View Responses</h1>
			<h2>Show the "Yes&rsquo;s", "No&rsquo;s" and "No Response Yet&rsquo;s"</h2>';
            // initialize vars
            $menu_slug = "cwsi_invites_viewresponses";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // Display functions
                require_once plugin_dir_path( __FILE__ ) . 'partials/cwsi-invites-admin-display.php';
                // get all events
                $results_events = cwsiGetEvents();
                $search_event_id = 1;
                //default
                if ( isset( $results_events['status'] ) && $results_events['status'] == 1 && isset( $results_events['data'] ) ) {
                    $events = $results_events['data'];
                    if ( isset( $_POST['search_event_id'] ) ) {
                        // search form
                        check_admin_referer( 'show-filter-events', 'showfiltereventsform' );
                        $search_event_id = sanitize_text_field( wp_unslash( $_POST['search_event_id'] ) );
                    } elseif ( isset( $_POST['event_id'] ) ) {
                        // email form
                        $search_event_id = sanitize_text_field( wp_unslash( $_POST['event_id'] ) );
                    } elseif ( is_array( $events ) && count( $events ) >= 1 ) {
                        $search_event_id = $events[0]->ID;
                    }
                }
                $results_yes = cwsiGetYesInvitesResponses( $search_event_id );
                $results_no = cwsiGetNoInvitesResponses( $search_event_id );
                $results_null = cwsiGetNullInvitesResponses( $search_event_id );
                // Display form
                cwsiShowFilterEvents( $menu_slug, $results_events, $search_event_id );
                cwsiShowInvitesViewResponses(
                    $menu_slug,
                    $results_yes,
                    $results_no,
                    $results_null
                );
            } else {
                echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    public function cwsiinvitessendreminders_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>Send Reminder Emails</h1>
			<h2>Send a reminder to anyone who has not responded. </h2>';
            // initialize vars
            $menu_slug = "cwsi_invites_sendreminders";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // END is premium
            } else {
                echo '<p class="failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

    // send save the date by email
    public function cwsiinvitessavethedate_page() {
        if ( is_user_logged_in() ) {
            echo '<h1>Send SAVE THE DATE Emails</h1>
			<h2>Send an email to anyone on the invite list. </h2>';
            // initialize vars
            $menu_slug = "cwsi_invites_savethedate";
            // get roles
            global $current_user;
            wp_get_current_user();
            $roles = $current_user->roles;
            if ( in_array( "administrator", $roles ) ) {
                // END is premium
            } else {
                echo '<p class="failmsg">You are not authorized to be here. </p>';
            }
        } else {
            echo '<p class="cwsi_failmsg">You are not authorized to be here. </p>';
        }
    }

}

//////////////////////////////////
// cwsi_invites Table FUNCTIONS
//////////////////////////////////
// Retrieve all invitees for the given event
function cwsiGetInvitesResponses(  $event_id  ) {
    global $wpdb;
    if ( $event_id < 0 ) {
        $values = $wpdb->get_results( 'SELECT * FROM ' . $wpdb->prefix . 'cwsi_invites ORDER BY full_name, email' );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    } else {
        $values = $wpdb->get_results( $wpdb->prepare( 'SELECT a.event_id, a.invitee_id, a.num_guests, a.rsvp_answer, a.comments, a.personal_note, a.admin_notes, a.date_save_sent, a.date_email_sent, a.date_responded, b.ID, b.full_name, b.email, b.email2, b.date_updated FROM ' . $wpdb->prefix . 'cwsi_events_invitees as a JOIN ' . $wpdb->prefix . 'cwsi_invites as b ON a.invitee_id=b.ID WHERE event_id=%d ORDER BY b.full_name, b.email', array(sanitize_text_field( $event_id )) ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    }
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'Failed to find Invitees for this event. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Success",
            "data"    => $values,
        );
    }
    return $results;
}

function cwsiGetInvitesNotEmailed(  $event_id  ) {
    global $wpdb;
    $values = $wpdb->get_results( $wpdb->prepare( 'SELECT a.event_id, a.invitee_id, a.num_guests, a.rsvp_answer, a.comments, a.personal_note, a.admin_notes, a.date_email_sent, a.date_responded, b.ID, b.full_name, b.email, b.email2, b.date_updated FROM ' . $wpdb->prefix . 'cwsi_events_invitees as a JOIN ' . $wpdb->prefix . 'cwsi_invites as b ON a.invitee_id=b.ID WHERE event_id=%d AND (date_email_sent IS NULL OR date_email_sent="0000-00-00") ORDER BY full_name, email', array(sanitize_text_field( $event_id )) ) );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'Failed to find Invitees not emailed. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Success",
            "data"    => $values,
        );
    }
    return $results;
}

// Retrieve invitees who are coming
function cwsiGetYesInvitesResponses(  $event_id  ) {
    global $wpdb;
    if ( $event_id < 0 ) {
        $values = $wpdb->get_results( 'SELECT * FROM ' . $wpdb->prefix . 'cwsi_invites WHERE rsvp_answer="Yes" ORDER BY full_name, email' );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    } else {
        $values = $wpdb->get_results( $wpdb->prepare( 'SELECT a.event_id, a.invitee_id, a.num_guests, a.rsvp_answer, a.comments, a.personal_note, a.admin_notes, a.date_email_sent, a.date_responded, b.ID, b.full_name, b.email, b.email2, b.date_updated FROM ' . $wpdb->prefix . 'cwsi_events_invitees as a JOIN ' . $wpdb->prefix . 'cwsi_invites as b ON a.invitee_id=b.ID WHERE event_id=%d AND rsvp_answer="Yes" ORDER BY full_name, email', array(sanitize_text_field( $event_id )) ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    }
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'Failed to find Invitees. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Success",
            "data"    => $values,
        );
    }
    return $results;
}

// Retrieve invitees who are NOT coming
function cwsiGetNoInvitesResponses(  $event_id  ) {
    global $wpdb;
    if ( $event_id < 0 ) {
        $values = $wpdb->get_results( 'SELECT * FROM ' . $wpdb->prefix . 'cwsi_invites WHERE rsvp_answer="No" ORDER BY full_name, email' );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    } else {
        $values = $wpdb->get_results( $wpdb->prepare( 'SELECT a.event_id, a.invitee_id, a.num_guests, a.rsvp_answer, a.comments, a.personal_note, a.admin_notes, a.date_email_sent, a.date_responded, b.ID, b.full_name, b.email, b.email2, b.date_updated FROM ' . $wpdb->prefix . 'cwsi_events_invitees as a JOIN ' . $wpdb->prefix . 'cwsi_invites as b ON a.invitee_id=b.ID WHERE event_id=%d AND rsvp_answer="No" ORDER BY full_name, email', array(sanitize_text_field( $event_id )) ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    }
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'Failed to find Invitees. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Success",
            "data"    => $values,
        );
    }
    return $results;
}

// Retrieve invitees who have not answered
function cwsiGetNullInvitesResponses(  $event_id  ) {
    global $wpdb;
    if ( $event_id < 0 ) {
        $values = $wpdb->get_results( 'SELECT * FROM ' . $wpdb->prefix . 'cwsi_invites (date_responded IS NULL OR date_responded="0000-00-00") AND (rsvp_answer IS NULL OR rsvp_answer="") ORDER BY full_name, email' );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    } else {
        $values = $wpdb->get_results( $wpdb->prepare( 'SELECT a.event_id, a.invitee_id, a.num_guests, a.rsvp_answer, a.comments, a.personal_note, a.admin_notes, a.date_email_sent, a.date_responded, b.ID, b.full_name, b.email, b.email2, b.date_updated FROM ' . $wpdb->prefix . 'cwsi_events_invitees as a JOIN ' . $wpdb->prefix . 'cwsi_invites as b ON a.invitee_id=b.ID WHERE event_id=%d AND (date_responded IS NULL OR date_responded="0000-00-00") AND (rsvp_answer IS NULL OR rsvp_answer="") ORDER BY full_name, email', array(sanitize_text_field( $event_id )) ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    }
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'Failed to find Invitees. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Success",
            "data"    => $values,
        );
    }
    return $results;
}

function cwsiAdminAddInvitee(  $values  ) {
    global $wpdb;
    // validate
    $fields = array(
        'full_name'  => "Name",
        'email'      => "Email",
        'num_guests' => "Number of Guests",
        'event_id'   => "Event",
    );
    foreach ( $fields as $field => $name ) {
        if ( !isset( $values[$field] ) || $values[$field] == "" ) {
            $results = array(
                "status"  => 0,
                "message" => $name . " is required.",
            );
            return $results;
        }
    }
    // check if already in invitees list
    $exists = $wpdb->get_results( $wpdb->prepare( 'SELECT email, ID FROM ' . $wpdb->prefix . 'cwsi_invites WHERE email=%s', array(sanitize_text_field( $values['email'] )) ) );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( is_object( $exists ) || is_array( $exists ) ) {
        if ( isset( $exists[0] ) ) {
            $invitee_id = $exists[0]->ID;
            // already assigned to this event?
            $assigned = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM ' . $wpdb->prefix . 'cwsi_events_invitees WHERE event_id=%d AND invitee_id=%d', array($values['event_id'], $invitee_id) ) );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            if ( is_object( $assigned ) || is_array( $assigned ) ) {
                if ( isset( $assigned[0] ) ) {
                    $results = array(
                        'status'  => 0,
                        'message' => sanitize_text_field( $values['email'] ) . ' has already been added to this event. ',
                    );
                    return $results;
                }
            }
        }
    }
    if ( !isset( $values['comments'] ) || $values['comments'] == "" ) {
        $values['comments'] = "";
    }
    // add to invitees table?
    if ( !isset( $invitee_id ) ) {
        $query = $wpdb->insert( $wpdb->prefix . 'cwsi_invites', array(
            'full_name' => sanitize_text_field( $values['full_name'] ),
            'email'     => sanitize_text_field( $values['email'] ),
        ), array('%s', '%s') );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        if ( $wpdb->insert_id < 0 ) {
            $results = array(
                'status'    => 0,
                'message'   => 'Could not add invitee ' . sanitize_text_field( $values['email'] ) . '. Error is ' . sanitize_text_field( $wpdb->last_error ),
                'insert_id' => 0,
            );
            return $results;
        }
        $invitee_id = $wpdb->insert_id;
    }
    if ( !isset( $invitee_id ) ) {
        $results = array(
            'status'    => 0,
            'message'   => 'Could not add invitee ' . sanitize_text_field( $values['email'] ) . '. ',
            'insert_id' => 0,
        );
    } else {
        // now assign it
        $results = cwsiAdminSaveInviteeToEvent( $values['event_id'], $invitee_id, $values );
    }
    return $results;
}

// save to invite table and to event_invites table
function cwsiAdminSaveInvitee(  $values  ) {
    global $wpdb;
    $now = current_datetime()->format( 'Y-m-d' );
    // validate
    $fields = array(
        'cwsinvitee_id' => "Invitee ID",
        'event_id'      => "Event ID",
        'full_name'     => "Name",
        'email'         => "Email",
        'num_guests'    => "Number of Guests",
    );
    foreach ( $fields as $field => $name ) {
        if ( !isset( $values[$field] ) || $values[$field] == "" ) {
            $results = array(
                "status"  => 0,
                "message" => $name . " is required.",
            );
            return $results;
        }
    }
    if ( !is_email( $values['email'] ) ) {
        $results = array(
            "status"  => 0,
            "message" => "That is not a valid email.",
        );
        return $results;
    }
    if ( !isset( $values['comments'] ) || $values['comments'] == "" ) {
        $values['comments'] = "";
    }
    $query = $wpdb->update(
        $wpdb->prefix . 'cwsi_invites',
        array(
            'full_name' => sanitize_text_field( $values['full_name'] ),
            'email'     => sanitize_text_field( $values['email'] ),
        ),
        array(
            'ID' => sanitize_text_field( $values['cwsinvitee_id'] ),
        ),
        array('%s', '%s'),
        array('%d')
    );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query ) {
        if ( $wpdb->last_error != "" ) {
            $results = array(
                "status"  => 0,
                "message" => "Invitee " . esc_html( $values['email'] ) . " was NOT updated. Error is " . $wpdb->last_error . '. ',
            );
            return $results;
        } else {
            $results = array(
                "status"  => 1,
                "message" => "",
            );
        }
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Invitee " . esc_html( $values['email'] ) . " was updated.",
        );
    }
    // events_invite table
    $query = $wpdb->update(
        $wpdb->prefix . 'cwsi_events_invitees',
        array(
            'num_guests'      => sanitize_text_field( $values['num_guests'] ),
            'rsvp_answer'     => sanitize_text_field( $values['rsvp_answer'] ),
            'comments'        => sanitize_textarea_field( $values['comments'] ),
            'personal_note'   => sanitize_text_field( $values['personal_note'] ),
            'admin_notes'     => sanitize_text_field( $values['admin_notes'] ),
            'date_email_sent' => sanitize_text_field( $values['date_email_sent'] ),
            'date_responded'  => sanitize_text_field( $values['date_responded'] ),
        ),
        array(
            'event_id'   => $values['event_id'],
            'invitee_id' => $values['cwsinvitee_id'],
        ),
        array(
            '%d',
            '%s',
            '%s',
            '%s',
            '%s',
            '%s',
            '%s'
        ),
        array('%d', '%d')
    );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query ) {
        if ( $wpdb->last_error != "" ) {
            $results = array(
                "status"  => 0,
                "message" => "Invitee " . esc_html( $values['email'] ) . " NOT updated for this event. Error is " . $wpdb->last_error . '. ',
            );
        } else {
            $results = array(
                "status" => 1,
            );
        }
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Invitee " . esc_html( $values['email'] ) . " was updated for this event.",
        );
    }
    return $results;
}

function cwsiAdminDeleteInvitee(  $cwsinvitee_id  ) {
    global $wpdb;
    if ( !isset( $cwsinvitee_id ) || $cwsinvitee_id < 0 ) {
        return array(
            'status' => 0,
            'No invite selected for deletion.',
        );
    }
    $query = $wpdb->delete( $wpdb->prefix . "cwsi_invites", array(
        'ID' => sanitize_text_field( $cwsinvitee_id ),
    ), array('%d') );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query && $wpdb->last_error != "" ) {
        $results = array(
            "status"  => 0,
            "message" => 'Could not delete invite. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => 'Successfully deleted invite. ',
        );
    }
    return $results;
}

function cwsiAdminInviteDownloadInvitees(  $results  ) {
    $ct = "";
    if ( isset( $results['status'] ) && $results['status'] == 1 && isset( $results['data'] ) ) {
        $data = $results['data'];
        $data_str = "Name,Email,Coming?,Num Guests,Comments, Admin Notes, Date Emailed, Date Responded\n";
        foreach ( $data as $row ) {
            $data_str .= esc_html( $row->full_name ) . "," . esc_html( $row->email ) . "," . esc_html( $row->rsvp_answer ) . "," . esc_html( $row->num_guests ) . "," . esc_html( wp_unslash( $row->comments ) ) . "," . esc_html( $row->admin_notes ) . "," . esc_html( $row->date_email_sent ) . "," . esc_html( $row->date_responded ) . "\n";
        }
        if ( $data_str == "" ) {
            $ct .= '<p class="cwsi_failmsg">No records found!</p>';
        } else {
            $upload_dir = wp_upload_dir();
            // Array of key => value pairs
            $time = current_datetime()->format( 'YmdHi' );
            $file_name = "invitees-" . $time . '.csv';
            if ( isset( $upload_dir['path'] ) ) {
                $csvFile = $upload_dir['path'] . '/' . $file_name;
                $linkFile = $upload_dir['url'] . '/' . $file_name;
                $access_type = get_filesystem_method();
                if ( $access_type === 'direct' ) {
                    $creds = request_filesystem_credentials(
                        site_url() . '/wp-admin/',
                        '',
                        false,
                        false,
                        array()
                    );
                    /* initialize the API */
                    if ( !WP_Filesystem( $creds ) ) {
                        return '<p class="cwsi_failmsg"Could not initialize file system. Not</p>';
                    }
                    global $wp_filesystem;
                    $wp_filesystem->put_contents( $csvFile, $data_str, FS_CHMOD_FILE );
                    if ( !file_exists( $csvFile ) ) {
                        return '<p class="cwsi_failmsg">Could not create file - does not exist: ' . $csvFile . '</p>';
                    }
                    // try to save to WP database
                    $csv_data = file_get_contents( $csvFile );
                    $filename = basename( $csvFile );
                    $attachment = array(
                        'guid'           => $upload_dir['url'] . '/' . $filename,
                        'post_mime_type' => 'text/csv',
                        'post_title'     => preg_replace( '/\\.[^.]+$/', '', $filename ),
                        'post_content'   => '',
                        'post_status'    => 'inherit',
                    );
                    $attach_id = wp_insert_attachment( $attachment, $csvFile );
                    // Generate attachment metadata and update the attachment
                    $attach_data = wp_generate_attachment_metadata( $attach_id, $csvFile );
                    wp_update_attachment_metadata( $attach_id, $attach_data );
                    return '<p class="cwsi_successmsg">Download CSV at <a href="' . $linkFile . '" style="color:white; font-weight:bold">' . $linkFile . '</a>. Once it is downloaded, you may want to go to the media library and delete the file. </p>';
                } else {
                    return '<p class="cwsi_failmsg">Need credentials for the file system. Cannot download the CSV. Sorry!</p>';
                }
            } else {
                return '<p class="cwsi_failmsg">Could not access the upload directory and so cannot download the CSV. Sorry!</p>';
            }
        }
    }
    return $ct;
}

// set date that invitation email was sent
function cwsiAdminUpdateAsSent(  $event_id, $invitee_id  ) {
    global $wpdb;
    $now = current_datetime()->format( 'Y-m-d' );
    $query = $wpdb->update(
        $wpdb->prefix . 'cwsi_events_invitees',
        array(
            'date_email_sent' => sanitize_text_field( $now ),
        ),
        array(
            'event_id'   => sanitize_text_field( $event_id ),
            'invitee_id' => sanitize_text_field( $invitee_id ),
        ),
        array('%s'),
        array('%d', '%d')
    );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query ) {
        if ( $wpdb->last_error != "" ) {
            $results = array(
                "status"  => 0,
                "message" => 'Could not update your response. Error is ' . $wpdb->last_error . '. ',
            );
        } else {
            $results = array(
                "status" => 1,
            );
        }
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Date email sent was saved.",
        );
    }
    return $results;
}

// set date that save the dateemail was sent
function cwsiAdminUpdateAsSaveSent(  $event_id, $invitee_id  ) {
    global $wpdb;
    $now = current_datetime()->format( 'Y-m-d' );
    $query = $wpdb->update(
        $wpdb->prefix . 'cwsi_events_invitees',
        array(
            'date_save_sent' => sanitize_text_field( $now ),
        ),
        array(
            'event_id'   => sanitize_text_field( $event_id ),
            'invitee_id' => sanitize_text_field( $invitee_id ),
        ),
        array('%s'),
        array('%d', '%d')
    );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query ) {
        if ( $wpdb->last_error != "" ) {
            $results = array(
                "status"  => 0,
                "message" => 'Could not update your response. Error is ' . $wpdb->last_error . '. ',
            );
        } else {
            $results = array(
                "status" => 1,
            );
        }
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Date email sent was saved.",
        );
    }
    return $results;
}

/////////////////////////////////////
// FILE UPLOAD
////////////////////////////////////
function cwsiSaveInvitesImport(  $post, $files  ) {
    global $wpdb;
    $allowed = array("text/plain");
    $message = "";
    $severe = false;
    $field = "upload_file";
    if ( isset( $files[$field] ) && $files[$field]['size'] > 0 && $files[$field]['error'] === UPLOAD_ERR_OK ) {
        if ( !function_exists( 'wp_handle_upload' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        $uploadedfile = $files[$field];
        $upload_overrides = array(
            'test_form' => false,
        );
        $movefile = wp_handle_upload( $uploadedfile, $upload_overrides );
        if ( $movefile && !isset( $movefile['error'] ) ) {
            if ( isset( $movefile['url'] ) ) {
                $fileurl = $movefile['url'];
            }
        } else {
            $results = array(
                'status'  => $status,
                'message' => "Failed to upload file. Error is " . $movefile['error'],
            );
            return $results;
        }
        $response = wp_remote_get( $fileurl );
        if ( is_array( $response ) && !is_wp_error( $response ) ) {
            $headers = $response['headers'];
            // array of http header lines
            $filedata = $response['body'];
            // use the content
        } else {
            $results = array(
                'status'  => $status,
                'message' => "Failed to read the file at " . $fileurl . ". Error is " . $response,
            );
            return $results;
        }
        wp_delete_file( $fileurl );
        // process
        /** This filter is documented in the WordPress file wp-admin/admin.php */
        @ini_set( 'memory_limit', apply_filters( 'admin_memory_limit', WP_MAX_MEMORY_LIMIT ) );
        // phpcs:ignore 	Squiz.PHP.DiscouragedFunctions.Discouraged
        @set_time_limit( 300 );
        // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged
        $lines = explode( PHP_EOL, $filedata );
        $array = array();
        if ( !is_array( $lines ) ) {
            $message .= 'Error breaking out the data from import file. ';
            $severe = true;
        }
        if ( !$severe ) {
            $ctr_line = 0;
            foreach ( $lines as $ctr => $line ) {
                $ctr_line++;
                $data = str_getcsv( $line );
                $results['status'] = 1;
                if ( !is_array( $data ) ) {
                    $message .= 'Error breaking out the data in the line ' . $ctr . '. ';
                    $severe = true;
                } else {
                    // now add to invitees table
                    if ( !isset( $data[0] ) || $data[0] == "" ) {
                        $data[0] = "";
                    }
                    $email = $data[0];
                    if ( $email != "" && $ctr_line > 1 ) {
                        if ( !is_email( $email ) ) {
                            $message = esc_html( $email ) . " is not a valid email.";
                            $results = array(
                                "status"  => 0,
                                "message" => $message,
                            );
                            return $results;
                        }
                        $values = array(
                            'event_id'        => $post['event_id'],
                            'email'           => $email,
                            'full_name'       => $data[1],
                            'num_guests'      => $data[2],
                            'rsvp_answer'     => $data[3],
                            'comments'        => $data[4],
                            'personal_note'   => $data[5],
                            'admin_notes'     => $data[6],
                            'date_email_sent' => "",
                            'date_responded'  => "",
                        );
                        $results = cwsiAdminAddInvitee( $values );
                        if ( $results['status'] == 0 ) {
                            $message .= $results['message'];
                        } else {
                            if ( $results['message'] != "" ) {
                                $message .= $results['message'];
                            }
                        }
                    }
                }
            }
        }
    } elseif ( isset( $files[$field]['error'] ) ) {
        switch ( esc_html( $files[$field]['error'] ) ) {
            case UPLOAD_ERR_OK:
                break;
            case UPLOAD_ERR_NO_FILE:
                $message .= 'No file sent.';
            case UPLOAD_ERR_INI_SIZE:
            case UPLOAD_ERR_FORM_SIZE:
                $message .= 'Exceeded filesize limit.';
            default:
                $message .= 'Unknown errors.';
        }
    } else {
        $message .= 'Unknown errors.';
    }
    if ( $severe ) {
        $status = 0;
    } else {
        $status = 1;
    }
    if ( $message == "" ) {
        $message = "Added. ";
    }
    $results = array(
        'status'  => $status,
        'message' => $message,
    );
    return $results;
}

function cwsiGetInviteByEmail(  $email  ) {
    global $wpdb;
    if ( !isset( $email ) || $email == "" || !is_email( $email ) ) {
        $results = array(
            'status'    => 0,
            'message'   => 'Cannot send email since email ' . $email . ' is invalid',
            'insert_id' => 0,
        );
        return $results;
    }
    $values = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM ' . $wpdb->prefix . 'cwsi_invites WHERE email=%s', array($email) ) );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    $results = array(
        'status'  => 0,
        'message' => "Could not find an invitee with the email " . $email,
    );
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            'status'  => 0,
            'message' => 'Failed to find Invitees. Error is ' . $wpdb->last_error . '. ',
        );
    } elseif ( count( $values ) < 1 ) {
        $results = array(
            'status'  => 0,
            'message' => 'There is no invitee with email ' . $email . '. ',
        );
    } elseif ( isset( $values[0] ) && isset( $values[0]->full_name ) && isset( $values[0]->email ) ) {
        $results = array(
            'status'  => 1,
            'message' => "",
            'data'    => $values[0],
        );
    } else {
        $results = array(
            'status'  => 0,
            'message' => 'There is no invitee with email ' . $email . '. ',
        );
    }
    return $results;
}

//////////////////////////////////
// events_invitees table FUNCTIONS
//////////////////////////////////
function cwsiAdminGetInviteeAllEvents(  $invitee_id  ) {
    global $wpdb;
    $values = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM ' . $wpdb->prefix . 'cwsi_events_invitees WHERE invitee_id=%d', array($invitee_id) ) );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            'status'  => 0,
            'message' => 'Failed to find Invitees. Error is ' . $wpdb->last_error . '. ',
        );
    } elseif ( count( $values ) < 1 ) {
        $results = array(
            'status'  => 0,
            'message' => 'The invitee is not assigned to any events. ',
        );
    } elseif ( isset( $values[0] ) ) {
        $results = array(
            'status'  => 1,
            'message' => "Assigned to at least 1 event",
            'data'    => $values,
        );
    } else {
        $results = array(
            'status'  => 0,
            'message' => 'The invitee is not assigned to any events. ',
        );
    }
    return $results;
}

function cwsiAdminSaveInviteeToEvent(  $event_id, $invitee_id, $values  ) {
    global $wpdb;
    $query = $wpdb->insert( $wpdb->prefix . 'cwsi_events_invitees', array(
        'event_id'        => sanitize_text_field( $event_id ),
        'invitee_id'      => sanitize_text_field( $invitee_id ),
        'num_guests'      => sanitize_textarea_field( $values['num_guests'] ),
        'rsvp_answer'     => sanitize_text_field( $values['rsvp_answer'] ),
        'comments'        => sanitize_text_field( $values['comments'] ),
        'personal_note'   => sanitize_text_field( $values['personal_note'] ),
        'admin_notes'     => sanitize_text_field( $values['admin_notes'] ),
        'date_email_sent' => sanitize_text_field( $values['date_email_sent'] ),
        'date_responded'  => sanitize_text_field( $values['date_responded'] ),
    ), array(
        '%d',
        '%d',
        '%d',
        '%s',
        '%s',
        '%s',
        '%s',
        '%s',
        '%s'
    ) );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( $wpdb->insert_id < 0 ) {
        $results = array(
            'status'    => 0,
            'message'   => 'Could not add invitee to the selected event. Error is ' . $wpdb->last_error,
            'insert_id' => 0,
        );
    } else {
        $results = array(
            'status'    => 1,
            'message'   => 'Added invitee to the selected event. ',
            'insert_id' => $wpdb->insert_id,
        );
    }
    return $results;
}

function cwsiAdminDeleteEventInvitee(  $event_id, $invitee_id  ) {
    global $wpdb;
    // delete from the assignment table. If the invitee is not assigned to any other events then delete them from the invites table as well
    if ( !isset( $event_id ) || !isset( $invitee_id ) ) {
        return array(
            'status'  => 0,
            'message' => 'Could not delete the invitee for that event.',
        );
    }
    $query = $wpdb->delete( $wpdb->prefix . "cwsi_events_invitees", array(
        'event_id'   => $event_id,
        'invitee_id' => $invitee_id,
    ), array('%d', '%d') );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query && $wpdb->last_error != "" ) {
        $results = array(
            "status"  => 0,
            "message" => 'Could not delete invitee for the event. Error is ' . $wpdb->last_error . '. ',
        );
        return $results;
    }
    $results = array(
        "status"  => 1,
        "message" => 'Successfully deleted invite. ',
    );
    // are then in the assignment table at all?
    $results_get = cwsiAdminGetInviteeAllEvents( $invitee_id );
    if ( isset( $results_get ) && isset( $results_get['status'] ) && $results_get['status'] == 0 ) {
        // delete invitee as well
        $results = cwsiAdminDeleteInvitee( $invitee_id );
    }
    return $results;
}

//////////////////////////////////
// Email FUNCTIONS
//////////////////////////////////
// send invitation email to the invitees
function cwsiAdminInvitesSendEmail(  $results_settings  ) {
    $bad = array();
    $good = array();
    if ( !isset( $results_settings['status'] ) || $results_settings['status'] == 0 || !isset( $results_settings['data'] ) || count( $results_settings['data'] ) == 0 ) {
        return $results_settings;
    }
    $settings = $results_settings['data'][0];
    $required = array(
        'email_fromname' => "Email From Name",
        'email_from'     => "Email From",
        'email_subject'  => "Subject",
        'email_body'     => "Body of Email",
        'event_id'       => "Event",
    );
    foreach ( $required as $field => $desc ) {
        if ( !isset( $settings->{$field} ) || $settings->{$field} == "" ) {
            $results = array(
                'status'  => 0,
                'message' => $desc . " is a required field. Cannot send invitations.",
            );
            return $results;
        }
    }
    $results = cwsiGetInvitesNotEmailed( $settings->event_id );
    if ( !isset( $results['status'] ) || $results['status'] == 0 ) {
        return $results;
    }
    $invitees = $results['data'];
    if ( count( $invitees ) == 0 ) {
        $results = array(
            'status'  => 0,
            'message' => "Everyone has been emailed already.",
        );
        return $results;
    }
    $headers = array("Content-Type: text/html; charset=UTF-8\r\n");
    $email_from = $settings->email_from;
    if ( !is_email( $email_from ) ) {
        $results = array(
            'status'  => 0,
            'message' => "The FROM email {$email_from} is invalid. Cannot send email.",
        );
        return $results;
    }
    $headers[] = "From: " . esc_html( $settings->email_fromname ) . ' <' . esc_html( $email_from ) . ">\r\n";
    if ( isset( $settings->email_replyto ) && $settings->email_replyto != "" ) {
        if ( !is_email( $settings->email_replyto ) ) {
            $results = array(
                "status"  => 0,
                "message" => esc_html( $settings->email_replyto ) . " is not a valid REPLY-TO email address. ",
            );
            return $results;
        }
        $headers[] = "Reply-to: " . $settings->email_replyto . "\r\n";
    }
    if ( isset( $settings->email_cc ) && $settings->email_cc != "" ) {
        if ( !is_email( $settings->email_cc ) ) {
            $results = array(
                "status"  => 0,
                "message" => esc_html( $settings->email_cc ) . " is not a valid Cc email address. ",
            );
            return $results;
        }
        $headers[] = "Cc: " . $settings->email_cc . "\r\n";
    }
    if ( isset( $settings->email_bcc ) && $settings->email_bcc != "" ) {
        if ( !is_email( $settings->email_bcc ) ) {
            $results = array(
                "status"  => 0,
                "message" => esc_html( $settings->email_bcc ) . " is not a valid Bcc email address. ",
            );
            return $results;
        }
        $headers[] = "Bcc: " . $settings->email_bcc . "\r\n";
    }
    $subject = wp_unslash( $settings->email_subject );
    foreach ( $invitees as $i => $row ) {
        $body = wp_unslash( nl2br( $settings->email_body ) );
        $body = str_replace( "[email]", $row->email, $body );
        $body = str_replace( "[full_name]", $row->full_name, $body );
        if ( $row->personal_note != "" ) {
            $body = str_replace( "[personal_note]", $row->personal_note, $body );
        } else {
            $body = str_replace( "[personal_note]", "", $body );
        }
        $to = esc_html( $row->email );
        $sent = @wp_mail(
            $to,
            $subject,
            $body,
            $headers
        );
        if ( !$sent ) {
            $bad[] = $row->email;
        } else {
            $good[] = $row->email;
            cwsiAdminUpdateAsSent( $settings->event_id, $row->invitee_id );
        }
    }
    $results = array(
        "status"  => 1,
        "message" => "Emails sent.",
        "bad"     => $bad,
        "good"    => $good,
    );
    return $results;
}

////////////////////////////////////////////////////////////////////////
// Send Emails
///////////////////////////////////////////////////////////////////////
function cwsiSendInviteeReminderEmails(  $values, $type  ) {
    global $wpdb;
    // validate
    $requireds = array(
        "email_subject"  => "Subject",
        "email_fromname" => "From name",
        "email_from"     => "From Email",
        "email_body"     => "Email Body",
    );
    foreach ( $requireds as $field => $nice ) {
        if ( !isset( $values[$field] ) ) {
            $results = array(
                "status"  => 0,
                "message" => $nice . " is required.",
            );
            return $results;
        }
    }
    $email_from = trim( $values['email_from'] );
    if ( $email_from == "" || !is_email( $email_from ) ) {
        $results = array(
            "status"  => 0,
            "message" => $email_from . " is not a valid FROM email address. Please only enter the email without the domain.",
        );
        return $results;
    }
    $email_from = $values['email_fromname'] . '<' . $email_from . '>';
    $headers = array("Content-Type: text/html; charset=UTF-8\r\n", "From: {$email_from}\r\n");
    if ( isset( $values['email_replyto'] ) && $values['email_replyto'] != "" ) {
        if ( !is_email( $values['email_replyto'] ) ) {
            $results = array(
                "status"  => 0,
                "message" => esc_html( $values['email_replyto'] ) . " is not a valid REPLY-TO email address. ",
            );
            return $results;
        }
        $headers[] = "Reply-to: " . $values['email_replyto'] . "\r\n";
    }
    $subject = wp_unslash( $values['email_subject'] );
    // who is it going to?
    if ( isset( $values['email_tofield'] ) && trim( $values['email_tofield'] ) != "" ) {
        $to_array = explode( ",", $values['email_tofield'] );
    } else {
        $to_array = array();
    }
    // Add emails from the list here
    if ( isset( $values['invite'] ) ) {
        foreach ( $values['invite'] as $email ) {
            if ( !in_array( $email, $to_array ) && is_email( $email ) ) {
                $to_array[] = $email;
            }
        }
    }
    if ( count( $to_array ) <= 0 ) {
        $results = array(
            "status"  => 0,
            "message" => "There were no TO emails selected. Nothing sent. ",
        );
        return $results;
    }
    if ( isset( $values['cwsi_attachment_id'] ) && $values['cwsi_attachment_id'] != "" ) {
        $url = get_attached_file( $values['cwsi_attachment_id'], true );
    } else {
        $url = "";
    }
    // Loop through to emails and if valid send email
    $status = 1;
    $message = "";
    $conn = "";
    $now = current_datetime()->format( 'Y-m-d' );
    foreach ( $to_array as $email ) {
        $to = trim( $email );
        if ( $to == "" || !is_email( $to ) ) {
            $message .= "Could not send to a blank or invalid email: " . esc_html( $email ) . ". Sending stopped. ";
            $results = array(
                "status"  => 0,
                "message" => $message,
            );
            return $results;
        }
        $body = $values['email_body'];
        $body = str_replace( "[email]", $to, $body );
        if ( stristr( $body, "[full_name]" ) ) {
            // have to get record
            $results = cwsiGetInviteByEmail( $to );
            if ( !isset( $results['status'] ) || $results['status'] != 1 || !isset( $results['data'] ) ) {
                return $results;
            }
            $data = $results['data'];
            $body = str_replace( "[full_name]", $data->full_name, $body );
        }
        if ( isset( $url ) && $url != "" ) {
            $sent = @wp_mail(
                $to,
                $subject,
                wp_unslash( nl2br( $body ) ),
                $headers,
                $url
            );
        } else {
            $sent = @wp_mail(
                $to,
                $subject,
                wp_unslash( nl2br( $body ) ),
                $headers
            );
        }
        if ( !isset( $sent ) || !$sent ) {
            $message .= "Could not send email to {$email}. Sending stopped. ";
            $results = array(
                "status"  => 0,
                "message" => $message,
            );
            return $results;
        } elseif ( $type != "" ) {
            // update as sent
            // need invitee id
            $invites = $wpdb->get_results( $wpdb->prepare( 'SELECT a.ID as events_invitees_id FROM ' . $wpdb->prefix . 'cwsi_events_invitees as a JOIN ' . $wpdb->prefix . 'cwsi_invites as b ON a.invitee_id=b.ID WHERE b.email=%s', array(sanitize_text_field( $to )) ) );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            if ( isset( $invites ) && is_array( $invites ) && count( $invites ) > 0 && isset( $invites[0] ) && isset( $invites[0]->events_invitees_id ) ) {
                $query = $wpdb->update(
                    $wpdb->prefix . 'cwsi_events_invitees',
                    array(
                        $type => sanitize_text_field( $now ),
                    ),
                    array(
                        'ID' => sanitize_text_field( $invites[0]->events_invitees_id ),
                    ),
                    array('%s'),
                    array('%d')
                );
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                if ( !$query && $wpdb->last_error != "" ) {
                    $results = array(
                        "status"  => 0,
                        "message" => 'Could not update your response. Error is ' . $wpdb->last_error . '. ',
                    );
                    return $results;
                }
            }
            // invitee found for email
        }
        $message .= $conn . $to;
        $conn = ", ";
    }
    $results = array(
        "status"  => 1,
        "message" => "Emails sent successfully to " . $message,
    );
    return $results;
}

///////////////////////////////////////////////////////////////////////////
// SETTINGS FUNCTIONS
//////////////////////////////////////////////////////////////////////////
// Check if settings table was updated in recent activation
function cwsiIsSettingsTableUpdated() {
    global $wpdb;
    $col_name = 'event_id';
    $col_type = 'bigint(20)';
    $values = $wpdb->get_results( "DESC " . $wpdb->prefix . 'cwsi_invitesettings' );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    foreach ( $values as $row ) {
        if ( $row->Field === $col_name ) {
            return "";
        }
        // End if found our column.
    }
    return '<p class="cwsi_failmsg">It looks like you have an older version of the plugin. Go to your plugins page, update if necessary, then deactivate and activate to get the database updates. You will not lose any invitees. </p>';
}

// retrieve the email settings
function cwsiAdminInviteGetEmailSettings(  $message_type, $event_id  ) {
    global $wpdb;
    if ( !isset( $message_type ) || $message_type != "invite" && $message_type != "confirmation" ) {
        $results = array(
            "status"  => 0,
            "message" => 'Could not retrieve email settings for invalid message type ' . $message_type . '. ',
        );
    }
    $values = $wpdb->get_results( $wpdb->prepare( 'SELECT a.*, b.event_name, b.organizer, b.event_date FROM ' . $wpdb->prefix . 'cwsi_invitesettings as a JOIN ' . $wpdb->prefix . 'cwsi_events as b ON a.event_id=b.ID WHERE message_type=%s AND b.ID=%d ORDER BY date_updated DESC LIMIT 1', array($message_type, $event_id) ) );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'Failed to find email settings. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Success",
            "data"    => $values,
        );
    }
    return $results;
}

// validate the settings form
function cwsiAdminInviteProcessEmailSettings(  $values  ) {
    $fields = array(
        "settings_id",
        "email_subject",
        "email_fromname",
        "email_from",
        "email_body",
        "message_type",
        "event_id"
    );
    foreach ( $fields as $field ) {
        if ( !isset( $values[$field] ) || $values[$field] == "" ) {
            $results = array(
                "status"  => 0,
                "message" => "Could not save the settings. The subject, from name, from email and email body must all be entered. The event must be selected. ",
            );
            return $results;
        }
    }
    if ( !isset( $values['email_body_2'] ) || $values['email_body_2'] == "" ) {
        if ( $values['message_type'] == "confirmation" ) {
            $results = array(
                "status"  => 0,
                "message" => "Enter the Email Body if the Answer Is No.",
            );
            return $results;
        }
        $values['email_body_2'] = "";
    }
    $email_from = trim( $values['email_from'] );
    if ( $email_from == "" || !is_email( $email_from ) ) {
        $results = array(
            "status"  => 0,
            "message" => $email_from . " is not a valid FROM email address. Please only enter the email without the domain.",
        );
        return $results;
    }
    if ( isset( $values['email_replyto'] ) && $values['email_replyto'] != "" && !is_email( $values['email_replyto'] ) ) {
        $results = array(
            "status"  => 0,
            "message" => esc_html( $values['email_replyto'] ) . " is not a valid REPLY-TO email address. ",
        );
        return $results;
    }
    if ( isset( $values['email_cc'] ) && $values['email_cc'] != "" && !is_email( $values['email_cc'] ) ) {
        $results = array(
            "status"  => 0,
            "message" => esc_html( $values['email_cc'] ) . " is not a valid CC email address. ",
        );
        return $results;
    }
    if ( isset( $values['email_bcc'] ) && $values['email_bcc'] != "" && !is_email( $values['email_bcc'] ) ) {
        $results = array(
            "status"  => 0,
            "message" => esc_html( $values['email_bcc'] ) . " is not a valid BCC email address. ",
        );
        return $results;
    }
    if ( $values['settings_id'] >= 0 ) {
        $results = cwsiAdminInviteUpdateEmailSettings( $values );
    } else {
        $results = cwsiAdminInviteInsertEmailSettings( $values );
    }
    return $results;
}

function cwsiAdminInviteInsertEmailSettings(  $values  ) {
    global $wpdb;
    $query = $wpdb->insert( $wpdb->prefix . 'cwsi_invitesettings', array(
        "event_id"       => sanitize_text_field( $values['event_id'] ),
        "email_subject"  => sanitize_text_field( wp_unslash( $values['email_subject'] ) ),
        "email_fromname" => sanitize_text_field( $values['email_fromname'] ),
        "email_from"     => sanitize_text_field( $values['email_from'] ),
        "email_replyto"  => sanitize_text_field( $values['email_replyto'] ),
        "email_cc"       => sanitize_text_field( $values['email_cc'] ),
        "email_bcc"      => sanitize_text_field( $values['email_bcc'] ),
        "message_type"   => sanitize_text_field( $values['message_type'] ),
        "email_body"     => wp_unslash( $values['email_body'] ),
        "email_body_2"   => wp_unslash( $values['email_body_2'] ),
    ), array(
        '%d',
        '%s',
        '%s',
        '%s',
        '%s',
        '%s',
        '%s',
        '%s',
        '%s',
        '%s'
    ) );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( $wpdb->insert_id < 0 ) {
        $results = array(
            "status"  => 0,
            "message" => 'Could not add settings. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"    => 1,
            "message"   => "Settings added successfully.",
            "insert_id" => $wpdb->insert_id,
        );
    }
    return $results;
}

function cwsiAdminInviteUpdateEmailSettings(  $values  ) {
    global $wpdb;
    $query = $wpdb->update(
        $wpdb->prefix . 'cwsi_invitesettings',
        array(
            "event_id"       => sanitize_text_field( $values['event_id'] ),
            "email_subject"  => sanitize_text_field( wp_unslash( $values['email_subject'] ) ),
            "email_fromname" => sanitize_text_field( $values['email_fromname'] ),
            "email_from"     => sanitize_text_field( $values['email_from'] ),
            "email_replyto"  => sanitize_text_field( $values['email_replyto'] ),
            "email_cc"       => sanitize_text_field( $values['email_cc'] ),
            "email_bcc"      => sanitize_text_field( $values['email_bcc'] ),
            "message_type"   => sanitize_text_field( $values['message_type'] ),
            "email_body"     => wp_unslash( $values['email_body'] ),
            "email_body_2"   => wp_unslash( $values['email_body_2'] ),
        ),
        array(
            'ID' => sanitize_text_field( $values['settings_id'] ),
        ),
        array(
            '%d',
            '%s',
            '%s',
            '%s',
            '%s',
            '%s',
            '%s',
            '%s',
            '%s',
            '%s'
        ),
        array('%d')
    );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query ) {
        if ( $wpdb->last_error != "" ) {
            $results = array(
                "status"  => 0,
                "message" => 'Could not save settings. Error is ' . $wpdb->last_error . '. ',
            );
        } else {
            $results = array(
                "status"  => 0,
                "message" => 'Updated',
            );
        }
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Settings saved successfully.",
        );
    }
    return $results;
}

// send a test email
function cwsiAdminInviteSendTestEmail(  $values, $results  ) {
    // validate
    if ( !isset( $values['email_to'] ) || $values['email_to'] == "" || !is_email( $values['email_to'] ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'Enter a correct To Email Address. ',
        );
        return $results;
    }
    if ( !isset( $results['data'] ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'No email settings found. ',
        );
        return $results;
    }
    $alldata = $results['data'];
    $data = $alldata[0];
    $to = trim( $values['email_to'] );
    $email_from = trim( $data->email_from );
    $email_from = $data->email_fromname . '<' . $email_from . '>';
    $subject = wp_unslash( $data->email_subject );
    $headers = array("Content-Type: text/html; charset=UTF-8\r\n", "From: {$email_from}\r\n");
    if ( isset( $data->email_replyto ) && $data->email_replyto != "" && is_email( $data->email_replyto ) ) {
        $headers[] = "Reply-to: " . trim( $data->email_replyto ) . "\r\n";
    }
    if ( isset( $data->email_cc ) && $data->email_cc != "" && is_email( $data->email_cc ) ) {
        $headers[] = "Cc: " . trim( $data->email_cc ) . "\r\n";
    }
    if ( isset( $data->email_bcc ) && $data->email_bcc != "" && is_email( $data->email_bcc ) ) {
        $headers[] = "Bcc: " . trim( $data->email_bcc ) . "\r\n";
    }
    $body = wp_kses( wp_unslash( nl2br( $data->email_body ) ), cwsi_invites_allowed_html() );
    if ( isset( $data->email_body_2 ) && $data->email_body_2 != "" ) {
        $body .= "<br /><br />" . wp_kses( wp_unslash( nl2br( $data->email_body_2 ) ), cwsi_invites_allowed_html() );
    }
    $sent = @wp_mail(
        $to,
        $subject,
        $body,
        $headers
    );
    if ( !isset( $sent ) || !$sent ) {
        $message = "Could not send email to {$to}. <br />From: {$email_from}. <br />Headers are: ";
        foreach ( $headers as $n => $v ) {
            $message .= "<br />{$n}: {$v}";
        }
        $results = array(
            "status"  => 0,
            "message" => $message,
        );
        return $results;
    }
    $results = array(
        "status"  => 1,
        "message" => "Email sent successfully to " . esc_html( $to ),
    );
    return $results;
}

///////////////////////////////////////////////////////////////////////////
// EVENTS FUNCTIONS
//////////////////////////////////////////////////////////////////////////
// Check if events table exists
function cwsiIsEventsTable() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'cwsi_events';
    if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) === $table_name ) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        return "";
    } else {
        return '<p class="cwsi_failmsg">It looks like you have an older version of the plugin. Go to your plugins page, update if necessary, then deactivate and activate to get the database updates. You will not lose any invitees. </p>';
    }
}

// Retrieve invitees who have not answered
function cwsiGetEvents() {
    global $wpdb;
    $values = $wpdb->get_results( 'SELECT * FROM ' . $wpdb->prefix . 'cwsi_events ORDER BY ID ASC' );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !is_object( $values ) && !is_array( $values ) ) {
        $results = array(
            "status"  => 0,
            "message" => 'Failed to find Events. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Success",
            "data"    => $values,
        );
    }
    return $results;
}

function cwsiAdminAddEvent(  $values  ) {
    global $wpdb;
    // validate
    $fields = array(
        'event_name' => "Event Name",
        'event_date' => "Event Date",
        'organizer'  => "Organizer",
    );
    foreach ( $fields as $field => $name ) {
        if ( !isset( $values[$field] ) || $values[$field] == "" ) {
            $results = array(
                "status"  => 0,
                "message" => $name . " is required.",
            );
            return $results;
        }
    }
    $exists = $wpdb->get_results( $wpdb->prepare( 'SELECT event_name, event_date, ID FROM ' . $wpdb->prefix . 'cwsi_events WHERE event_name=%s AND event_date=%s', array(sanitize_text_field( $values['event_name'] ), sanitize_text_field( $values['event_date'] )) ) );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( is_object( $exists ) || is_array( $exists ) ) {
        if ( isset( $exists[0] ) ) {
            $results = array(
                'status'  => 0,
                'message' => sanitize_text_field( $values['event_name'] ) . ' on ' . sanitize_text_field( $values['event_date'] ) . ' already exists in invitee table as ID ' . $exists[0]->ID,
            );
            return $results;
        }
    }
    $query = $wpdb->insert( $wpdb->prefix . 'cwsi_events', array(
        'event_name' => sanitize_text_field( $values['event_name'] ),
        'event_date' => sanitize_text_field( $values['event_date'] ),
        'organizer'  => sanitize_textarea_field( $values['organizer'] ),
    ), array('%s', '%s', '%s') );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( $wpdb->insert_id < 0 ) {
        $results = array(
            'status'    => 0,
            'message'   => 'Could not add event ' . sanitize_text_field( $values['event_name'] ) . ' on ' . sanitize_text_field( $values['event_date'] ) . '. Error is ' . sanitize_text_field( $wpdb->last_error ),
            'insert_id' => 0,
        );
    } else {
        $results = array(
            'status'    => 1,
            'message'   => "",
            'insert_id' => $wpdb->insert_id,
        );
    }
    return $results;
}

function cwsiAdminSaveEvent(  $values  ) {
    global $wpdb;
    // validate
    $fields = array(
        'cwsevent_id' => "Event ID",
        'event_name'  => "Event Name",
        'event_date'  => "Event Date",
        'organizer'   => "Organizer",
    );
    foreach ( $fields as $field => $name ) {
        if ( !isset( $values[$field] ) || $values[$field] == "" ) {
            $results = array(
                "status"  => 0,
                "message" => $name . " is required.",
            );
            return $results;
        }
    }
    $query = $wpdb->update(
        $wpdb->prefix . 'cwsi_events',
        array(
            'event_name' => sanitize_text_field( $values['event_name'] ),
            'event_date' => sanitize_text_field( $values['event_date'] ),
            'organizer'  => sanitize_text_field( $values['organizer'] ),
        ),
        array(
            'ID' => sanitize_text_field( $values['cwsevent_id'] ),
        ),
        array('%s', '%s', '%s'),
        array('%d')
    );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query ) {
        if ( $wpdb->last_error != "" ) {
            $results = array(
                "status"  => 0,
                "message" => "Event " . esc_html( $values['event_name'] ) . ' on ' . esc_html( $values['event_date'] ) . " was updated. Error is " . $wpdb->last_error . '. ',
            );
        } else {
            $results = array(
                "status"  => 1,
                "message" => "",
            );
        }
    } else {
        $results = array(
            "status"  => 1,
            "message" => "Event " . esc_html( $values['event_name'] ) . ' on ' . esc_html( $values['event_date'] ) . " was updated.",
        );
    }
    return $results;
}

function cwsiAdminDeleteEvent(  $cwsevent_id  ) {
    global $wpdb;
    if ( !isset( $cwsevent_id ) || $cwsevent_id < 0 ) {
        return array(
            'status'  => 0,
            'message' => 'No event selected for deletion.',
        );
    }
    $query = $wpdb->delete( $wpdb->prefix . "cwsi_events", array(
        'ID' => sanitize_text_field( $cwsevent_id ),
    ), array('%d') );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    if ( !$query && $wpdb->last_error != "" ) {
        $results = array(
            "status"  => 0,
            "message" => 'Could not delete event. Error is ' . $wpdb->last_error . '. ',
        );
    } else {
        $results = array(
            "status"  => 1,
            "message" => 'Successfully deleted event. ',
        );
    }
    return $results;
}

///////////////////////////////////////////////////////////////////////////
// HELPER FUNCTIONS
//////////////////////////////////////////////////////////////////////////
// define allows html tags
function cwsi_invites_allowed_html() {
    $allowed_tags = array(
        'a'          => array(
            'class' => array(),
            'href'  => array(),
            'rel'   => array(),
            'style' => array(),
            'title' => array(),
        ),
        'abbr'       => array(
            'title' => array(),
        ),
        'b'          => array(),
        'br'         => array(),
        'blockquote' => array(
            'cite' => array(),
        ),
        'button'     => array(
            'class' => array(),
            'href'  => array(),
            'rel'   => array(),
            'style' => array(),
            'title' => array(),
        ),
        'cite'       => array(
            'title' => array(),
        ),
        'code'       => array(),
        'del'        => array(
            'datetime' => array(),
            'title'    => array(),
        ),
        'dd'         => array(),
        'div'        => array(
            'class' => array(),
            'title' => array(),
            'style' => array(),
        ),
        'dl'         => array(),
        'dt'         => array(),
        'em'         => array(),
        'h1'         => array(),
        'h2'         => array(),
        'h3'         => array(),
        'h4'         => array(),
        'h5'         => array(),
        'h6'         => array(),
        'i'          => array(),
        'img'        => array(
            'alt'    => array(),
            'class'  => array(),
            'height' => array(),
            'src'    => array(),
            'width'  => array(),
        ),
        'li'         => array(
            'class' => array(),
        ),
        'ol'         => array(
            'class' => array(),
        ),
        'p'          => array(
            'class' => array(),
        ),
        'q'          => array(
            'cite'  => array(),
            'title' => array(),
        ),
        'span'       => array(
            'class' => array(),
            'title' => array(),
            'style' => array(),
        ),
        'strike'     => array(),
        'strong'     => array(),
        'table'      => array(
            'class' => array(),
            'align' => array(),
            'style' => array(),
        ),
        'thead'      => array(),
        'tbody'      => array(),
        'tr'         => array(),
        'th'         => array(
            'class' => array(),
            'align' => array(),
            'style' => array(),
        ),
        'td'         => array(
            'class' => array(),
            'align' => array(),
            'style' => array(),
        ),
        'ul'         => array(
            'class' => array(),
        ),
    );
    return $allowed_tags;
}
