<?php
if (!defined('ABSPATH'))
    exit;

/**
 * Check if the provided slot is already booked (overlap check).
 */
function cutqueue_slot_is_booked($store_id, $date, $time, $duration, $worker = 'any')
{
    $bookings = get_option("cutqueue_booking_system_bookings_{$store_id}", []);
    if (!is_array($bookings))
        $bookings = [];

    $stores = get_option('cutqueue_booking_system_stores', []);
    $store_timezone = 'UTC';
    if (is_array($stores)) {
        foreach ($stores as $s) {
            if (($s['id'] ?? '') === $store_id) {
                $store_timezone = $s['timezone'] ?? 'UTC';
                break;
            }
        }
    }

    $dt = date_create_from_format(
        'Y-m-d H:i',
        "$date $time",
        wp_timezone() // Uses site timezone safely
    );

    if (!$dt) {
        return true; // invalid date/time → treat as booked
    }

    $new_start = $dt->getTimestamp();
    $new_end = $new_start + intval($duration) * 60;

    if ($new_start === false)
        return true; // invalid new booking time → consider booked

    $workers = get_option("cutqueue_booking_system_workers_{$store_id}", []);
    $total_workers = is_array($workers) ? count($workers) : 0;
    $disable_worker_selection = get_option("cutqueue_booking_system_disable_worker_selection_{$store_id}", false);
    $slots_per_booking = intval(get_option("cutqueue_booking_system_slots_per_booking_{$store_id}", 1));

    if (!$disable_worker_selection && $total_workers > 0) {
        $selected_worker = ($worker !== 'any') ? $worker : null;

        foreach ($bookings as $b) {
            if (!isset($b['worker'], $b['date'], $b['time'], $b['duration']))
                continue;
            if ($b['worker'] !== $selected_worker)
                continue;

            $b_start = strtotime("{$b['date']} {$b['time']}");
            $b_end = $b_start + intval($b['duration']) * 60;
            if ($b_start === false)
                continue;

            if ($b['date'] === $date && $new_start < $b_end && $new_end > $b_start) {
                return true;
            }
        }
        return false;
    }

    // Worker selection disabled → count slots
    $booked_count = 0;
    foreach ($bookings as $b) {
        if (!isset($b['date'], $b['time'], $b['duration']))
            continue;

        $b_start = strtotime("{$b['date']} {$b['time']}");
        $b_end = $b_start + intval($b['duration']) * 60;
        if ($b_start === false)
            continue;

        if ($b['date'] === $date && $new_start < $b_end && $new_end > $b_start) {
            $booked_count++;
        }
    }

    return ($booked_count >= $slots_per_booking);
}

/**
 * Main AJAX handler for saving bookings.
 */
function cutqueue_booking_save()
{
    if (
        !isset($_POST['cutqueue_booking_system_nonce']) ||
        !wp_verify_nonce(
            sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_nonce'])),
            'cutqueue_booking_system_booking_nonce'
        )
    ) {
        wp_send_json_error('Nonce verification failed.');
    }

    // Collect and sanitize fields
    $name = isset($_POST['cutqueue_booking_system_name']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_name'])) : '';
    $email = isset($_POST['cutqueue_booking_system_email']) ? sanitize_email(wp_unslash($_POST['cutqueue_booking_system_email'])) : '';
    $mobile = isset($_POST['cutqueue_booking_system_mobile']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_mobile'])) : '';
    $service = isset($_POST['cutqueue_booking_system_service']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_service'])) : '';
    $worker = isset($_POST['cutqueue_booking_system_worker']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_worker'])) : '';
    $date = isset($_POST['cutqueue_booking_system_date']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_date'])) : '';
    $time = isset($_POST['cutqueue_booking_system_time']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_time'])) : '';
    $store_id = isset($_POST['cutqueue_booking_system_store_id']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_store_id'])) : 'store_default';
    $payment_mode = isset($_POST['cutqueue_booking_system_payment_mode']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_payment_mode'])) : 'disabled';

    if (!$name || !$email || !$mobile || !$service || !$worker || !$date || !$time) {
        wp_send_json_error('Please fill in all fields.');
    }

    // Resolve duration from services option
    $services = get_option("cutqueue_booking_system_services_{$store_id}", []);
    $slot_duration = 30;
    if (is_array($services)) {
        foreach ($services as $s) {
            if ((isset($s['name']) && $s['name'] === $service) || (isset($s['slug']) && $s['slug'] === $service)) {
                $slot_duration = intval($s['duration'] ?? $slot_duration);
                break;
            }
        }
    }

    // Final safety server-side slot check BEFORE any payment step
    if (cutqueue_slot_is_booked($store_id, $date, $time, $slot_duration, $worker)) {
        wp_send_json_error('This time slot is already booked. Please choose another time.');
    }

    // Build a lock key to stop near-simultaneous attempts
    $lock_key = md5($store_id . '|' . $date . '|' . $time . '|' . $worker);
    $transient_key = "cutqueue_booking_system_lock_{$lock_key}";

    if (get_transient($transient_key)) {
        wp_send_json_error('This time slot is currently being booked. Try again in a few seconds.');
    }

    // Set transient lock for 30 seconds
    set_transient($transient_key, 1, 30);

    if ($payment_mode === 'online') {
        $paymentIntentId = isset($_POST['paymentIntentId']) ? sanitize_text_field(wp_unslash($_POST['paymentIntentId'])) : '';
        $stripe_secret_key = get_option("cutqueue_booking_system_stripe_secret_key_{$store_id}", '');

        if ($paymentIntentId) {
            $response = wp_remote_get(
                'https://api.stripe.com/v1/payment_intents/' . rawurlencode($paymentIntentId),
                [
                    'headers' => [
                        'Authorization' => 'Bearer ' . $stripe_secret_key,
                        'Content-Type' => 'application/x-www-form-urlencoded'
                    ],
                    'timeout' => 20,
                ]
            );

            if (is_wp_error($response)) {
                wp_send_json_error('Unable to verify payment at this time.');
            }

            $pi = json_decode(wp_remote_retrieve_body($response), true);
            if (empty($pi) || ($pi['status'] ?? '') !== 'succeeded') {
                wp_send_json_error('Payment not completed.');
            }
        }
    }

    $bookings = get_option("cutqueue_booking_system_bookings_{$store_id}", []);
    if (!is_array($bookings))
        $bookings = [];

    $bookings[] = [
        'name' => $name,
        'email' => $email,
        'mobile' => $mobile,
        'service' => $service,
        'worker' => $worker,
        'date' => $date,
        'time' => $time,
        'duration' => $slot_duration,
        'created' => current_time('mysql'),
        'paymentIntentId' => $paymentIntentId,
        'status' => $payment_mode === 'online' ? 'pending' : 'confirmed'
    ];

    update_option("cutqueue_booking_system_bookings_{$store_id}", $bookings);

    delete_transient($transient_key);

    if ($payment_mode === 'online') {
        wp_schedule_single_event(time() + 1, 'cutqueue_async_post_save', [
            $store_id,
            $paymentIntentId
        ]);
    } else {
        $last_index = array_key_last($bookings);
        if ($last_index !== null) {
            $last_booking = $bookings[$last_index];
            cutqueue_send_confirmation_email($store_id, $last_booking);
        }
    }

    wp_send_json_success('Your booking was successful! A confirmation email has been sent.');
}

add_action('wp_ajax_cutqueue_booking_save', 'cutqueue_booking_save');
add_action('wp_ajax_nopriv_cutqueue_booking_save', 'cutqueue_booking_save');


add_action('wp_ajax_cutqueue_create_intent', 'cutqueue_create_intent');
add_action('wp_ajax_nopriv_cutqueue_create_intent', 'cutqueue_create_intent');

function cutqueue_create_intent()
{
    if (
        !isset($_POST['cutqueue_booking_system_nonce']) ||
        !wp_verify_nonce(
            sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_nonce'])),
            'cutqueue_booking_system_booking_nonce'
        )
    ) {
        wp_send_json_error('Nonce verification failed.');
    }

    $store_id = isset($_POST['cutqueue_booking_system_store_id']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_store_id'])) : 'store_default';
    $service_price = isset($_POST['cutqueue_booking_system_service_price']) ? floatval(sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_service_price']))) : 0;
    $amount = intval(round($service_price * 100));
    $stripe_secret_key = get_option("cutqueue_booking_system_stripe_secret_key_{$store_id}", '');

    $customer_name = isset($_POST['cutqueue_booking_system_name']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_name'])) : '';
    $customer_email = isset($_POST['cutqueue_booking_system_email']) ? sanitize_email(wp_unslash($_POST['cutqueue_booking_system_email'])) : '';
    $customer_phone = isset($_POST['cutqueue_booking_system_mobile']) ? sanitize_text_field(wp_unslash($_POST['cutqueue_booking_system_mobile'])) : '';

    $body = [
        'amount' => $amount,
        'currency' => 'eur',
        'receipt_email' => $customer_email,
        'automatic_payment_methods[enabled]' => 'true',
        'payment_method_options[card][request_three_d_secure]' => 'automatic',
        'metadata[name]' => $customer_name,
        'metadata[email]' => $customer_email,
        'metadata[phone]' => $customer_phone,
    ];

    $response = wp_remote_post(
        'https://api.stripe.com/v1/payment_intents',
        [
            'headers' => [
                'Authorization' => 'Bearer ' . $stripe_secret_key,
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => http_build_query($body),
            'timeout' => 20,
        ]
    );

    if (is_wp_error($response)) {
        wp_send_json([
            'success' => false,
            'error' => $response->get_error_message()
        ]);
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);

    if (!empty($data['client_secret'])) {
        wp_send_json([
            'success' => true,
            'clientSecret' => $data['client_secret'],
            'paymentIntentId' => $data['id'],
        ]);
    }

    wp_send_json([
        'success' => false,
        'error' => $data
    ]);
}

add_action('cutqueue_async_post_save', 'cutqueue_async_post_save_handler', 10, 2);

function cutqueue_async_post_save_handler($store_id, $paymentIntentId)
{
    if (!$store_id || !$paymentIntentId)
        return;

    $bookings = get_option("cutqueue_booking_system_bookings_{$store_id}", []);
    if (!is_array($bookings))
        $bookings = [];

    foreach ($bookings as &$b) {
        if (!isset($b['paymentIntentId']))
            continue;

        if ($b['paymentIntentId'] === $paymentIntentId) {
            $b['status'] = 'confirmed';
            cutqueue_send_confirmation_email($store_id, $b);
            break;
        }
    }
    update_option("cutqueue_booking_system_bookings_{$store_id}", $bookings);
}

function cutqueue_send_confirmation_email($store_id, $booking)
{
    $adminEmail = get_option("cutqueue_booking_system_admin_email_{$store_id}", '');
    $subject = "Booking Confirmation – {$booking['service']} on {$booking['date']} at {$booking['time']}";
    $message_body = "Hi {$booking['name']},\n\nYour appointment has been confirmed!\n\n" .
        "📅 Date: {$booking['date']}\n⏰ Time: {$booking['time']}\n" .
        "💈 Service: {$booking['service']}\n💇 Worker: {$booking['worker']}\n\n" .
        "Thank you for booking with us!";
    $server_name = isset($_SERVER['SERVER_NAME'])
        ? sanitize_text_field(wp_unslash($_SERVER['SERVER_NAME']))
        : 'localhost';
    $headers = ['Content-Type: text/plain; charset=UTF-8', 'From: Booking <no-reply@' . $server_name . '>'];

    if (!empty($booking['email']))
        wp_mail($booking['email'], $subject, $message_body, $headers);
    if (!empty($adminEmail))
        wp_mail($adminEmail, "New Booking from {$booking['name']}", "Booking Details:\nEmail: {$booking['email']}\nMobile: {$booking['mobile']}\nDate: {$booking['date']}\nTime: {$booking['time']}", $headers);
}
