<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Sanitize color values for dark mode settings.
 *
 * Validates and sanitizes color input to ensure it's a valid CSS color format.
 * Supports hex colors, RGB, and RGBA formats.
 *
 * @since 1.0.0
 *
 * @param string $input The color value to sanitize.
 * @return string Sanitized color value or default fallback.
 */
function dmtf_sanitize_color($input) {
    // Remove any non-hex characters
    $input = sanitize_hex_color($input);
    
    // If sanitize_hex_color returns empty, check if it's a valid CSS color
    if (empty($input)) {
        // Allow common CSS color formats
        $input = sanitize_text_field($input);
        if (preg_match('/^#[0-9a-fA-F]{3,6}$|^rgb\(\s*\d+\s*,\s*\d+\s*,\s*\d+\s*\)$|^rgba\(\s*\d+\s*,\s*\d+\s*,\s*\d+\s*,\s*[\d.]+\s*\)$/', $input)) {
            return $input;
        }
        return '#222222'; // Default fallback
    }
    
    return $input;
}

/**
 * Sanitize custom CSS input for security.
 *
 * Removes potentially dangerous content and allows only safe CSS properties.
 * Prevents XSS attacks through CSS injection.
 *
 * @since 1.0.0
 *
 * @param string $input The CSS content to sanitize.
 * @return string Sanitized CSS content.
 */
function dmtf_sanitize_css($input) {
    // Basic CSS sanitization - allow common CSS properties
    $allowed_css = array(
        'color', 'background-color', 'background', 'border', 'border-color',
        'font-size', 'font-weight', 'text-align', 'margin', 'padding',
        'width', 'height', 'display', 'position', 'top', 'left', 'right', 'bottom',
        'z-index', 'opacity', 'transition', 'transform', 'box-shadow'
    );
    
    // Remove potentially dangerous content
    $input = wp_strip_all_tags($input);
    $input = str_replace(array('<script', '</script>', 'javascript:', 'onclick', 'onload'), '', $input);
    
    return $input;
}

/**
 * Register plugin settings and their sanitization callbacks.
 *
 * Sets up all the plugin options with proper sanitization functions
 * to ensure data integrity and security.
 *
 * @since 1.0.0
 *
 * @return void
 */
function dmtf_register_settings() {
    add_option('dmtf_dark_bg_color', '#222222');
    add_option('dmtf_dark_text_color', '#ffffff');
    add_option('dmtf_toggle_button_bg', '#444444');
    add_option('dmtf_toggle_button_text', '#ffffff');
    add_option('dmtf_custom_css', '');

    register_setting('dmtf_options_group', 'dmtf_dark_bg_color', array(
        'sanitize_callback' => 'dmtf_sanitize_color'
    ));
    register_setting('dmtf_options_group', 'dmtf_dark_text_color', array(
        'sanitize_callback' => 'dmtf_sanitize_color'
    ));
    register_setting('dmtf_options_group', 'dmtf_toggle_button_bg', array(
        'sanitize_callback' => 'dmtf_sanitize_color'
    ));
    register_setting('dmtf_options_group', 'dmtf_toggle_button_text', array(
        'sanitize_callback' => 'dmtf_sanitize_color'
    ));
    register_setting('dmtf_options_group', 'dmtf_custom_css', array(
        'sanitize_callback' => 'dmtf_sanitize_css'
    ));
}
add_action('admin_init', 'dmtf_register_settings');

/**
 * Register the admin options page.
 *
 * Adds a submenu page under Settings for the dark mode toggle configuration.
 *
 * @since 1.0.0
 *
 * @return void
 */
function dmtf_register_options_page() {
    add_options_page(
        __('Dark Mode Toggle Settings', 'customizable-dark-mode-toggle'),
        __('Dark Mode Toggle', 'customizable-dark-mode-toggle'),
        'manage_options',
        'dmtf',
        'dmtf_options_page'
    );
}
add_action('admin_menu', 'dmtf_register_options_page');

/**
 * Render the admin options page content.
 *
 * Displays the form with all plugin settings including color pickers
 * and custom CSS textarea.
 *
 * @since 1.0.0
 *
 * @return void
 */
function dmtf_options_page() {
    ?>
    <div class="wrap">
        <h2><?php esc_html_e('Dark Mode Toggle Settings', 'customizable-dark-mode-toggle'); ?></h2>
        <form method="post" action="options.php">
            <?php settings_fields('dmtf_options_group'); ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Dark Background Color', 'customizable-dark-mode-toggle'); ?></th>
                    <td><input type="color" name="dmtf_dark_bg_color" value="<?php echo esc_attr(get_option('dmtf_dark_bg_color')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Dark Text Color', 'customizable-dark-mode-toggle'); ?></th>
                    <td><input type="color" name="dmtf_dark_text_color" value="<?php echo esc_attr(get_option('dmtf_dark_text_color')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Toggle Button Background', 'customizable-dark-mode-toggle'); ?></th>
                    <td><input type="color" name="dmtf_toggle_button_bg" value="<?php echo esc_attr(get_option('dmtf_toggle_button_bg')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Toggle Button Text Color', 'customizable-dark-mode-toggle'); ?></th>
                    <td><input type="color" name="dmtf_toggle_button_text" value="<?php echo esc_attr(get_option('dmtf_toggle_button_text')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Custom CSS', 'customizable-dark-mode-toggle'); ?></th>
                    <td><textarea name="dmtf_custom_css" rows="5" cols="50"><?php echo esc_textarea(get_option('dmtf_custom_css')); ?></textarea></td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
} 