(function() {
    let originalStyles = new WeakMap();
    let observer = null;
    const PSEUDO_OVERRIDE_ID = 'dmtf-darkmode-pseudo-overrides';

    function isExcluded(el) {
        return el.classList && el.classList.contains('dmtf-dark-mode-ignore');
    }

    function hasColorStyle(el) {
        const style = window.getComputedStyle(el);
        return (
            (style.backgroundColor && style.backgroundColor !== 'rgba(0, 0, 0, 0)' && style.backgroundColor !== 'transparent') ||
            (style.color && style.color !== '')
        );
    }

    function overrideColors(el, darkBg, darkText) {
        if (isExcluded(el)) return;
        if (!originalStyles.has(el)) {
            originalStyles.set(el, {
                backgroundColor: el.style.backgroundColor,
                color: el.style.color
            });
        }
        const style = window.getComputedStyle(el);
        if (style.backgroundColor && style.backgroundColor !== 'rgba(0, 0, 0, 0)' && style.backgroundColor !== 'transparent') {
            el.style.backgroundColor = darkBg;
        }
        if (style.color && style.color !== '') {
            el.style.color = darkText;
        }
    }

    function restoreColors(el) {
        if (originalStyles.has(el)) {
            const orig = originalStyles.get(el);
            el.style.backgroundColor = orig.backgroundColor;
            el.style.color = orig.color;
        }
    }

    // --- Pseudo-element override logic with alpha preservation ---
    function hexToRgb(hex) {
        // Remove # if present
        hex = hex.replace(/^#/, '');
        if (hex.length === 3) {
            hex = hex.split('').map(x => x + x).join('');
        }
        const num = parseInt(hex, 16);
        return [
            (num >> 16) & 255,
            (num >> 8) & 255,
            num & 255
        ];
    }

    function parseRgbString(rgbStr) {
        // Accepts rgb(a) or hsl(a) and returns [r, g, b, a]
        let m = rgbStr.match(/rgba?\(([^)]+)\)/);
        if (m) {
            let parts = m[1].split(',').map(s => s.trim());
            let r = parseInt(parts[0], 10);
            let g = parseInt(parts[1], 10);
            let b = parseInt(parts[2], 10);
            let a = parts[3] !== undefined ? parseFloat(parts[3]) : 1;
            return [r, g, b, a];
        }
        m = rgbStr.match(/hsla?\(([^)]+)\)/);
        if (m) {
            // Convert hsl to rgb
            let parts = m[1].split(',').map(s => s.trim());
            let h = parseFloat(parts[0]);
            let s = parseFloat(parts[1]) / 100;
            let l = parseFloat(parts[2]) / 100;
            let a = parts[3] !== undefined ? parseFloat(parts[3]) : 1;
            // hsl to rgb conversion
            let c = (1 - Math.abs(2 * l - 1)) * s;
            let x = c * (1 - Math.abs((h / 60) % 2 - 1));
            let m_ = l - c / 2;
            let r1, g1, b1;
            if (h < 60) { r1 = c; g1 = x; b1 = 0; }
            else if (h < 120) { r1 = x; g1 = c; b1 = 0; }
            else if (h < 180) { r1 = 0; g1 = c; b1 = x; }
            else if (h < 240) { r1 = 0; g1 = x; b1 = c; }
            else if (h < 300) { r1 = x; g1 = 0; b1 = c; }
            else { r1 = c; g1 = 0; b1 = x; }
            let r = Math.round((r1 + m_) * 255);
            let g = Math.round((g1 + m_) * 255);
            let b = Math.round((b1 + m_) * 255);
            return [r, g, b, a];
        }
        return null;
    }

    function getDarkRgbArray() {
        let darkBg = dmtfSettings.dark_bg_color;
        if (darkBg.startsWith('rgb')) {
            let arr = parseRgbString(darkBg);
            if (arr) return arr;
        }
        // Assume hex
        return hexToRgb(darkBg);
    }

    function getAllPseudoBgRules() {
        let rules = [];
        for (let sheet of document.styleSheets) {
            let cssRules;
            try {
                cssRules = sheet.cssRules;
            } catch (e) {
                continue; // CORS or inaccessible stylesheet
            }
            if (!cssRules) continue;
            for (let rule of cssRules) {
                if (rule instanceof CSSStyleRule) {
                    const selector = rule.selectorText;
                    if (selector && (selector.endsWith(':before') || selector.endsWith(':after'))) {
                        const bg = rule.style.getPropertyValue('background-color');
                        if (bg && bg !== 'transparent' && bg !== 'rgba(0, 0, 0, 0)') {
                            rules.push({ selector, bg });
                        }
                    }
                }
            }
        }
        return rules;
    }

    function injectPseudoBgOverrides(darkBg) {
        // Remove old style if exists
        let old = document.getElementById(PSEUDO_OVERRIDE_ID);
        if (old) old.remove();
        const rules = getAllPseudoBgRules();
        if (rules.length === 0) return;
        const darkRgb = getDarkRgbArray();
        const css = rules.map(({selector, bg}) => {
            let alpha = 1;
            let parsed = parseRgbString(bg);
            if (parsed) {
                alpha = parsed[3];
            }
            // If darkBg is rgb array, use it, else fallback to hex
            let color = (alpha < 1)
                ? `rgba(${darkRgb[0]}, ${darkRgb[1]}, ${darkRgb[2]}, ${alpha})`
                : (dmtfSettings.dark_bg_color.startsWith('rgb') ? dmtfSettings.dark_bg_color : `#${dmtfSettings.dark_bg_color.replace(/^#/, '')}`);
            return `body.dmtf-dark-mode ${selector} { background-color: ${color} !important; }`;
        }).join('\n');
        const style = document.createElement('style');
        style.id = PSEUDO_OVERRIDE_ID;
        style.textContent = css;
        document.head.appendChild(style);
    }

    function removePseudoBgOverrides() {
        let old = document.getElementById(PSEUDO_OVERRIDE_ID);
        if (old) old.remove();
    }

    function updateToggleSwitch(enabled) {
        const toggle = document.getElementById('dmtf-toggle');
        if (!toggle) return;
        
        const thumb = toggle.querySelector('.dmtf-switch-thumb');
        const sunIcon = toggle.querySelector('.dmtf-sun-icon');
        
        if (enabled) {
            // Dark mode enabled - move thumb to right
            thumb.style.transform = 'translateX(24px)';
            // Optional: slightly dim the sun icon in dark mode
            sunIcon.style.opacity = '0.7';
        } else {
            // Dark mode disabled - move thumb to left
            thumb.style.transform = 'translateX(0)';
            // Full brightness for light mode
            sunIcon.style.opacity = '1';
        }
    }

    function applyDarkMode(enabled) {
        if (enabled) {
            document.body.classList.add('dmtf-dark-mode');
            document.documentElement.style.setProperty('--dmtf-dark-bg', dmtfSettings.dark_bg_color);
            document.documentElement.style.setProperty('--dmtf-dark-text', dmtfSettings.dark_text_color);
            document.documentElement.style.setProperty('--dmtf-toggle-bg', dmtfSettings.toggle_button_bg);
            document.documentElement.style.setProperty('--dmtf-toggle-text', dmtfSettings.toggle_button_text);

            // Traverse and override
            document.querySelectorAll('*').forEach(el => {
                overrideColors(el, dmtfSettings.dark_bg_color, dmtfSettings.dark_text_color);
            });

            // Observe for dynamic content
            if (!observer) {
                observer = new MutationObserver(mutations => {
                    mutations.forEach(mutation => {
                        mutation.addedNodes.forEach(node => {
                            if (node.nodeType === 1) { // ELEMENT_NODE
                                overrideColors(node, dmtfSettings.dark_bg_color, dmtfSettings.dark_text_color);
                                node.querySelectorAll && node.querySelectorAll('*').forEach(child => {
                                    overrideColors(child, dmtfSettings.dark_bg_color, dmtfSettings.dark_text_color);
                                });
                            }
                        });
                    });
                });
                observer.observe(document.body, { childList: true, subtree: true });
            }

            // Inject pseudo-element background overrides
            injectPseudoBgOverrides(dmtfSettings.dark_bg_color);

            // Custom CSS
            if (dmtfSettings.custom_css) {
                let style = document.getElementById('dmtf-custom-css');
                if (!style) {
                    style = document.createElement('style');
                    style.id = 'dmtf-custom-css';
                    document.head.appendChild(style);
                }
                style.innerHTML = dmtfSettings.custom_css;
            }
        } else {
            document.body.classList.remove('dmtf-dark-mode');
            document.querySelectorAll('*').forEach(el => {
                restoreColors(el);
            });
            if (observer) {
                observer.disconnect();
                observer = null;
            }
            removePseudoBgOverrides();
        }
        
        // Update toggle switch appearance
        updateToggleSwitch(enabled);
    }

    function getPref() {
        return localStorage.getItem('dmtf-dark-mode') === 'enabled';
    }

    function setPref(enabled) {
        localStorage.setItem('dmtf-dark-mode', enabled ? 'enabled' : 'disabled');
    }

    document.addEventListener('DOMContentLoaded', function() {
        var toggle = document.getElementById('dmtf-toggle');
        if (!toggle) return;
        var enabled = getPref();
        applyDarkMode(enabled);
        
        // Handle click events
        toggle.addEventListener('click', function() {
            enabled = !enabled;
            applyDarkMode(enabled);
            setPref(enabled);
        });
        
        // Handle keyboard events for accessibility
        toggle.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                enabled = !enabled;
                applyDarkMode(enabled);
                setPref(enabled);
            }
        });
    });
})(); 