<?php
/*
Plugin Name: Customer Export CSV for Woo
Description: Export WooCommerce customers to CSV with advanced filtering options and field selection.
Version: 12.0
Author: emilion.co.il
Author URI: https://emilion.co.il
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// Make sure WooCommerce is active
if ( ! in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
    return;
}

// Add menu item under WooCommerce menu
add_action('admin_menu', 'cecfw_export_menu');
function cecfw_export_menu() {
    add_submenu_page(
        'woocommerce',
        'Export Customers',
        'Export Customers',
        'manage_options',
        'customer-export-csv-for-woo',
        'cecfw_export_page'
    );
}

// Create export page
function cecfw_export_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'customer-export-csv-for-woo'));
    }

    // Check if export request exists
    if (isset($_POST['cecfw_customer_export_submit']) && check_admin_referer('cecfw_customer_export_nonce', 'cecfw_customer_export_nonce')) {
        cecfw_export_process();
        return; // Important: stops the function execution here
    }

    $available_fields = array(
        'email' => 'Email',
        'phone' => 'Phone',
        'full_name' => 'Full Name',
        'date_created' => 'Registration Date',
        'total_spent' => 'Total Amount Spent',
        'order_count' => 'Number of Orders',
        'billing_address' => 'Billing Address',
        'shipping_address' => 'Shipping Address',
        'last_order_date' => 'Last Order Date'
    );
    ?>
    <div class="wrap">
        <h1>Export WooCommerce Customers</h1>
        <form method="post" action="">
            <?php wp_nonce_field('cecfw_customer_export_nonce', 'cecfw_customer_export_nonce'); ?>
            
            <h3>Select Fields to Export</h3>
            <fieldset>
                <legend class="screen-reader-text"><span>Select Fields to Export</span></legend>
                <?php foreach ($available_fields as $field_key => $field_label) : ?>
                    <label for="field_<?php echo esc_attr($field_key); ?>">
                        <input type="checkbox" id="field_<?php echo esc_attr($field_key); ?>" name="export_fields[]" value="<?php echo esc_attr($field_key); ?>" checked>
                        <?php echo esc_html($field_label); ?>
                    </label><br>
                <?php endforeach; ?>
            </fieldset>

            <h3>Filter Options</h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="date_from">Registration Date From:</label></th>
                    <td><input type="date" id="date_from" name="date_from"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="date_to">Registration Date To:</label></th>
                    <td><input type="date" id="date_to" name="date_to"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="total_spent_min">Minimum Amount Spent:</label></th>
                    <td><input type="number" id="total_spent_min" name="total_spent_min" min="0" step="0.01"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="total_spent_max">Maximum Amount Spent:</label></th>
                    <td><input type="number" id="total_spent_max" name="total_spent_max" min="0" step="0.01"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="order_count_min">Minimum Order Count:</label></th>
                    <td><input type="number" id="order_count_min" name="order_count_min" min="0"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="order_count_max">Maximum Order Count:</label></th>
                    <td><input type="number" id="order_count_max" name="order_count_max" min="0"></td>
                </tr>
            </table>
            
            <p class="submit">
                <input type="submit" name="cecfw_customer_export_submit" class="button button-primary" value="Export Customers">
            </p>
        </form>
    </div>
    <?php
}

// Process export
function cecfw_export_process() {
    if (!check_admin_referer('cecfw_customer_export_nonce', 'cecfw_customer_export_nonce')) {
        wp_die(esc_html__('Security verification failed.', 'customer-export-csv-for-woo'));
    }

    // Prevent HTML output
    if (ob_get_length()) {
        ob_end_clean();
    }

    $filename = sanitize_file_name('woocommerce_customers_export_' . gmdate('Y-m-d') . '.csv');
    
    // Create CSV content
    $csv_content = '';
    
    // Add BOM for UTF-8
    $csv_content .= "\xEF\xBB\xBF";

    $selected_fields = isset($_POST['export_fields']) ? array_map('sanitize_text_field', wp_unslash($_POST['export_fields'])) : array();
    
    // Create header row
    $header_row = array();
    foreach ($selected_fields as $field) {
        $header_row[] = cecfw_sanitize_csv_field(cecfw_get_field_label($field));
    }
    $csv_content .= implode(',', array_map('cecfw_escape_csv', $header_row)) . "\n";

    // Get user data
    $args = array(
        'role' => 'customer',
        'fields' => 'ID'
    );

    if (!empty($_POST['date_from'])) {
        $args['date_query'][0]['after'] = sanitize_text_field(wp_unslash($_POST['date_from']));
    }
    if (!empty($_POST['date_to'])) {
        $args['date_query'][0]['before'] = sanitize_text_field(wp_unslash($_POST['date_to']));
    }

    $customer_ids = get_users($args);

    foreach ($customer_ids as $customer_id) {
        $customer = new WC_Customer($customer_id);
        
        $row = array();
        foreach ($selected_fields as $field) {
            $value = cecfw_get_customer_field_value($customer, $field);
            $row[] = cecfw_escape_csv(cecfw_sanitize_csv_field($value));
        }
        $csv_content .= implode(',', $row) . "\n";
    }

    // Set headers
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="' . esc_attr($filename) . '"');
    
    // Print sanitized content
    echo esc_html($csv_content);
    exit();
}

// New function for sanitizing CSV fields
function cecfw_sanitize_csv_field($value) {
    // Remove dangerous characters
    $value = wp_strip_all_tags($value);
    
    // Remove characters that might be dangerous in CSV
    $value = str_replace(array("\r", "\n", "\t"), ' ', $value);
    
    // Remove invisible characters
    $value = preg_replace('/[\x00-\x1F\x7F]/', '', $value);
    
    return $value;
}

// Update escape function
function cecfw_escape_csv($value) {
    $value = (string)$value; // Ensure value is string
    if (strpos($value, ',') !== false || strpos($value, '"') !== false || strpos($value, "\n") !== false) {
        return '"' . str_replace('"', '""', $value) . '"';
    }
    return $value;
}

function cecfw_get_field_label($field) {
    $labels = array(
        'email' => 'Email',
        'phone' => 'Phone',
        'full_name' => 'Full Name',
        'date_created' => 'Registration Date',
        'total_spent' => 'Total Amount Spent',
        'order_count' => 'Number of Orders',
        'billing_address' => 'Billing Address',
        'shipping_address' => 'Shipping Address',
        'last_order_date' => 'Last Order Date'
    );
    return isset($labels[$field]) ? $labels[$field] : $field;
}

function cecfw_get_customer_field_value($customer, $field) {
    switch ($field) {
        case 'email':
            return $customer->get_email();
        case 'phone':
            return $customer->get_billing_phone();
        case 'full_name':
            return $customer->get_first_name() . ' ' . $customer->get_last_name();
        case 'date_created':
            return $customer->get_date_created()->date('Y-m-d');
        case 'total_spent':
            return $customer->get_total_spent();
        case 'order_count':
            return $customer->get_order_count();
        case 'billing_address':
            return $customer->get_billing_address_1() . ', ' . $customer->get_billing_city() . ', ' . $customer->get_billing_country();
        case 'shipping_address':
            return $customer->get_shipping_address_1() . ', ' . $customer->get_shipping_city() . ', ' . $customer->get_shipping_country();
        case 'last_order_date':
            $last_order = $customer->get_last_order();
            return $last_order ? $last_order->get_date_created()->date('Y-m-d') : '';
        default:
            return '';
    }
}