<?php
/*
Plugin Name: CubeMage Smart Link Checker
Plugin URI: https://www.cubemage.com/plugins/smart-link-checker/
Description: A simple and efficient broken link checker by CubeMage. Auto-scan and detect broken links locally.
Version: 1.0.0
Author: CubeMage
Author URI: https://www.cubemage.com
License: GPLv2 or later
Text Domain: cubemage-smart-link-checker
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class CubeMage_Link_Checker
 */
class CubeMage_Link_Checker {

    private $table_name;

    const OPTION_DELAY        = 'cubemage_check_delay';
    const OPTION_ENABLE_EMAIL = 'cubemage_enable_email';
    const OPTION_EMAIL_ADDR   = 'cubemage_email_address';
    const OPTION_UA_MODE      = 'cubemage_ua_mode';
    const OPTION_GROUP        = 'cubemage_options_group';

    // 必须与 URL Slug 保持一致
    const MENU_SLUG = 'cubemage-smart-link-checker';

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'cubemage_broken_links';

        register_activation_hook( __FILE__, [ $this, 'activate_plugin' ] );

        add_action( 'admin_menu', [ $this, 'create_menu' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );

        // 【新增】专门用于加载后台脚本的 Hook
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_assets' ] );

        add_action( 'wp_ajax_cubemage_scan_links', [ $this, 'ajax_scan_links' ] );
        add_action( 'wp_ajax_cubemage_check_status', [ $this, 'ajax_check_status' ] );
    }

    public function activate_plugin() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $table_name      = $this->table_name;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            url text NOT NULL,
            post_id bigint(20) NOT NULL,
            link_type varchar(20) DEFAULT 'external',
            status_code int DEFAULT 0,
            last_checked datetime DEFAULT '0000-00-00 00:00:00',
            is_broken boolean DEFAULT 0,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );

        if ( false === get_option( self::OPTION_DELAY ) ) {
            update_option( self::OPTION_DELAY, '0.5' );
        }
        if ( false === get_option( self::OPTION_UA_MODE ) ) {
            update_option( self::OPTION_UA_MODE, 'system' );
        }
    }

    public function create_menu() {
        add_menu_page( 'Smart Link Checker', 'Link Checker', 'manage_options', self::MENU_SLUG, [ $this, 'render_admin_page' ], 'dashicons-admin-links', 80 );
    }

    /**
     * 【新增】加载后台脚本
     */
    public function enqueue_admin_assets( $hook ) {
        // 确保只在插件自己的设置页面加载 JS，避免冲突
        if ( strpos( $hook, self::MENU_SLUG ) === false ) {
            return;
        }

        wp_enqueue_script(
            'cm-admin-script',
            plugins_url( 'assets/js/cubemage.js', __FILE__ ),
            [ 'jquery' ], // 依赖 jQuery
            '1.0.0',
            true // 在底部加载
        );

        // 将 PHP 变量传递给 JS (替代直接 echo)
        wp_localize_script( 'cm-admin-script', 'cm_vars', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
        ]);
    }

    public function register_settings() {
        register_setting(
            self::OPTION_GROUP,
            self::OPTION_DELAY,
            [
                'type'              => 'number',
                'sanitize_callback' => [ $this, 'sanitize_delay_value' ],
                'default'           => 0.5,
            ]
        );
        register_setting(
            self::OPTION_GROUP,
            self::OPTION_ENABLE_EMAIL,
            [
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ]
        );
        register_setting(
            self::OPTION_GROUP,
            self::OPTION_EMAIL_ADDR,
            [
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_email',
            ]
        );
        register_setting(
            self::OPTION_GROUP,
            self::OPTION_UA_MODE,
            [
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_key',
            ]
        );
    }

    public function sanitize_delay_value( $input ) {
        $value = floatval( $input );
        if ( $value < 0.1 ) {
            return 0.1;
        }
        if ( $value > 10 ) {
            return 10;
        }
        return $value;
    }

    public function render_admin_page() {
        $current_delay = get_option( self::OPTION_DELAY, '0.5' );
        $enable_email  = get_option( self::OPTION_ENABLE_EMAIL, '' );
        $admin_email   = get_option( 'admin_email' );
        $custom_email  = get_option( self::OPTION_EMAIL_ADDR, $admin_email );
        $ua_mode       = get_option( self::OPTION_UA_MODE, 'system' );

        // 【修正】所有 Text Domain 改为 cubemage-smart-link-checker
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( 'Smart Link Checker', 'cubemage-smart-link-checker' ); ?> <span style="font-size:12px;">by <a href="https://www.cubemage.com" target="_blank"><?php echo esc_html__( 'CubeMage', 'cubemage-smart-link-checker' ); ?></a></span></h1>
            <div style="background:#fff; padding:20px; border:1px solid #ccc; margin-bottom:20px; max-width: 800px;">
                <h3><?php echo esc_html__( '⚙️ Settings', 'cubemage-smart-link-checker' ); ?></h3>
                <form method="post" action="options.php">
                    <?php settings_fields( self::OPTION_GROUP ); ?>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><?php echo esc_html__( 'Check Interval (Seconds)', 'cubemage-smart-link-checker' ); ?></th>
                            <td>
                                <input type="number" step="0.1" min="0.1" max="10" name="<?php echo esc_attr( self::OPTION_DELAY ); ?>" value="<?php echo esc_attr( $current_delay ); ?>" style="width: 100px;" />
                                <span class="description"> <?php echo esc_html__( 'Rec: 0.5', 'cubemage-smart-link-checker' ); ?></span>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><?php echo esc_html__( 'User-Agent Mode', 'cubemage-smart-link-checker' ); ?></th>
                            <td>
                                <select name="<?php echo esc_attr( self::OPTION_UA_MODE ); ?>">
                                    <option value="system" <?php selected( $ua_mode, 'system' ); ?>><?php echo esc_html__( 'System Auto', 'cubemage-smart-link-checker' ); ?></option>
                                    <option value="browser" <?php selected( $ua_mode, 'browser' ); ?>><?php echo esc_html__( 'My Current Browser', 'cubemage-smart-link-checker' ); ?></option>
                                </select>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><?php echo esc_html__( 'Email Notification', 'cubemage-smart-link-checker' ); ?></th>
                            <td>
                                <fieldset>
                                    <label><input name="<?php echo esc_attr( self::OPTION_ENABLE_EMAIL ); ?>" type="checkbox" id="enable_email" value="1" <?php checked( '1', $enable_email ); ?> /> <?php echo esc_html__( 'Send me an email when scan is complete', 'cubemage-smart-link-checker' ); ?></label>
                                    <br><br>
                                    <input type="email" name="<?php echo esc_attr( self::OPTION_EMAIL_ADDR ); ?>" value="<?php echo esc_attr( $custom_email ); ?>" class="regular-text" placeholder="<?php echo esc_attr( $admin_email ); ?>" />
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button(); ?>
                </form>
            </div>

            <div style="margin-bottom: 20px;">
                <button id="btn-auto-process" class="button button-primary button-hero"><?php echo esc_html__( 'Start Full Scan & Check', 'cubemage-smart-link-checker' ); ?></button>
                <button id="btn-stop" class="button button-secondary button-hero" style="display:none; margin-left:10px; color:#d63638; border-color:#d63638;"><?php echo esc_html__( '🛑 Stop Process', 'cubemage-smart-link-checker' ); ?></button>
                <div id="cm-progress-wrapper" style="margin-top:20px; display:none;">
                    <div style="display:flex; justify-content:space-between; margin-bottom:5px; font-weight:bold;">
                        <span id="cm-progress-text"><?php echo esc_html__( 'Initializing...', 'cubemage-smart-link-checker' ); ?></span>
                        <span id="cm-progress-percent">0%</span>
                    </div>
                    <div style="width:100%; background:#f0f0f1; border-radius:10px; height:20px; overflow:hidden;">
                        <div id="cm-progress-bar" style="width:0%; height:100%; background:#2271b1; transition:width 0.5s ease;"></div>
                    </div>
                </div>
            </div>
            <div id="cm-log" style="margin-top:20px; padding:15px; background:#1e1e1e; color:#0f0; font-family:monospace; max-height:300px; overflow-y:scroll; border-radius:5px; display:none;"></div>
            <h3><?php echo esc_html__( 'Link Report', 'cubemage-smart-link-checker' ); ?></h3>
            <?php $this->render_link_list(); ?>
        </div>

        <?php
    }

    public function ajax_scan_links() {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $posts      = $wpdb->get_results( "SELECT ID, post_content FROM $wpdb->posts WHERE post_status='publish' AND post_type='post'" );
        $table_name = $this->table_name;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query( "TRUNCATE TABLE $table_name" );

        $total_count = 0;
        $home_url    = home_url();

        foreach ( $posts as $post ) {
            preg_match_all( '/href=["\']([^"\']+)["\']/i', $post->post_content, $matches );
            if ( ! empty( $matches[1] ) ) {
                foreach ( $matches[1] as $url ) {
                    $url = trim( $url );
                    if ( empty( $url ) || $url[0] === '#' || stripos( $url, 'mailto:' ) === 0 ) {
                        continue;
                    }
                    if ( strpos( $url, 'http' ) !== 0 ) {
                        $url = ( $url[0] === '/' ) ? home_url( $url ) : home_url( '/' . $url );
                    }
                    $type = ( strpos( $url, $home_url ) !== false ) ? 'internal' : 'external';

                    $sql = "SELECT id FROM $table_name WHERE url = %s AND post_id = %d";
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.NotPrepared
                    $exists = $wpdb->get_var( $wpdb->prepare( $sql, $url, $post->ID ) );

                    if ( ! $exists ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                        $wpdb->insert(
                            $table_name,
                            [
                                'url'         => $url,
                                'post_id'     => $post->ID,
                                'link_type'   => $type,
                                'status_code' => 0,
                            ],
                            [ '%s', '%d', '%s', '%d' ]
                        );
                        $total_count++;
                    }
                }
            }
        }
        wp_send_json_success( [ 'message' => 'Scan Complete.', 'total' => $total_count ] );
    }

    public function ajax_check_status() {
        global $wpdb;
        $table_name = $this->table_name;
        $delay      = (float) get_option( self::OPTION_DELAY, '0.5' );
        $delay      = ( $delay < 0.1 ) ? 0.1 : ( ( $delay > 10 ) ? 10 : $delay );

        $ua_mode = get_option( self::OPTION_UA_MODE, 'system' );
        $ua      = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36';

        if ( $ua_mode === 'browser' && isset( $_SERVER['HTTP_USER_AGENT'] ) ) {
            $raw_ua = wp_unslash( $_SERVER['HTTP_USER_AGENT'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $ua     = sanitize_text_field( $raw_ua );
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $links = $wpdb->get_results( "SELECT id, url FROM $table_name WHERE status_code = 0 LIMIT 5" );

        if ( empty( $links ) ) {
            $email_res = $this->maybe_send_email();
            wp_send_json_success( [ 'message' => 'Check complete.', 'remaining' => 0, 'email_result' => $email_res ] );
        }

        foreach ( $links as $link ) {
            $resp = wp_remote_head( $link->url, [ 'timeout' => 10, 'redirection' => 5, 'user-agent' => $ua ] );
            $code = is_wp_error( $resp ) ? 0 : wp_remote_retrieve_response_code( $resp );
            if ( $code == 405 || $code == 403 ) {
                $resp_get = wp_remote_get( $link->url, [ 'timeout' => 10, 'redirection' => 5, 'user-agent' => $ua ] );
                if ( ! is_wp_error( $resp_get ) ) {
                    $code = wp_remote_retrieve_response_code( $resp_get );
                }
            }
            $broken = ( $code >= 400 || $code == 0 ) ? 1 : 0;

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $wpdb->update( $table_name, [ 'status_code' => $code, 'is_broken' => $broken, 'last_checked' => current_time( 'mysql' ) ], [ 'id' => $link->id ] );
            usleep( (int) ( $delay * 1000000 ) );
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $rem = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name WHERE status_code = 0" );

        $msg       = "Checked " . count( $links ) . " links.";
        $email_res = ( $rem == 0 ) ? $this->maybe_send_email() : null;
        wp_send_json_success( [ 'message' => $msg, 'remaining' => $rem, 'email_result' => $email_res ] );
    }

    private function maybe_send_email() {
        global $wpdb;
        $table_name = $this->table_name;

        if ( get_option( self::OPTION_ENABLE_EMAIL ) !== '1' ) {
            return null;
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        if ( $wpdb->get_var( "SELECT COUNT(*) FROM $table_name WHERE status_code = 0" ) > 0 ) {
            return null;
        }

        $to = get_option( self::OPTION_EMAIL_ADDR, get_option( 'admin_email' ) );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $broken = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name WHERE is_broken = 1" );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $total = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );

        return wp_mail( $to, '[Smart Link Checker] Scan Completed', "Total: $total\nBroken: $broken\nView: " . site_url( '/wp-admin/admin.php?page=' . self::MENU_SLUG ) );
    }

    public function render_link_list() {
        global $wpdb;
        $table_name = $this->table_name;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $links = $wpdb->get_results( "SELECT * FROM $table_name LIMIT 100" );

        if ( empty( $links ) ) {
            echo '<div style="padding:10px; background:#fff; border-left:4px solid #ccc;"><p>' . esc_html__( 'Ready.', 'cubemage-smart-link-checker' ) . '</p></div>';
            return;
        }

        echo '<table class="wp-list-table widefat fixed striped"><thead><tr><th>' . esc_html__( 'Type', 'cubemage-smart-link-checker' ) . '</th><th>' . esc_html__( 'URL', 'cubemage-smart-link-checker' ) . '</th><th>' . esc_html__( 'Status', 'cubemage-smart-link-checker' ) . '</th><th>' . esc_html__( 'Found In', 'cubemage-smart-link-checker' ) . '</th><th>' . esc_html__( 'Actions', 'cubemage-smart-link-checker' ) . '</th></tr></thead><tbody>';

        foreach ( $links as $link ) {
            $edit   = get_edit_post_link( $link->post_id );
            $status = ( $link->status_code == 0 ) ? '<span style="color:gray;">Pending</span>' : ( ( $link->is_broken ) ? "<span style='color:red; font-weight:bold;'>{$link->status_code}</span>" : "<span style='color:green;'>{$link->status_code}</span>" );
            $type   = ( $link->link_type === 'internal' ) ? 'Internal' : 'External';

            echo "<tr>
                <td>" . esc_html( $type ) . "</td>
                <td style='word-break:break-all;'><a href='" . esc_url( $link->url ) . "' target='_blank'>" . esc_html( $link->url ) . "</a></td>
                <td>" . wp_kses_post( $status ) . "</td>
                <td><a href='" . esc_url( $edit ) . "' target='_blank'>Post #" . intval( $link->post_id ) . "</a></td>
                <td><a href='" . esc_url( $edit ) . "' target='_blank' class='button button-small'>" . esc_html__( 'Fix', 'cubemage-smart-link-checker' ) . "</a></td>
            </tr>";
        }
        echo '</tbody></table>';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $count = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );
        if ( $count > 100 ) {
            /* translators: %d: number of links */
            echo "<p><i>" . sprintf( esc_html__( 'Showing first 100 of %d links.', 'cubemage-smart-link-checker' ), intval( $count ) ) . "</i></p>";
        }
    }
}

new CubeMage_Link_Checker();