<?php

/**
 * Plugin Name: CRS Booker
 * Plugin URI: http://www.car-rental-software.it
 * Description: CRS Booker
 * Author: My Appy
 * Author URI: http://www.myappy.it/
 * Version: 1.2.8
 * Text Domain: crs-booker
 * License: GPLv2 or later
 *
 * Copyright: (c) 2024-2025 My Appy (info@myappy.net)
 *
 * @author    My Appy
 * @copyright Copyright (c) 2024-2025, My Appy
 *
 */

/**

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

Copyright (c) 2024-2025, My Appy

**/

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

define('CRSBKR_VERSION', '1.2.8');

class CRSBKR_BOOKER
{
	function __construct($locale = NULL)
	{
		$defLang = array_keys($this->STRINGS);
		$defLang = array_shift($defLang);

		$lang = get_query_var('crsbkr_lang');
		if ($lang == NULL && $locale)
		{
			if (($p = strpos($locale, '_')) !== FALSE)
			{
				$locale = substr($locale, 0, $p);
			}

			if (($p = strpos($locale, '-')) !== FALSE)
			{
				$locale = substr($locale, 0, $p);
			}

			if (isset($this->STRINGS[$locale]))
			{
				$lang = $locale;
			}
		}

		if (!$lang)
		{
			$lang = $defLang;
		}

		$this->LANG = $lang;

		// Set the default colors.
		if (!get_option('crsbkr_hue') || get_option('crsbkr_hue') == '')
		{
			if (get_option('crsbkr_hue') === NULL)
			{
				add_option('crsbkr_hue', '120');
			}
			else
			{
				update_option('crsbkr_hue', '120');
			}
		}

		if (!get_option('crsbkr_sat') || get_option('crsbkr_sat') == '')
		{
			if (get_option('crsbkr_sat') === NULL)
			{
				add_option('crsbkr_sat', '100%');
			}
			else
			{
				update_option('crsbkr_sat', '100%');
			}
		}

		if (!get_option('crsbkr_lig') || get_option('crsbkr_lig') == '')
		{
			if (get_option('crsbkr_lig') === NULL)
			{
				add_option('crsbkr_lig', '35%');
			}
			else
			{
				update_option('crsbkr_lig', '35%');
			}
		}
	}

	function add_scripts()
	{
		wp_enqueue_script('crsbkr_script');
		wp_enqueue_style('crsbkr_style');

		$inline = ':root
		{
			--crsbkr-hue: ' . get_option('crsbkr_hue') . ';
			--crsbkr-sat: ' . get_option('crsbkr_sat') . ';
			--crsbkr-lig: ' . get_option('crsbkr_lig') . ';

			--crsbkr-main: hsl(var(--crsbkr-hue), var(--crsbkr-sat), var(--crsbkr-lig));
			--crsbkr-dark: hsl(var(--crsbkr-hue), var(--crsbkr-sat), calc(var(--crsbkr-lig) * 0.75));
			--crsbkr-darker: hsl(var(--crsbkr-hue), var(--crsbkr-sat), calc(var(--crsbkr-lig) * 0.5));
			--crsbkr-lighter : #f5f5f5; /* rgba(0, 0, 0, 0.04); /* hsl(var(--hue), var(--sat), 97%); /* faf3e8 */
			--crsbkr-light : #eaeaea; /* rgba(0, 0, 0, 0.1); /* hsl(var(--hue), var(--sat), 90%); /* fadfbc */
			--crsbkr-medium : hsl(var(--hue), var(--sat), 50%); /* fadfbc */
			--crsbkr-transparent: hsla(var(--crsbkr-hue), var(--crsbkr-sat), calc(var(--crsbkr-lig)), 0.1);
		}';

		wp_add_inline_style('crsbkr_style', esc_html($inline));
	}

	function ajax($params = NULL)
	{
		check_ajax_referer('ajax');

		$error = NULL;
		$res = [ 'success' => FALSE ];

		$endpoint = NULL;
		$headers = NULL;
		$method = NULL;
		$request = NULL;
		$data = NULL;

		$noOutput = FALSE;
		if (!$params)
		{
			$endpoint = isset($_POST['endpoint']) ? sanitize_text_field(wp_unslash($_POST['endpoint'])) : NULL;
			$method = isset($_POST['method']) ? sanitize_text_field(wp_unslash($_POST['method'])) : 'GET';
			$request = isset($_POST['request']) ? sanitize_text_field(wp_unslash($_POST['request'])) : '';
			$data = isset($_POST['params']) ? urldecode(base64_decode(sanitize_text_field(wp_unslash($_POST['params'])))) : '';
		}
		else
		{
			$endpoint = isset($params['endpoint']) ? sanitize_text_field(wp_unslash($params['endpoint'])) : NULL;
			$method = isset($params['method']) ? $params['method'] : 'GET';
			$request = isset($params['request']) ? $params['request'] : '';
			$data = isset($params['params']) ? base64_decode($params['params']) : '';

			$noOutput = TRUE;
		}

		if ($request == 'fbq' && $data)
		{
			$jsonData = json_decode($data, TRUE);
			if ($jsonData && isset($jsonData['custom_data']) && isset($jsonData['custom_data']['controller_form']) && isset($_SESSION['CRSBKR_COF_' . $jsonData['custom_data']['controller_form'] . '_META_PIXEL']))
			{
				$pixelId = sanitize_text_field(isset($_SESSION['CRSBKR_COF_' . $jsonData['custom_data']['controller_form'] . '_META_PIXEL']['id']) ? $_SESSION['CRSBKR_COF_' . $jsonData['custom_data']['controller_form'] . '_META_PIXEL']['id'] : '');
				$token = sanitize_text_field(isset($_SESSION['CRSBKR_COF_' . $jsonData['custom_data']['controller_form'] . '_META_PIXEL']['token']) ? $_SESSION['CRSBKR_COF_' . $jsonData['custom_data']['controller_form'] . '_META_PIXEL']['token'] : '');

				$endpoint = 'https://graph.facebook.com/v23.0/' . $pixelId;
				$request = 'events?access_token=' . $token;

				$data = [ 'data' => json_encode([ $jsonData ]) ];

				$headers = [ 'Authorization' => '  ', 'Content-Type' => 'application/json' ];
			}
			else
			{
				$res = [ 'error' => 'config error' ];

				if ($noOutput)
				{
					return $res;
				}
				else
				{
					die(wp_json_encode($res));
				}
			}
		}

		$jsonData = NULL;
		if ($method == 'GET' && $data)
		{
			$jsonData = json_decode($data, TRUE);
			if ($jsonData)
			{
				$request .= (strpos($request, '?') !== FALSE ? '&' : '?') . http_build_query($jsonData);
			}
		}

		$res = $this->curl($method, $request, $headers, $data, $endpoint);

		if ($error || !$res)
		{
			$res = [ 'error' => $error, 'success' => FALSE ];
		}

		if ($noOutput)
		{
			return $res;
		}
		else
		{
			die(wp_json_encode($res));
		}
	}

	function curl($method, $url, $headers = NULL, $data = NULL, $endpoint = '', &$httpCode = NULL)
	{
		$res = NULL;
		$error = NULL;

		if ($endpoint == '' && strpos($url, 'client_reservation') !== FALSE)
		{
			$endpoint = get_option('crsbkr_booking_endpoint');
			if (!$endpoint || $endpoint == '')
			{
				$endpoint = 'https://carrentalsoftware.myappy.it/api/v1';
			}
		}
		else if ($endpoint == '')
		{
			$endpoint = get_option('crsbkr_endpoint');
			if (!$endpoint || $endpoint == '')
			{
				$endpoint = 'https://crsbrk00.myappy.it/api/v1';
			}
		}

		$url = $endpoint . '/' . $url;
		if (!$headers)
		{
			$headers = [];
		}

		$key = 'CRSBKR_' . get_option('crsbkr_uuid') . (strpos($url, 'client_reservation') !== FALSE ? '_booking' : '');
		$session = NULL;
		$token = NULL;

		$gotAuth = isset($headers['Authorization']) && strlen($headers['Authorization']) > 0;
		if (!$gotAuth)
		{
			if (!isset($_SESSION[$key]) || !isset($_SESSION[$key]['token']) || !isset($_SESSION[$key]['token_expiry'])
					|| $_SESSION[$key]['token_expiry'] < (new DateTime('now', new DateTimeZone('UTC')))->format('Y-m-d H:i:s'))
			{
				$session = $this->curl('GET', 'session/token', [ 'Authorization' => 'Basic ' . base64_encode(get_option('crsbkr_uuid') . ':' . get_option('crsbkr_api_key')) ], NULL, $endpoint);
				if ($session && isset($session['success']) && $session['success'] && isset($session['result']) && isset($session['result']['session']))
				{
					$_SESSION[$key] = $session['result']['session'];
					$token = $session['result']['session']['token'];
				}
				else
				{
					$res = $session;

					if (isset($_SESSION[$key]))
					{
						unset($_SESSION[$key]);
					}
				}
			}
			else if (isset($_SESSION[$key]) && isset($_SESSION[$key]['token']) && strlen(sanitize_text_field($_SESSION[$key]['token'])) > 0 && isset($_SESSION[$key]['token_expiry'])
					&& $_SESSION[$key]['token_expiry'] >= (new DateTime('now', new DateTimeZone('UTC')))->format('Y-m-d H:i:s'))
			{
				$token = sanitize_text_field($_SESSION[$key]['token']);
			}

			if ($token)
			{
				$gotAuth = TRUE;
				$headers['Authorization'] = 'Bearer ' . $token;
			}
		}

		if ($gotAuth)
		{
			$params =
			[
				'headers' => $headers,
				'method' => $method,
				'timeout' => 60,
			];

			if ($method != 'GET' && $data)
			{
				$params['body'] = $data;
			}

			$httpCode = NULL;
			$result = wp_remote_get($url, $params);

			if (!is_wp_error($result))
			{
				$httpCode = wp_remote_retrieve_response_code($result);
				$result = $result['body'];
			}
			else
			{
				$error = $result->get_error_message();
				$result = NULL;
			}

			$json = $result && $result != '' ? json_decode($result, TRUE) : NULL;
			if ($httpCode && substr('' . $httpCode, 0, 1) != '2')
			{
				if ($json && isset($json['error']))
				{
					if (is_array($json['error']))
					{
						$json['debug'] = [ 'url' => $url, 'params' => $params ];
					}
					else
					{
						// If there's a problem with the session, unset it.
						if (($p = strpos($json['error'], '_A0')) !== FALSE && $p == 0)
						{
							$session = NULL;
							if (isset($_SESSION[$key]))
							{
								unset($_SESSION[$key]);
							}
						}
					}

					$res = $json;
				}
				else
				{
					$res = [ 'error' => '_C0001: ' . $httpCode . ' :: ' . $result, 'url' => $url ];
				}
			}
			else if ($error)
			{
				$res = [ 'error' => '_C0002: ' . $error, 'url' => $url ];
			}
			else
			{
				$res = $json;

				if ($session)
				{
					$now = new DateTime('now', new DateTimeZone('UTC'));
					$now->add(new DateInterval('PT30M'));

					$_SESSION[$key]['token_expiry'] = $now->format('Y-m-d H:i:s');
				}
			}
		}

		if (!$res || trim($headers['Authorization']) != '' && (!isset($res['success']) || !$res['success']))
		{
			$debug = [ 'method' => $method, 'url' => $url, 'data' => $data, 'endpoint' => $endpoint, 'result' => $res ];
			$debug = json_encode($debug, JSON_PRETTY_PRINT);

			$referer = sanitize_text_field(wp_unslash(isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : (isset($_SERVER['QUERY_STRING']) ? $_SERVER['QUERY_STRING'] : '###')));

			$headers = [];
			$to = 'support@myappy.net';

			wp_mail($to, 'CRS Booker error on page ' . $referer,
					str_replace(['{page}', '{debug}'], [$referer, $debug],
					"Ciao,

abbiamo riscontrato un problema sul sito web collegato con CRS tramite il plugin CRS Booker sulla pagina: {page}.

Saluti,
Il tuo plugin CRS Booker

INFORMAZIONI AGGIUNTIVI:

{debug}"), $headers);
		}

		return $res;
	}

	function crsbkr_field_input($key)
	{
		$type = 'text';
		if ($key == 'crsbkr_api_key')
		{
			$type = 'password';
		}
		else if ($key == 'checkbox')
		{
			$type = 'checkbox';
		}
		else if ($key == 'textarea')
		{
			$type = 'textarea';
		}

		if ($type == 'checkbox')
		{
			echo '<input type="checkbox" id="' . esc_attr($key) . '" name="' . esc_attr($key) . '" ' . (get_option($key) || get_option($key) == 'on' ? ' checked="checked"' : '') . ' />';
		}
		else if ($type == 'textarea')
		{
			echo '<textarea id="' . esc_attr($key) . '" name="' . esc_attr($key) . '">' . esc_html(get_option($key)) . '</textarea>';
		}
		else
		{
 			echo '<input type="' . esc_attr($type) . '" id="' . esc_attr($key) . '" name="' . esc_attr($key) . '" value="' . esc_attr(get_option($key)) . '" />';
 		}

 		if (isset($this->STRINGS[$this->LANG][$key . '.desc']))
 		{
 			echo '<p>' . wp_kses_post($this->STRINGS[$this->LANG][$key . '.desc']) . '</p>';
 		}
	}

	function crsbkr_field_page($key)
	{
		echo '<select id="' . esc_attr($key)  . '" name="' . esc_attr($key) . '">';

		$option = get_option($key);
		if ($pages = get_pages())
		{
			foreach ($pages as $page)
			{
				echo '<option value="' . esc_attr($page->ID) . '" ' . selected($page->ID, $option) . '>' . esc_html($page->post_title) . '</option>';
			}
		}

		echo '</select>';

		if (isset($this->STRINGS[$this->LANG][$key . '.desc']))
		{
			echo '<p>' . wp_kses_post($this->STRINGS[$this->LANG][$key . '.desc']) . '</p>';
		}
	}

	function crsbkr_i18n_page_link($id)
	{
		if (function_exists('pll_get_post'))
		{
			$id = pll_get_post($id, $this->LANG);
		}

		return get_page_link($id);
	}

	function crsbkr_register_settings()
	{
		$fields =
		[
			'crsbkr_auth' =>
			[
				'name' => $this->STRINGS[$this->LANG]['crsbkr_auth'],
				'fields' =>
				[
					'crsbkr_endpoint', 'crsbkr_booking_endpoint', 'crsbkr_uuid', 'crsbkr_api_key',
				],
			],

			'crsbkr_styles' =>
			[
				'name' => $this->STRINGS[$this->LANG]['crsbkr_styles'],
				'fields' =>
				[
					'crsbkr_hue', 'crsbkr_sat', 'crsbkr_lig'
				],
			],

			'crsbkr_search' =>
			[
				'name' => $this->STRINGS[$this->LANG]['crsbkr_search'],
				'fields' =>
				[
					'crsbkr_search_results_page'
				],
			],
		];

		foreach ($fields as $k => $v)
		{
			add_settings_section($k, $v['name'], '', 'crsbkr_slug');

			foreach ($v['fields'] as $kk)
			{
				register_setting('crsbkr_options', $kk, 'sanitize_text_field');

				$type = NULL;

				if ($kk == 'crsbkr_search_results_page')
				{
					$type = 'crsbkr_field_page';
				}
				else if (strcmp($kk, 'crsbkr_form_cache_update') == 0)
				{
					$type = 'crsbkr_field_cache_update';
				}
				else
				{
					$type = 'crsbkr_field_input';
				}

				add_settings_field($kk, $this->STRINGS[$this->LANG][$kk], [ $this, $type ], 'crsbkr_slug', $k, $kk);
			}
		}
	}

	function crsbkr_options_page()
	{
		$this->crsbkr_register_settings();

		echo '
			<div class="wrap">
				<h1>' . wp_kses_post($this->STRINGS[$this->LANG]['crsbkr_settings']) . '</h1>
				<form method="post" action="options.php">';

		settings_fields('crsbkr_options'); // settings group name
		do_settings_sections('crsbkr_slug'); // just a page slug

		submit_button();

		echo wp_kses_post($this->STRINGS[$this->LANG]['crsbkr_footer']);
		echo '</div></form>';
	}

	function crsbkr_register_options_page()
	{
		add_options_page('CRS Booker', 'CRS Booker', 'manage_options', 'crsbkr', array($this, 'crsbkr_options_page'));
	}

	function crsbkr_module($attr)
	{
		global $wp;

		$this->add_scripts();

		$data = $this->curl('GET', 'controller_form/' . $attr['code'] . '?expand=acriss_code_web,base,base_opening_hours,base_transfer_time');
		$form = '';

		if ($data && isset($data['success']) && $data['success'])
		{
			$metaPixelId = sanitize_text_field(isset($data['result']) && isset($data['result']['controller_form']) && isset($data['result']['controller_form']['params']) && isset($data['result']['controller_form']['params']['meta_pixel']) && isset($data['result']['controller_form']['params']['meta_pixel']['pixel_id']) ? $data['result']['controller_form']['params']['meta_pixel']['pixel_id'] : '');
			if ($metaPixelId && strlen($metaPixelId) > 0)
			{
				$code = '<!-- Meta Pixel Code -->
	!function(f,b,e,v,n,t,s)
	{if(f.fbq)return;n=f.fbq=function(){n.callMethod?
	n.callMethod.apply(n,arguments):n.queue.push(arguments)};
	if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version=\'2.0\';
	n.queue=[];t=b.createElement(e);t.async=!0;
	t.src=v;s=b.getElementsByTagName(e)[0];
	s.parentNode.insertBefore(t,s)}(window, document,\'script\',
	\'https://connect.facebook.net/en_US/fbevents.js\');
	fbq(\'init\', \'' . $metaPixelId . '\');
	fbq(\'track\', \'PageView\');
	<!-- End Meta Pixel Code -->';

				wp_add_inline_script('crsbkr_script', wp_kses_post($code));
				unset($data['result']['controller_form']['params']['meta_pixel']['pixel_id']);
			}
			else
			{
				$metaPixelId = NULL;
			}

			$metaConversionsApiToken = sanitize_text_field(isset($data['result']) && isset($data['result']['controller_form']) && isset($data['result']['controller_form']['params']) && isset($data['result']['controller_form']['params']['meta_pixel']) && isset($data['result']['controller_form']['params']['meta_pixel']['conversions_api_access_token']) ? $data['result']['controller_form']['params']['meta_pixel']['conversions_api_access_token'] : '');
			if ($metaPixelId && $metaConversionsApiToken && strlen($metaConversionsApiToken) > 0)
			{
				$_SESSION['CRSBKR_COF_' . $data['result']['controller_form']['code'] . '_META_PIXEL'] = [ 'id' => $metaPixelId, 'token' => $metaConversionsApiToken ];
				unset($data['result']['controller_form']['params']['meta_pixel']['conversions_api_access_token']);
			}

			wp_add_inline_script('crsbkr_script', wp_kses_post('jQuery(function( $ )
				{
					CRSBKR_DIR = \'' . plugin_dir_url(__FILE__) . '\';
					CRSBKR_GW_URL = \'' . wp_nonce_url(admin_url('admin-ajax.php'), 'ajax') . '\';
					CRSBKR_RESULTS_PAGE = \'' . wp_nonce_url($this->crsbkr_i18n_page_link(get_option('crsbkr_search_results_page')), 'results') . '\';
					CRSBKR_STRINGS = ' . wp_json_encode($this->STRINGS[$this->LANG]) . ';

					crsbkr_module(' . wp_json_encode(isset($data['result']) ? $data['result'] : $data) . ');
				});'));

			$form = '<div class="crsbkr" id="crsbkr_module"></div>';
		}
		else
		{
			$form = '<div class="crsbkr" id="crsbkr_module"><span class="error">' . $this->STRINGS[$this->LANG]['error'] . (isset($data['error']) ? '<br />' . $data['error'] : '') . '</span></div>';
		}

		return wp_kses_post($form);
	}

	function crsbkr_search_results($attr)
	{
		global $wp;

		$nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : NULL;
		if (!wp_verify_nonce($nonce, 'results'))
		{
			return '';
		}

		$q = !isset($_GET['q']) || $_GET['q'] == '' ? '' : sanitize_text_field(wp_unslash($_GET['q']));
		$f = isset($_GET['f']) ? sanitize_text_field(wp_unslash($_GET['f'])) : '';

		if ($q == '')
		{
			$attr['code'] = $f;
			return $this->crsbkr_module($attr);
		}

		$this->add_scripts();

		$data = $this->curl('GET', 'controller_form/' . $f . '?expand=accessory,acriss_code_web,agent,base,base_opening_hours,base_transfer_time,broker,user');
		$input = json_decode(base64_decode($q), TRUE);
		$rid = isset($_GET['rid']) ? sanitize_text_field(wp_unslash($_GET['rid'])) : NULL;

		$metaPixelId = sanitize_text_field(isset($data['result']) && isset($data['result']['controller_form']) && isset($data['result']['controller_form']['params']) && isset($data['result']['controller_form']['params']['meta_pixel']) && isset($data['result']['controller_form']['params']['meta_pixel']['pixel_id']) ? $data['result']['controller_form']['params']['meta_pixel']['pixel_id'] : '');
		if ($metaPixelId && strlen($metaPixelId) > 0)
		{
			$code = '<!-- Meta Pixel Code -->
!function(f,b,e,v,n,t,s)
{if(f.fbq)return;n=f.fbq=function(){n.callMethod?
n.callMethod.apply(n,arguments):n.queue.push(arguments)};
if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version=\'2.0\';
n.queue=[];t=b.createElement(e);t.async=!0;
t.src=v;s=b.getElementsByTagName(e)[0];
s.parentNode.insertBefore(t,s)}(window, document,\'script\',
\'https://connect.facebook.net/en_US/fbevents.js\');
fbq(\'init\', \'' . $metaPixelId . '\');
fbq(\'track\', \'PageView\');
<!-- End Meta Pixel Code -->';

			wp_add_inline_script('crsbkr_script', wp_kses_post($code));
			unset($data['result']['controller_form']['params']['meta_pixel']['pixel_id']);
		}
		else
		{
			$metaPixelId = NULL;
		}

		$metaConversionsApiToken = sanitize_text_field(isset($data['result']) && isset($data['result']['controller_form']) && isset($data['result']['controller_form']['params']) && isset($data['result']['controller_form']['params']['meta_pixel']) && isset($data['result']['controller_form']['params']['meta_pixel']['conversions_api_access_token']) ? $data['result']['controller_form']['params']['meta_pixel']['conversions_api_access_token'] : '');
		if ($metaPixelId && $metaConversionsApiToken && strlen($metaConversionsApiToken) > 0)
		{
			$_SESSION['CRSBKR_COF_' . $data['result']['controller_form']['code'] . '_META_PIXEL'] = [ 'id' => $metaPixelId, 'token' => $metaConversionsApiToken ];
			unset($data['result']['controller_form']['params']['meta_pixel']['conversions_api_access_token']);
		}

		$params =
		[
			'data' => isset($data['result']) ? $data['result'] : NULL,
			'input' => $input,
			'mpe' => isset($_GET['mpe']) ? sanitize_text_field(wp_unslash($_GET['mpe'])) : NULL,
			'reservation' => $rid
		];

		$content = '';
		if (!isset($input['checkout_time']) || !isset($input['checkin_time']) || $input['checkout_time'] <= (new DateTime('now', new DateTimeZone('UTC')))->format('Y-m-d H:i:s') || $input['checkin_time'] <= $input['checkout_time'])
		{
		}
		else
		{
			$content = '<div class="crsbkr" id="crsbkr_search_results"></div>';
		}

		wp_add_inline_script('crsbkr_script', wp_kses_post('jQuery(function( $ )
			{
				CRSBKR_DIR = \'' . plugin_dir_url(__FILE__) . '\';
				CRSBKR_GW_URL = \'' . wp_nonce_url(admin_url('admin-ajax.php'), 'ajax') . '\';
				CRSBKR_RESULTS_PAGE = \'' . wp_nonce_url($this->crsbkr_i18n_page_link(get_option('crsbkr_search_results_page')), 'results') . '\';
				CRSBKR_STRINGS = ' . wp_json_encode($this->STRINGS[$this->LANG]) . ';

				crsbkr_module(' . wp_json_encode(isset($data['result']) ? $data['result'] : NULL) . ');
				' . ($content != '' ? 'crsbkr_search_results(' . wp_json_encode($params) . ');' : '') . '
			});'));

		$form = '<div class="crsbkr" id="crsbkr_module"></div>' . $content;

		return wp_kses_post($form);
	}

	function crsbkr_webcheck($attr)
	{
		global $wp;

		$this->add_scripts();

		wp_add_inline_script('crsbkr_script', wp_kses_post('jQuery(function( $ )
			{
				CRSBKR_DIR = \'' . plugin_dir_url(__FILE__) . '\';
				CRSBKR_GW_URL = \'' . wp_nonce_url(admin_url('admin-ajax.php'), 'ajax') . '\';
				CRSBKR_STRINGS = ' . wp_json_encode($this->STRINGS[$this->LANG]) . ';

				crsbkr_webcheck(' . wp_json_encode(get_query_var('q')) . ');
			});'));

		$form = '<div class="crsbkr" id="crsbkr_webcheck"></div>';

		return wp_kses_post($form);
	}

	var $COUNTRIES = [["code"=>"IT","name"=>"ITALIA"],["code"=>"AF","name"=>"AFGHANISTAN"],["code"=>"AL","name"=>"ALBANIA"],["code"=>"DZ","name"=>"ALGERIA"],["code"=>"AD","name"=>"ANDORRA"],["code"=>"AO","name"=>"ANGOLA"],["code"=>"AI","name"=>"ANGUILLA"],["code"=>"AG","name"=>"ANTIGUA E BARBUDA"],["code"=>"AN","name"=>"ANTILLE OLANDESI"],["code"=>"SA","name"=>"ARABIA SAUDITA"],["code"=>"AR","name"=>"ARGENTINA"],["code"=>"AM","name"=>"ARMENIA"],["code"=>"AW","name"=>"ARUBA"],["code"=>"AU","name"=>"AUSTRALIA"],["code"=>"AT","name"=>"AUSTRIA"],["code"=>"AZ","name"=>"AZERBAIGIAN"],["code"=>"BS","name"=>"BAHAMAS"],["code"=>"BH","name"=>"BAHRAIN"],["code"=>"BD","name"=>"BANGLADESH"],["code"=>"BB","name"=>"BARBADOS"],["code"=>"BE","name"=>"BELGIO"],["code"=>"BZ","name"=>"BELIZE"],["code"=>"BJ","name"=>"BENIN"],["code"=>"BM","name"=>"BERMUDA"],["code"=>"BT","name"=>"BHUTAN"],["code"=>"BY","name"=>"BIELORUSSIA"],["code"=>"MM","name"=>"BIRMANIA"],["code"=>"BO","name"=>"BOLIVIA"],["code"=>"BQ","name"=>"BONAIRE"],["code"=>"BA","name"=>"BOSNIA ED ERZEGOVINA"],["code"=>"BW","name"=>"BOTSWANA"],["code"=>"BR","name"=>"BRASILE"],["code"=>"BN","name"=>"BRUNEI"],["code"=>"BG","name"=>"BULGARIA"],["code"=>"BF","name"=>"BURKINA FASO"],["code"=>"BI","name"=>"BURUNDI"],["code"=>"KH","name"=>"CAMBOGIA"],["code"=>"CM","name"=>"CAMERUN"],["code"=>"CA","name"=>"CANADA"],["code"=>"CV","name"=>"CAPO VERDE"],["code"=>"TD","name"=>"CIAD"],["code"=>"CL","name"=>"CILE"],["code"=>"CN","name"=>"CINA"],["code"=>"CY","name"=>"CIPRO"],["code"=>"VA","name"=>"CITTÃ€ DEL VATICANO"],["code"=>"CO","name"=>"COLOMBIA"],["code"=>"KM","name"=>"COMORE"],["code"=>"KP","name"=>"COREA DEL NORD"],["code"=>"KR","name"=>"COREA DEL SUD"],["code"=>"CI","name"=>"COSTA D'AVORIO"],["code"=>"CR","name"=>"COSTA RICA"],["code"=>"HR","name"=>"CROAZIA"],["code"=>"CU","name"=>"CUBA"],["code"=>"CW","name"=>"CURAÃ‡AO"],["code"=>"DK","name"=>"DANIMARCA"],["code"=>"DM","name"=>"DOMINICA"],["code"=>"EC","name"=>"ECUADOR"],["code"=>"EG","name"=>"EGITTO"],["code"=>"SV","name"=>"EL SALVADOR"],["code"=>"AE","name"=>"EMIRATI ARABI UNITI"],["code"=>"ER","name"=>"ERITREA"],["code"=>"EE","name"=>"ESTONIA"],["code"=>"ET","name"=>"ETIOPIA"],["code"=>"FJ","name"=>"FIGI"],["code"=>"PH","name"=>"FILIPPINE"],["code"=>"FI","name"=>"FINLANDIA"],["code"=>"FR","name"=>"FRANCIA"],["code"=>"GA","name"=>"GABON"],["code"=>"GM","name"=>"GAMBIA"],["code"=>"GE","name"=>"GEORGIA"],["code"=>"GS","name"=>"GEORGIA DEL SUD E ISOLE SANDWICH MERIDIONALI"],["code"=>"DE","name"=>"GERMANIA"],["code"=>"GH","name"=>"GHANA"],["code"=>"JM","name"=>"GIAMAICA"],["code"=>"JP","name"=>"GIAPPONE"],["code"=>"DJ","name"=>"GIBUTI"],["code"=>"JO","name"=>"GIORDANIA"],["code"=>"GR","name"=>"GRECIA"],["code"=>"GD","name"=>"GRENADA"],["code"=>"GL","name"=>"GROENLANDIA"],["code"=>"GP","name"=>"GUADALUPA"],["code"=>"GU","name"=>"GUAM"],["code"=>"GT","name"=>"GUATEMALA"],["code"=>"GN","name"=>"GUINEA"],["code"=>"GQ","name"=>"GUINEA EQUATORIALE"],["code"=>"GW","name"=>"GUINEA-BISSAU"],["code"=>"GY","name"=>"GUYANA"],["code"=>"GF","name"=>"GUYANA FRANCESE"],["code"=>"HT","name"=>"HAITI"],["code"=>"HN","name"=>"HONDURAS"],["code"=>"HK","name"=>"HONG KONG"],["code"=>"IN","name"=>"INDIA"],["code"=>"ID","name"=>"INDONESIA"],["code"=>"IR","name"=>"IRAN"],["code"=>"IQ","name"=>"IRAQ"],["code"=>"IE","name"=>"IRLANDA"],["code"=>"IS","name"=>"ISLANDA"],["code"=>"CX","name"=>"ISOLA DEL NATALE"],["code"=>"NF","name"=>"ISOLA NORFOLK"],["code"=>"KY","name"=>"ISOLE CAYMAN"],["code"=>"CC","name"=>"ISOLE COCOS E KEELING"],["code"=>"CK","name"=>"ISOLE COOK"],["code"=>"FK","name"=>"ISOLE FALKLAND"],["code"=>"MP","name"=>"ISOLE MARIANNE SETTENTRIONALI"],["code"=>"MH","name"=>"ISOLE MARSHALL"],["code"=>"PN","name"=>"ISOLE PITCAIRN"],["code"=>"SB","name"=>"ISOLE SALOMONE"],["code"=>"TF","name"=>"ISOLE SPARSE NELL'OCEANO INDIANO"],["code"=>"VI","name"=>"ISOLE VERGINI AMERICANE"],["code"=>"VG","name"=>"ISOLE VERGINI BRITANNICHE"],["code"=>"IL","name"=>"ISRAELE"],["code"=>"KZ","name"=>"KAZAKISTAN"],["code"=>"KE","name"=>"KENYA"],["code"=>"KG","name"=>"KIRGHIZISTAN"],["code"=>"KI","name"=>"KIRIBATI"],["code"=>"XK","name"=>"KOSOVO"],["code"=>"KW","name"=>"KUWAIT"],["code"=>"LA","name"=>"LAOS"],["code"=>"LS","name"=>"LESOTHO"],["code"=>"LV","name"=>"LETTONIA"],["code"=>"LB","name"=>"LIBANO"],["code"=>"LR","name"=>"LIBERIA"],["code"=>"LY","name"=>"LIBIA"],["code"=>"LI","name"=>"LIECHTENSTEIN"],["code"=>"LT","name"=>"LITUANIA"],["code"=>"LU","name"=>"LUSSEMBURGO"],["code"=>"MO","name"=>"MACAO"],["code"=>"MK","name"=>"MACEDONIA"],["code"=>"MG","name"=>"MADAGASCAR"],["code"=>"MW","name"=>"MALAWI"],["code"=>"MV","name"=>"MALDIVE"],["code"=>"MY","name"=>"MALESIA"],["code"=>"ML","name"=>"MALI"],["code"=>"MT","name"=>"MALTA"],["code"=>"MA","name"=>"MAROCCO"],["code"=>"MQ","name"=>"MARTINICA"],["code"=>"MR","name"=>"MAURITANIA"],["code"=>"MU","name"=>"MAURITIUS"],["code"=>"YT","name"=>"MAYOTTE"],["code"=>"MX","name"=>"MESSICO"],["code"=>"MD","name"=>"MOLDAVIA"],["code"=>"MN","name"=>"MONGOLIA"],["code"=>"ME","name"=>"MONTENEGRO"],["code"=>"MS","name"=>"MONTSERRAT"],["code"=>"MZ","name"=>"MOZAMBICO"],["code"=>"NA","name"=>"NAMIBIA"],["code"=>"NR","name"=>"NAURU"],["code"=>"NP","name"=>"NEPAL"],["code"=>"NI","name"=>"NICARAGUA"],["code"=>"NE","name"=>"NIGER"],["code"=>"NG","name"=>"NIGERIA"],["code"=>"NU","name"=>"NIUE"],["code"=>"NO","name"=>"NORVEGIA"],["code"=>"NC","name"=>"NUOVA CALEDONIA"],["code"=>"NZ","name"=>"NUOVA ZELANDA"],["code"=>"OM","name"=>"OMAN"],["code"=>"NL","name"=>"PAESI BASSI"],["code"=>"PK","name"=>"PAKISTAN"],["code"=>"PW","name"=>"PALAU"],["code"=>"PS","name"=>"PALESTINA"],["code"=>"PA","name"=>"PANAMA"],["code"=>"PG","name"=>"PAPUA NUOVA GUINEA"],["code"=>"PY","name"=>"PARAGUAY"],["code"=>"PE","name"=>"PERÃ™"],["code"=>"PF","name"=>"POLINESIA FRANCESE"],["code"=>"PL","name"=>"POLONIA"],["code"=>"PR","name"=>"PORTO RICO"],["code"=>"PT","name"=>"PORTOGALLO"],["code"=>"MC","name"=>"PRINCIPATO DI MONACO"],["code"=>"QA","name"=>"QATAR"],["code"=>"GB","name"=>"REGNO UNITO"],["code"=>"CZ","name"=>"REPUBBLICA CECA"],["code"=>"CF","name"=>"REPUBBLICA CENTRAFRICANA"],["code"=>"CG","name"=>"REPUBBLICA DEL CONGO"],["code"=>"CD","name"=>"REPUBBLICA DEMOCRATICA DEL CONGO"],["code"=>"DO","name"=>"REPUBBLICA DOMINICANA"],["code"=>"RE","name"=>"RIUNIONE"],["code"=>"RO","name"=>"ROMANIA"],["code"=>"RW","name"=>"RUANDA"],["code"=>"RU","name"=>"RUSSIA"],["code"=>"BQ","name"=>"SABA"],["code"=>"KN","name"=>"SAINT KITTS E NEVIS"],["code"=>"VC","name"=>"SAINT VINCENT E GRENADINE"],["code"=>"BL","name"=>"SAINT-BARTHÃ‰LEMY"],["code"=>"MF","name"=>"SAINT-MARTIN"],["code"=>"PM","name"=>"SAINT-PIERRE E MIQUELON"],["code"=>"WS","name"=>"SAMOA"],["code"=>"AS","name"=>"SAMOA AMERICANE"],["code"=>"SM","name"=>"SAN MARINO"],["code"=>"SH","name"=>"SANT'ELENA, ASCENSIONE E TRISTAN DA CUNHA"],["code"=>"LC","name"=>"SANTA LUCIA"],["code"=>"ST","name"=>"SÃO TOMÉ E PRÍNCIPE"],["code"=>"SN","name"=>"SENEGAL"],["code"=>"RS","name"=>"SERBIA"],["code"=>"SC","name"=>"SEYCHELLES"],["code"=>"SL","name"=>"SIERRA LEONE"],["code"=>"SG","name"=>"SINGAPORE"],["code"=>"BQ","name"=>"SINT EUSTATIUS"],["code"=>"SX","name"=>"SINT MAARTEN"],["code"=>"SY","name"=>"SIRIA"],["code"=>"SK","name"=>"SLOVACCHIA"],["code"=>"SI","name"=>"SLOVENIA"],["code"=>"SO","name"=>"SOMALIA"],["code"=>"ES","name"=>"SPAGNA"],["code"=>"LK","name"=>"SRI LANKA"],["code"=>"FM","name"=>"STATI FEDERATI DI MICRONESIA"],["code"=>"US","name"=>"STATI UNITI D'AMERICA"],["code"=>"ZA","name"=>"SUDAFRICA"],["code"=>"SD","name"=>"SUDAN"],["code"=>"SS","name"=>"SUDAN DEL SUD"],["code"=>"SR","name"=>"SURINAME"],["code"=>"SE","name"=>"SVEZIA"],["code"=>"CH","name"=>"SVIZZERA"],["code"=>"SZ","name"=>"SWAZILAND"],["code"=>"TJ","name"=>"TAGIKISTAN"],["code"=>"TW","name"=>"TAIWAN"],["code"=>"TZ","name"=>"TANZANIA"],["code"=>"TH","name"=>"THAILANDIA"],["code"=>"TL","name"=>"TIMOR EST"],["code"=>"TG","name"=>"TOGO"],["code"=>"TK","name"=>"TOKELAU"],["code"=>"TO","name"=>"TONGA"],["code"=>"TT","name"=>"TRINIDAD E TOBAGO"],["code"=>"TN","name"=>"TUNISIA"],["code"=>"TR","name"=>"TURCHIA"],["code"=>"TM","name"=>"TURKMENISTAN"],["code"=>"TC","name"=>"TURKS E CAICOS"],["code"=>"TV","name"=>"TUVALU"],["code"=>"UA","name"=>"UCRAINA"],["code"=>"UG","name"=>"UGANDA"],["code"=>"HU","name"=>"UNGHERIA"],["code"=>"UY","name"=>"URUGUAY"],["code"=>"UZ","name"=>"UZBEKISTAN"],["code"=>"VU","name"=>"VANUATU"],["code"=>"VE","name"=>"VENEZUELA"],["code"=>"VN","name"=>"VIETNAM"],["code"=>"WF","name"=>"WALLIS E FUTUNA"],["code"=>"YE","name"=>"YEMEN"],["code"=>"ZM","name"=>"ZAMBIA"],["code"=>"ZW","name"=>"ZIMBABWE"]];

	var $LANG = NULL;

	var $PARAMS = [];

	var $STRINGS = array
	(
		'it' => array
		(
			'lang' => 'it',
			'version' => CRSBKR_VERSION,

			'error' => 'Ci dispiace, sembra che il sito stia incontrando difficoltà tecniche. Prova a ricaricare la pagina o collegati di nuovo più tardi in caso che il problema persista.',
			'error.error' => 'Errore',

			'alert.file.camera' => 'Scatta foto',
			'alert.file.camera.confirm' => 'Carica',
			'alert.file.camera.none' => 'Non è stata rilavata nessuna fotocamera.',
			'alert.file.delete' => 'Cancella file',
			'alert.file.deleteConfirm' => 'Clicca sul tasto "Invia" per cancellare il file caricato prima.',
			'alert.file.drag' => 'Clicca qui o trascina un file in questa zona per caricarlo',
			'alert.file.drop' => 'Rilascia il file qui per caricarlo',
			'alert.file.dropping' => 'Dovresti aspettare la fine del caricamento.',
			'alert.file.existing' => 'File caricato',
			'alert.file.intro' => 'Seleziona una delle opzioni:',
			'alert.file.upload' => 'Carica file',
			'alert.file.url' => 'Usa da web',
			'alert.file.url.intro' => 'Se vuoi usare un\'immagine dal WEB, copia il suo indirizzo (URL) dalla barra e incollalo nel campo di seguito.',

			'label.cancel' => 'Annulla',
			'label.close' => 'Chiudi',
			'label.fileLink' => 'Link',
			'label.fileMime' => 'Tipo',
			'label.fileName' => 'Nome file',
			'label.fileSize' => 'Dimensioni',
			'label.loading' => 'Caricamento…',
			'label.modify' => 'Modifica',
			'label.ok' => 'OK',
			'label.retry' => 'Riprova',
			'label.search' => 'Cerca',
			'label.select' => 'Seleziona…',
			'label.submit' => 'Invia',
			'label.uploading' => 'Caricamento',

			'crsbkr_settings' => 'Impostazioni CRS Controller',
			'crsbkr_footer' => '',

			'crsbkr_api_key' => 'API Key',
			'crsbkr_api_key.desc' => 'Inserisci l\'API Key del controller esattamente come viene visualizzato su CRS.',
			'crsbkr_auth' => 'Authenticazione',
			'crsbkr_uuid' => 'UUID',
			'crsbkr_endpoint' => 'Endpoint',
			'crsbkr_endpoint.desc' => 'Inserisci l\'endpoint dell\'interfaccia come comunicato da My Appy.<br /><b>Default:</b> https://crsbrk00.myappy.it/api/v1',
			'crsbkr_booking_endpoint' => 'Booking Endpoint',
			'crsbkr_booking_endpoint.desc' => 'Inserisci l\'url del booking endpoint dell\'interfaccia come comunicato da My Appy.<br /><b>Default:</b> https://carrentalsoftware.myappy.it/api/v1',
			'crsbkr_uuid.desc' => 'Inserisci l\'identificatore del controller come visualizzato su CRS.',

			'crsbkr_search' => 'Impostazioni modulo di ricerca',
			'crsbkr_search_results_page' => 'Pagina con i risultati',
			'crsbkr_search_results_page.desc' => 'Sulla pagina basta inserire il shortcode <b>[crsbkr_search_results]</b> per mostrare i risultati dai vari moduli di ricerca.',

			'crsbkr_styles' => 'Impostazioni design',
			'crsbkr_hue' => 'Valore H del color principale in HSL',
			'crsbkr_sat' => 'Valore S del color principale in HSL',
			'crsbkr_lig' => 'Valore L del color principale in HSL',

			'id_type.id' => 'Carta di identità',
			'id_type.id_diplomatic' => 'Carta di identità diplomatica',
			'id_type.id_electronic' => 'Carta di identità elettronica',
			'id_type.license' => 'Patente di guida',
			'id_type.passport' => 'Passaporto',
			'id_type.passport_diplomatic' => 'Passaporto diplomatico',
			'id_type.passport_service' => 'Passaporto di servizio',

			'month.0' => 'Gennaio',
			'month.1' => 'Febbraio',
			'month.2' => 'Marzo',
			'month.3' => 'Aprile',
			'month.4' => 'Maggio',
			'month.5' => 'Giugno',
			'month.6' => 'Luglio',
			'month.7' => 'Agosto',
			'month.8' => 'Settembre',
			'month.9' => 'Ottobre',
			'month.10' => 'Novembre',
			'month.11' => 'Dicembre',

			'booking.invoice' => 'Dati fatturazione',
			'booking.invoice.desc' => 'Se l\'anagrafica da usare per la fatturazione è diversa da quella del conducente, compila i campi di sotto.',
			'booking.invoice.address' => 'Indirizzo domicilio / residenza',
			'booking.invoice.business' => 'Nome azienda',
			'booking.invoice.city' => 'Comune',
			'booking.invoice.contact_person' => 'Referente',
			'booking.invoice.country_id' => 'Nazione domicilio / residenza',
			'booking.invoice.date_of_birth' => 'Data di nascita (GG/MM/AAAA)',
			'booking.invoice.client_driver' => 'Conducenti',
			'booking.invoice.email' => 'Email',
			'booking.invoice.first_name' => 'Nome',
			'booking.invoice.name' => 'Cognome',
			'booking.invoice.notes' => 'Note',
			'booking.invoice.pec' => 'PEC',
			'booking.invoice.phone' => 'Telefono',
			'booking.invoice.place_of_birth' => 'Luogo di nascita',
			'booking.invoice.province' => 'Sigla Prov (tutto maiuscolo)',
			'booking.invoice.sdi_code' => 'Codice destinatario SdI',
			'booking.invoice.tax_number' => 'Codice fiscale',
			'booking.invoice.vat_number' => 'Partita IVA',
			'booking.invoice.zip_code' => 'CAP',

			'booking.driver' => 'Compila il modulo e prenota',
			'booking.driver.desc' => 'Compila il form di sotto con i dati del conducente principale per inviare la tua prenotazione.',
			'booking.driver.address' => 'Indirizzo domicilio/residenza',
			'booking.driver.city' => 'Comune',
			'booking.driver.country_id' => 'Nazione domicilio/residenza',
			'booking.driver.date_of_birth' => 'Data di nascita (GG/MM/AAAA)',
			'booking.driver.place_of_birth' => 'Luogo di nascita',
			'booking.driver.email' => 'E-mail',
			'booking.driver.email_confirm' => 'Ripeti E-Mail',
			'booking.driver.first_name' => 'Nome',
			'booking.driver.name' => 'Cognome',
			'booking.driver.nationality' => 'Cittadinanza',
			'booking.driver.notes' => 'Note prenotazione',
			'booking.driver.phone' => 'Telefono',
			'booking.driver.province' => 'Sigla Prov (tutto maiuscolo)',
			'booking.driver.tax_number' => 'Cod Fiscale',
			'booking.driver.zip_code' => 'CAP',

			'quote.accessories' => 'Extra aggiuntivi',
			'quote.accessories.desc' => 'Seleziona gli extra aggiuntivi che vuoi prenotare.',
			'quote.cancel' => 'Annulla prenotazione',
			'quote.canceled' => 'La tua prenotazione {reservation} è stata annullata.',
			'quote.checkin_base' => 'Località di riconsegna:',
			'quote.checkin_date' => 'Data di riconsegna:',
			'quote.checkout_base' => 'Località di ritiro:',
			'quote.checkout_date' => 'Data di ritiro:',
			'quote.confirmation' => 'Complimenti, la tua prenotazione {reservation} è stata confermata. Grazie per aver scelto i nostri servizi!',
			'quote.error' => 'Si è verificato un errore durante l\'elaborazione della tua richiesta. Riprova più tardi per favore o mettiti in contatto con il gestore del sito per risolvere il problema, grazie!',
			'quote.error.emailNotValid' => 'L\'indirizzo email {email} non sembra valido.',
			'quote.error.emailsNotMatching' => 'L\'indirizzo email inserito nel campo "Ripeti E-Mail" non corrisponde all\'indirizzo email inserito.',
			'quote.error.privacy' => 'Bisogna autorizzare al trattamento dei propri dati personali.',
			'quote.error.required' => 'Per favore, compila tutti i campi obbligatori per effettuare la prenotazione.',
			'quote.error.terms' => 'Bisogna accettare i termini e condizioni di noleggio.',
			'quote.group' => 'La tua selezione:',
			'quote.insurances' => 'Copertura',
			'quote.insurances.desc' => 'Scegli la copertura che vuoi prenotare.',
			'quote.insurances.excess' => 'Deposito cauzionale',
			'quote.insurances.excess.damage' => 'Franchigia danni',
			'quote.insurances.excess.info' => 'Franchigia danni/furto',
			'quote.insurances.excess.info.link' => 'Leggi le condizioni generali di noleggio per ulteriori info',
			'quote.insurances.excess.theft' => 'Franchigia furto',
			'quote.insurances.included' => 'Incluso',
			'quote.insurances.rate' => 'Prezzo',
			'quote.insurances.selected' => 'Selezionato',
			'quote.insurances.select' => 'Seleziona',
			'quote.km_limit' => 'KM Inclusi:',
			'quote.km_limit.day' => 'gg',
			'quote.km_limit.unlimited' => 'Illimitati',
			'quote.modify' => '[cambia]',
			'quote.pay' => 'Paga adesso {totals}',
			'quote.privacy' => 'Autorizzo al trattamento dei propri dati personali secondo il Regolamento UE n. 679 del 2016 (GDPR) e D.Lgs. n. 101 del 2018 e successive modifiche.',
			'quote.privacy.link' => 'trattamento dei propri dati personali',
			'quote.rate' => 'Riepilogo',
			'quote.submit' => 'Prenota e paga adesso {totals}',
			'quote.terms' => 'Desidero noleggiare il veicolo indicato secondo i Termini e le Condizioni di noleggio e alle tariffe in vigore. Ho letto ed accettato i Termini e le Condizioni di noleggio, tariffarie e assicurative.',
			'quote.terms.link' => 'Termini e le Condizioni di noleggio',
			'quote.totals' => 'Totale',
			'quote.totals.vat' => 'IVA',
			'quote.totals.vatExcl' => 'Imponibile',

			'results.change' => 'Modifica la ricerca',
			'results.checkout' => 'Ritiro',
			'results.checkin' => 'Riconsegna',
			'results.day' => 'Giorno',
			'results.days' => 'Giorni',
			'results.details' => '⤼ Dettagli',
			'results.details.back' => '⤺ Indietro',
			'results.extra' => '* Suppl. {name}: {price}',
			'results.group.rate' => 'Da {rate}',
			'results.km_limit' => 'KM Inclusi',
			'results.km_limit.day' => 'gg',
			'results.km_limit.unlimited' => 'Illimitati',
			'results.loading' => 'Stiamo verificando la disponibilità…',
			'results.none.title' => 'Nessuna disponibilità trovata',
			'results.none.text' => 'Ci dispiace, ma non è stato trovato nessun risultato con i criteri di ricerca inseriti.',
			'results.options.day' => 'Tariffa valida per <b>1 giorno</b>',
			'results.options.days' => 'Tariffa valida per <b>{days} giorni</b>',
			'results.options.hint' => 'Clicca sulla tariffa che fa per te per andare avanti.',
			'results.rate' => 'Prezzo',
			'results.rate.included' => 'Incluso',
			'results.remove' => 'Elimina',
			'results.required' => 'Obbligatorio',
			'results.select' => 'Scegli',

			'results.params.chm_characteristics' => 'Characteristics',
			'results.params.chm_characteristics.engine' => 'Motore',
			'results.params.chm_characteristics.length_overall' => 'L.O.A.',
			'results.params.chm_characteristics.length_overall.unit' => 'm',
			'results.params.chm_characteristics.tank_size' => 'Serbatoio Diesel',
			'results.params.chm_characteristics.tank_size.unit' => 'L',
			'results.params.chm_characteristics.type' => 'Tipo',
			'results.params.chm_characteristics.water_tank_size' => 'Serbatoio Acqua',
			'results.params.chm_characteristics.water_tank_size.unit' => 'L',
			'results.params.chm_characteristics.year' => 'Anno',

			'results.params.chm_equipment' => 'Equipment',
			'results.params.chm_equipment.bow_thruster' => 'Bow thruster',
			'results.params.chm_equipment.bow_thruster.1' => 'Sì',
			'results.params.chm_equipment.bow_thruster.0' => 'No',
			'results.params.chm_equipment.gps' => 'GPS',
			'results.params.chm_equipment.gps.bg_vulcan' => 'GPS B&G Vulcan',
			'results.params.chm_equipment.gps.gps_plotter' => 'GPS Plotter',
			'results.params.chm_equipment.gps.navionics' => 'iPad (Navionics App)',
			'results.params.chm_equipment.mainsail' => 'Mainsail',
			'results.params.chm_equipment.mainsail.furling' => 'Furling',
			'results.params.chm_equipment.mainsail.lazy_bag' => 'Lazy bag',
			'results.params.chm_equipment.teak' => 'Teak',
			'results.params.chm_equipment.teak.cockpit' => 'Cockpit',
			'results.params.chm_equipment.teak.full' => 'Full',
			'results.params.chm_equipment.teak.none' => 'None',

			'results.params.vehicle_filters.berths_no' => 'Cuccette',
			'results.params.vehicle_filters.big_bags' => 'Grandi',
			'results.params.vehicle_filters.cabins_no' => 'Cabine',
			'results.params.vehicle_filters.doors' => 'Porte',
			'results.params.vehicle_filters.fuel_type' => 'Alimentazione',
			'results.params.vehicle_filters.fuel_type.diesel' => 'Diesel',
			'results.params.vehicle_filters.tuel_type.electric' => 'Elettrica',
			'results.params.vehicle_filters.fuel_type.gpl' => 'GPL',
			'results.params.vehicle_filters.fuel_type.hybrid' => 'Ibrida',
			'results.params.vehicle_filters.fuel_type.methane' => 'Metano',
			'results.params.vehicle_filters.fuel_type.petrol' => 'Benzina',
			'results.params.vehicle_filters.gearshift.automatic' => 'Autom.',
			'results.params.vehicle_filters.gearshift.manual' => 'Manuale',
			'results.params.vehicle_filters.km_limit.none' => 'Km illimitati',
			'results.params.vehicle_filters.seats' => 'Posti',
			'results.params.vehicle_filters.seats_no' => 'Pers. max.',
			'results.params.vehicle_filters.small_bags' => 'Piccole',
			'results.params.vehicle_filters.toilets_no' => 'WC',

			'results.params.description' => '',
			'results.params.description.text' => '',

			'results.params.vehicle_details' => 'Caratteristiche',
			'results.params.vehicle_details.air' => 'Climatizzatore',
			'results.params.vehicle_details.air.0' => 'No',
			'results.params.vehicle_details.air.1' => 'Sì',
			'results.params.vehicle_details.air.false' => 'No',
			'results.params.vehicle_details.air.true' => 'Sì',
			'results.params.vehicle_details.doors' => 'Porte',
			'results.params.vehicle_details.lifting_platform' => 'Pedana sollevatrice',
			'results.params.vehicle_details.lifting_platform.0' => 'No',
			'results.params.vehicle_details.lifting_platform.1' => 'Sì',
			'results.params.vehicle_details.lifting_platform.false' => 'No',
			'results.params.vehicle_details.lifting_platform.true' => 'Sì',
			'results.params.vehicle_details.emission_class' => 'Euro',
			'results.params.vehicle_details.engine_size' => 'Cilindrata',
			'results.params.vehicle_details.fuel_type' => 'Alimentazione',
			'results.params.vehicle_details.fuel_type.diesel' => 'Diesel',
			'results.params.vehicle_details.fuel_type.electric' => 'Elettrica',
			'results.params.vehicle_details.fuel_type.gpl' => 'GPL',
			'results.params.vehicle_details.fuel_type.hybrid' => 'Ibrida',
			'results.params.vehicle_details.fuel_type.methane' => 'Metano',
			'results.params.vehicle_details.fuel_type.petrol' => 'Benzina',
			'results.params.vehicle_details.pax' => 'Passeggeri',
			'results.params.vehicle_details.radio' => 'Autoradio',
			'results.params.vehicle_details.radio.0' => 'No',
			'results.params.vehicle_details.radio.1' => 'Sì',
			'results.params.vehicle_details.radio.false' => 'No',
			'results.params.vehicle_details.radio.true' => 'Sì',
			'results.params.vehicle_details.tank_size' => 'Capacità serbatoio',

			'results.params.interior_dimensions' => 'Dimensioni interni',
			'results.params.interior_dimensions.weight_limit' => 'Portata',
			'results.params.interior_dimensions.length' => 'Lunghezza',
			'results.params.interior_dimensions.width' => 'Larghezza',
			'results.params.interior_dimensions.wheel_arch_width' => 'Larghezza passaruote',
			'results.params.interior_dimensions.volume' => 'Volume',
			'results.params.interior_dimensions.height' => 'Altezza',

			'results.params.exterior_dimensions' => 'Dimensioni esterne',
			'results.params.exterior_dimensions.wheelbase' => 'Passo',
			'results.params.exterior_dimensions.width' => 'Larghezza',
			'results.params.exterior_dimensions.length' => 'Lunghezza',
			'results.params.exterior_dimensions.height' => 'Altezza',

			'search.checkout_base' => 'Località di ritiro',
			'search.checkout_base.placeholder' => 'Seleziona…',
			'search.checkout_date' => 'Data di ritiro',
			'search.checkout_date.select' => 'Seleziona data di ritiro',
			'search.checkout_time' => 'Orario ritiro',
			'search.checkin_base' => 'Località di riconsegna',
			'search.checkin_base.placeholder' => 'Seleziona…',
			'search.checkin_base.tick' => 'Effettua la riconsegna in una località diversa',
			'search.checkin_date' => 'Data di riconsegna',
			'search.checkin_date.select' => 'Seleziona data di riconsegna',
			'search.checkin_time' => 'Orario riconsegna',
			'search.error.baseClosed' => 'La località risulta chiusa per il giorno scelto',
			'search.error.baseClosedHours.checkin' => 'La riconsegne a quest\'ora non è disponibile.',
			'search.error.baseClosedHours.checkout' => 'Il ritiro a quest\'ora non è disponibile.',
			'search.error.baseOutOfHours.checkin' => 'Potrebbero essere applicati dei costi aggiuntivi per la riconsegna a quest\'ora.',
			'search.error.baseOutOfHours.checkout' => 'Potrebbero essere applicati dei costi aggiuntivi per il ritiro a quest\'ora.',
			'search.error.checkinBeforeCheckout' => 'La data e ora della riconsegna non può precedere e non può essere uguale a quella del ritiro.',
			'search.error.pastDate' => 'La data/ora scelta è nel passato.',
			'search.promo_code.select' => 'Ho un codice promo',
			'search.promo_code.placeholder' => 'Inserisci codice promo…',
			'search.search' => 'Cerca',
			'search.selectBase' => 'Seleziona località',
			'search.selectDate' => 'Seleziona la data',
			'search.selectTime' => 'Seleziona gli orari',
			'search.selectTime.hide' => 'Nascondi',
			'search.selectTime.hint' => 'Scorri a destra o sinistra',
			'search.selectTime.show' => 'Mostra',

			'webcheck.form.email' => 'Indirizzo email',
			'webcheck.form.error' => 'Ci dispiace, purtroppo non siamo riusciti a trovare la tua prenotazione.<br />Errore: {{error}}<br />Per favore controlla i dati di sotto e provaci di nuovo.',
			'webcheck.form.error.incomplete' => 'Compila tutti i campi del modulo prima di inviare la richiesta.',
			'webcheck.form.intro' => 'Compila i campi di sotto e clicca su Invia per recuperare la tua prenotazione. Una volta caricato potrai aggiungere i dati mancanti per velocizzare il ritiro del mezzo.',
			'webcheck.form.surname' => 'Cognome cliente/conducente principale',
			'webcheck.form.voucher' => 'N° voucher (p.es. "AB-12345678")',

			'webcheck.attachment.select' => 'Carica file…',

			'webcheck.client' => 'Dati fatturazione',
			'webcheck.client.desc' => 'Se desideri usare un indirizzo diverso da quello del conducente principale per eventuale fatture emesse dall\'autonoleggio compila i modulo seguente',
			'webcheck.client.show' => 'Aggiungi dati di fatturazione',
			'webcheck.client_driver.0' => 'Conducente principale',
			'webcheck.client_driver.0.desc' => 'Inserisci tutti i dati del conducente principale.',
			'webcheck.client_driver.0.id' => 'Documento di riconoscimento',
			'webcheck.client_driver.0.id.desc' => 'Fornisci tutti i dettagli del documento di riconoscimento (carta d\'identità, passaporto) del conducente principale e allega una foto per ciascuno dei lati.',
			'webcheck.client_driver.0.license' => 'Patente di guida',
			'webcheck.client_driver.0.license.desc' => 'Fornisci tutti i dettagli della patente di guida del conducente principale e allega una foto per ciascuno dei lati.',
			'webcheck.client_driver.1' => 'Conducente secondario',
			'webcheck.client_driver.1.desc' => 'Inserisci tutti i dati del conducente secondario. Si nota che aggiungere un altro conducente al noleggio potrebbe implicare dei costi aggiuntivi.',
			'webcheck.client_driver.1.id' => 'Documento di riconoscimento',
			'webcheck.client_driver.1.id.desc' => 'Fornisci tutti i dettagli del documento di riconoscimento (carta d\'identità, passaporto) del conducente secondario e allega una foto per ciascuno dei lati.',
			'webcheck.client_driver.1.license' => 'Patente di guida',
			'webcheck.client_driver.1.license.desc' => 'Fornisci tutti i dettagli della patente di guida del conducente secondario e allega una foto per ciascuno dei lati.',
			'webcheck.client_driver.1.show' => 'Aggiungi un altro conducente…',

			'webcheck.client.address' => 'Indirizzo',
			'webcheck.client.city' => 'Comune',
			'webcheck.client.country_id' => 'Nazione',
			'webcheck.client.date_of_birth' => 'Data di nascita (GG/MM/YYYY)',
			'webcheck.client.place_of_birth' => 'Luogo di nascita',
			'webcheck.client.email' => 'E-mail',
			'webcheck.client.first_name' => 'Nome',
			'webcheck.client.name' => 'Cognome',
			'webcheck.client.phone' => 'Telefono',
			'webcheck.client.province' => 'Sigla Prov (tutto maiuscolo)',
			'webcheck.client.tax_number' => 'Cod Fiscale',
			'webcheck.client.vat_number' => 'Partita IVA',
			'webcheck.client.zip_code' => 'CAP',

			'webcheck.client_driver.address' => 'Indirizzo domicilio/residenza',
			'webcheck.client_driver.city' => 'Comune',
			'webcheck.client_driver.country_id' => 'Nazione domicilio/residenza',
			'webcheck.client_driver.date_of_birth' => 'Data di nascita (GG/MM/YYYY)',
			'webcheck.client_driver.place_of_birth' => 'Luogo di nascita',
			'webcheck.client_driver.email' => 'E-mail',
			'webcheck.client_driver.file_id' => 'Allega copia documento (fronte)',
			'webcheck.client_driver.file_id_2' => 'Allega copia documento (retro)',
			'webcheck.client_driver.file_license' => 'Allega copia patente (fronte)',
			'webcheck.client_driver.file_license_2' => 'Allega copia patente (retro)',
			'webcheck.client_driver.first_name' => 'Nome',
			'webcheck.client_driver.id_expiry_date' => 'Data di scadenza (GG/MM/YYYY)',
			'webcheck.client_driver.id_issue_date' => 'Data di rilascio (GG/MM/YYYY)',
			'webcheck.client_driver.id_issuer' => 'Rilasciato da',
			'webcheck.client_driver.id_issuer_locality' => 'Luogo rilascio (Comune o stato estero)',
			'webcheck.client_driver.id_number' => 'Numero documento',
			'webcheck.client_driver.id_type' => 'Tipo documento',
			'webcheck.client_driver.license_expiry_date' => 'Data di scadenza patente (GG/MM/YYYY)',
			'webcheck.client_driver.license_issue_date' => 'Data di rilascio patente (GG/MM/YYYY)',
			'webcheck.client_driver.license_issuer' => 'Rilasciata da',
			'webcheck.client_driver.license_issuer_locality' => 'Luogo rilascio (Comune o stato estero)',
			'webcheck.client_driver.license_number' => 'Numero patente',
			'webcheck.client_driver.name' => 'Cognome',
			'webcheck.client_driver.nationality' => 'Cittadinanza',
			'webcheck.client_driver.phone' => 'Telefono',
			'webcheck.client_driver.province' => 'Sigla Prov (tutto maiuscolo)',
			'webcheck.client_driver.tax_number' => 'Cod Fiscale',
			'webcheck.client_driver.zip_code' => 'CAP',

			'webcheck.error' => 'Abbiamo rilevato degli errori nel modulo. Per favore assicurati di aver inserito tutti i dati nel formato corretto e riprova.',
			'webcheck.error.expired' => 'Documento scaduto',
			'webcheck.error.invalidFormat' => 'Formato invalido',
			'webcheck.intro' => 'Di sotto puoi visualizzare i dati richiesti per la tua prenotazione {{ref_id}}. Completa i campi vuoti compreso gli allegati e clicca sul tasto "Invia Dati" per completare la tua prenotazione.',
			'webcheck.loading' => '<b>Attendi.<br />Stiamo elaborando la tua richiesta.<br />Non chiudere questa pagina.</b><br /><br />Tempo di attesa stimato: 30"',
			'webcheck.outro' => 'Clicca su "Invia dati" per inviare i dati forniti al nostro sistema. Inviando i dati confermi di aver preso visione della nostra Privacy Policy.',
			'webcheck.outro.title' => 'Invio dati',
			'webcheck.submit' => 'Invia dati',
			'webcheck.submit.progress' => 'Stiamo trasferendo i dati forniti al nostro sistema…',
			'webcheck.submit.success' => 'La trasmissione è andata a buon fine. Grazie per averci scelto!',

			'weekday.0' => 'Lunedì',
			'weekday.1' => 'Martedì',
			'weekday.2' => 'Mercoledì',
			'weekday.3' => 'Giovedì',
			'weekday.4' => 'Venerdì',
			'weekday.5' => 'Sabato',
			'weekday.6' => 'Domenica',
		),

		'en' =>
		[
			'lang' => 'en',
			'version' => CRSBKR_VERSION,
			'error' => 'We\'re sorry, seems like the website is having some problems right now. Try to reload the page o try again later in case the problem persists.',
			'error.error' => 'Error',

			'alert.file.camera' => 'Take photo',
			'alert.file.camera.confirm' => 'Upload',
			'alert.file.camera.none' => 'No camera found.',
			'alert.file.delete' => 'Delete file',
			'alert.file.deleteConfirm' => 'Click on the "Submit" button to delete the file.',
			'alert.file.drag' => 'Click or drag a file here to upload it.',
			'alert.file.drop' => 'Drop the file here to upload it.',
			'alert.file.dropping' => 'You have to wait until the upload is complete.',
			'alert.file.existing' => 'Existing file',
			'alert.file.intro' => 'Select one of the following options:',
			'alert.file.upload' => 'Upload file',
			'alert.file.url' => 'Use from web',
			'alert.file.url.intro' => 'If you want to use a file from the WEB, paste its address/URL into the field below.',

			'label.cancel' => 'Cancel',
			'label.close' => 'Close',
			'label.fileLink' => 'Link',
			'label.fileMime' => 'Type',
			'label.fileName' => 'Filename',
			'label.fileSize' => 'Filesize',
			'label.loading' => 'Loading…',
			'label.modify' => 'Modify',
			'label.ok' => 'OK',
			'label.retry' => 'Retry',
			'label.search' => 'Search',
			'label.select' => 'Select…',
			'label.submit' => 'Submit',
			'label.uploading' => 'Uploading',

			'crsbkr_settings' => 'CRS Controller settings',
			'crsbkr_footer' => '',

			'crsbkr_api_key' => 'API Key',
			'crsbkr_api_key.desc' => 'Insert the controller API key exactly how it\'s displayed on CRS.',
			'crsbkr_auth' => 'Authentication',
			'crsbkr_endpoint' => 'Endpoint',
			'crsbkr_endpoint.desc' => 'Insert the gateway endpoint exactly how it was communicated to you by My Appy.<br /><b>Default:</b> https://crsbrk00.myappy.it/api/v1',
			'crsbkr_booking_endpoint' => 'Booking Endpoint',
			'crsbkr_booking_endpoint.desc' => 'Insert the booking gateway endpoint exactly how it was communicated to you by My Appy.<br /><b>Default:</b> https://carrentalsoftware.myappy.it/api/v1',
			'crsbkr_uuid' => 'UUID',
			'crsbkr_uuid.desc' => 'Insert the controller id as displayed on CRS.',

			'crsbkr_search' => 'Search module settings',
			'crsbkr_search_results_page' => 'Search results page',
			'crsbkr_search_results_page.desc' => 'On this page you\'ll only have to insert the shortcode <b>[crsbkr_search_results]</b> to show the results from the various search modules.',

			'crsbkr_styles' => 'Design settings',
			'crsbkr_hue' => 'H value of main color HSL',
			'crsbkr_sat' => 'S value of main color HSL',
			'crsbkr_lig' => 'L value of main color HSL',

			'id_type.id' => 'Identity card',
			'id_type.id_diplomatic' => 'Diplomatic identity card',
			'id_type.id_electronic' => 'Electronic identity card',
			'id_type.license' => 'Driver\'s license',
			'id_type.passport' => 'Passport',
			'id_type.passport_diplomatic' => 'Diplomatic passport',
			'id_type.passport_service' => 'Service passport',

			'month.0' => 'January',
			'month.1' => 'February',
			'month.2' => 'March',
			'month.3' => 'April',
			'month.4' => 'May',
			'month.5' => 'June',
			'month.6' => 'July',
			'month.7' => 'August',
			'month.8' => 'September',
			'month.9' => 'October',
			'month.10' => 'November',
			'month.11' => 'December',

			'booking.invoice' => 'Invoicing details',
			'booking.invoice.desc' => 'If you\'re invoicing details are different than the driver details, fill in the following fields.',
			'booking.invoice.address' => 'Address',
			'booking.invoice.business' => 'Business name',
			'booking.invoice.city' => 'City',
			'booking.invoice.contact_person' => 'Contact person',
			'booking.invoice.country_id' => 'Country',
			'booking.invoice.date_of_birth' => 'Date of birth (DD/MM/YYYY)',
			'booking.invoice.client_driver' => 'Drivers',
			'booking.invoice.email' => 'Email',
			'booking.invoice.first_name' => 'Given name',
			'booking.invoice.name' => 'Surname',
			'booking.invoice.notes' => 'Notes',
			'booking.invoice.pec' => 'PEC',
			'booking.invoice.phone' => 'Phone',
			'booking.invoice.place_of_birth' => 'Place of birth',
			'booking.invoice.province' => 'Province',
			'booking.invoice.sdi_code' => 'SdI Code',
			'booking.invoice.tax_number' => 'Tax number',
			'booking.invoice.vat_number' => 'VAT number',
			'booking.invoice.zip_code' => 'Postal code',

			'booking.driver' => 'Fill out the form and book',
			'booking.driver.desc' => 'Fill out the following form with the contact details of the main driver in order to submit your booking.',
			'booking.driver.address' => 'Address',
			'booking.driver.city' => 'City',
			'booking.driver.country_id' => 'Country',
			'booking.driver.date_of_birth' => 'Date of birth (DD/MM/YYYY)',
			'booking.driver.place_of_birth' => 'Place of birth',
			'booking.driver.email' => 'Email',
			'booking.driver.email_confirm' => 'Repeat Email',
			'booking.driver.first_name' => 'Given name',
			'booking.driver.name' => 'Surname',
			'booking.driver.nationality' => 'Nationality',
			'booking.driver.notes' => 'Notes',
			'booking.driver.phone' => 'Phone',
			'booking.driver.province' => 'Province',
			'booking.driver.tax_number' => 'Tax number',
			'booking.driver.zip_code' => 'Postal code',

			'quote.accessories' => 'Additional extras',
			'quote.accessories.desc' => 'Select the extras you would like to add to your booking.',
			'quote.cancel' => 'Cancel booking',
			'quote.canceled' => 'Your booking n. {reservation} was canceled.',
			'quote.checkin_base' => 'Drop-off location:',
			'quote.checkin_date' => 'Drop-off date:',
			'quote.checkout_base' => 'Pick-up location:',
			'quote.checkout_date' => 'Pick-up date:',
			'quote.confirmation' => 'Congratulations, your booking n. {reservation} was confirmed. Thanks for using our service!',
			'quote.error' => 'We\'ve encountered an error during the elaboration of your request. Please try again later or get in contact with the website owner in order to have it fixed. Thank you!',
			'quote.error.emailNotValid' => 'The email address {email} is not valid.',
			'quote.error.emailsNotMatching' => 'The email address from the field "Repeat Email" is not the same as the entered email.',
			'quote.error.privacy' => 'You have to accept the privacy policy.',
			'quote.error.required' => 'Fill out all required fields to submit your booking.',
			'quote.error.terms' => 'You have to accept the terms and conditions.',
			'quote.group' => 'Your selection:',
			'quote.insurances' => 'Coverages',
			'quote.insurances.desc' => 'Select the coverage you\'d like to book.',
			'quote.insurances.excess' => 'Security deposit',
			'quote.insurances.excess.damage' => 'Excess damages',
			'quote.insurances.excess.info' => 'Excess damages/theft',
			'quote.insurances.excess.info.link' => 'Read the rental conditions for more information',
			'quote.insurances.excess.theft' => 'Excess theft',
			'quote.insurances.included' => 'Included',
			'quote.insurances.rate' => 'Price',
			'quote.insurances.selected' => 'Selected',
			'quote.insurances.select' => 'Select',
			'quote.km_limit' => 'Included KM:',
			'quote.km_limit.day' => 'day',
			'quote.km_limit.unlimited' => 'Unlimited',
			'quote.modify' => '[change]',
			'quote.pay' => 'Pay now {totals}',
			'quote.privacy' => 'I hereby authorise the processing of my personal data according to the privacy policy in compliance with the GDPR (UE Regulation 2016/679) and the D.LGS. 2018/101 and later modifications.',
			'quote.privacy.link' => 'privacy policy',
			'quote.rate' => 'Summary',
			'quote.submit' => 'Book and pay now {totals}',
			'quote.terms' => 'I wish to rent the indicated vehicle according to the terms and conditions and the selected tariff. I have read and understood the terms and the tariff and coverage options.',
			'quote.terms.link' => 'terms and conditions',
			'quote.totals' => 'Totals',
			'quote.totals.vat' => 'VAT',
			'quote.totals.vatExcl' => 'Taxable',

			'results.change' => 'Modify parameters',
			'results.checkout' => 'Pickup',
			'results.checkin' => 'Dropoff',
			'results.day' => 'Day',
			'results.days' => 'Days',
			'results.details' => '⤼ Details',
			'results.details.back' => '⤺ Back',
			'results.extra' => '* {name} supplement: {price}',
			'results.group.rate' => 'From {rate}',
			'results.km_limit.day' => 'day',
			'results.loading' => 'Checking availability…',
			'results.none.title' => 'No results found',
			'results.none.text' => 'We\'re sorry but we have not found any results matching your search criteria.',
			'results.options.day' => 'Tariff valid for <b>1 day</b>',
			'results.options.days' => 'Tariff valid for <b>{days} days</b>',
			'results.options.hint' => 'Select your tariff option to proceed.',
			'results.rate' => 'Price',
			'results.rate_included' => 'Included',
			'results.remove' => 'Eliminate',
			'results.required' => 'Required',
			'results.select' => 'Select',

			'results.params.chm_characteristics' => 'Characteristics',
			'results.params.chm_characteristics.engine' => 'Engine',
			'results.params.chm_characteristics.length_overall' => 'L.O.A.',
			'results.params.chm_characteristics.length_overall.unit' => 'm',
			'results.params.chm_characteristics.tank_size' => 'Tank size',
			'results.params.chm_characteristics.tank_size.unit' => 'L',
			'results.params.chm_characteristics.type' => 'Type',
			'results.params.chm_characteristics.water_tank_size' => 'Water tank',
			'results.params.chm_characteristics.water_tank_size.unit' => 'L',
			'results.params.chm_characteristics.year' => 'Year',

			'results.params.chm_equipment' => 'Equipment',
			'results.params.chm_equipment.bow_thruster' => 'Bow thruster',
			'results.params.chm_equipment.bow_thruster.1' => 'Yes',
			'results.params.chm_equipment.bow_thruster.0' => 'No',
			'results.params.chm_equipment.bow_thruster.true' => 'Yes',
			'results.params.chm_equipment.bow_thruster.false' => 'No',
			'results.params.chm_equipment.gps' => 'GPS',
			'results.params.chm_equipment.gps.bg_vulcan' => 'GPS B&G Vulcan',
			'results.params.chm_equipment.gps.gps_plotter' => 'GPS Plotter',
			'results.params.chm_equipment.gps.navionics' => 'iPad (Navionics App)',
			'results.params.chm_equipment.mainsail' => 'Mainsail',
			'results.params.chm_equipment.mainsail.furling' => 'Furling',
			'results.params.chm_equipment.mainsail.lazy_bag' => 'Lazy bag',
			'results.params.chm_equipment.teak' => 'Teak',
			'results.params.chm_equipment.teak.cockpit' => 'Cockpit',
			'results.params.chm_equipment.teak.full' => 'Full',
			'results.params.chm_equipment.teak.none' => 'None',

			'results.params.vehicle_filters.berths_no' => 'Berths',
			'results.params.vehicle_filters.big_bags' => 'Big',
			'results.params.vehicle_filters.cabins_no' => 'Cabins',
			'results.params.vehicle_filters.doors' => 'Doors',
			'results.params.vehicle_filters.fuel_type' => 'Fuel',
			'results.params.vehicle_filters.fuel_type.diesel' => 'Diesel',
			'results.params.vehicle_filters.fuel_type.electric' => 'Electric',
			'results.params.vehicle_filters.fuel_type.gpl' => 'GPL',
			'results.params.vehicle_filters.fuel_type.hybrid' => 'Hybrid',
			'results.params.vehicle_filters.fuel_type.methane' => 'Methan',
			'results.params.vehicle_filters.fuel_type.petrol' => 'Petrol',
			'results.params.vehicle_filters.gearshift.automatic' => 'Autom.',
			'results.params.vehicle_filters.gearshift.manual' => 'Manual',
			'results.params.vehicle_filters.km_limit.none' => 'Unlimited Km',
			'results.params.vehicle_filters.seats' => 'Seats',
			'results.params.vehicle_filters.seats_no' => 'Pers. max.',
			'results.params.vehicle_filters.small_bags' => 'Small',
			'results.params.vehicle_filters.toilets_no' => 'WC',

			'results.params.description' => '',
			'results.params.description.text' => '',

			'results.params.vehicle_details' => 'Characteristics',
			'results.params.vehicle_details.air' => 'Air conditioning',
			'results.params.vehicle_details.air.0' => 'No',
			'results.params.vehicle_details.air.1' => 'Yes',
			'results.params.vehicle_details.air.false' => 'No',
			'results.params.vehicle_details.air.true' => 'Yes',
			'results.params.vehicle_details.doors' => 'Doors',
			'results.params.vehicle_details.lifting_platform' => 'Lifting platform',
			'results.params.vehicle_details.lifting_platform.false' => 'No',
			'results.params.vehicle_details.lifting_platform.true' => 'Yes',
			'results.params.vehicle_details.lifting_platform.0' => 'No',
			'results.params.vehicle_details.lifting_platform.1' => 'Yes',
			'results.params.vehicle_details.emission_lass' => 'Euro',
			'results.params.vehicle_details.engine_size' => 'Engine size',
			'results.params.vehicle_details.fuel_type' => 'Fuel',
			'results.params.vehicle_details.fuel_type.diesel' => 'Diesel',
			'results.params.vehicle_details.fuel_type.electric' => 'Electric',
			'results.params.vehicle_details.fuel_type.gpl' => 'GPL',
			'results.params.vehicle_details.fuel_type.hybrid' => 'Hybrid',
			'results.params.vehicle_details.fuel_type.methane' => 'Methan',
			'results.params.vehicle_details.fuel_type.petrol' => 'Petrol',
			'results.params.vehicle_details.pax' => 'Passengers',
			'results.params.vehicle_details.radio' => 'Radio',
			'results.params.vehicle_details.radio.0' => 'No',
			'results.params.vehicle_details.radio.1' => 'Yes',
			'results.params.vehicle_details.radio.false' => 'No',
			'results.params.vehicle_details.radio.true' => 'Yes',
			'results.params.vehicle_details.tank_size' => 'Tank size',

			'results.params.interior_dimensions' => 'Internal dimensions',
			'results.params.interior_dimensions.weight_limit' => 'Weight limit',
			'results.params.interior_dimensions.length' => 'Length',
			'results.params.interior_dimensions.width' => 'Width',
			'results.params.interior_dimensions.wheel_arch_width' => 'Wheel arch width',
			'results.params.interior_dimensions.volume' => 'Volume',
			'results.params.interior_dimensions.height' => 'Height',

			'results.params.exterior_dimensions' => 'Exterior dimensions',
			'results.params.exterior_dimensions.wheelbase' => 'Wheelbase',
			'results.params.exterior_dimensions.width' => 'Width',
			'results.params.exterior_dimensions.length' => 'Length',
			'results.params.exterior_dimensions.height' => 'Height',

			'search.checkout_base' => 'Pickup location',
			'search.checkout_base.placeholder' => 'Select…',
			'search.checkout_date' => 'Pickup date',
			'search.checkout_date.select' => 'Select pickup date',
			'search.checkout_time' => 'Pickup time',
			'search.checkin_base' => 'Dropoff location',
			'search.checkin_base.placeholder' => 'Select…',
			'search.checkin_base.tick' => 'Leave the vehicle at another location',
			'search.checkin_date' => 'Dropoff date',
			'search.checkin_date.select' => 'Select dropoff date',
			'search.checkin_time' => 'Dropoff time',
			'search.error.baseClosed' => 'The location seems to be closed on the selected day',
			'search.error.baseClosedHours.checkin' => 'Dropoffs at this time are not available',
			'search.error.baseClosedHours.checkout' => 'Pickups at this time are not available',
			'search.error.baseOutOfHours.checkin' => 'Dropoffs at this time might come with additional fees',
			'search.error.baseOutOfHours.checkout' => 'Pickups at this time might come with additional fees',
			'search.error.checkinBeforeCheckout' => 'The dropoff date and time has to be after the pickup.',
			'search.error.pastDate' => 'The chosen date has already passed.',
			'search.promo_code.select' => 'I have a promo code',
			'search.promo_code.placeholder' => 'Insert promo code…',
			'search.search' => 'Search',
			'search.selectBase' => 'Select location',
			'search.selectDate' => 'Select dates',
			'search.selectTime' => 'Select times',
			'search.selectTime.hide' => 'Hide',
			'search.selectTime.hint' => 'Move the sliders to the left or right',
			'search.selectTime.show' => 'Show',

			'webcheck.form.email' => 'Email address',
			'webcheck.form.error' => 'Unfortunately we haven\'t found your reservation.<br />Error: {{error}}<br />Please verify the inserted details are correct and try again.',
			'webcheck.form.error.incomplete' => 'Fill in all the fields in order to submit the request.',
			'webcheck.form.intro' => 'Fill in the fields below and click on "Submit" to load the details of your reservation. Once loaded, you\'ll be able to add the missing information in order to speed up the vehicle collection on the day your rental starts.',
			'webcheck.form.surname' => 'Surname of client/main driver',
			'webcheck.form.voucher' => 'Voucher No. (e.g. "AB-12345678")',

			'webcheck.attachment.select' => 'Select file…',

			'webcheck.client_driver.0' => 'Main driver',
			'webcheck.client_driver.0.desc' => 'Insert the details of the main driver.',
			'webcheck.client_driver.0.id' => 'Type of identification',
			'webcheck.client_driver.0.id.desc' => 'Insert the details of the identification document (identity card, passport, …) of the main driver and attach a photo of both sides (front and back).',
			'webcheck.client_driver.0.license' => 'Driver\'s license',
			'webcheck.client_driver.0.license.desc' => 'Insert the details of the main driver\'s license and attach a photo of both sides (front and back)',
			'webcheck.client_driver.1' => 'Secondary driver',
			'webcheck.client_driver.1.desc' => 'Insert the details of the secondary driver. Please note that adding a secondary driver to your rental might imply additional costs.',
			'webcheck.client_driver.1.id' => 'Type of identification',
			'webcheck.client_driver.1.id.desc' => 'Insert the details of the identification document (identity card, passport, …) of the secondary driver and attach a photo of both sides (front and back).',
			'webcheck.client_driver.1.license' => 'Driver\'s license',
			'webcheck.client_driver.1.license.desc' => 'Insert the details of the secondary driver\'s license and attach a photo of both sides (front and back)',
			'webcheck.client_driver.1.show' => 'Add another driver…',

			'webcheck.client.address' => 'Address',
			'webcheck.client.city' => 'City',
			'webcheck.client.country_id' => 'Country',
			'webcheck.client.date_of_birth' => 'Date of birth (DD/MM/YYYY)',
			'webcheck.client.place_of_birth' => 'Place of birth',
			'webcheck.client.email' => 'Email',
			'webcheck.client.first_name' => 'Name',
			'webcheck.client.name' => 'Surname',
			'webcheck.client.phone' => 'Phone',
			'webcheck.client.province' => 'Province/State',
			'webcheck.client.tax_number' => 'Tax code',
			'webcheck.client.vat_number' => 'VAT number',
			'webcheck.client.zip_code' => 'Postal code',

			'webcheck.client_driver.address' => 'Address',
			'webcheck.client_driver.city' => 'City',
			'webcheck.client_driver.country_id' => 'Country',
			'webcheck.client_driver.date_of_birth' => 'Date of birth (DD/MM/YYYY)',
			'webcheck.client_driver.place_of_birth' => 'Place of birth',
			'webcheck.client_driver.email' => 'Email',
			'webcheck.client_driver.file_id' => 'Attach copy of ID (front)',
			'webcheck.client_driver.file_id_2' => 'Attach copy of ID (back)',
			'webcheck.client_driver.file_license' => 'Attach license (front)',
			'webcheck.client_driver.file_license_2' => 'Attach license (back)',
			'webcheck.client_driver.first_name' => 'Name',
			'webcheck.client_driver.id_expiry_date' => 'Expiry date (DD/MM/YYYY)',
			'webcheck.client_driver.id_issue_date' => 'Issue date (DD/MM/YYYY)',
			'webcheck.client_driver.id_issuer' => 'Issuer',
			'webcheck.client_driver.id_issuer_locality' => 'Issue location (Italian city or foreign country)',
			'webcheck.client_driver.id_number' => 'Document no.',
			'webcheck.client_driver.id_type' => 'Document type',
			'webcheck.client_driver.license_expiry_date' => 'License expiry date (DD/MM/YYYY)',
			'webcheck.client_driver.license_issue_date' => 'License issue date (DD/MM/YYYY)',
			'webcheck.client_driver.license_issuer' => 'License issuer',
			'webcheck.client_driver.license_issuer_locality' => 'License issue location (Italian city or foreign country)',
			'webcheck.client_driver.license_number' => 'License no.',
			'webcheck.client_driver.name' => 'Surname',
			'webcheck.client_driver.nationality' => 'Nationality',
			'webcheck.client_driver.phone' => 'Phone',
			'webcheck.client_driver.province' => 'Province/State',
			'webcheck.client_driver.tax_number' => 'Tax number',
			'webcheck.client_driver.zip_code' => 'Postal code',

			'webcheck.error' => 'We\'ve found some issues with your inserted details. Please make sure you have inserted all the requested data in the correct format and try again.',
			'webcheck.error.expired' => 'Expired document',
			'webcheck.error.invalidFormat' => 'Invalid format',
			'webcheck.intro' => 'Below you can find the requested details for your reservation {{ref_id}}. Fill in the empty fields, attach the requested documents and click on "Submit request" in order to complete your reservation.',
			'webcheck.loading' => '<b>Please wait…<br />We\'re processing your request.<br />Do not close this page.</b><br /><br />Estimated waiting time: 30"',
			'webcheck.outro' => 'Click on "Submit request" to transfer the provided information to our system. By clicking on "Submit request" you confirm that you have read our Privacy Policy.',
			'webcheck.outro.title' => 'Submit request',
			'webcheck.submit' => 'Submit request',
			'webcheck.submit.progress' => 'Submitting data to our system…',
			'webcheck.submit.success' => 'The transfer has completed successfully. Thanks for choosing us!',

			'weekday.0' => 'Monday',
			'weekday.1' => 'Tuesday',
			'weekday.2' => 'Wednesday',
			'weekday.3' => 'Thursday',
			'weekday.4' => 'Friday',
			'weekday.5' => 'Saturday',
			'weekday.6' => 'Sunday',
		]
	);
}

function crsbkr_session_new()
{
	if(!session_id())
	{
		session_start();
	}
}

add_action('init', 'crsbkr_session_new');

function crsbkr_init()
{
	$obj = new CRSBKR_BOOKER(get_user_locale());

	wp_register_script('crsbkr_script', plugins_url('/crs_booker.js', __FILE__), [ 'jquery' ], CRSBKR_VERSION, [ 'in_footer' => FALSE ]);
	wp_register_style('crsbkr_style', plugins_url('/crs_booker.css', __FILE__), [], CRSBKR_VERSION);

// 	add_action('wp_enqueue_scripts', array($obj, 'add_scripts'));

	add_action('wp_ajax_nopriv_crsbkr_ajax', array($obj, 'ajax'));
	add_action('wp_ajax_crsbkr_ajax', array($obj, 'ajax'));

	add_shortcode('crsbkr_module', array($obj, 'crsbkr_module'));
	add_shortcode('crsbkr_search_results', array($obj, 'crsbkr_search_results'));
	add_shortcode('crsbkr_webcheck', array($obj, 'crsbkr_webcheck'));

	add_action('admin_init', array($obj, 'crsbkr_register_settings'));
	add_action('admin_menu', array($obj, 'crsbkr_register_options_page'));

	// Settings link on plugin page.
	add_filter('plugin_action_links', 'crsbkr_settings_link', 10, 4);

	function crsbkr_settings_link($actions, $pluginFile, $pluginData, $context)
	{
		if ($pluginFile != 'crs-booker/crsbkr.php')
		{
			return $actions;
		}

		// Build and escape the URL.
		$url = esc_url(add_query_arg('page', 'crsbkr', get_admin_url() . 'admin.php'));

		// Create the link.
		$actions[] = "<a href='$url'>" . __('Settings', 'crs-booker') . '</a>';

		return $actions;
	}

	global $wp;
	$wp->add_query_var('q');
	$wp->add_query_var('mpi');
	$wp->add_query_var('mpe');
}

add_action('init', 'crsbkr_init');

?>
